/******************************************************************
 * @file   ak_led.c
 * @author chunfeng wei
 * @date   2008-10-10
 * 
 * @brief  Thunderbird led driver
 * 
 ****************************************************************** 
 */

#include <linux/module.h>
#include <linux/moduleparam.h>

#include <linux/init.h>
#include <linux/poll.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>

#include <linux/kernel.h>
#include <asm/arch/gpio.h>
#include <asm/arch/at91_pio.h>
#include <linux/proc_fs.h>
#include <linux/cdev.h>
#include "ak_led.h"

static atomic_t led_available = ATOMIC_INIT(1);

static int ak_led_major;
module_param(ak_led_major, int, 0);

MODULE_LICENSE("Dual BSD/GPL");

typedef struct {
    int pin_led_;
} led_pins_t;

typedef struct {
    led_pins_t *led_;
    int num_;
    struct cdev cdev_;
}led_dev_t;

static led_pins_t g_led_pins[] = {
    {
        .pin_led_ = SYS_LED,
    },
};

led_dev_t ak_led_dev = {
    .led_ = g_led_pins,
    .num_ = ARRAY_SIZE(g_led_pins),
};


int led_read_procmem (char *buf, char **start, off_t offset,
                         int count, int *eof, void *data)
{
    int n;
    n = sprintf(buf, "%s", "hello led's proc \n");
	*eof = 1;
	return n;
}

static int ak_led_release (struct inode *inode, struct file *filp)
{
	atomic_inc(&led_available); /* release the device */
	return 0;
}

static int ak_led_open (struct inode *inode, struct file *filp)
{

	led_dev_t *dev; /* device information */

/* 	if (! atomic_dec_and_test (&led_available)) { */
/* 		atomic_inc(&led_available); */
/* 		return -EBUSY; /\* already open *\/ */
/* 	} */

	dev = container_of(inode->i_cdev, led_dev_t, cdev_);
	filp->private_data = dev; /* for other methods */

	return 0;
}

static int ak_led_ioctl (struct inode *inode, struct file *filp,
                         unsigned int cmd, unsigned long arg)
{

	int err = 0;
	led_dev_t *dev;
    int led_status;

	/* don't even decode wrong cmds: better returning  ENOTTY than EFAULT */
	if (_IOC_TYPE(cmd) != AK_LED_IOC_MAGIC) return -ENOTTY;
	if (_IOC_NR(cmd) > AK_LED_IOC_MAXNR) return -ENOTTY;

	/*
	 * the type is a bitmask, and VERIFY_WRITE catches R/W
	 * transfers. Note that the type is user-oriented, while
	 * verify_area is kernel-oriented, so the concept of "read" and
	 * "write" is reversed
	 */
	if (_IOC_DIR(cmd) & _IOC_READ)
		err = !access_ok(VERIFY_WRITE, (void __user *)arg, _IOC_SIZE(cmd));
	else if (_IOC_DIR(cmd) & _IOC_WRITE)
		err =  !access_ok(VERIFY_READ, (void __user *)arg, _IOC_SIZE(cmd));
	if (err)
		return -EFAULT;

	switch(cmd) {
	case AK_SYS_LED_FLASH:
        dev = filp->private_data;
        led_status = at91_get_gpio_value(dev->led_[SYS_LED_ID].pin_led_);
        at91_set_gpio_value(dev->led_[SYS_LED_ID].pin_led_, !(led_status));
        break;
        
	default:  /* redundant, as cmd was checked against MAXNR */
		return -ENOTTY;
	}

    return 0;
}

                            

struct file_operations led_fops = {
	.owner =     THIS_MODULE,
	.open =	     ak_led_open,
    .ioctl =     ak_led_ioctl,
	.release =   ak_led_release,
};

/** 
 * @param dev 
 * @param devno 
 * 
 * @return 0 on success
 */
static int led_dev_setup (led_dev_t *dev, dev_t devno)
{
    int err, i;
    
    for (i = 0; i < dev->num_; ++i)
        at91_set_gpio_output(dev->led_[i].pin_led_, 0);

	cdev_init(&dev->cdev_, &led_fops);
	dev->cdev_.owner = THIS_MODULE;
	err = cdev_add (&dev->cdev_, devno, 1);

	if (err) {
		printk(KERN_NOTICE "Error %d LED cdev setup \n", err);
        return -1;
    }

    return 0;
}

static void led_dev_destroy (led_dev_t *dev)
{
    cdev_del(&dev->cdev_);
	unregister_chrdev_region(MKDEV(ak_led_major, 0), 1);
}

static int ak_led_init (void)
{
	int result;
	dev_t dev = MKDEV(ak_led_major, 0);
	
	/*
	 * Register your major, and accept a dynamic number.
	 */
	if (ak_led_major)
		result = register_chrdev_region(dev, 1, "ak_led");
	else {
		result = alloc_chrdev_region(&dev, 0, 1, "ak_led");
		ak_led_major = MAJOR(dev);
	}
    
	if (result < 0)
		return result;

    result = led_dev_setup(&ak_led_dev, dev);

    if (result)
        unregister_chrdev_region(dev, 1);
    
#ifdef LED_USE_PROC /* only when available */
	create_proc_read_entry("ak_led", 0, NULL, led_read_procmem, NULL);
#endif

	return result;
}

static void ak_led_exit (void)
{

    led_dev_destroy(&ak_led_dev);
	printk(KERN_ALERT "Goodbye, LED \n");
}
 
module_init(ak_led_init);
module_exit(ak_led_exit);
