/******************************************************************
 * @file   token.h
 * @author Richard Luo
 * @date   2008-11-21
 * 
 * @brief  
 * 
 ****************************************************************** 
 */

#ifndef _TOKEN_H
#define _TOKEN_H

#include <linux/circ_buf.h>

struct token_head {
    u8 magic[2];                // ky
    u16 check;
    u16 len;
    u16 data[10];
};

#define FNET_SERIAL_BUF_SIZE 2048

static inline void serial_buf_clear(struct circ_buf *cb)
{
	cb->head = cb->tail = 0;
}

/*
 * serial_buf_alloc
 *
 * Allocate a circular buffer and all associated memory.
 */
static inline struct circ_buf *serial_buf_alloc(void)
{
	struct circ_buf *cb;
	cb = kmalloc(sizeof(struct circ_buf), GFP_KERNEL);
	if (cb == NULL)
		return NULL;
	cb->buf = kmalloc(FNET_SERIAL_BUF_SIZE, GFP_KERNEL);
	if (cb->buf == NULL) {
		kfree(cb);
		return NULL;
	}
	serial_buf_clear(cb);
	return cb;
}

/*
 * serial_buf_free
 *
 * Free the buffer and all associated memory.
 */
static inline void serial_buf_free(struct circ_buf *cb)
{
	kfree(cb->buf);
	kfree(cb);
}

/*
 * serial_buf_data_avail
 *
 * Return the number of bytes of data available in the circular
 * buffer.
 */
static inline int serial_buf_data_avail(struct circ_buf *cb)
{
	return CIRC_CNT(cb->head,cb->tail,FNET_SERIAL_BUF_SIZE);
}

/*
 * serial_buf_put
 *
 * Copy data data from a user buffer and put it into the circular buffer.
 * Restrict to the amount of space available.
 *
 * Return the number of bytes copied.
 */
static inline int serial_buf_put(struct circ_buf *cb, const char *buf, int count)
{
	int c, ret = 0;
	while (1) {
		c = CIRC_SPACE_TO_END(cb->head, cb->tail, FNET_SERIAL_BUF_SIZE);
		if (count < c)
			c = count;
		if (c <= 0)
			break;
		memcpy(cb->buf + cb->head, buf, c);
		cb->head = (cb->head + c) & (FNET_SERIAL_BUF_SIZE-1);
		buf += c;
		count -= c;
		ret= c;
	}
	return ret;
}

/*
 * serial_buf_get
 *
 * Get data from the circular buffer and copy to the given buffer.
 * Restrict to the amount of data available.
 *
 * Return the number of bytes copied.
 */
static inline int serial_buf_get(struct circ_buf *cb, char *buf, int count)
{
	int c, ret = 0;
	while (1) {
		c = CIRC_CNT_TO_END(cb->head, cb->tail, FNET_SERIAL_BUF_SIZE);
		if (count < c)
			c = count;
		if (c <= 0)
			break;
		memcpy(buf, cb->buf + cb->tail, c);
		cb->tail = (cb->tail + c) & (FNET_SERIAL_BUF_SIZE-1);
		buf += c;
		count -= c;
		ret= c;
	}
	return ret;
}

static inline int serial_buf_printf (struct circ_buf *cb, const char * fmt, ...)
{
	va_list args;
    char tmpbuf[128];
    int n, need, limit;

    limit = serial_buf_data_avail(cb);

    if (limit > sizeof(tmpbuf) )
        limit = sizeof(tmpbuf);

	/* 
	 * First, try the static array 
	 */
	va_start(args,fmt);
	need = vsnprintf(tmpbuf, limit, fmt, args);
	va_end(args);

    n = need < limit ? need : limit;

    serial_buf_put(cb, tmpbuf, n);

    return n;
}


#endif /* _TOKEN_H */



