/*
 * (C) Copyright 2003
 * Author : Hamid Ikdoumi (Atmel)
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <at91_net.h>
#include <net.h>
#include <phy.h>

#ifdef CONFIG_DRIVER_ETHER

#if (CONFIG_COMMANDS & CFG_CMD_NET)

static unsigned char AT91C_PHY_ADDR;

/*
 * Name:
 *	dm9161a_IsPhyConnected
 * Description:
 *	Reads the 2 PHY ID registers
 * Arguments:
 *	p_mac - pointer to AT91S_EMAC struct
 * Return value:
 *	TRUE - if id read successfully
 *	FALSE- if error
 */
static unsigned int IsPhyConnected (AT91PS_EMAC p_mac)
{
	unsigned short Id1, Id2;
    unsigned short OUI_high, OUI_low;
    unsigned char model_number, revision_number;
    unsigned char phy_addr;
    unsigned char rc = FALSE;

    for (phy_addr=0; phy_addr<0x20; phy_addr++) {
        at91_EmacEnableMDIO (p_mac);
        at91_EmacReadPhy (p_mac, phy_addr, PHYID1, &Id1);
        at91_EmacReadPhy (p_mac, phy_addr, PHYID2, &Id2);
        at91_EmacDisableMDIO (p_mac);

        OUI_low = Id1;
        OUI_high = (Id2 & 0xfc00) >> 10;
        model_number = (Id2 & 0x03f0) >> 4;
        revision_number = Id2 & 0x0f;

        if ((OUI_low == KSZ8041NL_OUI_LOW) && (OUI_high == KSZ8041NL_OUI_HIGH)) {
            printf("PHY Detected: MICREL KSZ8041NL, Model Number = %x, Revsion Number = %x\n", model_number, revision_number);
            rc = TRUE;
            break;
        }
        else if ((OUI_low == LAN8700_OUI_LOW) && (OUI_high == LAN8700_OUI_HIGH)) {
            printf("PHY Detected: SMSC LAN8700, Model Number = %x, Revsion Number = %x\n", model_number, revision_number);
            rc =  TRUE;
            break;
        }
        else if ((OUI_low == RTL8201CP_OUI_LOW) && (OUI_high == RTL8201CP_OUI_HIGH)) {
            printf("PHY Detected: Realtek RTL8201CP, Model Number = %x, Revsion Number = %x\n", model_number, revision_number);
            rc =  TRUE;
            break;
        }
    }
    
    if (rc)
        AT91C_PHY_ADDR = phy_addr;
    else   
        printf("PHY Detected: Can't Find PHY\n");

	return rc;
}

/*
 * Name:
 *	dm9161a_GetLinkSpeed
 * Description:
 *	Link parallel detection status of MAC is checked and set in the
 *	MAC configuration registers
 * Arguments:
 *	p_mac - pointer to MAC
 * Return value:
 *	TRUE - if link status set succesfully
 *	FALSE - if link status not set
 */
static uchar GetLinkSpeed (AT91PS_EMAC p_mac)
{
	unsigned short stat;

	if (!at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, BMSR, &stat))
		return FALSE;
    
	if (!(stat & LINK_STATUS))	/* link status up? */
		return FALSE;

	if (stat & SPEED_100BASE_TX_FD) {
		/*set Emac for 100BaseTX and Full Duplex  */
		p_mac->EMAC_NCFGR |= AT91C_EMAC_SPD | AT91C_EMAC_FD;
		return TRUE;
	}

	if (stat & SPEED_10BASE_T_FD) {
		/*set MII for 10BaseT and Full Duplex  */
		p_mac->EMAC_NCFGR = (p_mac->EMAC_NCFGR &
				~(AT91C_EMAC_SPD | AT91C_EMAC_FD))
				| AT91C_EMAC_FD;
		return TRUE;
	}

	if (stat & SPEED_100BASE_TX_HD) {
		/*set MII for 100BaseTX and Half Duplex  */
		p_mac->EMAC_NCFGR = (p_mac->EMAC_NCFGR &
				~(AT91C_EMAC_SPD | AT91C_EMAC_FD))
				| AT91C_EMAC_SPD;
		return TRUE;
	}

	if (stat & SPEED_10BASE_T_HD) {
		/*set MII for 10BaseT and Half Duplex  */
		p_mac->EMAC_NCFGR &= ~(AT91C_EMAC_SPD | AT91C_EMAC_FD);
		return TRUE;
	}

	return FALSE;
}


/*
 * Name:
 *	dm9161a_ResetPhy
 * Description:
 *	Resets PHY
 * Arguments:
 *	p_mac - pointer to struct AT91S_EMAC
 * Return value:
 *	TRUE - if link status set succesfully
 *	FALSE - if link status not set
 */
static uchar ResetPhy(AT91PS_EMAC p_mac)
{
    unsigned short bmcr;
	int timeout = 10;
	uchar ret = TRUE;

	bmcr = RESET;
	at91_EmacEnableMDIO (p_mac);

	at91_EmacWritePhy (p_mac, AT91C_PHY_ADDR, BMCR, &bmcr);

	do {
	  at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, BMCR, &bmcr);
	  at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, BMCR, &bmcr);
	  timeout--;
	} while ((bmcr & RESET) && timeout);

	at91_EmacDisableMDIO (p_mac);

	if (!timeout)
	  ret = FALSE;

	return ret;
}


/*
 * Name:
 *	dm9161a_InitPhy
 * Description:
 *	MAC starts checking its link by using parallel detection and
 *	Autonegotiation and the same is set in the MAC configuration registers
 * Arguments:
 *	p_mac - pointer to struct AT91S_EMAC
 * Return value:
 *	TRUE - if link status set succesfully
 *	FALSE - if link status not set
 */
static uchar InitPhy (AT91PS_EMAC p_mac)
{
	unsigned int tick = 0;
	unsigned short IntValue;
	uchar ret = TRUE;

	at91_EmacEnableMDIO (p_mac);

	do {
		/* Link status is latched, so read twice to get current value */
		at91_EmacReadPhy(p_mac, AT91C_PHY_ADDR, BMSR, &IntValue);
		at91_EmacReadPhy(p_mac, AT91C_PHY_ADDR, BMSR, &IntValue);
	
		tick++;
	} while (!(IntValue & LINK_STATUS) && (tick < AT91C_ETH_TIMEOUT));

	if (!(IntValue & LINK_STATUS)){
		at91_EmacDisableMDIO (p_mac);
		printf ("No link\n\r");
		ret = FALSE;
		return ret;
	}

	if (!GetLinkSpeed (p_mac)) {
		/* Try another time */
		ret = GetLinkSpeed (p_mac);
	}

	return (ret);
}


/*
 * Name:
 *	ksz8041nl_AutoNegotiate
 * Description:
 *	MAC Autonegotiates with the partner status of same is set in the
 *	MAC configuration registers
 * Arguments:
 *	dev - pointer to struct net_device
 * Return value:
 *	TRUE - if link status set successfully
 *	FALSE - if link status not set
 */
static uchar AutoNegotiate (AT91PS_EMAC p_mac, int *status)
{
	unsigned short value;
	unsigned short PhyAnar;
	unsigned short PhyAnalpar;

	/* Set dm9161 control register */
	if (!at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, BMCR, &value))
		return FALSE;
	value &= ~AUTONEG;	/* remove autonegotiation enable */
	value |= ISOLATE;	/* Electrically isolate PHY */
	if (!at91_EmacWritePhy (p_mac, AT91C_PHY_ADDR, BMCR, &value))
		return FALSE;

	/* Set the Auto_negotiation Advertisement Register */
	/* MII advertising for Next page, 100BaseTxFD and HD, 10BaseTFD and HD, IEEE 802.3 */
	PhyAnar = NP | TX_FDX | TX_HDX | SPEED_10_FDX | SPEED_10_HDX | AN_IEEE_802_3;
	if (!at91_EmacWritePhy (p_mac, AT91C_PHY_ADDR, ANAR, &PhyAnar))
		return FALSE;

	/* Read the Control Register     */
	if (!at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, BMCR, &value))
		return FALSE;

	value |= SPEED_SELECT | AUTONEG | DUPLEX_MODE;
	if (!at91_EmacWritePhy (p_mac, AT91C_PHY_ADDR, BMCR, &value))
		return FALSE;
	/* Restart Auto_negotiation  */
	value |= RESTART_AUTONEG;
	if (!at91_EmacWritePhy (p_mac, AT91C_PHY_ADDR, BMCR, &value))
		return FALSE;

	/*check AutoNegotiate complete */
	udelay (10000);
	at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, BMSR, &value);
	if (!(value & AUTONEG_COMP))
		return FALSE;

	/* Get the AutoNeg Link partner base page */
	if (!at91_EmacReadPhy (p_mac, AT91C_PHY_ADDR, ANLPAR, &PhyAnalpar))
		return FALSE;

	if ((PhyAnar & TX_FDX) && (PhyAnalpar & TX_FDX)) {
		/*set MII for 100BaseTX and Full Duplex  */
		p_mac->EMAC_NCFGR |= AT91C_EMAC_SPD | AT91C_EMAC_FD;
		return TRUE;
	}

	if ((PhyAnar & SPEED_10_FDX) && (PhyAnalpar & SPEED_10_FDX)) {
		/*set MII for 10BaseT and Full Duplex  */
		p_mac->EMAC_NCFGR = (p_mac->EMAC_NCFGR &
				~(AT91C_EMAC_SPD | AT91C_EMAC_FD))
				| AT91C_EMAC_FD;
		return TRUE;
	}
	return FALSE;
}


static unsigned char GetAddr (void)
{
    return AT91C_PHY_ADDR;
}


/*
 * Name:
 *	at91_GetPhyInterface
 * Description:
 *	Initialise the interface functions to the PHY
 * Arguments:
 *	None
 * Return value:
 *	None
 */
void at91_GetPhyInterface(AT91PS_PhyOps p_phyops)
{
	p_phyops->Init           = InitPhy;
	p_phyops->IsPhyConnected = IsPhyConnected;
	p_phyops->GetLinkSpeed   = GetLinkSpeed;
	p_phyops->AutoNegotiate  = AutoNegotiate;
	p_phyops->Reset          = ResetPhy;
    p_phyops->GetAddr        = GetAddr;   
}

#endif	/* CONFIG_COMMANDS & CFG_CMD_NET */

#endif	/* CONFIG_DRIVER_ETHER */

