/*
 * $Id: ak4.c,v 1.14 2005/11/07 11:14:27 gleixner Exp $
 *
 * Handle mapping of the NOR flash on Cogent AK4 boards
 *
 * Copyright 2002 SYSGO Real-Time Solutions GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/autoconf.h>

#ifdef CONFIG_MTD_PARTITIONS
#include <linux/mtd/partitions.h>
#endif

#ifdef CONFIG_MTD_AK2_MAP
#include "ak2_partition_params.h"
#else
#error "CONFIG_MTD_AK4 or  CONFIG_MTD_AK2 must be defined!"
#endif


#define FLASH_BUS_WIDTH    2

/* can be "cfi_probe", "jedec_probe", "map_rom", NULL }; */
#define PROBETYPES { "cfi_probe", NULL }

#if defined CONFIG_MTD_AK4_MAP
#define AK_PREFIX "ak4"
#elif defined CONFIG_MTD_AK2_MAP
#define AK_PREFIX "ak2"
#endif

#define MSG_PREFIX AK_PREFIX"_nor:"   /* prefix for our printk()'s */
#define MTDID      AK_PREFIX"_mtd_id"    /* for mtdparts= partitioning */

static struct mtd_info *mymtd;

struct map_info ak4nor_map = {
	.name = "ak2_part_map",
	.bankwidth = FLASH_BUS_WIDTH,
	.size = PARTITION_FLASH_END_OFFSET,
    .phys = NORFLASH1_START_ADDR,
};

/*
 * MTD partitioning stuff
 */
static struct mtd_partition static_partitions[] =
{
	{                           /* mtd1 */
		.name = "bsp1 (4K)",
		.size = PARTITION_BSP1_SIZE,
		.offset = PARTITION_BSP1_OFFSET,
	},

	{                           /* mtd2 */
		.name = "bsp2 (4K)",
		.size = PARTITION_BSP2_SIZE,
		.offset = PARTITION_BSP2_OFFSET,
	},

	{                           /* mtd3 */
		.name = "uboot-env (4k)",
		.size = PARTITION_UBOOT_ENV_SIZE,
		.offset = PARTITION_UBOOT_ENV_OFFSET,
	},

	{                           /* mtd4 */
		.name = "u-boot (116k)",
		.size = PARTITION_UBOOT_SIZE,
		.offset = PARTITION_UBOOT_OFFSET,
	},

	{                           /* mtd5 */
		.name = "datafs (6.8M)",
		.size = PARTITION_DATAFS_SIZE,
		.offset = PARTITION_DATAFS_OFFSET,
	},

	{                           /* mtd6 */
		.name = "imagefs (9M)",
		.size = PARTITION_IMAGEFS_SIZE,
		.offset = PARTITION_IMAGEFS_OFFSET,
	},

};

//static const char *probes[] = { "RedBoot", "cmdlinepart", NULL };
static const char *probes[] = { NULL };

static int                   mtd_parts_nb = 0;
static struct mtd_partition *mtd_parts    = 0;

int __init init_ak4nor(void)
{
	static const char *rom_probe_types[] = PROBETYPES;
	const char **type;
	const char *part_type = 0;

    printk(KERN_NOTICE MSG_PREFIX "0x%08x at 0x%08x\n", PARTITION_DATAFS_SIZE, PARTITION_DATAFS_START);

	ak4nor_map.virt = ioremap(ak4nor_map.phys, ak4nor_map.size);

	if (!ak4nor_map.virt) {
		printk(MSG_PREFIX "failed to ioremap\n");
		return -EIO;
	}

	simple_map_init(&ak4nor_map);

	mymtd = 0;
	type = rom_probe_types;
	for(; !mymtd && *type; type++) {
		mymtd = do_map_probe(*type, &ak4nor_map);
	}
	if (mymtd) {
		mymtd->owner = THIS_MODULE;

#ifdef CONFIG_MTD_PARTITIONS
		mtd_parts_nb = parse_mtd_partitions(mymtd, probes, &mtd_parts, MTDID);
		if (mtd_parts_nb > 0)
            part_type = "detected";

		if (mtd_parts_nb == 0)
		{
			mtd_parts = static_partitions;
			mtd_parts_nb = ARRAY_SIZE(static_partitions);
			part_type = "static";
		}
#endif
		add_mtd_device(mymtd);
		if (mtd_parts_nb == 0)
            printk(KERN_NOTICE MSG_PREFIX "no partition info available\n");
		else
		{
			printk(KERN_NOTICE MSG_PREFIX
			       "using %s partition definition\n", part_type);
			add_mtd_partitions(mymtd, mtd_parts, mtd_parts_nb);
		}
		return 0;
	}

	iounmap((void *)ak4nor_map.virt);
	return -ENXIO;
}

static void __exit cleanup_ak4nor(void)
{
	if (mymtd) {
		del_mtd_device(mymtd);
		map_destroy(mymtd);
	}
	if (ak4nor_map.virt) {
		iounmap((void *)ak4nor_map.virt);
		ak4nor_map.virt = 0;
	}
}

module_init(init_ak4nor);
module_exit(cleanup_ak4nor);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Marius Groeger <mag@sysgo.de>");
MODULE_DESCRIPTION("Generic configurable MTD map driver");
