/*
 * snull.c --  the Simple Network Utility
 *
 * Copyright (C) 2001 Alessandro Rubini and Jonathan Corbet
 * Copyright (C) 2001 O'Reilly & Associates
 *
 * The source code in this file can be freely used, adapted,
 * and redistributed in source or binary form, so long as an
 * acknowledgment appears in derived source files.  The citation
 * should list that the code comes from the book "Linux Device
 * Drivers" by Alessandro Rubini and Jonathan Corbet, published
 * by O'Reilly & Associates.   No warranty is attached;
 * we cannot take responsibility for errors or fitness for use.
 *
 * $Id: snull.c,v 1.21 2004/11/05 02:36:03 rubini Exp $
 */

#include <linux/tty.h>
#include <linux/ioport.h>
#include <linux/serial.h>
#include <linux/clk.h>
#include <linux/console.h>
#include <linux/sysrq.h>
#include <linux/tty_flip.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/atmel_pdc.h>
#include <linux/serial_core.h>
#include <asm/io.h>
#include <asm/mach/serial_at91.h>
#include <asm/arch/board.h>
#include <asm/arch/cpu.h>
#include <asm/arch/gpio.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/moduleparam.h>
#include <linux/sched.h>
#include <linux/kernel.h> /* printk() */
#include <linux/slab.h> /* kmalloc() */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/interrupt.h> /* mark_bh */
#include <linux/in.h>
#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/tcp.h>         /* struct tcphdr */
#include <linux/skbuff.h>
#include <linux/in6.h>
#include <asm/checksum.h>
#include <linux/etherdevice.h>

#include "fntnet.h"
#include "akdebug.h"
#include "atmel_serial.h"
#include "token.h"

MODULE_AUTHOR("Richard Luo");
MODULE_DESCRIPTION("peer to peer over RS485");
MODULE_LICENSE("GPL");

/*
 * Transmitter lockup simulation, normally disabled.
 */
static int lockup = 0;
module_param(lockup, int, 0);

static int g_timeout = FNTNET_TIMEOUT;
module_param(g_timeout, int, 0);

int pool_size = 8;
module_param(pool_size, int, 0);


/*
 * A structure representing an in-flight packet.
 */
struct fntnet_packet {
	struct fntnet_packet *next;
	struct net_device *dev;
	int	datalen;
	u8 data[ETH_DATA_LEN];
};


struct net_device *fntnet_devs = 0;


static void fntnet_tx_timeout(struct net_device *dev);

/*
 * Set up a device's packet pool.
 */
void fntnet_setup_pool (struct net_device *dev)
{
	struct fntnet_priv *priv = netdev_priv(dev);
	int i;
	struct fntnet_packet *pkt;

	priv->ppool = NULL;
	for (i = 0; i < pool_size; i++) {
		pkt = kmalloc (sizeof (struct fntnet_packet), GFP_KERNEL);
		if (pkt == NULL) {
			printk (KERN_NOTICE "Ran out of memory allocating packet pool\n");
			return;
		}
		pkt->dev = dev;
		pkt->next = priv->ppool;
		priv->ppool = pkt;
	}
}

void fntnet_teardown_pool (struct net_device *dev)
{
	struct fntnet_priv *priv = netdev_priv(dev);
	struct fntnet_packet *pkt;
    
	while ((pkt = priv->ppool)) {
		priv->ppool = pkt->next;
		kfree (pkt);
		/* FIXME - in-flight packets ? */
	}
}    

/*
 * Buffer/pool management.
 */
struct fntnet_packet *fntnet_get_tx_buffer (struct net_device *dev)
{
	struct fntnet_priv *priv = netdev_priv(dev);
	unsigned long flags;
	struct fntnet_packet *pkt;
    
	spin_lock_irqsave(&priv->lock, flags);
	pkt = priv->ppool;
	priv->ppool = pkt->next;
	if (priv->ppool == NULL) {
		printk (KERN_INFO "Pool empty\n");
		netif_stop_queue(dev);
	}
	spin_unlock_irqrestore(&priv->lock, flags);
	return pkt;
}


void fntnet_release_buffer (struct fntnet_packet *pkt)
{
	unsigned long flags;
	struct fntnet_priv *priv = netdev_priv(pkt->dev);
	
	spin_lock_irqsave(&priv->lock, flags);
	pkt->next = priv->ppool;
	priv->ppool = pkt;
	spin_unlock_irqrestore(&priv->lock, flags);
	if (netif_queue_stopped(pkt->dev) && pkt->next == NULL)
		netif_wake_queue(pkt->dev);
}

void fntnet_enqueue_buf (struct net_device *dev, struct fntnet_packet *pkt)
{
	unsigned long flags;
	struct fntnet_priv *priv = netdev_priv(dev);

	spin_lock_irqsave(&priv->lock, flags);
	pkt->next = priv->rx_queue;  /* FIXME - misorders packets */
	priv->rx_queue = pkt;
	spin_unlock_irqrestore(&priv->lock, flags);
}

struct fntnet_packet *fntnet_dequeue_buf  (struct net_device *dev)
{
	struct fntnet_priv *priv = netdev_priv(dev);
	struct fntnet_packet *pkt;
	unsigned long flags;

	spin_lock_irqsave(&priv->lock, flags);
	pkt = priv->rx_queue;
	if (pkt != NULL)
		priv->rx_queue = pkt->next;
	spin_unlock_irqrestore(&priv->lock, flags);
	return pkt;
}

/*
 * Enable and disable receive interrupts.
 */
static void fntnet_rx_ints  (struct net_device *dev, int enable)
{
	struct fntnet_priv *priv = netdev_priv(dev);
	priv->rx_int_enabled = enable;
}

    
/*
 * Open and close
 */
int fntnet_open (struct net_device *dev)
{
	struct fntnet_priv *priv = netdev_priv(dev);

	ERR_INFO_RETURN (rs485_bus_ctor(priv),
                     "error for rs485_bus_ctor");

	memcpy(dev->dev_addr, "\0SNUL0", ETH_ALEN);

	netif_start_queue(dev);
	return 0;
}

int fntnet_release (struct net_device *dev)
{
    /* release ports, irq and such -- like fops->close */
	struct fntnet_priv *priv = netdev_priv(dev);

	netif_stop_queue(dev); /* can't transmit any more */
    rs485_bus_dtor(priv);
	return 0;
}

/*
 * Configuration changes (passed on by ifconfig)
 */
int fntnet_config (struct net_device *dev, struct ifmap *map)
{
	if (dev->flags & IFF_UP) /* can't act on a running interface */
		return -EBUSY;

	/* Don't allow changing the I/O address */
	if (map->base_addr != dev->base_addr) {
		printk(KERN_WARNING "snull: Can't change I/O address\n");
		return -EOPNOTSUPP;
	}

	/* Allow changing the IRQ */
	if (map->irq != dev->irq) {
		dev->irq = map->irq;
        	/* request_irq() is delayed to open-time */
	}

	/* ignore other fields */
	return 0;
}

/*
 * Receive a packet: retrieve, encapsulate and pass over to upper levels
 */
void fntnet_rx (struct net_device *dev, struct fntnet_packet *pkt)
{
	struct sk_buff *skb;
	struct fntnet_priv *priv = netdev_priv(dev);

	/*
	 * The packet has been retrieved from the transmission
	 * medium. Build an skb around it, so upper layers can handle it
	 */
	skb = dev_alloc_skb(pkt->datalen + 2);
	if (!skb) {
		if (printk_ratelimit())
			printk(KERN_NOTICE "snull rx: low on mem - packet dropped\n");
		priv->stats.rx_dropped++;
		goto out;
	}
	skb_reserve(skb, 2); /* align IP on 16B boundary */  
	memcpy(skb_put(skb, pkt->datalen), pkt->data, pkt->datalen);

	/* Write metadata, and then pass to the receive level */
	skb->dev = dev;
	skb->protocol = eth_type_trans(skb, dev);
	skb->ip_summed = CHECKSUM_UNNECESSARY; /* don't check it */
	priv->stats.rx_packets++;
	priv->stats.rx_bytes += pkt->datalen;
	netif_rx(skb);
  out:
	return;
}
    

#if 0
/*
 * The poll implementation.
 */
static int fntnet_poll (struct net_device *dev, int *budget)
{
	int npackets = 0, quota = min(dev->quota, *budget);
	struct sk_buff *skb;
	struct fntnet_priv *priv = netdev_priv(dev);
	struct fntnet_packet *pkt;
    
	while (npackets < quota && priv->rx_queue) {
		pkt = fntnet_dequeue_buf(dev);
		skb = dev_alloc_skb(pkt->datalen + 2);
		if (! skb) {
			if (printk_ratelimit())
				printk(KERN_NOTICE "snull: packet dropped\n");
			priv->stats.rx_dropped++;
			fntnet_release_buffer(pkt);
			continue;
		}
		skb_reserve(skb, 2); /* align IP on 16B boundary */  
		memcpy(skb_put(skb, pkt->datalen), pkt->data, pkt->datalen);
		skb->dev = dev;
		skb->protocol = eth_type_trans(skb, dev);
		skb->ip_summed = CHECKSUM_UNNECESSARY; /* don't check it */
		netif_receive_skb(skb);
		
        	/* Maintain stats */
		npackets++;
		priv->stats.rx_packets++;
		priv->stats.rx_bytes += pkt->datalen;
		fntnet_release_buffer(pkt);
	}
	/* If we processed all packets, we're done; tell the kernel and reenable ints */
	*budget -= npackets;
	dev->quota -= npackets;
	if (! priv->rx_queue) {
		netif_rx_complete(dev);
		fntnet_rx_ints(dev, 1);
		return 0;
	}
	/* We couldn't process everything. */
	return 1;
}
#endif
	    
        
/*
 * The typical interrupt entry point
 */
static void fntnet_regular_interrupt (int irq, void *dev_id, struct pt_regs *regs)
{
	int statusword;
	struct fntnet_priv *priv;
	struct fntnet_packet *pkt = NULL;
	/*
	 * As usual, check the "device" pointer to be sure it is
	 * really interrupting.
	 * Then assign "struct device *dev"
	 */
	struct net_device *dev = (struct net_device *)dev_id;
	/* ... and check with hw if it's really ours */

	/* paranoid */
	if (!dev)
		return;

	/* Lock the device */
	priv = netdev_priv(dev);
	spin_lock(&priv->lock);

	/* retrieve statusword: real netdevices use I/O instructions */
	statusword = priv->status;
	priv->status = 0;
	if (statusword & FNTNET_RX_INTR) {
		/* send it to fntnet_rx for handling */
		pkt = priv->rx_queue;
		if (pkt) {
			priv->rx_queue = pkt->next;
			fntnet_rx(dev, pkt);
		}
	}
	if (statusword & FNTNET_TX_INTR) {
		/* a transmission is over: free the skb */
		priv->stats.tx_packets++;
		priv->stats.tx_bytes += priv->tx_packetlen;
		dev_kfree_skb(priv->skb);
	}

	/* Unlock the device and we are done */
	spin_unlock(&priv->lock);
	if (pkt) fntnet_release_buffer(pkt); /* Do this outside the lock! */
	return;
}

/*
 * Transmit a packet (called by the kernel)
 */
int fntnet_tx (struct sk_buff *skb, struct net_device *dev)
{
	struct fntnet_priv *priv = netdev_priv(dev);

#ifdef TEST_TRANSFER_ERRORS_PERC
    static struct token_head tok = {
        .magic = { 'k', 'y' },
        .check = 0,
        .len = sizeof(tok),
        .data = { 0xaaaa, 0xbbbb, 0xcccc, 0xdddd, 0xeeee, 0xffff, 0x0000, 0x1111, 0x2222, 0x3333 }
    };

    kfree_skb(skb);

    tok.check = 0;         /* and rebuild the checksum (ip needs it) */
	tok.check = ip_fast_csum((unsigned char *)&tok, tok.len);

    skb = dev_alloc_skb(sizeof(tok) + 10);
    memcpy(skb_put(skb, sizeof(tok)), &tok, sizeof(tok));
    show_bytes("tx skb contents(tok): ", skb->data, skb->len);
#else

    PDEBUG("header_room: %d \n", skb_headroom(skb) );
    show_bytes("tx skb contents: ", skb->data, skb->len);

#endif    

    rs485_enqueue(&priv->rs485, skb);
	dev->trans_start = jiffies; /* save the timestamp */
    rs485_priv_start_tx(&priv->rs485);

	return 0; /* Our simple device can not fail */
}

/*
 * Deal with a transmit timeout.
 */
void fntnet_tx_timeout (struct net_device *dev)
{
	struct fntnet_priv *priv = netdev_priv(dev);

	PDEBUG("Transmit timeout at %ld, latency %ld\n", jiffies,
			jiffies - dev->trans_start);
        /* Simulate a transmission interrupt to get things moving */
	priv->status = FNTNET_TX_INTR;
	fntnet_regular_interrupt(0, dev, NULL);
	priv->stats.tx_errors++;
	netif_wake_queue(dev);
	return;
}

/*
 * Ioctl commands 
 */
int fntnet_ioctl (struct net_device *dev, struct ifreq *rq, int cmd)
{
	struct fntnet_priv *priv = netdev_priv(dev);

	if (!netif_running(dev))
		return -EINVAL;

	spin_lock_irq(&priv->lock);
	spin_unlock_irq(&priv->lock);
	return 0;
}

/*
 * Return statistics to the caller
 */
struct net_device_stats *fntnet_stats (struct net_device *dev)
{
	struct fntnet_priv *priv = netdev_priv(dev);
	return &priv->stats;
}

/*
 * This function is called to fill up an eth header, since arp is not
 * available on the interface
 */
int fntnet_rebuild_header (struct sk_buff *skb)
{
	u32 *s_addr, *d_addr;
	struct ethhdr *eth = (struct ethhdr *) skb->data;
	struct net_device *dev = skb->dev;
    struct iphdr *ih = (struct iphdr*)( ((char*)eth) + sizeof(struct ethhdr));

	memcpy(eth->h_source, dev->dev_addr, dev->addr_len);
	memcpy(eth->h_dest,   dev->dev_addr, dev->addr_len);

	s_addr = &ih->saddr;
	d_addr = &ih->daddr;

	eth->h_dest[ETH_ALEN-1] = ((u8 *)d_addr)[3];
	eth->h_source[ETH_ALEN-1] = ((u8 *)s_addr)[3];

	return 0;
}


int fntnet_header (struct sk_buff *skb, struct net_device *dev,
                   unsigned short type, void *daddr, void *saddr,
                   unsigned int len)
{
    unsigned char *pbeg = skb_push(skb,ETH_HLEN);
	struct ethhdr *eth = (struct ethhdr *) pbeg;
    struct iphdr *ih = (struct iphdr*)(pbeg + sizeof(struct ethhdr));
	u32 *s_addr, *d_addr;

	eth->h_proto = htons(type);
	memcpy(eth->h_source, saddr ? saddr : dev->dev_addr, dev->addr_len);
	memcpy(eth->h_dest,   daddr ? daddr : dev->dev_addr, dev->addr_len);

	s_addr = &ih->saddr;
	d_addr = &ih->daddr;

	eth->h_dest[ETH_ALEN-1] = ((u8 *)d_addr)[3];
	eth->h_source[ETH_ALEN-1] = ((u8 *)s_addr)[3];
	return (dev->hard_header_len);

}

/*
 * The "change_mtu" method is usually not needed.
 * If you need it, it must be like this.
 */
int fntnet_change_mtu (struct net_device *dev, int new_mtu)
{
	unsigned long flags;
	struct fntnet_priv *priv = netdev_priv(dev);
	spinlock_t *lock = &priv->lock;
    
	/* check ranges */
	if ((new_mtu < 68) || (new_mtu > 1500))
		return -EINVAL;
	/*
	 * Do anything you need, and the accept the value
	 */
	spin_lock_irqsave(lock, flags);
	dev->mtu = new_mtu;
	spin_unlock_irqrestore(lock, flags);
	return 0; /* success */
}

/*
 * Store the new hardware address in dev->dev_addr, and update the MAC.
 */
static int fntnet_set_mac_address (struct net_device *dev, void* addr)
{
    void dump_debug_cir_buf (void);
    void test_enable_retto (void);
	struct sockaddr *address = addr;

	if (!is_valid_ether_addr(address->sa_data))
		return -EADDRNOTAVAIL;

	memcpy(dev->dev_addr, address->sa_data, dev->addr_len);

	printk("%s: Setting MAC address to %02x:%02x:%02x:%02x:%02x:%02x\n", dev->name,
		dev->dev_addr[0], dev->dev_addr[1], dev->dev_addr[2],
		dev->dev_addr[3], dev->dev_addr[4], dev->dev_addr[5]);

//    dump_debug_cir_buf();

	return 0;
}



/*
 * The init function (sometimes called probe).
 * It is invoked by register_netdev()
 */
void fntnet_init (struct net_device *dev)
{
	struct fntnet_priv *priv;
#if 0
    	/*
	 * Make the usual checks: check_region(), probe irq, ...  -ENODEV
	 * should be returned if no device found.  No resource should be
	 * grabbed: this is done on open(). 
	 */
#endif

    	/* 
	 * Then, assign other fields in dev, using ether_setup() and some
	 * hand assignments
	 */
	ether_setup(dev); /* assign some of the fields */

	dev->open            = fntnet_open;
	dev->stop            = fntnet_release;
	dev->set_config      = fntnet_config;
	dev->hard_start_xmit = fntnet_tx;
	dev->do_ioctl        = fntnet_ioctl;
	dev->get_stats       = fntnet_stats;
	dev->change_mtu      = fntnet_change_mtu;  
	dev->rebuild_header  = fntnet_rebuild_header;
	dev->hard_header     = fntnet_header;
	dev->tx_timeout      = fntnet_tx_timeout;
	dev->set_mac_address = fntnet_set_mac_address;
    
	dev->watchdog_timeo = g_timeout;

	/* keep the default flags, just add NOARP */
	dev->flags           |= IFF_NOARP;
	dev->features        |= NETIF_F_NO_CSUM;
	dev->hard_header_cache = NULL;      /* Disable caching */

	/*
	 * Then, initialize the priv field. This encloses the statistics
	 * and a few private fields.
	 */
	priv = netdev_priv(dev);
	memset(priv, 0, sizeof(struct fntnet_priv));
	spin_lock_init(&priv->lock);
	fntnet_rx_ints(dev, 1);		/* enable receive interrupts */
	fntnet_setup_pool(dev);
}

/*
 * Finally, the module stuff
 */
static void __exit fntnet_cleanup (void)
{
    if (fntnet_devs) {
        unregister_netdev(fntnet_devs);
        fntnet_teardown_pool(fntnet_devs);
        free_netdev(fntnet_devs);
    }
}

static int __init fntnet_init_module (void)
{
	int result, ret = -ENOMEM;

	/* Allocate the devices */
	fntnet_devs = alloc_netdev (sizeof(struct fntnet_priv), "lando%d",
                                fntnet_init);
	if (fntnet_devs == NULL)
		goto out;

	ret = -ENODEV;

    if ( (result = register_netdev(fntnet_devs)) )
        printk("snull: error %i registering device \"%s\"\n",
               result, fntnet_devs->name);
    else
        ret = 0;
out:
	if (ret) 
		fntnet_cleanup();
	return ret;
}


module_init(fntnet_init_module);
module_exit(fntnet_cleanup);
