/******************************************************************
 * @file   psoc_prog.c
 * @author Richard Luo
 * @date   2008-09-17
 * 
 * @brief  
 * 
 ****************************************************************** 
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <asm/arch/at91_pmc.h>
#include <asm/arch/io.h>
#include <asm/arch/at91sam9260.h>
#include <linux/device.h>
#include <linux/firmware.h>

#include "psoc_bitbang.h"
#include "psoc_data.h"
#include "psoc_fw.h"
#include "akdebug.h"

MODULE_LICENSE("Dual BSD/GPL");

static int psoc_data_verify_procedure (void);
static void psoc_hw_init (void);
void psoc_dump_one_fw_record (const psoc_fw_record_t *prec);
void psoc_dump_one_fw_record_prefix (const char *prefix, const psoc_fw_record_t *prec);


static psoc_vector_t init_v1 = { 
    .data_ = {
        0x00000053, 0x00000000,
        0x00000000, 0x00000000,
        0xef8077b0, 0xe0f9e00d,
        0xbf813e7a, 0x05ef8057,
        0xf80ef9f8, 0x2fbe193e,
        0xf201ef81, 0xfef800fb,
        0x00000e91, 
    },
    .n_ = 396,
    .vec_type_ = VEC_TYPE_PURE_OUTPUT,
};

static psoc_vector_t init_v2 = { 
    .data_ = {
        0xdef8077b, 0xae0f9e00,
        0x7bf813e7, 0x805ef805,
        0xef80ef9f, 0x12fbe193,
        0xbf00be78, 0x03efb007,
        0x3a47fbe0, 
    },
    .n_ = 286,
    .vec_type_ = VEC_TYPE_PURE_OUTPUT,
};


static psoc_vector_t init_v3 = { 
    .data_ = {
        0x5ef8077b, 0x8057bf80,
        0xfbe20def, 0x7f3efc50,
        0xef862fbe, 0x0000e91f,
        0xbe01dec0, 0x15efe017,
        0xf8837be0, 0xcfbe0c3e,
        0xe18befab, 0xfef8477b,
        0x00000e91, 0x7be01dec,
        0x015efe01, 0xef8837be,
        0xfcfbf143, 0xbe18befd,
        0x0003a47f, 0xf8077b00,
        0x57bf805e, 0xe20def80,
        0x3ef830fb, 0x862fbe2f,
        0xfbe11def, 0x00003a47,
        0x00000000, 
    },
    .n_ = 836,
    .vec_type_ = VEC_TYPE_PURE_OUTPUT,
};


static psoc_vector_t id_setup_vector = { 
    .data_ = {
        0x0ef8477b, 0x8077be80,
        0xf9e00def, 0x813e7ae0,
        0xef8057bf, 0x0ef9f805,
        0xbe193ef8, 0x0be7812f,
        0xfb007be0, 0x7fbe003e,
        0x000003a4, 
    },
    .n_ = 330,
    .vec_type_ = VEC_TYPE_PURE_OUTPUT,
};

static psoc_vector_t read_id_out_v1 = { 
    .data_ = {
        0x000000fd, 
    },
    .n_ = 11,
    .vec_type_ = VEC_TYPE_PURE_OUTPUT,
};

static psoc_vector_t read_id_samp_v1 = { 
    .n_ = 10,
    .sp_ = {
        {0, SP_HIGH_Z_BEG},
        {9, SP_HIGH_Z_END},
        {0, 0},
    },
    .vec_type_ = SAMP_ENOUGH_NUM,
};

static psoc_vector_t read_id_out_v2 = { 
    .data_ = {
        0x000009fb, 
    },
    .n_ = 12,
    .vec_type_ = VEC_TYPE_PURE_OUTPUT,
};

static psoc_vector_t read_id_samp_v2 = { 
    .n_ = 10,
    .sp_ = {
        {0, SP_HIGH_Z_BEG},
        {9, SP_HIGH_Z_END},
        {0, 0},
    },
    .vec_type_ = SAMP_ENOUGH_NUM,
};

static psoc_vector_t read_id_out_v3 = { /* just one bit 1, wo Fnt ;) */
    .data_ = { 0x00001 },
    .n_ = 1,
    .vec_type_ = VEC_TYPE_PURE_OUTPUT,
};

static vector_grp_t read_group = {
    .grp_ = {
        &read_id_out_v1,
        &read_id_samp_v1,
        &read_id_out_v2,
        &read_id_samp_v2,
        &read_id_out_v3,
    },
    .n_ = 4,
};


static psoc_vector_t erase_all_vec = { 
    .data_ = {
        0xfe7d41f9, 0x8077bed4,
        0xf9e00def, 0x813e7ae0,
        0xef8057bf, 0x0ef9f805,
        0xbe193ef8, 0x01ef812f,
        0xf800fbf4, 0x000e91fe,
    },
    .n_ = 308,
    .vec_type_ = VEC_TYPE_PURE_OUTPUT,    
};

#ifdef UNIT_TEST
#include "testfuncs.h"
#endif

/** 
 * 
 * 
 * @param rec_addr addresss format in psoc_fw_record_t
 * @param block_no 
 * @param sram_addr 
 * 
 * @return 
 */
static int translate_address (uint16_t rec_addr, uint8_t *block_no, uint8_t *sram_addr)
{
    uint16_t v;
    *sram_addr = (rec_addr & 0x003f);   /* 0b:0000_0000 0011_1111 */

    v =  (rec_addr & 0x3fc0);   /* 0b:0011_1111 1100_0000 */
    v >>= 6;
    *block_no = v;
    
    if (*sram_addr > 64) return -1;
    return 0;
}

static int psoc_write_data_record (const psoc_fw_record_t *prec)
{
    uint8_t block_no, start_addr;

    if (prec->type_)
        return -1;              /* must be datablock */

    ERR_INFO_RETURN (translate_address(prec->saddr_, &block_no, &start_addr),
                     "error for translate_address");
    
    ak_debug("before write block, block_no:%d, start_addr:%d size:%d \n",
             block_no, start_addr, prec->num_);
    
    /* show_bytes("block data:", prec->data_, prec->num_); */

    /* ERR_INFO_RETURN (psoc_safe_write_data_block(prec->data_, prec->num_, block_no, start_addr), */
    /*                  "error: psoc_safe_write_data_block"); */

    ERR_INFO_RETURN (psoc_write_block(BLOCK_TYPE_DATA, prec->data_, prec->num_, block_no, start_addr),
                    "error for psoc_write_block");

    return 0;
}

static int psoc_validate_data_record (const psoc_fw_record_t *prec)
{
    int ret;
    uint8_t block_no, start_addr;
    uint8_t valid_buf[prec->num_];

    BUG_ON(prec->num_ > 64);

    ERR_INFO_RETURN ((prec->type_), "error for wrong record type");

    ERR_INFO_RETURN(translate_address(prec->saddr_, &block_no, &start_addr),
                    "error for translate_address");

    ERR_INFO_RETURN(read_from_block(valid_buf, sizeof(valid_buf), block_no, start_addr),
                    "error for read_from_block");

    ret = (0 == memcmp(valid_buf, prec->data_, prec->num_) ? 0 : -1);

    if (ret) {
        aways_show_bytes("==== right data:", prec->data_, prec->num_);
        aways_show_bytes("==== wrong data:", valid_buf, sizeof(valid_buf) );
    } else
        printk("verify block:%d ok \n", block_no);
    return ret;
}


static int psoc_write_secure_record (const psoc_fw_record_t *first, const psoc_fw_record_t *second)
{
    uint8_t block_no, start_addr;

    ERR_INFO_RETURN (translate_address(second->saddr_, &block_no, &start_addr),
                     "invalid address");
    
    printk("%s: block_no:%d start_addr:%d \n", __func__, block_no, start_addr);

    return psoc_write_block(BLOCK_TYPE_SECURE_DATA, second->data_, second->num_, block_no, start_addr);
}

static int psoc_dev_checksum_verify (const psoc_fw_record_t *first, const psoc_fw_record_t *second)
{
    uint8_t ch, cl;
    vector_grp_t *vgrp;

    ERR_INFO_RETURN ( send_wait_poll_vector(get_program_vector(BLOCK_TYPE_DEV_CHECKSUM_DATA)),
                      "error for send_wait_poll_vector");

    vgrp = make_read_checksum_vector_group();
    psoc_process_vector_grp(vgrp);

    psoc_dump_vector("checksum read, byte1", vgrp->grp_[1]);
    psoc_dump_vector("checksum read, byte2", vgrp->grp_[3]);

    retrieve_read_byte(vgrp->grp_[1], &ch);
    retrieve_read_byte(vgrp->grp_[3], &cl);

    printk("checksum, ch:%02x cl:%02x \n", ch, cl);

    return 0;

}

static void psoc_hw_init (void)
{
    psoc_init_gpio();
}

static int psoc_check_setup_id (void)
{
    int i;
    uint16_t id = 0;
    psoc_vector_t *samp_v1 = read_group.grp_[1];
    psoc_vector_t *samp_v2 = read_group.grp_[3];

    psoc_process_vector_grp (&read_group);

    for (i=8; i >=1; --i, id <<= 1) 
        id |= test_bit(i, (void*)samp_v1->data_);    

    for (i=8; i >=1; --i, id <<= 1) 
        id |= test_bit(i, (void*)samp_v2->data_);

    printk("id:%04x \n", id);
    return (id == 0x00A0 ? 0 : -1);
}

static int psoc_initialize_procedure (void)
{
    psoc_reset();

    ERR_INFO_RETURN(send_wait_poll_vector(&init_v1),
                    "error for send_wait_poll_vector, init_v1");
    ERR_INFO_RETURN(send_wait_poll_vector(&init_v2),
                    "error for send_wait_poll_vector, init_v2");
    psoc_send_output_vecor(&init_v3);
    send_40_zeros();
    return 0;
}

static int psoc_verify_silicon_id_procedure (void)
{
    ERR_INFO_RETURN (send_wait_poll_vector(&id_setup_vector),
                     "error for id_setup_vector" );

    ERR_INFO_RETURN (psoc_check_setup_id(),
                     "error for error for psoc_check_setup_id");

    /* ERR_INFO_RETURN (psoc_wait_and_poll(), */
    /*                  "error psoc_wait_and_poll for silicon id"); */
    
    send_40_zeros();

    return 0;
}


static struct device psoc_prog_dev = {
	.bus_id    = "psoc_prog_dev",
};

void psoc_dump_one_fw_record (const psoc_fw_record_t *prec)
{
    int nline = prec->num_/16, i = 0, nremain = prec->num_%16;

    if ( nline > 0 ) 
        for (; i < nline; ++i) {
            printk("\t%04x: ", prec->saddr_+(i*16));
            show_bytes_f(prec->data_+(i*16), 16);
        }

    if ( nremain > 0 ) {
        printk("\t%04x: ", prec->saddr_+(nline*16));
        show_bytes_f(prec->data_+(nline*16), nremain);
        printk("\n");
    }
    
}



static int psoc_verify_data_record_cb (const psoc_fw_record_t *prec)
{
    psoc_fw_record_t *next_rec = 0;

    switch (prec->type_) {
    case FWV_DATA:
        ERR_INFO (psoc_validate_data_record(prec),
                  "error for psoc_validate_data_record");
        break;
    case FWV_END_OF_FILE:
        printk("no need to verify FWV_END_OF_FILE record \n");
        break;
        
    case FWV_EXTENDED_ADDR:
        ERR_INFO_RETURN ((1 != psoc_record_next(&next_rec)),
                         "EXTENDED record must have next one");

        if (next_rec->num_ == 64)
            printk("no need to verify secure_record \n");
        else if (next_rec->num_ == 2)
            printk("no need to verify dev_checksum_record \n");
        else ERR_INFO_RETURN (1, "invalid logic record in psoc_verify_data_record_cb");
        break;
    default:
        printk("invalid record type: %d \n", prec->type_);
        return -1;
    };
    return 0;
}

/** 
 * @brief to write one data record into the flash
 * 
 * @return 0 on successful writing
 */
static int psoc_write_data_record_cb (const psoc_fw_record_t *prec)
{
    switch (prec->type_) {
    case FWV_DATA:
        ERR_INFO_RETURN (psoc_write_data_record(prec),
                         "error for psoc_write_data_record");
        break;
    case FWV_END_OF_FILE:
        printk("reach the FWV_END_OF_FILE record \n");
        return 1;
        
    case FWV_EXTENDED_ADDR:
        printk("reached FWV_EXTENDED_ADDR record \n");
        return 1;

    default:
        printk("invalid record type: %d \n", prec->type_);
        return -1;
    };
    
    return 0;
}


static int psoc_dev_checksum_verify_cb (const psoc_fw_record_t *prec)
{
    psoc_fw_record_t *next_rec = 0;

    switch (prec->type_) {

    case FWV_DATA:
        printk("%s:ignore the FWV_DATA record \n", __func__);
        break;
    case FWV_END_OF_FILE:
        printk("ignore the FWV_END_OF_FILE record \n");
        break;
        
    case FWV_EXTENDED_ADDR:
        ERR_INFO_RETURN (1 != psoc_record_next(&next_rec),
                         "must can be continued");

        if (next_rec->num_ == 2) {
            printk("before psoc_dev_checksum_verify \n");            
            ERR_INFO_RETURN (psoc_dev_checksum_verify(prec, next_rec),
                             "error for psoc_dev_checksum_verify");
            return 1;
        }
        break;
    default:
        printk("invalid record type: %d \n", prec->type_);
        return -1;
    };
    
    return 0;
}

static int psoc_write_secure_data_cb (const psoc_fw_record_t *prec)
{
    psoc_fw_record_t *next_rec = 0;

    switch (prec->type_) {

    case FWV_DATA:
        printk("%s:ignore the FWV_DATA record \n", __func__);
        break;
    case FWV_END_OF_FILE:
        printk("ignore the FWV_END_OF_FILE record \n");
        break;
        
    case FWV_EXTENDED_ADDR:
        ERR_INFO_RETURN (1 != psoc_record_next(&next_rec),
                         "must can be continued");
        if (next_rec->num_ == 64) {
            printk("before psoc_write_secure_record \n");
            ERR_INFO_RETURN (psoc_write_secure_record(prec, next_rec),
                             "error for psoc_write_secure_record");
            return 1;           /* ended */
        }
        break;
    default:
        printk("invalid record type: %d \n", prec->type_);
        return -1;
    };
    
    return 0;
}


typedef int psoc_record_func_t (const psoc_fw_record_t *prec);
    
static int psoc_fw_foreach_record (const struct firmware *fw_entry,
                                   psoc_record_func_t *func, int start, int end)
{
    int ret, i;
    psoc_fw_record_t *prec;

    psoc_fw_set_firmware(fw_entry);

    /* go to the @c start record */
    for (i = 0, ret = psoc_record_next(&prec);
         ret > 0 && i < start;
         ret = psoc_record_next(&prec), ++i);

    ERR_INFO_RETURN ( (i != start), "error to find start in psoc_fw_foreach_record");

    /* prec alread point to record specified by @c start  */
    for ( ;ret >= 0 && i < end;
          ret = psoc_record_next(&prec), ++i) {
        int r = func(prec);
        ERR_INFO_RETURN ( (r < 0),
                         "error for psoc_fw_foreach_record.func");
        if (r == 1) return 0;
    }

    return 0;
}


static int psoc_program_procedure (void)
{

#if 0

    ERR_INFO_RETURN (psoc_data_verify_procedure(),
                     "error for psoc_data_verify_procedure");
    printk("verify prog  ok \n");

#else    


    ERR_INFO_RETURN (send_wait_poll_vector(&erase_all_vec),
                     "error for erase_all_vec");
    printk("erase ok \n");
    ERR_INFO_RETURN (psoc_fw_foreach_record(0, &psoc_write_data_record_cb, 0, 256),
                     "error for psoc_fw_foreach_record");
    printk("prog ok \n");
    
    ERR_INFO_RETURN (psoc_data_verify_procedure(),
                     "error for psoc_data_verify_procedure");
    printk("verify prog  ok \n");

    ERR_INFO_RETURN (psoc_fw_foreach_record(0, &psoc_write_secure_data_cb, 256, 300),
                     "error for psoc_fw_foreach_record");
    printk("security prog  ok \n");
    ERR_INFO_RETURN (psoc_fw_foreach_record(0, &psoc_dev_checksum_verify_cb, 256, 300),
                     "error for psoc_fw_foreach_record");

#endif    

    return 0;
}


static int psoc_data_verify_procedure (void)
{
    ERR_INFO_RETURN (psoc_fw_foreach_record(0, &psoc_verify_data_record_cb, 0, 300),
                     "error for psoc_fw_foreach_record");
    return 0;
}


#ifdef DUMP_RECORD
static int psoc_dump_one_logic_fw_record (const psoc_fw_record_t *prec)
{
    if (prec->type_ == FWV_DATA)
        psoc_dump_one_fw_record_prefix("FWV_DATA", prec);
    else if (prec->type_ == FWV_END_OF_FILE)
        psoc_dump_one_fw_record_prefix("FWV_END_OF_FILE", prec);
    else {
        psoc_fw_record_t *next_rec = 0;

        if (1 != psoc_record_next(&next_rec))
            return -1;              /* must can be continued */
        
        if (next_rec->num_ == 64) {
            psoc_dump_one_fw_record_prefix("FWV_SECURE_0", prec);
            psoc_dump_one_fw_record_prefix("FWV_SECURE_1", next_rec);
        }
        else if (next_rec->num_ == 2) {
            psoc_dump_one_fw_record_prefix("FWV_CHECKSUM_0", prec);
            psoc_dump_one_fw_record_prefix("FWV_CHECKSUM_1", next_rec);
        }
        else ERR_INFO_RETURN (1, "invalid logic record");
    }
    return 0;
}


/** 
 * @brief dump [start, end)
 * 
 * @param fw_entry 
 * @param start start record index
 * @param end start record index
 * 
 * @return 0 on success
 */
static int psoc_fw_dump ( const struct firmware *fw_entry, int start, int end )
{
    ERR_INFO_RETURN (psoc_fw_foreach_record(fw_entry, &psoc_dump_one_logic_fw_record, start, end),
                     "error for psoc_fw_foreach_record");
    return 0;
}
#endif

void psoc_dump_one_fw_record_prefix (const char *prefix, const psoc_fw_record_t *prec)
{
    printk("%s:\n", prefix);
    return psoc_dump_one_fw_record(prec);
}

static int psoc_firmware_probe (void)
{
    int ret = 0;
    const struct firmware *fw_entry;

    ERR_INFO_RETURN (request_firmware(&fw_entry, "psoc_prog", &psoc_prog_dev),
                     "firmware_sample_driver: Firmware not available");

    ERR_INFO_GOTO (psoc_validate_firmware(fw_entry),
                   "error for psoc_validate_firmware",
                   prog_failed);

    /* ERR_INFO_GOTO (psoc_fw_dump(fw_entry, 200, 258), */
    /*                "error for psoc_validate_firmware", */
    /*                prog_failed); */

    ERR_INFO_GOTO (psoc_initialize_procedure(),
                   "error for psoc_initialize_procedure",
                   prog_failed);

    ERR_INFO_GOTO (psoc_verify_silicon_id_procedure(),
                   "error for psoc_verify_silicon_id_procedure",
                   prog_failed);

    ERR_INFO_GOTO (psoc_program_procedure(),
                   "error for psoc_program_procedure",
                   prog_failed);
prog_failed:
    printk("=>psoc_exit_gpio and  psoc_reset\n");
    psoc_reset();
    psoc_exit_gpio();
	release_firmware(fw_entry);
    return ret;
}


static int psoc_prog_init (void)
{

#ifdef CONV_VECTOR
    issp_verctor_convert();
#elif defined UNIT_TEST
    run_unit_test();
#else    
	printk(KERN_ALERT "hello psoc programmer \n");
    psoc_hw_init();
	device_initialize(&psoc_prog_dev);

	ERR_INFO_RETURN (psoc_firmware_probe(),
                     "psoc_firmware_probe failed");
#endif
    return -1;
}

static void psoc_prog_exit (void)
{
	printk(KERN_ALERT "Goodbye, psoc programmer \n");
}

module_init(psoc_prog_init);
module_exit(psoc_prog_exit);
