/*
 * Copyright (C) 2017 Timesys Corp All Rights Reserved.
 *
 *    derived from ov5640.c - Copyright (C) 2012-2015 Freescale Semiconductor, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/pinctrl/consumer.h>
#include <linux/v4l2-mediabus.h>
#include <media/v4l2-device.h>
#include "ev76c454.h"

struct ev76c454 {
	struct v4l2_subdev		subdev;
	struct i2c_client *i2c_client;
	struct v4l2_mbus_framefmt fmt;
	struct clk *sensor_clk;
};

// main data structure
static struct ev76c454 ev76c454_data;

// gpios - pulled from device tree
static int pwr_gpio, rst_gpio;

// local forward references
static int ev76c454_probe(struct i2c_client *adapter, const struct i2c_device_id *device_id);
static int ev76c454_remove(struct i2c_client *client);
static s32 ev76c454_read_reg(u8 reg);
static s32 ev76c454_write_reg(u8 reg, u16 val);

static const struct i2c_device_id ev76c454_id[] = {
	{"ev76c454", 0},
	{},
};

MODULE_DEVICE_TABLE(i2c, ev76c454_id);

static struct i2c_driver ev76c454_i2c_driver = {
	.driver = {
		  .owner = THIS_MODULE,
		  .name  = "ev76c454",
		  },
	.probe  = ev76c454_probe,
	.remove = ev76c454_remove,
	.id_table = ev76c454_id,
};

static struct ev76c454 *to_ev76c454(const struct i2c_client *client)
{
	return container_of(i2c_get_clientdata(client), struct ev76c454, subdev);
}

static void ev76c454_hardware_reset(void)
{
	gpio_set_value_cansleep(rst_gpio, 1);
	msleep(1);
	gpio_set_value_cansleep(rst_gpio, 0);
	msleep(1);
	gpio_set_value_cansleep(rst_gpio, 1);
	msleep(1);
}

static s32 ev76c454_read_reg(u8 reg)
{
    struct i2c_msg msgs[2];
	u8 rdBuf[2];
	
	// register address of 0x80 and above are 16 bit
	int rdLength = reg & 0x80 ? 2 : 1;
    
    // send register address
    msgs[0].addr = ev76c454_data.i2c_client->addr;
    msgs[0].flags = 0;
    msgs[0].len = 1;
    msgs[0].buf = &reg;
    
    // receive data
    msgs[1].addr = ev76c454_data.i2c_client->addr;
    msgs[1].flags = I2C_M_RD;
    msgs[1].len = rdLength;
    msgs[1].buf = rdBuf;
    
    if(2 != i2c_transfer(ev76c454_data.i2c_client->adapter, msgs, 2)) {
		pr_err("%s error: reg=%x\n", __func__, reg);
		return -1;
	}	

	// flip endian for 16 bit
	return rdLength == 2 ? rdBuf[0] << 8 | rdBuf[1] : rdBuf[0];
}

static s32 ev76c454_write_reg(u8 reg, u16 val)
{
    struct i2c_msg msg;
	u8 wrBuf[3];
	
	// register address of 0x80 and above are 16 bit
	int wrLength = 1 + (reg & 0x80 ? 2 : 1);
    
    // send register address
    msg.addr = ev76c454_data.i2c_client->addr;
    msg.flags = 0; // send
    msg.len = wrLength;
    msg.buf = wrBuf;
	
	// register
	wrBuf[0] = reg;

	// data
	if(wrLength == 3) {	// 16 bit
		wrBuf[1] = val >> 8;
		wrBuf[2] = val & 0xff;
	}
	else {	// 8 bit
		wrBuf[1] = val;
	}
    
    if(1 != i2c_transfer(ev76c454_data.i2c_client->adapter, &msg, 1)) {
		pr_err("%s error: reg=%x\n", __func__, reg);
		return -1;
	}
	
	return 0;
}

#ifdef CONFIG_VIDEO_ADV_DEBUG
static int ev76c454_get_register(struct v4l2_subdev *sd, struct v4l2_dbg_register *reg)
{
	s32 val;

	// register address always 8 bits
	if (reg->reg & ~0xff)
		return -EINVAL;

	reg->size = reg->reg & 0x80 ? 2 : 1;

	val = ev76c454_read_reg(reg->reg);
	if (val < 0)
		return -EIO;

	reg->val = (__u64)val;

	return 0;
}

static int ev76c454_set_register(struct v4l2_subdev *sd, const struct v4l2_dbg_register *reg)
{
	s32 err;

	// register always 8 bits, value 8 or 16
	if (reg->reg & ~0xff)
		return -EINVAL;

	if(reg->val & (reg->reg & 0x80 ? ~0xffff : ~0xff))
		return -EINVAL;

	err = ev76c454_write_reg(reg->reg, reg->val);
	return err < 0 ? -EIO : 0;
}
#endif

// This is called at user-space open() and close()
static int ev76c454_s_power(struct v4l2_subdev *sd, int on)
{
	if (on) {
		clk_enable(ev76c454_data.sensor_clk);
		ev76c454_hardware_reset();
	}
	else
		clk_disable(ev76c454_data.sensor_clk);

	return 0;
}

static int ev76c454_s_fmt(struct v4l2_subdev *sd, struct v4l2_mbus_framefmt *mf)
{
	struct i2c_client *client = v4l2_get_subdevdata(sd);
	struct ev76c454 *sensor = to_ev76c454(client);

	dev_dbg(&client->dev, "%s\n", __func__);

	//ev76c454_try_fmt(sd, mf);
	sensor->fmt = *mf;

	return 0;
}

static int ev76c454_g_fmt(struct v4l2_subdev *sd, struct v4l2_mbus_framefmt *mf)
{
	struct i2c_client *client = v4l2_get_subdevdata(sd);
	struct ev76c454 *sensor = to_ev76c454(client);

	dev_dbg(&client->dev, "%s\n", __func__);

	*mf = sensor->fmt;

	return 0;
}

static struct v4l2_subdev_video_ops ev76c454_subdev_video_ops = {
	.s_mbus_fmt	= ev76c454_s_fmt,
	.g_mbus_fmt	= ev76c454_g_fmt,
};

static struct v4l2_subdev_core_ops ev76c454_subdev_core_ops = {
	.s_power	= ev76c454_s_power,
#ifdef CONFIG_VIDEO_ADV_DEBUG
	.g_register	= ev76c454_get_register,
	.s_register	= ev76c454_set_register,
#endif
};

static struct v4l2_subdev_ops ev76c454_subdev_ops = {
	.core	= &ev76c454_subdev_core_ops,
	.video	= &ev76c454_subdev_video_ops,
};

static int ev76c454_probe(struct i2c_client *client,
			const struct i2c_device_id *id)
{
	struct pinctrl *pinctrl;
	struct device *dev = &client->dev;
	int ret;
	u16 chip_id;
	u32 mclk;
	
	// *********************** main data structure *******************************

	memset(&ev76c454_data, 0, sizeof(ev76c454_data));
	ev76c454_data.i2c_client = client;
	ev76c454_data.fmt.width = DEFAULT_HORIZONTAL_WIDTH;
	ev76c454_data.fmt.height = DEFAULT_VERTICAL_HEIGHT;
	ev76c454_data.fmt.code = MEDIA_BUS_FMT_SBGGR8_1X8;
	ev76c454_data.fmt.field = V4L2_FIELD_NONE;
	ev76c454_data.fmt.colorspace = V4L2_COLORSPACE_JPEG;

	// **************************** pinmuxing ************************************

	pinctrl = devm_pinctrl_get_select_default(dev);
	if (IS_ERR(pinctrl)) {
		dev_err(dev, "setup pinctrl failed\n");
		return PTR_ERR(pinctrl);
	}

	// power enable pin
	pwr_gpio = of_get_named_gpio(dev->of_node, "pwr-gpios", 0);
	if (!gpio_is_valid(pwr_gpio)) {
		dev_err(dev, "no sensor pwr pin available\n");
		return -EINVAL;
	}
	ret = devm_gpio_request_one(dev, pwr_gpio, GPIOF_OUT_INIT_HIGH, "ev76c454_pwr");
	if (ret < 0)
		return ret;

	gpio_export(pwr_gpio, false);

	//reset pin
	rst_gpio = of_get_named_gpio(dev->of_node, "rst-gpios", 0);
	if (!gpio_is_valid(rst_gpio)) {
		dev_err(dev, "no sensor reset pin available\n");
		return -EINVAL;
	}
	ret = devm_gpio_request_one(dev, rst_gpio, GPIOF_OUT_INIT_LOW, "ev76c454_reset");
	if (ret < 0)
		return ret;

	
	gpio_export(rst_gpio, false);


	// ****************************** clock **************************************

	ev76c454_data.sensor_clk = devm_clk_get(dev, "csi_mclk");
	if (IS_ERR(ev76c454_data.sensor_clk)) {
		dev_err(dev, "get mclk failed\n");
		return PTR_ERR(ev76c454_data.sensor_clk);
	}

	ret = of_property_read_u32(dev->of_node, "mclk", &mclk);
	if (ret) {
		dev_err(dev, "mclk frequency is invalid\n");
		return ret;
	}

	dev_info(dev, "   Setting mclk from %d MHz to %d MHz\n",
			 (int)(clk_get_rate(ev76c454_data.sensor_clk) / 1000000), mclk / 1000000);
	ret = clk_set_rate(ev76c454_data.sensor_clk, mclk);
	if (ret < 0) {
		dev_err(dev, "set rate failed, rate=%d\n", mclk);
		return ret;
	}

	clk_prepare_enable(ev76c454_data.sensor_clk);

	dev_info(dev, "mclk now: %lu\n", clk_get_rate(ev76c454_data.sensor_clk));

	// ************************** hardware init **********************************

	ev76c454_hardware_reset();

	chip_id = ev76c454_read_reg(REG_fb_chip_id);
	if (chip_id != EV76C454_ID) {
		clk_disable_unprepare(ev76c454_data.sensor_clk);
		dev_err(dev, "camera ev76c454 is not found: 0x%04x\n", chip_id);
		return -ENODEV;
	}

	// ******************** v4l subdev registration  *****************************

	v4l2_i2c_subdev_init(&ev76c454_data.subdev, client, &ev76c454_subdev_ops);

	ret = v4l2_async_register_subdev(&ev76c454_data.subdev);
	if (ret < 0)
		dev_err(dev, "%s--Async register failed, ret=%d\n", __func__, ret);

	dev_info(dev, "camera ev76c454, is found\n");
	return ret;
}

static int ev76c454_remove(struct i2c_client *client)
{
	struct v4l2_subdev *sd = i2c_get_clientdata(client);

	v4l2_async_unregister_subdev(sd);

	clk_unprepare(ev76c454_data.sensor_clk);

	//ev76c454_power_down(1);

	return 0;
}

module_i2c_driver(ev76c454_i2c_driver);

MODULE_AUTHOR("Timesys Corporation");
MODULE_DESCRIPTION("EV76C454 Imager Driver");
MODULE_LICENSE("GPL");
MODULE_VERSION("1.0");
MODULE_ALIAS("CSI");
