/*!
 * \file    prng_test.c
 * \brief   Top-level test code for "gen_entropy.c".
 *
 * \par File contents
 * This file contains the code to set up the array (including array size) that
 * will hold the generated entropy values and call the "GenEntropy()" function.
 * Pointers to the key and the entropy array along with the required number of
 * bits for the entropy are passed to the "GenEntropy()" function as these are
 * set by the user. Although the entropy array could be declared within the
 * "GenEntropy()" function, it may be useful to observe the generated entropy,
 * thus it is declared in "prng_test.c". If TEST mode is selected, the pointers
 * to the key and the entropy array and the required number of bits for the
 * entropy are also passed to the "GenEntropyTest()" function and the entropy of
 * both "GenEntropy()" and "GenEntropyTest()" are displayed and passed to the
 * "RsaEntropyTest()" function to check if the generated entropy is suitable to
 * be used for encryption and decryption.
 */
#include "gen_entropy.h"

/*!
 * The number of bytes to be used to set up the "entropy" array.
 * 32 is chosen here so that 2 blocks of NIST validated
 * plaintext are read in and encrypted.
 * 32 bytes = 256 bits: the number of bits required by the
 * crypto library prng.
 */
#define NUM_BYTES_ENTROPY 32

int main( void )
{
    /*!
     * Set up arrays for the key and the entropy.
     * 128 bit NIST validated key - Hex value: 2b7e151628aed2a6abf7158809cf4f3c
     */
    static const unsigned char _ckey   [AES_BLOCK_SIZE] =
        {43,126,21,22,40,174,210,166,171,247,21,136,9,207,79,60};
    static unsigned char _entropy[NUM_BYTES_ENTROPY];

    GenEntropy( _ckey, _entropy, NUM_BYTES_ENTROPY );
#if TEST

    printf( "***************************************************************\n" );

    DisplayEntropy( _entropy, NUM_BYTES_ENTROPY );
    RsaEntropyTest( _entropy, NUM_BYTES_ENTROPY );
    GenEntropyTest( _ckey, _entropy, NUM_BYTES_ENTROPY );
    DisplayEntropy( _entropy, NUM_BYTES_ENTROPY );
    RsaEntropyTest( _entropy, NUM_BYTES_ENTROPY );

    printf( "***************************************************************\n" );

#endif /* TEST */

    return 0;
}
/******************************************************************************/
