/*!
 * \file    gen_entropy.c
 * \brief   Pseudo Random Number Generator for Entropy Generation
 *
 * \par File contents
 * This file contains the code to encrypt the plaintext and generate the entropy
 * required by the RAND functions. TEST mode allows known, validated values to
 * be encrypted and then decrypted to ensure that the code works properly.
 *
 * \note
 * By passing TEST = 1 from the Makefile to the preprocessor, test code is
 * compiled to do the following:
 * - Generate a "plaintext.txt" file with the validated NIST value.
 * - Generate a "nonce.txt" file with the validated NIST value.
 * - Initialse the "padding" and "local_ctr" to the validated NIST values.
 * - Create "ciphertext.txt" and write the encrypted ciphertext to it.
 * - Reset the "input_block" value to run decryption.
 * - Decrypt "ciphertext.txt" to retrieve the plaintext value.
 * - Create "decrypted_text.txt" and write the decrypted text to it.
 * - Display a table showing the "_ckey", "input_block", "decrypted_text",
 *   "ciphertext" along with the validated plaintext and ciphertext values for
 *   each block of plaintext read in and print a message informing whether the
 *   ciphertext and decrypted text match the validated NIST cipertext and
 *   plaintext values respectively.
 */
#include "gen_entropy.h"

/*!
 * Function prototypes / Forward declarations.
 * These are listed alphabetically here and in the main code.
 */
static void _CtrInc( unsigned char *input_block, unsigned int ctr );

static void _DecToHex( unsigned char *temp_large, unsigned char *temp_small,
                       unsigned int d2h_size );

static void _DisplayError( signed int error );

static void _Encrypt( const unsigned char *_ckey, unsigned char *_entropy,
                      unsigned char *input_block, unsigned char *temp_large,
                      unsigned char *temp_small,
                      unsigned int NUM_BYTES_ENTROPY );

static void _HexToDec( unsigned char *temp_large, unsigned char *temp_small,
                       unsigned int h2d_size );

static void _InBlkInit( unsigned char *input_block, unsigned char *temp_large,
                        unsigned char *temp_small );

static unsigned int _Weight( unsigned int power );

/*!
 * Structure instance defined in "aes.h" (included in "crypt.h").
 */
AES_KEY key;
/******************************************************************************/
/*!
 * \par Description
 * Call the "_InBlkInit()" function.
 * Call the "_Encrypt()" function.
 *
 * \param *_ckey
 * The pointer to the array holding the key value.
 *
 * \param *entropy
 * The pointer to the array set up to hold the generated entropy.
 *
 * \param NUM_BYTES_ENTROPY
 * The size of the "entropy" array - required by the "RAND_add()" and
 * "RAND_seed()" functions.
 *
 * \returns
 * N/A.
 */
void GenEntropy( const unsigned char *_ckey, unsigned char *_entropy,
                 unsigned int NUM_BYTES_ENTROPY )
{
    /*!
     * Declare arrays of set lengths.
     * AES_BLOCK_SIZE:   16 (bytes): defined in "aes.h".
     * AES_BLOCK_SIZE_2: 32 (bytes): defined above.
     * Declare a small and a large temporary array to store data that has been
     * read from a file or is to be written to a file or converted to a
     * different base.
     */
    unsigned char input_block   [AES_BLOCK_SIZE];
    unsigned char temp_small    [AES_BLOCK_SIZE];
    unsigned char temp_large    [AES_BLOCK_SIZE_2];

    _InBlkInit( input_block, temp_large, temp_small );
    _Encrypt( _ckey, _entropy, input_block, temp_large, temp_small, NUM_BYTES_ENTROPY );

    return;
}
/******************************************************************************/
/*!
 * \par Description
 * Calculate the value of "tracker" based on whether the function is to
 * increment the local or persistant counter, then increment the chosen counter
 * based on the value of "tracker".
 *
 * \param *input_block
 * The pointer to the array holding the "input_block" values.
 *
 * \param ctr
 * The ascii number representing either the character 'L' or the character 'P'.
 *
 * \returns
 * N/A.
 */
static void _CtrInc( unsigned char *input_block, unsigned int ctr )
{
    /*!
     * ctr_i   : Used to access individual elements of the arrays used in the
     *           "_CtrInc()" function.
     * ctr_j   : Used as the exponent when calculating the weight of the current
     *           counter element.
     * tracker : Keep track of the weight of the current counter element.
     */
    unsigned int ctr_i   = 0;
    unsigned int ctr_j   = 0;
    unsigned int tracker = 0;

    /*!
     * Calculate the value of "tracker" based on the data in the local counter
     * elements of the "input_block" array.
     */
    if ( ctr == 'L' )
    {
        for ( ctr_i = TWELVE, ctr_j = THREE; ctr_i < AES_BLOCK_SIZE; ctr_i++, ctr_j-- )
        {
            tracker += ( input_block[ctr_i] == ASCII_LIMIT ) ? ( _Weight( ctr_j ) ) : ( 0 );
        }
        ctr_i = AES_BLOCK_SIZE - ONE;
    }

    /*!
     * Calculate the value of "tracker" based on the data in the persistant
     * counter elements of the "input_block" array.
     */
    if ( ctr == 'P' )
    {
        for ( ctr_i = SIX, ctr_j = THREE; ctr_i < NONCE_BLOCK_SIZE; ctr_i++, ctr_j-- )
        {
            tracker += ( input_block[ctr_i] == ASCII_LIMIT ) ? ( _Weight( ctr_j ) ) : ( 0 );
        }
        ctr_i = NONCE_BLOCK_SIZE - ONE;
    }

    /*!
     * Based on the "ctr" value, increment or reset the elements of the
     * "input_block" array that correspond to the chosen counter.
     * The fallthrough can be used as the incrementation for each element
     * follows a pattern.
     * The default exits the program to ensure that no data is corrupted.
     */
    switch ( tracker )
    {
        case ZERO    :
        case TWO     :
        case FOUR    :
        case SIX     :
        case EIGHT   :
        case TEN     :
        case TWELVE  :
        case FOURTEEN:
            input_block[ctr_i]++;
            break;

        case ONE     :
        case FIVE    :
        case NINE    :
        case THIRTEEN:
            input_block[ctr_i]         = 0;
            input_block[ctr_i - ONE]++;
            break;

        case THREE   :
        case ELEVEN  :
            input_block[ctr_i]         = 0;
            input_block[ctr_i - ONE]   = 0;
            input_block[ctr_i - TWO]++;
            break;

        case SEVEN   :
            input_block[ctr_i]         = 0;
            input_block[ctr_i - ONE]   = 0;
            input_block[ctr_i - TWO]   = 0;
            input_block[ctr_i - THREE]++;
            break;

        case FIFTEEN :
            input_block[ctr_i]         = 0;
            input_block[ctr_i - ONE]   = 0;
            input_block[ctr_i - TWO]   = 0;
            input_block[ctr_i - THREE] = 0;
            break;

        default      :
            _DisplayError( FOUR );
            break;
    }

    return;
}
/******************************************************************************/
/*!
 * \par Description
 * Convert the decimal value of either "ciphertext", "decrypted_text" or
 * "input_block" to hexadecimal format.
 *
 * \param *temp_large
 * The pointer to the temporary array holding the hexadecimal value of either
 * "ciphertext_hex", "decrypted_text_hex" or "nonce".
 *
 * \param *temp_small
 * The pointer to the temporary array holding the decimal value of either
 * "ciphertext", "decrypted_text" or "input_block".
 *
 * \param d2h_size
 * The size of the array to be manipulated.
 *
 * \returns
 * N/A.
 */
static void _DecToHex( unsigned char *temp_large, unsigned char *temp_small,
                       unsigned int d2h_size )
{
    /*!
     * A temporary array that is needed due to the format of the "_DecToHex()"
     * function.
     */
    unsigned char d2h_temp[AES_BLOCK_SIZE_2];

    /*!
     * Used to access individual elements of the arrays used in the
     * "_DecToHex()" function.
     */
    unsigned int d2h_i = 0;
    unsigned int d2h_j = 0;

    /*!
     * For the least significant half of "d2h_temp" (elements [n/2] - [n-1]):
     * Each element of "d2h_temp" is made equal to the modulus of the
     * corresponding element of "temp_small".
     *
     * For the most significant half of "d2h_temp" (elements [0] - [(n/2)-1]):
     * Each element of "d2h_temp" is made equal to the modulus of the
     * corresponding element of "temp_small" minus the modulus of that element.
     *
     * The values of each element of "d2h_temp" are then checked and converted
     * to the corresponding hexadecimal equivalent. This is done in two parts
     * due to the setup of the for loop.
     *
     * temp_small   aaaaaaaa aaaaaaaa                       a = ascii value
     * d2h_temp     ssssssss ssssssss %%%%%%%% %%%%%%%%     s = subtracted value  % = modulus value
     * d2h_temp   h ssssssss ssssssss %%%%%%%% %%%%%%%%     h = hex value
     * temp_large   AAAAAAAA AAAAAAAA AAAAAAAA AAAAAAAA     A = added value where A = s[i] + %[i+16]
     */
    for ( d2h_i = 0; d2h_i < d2h_size; d2h_i++ )
    {
        d2h_temp[d2h_i + d2h_size] = temp_small[d2h_i] % HEX_BASE;
        d2h_temp[d2h_i] = ( temp_small[d2h_i] - d2h_temp[d2h_i + d2h_size] ) / HEX_BASE;

        d2h_temp[d2h_i] += ( d2h_temp[d2h_i] <= NINE ) ? ( '0' ) : ( '7' );
        d2h_temp[d2h_i + d2h_size] += ( d2h_temp[d2h_i + d2h_size] <= NINE ) ? ( '0' ) : ( '7' );
    }

    /*!
     * Sort the values stored in "d2h_temp" into the correct order.
     * This is done in two parts due to the setup of the previous for loop.
     */
    for ( d2h_i = 0, d2h_j = 0; d2h_i < d2h_size; d2h_i++, d2h_j += TWO )
    {
        temp_large[d2h_j] = d2h_temp[d2h_i];
    }

    for ( d2h_i = ( ( d2h_size * TWO ) - ONE ), d2h_j =
        ( ( d2h_size * TWO ) - ONE ); d2h_i > ( d2h_size - ONE ); d2h_i--, d2h_j -= TWO )
    {
        temp_large[d2h_j] = d2h_temp[d2h_i];
    }

    return;
}
/******************************************************************************/
/*!
 * \par Description
 * Normal operation:
 * Exits the program to ensure that no data is corrupted.
 *
 * Debug operation:
 * Prints an error message to the screen and then exits the program to ensure
 * that no data is corrupted. No data is printed during normal operation as it
 * is assumed the program will a part of a closed system.
 *
 * \param error
 * The error number denotes which error message to display.
 * Positive numbers indicate errors during the normal operation of the program.
 * Negative numbers indicate errors during the debug operation of the program.
 *
 * \returns
 * N/A.
 */
static void _DisplayError( signed int error )
{
    exit ( 1 );
}
/******************************************************************************/
/*!
 * \par Description
 * Read in the nonce and plaintext. Write out the nonce. Write out the
 * ciphertext if TEST is set to 1. Increment the local and persistant counters.
 * Generate the entropy for the "RAND_add()" and "RAND_seed()" functions.
 * Call functions to do the following:
 * - Convert the plaintext from hexadecimal to decimal.
 * - Initialise the encryption key.
 * - Run the encryption algorithm.
 * - Convert the nonce and ciphertext from decimal to hexadecimal.
 * - Seed the crypto library "RAND_add()" and "RAND_seed()" functions.
 *
 * \param *_ckey
 * The pointer to the array holding the key value.
 *
 * \param *entropy
 * The pointer to the array set up to hold the generated entropy.
 *
 * \param *input_block
 * The pointer to the array holding the "input_block" value.
 *
 * \param *temp_large
 * The pointer to the temporary array holding the hexadecimal value of either
 * "ciphertext_hex", "decrypted_text_hex" or "nonce".
 *
 * \param *temp_small
 * The pointer to the temporary array holding the decimal value of either
 * "ciphertext", "decrypted_text" or "input_block".
 *
 * \param NUM_BYTES_ENTROPY
 * The size of the "entropy" array - required by the "RAND_add()" and
 * "RAND_seed()" functions.
 *
 * \returns
 * N/A.
 */
static void _Encrypt( const unsigned char *_ckey, unsigned char *_entropy,
                      unsigned char *input_block, unsigned char *temp_large,
                      unsigned char *temp_small,
                      unsigned int NUM_BYTES_ENTROPY )
{
    FILE *fp_nc;
    FILE *fp_pt;

    /*!
     * Declare arrays of set lengths.
     * AES_BLOCK_SIZE:   16 (bytes): defined in "aes.h".
     * AES_BLOCK_SIZE_2: 32 (bytes): defined above.
     * NONCE_BLOCK_SIZE: 10 (bytes): defined above.
     * "ecount" and "ivec" arrays are needed by the "AES_ctr128_encrypt()"
     * function.
     */
    unsigned char ecount        [AES_BLOCK_SIZE];
    unsigned char ivec          [AES_BLOCK_SIZE];
    unsigned char ciphertext    [AES_BLOCK_SIZE];
    unsigned char plaintext     [AES_BLOCK_SIZE];
    unsigned char ciphertext_hex[AES_BLOCK_SIZE_2];
    unsigned char nonce         [NONCE_BLOCK_SIZE];

    /*!
     * Declare variables.
     * ent_i            : Used to access individual elements of the "entropy"
     *                    array.
     * entropy_ctr      : Keeps track of the current plaintext block.
     * entropy_limit    : The number of loops required to generate the specified
     *                    entropy.
     * num              : Required to be zero by the AES_ctr128_encrypt
     *                    function.
     * nc_bytes_written : Number of bytes written to the nonce file.
     * pt_bytes_read    : Number of bytes read from the plaintext file.
     * ct_bytes_written : Number of bytes written to the ciphertext file.
     */
    unsigned int ent_i            = 0;
    unsigned int entropy_ctr      = 0;
    unsigned int entropy_limit    = 0;
    unsigned int num              = 0;
    unsigned int nc_bytes_written = 0;
    unsigned int pt_bytes_read    = 0;

    fp_pt = fopen( "plaintext.txt", "rb" );

    if ( fp_pt == NULL )
    {
        _DisplayError( TWO );
    }

    /*!
     * Set all elements of "ecount" to zero as required by the
     * "AES_ctr128_encrypt()" function.
     * Copy the data in "input_block" into "ivec". This is required as the
     * "ivec" value will be changed during the encryption process, therefore the
     * "input_block" value cannot be used as it must be preserved until it is
     * changed by the local or persistant counters.
     */
    memset( ecount, 0, AES_BLOCK_SIZE );
    memcpy( ivec, input_block, AES_BLOCK_SIZE );

    /*!
     * Initialise the encryption KEY.
     * \par Format of the function:
     *
     * \verbatim

     int AES_set_encrypt_key( const unsigned char *userKey,
                              const int bits,
                              AES_KEY *key );

     \endverbatim
     */
    AES_set_encrypt_key( _ckey, AES_KEY_SIZE, &key );

    /*!
     * Calculate the "entropy_limit" based on the values of "NUM_BYTES_ENTROPY"
     * and "AES_BLOCK_SIZE". This makes sure the correct number of ciphertext
     * elements are copied to the entropy array.
     */
    if ( NUM_BYTES_ENTROPY % AES_BLOCK_SIZE == 0 )
    {
        entropy_limit = NUM_BYTES_ENTROPY / AES_BLOCK_SIZE;
    }
    else
    {
        entropy_limit = ( NUM_BYTES_ENTROPY / AES_BLOCK_SIZE ) + ONE;
    }

    for ( entropy_ctr = 0; entropy_ctr < entropy_limit; entropy_ctr++ )
    {
        /*!
         * Read in one block (16 bytes) from the plaintext file. If the number
         * of bytes read is equal to zero, the last block of plaintext has been
         * encrypted, therefore exit the while loop. This also guards against a
         * plaintext file that contains nothing and a plaintext file that
         * contains an exact multiple of AES_BLOCK_SIZE_2 bytes.
         */
        pt_bytes_read = fread( temp_large, 1, AES_BLOCK_SIZE_2, fp_pt );

        if ( pt_bytes_read == 0 )
        {
            break;
        }

        /*!
         * Convert the plaintext from base 16 ("temp_large") to base 256
         * ("temp_small").
         * Copy the "temp_small" array to the "plaintext" array.
         */
        _HexToDec( temp_large, temp_small, AES_BLOCK_SIZE_2 );
        memcpy( plaintext, temp_small, AES_BLOCK_SIZE );

        /*!
         * Call the encryption function.
         * \par Format of the function:
         *
         * \verbatim

         void AES_ctr128_encrypt( const unsigned char *in,
                                  unsigned char *out,
                                  size_t length,
                                  const AES_KEY *key,
                                  unsigned char ivec[AES_BLOCK_SIZE],
                                  unsigned char ecount_buf[AES_BLOCK_SIZE],
                                  unsigned int *num );

         \endverbatim
         */
        AES_ctr128_encrypt( plaintext, ciphertext, ( pt_bytes_read / TWO ),
                            &key, ivec, ecount, &num );

        /*!
         * Copy the "ciphertext" array to the "temp_small" array.
         * Convert the ciphertext from base 256 ("temp_small") to base 16
         * ("temp_large").
         * Copy the "temp_large" array to the "ciphertext_hex" array.
         */
        memcpy( temp_small, ciphertext, AES_BLOCK_SIZE );
        _DecToHex( temp_large, temp_small, AES_BLOCK_SIZE );
        memcpy( ciphertext_hex, temp_large, AES_BLOCK_SIZE_2 );

        /*!
         * Copy the number of user specified "ciphertext" elements to produce
         * the generated entropy.
         */
        for ( ent_i = 0; ent_i < AES_BLOCK_SIZE; ent_i++)
        {
            _entropy[ent_i + ( entropy_ctr * AES_BLOCK_SIZE )] = ciphertext[ent_i];
        }

        /*!
         * If the number of bytes read is less than the AES_BLOCK_SIZE_2, the
         * last block of plaintext has been encrypted, therefore exit the while
         * loop.
         */
        if ( pt_bytes_read < AES_BLOCK_SIZE_2 )
        {
            break;
        }

        /*!
         * Increment the local counter.
         */
        _CtrInc( input_block, 'L' );
    }

    /*!
     * Increment the persistant counter.
     * Copy the "input_block" array to the "temp_small" array.
     * Convert the nonce from base 256 ("temp_small") to base 16 ("temp_large").
     * Copy the "temp_large" array to the "nonce" array.
     */
    _CtrInc( input_block, 'P' );
    memcpy( temp_small, input_block, NONCE_BLOCK_SIZE );
    _DecToHex( temp_large, temp_small, NONCE_BLOCK_SIZE );
    memcpy( nonce, temp_large, NONCE_BLOCK_SIZE_2 );

    fp_nc = fopen( "nonce.txt", "wb" );

    if ( fp_nc == NULL )
    {
        _DisplayError( THREE );
    }

    nc_bytes_written = fwrite( nonce, 1, NONCE_BLOCK_SIZE_2, fp_nc );
    fclose( fp_nc );

    RAND_add ( _entropy, NUM_BYTES_ENTROPY, 0.0 );
    RAND_seed( _entropy, NUM_BYTES_ENTROPY );

    return;
}
/******************************************************************************/
/*!
 * \par Description
 * Convert the hexadecimal value of either "ciphertext", "nonce" or "plaintext"
 * to decimal format.
 *
 * \param *temp_large
 * The pointer to the temporary array holding the decimal value of either
 * "ciphertext", "nonce" or "plaintext".
 *
 * \param *temp_small
 * The pointer to the temporary array holding the hexadecimal value of either
 * "ciphertext", "nonce" or "plaintext".
 *
 * \param h2d_size
 * The size of the array to be manipulated.
 *
 * \returns
 * N/A.
 */
static void _HexToDec( unsigned char *temp_large, unsigned char *temp_small,
                       unsigned int h2d_size )
{
    /*!
     * Used to access individual elements of the arrays used in the
     * "_HexToDec()" function.
     */
    unsigned int h2d_i = 0;

    /*!
     * Convert each element to uppercase if it contains a letter.
     * Convert each hex value to its corresponding ascii value by subtracting
     * the appropriate ascii number.
     * Multiply each even element by 16.
     *
     * temp_large   hhhhhhhh hhhhhhhh hhhhhhhh hhhhhhhh     h = hex value
     * temp_large   aaaaaaaa aaaaaaaa aaaaaaaa aaaaaaaa     a = ascii value
     * temp_large   xaxaxaxa xaxaxaxa xaxaxaxa xaxaxaxa     x = multiplied value
     * temp_small   AAAAAAAA AAAAAAAA                       A = final added value where
     *                                                      A = x[i] + a[i+1]
     */
    for ( h2d_i = 0; h2d_i < h2d_size; h2d_i++ )
    {
        temp_large[h2d_i] = toupper( temp_large[h2d_i] );
        temp_large[h2d_i] -=
            ( temp_large[h2d_i] >= '0' && temp_large[h2d_i] <= '9' ) ? ( '0' ) : ( '7' );
        temp_large[h2d_i] =
            ( h2d_i % TWO == 0 ) ? ( temp_large[h2d_i] * HEX_BASE ) : ( temp_large[h2d_i] );
    }

    /*!
     * Add each even element to the next odd element in the array to produce the
     * actual ascii number of each hexadecimal byte.
     */
    for ( h2d_i = 0; h2d_i < ( h2d_size / TWO ); h2d_i ++ )
    {
        temp_small[h2d_i] = temp_large[h2d_i * TWO] + temp_large[( h2d_i * TWO ) + ONE];
    }

    return;
}
/******************************************************************************/
/*!
 * \par Description
 * Initialise the "input_block" by adding the "nonce" to it.
 *
 * \param *input_block
 * The pointer to the array holding the "input_block" value.
 *
 * \param *temp_large
 * The pointer to the temporary array holding the hexadecimal value of either
 * "ciphertext_hex", "decrypted_text_hex" or "nonce".
 *
 * \param *temp_small
 * The pointer to the temporary array holding the decimal value of either
 * "ciphertext", "decrypted_text" or "input_block".
 *
 * \returns
 * N/A.
 */
static void _InBlkInit( unsigned char *input_block, unsigned char *temp_large,
                        unsigned char *temp_small )
{
    unsigned char nc_bytes_read = 0;

    /*!
     * Read data into "temp_large" from "nonce.txt" using binary read.
     * Close the file after reading to prevent file corruption if something goes
     * wrong as the following while loop may take a long time to execute
     * depending on the size of the plaintext file.
     */
    FILE *fp_nc;
    fp_nc = fopen( "nonce.txt", "rb" );

    if ( fp_nc == NULL )
    {
        _DisplayError( ONE );
    }

    nc_bytes_read = fread( temp_large, 1, NONCE_BLOCK_SIZE_2, fp_nc );
    fclose( fp_nc );

    /*!
     * Convert the nonce from base 16 ("temp_large") to base 256 ("temp_small").
     * Set all elements of "input_block" to zero.
     * Copy the "temp_large" array (nonce) to the corresponding "input_block"
     * elements.
     */
    _HexToDec( temp_large, temp_small, NONCE_BLOCK_SIZE_2 );
    memset( input_block, 0, AES_BLOCK_SIZE );
    memcpy( input_block, temp_small, NONCE_BLOCK_SIZE );

    return;
}
/******************************************************************************/
/*!
 * \par Description
 * Implements an integer only equivalent of the floating point "pow()" function
 * for base 2.
 *
 * \param exponent
 * The power, where result = 2 ^ "exponent" (or, result = pow(2, "exponent") in
 * equivalent form).
 *
 * \returns
 * The "tracker_weight" for the given "input_block" element.
 */
static unsigned int _Weight( unsigned int exponent )
{
    unsigned int power_ctr = 0;
    unsigned int tracker_weight = ONE;

    if ( exponent == 0 )
    {
        return tracker_weight;
    }
    else
    {
        for (power_ctr = 0; power_ctr < exponent; power_ctr++)
        {
            tracker_weight *= TWO;
        }
    }

    return tracker_weight;
}
/******************************************************************************/
