//==========================================================================
//
//        timer2.cxx
//
//        POSIX timer test 2
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2007 Free Software Foundation, Inc.
// Copyright (C) 2007 eCosCentric Limited                                   
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):     nickg
// Contributors:  nickg
// Date:          2007-05-22
// Description:   Tests POSIX timer/clock functionality. In particular this
//                tests what happens when clock_settime() is used to change
//                CLOCK_REALTIME and checks that pending alarms behave correctly.
//
//####DESCRIPTIONEND####
//==========================================================================

#include <cyg/infra/testcase.h>
#include <pkgconf/posix.h>

#ifndef CYGPKG_POSIX_SIGNALS
#define NA_MSG "No POSIX signals"
#elif !defined(CYGPKG_POSIX_TIMERS)
#define NA_MSG "No POSIX timers"
#elif !defined(CYGPKG_POSIX_CLOCKS_USE_WALLCLOCK)
#define NA_MSG "POSIX clock wallclock access not enabled"
#endif

#ifdef NA_MSG
void
cyg_start(void)
{
    CYG_TEST_INIT();
    CYG_TEST_NA(NA_MSG);
}
#else

#include <sys/types.h>
#include <pthread.h>
#include <signal.h>
#include <semaphore.h>
#include <time.h>
#include <errno.h>

//--------------------------------------------------------------------------
// Local variables

timer_t timer1;
timer_t timer2;

volatile int sigusr1_called = 0;
volatile int sigusr2_called = 0;

volatile int fails = 0;

struct timespec timer2_trigger;

// Swap the following defines over for some debug output.
//#define db_printf diag_printf
#define db_printf(__fmt, ... ) CYG_EMPTY_STATEMENT

//--------------------------------------------------------------------------
// Signal handler functions

static void sigusr1( int signo, siginfo_t *info, void *context )
{
    CYG_TEST_INFO( "sigusr1() handler called" );
    CYG_TEST_CHECK( signo == SIGUSR1, "Signal not SIGUSR1");
    CYG_TEST_CHECK( signo == info->si_signo, "Bad signal number in siginfo" );
    CYG_TEST_CHECK( info->si_code == SI_TIMER, "Siginfo code not SI_TIMER" );
    CYG_TEST_CHECK( info->si_value.sival_int == 0xABCDEF01, "Siginfo value wrong");
//    CYG_TEST_CHECK( pthread_equal(pthread_self(), thread1), "Not called in thread1");

    sigusr1_called++;
}

static void sigusr2( int signo, siginfo_t *info, void *context )
{
    struct timespec ts;
    
    CYG_TEST_INFO( "sigusr2() handler called" );
    CYG_TEST_CHECK( signo == SIGUSR2, "Signal not SIGUSR2");
    CYG_TEST_CHECK( signo == info->si_signo, "Bad signal number in siginfo" );
    CYG_TEST_CHECK( info->si_code == SI_TIMER, "Siginfo code not SI_TIMER" );
    CYG_TEST_CHECK( info->si_value.sival_int == 0xABCDEF02, "Siginfo value wrong");
//    CYG_TEST_CHECK( pthread_equal(pthread_self(), thread2), "Not called in thread2");

    clock_gettime( CLOCK_REALTIME, &ts );
    db_printf("ts: %d %d\n", ts.tv_sec, ts.tv_nsec );
    db_printf("timer2_trigger: %d %d\n", timer2_trigger.tv_sec, timer2_trigger.tv_nsec );
        
    if( timer2_trigger.tv_sec != ts.tv_sec ||
        (timer2_trigger.tv_sec == ts.tv_sec &&
         timer2_trigger.tv_nsec > ts.tv_nsec) )
    {
        CYG_TEST_INFO("SIGUSR2 event before timeout");
        fails++;
    }

    timer2_trigger.tv_sec += 4;
    
    sigusr2_called++;
}

//--------------------------------------------------------------------------

int main(int argc, char **argv)
{
    struct timespec tp0;
    int ret;
    sigset_t mask;
    
    CYG_TEST_INIT();

    // Make a full signal set
    sigfillset( &mask );

    
    // Install signal handlers
    {
        struct sigaction sa;

        sa.sa_sigaction = sigusr1;
        sa.sa_mask = mask;
        sa.sa_flags = SA_SIGINFO;

        ret = sigaction( SIGUSR1, &sa, NULL );

        CYG_TEST_CHECK( ret == 0 , "sigaction returned error");
    }
    {
        struct sigaction sa;

        sa.sa_sigaction = sigusr2;
        sa.sa_mask = mask;
        sa.sa_flags = SA_SIGINFO;

        ret = sigaction( SIGUSR2, &sa, NULL );

        CYG_TEST_CHECK( ret == 0 , "sigaction returned error");
    }


    // First, check that clock_settime() functions as expected
    {
        struct timespec tp1, tp2, ns;
        int res;

        CYG_TEST_INFO("Set CLOCK_REALTIME to 1 Billion seconds");
        
        // Get current time for information
        res = clock_gettime( CLOCK_REALTIME, &tp0 );
        CYG_TEST_CHECK( res == 0, "clock_gettime returned error" );
        db_printf("tp0: %d %d\n", tp0.tv_sec, tp0.tv_nsec );

        tp1.tv_sec = 1000000000;
        tp1.tv_nsec = 0;
        res = clock_settime( CLOCK_REALTIME, &tp1 );
        CYG_TEST_CHECK( res == 0, "clock_settime returned error" );
        
        res = clock_gettime( CLOCK_REALTIME, &tp2 );
        CYG_TEST_CHECK( res == 0, "clock_gettime returned error" );
        db_printf("tp2: %d %d\n", tp2.tv_sec, tp2.tv_nsec );

        CYG_TEST_INFO("Wait 7 seconds and check CLOCK_REALTIME agrees");
        
        ns.tv_sec = 7;
        ns.tv_nsec = 0;
        nanosleep( &ns, NULL );

        res = clock_gettime( CLOCK_REALTIME, &tp2 );
        CYG_TEST_CHECK( res == 0, "clock_gettime returned error" );
        db_printf("tp: %d %d\n", tp2.tv_sec, tp2.tv_nsec );

        CYG_TEST_CHECK( tp2.tv_sec-tp1.tv_sec == ns.tv_sec, "clock increment failed" );
        if ( tp2.tv_sec-tp1.tv_sec != ns.tv_sec )
            fails++;

        CYG_TEST_INFO("Set CLOCK_REALTIME back to original time (+7s)");

        // Set the wallclock back to the original setting, plus the
        // time that has elapsed since.
        tp0.tv_sec += tp2.tv_sec-tp1.tv_sec;
        tp0.tv_nsec += tp2.tv_nsec-tp1.tv_nsec;
        if( tp0.tv_nsec > 1000000000 )
            tp0.tv_nsec -= 1000000000, tp0.tv_sec++;
        db_printf("tp0: %d %d\n", tp0.tv_sec, tp0.tv_nsec );
        
        res = clock_settime( CLOCK_REALTIME, &tp0 );
        CYG_TEST_CHECK( res == 0, "clock_settime returned error" );

        res = clock_gettime( CLOCK_REALTIME, &tp2 );
        CYG_TEST_CHECK( res == 0, "clock_gettime returned error" );
        db_printf("tp: %d %d\n", tp2.tv_sec, tp2.tv_nsec );
    }
    

    // Create an absolute timer set to expire in 10s.
    {
        struct sigevent sev;
        struct itimerspec value;

        CYG_TEST_INFO("Create absolute timer to expire in 10s");
        
        sev.sigev_notify                = SIGEV_SIGNAL;
        sev.sigev_signo                 = SIGUSR1;
        sev.sigev_value.sival_int       = 0xABCDEF01;

        ret = clock_gettime( CLOCK_REALTIME, &value.it_value );
        value.it_value.tv_sec           += 10;
        value.it_interval.tv_sec        = 0;
        value.it_interval.tv_nsec       = 0;
        
        ret = timer_create( CLOCK_REALTIME, &sev, &timer1 );
        CYG_TEST_CHECK( ret == 0 , "timer_create returned error");

        ret = timer_settime( timer1, TIMER_ABSTIME, &value, NULL );
        CYG_TEST_CHECK( ret == 0 , "timer_settime returned error");
    }

    // Create an absolute initial value timer set to expire in 25s,
    // with a 4s retrigger interval.
    {
        struct sigevent sev;
        struct itimerspec value;
        
        sev.sigev_notify                = SIGEV_SIGNAL;
        sev.sigev_signo                 = SIGUSR2;
        sev.sigev_value.sival_int       = 0xABCDEF02;

        CYG_TEST_INFO("Create absolute timer to expire in 25s and every 4s thereafter");
        
        ret = clock_gettime( CLOCK_REALTIME, &value.it_value );
        value.it_value.tv_sec           += 25;
        timer2_trigger = value.it_value;
        value.it_interval.tv_sec        = 4;
        value.it_interval.tv_nsec       = 0;
        
        ret = timer_create( CLOCK_REALTIME, &sev, &timer2 );
        CYG_TEST_CHECK( ret == 0 , "timer_create returned error");

        ret = timer_settime( timer2, TIMER_ABSTIME, &value, NULL );
        CYG_TEST_CHECK( ret == 0 , "timer_settime returned error");
    }

    
    // Now change clock to a time after the first timer but before the
    // second and check that the first timer triggers as a result and
    // that the second doesn't.
    {
        struct timespec tp;
        int res;

        CYG_TEST_INFO("Move CLOCK_REALTIME on 20s, past first timer");
        CYG_TEST_INFO("The first timer should expire immediately, the second should not");
        
        res = clock_gettime( CLOCK_REALTIME, &tp );
        CYG_TEST_CHECK( res == 0, "clock_gettime returned error" );
        db_printf("tp: %d %d\n", tp.tv_sec, tp.tv_nsec );

        tp.tv_sec += 20;
        res = clock_settime( CLOCK_REALTIME, &tp );
        CYG_TEST_CHECK( res == 0, "clock_settime returned error" );

        CYG_TEST_CHECK( sigusr1_called > 0, "SIGUSR1 handler not called" );
        CYG_TEST_CHECK( sigusr2_called == 0, "SIGUSR2 handler called" );
        if( sigusr1_called == 0 )
            fails++;
        if( sigusr2_called != 0 )
            fails++;
    }    

    // Wait for second timer to expire, which should be in 5 seconds
    // now, 20s earlier than when it was set.
    {
        struct timespec ns;

        CYG_TEST_INFO("Wait for second timer to expire at new absolute time, in 5s");
        
        ns.tv_sec = 6;
        ns.tv_nsec = 0;
        nanosleep( &ns, NULL );

        CYG_TEST_CHECK( sigusr2_called > 0, "SIGUSR2 handler not called" );        
    }

    // Check that timer2 intervals are still correct
    {
        struct timespec ns;

        CYG_TEST_INFO("Wait for second timer to expire 5 more times on 4s interval");
        
        ns.tv_sec = 21;
        ns.tv_nsec = 0;
        while( nanosleep( &ns, &ns ) != 0 && errno == EINTR )
            continue;

        CYG_TEST_CHECK( sigusr2_called == 6, "SIGUSR2 handler not called 5 more times" );        
    }

    
    // Move CLOCK_REALTIME back 20s to restore it to where it started from
    {
        struct timespec tp;
        int res;

        CYG_TEST_INFO("Move CLOCK_REALTIME back 20s to original real time");
        
        res = clock_gettime( CLOCK_REALTIME, &tp );
        CYG_TEST_CHECK( res == 0, "clock_gettime returned error" );
        db_printf("tp: %d %d\n", tp.tv_sec, tp.tv_nsec );

        tp.tv_sec -= 20;
        res = clock_settime( CLOCK_REALTIME, &tp );
        CYG_TEST_CHECK( res == 0, "clock_settime returned error" );
    }    
    
    if( fails > 0 )
        CYG_TEST_FAIL_FINISH( "timer2" );
    else
        CYG_TEST_PASS_FINISH( "timer2" );
}

#endif

//--------------------------------------------------------------------------
// end of timer2.c
