//==========================================================================
//
//      if_lpc2xxx-lwip.c
//
//      lwIP-specific Ethernet driver for NXP LPC2XXX family on-chip EMAC.
//
//==========================================================================
//####ECOSGPLCOPYRIGHTBEGIN####
// -------------------------------------------
// This file is part of eCos, the Embedded Configurable Operating System.
// Copyright (C) 2003, 2004, 2006, 2007 Free Software Foundation, Inc.      
// Copyright (C) 2003, 2004, 2006, 2007, 2009 eCosCentric Limited                 
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------
//####ECOSGPLCOPYRIGHTEND####
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    jlarmour
// Contributors:
// Date:         2007-08-13
// Description:  hardware driver for LPC2XXX ethernet devices specifically
//               and solely for lwIP TCP/IP stack.
//               Derived directly from if_at91-lwip.c, with some LPC2XXX portions
//               imported from if_lpc2xxx.c and then much munged.
//
//               At the moment, this driver has only been tested with the LPC2387.
//
//####DESCRIPTIONEND####
//==========================================================================

#include <pkgconf/system.h>
#include <pkgconf/hal.h>
#include <pkgconf/devs_eth_arm_lpc2xxx.h>
#include <pkgconf/io_eth_drivers.h>

#include <cyg/infra/cyg_type.h>
#include <cyg/infra/cyg_ass.h>
#include <cyg/hal/hal_arch.h>
#include <cyg/hal/hal_intr.h>
#include <cyg/hal/hal_cache.h>
#include <cyg/hal/hal_if.h>
#include <cyg/infra/diag.h>
#include <cyg/hal/drv_api.h>
#include <cyg/io/eth/netdev.h>
#include <cyg/io/eth/eth_drv.h>
#include <string.h>
#include <cyg/io/eth_phy.h>

#include <pkgconf/net_lwip.h>
#include "lwip/opt.h"
#include "lwip/err.h"
#include "lwip/netif.h"
#include "lwip/pbuf.h"

//==========================================================================

#ifdef CYGACC_CALL_IF_DELAY_US
# define LPC2XXX_ETH_DELAY_US(us) CYGACC_CALL_IF_DELAY_US(us)
#else
# include <cyg/hal/hal_diag.h>
# define LPC2XXX_ETH_DELAY_US(us) HAL_DELAY_US(us)
#endif

#ifndef CYGARC_PHYSICAL_ADDRESS
#define CYGARC_PHYSICAL_ADDRESS(__x) (__x)
#endif

#ifndef CYGARC_UNCACHED_ADDRESS
#define CYGARC_UNCACHED_ADDRESS(__x) (__x)
#endif

#define LPC2XXX_ETH_RAM 0x7FE00000

#ifndef LPC2XXX_EMAC_TX_BUFF_SIZE
#define LPC2XXX_EMAC_TX_BUFF_SIZE  0x600
#endif

//==========================================================================
// Debug support, in the form of diagnostics and a simple trace
// buffer. Also make it easy to enable stats without messing about
// with the configuration.

#define DIAG_LEVEL     0
#define DIAG_BUFFERED  0
#define DIAG_WRAP      1
#define DIAG_PHY       0

/* Here's two useful GDB command fragments. The first dumps
 * the rxbds. Change 32 to the RXBD count:
set $foo=0
while ($foo < 32)
p/x lpc2xxx_eth_info.rxbds[$foo]
set $foo=$foo+1
end

* This one dumps the trace buffer. Change 300 to the
* diag buffer size:
set $foo=lpc2xxx_eth_trace_next
while ($foo >= 0)
p lpc2xxx_eth_trace_data[$foo]
set $foo=$foo-1
end
set $foo=300-1
while ($foo > lpc2xxx_eth_trace_next)
p lpc2xxx_eth_trace_data[$foo]
set $foo=$foo-1
end
 */


#if (DIAG_LEVEL <= 0)

# define DIAG(_level_, _fmt_, ...)                  \
    CYG_MACRO_START                                 \
    CYG_MACRO_END

# define DIAGPKT(_level_, _msg_, _pkt_, _len_)      \
    CYG_MACRO_START                                 \
    CYG_MACRO_END

#define DIAGDUMP(_level_, _msg_, _addr_, _size_ )       \
    CYG_MACRO_START                                     \
    CYG_MACRO_END

#define DIAGDUMP32(_level_, _msg_, _addr_, _size_ )     \
    CYG_MACRO_START                                     \
    CYG_MACRO_END

#elif (! DIAG_BUFFERED)

#  define DIAG(_level_, _fmt_, ...)                                     \
    CYG_MACRO_START                                                     \
    if ((_level_) <= DIAG_LEVEL) {                                      \
        cyg_uint32 __istate;                                            \
        HAL_DISABLE_INTERRUPTS(__istate);                               \
        diag_printf("ETH %30s[%3d] : " _fmt_ "\n", __func__, __LINE__, ## __VA_ARGS__); \
        HAL_RESTORE_INTERRUPTS(__istate);                               \
    }                                                                   \
    CYG_MACRO_END

# define DIAGPKT(_level_, _msg_, _pkt_, _len_)                          \
    CYG_MACRO_START                                                     \
    if ((_level_) <= DIAG_LEVEL) {                                      \
        cyg_uint32 __istate;                                            \
        HAL_DISABLE_INTERRUPTS(__istate);                               \
        diag_printf("ETH %30s[%3d] : %s : pkt %p, len %d\n", __func__, __LINE__, _msg_, \
                    (cyg_uint8*)_pkt_, _len_);                          \
        HAL_RESTORE_INTERRUPTS(__istate);                               \
    }                                                                   \
    CYG_MACRO_END

#define DIAGDUMP(_level_, _msg_, _addr_, _size_ )                       \
    CYG_MACRO_START                                                     \
    if ((_level_) <= DIAG_LEVEL) {                                      \
        cyg_uint32 __istate;                                            \
        HAL_DISABLE_INTERRUPTS(__istate);                               \
        diag_printf("ETH %30s[%3d] : %s:\n", __func__, __LINE__, _msg_ ); \
        diag_dump_buf( _addr_, _size_ );                                \
        HAL_RESTORE_INTERRUPTS(__istate);                               \
    }                                                                   \
    CYG_MACRO_END
    
#define DIAGDUMP32(_level_, _msg_, _addr_, _size_ )                     \
    CYG_MACRO_START                                                     \
    if ((_level_) <= DIAG_LEVEL) {                                      \
        cyg_uint32 __istate;                                            \
        HAL_DISABLE_INTERRUPTS(__istate);                               \
        diag_printf("ETH %30s[%3d] : %s:\n", __func__, __LINE__, _msg_ ); \
        diag_dump_buf_32bit( _addr_, _size_ );                          \
        HAL_RESTORE_INTERRUPTS(__istate);                               \
    }                                                                   \
    CYG_MACRO_END


#else
// Trace buffer size.
# define DIAG_BUFFER_SIZE      150
typedef struct lpc2xxx_eth_trace {
    const char*     fn;
    char            msg[100];
    cyg_uint32      len;
    cyg_uint8       packet[14];
} lpc2xxx_eth_trace_entry;
static lpc2xxx_eth_trace_entry     lpc2xxx_eth_trace_data[DIAG_BUFFER_SIZE];
static int                      lpc2xxx_eth_trace_next      = DIAG_BUFFER_SIZE - 1;
static cyg_bool                 lpc2xxx_eth_trace_wrapped   = false;

static void
lpc2xxx_eth_trace(const char* fn, const char* fmt, cyg_uint8* packet, cyg_uint32 len, ...)
{
    lpc2xxx_eth_trace_entry* entry = &(lpc2xxx_eth_trace_data[lpc2xxx_eth_trace_next]);
    va_list ap;

# ifdef DIAG_WRAP
    if (0 == lpc2xxx_eth_trace_next) {
        lpc2xxx_eth_trace_next = DIAG_BUFFER_SIZE - 1;
        lpc2xxx_eth_trace_wrapped = true;
    } else {
        lpc2xxx_eth_trace_next -= 1;
    }
# else
    if (lpc2xxx_eth_trace_next < 0) {
        return;
    }
    lpc2xxx_eth_trace_next -= 1;
# endif

    entry->fn   = fn;
    va_start(ap, len);
    memset(entry->msg, 0, sizeof(entry->msg));
    diag_vsnprintf(entry->msg, sizeof(entry->msg), fmt, ap);
    va_end(ap);
    entry->len  = len;
    if ((cyg_uint8*)0 == packet) {
        memset(entry->packet, 0, 14);
    } else {
        memcpy(entry->packet, packet, 14);
    }
}

#  define DIAG(_level_, _fmt_, ...)                                         \
    CYG_MACRO_START                                                         \
    if ((_level_) <= DIAG_LEVEL) {                                          \
        lpc2xxx_eth_trace(__func__, _fmt_, (cyg_uint8*)0, 0, ## __VA_ARGS__); \
    }                                                                       \
    CYG_MACRO_END

# define DIAGPKT(_level_, _msg_, _pkt_, _len_)                              \
    CYG_MACRO_START                                                         \
    if ((_level_) <= DIAG_LEVEL) {                                          \
        lpc2xxx_eth_trace(__func__, _msg_, (cyg_uint8*)_pkt_, _len_);       \
    }                                                                       \
    CYG_MACRO_END

#endif

#if (DIAG_LEVEL < 3)
# define WRITE32(_addr_, _val_)         HAL_WRITE_UINT32(_addr_, _val_)
# define READ32( _addr_, _var_)         HAL_READ_UINT32(_addr_, _var_)
#else

# define WRITE32(_addr_, _val_)                                             \
    CYG_MACRO_START                                                         \
    DIAG(DIAG_LEVEL, "WRITE %s %08x <= 0x%08x", # _addr_, _addr_, _val_) ;  \
    HAL_WRITE_UINT32(_addr_, _val_);                                        \
    CYG_MACRO_END

#define READ32(_addr_, _var_)                                               \
    CYG_MACRO_START                                                         \
    HAL_READ_UINT32(_addr_, _var_);                                         \
    DIAG(DIAG_LEVEL, "READ  %s %08x == 0x%08x", # _addr_, _addr_, _var_) ;  \
    CYG_MACRO_END

#endif

#if 0
// A function that gets placed in RAM. This can be called by flash-resident
// code, e.g. RedBoot, when I need a breakpoint on a specific condition.
static void lpc2xxx_eth_ramfn(void) __attribute__((section (".2ram.lpc2xxx_eth_ramfn")));
static int  lpc2xxx_eth_ramfn_calls;
static void
lpc2xxx_eth_ramfn(void)
{
    lpc2xxx_eth_ramfn_calls += 1;
}
#endif

//======================================================================

#if DIAG_PHY
// Include PHY stuff after debug macro defs.
# define phy_debug_printf( __fmt, ... ) DIAG(2, "LPC2XXX ETH: %30s[%4d]: " __fmt, __FUNCTION__, __LINE__, ## __VA_ARGS__ )
#else
# define phy_debug_printf( __fmt, ... )
#endif


#include "lpc2xxx_phy.h"

// ----------------------------------------------------------------------------
// The driver currently support a configurable number of rx buffers. The
// size is fixed by the hardware (so far, seen only at 128 bytes). Incoming
// frames are passed into lwIP as pbuf chains.
//
// For tx the current code supports a single outgoing transmission at a
// time. This does limit outgoing bandwidth a bit, but only a bit, and
// saves memory and complexity.

#if defined(CYGPKG_HAL_ARM_ARM9)  // Maybe it works. It's certainly not yet tested.
# error ARM9 support is work in progress
#endif

#if defined(HAL_DCACHE_LINE_SIZE) && (HAL_DCACHE_LINE_SIZE > 4)
# define BD_ALIGNMENT (HAL_DCACHE_LINE_SIZE)
#else
# define BD_ALIGNMENT (4)
#endif

// Receive Buffer Descriptor
typedef struct rbd_s
{
   cyg_uint32 addr;
   cyg_uint32 ctrl;
} rbd_t;

// Receive Status Descriptor
typedef struct rsd_s
{
   cyg_uint32 info;
   cyg_uint32 hash_crc;
} rsd_t;


// Transmit Buffer Descriptor
typedef struct tbd_s
{
   cyg_uint32 addr;
   cyg_uint32 ctrl;
} tbd_t;

// Transmit Status Descriptor
typedef struct tsd_s
{
   cyg_uint32 info;
} tsd_t;

#define RXBD_BUFFER_ADDR(rxbd) ((cyg_uint8*)((rxbd).addr))

#define TX_BUFFER_DESCRIPTOR_COUNT CYGNUM_DEVS_ETH_ARM_LPC2XXX_TX_BUFS

//==========================================================================
// LPC2XXX Ethernet private data
//
// All LPC2XXXs seen so far have a single on-chip ethernet device, so all
// device-specific data could be held in statics. However, in case this
// driver ever gets re-used for a chip with multiple ethernet devices that
// data is accessed via macros which can choose to use statics or access
// the internal data.

typedef struct lpc2xxx_eth
{
    // Interrupt vector details.
    cyg_handle_t    interrupt_handle;
    cyg_interrupt   interrupt_data;

    // Holds pbuf that needs freeing after send.
    struct pbuf *   tx_pbuf;

    // The buffer descriptors. These will be allocated in the on-chip
    // ethernet RAM.
    volatile tbd_t *txbds;
    volatile tsd_t *tsbds;
    volatile rbd_t *rxbds;
    volatile rsd_t *rsbds;

    cyg_uint8 *txbuf;           // Transmit buffer.
    
    struct netif    netif;
    
    volatile cyg_uint8 tx_index;

    /* Maybe in future we could consider consolidating the booleans below
     * into a single flags bitmask. However, firstly there are potential
     * problems with atomic set/get to be considered, and secondly at present
     * there are only 3 of them - less than one word, so no real gain.
     */
    cyg_uint8          started;
    volatile cyg_uint8 tx_can_send;

    // A flag to indicate whether we ran out of buffers, and need to try and refill on pbuf free.
//    cyg_uint8          rx_ran_out_of_buffers;

    // rx_next_buffer tracks the next rx buffer descriptor which the
    // hardware may process.
    cyg_uint8          rx_next_buffer;

    cyg_uint8          mac[6];

} lpc2xxx_eth;

//==========================================================================
// The following macros take an lpc2xxx_eth structure as an argument, for
// a degree of future-proofing should there ever be multiple EMACs.

#define LPC2XXX_ETH_INTR_VECTOR(eth)    (CYGNUM_HAL_INTERRUPT_ETHERNET)
#define LPC2XXX_ETH_BASE(eth)           (CYGARC_HAL_LPC2XXX_REG_ETH_BASE)
#define LPC2XXX_ETH_PHY(eth)            (&lpc2xxx_phy)

//==========================================================================
// This structure is held in bss so everything is automatically zeroed.

static lpc2xxx_eth lpc2xxx_eth_info;

//==========================================================================
// Define a type for pool used for RX packet data pbufs.
// Unhelpfully lwIP requires the payload to directly follow the pbuf.
// This has consequences for CPUs with caches as we can't share the struct pbuf with the payload
// of any *preceding* packet. This means we waste HAL_DCACHE_LINE_SIZE - sizeof(struct pbuf) bytes :(.
// Fixing this requirement in lwIP would require some major surgery that is best done upstream, and
// has been discussed there so may resolved in due course.
// BD_ALIGNMENT is also suitable to use as the alignment here.
//
// The extra 16 bytes in the payload put a spacer between the end of
// one buffer and the next pbuf. It appears that occasionally the next
// two bytes of a multi-buffer packet get written to the word just
// after the end of a buffer. If this is the start of a pbuf then it
// messes up the link pointer. It is 16 bytes to keep the pbufs
// 16-byte aligned.


struct lpc2xxx_pbuf_pool_and_payload {
    struct pbuf pbuf;
    char payload[CYGNUM_LWIP_PBUF_POOL_BUFSIZE+16];
} CYGBLD_ATTRIB_ALIGN(BD_ALIGNMENT);

struct lpc2xxx_pbuf_pool_and_payload *lpc2xxx_pbuf_pool = (struct lpc2xxx_pbuf_pool_and_payload *)LPC2XXX_ETH_RAM;

#define NEXT_RXBD(current) (((current) == (CYGNUM_DEVS_ETH_ARM_LPC2XXX_RX_BUFS - 1)) ? 0 : ((current) + 1))
#define PREV_RXBD(current) (((current) == 0) ? (CYGNUM_DEVS_ETH_ARM_LPC2XXX_RX_BUFS - 1) : ((current) - 1))
#define NEXT_TXBD(current) (((current) == (TX_BUFFER_DESCRIPTOR_COUNT - 1)) ? 0 : ((current) + 1))
#define PREV_TXBD(current) (((current) == 0) ? (TX_BUFFER_DESCRIPTOR_COUNT - 1) : ((current) - 1))

static cyg_uint32 lpc2xxx_eth_isr(cyg_vector_t vector, cyg_addrword_t data);
static void       lpc2xxx_eth_dsr(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data);

// ----------------------------------------------------------------------------
// The MAC address. Comes from RedBoot's config info, if CDL indicates we should
// try. Otherwise this will be supplied by the platform HAL, but some
// platforms may not have appropriate hardware. Worst case, we use the CDL-defined
// configurable default value.

static cyg_uint8    lpc2xxx_eth_default_mac[6]  = { CYGDAT_DEVS_ETH_ARM_LPC2XXX_MACADDR } ;

static void
lpc2xxx_eth_get_config_mac_address(cyg_uint8* mac)
{
    int mac_ok  = 0;
    // Do we have virtual vectors?
#if defined(CYGPKG_DEVS_ETH_ARM_LPC2XXX_REDBOOT_HOLDS_ESA) // implies VV support
    // There has been an incompatible change to hal_if.h. Cope with both new and old names
# ifdef CYGNUM_FLASH_CFG_TYPE_CONFIG_ESA    
    mac_ok  = CYGACC_CALL_IF_FLASH_CFG_OP(CYGNUM_CALL_IF_FLASH_CFG_GET, "eth0_esa_data", mac, CYGNUM_FLASH_CFG_TYPE_CONFIG_ESA);
# else    
    mac_ok  = CYGACC_CALL_IF_FLASH_CFG_OP(CYGNUM_CALL_IF_FLASH_CFG_GET, "eth0_esa_data", mac, CYGNUM_FLASH_CFG_OP_CONFIG_ESA);
# endif
#endif
#ifdef CYGHWR_DEVS_ETH_ARM_LPC2XXX_GET_ESA
    if (!mac_ok) {
        CYGHWR_DEVS_ETH_ARM_LPC2XXX_GET_ESA(mac, mac_ok);
    }
#endif
    if (!mac_ok) {
        memcpy(mac, lpc2xxx_eth_default_mac, 6);
    }
}

// ----------------------------------------------------------------------------

/* lwIP PBUF_POOL creation override */
__externC struct pbuf *
cyg_lpc2xxx_eth_lwip_pool_init(void)
{
    struct pbuf *p;
    cyg_ucount32 i;

    DIAG(2, "Create pool" );
    
    for (i=0; i<CYGNUM_LWIP_PBUF_POOL_SIZE; i++)
    {
        p = &lpc2xxx_pbuf_pool[i].pbuf;
        p->payload = lpc2xxx_pbuf_pool[i].payload;
        p->next = &lpc2xxx_pbuf_pool[i+1].pbuf;
        p->len = p->tot_len = CYGNUM_LWIP_PBUF_POOL_BUFSIZE;
        p->flags = PBUF_FLAG_POOL;
        DIAG(3, "pbuf 0x%08x payload 0x%08x[%d]", p, p->payload, p->len );
    }

    // Last pbuf's next field in fact gets set to NULL to denote end.
    p->next = NULL;

    return &lpc2xxx_pbuf_pool[0].pbuf;
}

// ----------------------------------------------------------------------------
/* Hook when freeing a pool pbuf. */
/* This is called inside a SYS_ARCH_PROTECT, so we should be m-t safe */

__externC void
cyg_lpc2xxx_eth_lwip_pool_free_hook( struct pbuf *p )
{
    struct  lpc2xxx_eth*    eth = &lpc2xxx_eth_info;
    cyg_uint32 cix, pix;
    
    CYG_UNUSED_PARAM(struct pbuf *, p);

    DIAG(2, "Freeing pbuf @ 0x%08x, with payload 0x%08x", (unsigned)p, (unsigned)p->payload);

    READ32( LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_RXPRODIX, pix );
    READ32( LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_RXCONSIX, cix );

    DIAG(3, "cix %d pix %d nxb %d", cix, pix, eth->rx_next_buffer);
    DIAGDUMP32(3, "RX Descriptors", eth->rxbds, 128 );
    
    while( cix != eth->rx_next_buffer && eth->rxbds[cix].addr == 0 )
    {
        struct pbuf *newp;

        newp = pbuf_alloc(PBUF_RAW, CYGNUM_LWIP_PBUF_POOL_BUFSIZE, PBUF_POOL);
    
        if( newp == NULL )
            break;
    
        DIAG(2, "Reprovision rxbd[%d], with pbuf @ 0x%08x (payload 0x%08x)",
             cix, (unsigned)newp, newp ? (unsigned)(p->payload) : 0);

        // we can only really replace it if we succeeded of course.
        eth->rxbds[cix].ctrl = (CYGNUM_LWIP_PBUF_POOL_BUFSIZE-1) | CYGARC_HAL_LPC2XXX_RXBD_CTRL_INTR;
        eth->rxbds[cix].addr = (cyg_uint32)newp->payload;
        eth->rsbds[cix].info = 0;
        eth->rsbds[cix].hash_crc = 0;
            
        cix = NEXT_RXBD(cix);
        WRITE32( LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_RXCONSIX, cix );
    }
}

//======================================================================
// Transmitter/Receiver enable

static void 
lpc2xxx_enable(lpc2xxx_eth *eth)
{
    cyg_uint32 cmd;

    READ32( LPC2XXX_ETH_BASE(eth)+CYGARC_HAL_LPC2XXX_REG_ETH_CMD, cmd);
    cmd |=  CYGARC_HAL_LPC2XXX_REG_ETH_CMD_RXENABLE;
    cmd |=  CYGARC_HAL_LPC2XXX_REG_ETH_CMD_TXENABLE;
    WRITE32( LPC2XXX_ETH_BASE(eth)+CYGARC_HAL_LPC2XXX_REG_ETH_CMD, cmd);

}

// ----------------------------------------------------------------------------

// Configure the pins so that the EMAC has control of them. 
static void
lpc2xxx_cfg_pins(void)
{

#if defined(CYGHWR_HAL_ARM_LPC2XXX_LPC2468) || \
    defined(CYGHWR_HAL_ARM_LPC2XXX_LPC2478) || \
    defined(CYGHWR_HAL_ARM_LPC2XXX_LPC23XX)
    
    cyg_uint32 base = CYGARC_HAL_LPC2XXX_REG_PIN_BASE;

    // The following is something of a kludge to detect the CPU
    // revision and set up PINSEL2 correctly. There is a bug in the
    // engineering samples that requires P1.6, ENET-TX_CLK, to be
    // allocated.
    // This works around the Ethernet.1 erratum.

    cyg_uint32 id;

    HAL_READ_UINT32( CYGARC_HAL_LPC2XXX_REG_ETH_BASE+0xFFC, id );

    if( id == ((0x3902 << 16) | 0x2000) )
    {
        // Engineering sample
        HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_PINSEL2, 0x50151105 );
    }
    else
    {
        // All other revisions
        HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_PINSEL2, 0x50150105 );
    }
    
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_PINSEL3, 0x00000005 );

#else

#error Unknown LPC2XXX variant
   
#endif
}

//======================================================================
// Set a specific address match to a given address. Packets received which
// match this address will be passed on.


static void
lpc2xxx_set_mac(lpc2xxx_eth *eth, cyg_uint8 * enaddr)
{
    cyg_uint32 hi, mid, lo;

    hi  = ((enaddr[1] << 8) |
           (enaddr[0] << 0));

    mid = ((enaddr[3] << 8) |
           (enaddr[2] << 0));
       
    lo  = ((enaddr[5] << 8) |
           (enaddr[4] << 0));

    HAL_WRITE_UINT32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_SA0, lo);
    HAL_WRITE_UINT32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_SA1, mid);
    HAL_WRITE_UINT32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_SA2, hi);
}

//======================================================================

__externC void cyg_lwip_eth_ecos_init(void)
{
    struct lpc2xxx_eth*     eth = &lpc2xxx_eth_info;
    cyg_uint32              base = LPC2XXX_ETH_BASE(eth);
    cyg_uint32              i;
    struct pbuf *           p;
    err_t err;
    //cyg_uint32 usrio, ncfg;
    unsigned short phy_state = 0;
    char do_dhcp;
    //const unsigned char enzero[6] = { 0x0, 0x0, 0x0, 0x0, 0x0, 0x0};
    cyg_uint32 cmd, mac1, mac2, supp, ipgt;
    
    DIAG(1, "entry");

    // Set pin functions correctly
    lpc2xxx_cfg_pins();

#if 0 //def CYGSEM_HAL_USE_ROM_MONITOR
    // If we are debugging over RedBoot, it is possible that there is an
    // outgoing packet still queued up. Give it a chance to get out, a
    // couple of milliseconds should suffice. The preceding initialisation
    // should not conflict with this.
    for (i = 0; i < 20; i++) {
        cyg_uint32 tsr;
        READ32( LPC2XXX_ETH_BASE(eth) + LPC2XXX_EMAC_TSR, tsr );
        if (0 == (tsr & LPC2XXX_EMAC_TSR_TGO )) {
            break;
        }
        HAL_DELAY_US(100);
    }
#endif


    // Reset all
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_MAC1, 0x0000CF00 );   
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_CMD, 0x00000038 );

    LPC2XXX_ETH_DELAY_US(10000);

    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_MAC1, 0x00000000 );

    // Disable TX and RX
    HAL_READ_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_CMD, cmd );
    cmd &= ~(CYGARC_HAL_LPC2XXX_REG_ETH_CMD_TXENABLE|CYGARC_HAL_LPC2XXX_REG_ETH_CMD_RXENABLE);
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_CMD, cmd );

    // Set default interpacket gap
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_IPGR, 18 );

    // Collision window/retry register. Magic value from EA/manual.
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_CLRT, 0x370F );


    // All interrupts were masked further up, so it's safe to attach interrupt handler.
    cyg_drv_interrupt_create(LPC2XXX_ETH_INTR_VECTOR(eth),
                             CYGNUM_DEVS_ETH_ARM_LPC2XXX_INTRPRI,
                             (CYG_ADDRWORD) eth,
                             lpc2xxx_eth_isr,
                             lpc2xxx_eth_dsr,
                             &eth->interrupt_handle,
                             &eth->interrupt_data);

    cyg_drv_interrupt_attach(eth->interrupt_handle);
    // Unmask overall eth interrupt (even though all individual sources
    // remain masked within EMAC)
    cyg_drv_interrupt_unmask(LPC2XXX_ETH_INTR_VECTOR(eth));

    // Determine the MAC address. This is done by either RedBoot or the platform HAL or CDL.
    lpc2xxx_eth_get_config_mac_address(eth->mac);

    DIAG(1, "Using ESA %02x:%02x:%02x:%02x:%02x:%02x",
         eth->mac[0],eth->mac[1],eth->mac[2],
         eth->mac[3],eth->mac[4],eth->mac[5]);

    // Give the EMAC its address
    lpc2xxx_set_mac(eth, eth->mac);

    // Now sort out the buffers and buffer descriptors for both TX and RX.

    // Set up the pointers to the descriptors and buffer pool in
    // ethernet memory.

    eth->txbds = (tbd_t *)CYGARC_UNCACHED_ADDRESS(LPC2XXX_ETH_RAM+(sizeof(struct lpc2xxx_pbuf_pool_and_payload)*CYGNUM_LWIP_PBUF_POOL_SIZE));
    eth->tsbds = (tsd_t *)CYGARC_UNCACHED_ADDRESS(eth->txbds+CYGNUM_DEVS_ETH_ARM_LPC2XXX_TX_BUFS);
    eth->rxbds = (rbd_t *)CYGARC_UNCACHED_ADDRESS(eth->tsbds+CYGNUM_DEVS_ETH_ARM_LPC2XXX_TX_BUFS);
    eth->rsbds = (rsd_t *)CYGARC_UNCACHED_ADDRESS(eth->rxbds+CYGNUM_DEVS_ETH_ARM_LPC2XXX_RX_BUFS);
    eth->txbuf = (cyg_uint8 *)CYGARC_UNCACHED_ADDRESS(eth->rsbds+CYGNUM_DEVS_ETH_ARM_LPC2XXX_RX_BUFS);
    
    DIAG(3,"txbds 0x%08x tsbds 0x%08x rxbds 0x%08x rsbds 0x%08x txbuf 0x%08x end 0x%08x", eth->txbds, eth->tsbds, eth->rxbds, eth->rsbds, eth->txbuf, eth->txbuf+LPC2XXX_EMAC_TX_BUFF_SIZE );

    CYG_ASSERT( ((long)(eth->txbuf+LPC2XXX_EMAC_TX_BUFF_SIZE)  < (LPC2XXX_ETH_RAM+(16*1024))), "Ethernet RAM size" );
    
    // We can fill in the buffer fields in the various rx buffer descriptors.
    for (i = 0; i < CYGNUM_DEVS_ETH_ARM_LPC2XXX_RX_BUFS; i++) {
        p = pbuf_alloc(PBUF_RAW, CYGNUM_LWIP_PBUF_POOL_BUFSIZE, PBUF_POOL);
        if (!p) {
            // By rights this could be an assert, due to the CDL requirement on PBUF_POOL_SIZE.
            // But for now I'll make it a soft error, and simply return, failing to start
            // the device.
            DIAG(1, "Failed to allocate initial rx pool pbufs");
            return;
        }
        // We don't need to track p itself - because we have complete control of the
        // pool we can derive it later.
        eth->rxbds[i].addr   = ((cyg_uint32)CYGARC_PHYSICAL_ADDRESS(p->payload));
        eth->rxbds[i].ctrl   = (CYGNUM_LWIP_PBUF_POOL_BUFSIZE-1) | CYGARC_HAL_LPC2XXX_RXBD_CTRL_INTR;
        eth->rsbds[i].info   = 0;
        eth->rsbds[i].hash_crc = 0;
    }

    // And tell the EMAC where the to find the descriptors
    HAL_WRITE_UINT32(base + CYGARC_HAL_LPC2XXX_REG_ETH_RXDESC, (CYG_ADDRWORD)eth->rxbds);
    HAL_WRITE_UINT32(base + CYGARC_HAL_LPC2XXX_REG_ETH_RXSTATUS, (CYG_ADDRWORD)eth->rsbds);
    HAL_WRITE_UINT32(base + CYGARC_HAL_LPC2XXX_REG_ETH_RXDESCNUM, CYGNUM_DEVS_ETH_ARM_LPC2XXX_RX_BUFS-1);
    HAL_WRITE_UINT32(base + CYGARC_HAL_LPC2XXX_REG_ETH_RXCONSIX, 0);
    
    // Ditto for the tx buffers.
    for (i=0; i < CYGNUM_DEVS_ETH_ARM_LPC2XXX_TX_BUFS; i++)
    {
        eth->txbds[i].addr   = 0; // nothing to put there yet
        eth->txbds[i].ctrl   = 0;
        eth->tsbds[i].info   = 0;
    }

    // And tell the EMAC where the to find the descriptors
    HAL_WRITE_UINT32(base + CYGARC_HAL_LPC2XXX_REG_ETH_TXDESC, (CYG_ADDRWORD)eth->txbds);
    HAL_WRITE_UINT32(base + CYGARC_HAL_LPC2XXX_REG_ETH_TXSTATUS, (CYG_ADDRWORD)eth->tsbds);
    HAL_WRITE_UINT32(base + CYGARC_HAL_LPC2XXX_REG_ETH_TXDESCNUM, CYGNUM_DEVS_ETH_ARM_LPC2XXX_TX_BUFS-1);
    HAL_WRITE_UINT32(base + CYGARC_HAL_LPC2XXX_REG_ETH_TXPRODIX, 0);

    DIAGDUMP(3, "Descriptors", eth->txbds, 256 );
    DIAGDUMP32(3, "Control Regs", base + 0x100, 0x78 );
   
    eth->tx_can_send = 1;

    // Set up the PHY
    CYG_ASSERTC(LPC2XXX_ETH_PHY(eth));

    if (!_eth_phy_init(LPC2XXX_ETH_PHY(eth)))
        return;


    // Reset PHY
    {
        long timeout = 100000;
        cyg_uint16 bmcr;

        _eth_phy_write(LPC2XXX_ETH_PHY(eth), 0, LPC2XXX_ETH_PHY(eth)->phy_addr, 0x8000 );       

        LPC2XXX_ETH_DELAY_US(100);    

        while (--timeout != 0)
        {
            _eth_phy_read(LPC2XXX_ETH_PHY(eth), 0, LPC2XXX_ETH_PHY(eth)->phy_addr, &bmcr );
            //debug2_printf("bmcr %04x\n", bmcr );
            if ((bmcr & 0x8000 ) == 0)
                break;
        }

        if (timeout == 0)
            diag_printf("  Error: failed to reset PHY\n");
    }

    // Force an autonegotiation
    {
        unsigned short reg;
        _eth_phy_read(LPC2XXX_ETH_PHY(eth), 0, LPC2XXX_ETH_PHY(eth)->phy_addr, &reg );
        reg |= (1<<12)|(1<<9);   // Set autonegotiation bits
        _eth_phy_write(LPC2XXX_ETH_PHY(eth), 0, LPC2XXX_ETH_PHY(eth)->phy_addr, reg );
    }
   
    // Get the current mode and print it. This will also wait for the
    // autonegotiation to complete.
    phy_state = _eth_phy_state(LPC2XXX_ETH_PHY(eth));

    //debug2_printf("phy_state %04x\n", phy_state);

    // If the link fails to come up, fail the initialization.
    if ((phy_state & ETH_PHY_STAT_LINK) == 0)
        return;

    // Set up MAC according to negotiated PHY characteristics
   
    mac2 = CYGARC_HAL_LPC2XXX_REG_ETH_MAC2_CRC_ENABLE |
        CYGARC_HAL_LPC2XXX_REG_ETH_MAC2_PAD;

    supp = 0;

    ipgt = 18;
   
    cmd = CYGARC_HAL_LPC2XXX_REG_ETH_CMD_PASSRUNT |
        CYGARC_HAL_LPC2XXX_REG_ETH_CMD_RMII;


    if( phy_state & ETH_PHY_STAT_100MB )
        supp |= CYGARC_HAL_LPC2XXX_REG_ETH_SUPP_SPEED_100;
       
    if( phy_state & ETH_PHY_STAT_FDX )
        mac2 |= CYGARC_HAL_LPC2XXX_REG_ETH_MAC2_FULL_DUPLEX,
            cmd |= CYGARC_HAL_LPC2XXX_REG_ETH_CMD_DUPLEX,
            ipgt = 21;
           

    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_MAC2, mac2 );
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_SUPP, supp );
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_CMD, cmd );
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_IPGT, ipgt );

    // Clear the Statistics counters;
//    lpc2xxx_clear_stats(eth);

    // Enable RX, pass all RX frames
    HAL_READ_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_MAC1, mac1 );
    mac1 |= CYGARC_HAL_LPC2XXX_REG_ETH_MAC1_RX_ENABLE;
    mac1 |= CYGARC_HAL_LPC2XXX_REG_ETH_MAC1_RX_PASS_ALL;
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_MAC1, mac1 );   

    // set up the Rx filter 
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_RXFILTERCTL, 0x0022 );

    // Clear all interrupts
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_INTCLEAR, 0xFFFF );      

    // Enable useful interrupts
    HAL_WRITE_UINT32( base + CYGARC_HAL_LPC2XXX_REG_ETH_INTENABLE, 0x00FF );         

    DIAG(1, "calling higher-level");
    
    // Initialize the upper level driver
    
    err = cyg_lwip_eth_drv_init_netif((void *)eth, eth->mac, &eth->netif, &do_dhcp);

    if (ERR_OK == err)
    {
        // Enable rx/tx operation.
        lpc2xxx_enable(eth);

        eth->started = 1;

#ifdef CYGFUN_LWIP_DHCP
        //
        // we call this after the driver was started successfully
        //
        if (do_dhcp)
            cyg_lwip_dhcp_init(&eth->netif);
#endif
    }

    DIAG(1, "done");
}

// ----------------------------------------------------------------------------
// The RX interrupt triggers whenever a whole frame has been received (or
// when an error has occurred). The hardware may be busy receiving into the
// next buffer.
//
// Because of the ring buffer several packets may arrive before the cpu gets
// around to processing them. It is possible to save a little bit of load
// by masking the interrupt inside the ethernet device, then doing the
// unmask inside the deliver() code.
//
// The TX interrupt triggers when a whole frame has been sent. Since at present
// we only send one frame at a time, the TX hardware will be idle.

static cyg_uint32
lpc2xxx_eth_isr(cyg_vector_t vector, cyg_addrword_t data)
{
    lpc2xxx_eth *eth = (lpc2xxx_eth *)data;

    CYG_UNUSED_PARAM(lpc2xxx_eth *, eth); // potentially unused due to macro use below

    cyg_drv_interrupt_mask(LPC2XXX_ETH_INTR_VECTOR(eth));

    // Ack now, to prevent issues with other interrupts.
    cyg_drv_interrupt_acknowledge(LPC2XXX_ETH_INTR_VECTOR(eth));

    return (CYG_ISR_HANDLED|CYG_ISR_CALL_DSR);  // Run the DSR
}

static void
lpc2xxx_eth_dsr(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data)
{
    // There is no need to worry about cleaning up the buffer descriptors.
    // The hardware will have set the ownership bit, which is the only one
    // which needs handling.
    DIAG(1, "LPC2XXX ETH DSR");

    cyg_lwip_eth_dsr();
}

// ----------------------------------------------------------------------------

// The driver only supports one transmit at a time, and a global is used
// to keep track of whether or not a transmit is in progress.
static __inline__ int
lpc2xxx_eth_can_send(struct lpc2xxx_eth* eth)
{
    return eth->tx_can_send;
}

// ----------------------------------------------------------------------------
// Perform deliveries on the transmit side. Called in delivery thread context.
// check_tsr is set if we should look at the tsr to see if anything needs doing;
// otherwise we consider the last packet finished regardless.

static void
lpc2xxx_eth_deliver_tx(lpc2xxx_eth *eth)
{
    DIAG(1, "tx done");

    // Only need to free if the original pbuf was used and therefore pbuf_ref()d.
    if (eth->tx_pbuf)
    {
        pbuf_free(eth->tx_pbuf);
        eth->tx_pbuf = NULL;
        eth->tx_index = NEXT_TXBD(eth->tx_index);    
        eth->tx_can_send = 1;
    }
    else
    {
#ifdef CYGPKG_INFRA_DEBUG
        // This is an important thing to catch, but could arise for all sorts
        // of reasons.
        diag_printf("LPC2XXX eth: delivered tx but no tx_pbuf?! Tell jifl\n");
#endif
    }
}

// ----------------------------------------------------------------------------
//
// cyg_lwip_eth_low_level_output():
//
// This is the function called by higher layers which should do the actual
// transmission of the packet. The packet is contained in the pbuf that is
// passed to the function. This pbuf might be chained.


__externC err_t
cyg_lwip_eth_low_level_output(struct netif *netif, struct pbuf *p)
{
    lpc2xxx_eth *eth = (lpc2xxx_eth *)netif->state;
    cyg_uint32 index;
    cyg_uint32 last_index;
    struct pbuf *q;

    // Sanity check p. Don't waste code returning on an error like this.
    // Just don't screw up in the first place!
    if (!p)
    {
        DIAG(1, "cyg_lwip_eth_low_level_output passed NULL pbuf");
        CYG_FAIL("cyg_lwip_eth_low_level_output passed NULL pbuf");
    }

    if (!p->len || !p->tot_len)
    {
        DIAG(1, "cyg_lwip_eth_low_level_output passed empty pbuf");
        CYG_FAIL("cyg_lwip_eth_low_level_output passed empty pbuf");
    }

    if (!eth->started)
        return ERR_IF;

    // Can we send?
    if (lpc2xxx_eth_can_send(eth) <= 0)
    {
        DIAG(2, "could not send");

        // If we can't send, we need to call the delivery function to make
        // sure that the reason we can't send isn't just because we are
        // either higher priority than the normal thread doing delivery,
        // and thus preventing it delivering; or that we _are_ the
        // delivery thread, in which case we would never see it complete
        // even if it had, because clearly this thread has been "doing
        // stuff" if it has stuff to send.
        lpc2xxx_eth_deliver_tx(eth);

#ifdef CYGIMP_IO_ETH_DRIVERS_LWIP_TX_FULL_WAIT
        while (lpc2xxx_eth_can_send(eth) <= 0)
        {
            // As above, call the delivery function each time. We may
            // be the delivery thread anyway, so if we didn't, looping here
            // and not calling it would otherwise cause everything to freeze.
            lpc2xxx_eth_deliver_tx(eth);

# if (CYGNUM_IO_ETH_DRIVERS_LWIP_TX_FULL_WAIT_DELAY > 0)
            // Give others a chance to run
            LPC2XXX_ETH_DELAY_US(CYGNUM_IO_ETH_DRIVERS_LWIP_TX_FULL_WAIT_DELAY);
# endif
        }
#else // !CYGIMP_IO_ETH_DRIVERS_LWIP_TX_FULL_WAIT
        /* Otherwise if we still can't send, just drop the packet completely. 
         * We claim success because there isn't necessarily really an error as such.
         * We may just be generating data too fast.
         */
        if (lpc2xxx_eth_can_send(eth) <= 0) {
            DIAG(1, "Couldn't send packet. Dropping");
            return ERR_OK;
        }
#endif
        DIAG(1, "Can now send");
    }

#if 1

    // This code copies the txpacket to a buffer in ethernet RAM.
    
    eth->tx_can_send    = 0;
    /* In the following line, last_index doesn't really need set, but it
     * silences a compiler warning.
     */
    last_index = index = eth->tx_index;

    // incr ref count of pbuf, so that caller's free will not really free up
    // memory. At least not until packet is definitely sent.
    pbuf_ref(p);
    eth->tx_pbuf = p;
    
    cyg_uint8 *txbufp = eth->txbuf;
    int tot_len = 0;
    
    for (q=p; q; q=q->next)
    {
        if (q->len)
        {
            DIAG(3, "pbuf %08x", q );
//            DIAGDUMP(3, "payload", p->payload, q->len );

            memcpy( txbufp, q->payload, q->len );
            txbufp += q->len;
            tot_len += q->len;
        }
    }

    eth->txbds[index].addr = (cyg_uint32)CYGARC_PHYSICAL_ADDRESS(eth->txbuf);
//    DIAGDUMP(3, "payload", eth->txbds[index].addr, tot_len );            
    eth->txbds[index].ctrl = (cyg_uint32)(tot_len & CYGARC_HAL_LPC2XXX_TXBD_CTRL_SIZE) - 1;
    last_index = index;
        
#else

    // This version tries to use the supplied buffers. However, this
    // doesn't seem to work; the ethernet ends up transmitting
    // 0x5555's. This code is being left in place in case we ever find
    // out why this is happening.
    
    cyg_uint32 pbuf_chain_length=1;
    
    // Quickly need to check the pbuf chain is not too long for us.
    // We could probably do this at the same time as the loop where
    // we fill in the txbds below, but to be honest, the chains should
    // usually be short enough (typically not a chain at all!) that 
    // this is faster.
    for (q=p->next; q; q=q->next)
        if (q->len)
            pbuf_chain_length++;

    if (pbuf_chain_length >= (CYGNUM_DEVS_ETH_ARM_LPC2XXX_TX_BUFS))
        return ERR_IF;

    eth->tx_can_send    = 0;
    /* In the following line, last_index doesn't really need set, but it
     * silences a compiler warning.
     */
    last_index = index = eth->tx_index;

    // incr ref count of pbuf, so that caller's free will not really free up
    // memory. At least not until packet is definitely sent.
    pbuf_ref(p);
    eth->tx_pbuf = p;
    
    for (q=p; q; q=q->next)
    {
        // Don't waste a txbd on empty chain entries
        if (q->len)
        {
            DIAG(3, "pbuf %08x", q );
//            DIAGDUMP(3, "payload", p->payload, q->len );
            
            eth->txbds[index].addr = (cyg_uint32)CYGARC_PHYSICAL_ADDRESS(q->payload);
//            DIAGDUMP(3, "payload", eth->txbds[index].addr, q->len );            
            eth->txbds[index].ctrl = (cyg_uint32)(q->len & CYGARC_HAL_LPC2XXX_TXBD_CTRL_SIZE) - 1;
            last_index = index;
            index = NEXT_TXBD(index);
        }
    }
#endif

    // Set interrupt and last bits in last descriptor of frame
    eth->txbds[last_index].ctrl |= CYGARC_HAL_LPC2XXX_TXBD_CTRL_INTR | CYGARC_HAL_LPC2XXX_TXBD_CTRL_LAST;

    eth->tx_index = index;

    // Ensure writeback cache data is written to physical memory
#ifdef HAL_DCACHE_SYNC
    HAL_DCACHE_SYNC();
#endif

    DIAGPKT(1, "start", p->payload, p->tot_len);

    // Start transmission by setting producer register in MAC to end
    // of new packet.
    WRITE32( LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_TXPRODIX, NEXT_TXBD(eth->tx_index) );
    
//    DIAGDUMP(3, "TX Descriptors", eth->txbds, 64+32 );
//    DIAGDUMP32(3, "Control Regs", LPC2XXX_ETH_BASE(eth) + 0x100, 0x78 );

    return ERR_OK;
}

// ----------------------------------------------------------------------------
// Perform deliveries on the RX side. Called in delivery thread context.

static void
lpc2xxx_eth_deliver_rx( lpc2xxx_eth *eth )
{
    int cache_state;
    int current_rxbd;
    int start_of_frame = -1;
    cyg_uint32 stat;
    cyg_uint32 pix, cix;
    int len = 0;
    
    DIAG(1, "rx_rdy");

    // Invalidate the cache's view of the rx buffer data space as this will
    // have changed. The rx buffer descriptors will be ok as they are accessed
    // via uncached space, and are in their own cache lines; but we have to
    // invalidate everything anyway. Since we invalidate, we need to sync dirty
    // cache lines to memory first.
    //
    // This could be done more selectively, but apparently the HAL_DCACHE_STORE
    // and HAL_DCACHE_INVALIDATE implementations don't work for ARM9 (and so
    // aren't defined). And for ARM7 these are no-ops.

    // If cache is always off (ARM7), cache_state will be set to constant 0 and
    // compiler should optimise the rest away hopefully, so we don't even get
    // the interrupt disable blip.
    HAL_DCACHE_IS_ENABLED(cache_state);
        
    if (cache_state) {
        cyg_drv_isr_lock(); // Need ISRs off during cache sync/invalidate
        HAL_DCACHE_SYNC();
        HAL_DCACHE_INVALIDATE_ALL();
        cyg_drv_isr_unlock();
    }

    // Loop while there are non-empty packets. This has to be done with a bit
    // of care. Packets should be processed in the right order where possible,
    // which means keeping track of the last packet received.
    current_rxbd = eth->rx_next_buffer;
    for (;;)
    {
        
        // We work on the assumption that the driver and the hardware agree
        // which rxbd is being filled next. Things will go wrong if this isn't true.
        // So we always assume the current rxbd should be the start, and you keep
        // going till you reached the End-Of-Frame.

        READ32( LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_RXPRODIX, pix );
        READ32( LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_RXCONSIX, cix );

        DIAG(3, "cix %d pix %d crxbd %d", cix, pix, current_rxbd);

        while( cix != eth->rx_next_buffer )
        {
            struct pbuf *newp;

            newp = pbuf_alloc(PBUF_RAW, CYGNUM_LWIP_PBUF_POOL_BUFSIZE, PBUF_POOL);

            if (newp)
            {
                DIAG(2, "Reprovision rxbd[%d], with pbuf @ 0x%08x (payload 0x%08x)",
                     current_rxbd, (unsigned)newp, newp ? (unsigned)(newp->payload) : 0);

                // we can only really replace it if we succeeded of course.
                eth->rxbds[cix].ctrl = (CYGNUM_LWIP_PBUF_POOL_BUFSIZE-1) | CYGARC_HAL_LPC2XXX_RXBD_CTRL_INTR;
                eth->rxbds[cix].addr = (cyg_uint32)newp->payload;
                eth->rsbds[cix].info = 0;
                eth->rsbds[cix].hash_crc = 0;

                cix = NEXT_RXBD(cix);
                WRITE32( LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_RXCONSIX, cix );
            }
            else
                break;
        }

        // See if the next descriptor is complete.
        
        if( current_rxbd == pix )
        {
            if ( start_of_frame >= 0 )
            {
                // We had found a start of frame. There are two possible reasons
                // we could find an empty rxbd now:
                // 1) The frame is in transit, and the EMAC is still filling the buffers.
                // 2) It's an error frame and will never complete.
                // We would purge it if we could be sure about 2, but it might be 1.
                // So we leave it for now. The next RX interrupt on frame completion
                // will cause us to notice.

                DIAG(2, "Had found SOF @ rxbd[%d], but now found empty rxbd[%d] - in transit?",
                     start_of_frame, current_rxbd);
                // leave eth->rx_next_buffer untouched
            }
            return;
        } // if

        
        // Otherwise, this is owned by software

        stat = eth->rsbds[current_rxbd].info;

        DIAG(3, "stat %08x", stat );
                
        if ( start_of_frame < 0 )
        {
            // start of a new frame
            DIAG(1, "Found SOF at rxbd[%d] (old SOF==rxbd[%d])", current_rxbd, start_of_frame);
            start_of_frame = current_rxbd;
            len = 0;
            eth->rx_next_buffer = start_of_frame; // new baseline
        } // if

        // Accumulate frame length from fragment lengths
        len += (stat & CYGARC_HAL_LPC2XXX_RXSD_INFO_SIZE) + 1;

        DIAG(3, "len frag %d total %d", (stat & CYGARC_HAL_LPC2XXX_RXSD_INFO_SIZE) + 1, len );
        
        // End of frame. A whole valid packet.
        if ( stat & CYGARC_HAL_LPC2XXX_RXSD_INFO_LAST )
        {
            struct pbuf *p, *first_pbuf;

            // Found end of frame. Might be same slot as SOF, and that's fine.
            // But if we didn't find the start....
            if ( start_of_frame < 0 )
            {
                // Oh dear. We found EOF, but no SOF. Something's gone wrong.
                DIAG(1, "Eek! EOF at rxbd[%d] but no SOF!", current_rxbd);
                CYG_FAIL("RXBD EOF but no SOF");
            }

            DIAG(2, "Detected whole frame. SOF @ rxbd[%d], EOF @ rxbd[%d], len %d",
                 start_of_frame, current_rxbd, len);

            // rewind rxbd to start of frame. We don't mind since we know we've got a consistent frame.
            current_rxbd = start_of_frame;

            // The pbuf structure always precedes the pbuf payload, from the way we set the pool up.
            p = first_pbuf = (struct pbuf *)(RXBD_BUFFER_ADDR(eth->rxbds[start_of_frame]) - sizeof(struct pbuf));

            first_pbuf->tot_len = len;

            for (;;)
            {
                if ( p->payload != RXBD_BUFFER_ADDR(eth->rxbds[current_rxbd]) )
                {
                    DIAG(1, "Eeek! Expected pbuf (==%08x) payload (==%08x) is not what's in rxbd[%d] (==%08x)",
                         (unsigned)p, (unsigned)p->payload, current_rxbd, eth->rxbds[current_rxbd].addr);
                    DIAGDUMP32(3, "RX Descriptors", eth->rxbds, 128 );
                    CYG_FAIL("Pbuf payload mismatch with rxbd buffer");
                }

                stat = eth->rsbds[current_rxbd].info;
                DIAG(3, "bd %d %08x stat %08x", eth->rxbds[current_rxbd].addr, stat );
                DIAGDUMP(3, "Buffer", eth->rxbds[current_rxbd].addr, (stat&CYGARC_HAL_LPC2XXX_RXSD_INFO_SIZE)+1);

                DIAG(1, "Adding pbuf 0x%08x from rxbd[%d] with buffer 0x%08x length %d stat 0x%08x to chain",
                     (unsigned)p, current_rxbd, (unsigned)RXBD_BUFFER_ADDR(eth->rxbds[current_rxbd]),
                     (stat & CYGARC_HAL_LPC2XXX_RXSD_INFO_SIZE) +1 ,  stat);

                // Mark RXBD empty, so it will be reprovisioned with a
                // pbuf in either the free hook or the next call/loop
                // of this routine.
                eth->rxbds[current_rxbd].addr = 0;                

                if ( 0 != (stat & CYGARC_HAL_LPC2XXX_RXSD_INFO_LAST) )
                {
                    // since it's the last, let's check the length
                    if ( len > CYGNUM_LWIP_PBUF_POOL_BUFSIZE )
                    {
                        DIAG(1, "Eeek! Length %d unexpectedly too large", len);
                        CYG_FAIL("When reached EOF fragment, remaining length was too large");
                    }
                    p->len = len;
                    p->next = NULL;
                    break; // finished the frame
                } else {
                    p->len = CYGNUM_LWIP_PBUF_POOL_BUFSIZE;
                    len -= CYGNUM_LWIP_PBUF_POOL_BUFSIZE;

                    // Now link to payload of next rxbd in frame
                    current_rxbd = NEXT_RXBD(current_rxbd);
                    // The pbuf structure always precedes the pbuf payload, from the way we set the pool up.
                    p->next = (struct pbuf *)(RXBD_BUFFER_ADDR(eth->rxbds[current_rxbd]) - sizeof(struct pbuf));
                    p = p->next;
                    p->tot_len = len;
                } // else
            } // for
            DIAGPKT(1, "packet received", first_pbuf->payload, first_pbuf->tot_len);

            // Reset pointers for next packet
            current_rxbd = eth->rx_next_buffer = NEXT_RXBD(current_rxbd);
            start_of_frame = -1;

            // Send this packet, i.e. pbuf chain, up.
            cyg_lwip_eth_drv_ecosif_input( &eth->netif, first_pbuf );
            
        } // if
        else // Not end of frame - a starting or intermediate packet frag
        {
            if (start_of_frame < 0)
            {
                DIAG(1, "Eek! found packet frag at rxbd[%d] with no start found", current_rxbd);
                CYG_FAIL("Packet frag found with no start");
            }
            current_rxbd = NEXT_RXBD(current_rxbd);
        }
    } // for
} /* lpc2xxx_eth_deliver_rx() */

// ----------------------------------------------------------------------------
// The delivery function is called from thread context, after the DSR has woken
// up the packet handling thread. It needs to report completed transmits so
// that higher-level code can release the tx pbuf if needed, and report all
// received frames.

__externC void
cyg_lwip_eth_run_deliveries(void)
{
    lpc2xxx_eth *eth = &lpc2xxx_eth_info;
    cyg_uint32 intstat;

    //    DIAG(1, "entry");

    for (;;)
    {
        // Read the interrupt status. 
        READ32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_INTSTATUS, intstat);


        DIAG(1, "intstat 0x%08x", intstat);

        // Abort if there are no events to process
        if( intstat == 0 )
            break;

        // Either successful frame transmission, or there was an error sending.
        // Either way, call tx delivery function.
        if (intstat & CYGARC_HAL_LPC2XXX_REG_ETH_INT_TXDONE )
        {
            WRITE32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_INTCLEAR, CYGARC_HAL_LPC2XXX_REG_ETH_INT_TXDONE);
            lpc2xxx_eth_deliver_tx(eth);
        }

        // Successful frame reception
        if (intstat & CYGARC_HAL_LPC2XXX_REG_ETH_INT_RXDONE )
        {
            WRITE32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_INTCLEAR, CYGARC_HAL_LPC2XXX_REG_ETH_INT_RXDONE);
            lpc2xxx_eth_deliver_rx(eth);
        }

        // Just report any errors that occur
        if (intstat & CYGARC_HAL_LPC2XXX_REG_ETH_INT_RXOVR )
        {
            WRITE32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_INTCLEAR, CYGARC_HAL_LPC2XXX_REG_ETH_INT_RXOVR);
            DIAG(1, "RX overrun");
        }
        if (intstat & CYGARC_HAL_LPC2XXX_REG_ETH_INT_RXERROR )
        {
            WRITE32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_INTCLEAR, CYGARC_HAL_LPC2XXX_REG_ETH_INT_RXERROR);
            DIAG(1, "RX error");
            DIAGDUMP32(3, "Control Regs", LPC2XXX_ETH_BASE(eth) + 0x100, 0x78 );
            DIAGDUMP32(3, "RX Descriptors", eth->rxbds, 128 );            
        }
        if (intstat & CYGARC_HAL_LPC2XXX_REG_ETH_INT_RXFINISH )
        {
            WRITE32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_INTCLEAR, CYGARC_HAL_LPC2XXX_REG_ETH_INT_RXFINISH);
            DIAG(1, "RX finish");
        }
        if (intstat & CYGARC_HAL_LPC2XXX_REG_ETH_INT_TXUNDR )
        {
            WRITE32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_INTCLEAR, CYGARC_HAL_LPC2XXX_REG_ETH_INT_TXUNDR);
            DIAG(1, "TX underflow");
        }
        if (intstat & CYGARC_HAL_LPC2XXX_REG_ETH_INT_TXERROR )
        {
            WRITE32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_INTCLEAR, CYGARC_HAL_LPC2XXX_REG_ETH_INT_TXERROR);
            DIAG(1, "TX error");
        }
        if (intstat & CYGARC_HAL_LPC2XXX_REG_ETH_INT_TXFINISH )
        {
            WRITE32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_INTCLEAR, CYGARC_HAL_LPC2XXX_REG_ETH_INT_TXFINISH);
            DIAG(1, "TX finish");
        }
        if (intstat & CYGARC_HAL_LPC2XXX_REG_ETH_INT_SOFT )
        {
            WRITE32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_INTCLEAR, CYGARC_HAL_LPC2XXX_REG_ETH_INT_SOFT);
            DIAG(1, "Soft interrupt");
        }
        if (intstat & CYGARC_HAL_LPC2XXX_REG_ETH_INT_WAKE )
        {
            WRITE32(LPC2XXX_ETH_BASE(eth) + CYGARC_HAL_LPC2XXX_REG_ETH_INTCLEAR, CYGARC_HAL_LPC2XXX_REG_ETH_INT_WAKE);
            DIAG(1, "Wake interrupt");
        }

        
        /* Nothing else wants explicit handling as far as I'm concerned. */
    }
    
    // Allow more interrupts.
    cyg_drv_interrupt_unmask(LPC2XXX_ETH_INTR_VECTOR(eth));
}

// ----------------------------------------------------------------------------

// SET_MAC_ADDRESS is straightforward, it just requires updating two registers.

__externC int
cyg_lwip_eth_ioctl(struct netif *netif, unsigned long key, void* data, int data_length)
{
    lpc2xxx_eth*    eth = (lpc2xxx_eth*) netif->state;

    DIAG(1, "entry");
    
    switch(key) {
      case ETH_DRV_SET_MAC_ADDRESS:
        {
            memcpy(eth->mac, data, 6);
            lpc2xxx_set_mac(eth, eth->mac);
            return 0;
        }

      default:
        return 1;
    }
}

// ----------------------------------------------------------------------------
/* EOF if_lpc2xxx-lwip.c */
