//==========================================================================
//
//      if_lan9118.c
//
//	    Ethernet driver for SMSC LAN9118
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2005 Free Software Foundation, Inc.                        
// Copyright (C) 2005 eCosCentric Limited                                   
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//=============================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):   bartv
// Date:        2005-07-20
//
//####DESCRIPTIONEND####
//=============================================================================

#include <pkgconf/system.h>
#include CYGBLD_HAL_PLATFORM_H
#include <pkgconf/devs_eth_smsc_lan9118.h>
#include <pkgconf/io_eth_drivers.h>

#include <cyg/infra/cyg_type.h>
#include <cyg/infra/cyg_ass.h>
#include <cyg/hal/hal_arch.h>
#include <cyg/hal/hal_intr.h>
#include <cyg/hal/hal_io.h>
#include <cyg/hal/hal_if.h>
#include <cyg/infra/diag.h>
#include <cyg/hal/drv_api.h>
#include <cyg/io/eth/netdev.h>
#include <cyg/io/eth/eth_drv.h>
#include <string.h>
#ifdef CYGPKG_NET
#include <pkgconf/net.h>
#include <net/if.h>  /* Needed for struct ifnet */
#endif
#include <cyg/io/eth/eth_drv_stats.h>
#include <cyg/io/lan9118.h>

// ----------------------------------------------------------------------------
// Compile-time options.

// The receive fifo is always read an int at a time. The data may be
// written to the receive buffer an int at a time if suitably aligned,
// otherwise byte operations are needed. The device supports optional
// initial padding to facilitate alignment.
//
// The full TCP/IP stacks have an sg_list[0].len of 14 for just the
// ethernet header, and the IP data goes into following buffers.
// Initial padding of 2 means the ethernet header requires byte ops
// but the main data uses integer ops. RedBoot just uses a single
// buffer so zero initial padding should give the desired alignment.
// LWIP uses linked lists of 128-byte buffers so again no padding
// is desired.
//
// Obviously the above is subject to change as the various packages
// get updated. The driver will still work if the alignments change,
// but at reduced performance. If the platform requires that all data
// be swapped anyway because of endianness issues then there is no
// point in optimizing for int operations.
#if defined(CYGPKG_IO_ETH_DRIVERS_NET) && !defined(HAL_LAN9118_SWAP_DATA)
# define RX_INITIAL_PADDING     2
#else
# define RX_INITIAL_PADDING     0
#endif

// Debug support, in the form of a simple trace buffer. Also make it
// easy to enable stats without messing about with the configuration.
// Debugging must be enabled before #include'ing the header which
// defines the lan9118 structure with its statistics fields.
#define DEBUG       0
#define DEBUG_WRAP  1

#ifdef CYGFUN_DEVS_ETH_SMSC_LAN9118_STATISTICS
# define INCR_STAT(_lan9118_, _which_)              \
    CYG_MACRO_START                                 \
    (_lan9118_)->_which_  += 1;                     \
    CYG_MACRO_END
#else
# define INCR_STAT(_lan9118_, _which_)  CYG_EMPTY_STATEMENT
#endif

#if DEBUG <= 0

# define TRACE(_level_, _msg_)                      \
    CYG_MACRO_START                                 \
    CYG_MACRO_END

# define TRACEPKT(_level_, _msg_, _pkt_, _len_)     \
    CYG_MACRO_START                                 \
    CYG_MACRO_END

#else
// Trace buffer size. This has to be kept smallish or RedBoot will overrun
// its RAM allocation. With 24 bytes per entry, 300 entries require a bit
// under 8K.
# define TRACE_BUFFER_SIZE      300
typedef struct lan9118_eth_trace {
    const char*     fn;
    const char*     msg;
    cyg_uint32      len;
    cyg_uint8       packet[14];
} lan9118_eth_trace_entry;
static lan9118_eth_trace_entry  lan9118_eth_trace_data[TRACE_BUFFER_SIZE];
static int                      lan9118_eth_trace_next      = TRACE_BUFFER_SIZE - 1;
static cyg_bool                 lan9118_eth_trace_wrapped   = false;

static void
lan9118_eth_trace(const char* fn, const char* msg, cyg_uint8* packet, cyg_uint32 len)
{
    lan9118_eth_trace_entry* entry = &(lan9118_eth_trace_data[lan9118_eth_trace_next]);

# ifdef DEBUG_WRAP
    if (0 == lan9118_eth_trace_next) {
        lan9118_eth_trace_next = TRACE_BUFFER_SIZE - 1;
        lan9118_eth_trace_wrapped = true;
    } else {
        lan9118_eth_trace_next -= 1;
    }
# else
    if (lan9118_eth_trace_next < 0) {
        return;
    }
    lan9118_eth_trace_next -= 1;
# endif

    entry->fn   = fn;
    entry->msg  = msg;
    entry->len  = len;
    if ((cyg_uint8*)0 == packet) {
        memset(entry->packet, 0, 14);
    } else {
        memcpy(entry->packet, packet, 14);
    }
}

#  define TRACE(_level_, _msg_)                                             \
    CYG_MACRO_START                                                         \
    if ((_level_) <= DEBUG) {                                               \
        lan9118_eth_trace(__func__, _msg_, (cyg_uint8*)0, 0);               \
    }                                                                       \
    CYG_MACRO_END

# define TRACEPKT(_level_, _msg_, _pkt_, _len_)                             \
    CYG_MACRO_START                                                         \
    if ((_level_) <= DEBUG) {                                               \
        lan9118_eth_trace(__func__, _msg_, (cyg_uint8*)_pkt_, _len_);       \
    }                                                                       \
    CYG_MACRO_END

// A function that gets placed in RAM. This can be called by flash-resident
// code, e.g. RedBoot, when I need a breakpoint on a specific condition.
static void lan9118_eth_ramfn(void) __attribute__((section (".2ram.lan9118_eth_ramfn")));
static int  lan9118_eth_ramfn_calls;

static void
lan9118_eth_ramfn(void)
{
    lan9118_eth_ramfn_calls += 1;
}

#endif

// ----------------------------------------------------------------------------
// Diagnostic messages.
//
// Flip this zero to a one to enable some simple diag_printf()s to
// help bring-up on new board ports.

#if 0
#define lan_diag( __fmt, ... ) diag_printf("LAN: %30s[%4d]: " __fmt, __FUNCTION__, __LINE__, __VA_ARGS__ );
#define lan_dump_buf( __buf, __size )  diag_dump_buf( __buf, __size )
#else
#define lan_diag( __fmt, ... ) 
#define lan_dump_buf( __buf, __size )
#endif

// ----------------------------------------------------------------------------
// Register access. If there is a single LAN9118 device on the board then
// its base address and interrupt vectors are known at compile-time.
// Otherwise those parameters have to come from the lan9118 instance
// structure. By default HAL_WRITE_UINT32() and HAL_READ_UINT32() are
// used for the low-level accesses, but the platform HAL can provide
// alternative macros.

#if (CYGNUM_DEVS_ETH_SMSC_LAN9118_COUNT > 1)
# define BASE(_inst_)       ((_inst_)->lan9118_base)
# define ISRVEC(_inst_)     ((_inst_)->lan9118_isrvec)
# define ISRPRI(_inst_)     ((_inst_)->lan9118_isrpri)
#else
# define BASE(_inst_)       HAL_LAN9118_BASE
# define ISRVEC(_inst_)     HAL_LAN9118_ISRVEC
# define ISRPRI(_inst_)     HAL_LAN9118_ISRPRI
#endif

#ifdef HAL_LAN9118_READ
# define READ(_inst_, _offset_, _value_)    HAL_LAN9118_READ(_inst_, _offset_, _value_)
#else
# define READ(_inst_, _offset_, _value_)    HAL_READ_UINT32(BASE(_inst) + _offset_, _value_)
#endif
#ifdef HAL_LAN9118_WRITE
# define WRITE(_inst_, _offset_, _value_)   HAL_LAN9118_WRITE(_inst_, _offset_, _value_)
#else
# define WRITE(_inst_, _offset_, _value_)   HAL_WRITE_UINT32(BASE(_inst) + _offset_, _value_)
#endif

// A WRITE may not take effect immediately. One or more dummy reads from
// the BYTE_TEST register are needed.
#define WAIT1(_lan9118_)                                \
    CYG_MACRO_START                                     \
    cyg_uint32 _byte_test_;                             \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    CYG_MACRO_END

#define WAIT2(_lan9118_)                                \
    CYG_MACRO_START                                     \
    cyg_uint32 _byte_test_;                             \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    CYG_MACRO_END

#define WAIT3(_lan9118_)                                \
    CYG_MACRO_START                                     \
    cyg_uint32 _byte_test_;                             \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    CYG_MACRO_END

#define WAIT4(_lan9118_)                                \
    CYG_MACRO_START                                     \
    cyg_uint32 _byte_test_;                             \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    CYG_MACRO_END

#define WAIT7(_lan9118_)                                \
    CYG_MACRO_START                                     \
    cyg_uint32 _byte_test_;                             \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    READ((_lan9118_), LAN9118_BYTE_TEST, _byte_test_);  \
    CYG_MACRO_END

void
cyg_lan9118_write_reg(lan9118_instance* lan9118, cyg_uint32 reg, cyg_uint32 val)
{
    WRITE(lan9118, reg, val);
}

cyg_uint32
cyg_lan9118_read_reg(lan9118_instance* lan9118, cyg_uint32 reg)
{
    cyg_uint32  val;
    READ(lan9118, reg, val);
    return val;
}

void
cyg_lan9118_write_mac_reg(lan9118_instance* lan9118, cyg_uint32 reg, cyg_uint32 val)
{
    cyg_uint32  mac_csr_cmd;

    do {
        READ(lan9118, LAN9118_MAC_CSR_CMD, mac_csr_cmd);
    } while( mac_csr_cmd & LAN9118_MAC_CSR_CMD_CSR_BUSY);
    WRITE(lan9118, LAN9118_MAC_CSR_DATA, val);
    WAIT1(lan9118);
    WRITE(lan9118, LAN9118_MAC_CSR_CMD,  LAN9118_MAC_CSR_CMD_CSR_BUSY | (reg << LAN9118_MAC_CSR_CMD_CSR_ADDRESS_SHIFT));
    WAIT1(lan9118);
    do {
        READ(inst, LAN9118_MAC_CSR_CMD, mac_csr_cmd);
    } while( mac_csr_cmd & LAN9118_MAC_CSR_CMD_CSR_BUSY);
}

cyg_uint32
cyg_lan9118_read_mac_reg(lan9118_instance* lan9118, cyg_uint32 reg)
{
    cyg_uint32  mac_csr_cmd;
    cyg_uint32  result;

    do {
        READ(lan9118, LAN9118_MAC_CSR_CMD, mac_csr_cmd);
    } while( mac_csr_cmd & LAN9118_MAC_CSR_CMD_CSR_BUSY);
    WRITE(lan9118, LAN9118_MAC_CSR_CMD, LAN9118_MAC_CSR_CMD_CSR_BUSY | LAN9118_MAC_CSR_CMD_RnW | (reg << LAN9118_MAC_CSR_CMD_CSR_ADDRESS_SHIFT));
    WAIT1(lan9118);
    do {
        READ(lan9118, LAN9118_MAC_CSR_CMD, mac_csr_cmd);
    } while( mac_csr_cmd & LAN9118_MAC_CSR_CMD_CSR_BUSY);
    READ(lan9118, LAN9118_MAC_CSR_DATA, result);
    return result;
}

void
cyg_lan9118_write_phy_reg(lan9118_instance* lan9118, cyg_uint32 reg, cyg_uint32 val)
{
    cyg_uint32  miiacc;
    do {
        miiacc = cyg_lan9118_read_mac_reg(lan9118, LAN9118_MAC_MIIACC);
    } while (miiacc & LAN9118_MAC_MIIACC_MIIBZY);
    cyg_lan9118_write_mac_reg(lan9118, LAN9118_MAC_MIIDATA, (LAN9118_SWAP16(val) << LAN9118_MAC_MII_DATA_SHIFT));
    cyg_lan9118_write_mac_reg(lan9118, LAN9118_MAC_MIIACC,
                              LAN9118_MAC_MIIACC_PHY_ADDRESS_MAGIC | LAN9118_MAC_MIIACC_MIIBZY |
                              (reg << LAN9118_MAC_MIIACC_MIIRINDA_SHIFT) | LAN9118_MAC_MIIACC_MIIWnR);
    do {
        miiacc = cyg_lan9118_read_mac_reg(lan9118, LAN9118_MAC_MIIACC);
    } while (miiacc & LAN9118_MAC_MIIACC_MIIBZY);
}

cyg_uint32
cyg_lan9118_read_phy_reg(lan9118_instance* lan9118, cyg_uint32 reg)
{
    cyg_uint32  miiacc;
    cyg_uint32  result;

    do {
        miiacc = cyg_lan9118_read_mac_reg(lan9118, LAN9118_MAC_MIIACC);
    } while (miiacc & LAN9118_MAC_MIIACC_MIIBZY);
    cyg_lan9118_write_mac_reg(lan9118, LAN9118_MAC_MIIACC,
                              LAN9118_MAC_MIIACC_PHY_ADDRESS_MAGIC | LAN9118_MAC_MIIACC_MIIBZY |
                              (reg << LAN9118_MAC_MIIACC_MIIRINDA_SHIFT));
    do {
        miiacc = cyg_lan9118_read_mac_reg(lan9118, LAN9118_MAC_MIIACC);
    } while (miiacc & LAN9118_MAC_MIIACC_MIIBZY);
    result  = cyg_lan9118_read_mac_reg(lan9118, LAN9118_MAC_MIIDATA) >> LAN9118_MAC_MII_DATA_SHIFT;
    result  = LAN9118_SWAP16(result);

    return result;
}


// ----------------------------------------------------------------------------
// NOTE: add a utility for manipulating the EEPROM if present, as an
// easy way for platform HALs to get the MAC address.

// ----------------------------------------------------------------------------
// The main implementation

// can_send() does not know the size of the packet being transmitted.
// Assume it is full-side, i.e. 1514 bytes. The packet may be split
// into multiple fragments, each of which can require up to 6 bytes of
// padding plus a 4-byte command word.
//
// In send(), once the packet is in the fifo the memory can be
// returned to higher-level code immediately. This can then call
// can_send() and send() again, so under heavy load there are recursion
// and stack usage considerations. Keeping track of the number of calls
// to send() avoids problems.
static int
lan9118_can_send(struct eth_drv_sc* sc)
{
    cyg_uint32  tx_fifo_inf;
    lan9118_instance*   lan9118 = (lan9118_instance*)sc->driver_private;

    if (lan9118->lan9118_up && (lan9118->lan9118_in_send_count < 4)) {
        READ( lan9118, LAN9118_TX_FIFO_INF, tx_fifo_inf);
        tx_fifo_inf  = (tx_fifo_inf & LAN9118_TX_FIFO_INF_TDFREE_MASK) >> LAN9118_TX_FIFO_INF_TDFREE_SHIFT;
        tx_fifo_inf  = LAN9118_SWAP16(tx_fifo_inf);
        if (tx_fifo_inf > (1514 + (MAX_ETH_DRV_SG * 10))) {
            TRACE(2, "yes");
            return 1;
        }
    }
    TRACE(3, "no");
    return 0;
}

// Transmit a packet. The data is in the form of an sg_list, which
// maps fairly well to the lan9118's multiple buffer support.
static void
lan9118_send(struct eth_drv_sc* sc,
                 struct eth_drv_sg* sg_list, int sg_len, int total_len,
                 unsigned long key)
{
    lan9118_instance*   lan9118 = (lan9118_instance*)sc->driver_private;
    int                 i, j;
    cyg_uint32          commandA, commandB;

    CYG_ASSERTC(total_len <= 1514);
    TRACEPKT(1, "start", sg_list[0].buf, total_len);

    lan_diag("len %d key %x\n",total_len, key);
    lan_dump_buf( sg_list[0].buf, 32 );
        
    // Prevent recursion problems during the tx_done() call.
    lan9118->lan9118_in_send_count    += 1;

    // commandB is invariant for the whole packet
    commandB    = (0    /* packet tag, not used */                              |
                   (LAN9118_SWAP16(total_len) << LAN9118_TX_COMMANDB_PACKET_LENGTH_SHIFT));
    
    for (i = 0; i < sg_len; i++) {
        int         len         = sg_list[i].len;
        cyg_uint32* buf32       = (cyg_uint32*)(((CYG_ADDRWORD)sg_list[i].buf) & ~0x03);
        int         start_pad   = (CYG_ADDRWORD)sg_list[i].buf - (CYG_ADDRWORD)buf32;

        CYG_LOOP_INVARIANT((i == 0) || (i == (sg_len - 1)) || (len >= 4), "h/w requires minimum buffer length");
        
        commandA    = (LAN9118_TX_COMMANDA_END_ALIGNMENT_4                          |
                       (start_pad << LAN9118_TX_COMMANDA_DATA_START_OFFSET_SHIFT)   |
                       ((i == 0) ? LAN9118_TX_COMMANDA_FIRST_SEGMENT : 0)           |
                       ((i == (sg_len - 1)) ? LAN9118_TX_COMMANDA_LAST_SEGMENT : 0) |
                       (LAN9118_SWAP16(len) << LAN9118_TX_COMMANDA_BUFFER_SIZE_SHIFT));

        WRITE( lan9118, LAN9118_TX_DATA_FIFO, commandA);
        WRITE( lan9118, LAN9118_TX_DATA_FIFO, commandB);

        len = (len + start_pad + 3) >> 2;
#ifndef HAL_LAN9118_SWAP_DATA
        // The bus is set up so that no swapping is needed.
        for (j = 0; j < len; j++) {
            WRITE(lan9118, LAN9118_TX_DATA_FIFO, *buf32++);
        }
#else
# error Data byte swapping not yet implemented.        
#endif        
    }
    
    // Tell higher-level code the packet is on its way. This may result
    // in a recursive call.
    (*sc->funs->eth_drv->tx_done)(sc, key, 1);
    INCR_STAT(lan9118, lan9118_tx_count);

    // All done.
    lan9118->lan9118_in_send_count    -= 1;
}

// Receive a known-good packet.
#if ((CYG_BYTEORDER == CYG_LSBFIRST) && !defined(HAL_LAN9118_SWAP_DATA)) || ((CYG_BYTEORDER == CYG_MSBFIRST) && defined(HAL_LAN9118_SWAP_DATA))
# define BYTE0(_data_)      (_data_ >>  0)
# define BYTE1(_data_)      (_data_ >>  8)
# define BYTE2(_data_)      (_data_ >> 16)
# define BYTE3(_data_)      (_data_ >> 24)
#else
# define BYTE3(_data_)      (_data_ >>  0)
# define BYTE2(_data_)      (_data_ >>  8)
# define BYTE1(_data_)      (_data_ >> 16)
# define BYTE0(_data_)      (_data_ >> 24)
#endif

static void
lan9118_recv(struct eth_drv_sc* sc, struct eth_drv_sg* sg_list, int sg_len)
{
    lan9118_instance*   lan9118 = (lan9118_instance*) sc->driver_private;
    cyg_uint32          fifo_align;
    int                 i, j;
    cyg_uint32          data = 0;

    TRACE(1, "packet in fifo\n");
    INCR_STAT(lan9118, lan9118_rx_good);

    lan_diag("sg_len %d\n", sg_len);
    
    // Fifo alignment varies with the TCP/IP stack in use, to allow for
    // maximum use of int operations.
    fifo_align = RX_INITIAL_PADDING;
#if (RX_INITIAL_PADDING > 0)
    READ(lan9118, LAN9118_RX_DATA_FIFO, data);
#endif    
    
    for (i = 0; i < sg_len; i++) {
        int         len         = (int) sg_list[i].len;
        cyg_uint8*  buf8        = (cyg_uint8*) sg_list[i].buf;
        cyg_uint32  buf_align   = ((CYG_ADDRWORD)buf8) & 0x03;

        if (0 == buf8) {
            // Higher-level code ran out of memory, discard the rest of the packet
            for (i = 0; i < ((len + 3) >> 2); i++) {
                READ(lan9118, LAN9118_RX_DATA_FIFO, data);
            }
            break;
        }
        
#ifndef HAL_LAN9118_SWAP_DATA
        if (buf_align == fifo_align) {
            // Most of the data can be handled by integer copies.
            // The start and end may need special attention.
            cyg_uint32* buf32 = (cyg_uint32*)(sg_list[i].buf & ~0x03);
            if (1 == fifo_align) {
                // There are three bytes at the start of the buffer
                // which can be filled in using data from the previous
                // iteration.
                *buf8++ = BYTE1(data);
                *buf8++ = BYTE2(data);
                *buf8   = BYTE3(data);
                buf32 += 1;
                len   -= 3;
            } else if (2 == fifo_align) {
                *buf8++ = BYTE2(data);
                *buf8   = BYTE3(data);
                buf32  += 1;
                len    -= 2;
            } else if (3 == fifo_align) {
                buf32 += 1;
                *buf8   = BYTE3(data);
                buf32  += 1;
                len    -= 1;
            }

            for (j = 0; j < (len >> 2); j++) {
                READ(lan9118, LAN9118_RX_DATA_FIFO, data);
                *buf32++ = data;
            }

            // How many extra bytes need to be filled in at the tail?
            len &= 0x03;
            if (0 != len) {
                buf8    = (cyg_uint8*)buf32;
                READ(lan9118, LAN9118_RX_DATA_FIFO, data);
                if (1 == len) {
                    *buf8   = BYTE0(data);
                } else if (2 == len) {
                    *buf8++ = BYTE0(data);
                    *buf8   = BYTE1(data);
                } else {
                    *buf8++ = BYTE0(data);
                    *buf8++ = BYTE1(data);
                    *buf8   = BYTE2(data);
                }
            }
            fifo_align = len;
        } else
#endif            
        {
            // The data is not nicely aligned or needs swapping.
            // Copying the packet from the fifos to memory is going to
            // involve either some serious shifting and masking, or
            // byte operations. It is not clear which would have the
            // better performance on any given platform. The shift
            // approach would have to cope with 12 misaligned
            // combinations which is a bit too painful. For
            // performance we have separate loops for the four fifo
            // align cases.
            switch(fifo_align) {
              case 0:
                {
                    // 0 of the required bytes already read
                    for (j = 0; j < (len >> 2); j++, buf8 += 4) {
                        READ(lan9118, LAN9118_RX_DATA_FIFO, data);
                        buf8[0] = BYTE0(data);
                        buf8[1] = BYTE1(data);
                        buf8[2] = BYTE2(data);
                        buf8[3] = BYTE3(data);
                    }
                    len &= 0x03;
                    if (len) {
                        READ(lan9118, LAN9118_RX_DATA_FIFO, data);
                        if (0x01 == len) {
                            *buf8       = BYTE0(data);
                            fifo_align  = 1;
                        } else if (0x02 == len) {
                            *buf8++     = BYTE0(data);
                            *buf8       = BYTE1(data);
                            fifo_align  = 2;
                        } else {
                            *buf8++     = BYTE0(data);
                            *buf8++     = BYTE1(data);
                            *buf8       = BYTE2(data);
                            fifo_align  = 3;
                        }
                    }
                    break;
                }
                
              case 1:
                {
                    // Three of the required bytes have already been
                    // read and remain in "data".
                    for (j = 0; j < (len >> 2); j++, buf8 += 4) {
                        buf8[0] = BYTE1(data);
                        buf8[1] = BYTE2(data);
                        buf8[2] = BYTE3(data);
                        READ(lan9118, LAN9118_RX_DATA_FIFO, data);
                        buf8[3] = BYTE0(data);
                    }
                    len &= 0x03;
                    if (len) {
                        if (0x01 == len) {
                            *buf8       = BYTE1(data);
                            fifo_align  = 2;
                        } else if (0x02 == len) {
                            *buf8++     = BYTE1(data);
                            *buf8       = BYTE2(data);
                            fifo_align  = 3;
                        } else {
                            *buf8++     = BYTE1(data);
                            *buf8++     = BYTE2(data);
                            *buf8       = BYTE3(data);
                            fifo_align  = 0;
                        }
                    }
                    break;
                }
              case 2:
                {
                    // Two bytes have already been read.
                    for (j = 0; j < (len >> 2); j++, buf8 += 4) {
                        buf8[0] = BYTE2(data);
                        buf8[1] = BYTE3(data);
                        READ(lan9118, LAN9118_RX_DATA_FIFO, data);
                        buf8[2] = BYTE0(data);
                        buf8[3] = BYTE1(data);
                    }
                    len &= 0x03;
                    if (len) {
                        if (0x01 == len) {
                            *buf8       = BYTE2(data);
                            fifo_align  = 3;
                        } else if (0x02 == len) {
                            *buf8++     = BYTE2(data);
                            *buf8       = BYTE3(data);
                            fifo_align  = 0;
                        } else {
                            *buf8++     = BYTE2(data);
                            *buf8++     = BYTE3(data);
                            READ(lan9118, LAN9118_RX_DATA_FIFO, data);
                            *buf8       = BYTE0(data);
                            fifo_align  = 1;
                        }
                    }
                    break;
                }
              case 3:
                {
                    // One byte has already been read.
                    for (j = 0; j < (len >> 2); j++, buf8 += 4) {
                        buf8[0] = BYTE3(data);
                        READ(lan9118, LAN9118_RX_DATA_FIFO, data);
                        buf8[1] = BYTE0(data);
                        buf8[2] = BYTE1(data);
                        buf8[3] = BYTE2(data);
                    }
                    len &= 0x03;
                    if (len) {
                        if (0x01 == len) {
                            *buf8       = BYTE3(data);
                            fifo_align  = 0;
                        } else if (0x02 == len) {
                            *buf8++     = BYTE3(data);
                            READ(lan9118, LAN9118_RX_DATA_FIFO, data);
                            *buf8       = BYTE0(data);
                            fifo_align  = 1;
                        } else {
                            *buf8++     = BYTE3(data);
                            READ(lan9118, LAN9118_RX_DATA_FIFO, data);
                            *buf8++     = BYTE0(data);
                            *buf8       = BYTE1(data);
                            fifo_align  = 2;
                        }
                    }
                    break;
                }
            }   // switch
        }   // else (misaligned)
    }   // foreach sg

    // Read and discard the ethernet CRC. The LAN9118 treats this as
    // part of the packet but eCos is not interested in it.
    READ(lan9118, LAN9118_RX_DATA_FIFO, data);

    TRACEPKT(1, "packet received\n", sg_list[0].buf, sg_list[0].len);
    lan9118 = lan9118;

    lan_diag("packet:\n",0);
    lan_dump_buf( sg_list[0].buf, 32 );
}

// deliver() is used for both polled and interrupt mode when there
// are one more rx_status words pending in the fifo. 
static void
lan9118_deliver(struct eth_drv_sc* sc)
{
    lan9118_instance*   lan9118 = (lan9118_instance*) sc->driver_private;
    cyg_uint32          rx_fifo_inf;
    cyg_uint32          rx_status;
    cyg_uint32          irq_cfg;
    int                 i, j;

    TRACE(1, "entry");

    READ(lan9118, LAN9118_RX_FIFO_INF, rx_fifo_inf);
    lan_diag("rx_fifo_inf %08x\n", rx_fifo_inf );    
    do {
        rx_fifo_inf = (rx_fifo_inf & LAN9118_RX_FIFO_INF_RXSUSED_MASK) >> LAN9118_RX_FIFO_INF_RXSUSED_SHIFT;
        for (i = rx_fifo_inf; i > 0 ; i--) {
            cyg_uint32  len;

            TRACE(1, "  rx ready");
            INCR_STAT(lan9118, lan9118_rx_count);
            READ(lan9118, LAN9118_RX_STATUS_FIFO, rx_status);
            len = (rx_status & LAN9118_RX_STATUS_PACKET_LENGTH_MASK) >> LAN9118_RX_STATUS_PACKET_LENGTH_SHIFT;
            len = LAN9118_SWAP16(len);
            if (0 == (rx_status & (LAN9118_RX_STATUS_FILTERING_FAIL             |
                                   LAN9118_RX_STATUS_ERROR                      |
                                   LAN9118_RX_STATUS_LENGTH_ERROR               |
                                   LAN9118_RX_STATUS_RUNT_FRAME                 |
                                   LAN9118_RX_STATUS_FRAME_TOO_LONG             |
                                   LAN9118_RX_STATUS_COLLISION_SEEN             |
                                   LAN9118_RX_STATUS_RECEIVE_WATCHDOG_TIMEOUT   |
                                   LAN9118_RX_STATUS_MII_ERROR                  |
                                                      LAN9118_RX_STATUS_CRC))) {
                // This appears to be a valid packet. The length includes the ethernet
                // CRC which we are not interested in.
                len -= 4;
                (*sc->funs->eth_drv->recv)(sc, len);
            } else {
                // An invalid packet. Discard it
                len += RX_INITIAL_PADDING;
                if (len < 16) {
                    // FFWD can only be used if there are at least 4
                    // dwords in the fifo.
                    for (j = 0; j < ((len + 3) >> 2); j++) {
                        cyg_uint32  discard;
                        READ(lan9118, LAN9118_RX_DATA_FIFO, discard);
                    }
                } else {
                    // Use the FFWD bit to remove an entire packet. This
                    // leaves the fifo unusable for a bit.
                    cyg_uint32  rx_dp_ctl;
                    WRITE(lan9118, LAN9118_RX_DP_CTL, LAN9118_RX_DP_CTRL_RX_FFWD);
                    WAIT1(lan9118);
                    do {
                        READ(lan9118, LAN9118_RX_DP_CTL, rx_dp_ctl);
                    } while (rx_dp_ctl & LAN9118_RX_DP_CTRL_RX_FFWD);
                }
#ifdef CYGFUN_DEVS_ETH_SMSC_LAN9118_STATISTICS
                if (rx_status & LAN9118_RX_STATUS_LENGTH_ERROR) {
                    // Is this reasonable?
                    lan9118->lan9118_rx_align_errors    += 1;
                }
                if (rx_status & LAN9118_RX_STATUS_RUNT_FRAME) {
                    lan9118->lan9118_rx_short_frames    += 1;
                }
                if (rx_status & (LAN9118_RX_STATUS_FRAME_TOO_LONG | LAN9118_RX_STATUS_RECEIVE_WATCHDOG_TIMEOUT)) {
                    lan9118->lan9118_rx_too_long_frames  += 1;
                }
                if (rx_status & LAN9118_RX_STATUS_COLLISION_SEEN) {
                    lan9118->lan9118_rx_collisions  += 1;
                }
                if (rx_status & LAN9118_RX_STATUS_MII_ERROR) {
                    // Does this map to anything?
                }
                if (rx_status & LAN9118_RX_STATUS_CRC) {
                    lan9118->lan9118_rx_crc_errors  += 1;
                }
#endif            
            }
        }
        // Clear the interrupt and re-enable interrupts. It should
        // now be safe, but to allow for race conditions we check
        // the fifo status again.
        WRITE(lan9118, LAN9118_INT_STS, LAN9118_INT_STS_RSFL);
        WAIT2(lan9118);
        READ(lan9118, LAN9118_IRQ_CFG, irq_cfg);
        WRITE(lan9118, LAN9118_IRQ_CFG, irq_cfg | LAN9118_IRQ_CFG_IRQ_EN);
        WAIT3(lan9118);
        READ(lan9118, LAN9118_RX_FIFO_INF, rx_fifo_inf);
    } while(rx_fifo_inf & LAN9118_RX_FIFO_INF_RXSUSED_MASK);
    
    lan9118 = lan9118;
}

#ifdef CYGFUN_DEVS_ETH_SMSC_LAN9118_STATISTICS
// A utility shared by the dsr() and poll() for processing
// tx status words, so there is no need to duplicate the
// statics code.
static void
lan9118_handle_tx_status(lan9118_instance* lan9118, cyg_uint32 tx_fifo_inf)
{
    int         i;
    cyg_uint32  tx_status;

    do {
        tx_fifo_inf >>= LAN9118_TX_FIFO_INF_TXSUSED_SHIFT;
        for (i = 0; i < tx_fifo_inf; i++) {
            READ(lan9118, LAN9118_TX_STATUS_FIFO, tx_status);
            if (0 == (tx_status & (LAN9118_TX_STATUS_ERROR                  |
                                   LAN9118_TX_STATUS_LOSS_OF_CARRIER        |
                                   LAN9118_TX_STATUS_NO_CARRIER             |
                                   LAN9118_TX_STATUS_LATE_COLLISION         |
                                   LAN9118_TX_STATUS_EXCESSIVE_COLLISIONS   |
                                   LAN9118_TX_STATUS_UNDERRUN))) {
                INCR_STAT(lan9118, lan9118_tx_good);
            }
            if (tx_status & (LAN9118_TX_STATUS_LOSS_OF_CARRIER || LAN9118_TX_STATUS_NO_CARRIER)) {
                INCR_STAT(lan9118, lan9118_tx_carrier_loss);
            }
            if (tx_status & LAN9118_TX_STATUS_LATE_COLLISION) {
                INCR_STAT(lan9118, lan9118_tx_late_collisions);
            }
            if (tx_status & LAN9118_TX_STATUS_EXCESSIVE_COLLISIONS) {
                lan9118->lan9118_tx_max_collisions       = 16;
                INCR_STAT(lan9118, lan9118_tx_total_collisions);
            } else {
                cyg_uint32  collisions = (tx_status & LAN9118_TX_STATUS_COLLISION_COUNT_MASK) >> LAN9118_TX_STATUS_COLLISION_COUNT_SHIFT;
                if (collisions > lan9118->lan9118_tx_max_collisions) {
                    INCR_STAT(lan9118, lan9118_tx_max_collisions);
                }
                if (1 == collisions) {
                    INCR_STAT(lan9118, lan9118_tx_total_collisions);
                    INCR_STAT(lan9118, lan9118_tx_single_collisions);
                } else if (1 < collisions) {
                    INCR_STAT(lan9118, lan9118_tx_total_collisions);
                    INCR_STAT(lan9118, lan9118_tx_mult_collisions);
                }
            }
            if (tx_status & LAN9118_TX_STATUS_UNDERRUN) {
                INCR_STAT(lan9118, lan9118_tx_underrun);
            }
            if (tx_status & LAN9118_TX_STATUS_DEFERRED) {
                INCR_STAT(lan9118, lan9118_tx_deferred);
            }
        }
        // Clear the interrupt.
        WRITE(lan9118, LAN9118_INT_STS, LAN9118_INT_STS_TSFL);
        WAIT2(lan9118);
        // And check that no new status words have arrived,
        READ(lan9118, LAN9118_TX_FIFO_INF, tx_fifo_inf);
        tx_fifo_inf &= LAN9118_TX_FIFO_INF_TXSUSED_MASK;
    } while (tx_fifo_inf);
}
#endif        

// For polling, process tx status words immediately. In
// interrupt-driven mode we want to handle those in the DSR so context
// switches only occur when really needed, and poll() has to be
// consistent. rx status words can be passed straight to deliver().
static void
lan9118_poll(struct eth_drv_sc* sc)
{
    lan9118_instance*   lan9118 = (lan9118_instance*) sc->driver_private;
    cyg_uint32  rx_fifo_inf;
    
    TRACE(2, "entry");
    
#ifdef CYGFUN_DEVS_ETH_SMSC_LAN9118_STATISTICS
    {
        cyg_uint32  tx_fifo_inf;
        READ(lan9118, LAN9118_TX_FIFO_INF, tx_fifo_inf);
        tx_fifo_inf &= LAN9118_TX_FIFO_INF_TXSUSED_MASK;
        if ( tx_fifo_inf ) {
            lan9118_handle_tx_status(lan9118, tx_fifo_inf);
        }
    }
#endif    
    
    READ(lan9118, LAN9118_RX_FIFO_INF, rx_fifo_inf);
    rx_fifo_inf &= LAN9118_RX_FIFO_INF_RXSUSED_MASK;
    if (rx_fifo_inf) {
        lan9118_deliver(sc);
    }

    // poll() may get called while the application is running, via
    // hal_default_isr() rather than lan9118_isr(), but the interrupt
    // still needs to be cleared.
    HAL_INTERRUPT_ACKNOWLEDGE(ISRVEC(lan9118));
    
    lan9118 = lan9118;
}

// Don't try to do anything in the ISR. Just mask the interrupt
// via the IRQ_EN bit.
static cyg_uint32
lan9118_isr(cyg_vector_t vec, cyg_addrword_t data)
{
    struct eth_drv_sc*  sc      = (struct eth_drv_sc*)data;
    lan9118_instance*   lan9118 = (lan9118_instance*) sc->driver_private;
    cyg_uint32  irq_cfg;

    INCR_STAT(lan9118, lan9118_interrupts);
    READ(lan9118, LAN9118_IRQ_CFG, irq_cfg);
//    lan_diag("irq_cfg %08x\n",irq_cfg );
    WRITE(lan9118, LAN9118_IRQ_CFG, irq_cfg & ~LAN9118_IRQ_CFG_IRQ_EN);
    WAIT3(lan9118);
    HAL_INTERRUPT_ACKNOWLEDGE(ISRVEC(lan9118));
    lan9118 = lan9118;
    return CYG_ISR_CALL_DSR|CYG_ISR_HANDLED;
}

// The deliver code should be called only if there are any
// pending rx status words. Tx can be handled here, saving
// the cost of a context switch.
static void
lan9118_dsr(cyg_vector_t vec, cyg_ucount32 count, cyg_addrword_t data)
{
    struct eth_drv_sc*  sc      = (struct eth_drv_sc*)data;
    lan9118_instance*   lan9118 = (lan9118_instance*) sc->driver_private;
    cyg_uint32          irq_cfg;
    cyg_uint32          rx_fifo_inf;

#ifdef CYGFUN_DEVS_ETH_SMSC_LAN9118_STATISTICS
    cyg_uint32          tx_fifo_inf;
    READ(lan9118, LAN9118_TX_FIFO_INF, tx_fifo_inf);
    tx_fifo_inf &= LAN9118_TX_FIFO_INF_TXSUSED_MASK;
    if (tx_fifo_inf) {
        lan9118_handle_tx_status(lan9118, tx_fifo_inf);
    }
#endif    

    READ(lan9118, LAN9118_RX_FIFO_INF, rx_fifo_inf);
//    lan_diag("rx_fifo_inf %08x\n", rx_fifo_inf );    
    if (rx_fifo_inf & LAN9118_RX_FIFO_INF_RXSUSED_MASK) {
        // This will cause deliver() to be called, and it will
        // re-enable interrupts.
        eth_drv_dsr(vec, count, data);
    } else {
        // Clear the rx interrupt bit anyway - in a mixed
        // polling/interrupt driven environment things can get
        // confused. Then re-enable interrupts.
        WRITE(lan9118, LAN9118_INT_STS, LAN9118_INT_STS_RSFL);
        WAIT2(lan9118);
        READ(lan9118, LAN9118_IRQ_CFG, irq_cfg);
        WRITE(lan9118, LAN9118_IRQ_CFG, irq_cfg | LAN9118_IRQ_CFG_IRQ_EN);
        WAIT3(lan9118);
    }
    lan9118 = lan9118;
}

static int
lan9118_int_vector(struct eth_drv_sc* sc)
{
    return ISRVEC((lan9118_instance*)sc->driver_private);
}

static int
lan9118_ioctl(struct eth_drv_sc* sc, unsigned long key, void* data, int data_length)
{
    lan9118_instance*   lan9118 = (lan9118_instance*)   sc->driver_private;
    
    TRACE(1, "entry");

    lan_diag("key %x\n", key);
    
    switch(key) {
      case ETH_DRV_SET_MAC_ADDRESS:
        {
            cyg_uint32  mac_addr;
            memcpy(lan9118->lan9118_mac, data, 6);
            mac_addr    = ((lan9118->lan9118_mac[5] << LAN9118_MAC_ADDRH_BYTE5_SHIFT) |
                           (lan9118->lan9118_mac[4] << LAN9118_MAC_ADDRH_BYTE4_SHIFT));
            cyg_lan9118_write_mac_reg( lan9118, LAN9118_MAC_ADDRH, mac_addr);
            mac_addr    = ((lan9118->lan9118_mac[3] << LAN9118_MAC_ADDRL_BYTE3_SHIFT) |
                           (lan9118->lan9118_mac[2] << LAN9118_MAC_ADDRL_BYTE2_SHIFT) |
                           (lan9118->lan9118_mac[1] << LAN9118_MAC_ADDRL_BYTE1_SHIFT) |
                           (lan9118->lan9118_mac[0] << LAN9118_MAC_ADDRL_BYTE0_SHIFT));
            cyg_lan9118_write_mac_reg( lan9118, LAN9118_MAC_ADDRL, mac_addr);
            return 0;
        }
      case ETH_DRV_SET_MC_ALL:
      case ETH_DRV_SET_MC_LIST:
        {
            cyg_uint32  mac_cr;
            mac_cr  = cyg_lan9118_read_mac_reg(lan9118, LAN9118_MAC_CR);
            cyg_lan9118_write_mac_reg(lan9118, LAN9118_MAC_CR, mac_cr | LAN9118_MAC_CR_MCPAS);
            return 0;
        }

#if defined(CYGFUN_DEVS_ETH_SMSC_LAN9118_STATISTICS) && defined(ETH_DRV_GET_IF_STATS_UD)
      case ETH_DRV_GET_IF_STATS_UD:
      case ETH_DRV_GET_IF_STATS:
        {
            struct ether_drv_stats* stats   = (struct ether_drv_stats*) data;
            cyg_uint32              phy_basic_ctrl;
            cyg_uint32              phy_basic_status;
            cyg_uint32              phy_special_control_status;

            strcpy(stats->description, "LAN9118 ethernet device");
            stats->snmp_chipset[0]  = '\0';
            phy_basic_status = cyg_lan9118_read_phy_reg(lan9118, LAN9118_PHY_BASIC_STATUS);
            if (phy_basic_status & LAN9118_PHY_BASIC_STATUS_LINK_STATUS) {
                stats->operational  = 3;
            } else {
                stats->operational  = 2;
            }
            phy_basic_ctrl   = cyg_lan9118_read_phy_reg(lan9118, LAN9118_PHY_BASIC_CTRL);
            if (phy_basic_ctrl & LAN9118_PHY_BASIC_CTRL_AUTO_ENABLE) {
                // auto-negotation enabled, check the
                // special_control_status register for the actual
                // speed setting.
                phy_special_control_status = cyg_lan9118_read_phy_reg(lan9118, LAN9118_PHY_SPECIAL_CONTROL_STATUS);
                phy_special_control_status &= LAN9118_PHY_SPECIAL_CONTROL_STATUS_SPEED_MASK;
                if ((phy_special_control_status == LAN9118_PHY_SPECIAL_CONTROL_STATUS_SPEED_10_FULL) ||
                    (phy_special_control_status == LAN9118_PHY_SPECIAL_CONTROL_STATUS_SPEED_100_FULL)) {
                    stats->duplex   = 3;
                } else {
                    stats->duplex   = 2;
                }
                if ((phy_special_control_status == LAN9118_PHY_SPECIAL_CONTROL_STATUS_SPEED_100_HALF) ||
                    (phy_special_control_status == LAN9118_PHY_SPECIAL_CONTROL_STATUS_SPEED_100_FULL)) {
                    stats->speed    = 100000000;
                } else {
                    stats->speed    = 10000000;
                }
            } else {
                // auto-negotation disabled, duplex and speed are indicated by
                // other bits in the basic ctrl register.
                if (phy_basic_ctrl & LAN9118_PHY_BASIC_CTRL_SPEED_SELECT) {
                    stats->speed    = 100000000;
                } else {
                    stats->speed    = 10000000;
                }
                if (phy_basic_ctrl & LAN9118_PHY_BASIC_CTRL_DUPLEX) {
                    stats->duplex   = 3;
                } else {
                    stats->duplex   = 2;
                }
            }
            
            stats->supports_dot3    = 1;
            stats->tx_queue_len     = 2;    // At least
            stats->tx_dropped       = 0;

            stats->interrupts           = lan9118->lan9118_interrupts;
            stats->tx_count             = lan9118->lan9118_tx_count;
            stats->tx_good              = lan9118->lan9118_tx_good;
            stats->tx_max_collisions    = lan9118->lan9118_tx_max_collisions;
            stats->tx_late_collisions   = lan9118->lan9118_tx_late_collisions;
            stats->tx_underrun          = lan9118->lan9118_tx_underrun;
            stats->tx_carrier_loss      = lan9118->lan9118_tx_carrier_loss;
            stats->tx_deferred          = lan9118->lan9118_tx_deferred;
            stats->tx_single_collisions = lan9118->lan9118_tx_single_collisions;
            stats->tx_mult_collisions   = lan9118->lan9118_tx_mult_collisions;
            stats->tx_total_collisions  = lan9118->lan9118_tx_total_collisions;
            stats->rx_count             = lan9118->lan9118_rx_count;
            stats->rx_good              = lan9118->lan9118_rx_good;
            stats->rx_crc_errors        = lan9118->lan9118_rx_crc_errors;
            stats->rx_align_errors      = lan9118->lan9118_rx_align_errors;
            stats->rx_overrun_errors    = lan9118->lan9118_rx_overrun_errors;
            stats->rx_collisions        = lan9118->lan9118_rx_collisions;
            stats->rx_short_frames      = lan9118->lan9118_rx_short_frames;
            stats->rx_too_long_frames   = lan9118->lan9118_rx_too_long_frames;

            return 0;
        }
#endif
    }
    return 1;
}

static void
lan9118_start(struct eth_drv_sc* sc, unsigned char* enaddr, int flags)
{
    lan9118_instance*   lan9118 = (lan9118_instance*)   sc->driver_private;
    cyg_uint32          phy_special_modes;
    cyg_uint32          mac_cr;
    cyg_uint32          irq_cfg;
    cyg_uint32          tx_cfg;

    TRACE(1, "entry");

    lan_diag("\n",0);
    
    // Allow the MAC to both send and receive packets. See full/half
    // duplex depending on the phy settings.
    phy_special_modes   = cyg_lan9118_read_phy_reg(lan9118, LAN9118_PHY_SPECIAL_MODES);
    mac_cr              = cyg_lan9118_read_mac_reg(lan9118, LAN9118_MAC_CR);
    phy_special_modes   &= LAN9118_PHY_SPECIAL_MODES_MODE_MASK;
    if ((phy_special_modes == LAN9118_PHY_SPECIAL_MODES_MODE_10BASET_FULL_NO_AUTO) ||
        (phy_special_modes == LAN9118_PHY_SPECIAL_MODES_MODE_100BASET_FULL_NO_AUTO)) {
        
        mac_cr &= ~(LAN9118_MAC_CR_RCVOWN);
        mac_cr |=  (LAN9118_MAC_CR_FDPX);
    } else {
        mac_cr |=  (LAN9118_MAC_CR_RCVOWN);
        mac_cr &= ~(LAN9118_MAC_CR_FDPX);
    }
    cyg_lan9118_write_mac_reg(lan9118, LAN9118_MAC_CR, mac_cr | (LAN9118_MAC_CR_RXEN | LAN9118_MAC_CR_TXEN));

    // Allow transmits.
    tx_cfg  = LAN9118_TX_CFG_TX_ON;
#ifndef CYGFUN_DEVS_ETH_SMSC_LAN9118_STATISTICS
    // The tx status words are only of interest if we are collecting
    // statistics.
    tx_cfg |= LAN9118_TX_CFG_TXSAO;
#endif    
    WRITE(lan9118, LAN9118_TX_CFG, tx_cfg);
    WAIT1(lan9118);

    // Packets can now come in and go out. Enable interrupts.
    READ(lan9118, LAN9118_IRQ_CFG, irq_cfg);
    WRITE(lan9118, LAN9118_IRQ_CFG, irq_cfg | LAN9118_IRQ_CFG_IRQ_EN);
    WAIT3(lan9118);

    // And finally allow can_send() to transmit packets.
    lan9118->lan9118_up = 1;
    TRACE(1, "done");
}

static void
lan9118_stop(struct eth_drv_sc* sc)
{
    lan9118_instance*   lan9118 = (lan9118_instance*)   sc->driver_private;
    cyg_uint32          mac_cr;
    cyg_uint32          irq_cfg;
    cyg_uint32          int_sts;
    int                 i;

    TRACE(1, "entry");
    lan_diag("\n",0);
    
    // Prevent any new transmissions, but don't bother to change the MAC
    // or fifos. All packets already queued are allowed to complete.
    lan9118->lan9118_up = 0;

    // Disable interrupts. IRQ_CFG is also manipulated by ISRs and DSRs,
    // but stop() should get called in thread context so any ISRs and DSRs
    // will have run to completion leaving IRQ_CFG unchanged.
    READ(lan9118, LAN9118_IRQ_CFG, irq_cfg);
    WRITE(lan9118, LAN9118_IRQ_CFG, irq_cfg & ~LAN9118_IRQ_CFG_IRQ_EN);
    WAIT3(lan9118);

    // Clear the RXSTOP_INT in the status register so that we can poll
    // it after updating MAC_CR.
    WRITE(lan9118, LAN9118_INT_STS, LAN9118_INT_STS_RXSTOP);
    WAIT2(lan9118);
    
    // Block new incoming packets in the MAC. Outgoing packets that are still
    // queued are allowed to go out, so no need to updata TX_CFG.
    mac_cr = cyg_lan9118_read_mac_reg(lan9118, LAN9118_MAC_CR);
    cyg_lan9118_write_mac_reg(lan9118, LAN9118_MAC_CR, mac_cr & ~LAN9118_MAC_CR_RXEN);

    // Wait for the new MAC setting to take effect, but not forever
    i = 0;
    do {
        READ(lan9118, LAN9118_INT_STS, int_sts);
    } while ((i++ < 64000) && !(int_sts & LAN9118_INT_STS_RXSTOP));
    
    // Clear out any incoming packets that are still queued.
    WRITE(lan9118, LAN9118_RX_CFG, LAN9118_RX_CFG_RX_DUMP);
    WAIT1(lan9118);
    // And set a data start offset appropriate for the configuration.
    if (0 != RX_INITIAL_PADDING) {
        WRITE(lan9118, LAN9118_RX_CFG, RX_INITIAL_PADDING << LAN9118_RX_CFG_RXDOFF_SHIFT);
        WAIT1(lan9118);
    }
    TRACE(1, "done");
}

bool
cyg_lan9118_eth_init(struct cyg_netdevtab_entry* ndp)
{
    struct eth_drv_sc*  sc      = (struct eth_drv_sc*) ndp->device_instance;
    lan9118_instance*   lan9118 = (lan9118_instance*)   sc->driver_private;
    cyg_uint32          pmt_ctrl;
    cyg_uint32          byte_test;
    cyg_uint32          irq_cfg;
    cyg_uint32          int_sts;
    cyg_uint32          old_mac_cr;
    cyg_uint32          mac_cr;
    cyg_uint32          gpio_cfg;
    cyg_uint32          e2p_cmd;
    cyg_uint32          mac_addr;
    int                 i;

    TRACE(1, "entry");
    lan_diag("base %p\n", BASE(lan9118));
    
    // At this stage the chip is not in a well-known state. It may
    // still be powering up, or it may be in a power management mode.
    // The only registers we are allowed to read are HW_CFG and
    // PMT_CTRL, and the only register we are allowed to write is
    // BYTE_TEST to get out of power management mode. Unfortunately
    // we cannot read or modify ENDIAN, which may be either in its
    // power-up state or already initialized from a previous run.
    // That means there is no sensible way to interpret the values
    // of HW_CFG or PMT_CTRL, especially the READY bit in the latter.
    //
    // We could poll PMT_CTRL for both bit 0 (READY) and bit 16
    // (reserved), thus coping with both endiannesses, breaking the
    // rule that reserved bits have undefined values. Or we can write
    // to ENDIAN anyway, on the assumption that this is safe. The
    // latter gives more sensible code. Also assume the BYTE_TEST can
    // be read at any time.
    READ(lan9118, LAN9118_PMT_CTRL, pmt_ctrl);
    WRITE(lan9118, LAN9118_BYTE_TEST, 0x00000000);
    WRITE(lan9118, LAN9118_ENDIAN,
          (lan9118->lan9118_hw_flags & LAN9118_HW_FLAGS_16BIT_BE) ? LAN9118_ENDIAN_BIG_MAGIC : LAN9118_ENDIAN_LITTLE_MAGIC);

    // A dummy read to allow the ENDIAN change to take effect, and
    // then a real read to make sure we are talking to the chip
    // correctly.
    READ(lan9118, LAN9118_BYTE_TEST, byte_test);
    READ(lan9118, LAN9118_BYTE_TEST, byte_test);
    lan_diag("byte_test %08x\n", byte_test);
    if (LAN9118_BYTE_TEST_MAGIC != byte_test) {
        return false;
    }

    // Next we loop until the PMT_CTRL READY bit is set, but not
    // quite forever.
    for (i = 0; !(pmt_ctrl & LAN9118_PMT_CTRL_READY) && (i < 10000000); i++) {
        READ(lan9118, LAN9118_PMT_CTRL, pmt_ctrl);
    }
    if (! (pmt_ctrl & LAN9118_PMT_CTRL_READY)) {
        return false;
    }

    // Now it is time to get the whole chip into a sane state.
    // There are two main scenarios:
    //
    // 1) stand-alone, i.e. !USE_ROM_MONITOR. We perform a full
    //    soft reset, including the phy. Anything else seems to
    //    cause trouble, e.g. spurious bytes appearing in the fifo.
    //
    // 2) application running on top of RedBoot, possibly with
    //    debugging over ethernet. There may still be outgoing
    //    debug-related packets which should not get discarded,
    //    and there may also be incoming packets. Interrupts
    //    are still disabled. TX activity is allowed to continue
    //    but rx activity is halted - a trade off between losing
    //    some debug-related messages and having lots of junk
    //    broadcasts stuck in the fifo. If debugging really is
    //    happening over ethernet then we can expect rx to be
    //    restarted very quickly so there is only a small window
    //    in which rx packets can get lost.

#ifndef CYGSEM_HAL_USE_ROM_MONITOR    
    {
        cyg_uint32  phy_basic_ctrl;
        cyg_uint32  hw_cfg;
        
        // Reset the phy
        cyg_lan9118_write_phy_reg(lan9118, LAN9118_PHY_BASIC_CTRL, LAN9118_PHY_BASIC_CTRL_RESET);
        i   = 100000;
        do {
            HAL_DELAY_US(10);
            phy_basic_ctrl  = cyg_lan9118_read_phy_reg(lan9118, LAN9118_PHY_BASIC_CTRL);
        } while (i-- && (phy_basic_ctrl & LAN9118_PHY_BASIC_CTRL_RESET));
        // As per SMSC example driver, the phy may not really have settled
        // down yet.
        HAL_DELAY_US(500);
        if( i == 0 )
            return false;
        
        // Autonegotiate speed
        phy_basic_ctrl  = cyg_lan9118_read_phy_reg(lan9118, LAN9118_PHY_BASIC_CTRL );
        phy_basic_ctrl  |= LAN9118_PHY_BASIC_CTRL_AUTO_ENABLE|LAN9118_PHY_BASIC_CTRL_RESTART_AUTO;
        cyg_lan9118_write_phy_reg(lan9118, LAN9118_PHY_BASIC_CTRL, phy_basic_ctrl );
        i   = 100000;
        do {
            HAL_DELAY_US(10);
            phy_basic_ctrl  = cyg_lan9118_read_phy_reg(lan9118, LAN9118_PHY_BASIC_STATUS);
        } while (i-- && ((phy_basic_ctrl & LAN9118_PHY_BASIC_STATUS_AUTO_COMPLETE) == 0));
        HAL_DELAY_US(500);
        if( i == 0 )
            return false;
        
        // And perform a soft reset
        WRITE(lan9118, LAN9118_HW_CFG, LAN9118_HW_CFG_SRST);
        WAIT1(lan9118);
        i = 100;
        do {
            HAL_DELAY_US(10);
            READ(lan9118, LAN9118_HW_CFG, hw_cfg);
        } while (i-- && (hw_cfg & LAN9118_HW_CFG_SRST) && !(hw_cfg & LAN9118_HW_CFG_SRST_TO));
        if (hw_cfg & LAN9118_HW_CFG_SRST_TO) {
            return false;
        }
    }
#endif

    // Check with the PHY that the link is actually up. If not then
    // return false, which should cause all ethernet support to be
    // disabled.
    {
        cyg_uint32  phy_basic_status;
        phy_basic_status  = cyg_lan9118_read_phy_reg(lan9118, LAN9118_PHY_BASIC_STATUS);

        if( (phy_basic_status & LAN9118_PHY_BASIC_STATUS_LINK_STATUS) == 0 )
            return false;
    }
    
    // The main interrupt config register. Disabling interrupts here
    // avoids surprises, and the polarity and type can be set as
    // required for the platform.
    irq_cfg = 0x00000000;
    if (lan9118->lan9118_hw_flags & LAN9118_HW_FLAGS_IRQ_POL_ACTIVE_HIGH) {
        irq_cfg |= LAN9118_IRQ_CFG_IRQ_POL;
    }
    if (lan9118->lan9118_hw_flags & LAN9118_HW_FLAGS_IRQ_PUSH_PULL) {
        irq_cfg |= LAN9118_IRQ_CFG_IRQ_TYPE;
    }
    WRITE(lan9118, LAN9118_IRQ_CFG, irq_cfg);
    WAIT3(lan9118);

    // Do not allow any new outgoing packets until start()
    lan9118->lan9118_up = 0;

    // Set up the main MAC ctrl register to sensible values, with
    // receives disabled. Transmits are stopped via the TX_CFG
    // register. If somehow RXEN was set before then a careful rx
    // shutdown process is needed.
    old_mac_cr  = cyg_lan9118_read_mac_reg(lan9118, LAN9118_MAC_CR);
    mac_cr  = LAN9118_MAC_CR_PADSTR | LAN9118_MAC_CR_TXEN;
    cyg_lan9118_write_mac_reg(lan9118, LAN9118_MAC_CR, mac_cr);
    // Wait for the stop to take effect.
    if (old_mac_cr & LAN9118_MAC_CR_RXEN) {
        WRITE(lan9118, LAN9118_INT_STS, LAN9118_INT_STS_RXSTOP);
        WAIT2(lan9118);
        i = 0;
        do {
            READ(lan9118, LAN9118_INT_STS, int_sts);
        } while ((i++ < 64000) && !(int_sts & LAN9118_INT_STS_RXSTOP));
    }
    // And discard any pending RX data and status. Set the initial
    // padding to maximise use of integer copies. Other RX_CFG
    // registers are left to their defaults.
    WRITE(lan9118, LAN9118_RX_CFG, LAN9118_RX_CFG_RX_DUMP);
    WAIT1(lan9118);
    if (0 != RX_INITIAL_PADDING) {
        WRITE(lan9118, LAN9118_RX_CFG, RX_INITIAL_PADDING << LAN9118_RX_CFG_RXDOFF_SHIFT);
        WAIT1(lan9118);
    }

    // Now for transmits. If USE_ROM_MONITOR we just leave this alone,
    // with new transmits blocked via the up flag. Otherwise we
    // perform a clean shutdown.
#ifndef CYGSEM_HAL_USE_ROM_MONITOR
    {
        cyg_uint32 tx_cfg;
        READ(lan9118, LAN9118_TX_CFG, tx_cfg);
        if (tx_cfg & LAN9118_TX_CFG_TX_ON) {
            WRITE(lan9118, LAN9118_INT_STS, LAN9118_INT_STS_TXSTOP);
            WAIT2(lan9118);
            WRITE(lan9118, LAN9118_TX_CFG, LAN9118_TX_CFG_STOP_TX);
            WAIT1(lan9118);
            i   = 0;
            do {
                READ(lan9118, LAN9118_INT_STS, int_sts);
            } while ((i++ < 64000) && !(int_sts & LAN9118_INT_STS_TXSTOP));
        }
        // The transmitter is stopped, clear out all pending data.
        WRITE(lan9118, LAN9118_TX_CFG, LAN9118_TX_CFG_TXS_DUMP | LAN9118_TX_CFG_TXD_DUMP);
        WAIT1(lan9118);
    }
#endif

    // Things should now be in a reasonably stable state.
    // Clear all pending interrupts
    WRITE(lan9118, LAN9118_INT_STS, 0xFFFFFFFF);
    WAIT2(lan9118);
    // The interrupts of interest are rx status fifo level and, if
    // we are collecting stats, tx status fifo level.
    //   SW_INT     - the driver does not use software interrupts
    //   TX_STOP    - the stop code does not actually halt the transmitter
    //   RX_STOP    - we don't worry about exactly when receives stop
    //   RXDFH      - don't care about dropped rx frames
    //   TX_IOC     - the IOC bit is not used
    //   RXD        - the driver does not try to process partial packets
    //   GPT        - the timer is not used
    //   PHY        - eCos does not currently support phy-level interrupts
    //   PME        - or power management
    //   TXSO       - unless collecting stats we don't carry about tx status at all
    //                If collecting stats, the transmitter will halt if tx status fifo is full
    //   RWT        - rx error conditions handled by main receive code, no need for a separate interrupt
    //   RXE        - ditto
    //   TXE        - ditto but for transmits
    //   TDFU       - impossible since we use store and forward mode
    //   TDFO       - the driver won't try to put too much data into the fifo
    //   TDFA       - could be used by a better can_send(), but does not fit the current eCos model
    //   TSFF       - we are already triggering on TSFL
    //   RXDF       - another error condition we don't need to worry about
    //   RDFL       - only complete incoming packets as per rx status is of interest
    //   RSFF       - we are already triggering on RSFL
    //   GPIOx      - not used by the driver.
#ifdef CYGFUN_DEVS_ETH_SMSC_LAN9118_STATISTICS
    WRITE(lan9118, LAN9118_INT_EN, LAN9118_INT_EN_TSFL | LAN9118_INT_EN_RSFL);
#else
    WRITE(lan9118, LAN9118_INT_EN, LAN9118_INT_EN_RSFL);
#endif    
    WAIT1(lan9118);
    // We want interrupts whenever there is a single entry in either
    // the RX or TX status fifos.
    WRITE(lan9118, LAN9118_FIFO_INT, (0x00FF << LAN9118_FIFO_INT_TX_AVAIL_SHIFT) );
    WAIT1(lan9118);

    // Enable store-and-forward mode since we cannot easily guarantee
    // that all TX data will be put into the fifo in time - at least,
    // not without potentially messing up real-time response
    // elsewhere. Set the TX fifo size to 4K, enough for at least two
    // full-size packets plus overheads plus status words. Without
    // DMA the ethernet chip will probably be fast enough that the cpu
    // cannot easily provide packets faster, so leave the resources
    // available for RX.
    WRITE(lan9118, LAN9118_HW_CFG, LAN9118_HW_CFG_SF + (4 << LAN9118_HW_CFG_TX_FIF_SZ_SHIFT));
    WAIT1(lan9118);

    // We know we are not in a power saving mode because that was taken
    // care of at the start of this function. Best leave the other
    // power management settings alone.

    // Enable LEDs
    gpio_cfg    = 0x00000000;
    if (lan9118->lan9118_hw_flags & LAN9118_HW_FLAGS_HAS_LED1) {
        gpio_cfg    |= LAN9118_GPIO_CFG_LED1_EN;
    }
    if (lan9118->lan9118_hw_flags & LAN9118_HW_FLAGS_HAS_LED2) {
        gpio_cfg    |= LAN9118_GPIO_CFG_LED2_EN;
    }
    if (lan9118->lan9118_hw_flags & LAN9118_HW_FLAGS_HAS_LED3) {
        gpio_cfg    |= LAN9118_GPIO_CFG_LED3_EN;
    }
    WRITE(lan9118, LAN9118_GPIO_CFG, gpio_cfg);
    WAIT1(lan9118);
    
    // Set flow control to default values. There is no simple
    // correlation between info available at the device driver
    // level and the optimal settings for any given environment.
    WRITE(lan9118, LAN9118_AFC_CFG, 0x00000000);
    WAIT1(lan9118);

    // Now for the MAC registers. MAC_CR has been filled in already.
    //
    // If the chip found an attached EEPROM and used that for the MAC
    // address, a bit will be set in E2P_CMD. Read that value and
    // store it in the lan9118 structure. Otherwise use the value
    // already in the lan9118 structure, which may have been provided
    // from e.g. an fconfig setting.
    READ(lan9118, LAN9118_E2P_CMD, e2p_cmd);
    
    if (e2p_cmd & LAN9118_E2P_CMD_MAC_ADDRESS_LOADED) {
        mac_addr = cyg_lan9118_read_mac_reg(lan9118, LAN9118_MAC_ADDRH);
        lan9118->lan9118_mac[5] = (mac_addr >>  LAN9118_MAC_ADDRH_BYTE5_SHIFT) & 0x0FF;
        lan9118->lan9118_mac[4] = (mac_addr >>  LAN9118_MAC_ADDRH_BYTE4_SHIFT) & 0x0FF;
        mac_addr = cyg_lan9118_read_mac_reg(lan9118, LAN9118_MAC_ADDRL);
        lan9118->lan9118_mac[3] = (mac_addr >>  LAN9118_MAC_ADDRL_BYTE3_SHIFT) & 0x0FF;
        lan9118->lan9118_mac[2] = (mac_addr >>  LAN9118_MAC_ADDRL_BYTE2_SHIFT) & 0x0FF;
        lan9118->lan9118_mac[1] = (mac_addr >>  LAN9118_MAC_ADDRL_BYTE1_SHIFT) & 0x0FF;
        lan9118->lan9118_mac[0] = (mac_addr >>  LAN9118_MAC_ADDRL_BYTE0_SHIFT) & 0x0FF;
    } else {
        mac_addr    = ((lan9118->lan9118_mac[5] << LAN9118_MAC_ADDRH_BYTE5_SHIFT) |
                       (lan9118->lan9118_mac[4] << LAN9118_MAC_ADDRH_BYTE4_SHIFT));
        cyg_lan9118_write_mac_reg( lan9118, LAN9118_MAC_ADDRH, mac_addr);
        mac_addr    = ((lan9118->lan9118_mac[3] << LAN9118_MAC_ADDRL_BYTE3_SHIFT) |
                       (lan9118->lan9118_mac[2] << LAN9118_MAC_ADDRL_BYTE2_SHIFT) |
                       (lan9118->lan9118_mac[1] << LAN9118_MAC_ADDRL_BYTE1_SHIFT) |
                       (lan9118->lan9118_mac[0] << LAN9118_MAC_ADDRL_BYTE0_SHIFT));
        cyg_lan9118_write_mac_reg( lan9118, LAN9118_MAC_ADDRL, mac_addr);
    }
    
    // Don't bother with the multicast hash table.
    // Leave the flow control settings alone.
    // Don't do anything with vlan or wake-up support.

    // Finally the phy.
    // FIXME: anything needed here? In particular, default to
    // autonegotiation?

    // Install an interrupt handler.
    cyg_drv_interrupt_create(ISRVEC(lan9118),
                             ISRPRI(lan9118),
                             (CYG_ADDRWORD)sc,
                             &lan9118_isr,
                             &lan9118_dsr,
                             &(lan9118->lan9118_isr_handle),
                             &(lan9118->lan9118_isr_data));
    cyg_drv_interrupt_attach(lan9118->lan9118_isr_handle);

    // Interrupts are masked via the IRQ_CFG_IRQ_EN bit, and that is
    // what used by the ISR and DSR, so it is safe to unmask
    // interrupts at the eCos level and leave them unmasked.
    cyg_drv_interrupt_unmask(ISRVEC(lan9118));
    
    // Initialize the upper level driver
    TRACE(1, "calling higher-level");
    (sc->funs->eth_drv->init)(sc, lan9118->lan9118_mac);

    TRACE(1, "done");
    return 1;
}

// ----------------------------------------------------------------------------
// The table of functions. This cannot be done via the usual ETH_DRV_SC macro
// because that cannot cope with the driver and the instantiation being in
// separate files.
struct eth_hwr_funs cyg_lan9118_eth_funs    = {
    .start          = &lan9118_start,
    .stop           = &lan9118_stop,
    .control        = &lan9118_ioctl,
    .can_send       = &lan9118_can_send,
    .send           = &lan9118_send,
    .recv           = &lan9118_recv,
    .deliver        = &lan9118_deliver,
    .poll           = &lan9118_poll,
    .int_vector     = &lan9118_int_vector,
    .eth_drv        = &eth_drv_funs,
    .eth_drv_old    = (struct eth_drv_funs*)0
};
