//==========================================================================
//
//      S29GL-N__aux.c
//
//      Flash driver for the SPANSION family - implementation. 
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2004, 2005, 2006, 2007, 2008 Free Software Foundation, Inc.
// Copyright (C) 2004, 2005, 2006, 2007, 2008 eCosCentric Limited           
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    bartv
// Contributors:
// Date:         2004-11-05
//              
//####DESCRIPTIONEND####
//
//==========================================================================

// This file is #include'd multiple times from the main am29xxxxx.c file,
// It serves to instantiate the various hardware operations in ways
// appropriate for all the bus configurations.
				 
// The following macros are used to construct suitable function names
// for the current bus configuration. S29_SUFFIX is #define'd before
// each #include of am29xxxxx_aux.c

#ifndef S29_STR
# define S29_STR1(_a_) # _a_
# define S29_STR(_a_) S29_STR1(_a_)
# define S29_CONCAT3_AUX(_a_, _b_, _c_) _a_##_b_##_c_
# define S29_CONCAT3(_a_, _b_, _c_) S29_CONCAT3_AUX(_a_, _b_, _c_)
#endif

#define S29_FNNAME(_base_) S29_CONCAT3(_base_, _,  S29_SUFFIX)

// Similarly construct a forward declaration, placing the function in
// the .2ram section. Each function must still be in a separate section
// for linker garbage collection.

# define S29_RAMFNDECL(_base_, _args_) \
  S29_FNNAME(_base_) _args_ __attribute__((section (".2ram." S29_STR(_base_) "_" S29_STR(S29_SUFFIX))))

// Calculate the various offsets, based on the device count.
// The main code may override these settings for specific
// configurations, e.g. 16as8
#ifndef S29_OFFSET_COMMAND
# define S29_OFFSET_COMMAND            0x0555
#endif
#ifndef S29_OFFSET_COMMAND2
# define S29_OFFSET_COMMAND2           0x02AA
#endif
#ifndef S29_OFFSET_MANUFACTURER_ID
# define S29_OFFSET_MANUFACTURER_ID    0x0000
#endif
#ifndef S29_OFFSET_DEVID
# define S29_OFFSET_DEVID              0x0001
#endif
#ifndef S29_OFFSET_DEVID2
# define S29_OFFSET_DEVID2             0x000E
#endif
#ifndef S29_OFFSET_DEVID3
# define S29_OFFSET_DEVID3             0x000F
#endif
#ifndef S29_OFFSET_CFI
# define S29_OFFSET_CFI                0x0055
#endif
#ifndef S29_OFFSET_CFI_DATA
# define S29_OFFSET_CFI_DATA(_idx_)    _idx_
#endif
#ifndef S29_OFFSET_AT49_LOCK_STATUS
# define S29_OFFSET_AT49_LOCK_STATUS   0x02
#endif

// For parallel operation commands are issued in parallel and status
// bits are checked in parallel.
#ifndef S29_PARALLEL
# define S29_PARALLEL(_cmd_)    (_cmd_)
#endif

// ----------------------------------------------------------------------------
// Diagnostic routines.

#if 0
#define spansion_diag( __fmt, ... ) diag_printf("SPANSION: %s[%d]: " __fmt, __FUNCTION__, __LINE__, ## __VA_ARGS__ );
#define spansion_dump_buf( __addr, __size ) diag_dump_buf( __addr, __size )
#else
#define spansion_diag( __fmt, ... )
#define spansion_dump_buf( __addr, __size )
#endif

// ----------------------------------------------------------------------------
// When performing the various low-level operations like erase the flash
// chip can no longer support ordinary data reads. Obviously this is a
// problem if the current code is executing out of flash. The solution is
// to store the key functions in RAM rather than flash, via a special
// linker section .2ram which usually gets placed in the same area as
// .data.
//
// In a ROM startup application anything in .2ram will consume space
// in both the flash and RAM. Hence it is desirable to keep the .2ram
// functions as small as possible, responsible only for the actual
// hardware manipulation.
//
// All these .2ram functions should be invoked with interrupts
// disabled. Depending on the hardware it may also be necessary to
// have the data cache disabled. The .2ram functions must be
// self-contained, even macro invocations like HAL_DELAY_US() are
// banned because on some platforms those could be implemented as
// function calls.

// gcc requires forward declarations with the attributes, then the actual
// definitions.
static int  S29_RAMFNDECL(s29_hw_query, (volatile S29_TYPE*));
static int  S29_RAMFNDECL(s29_hw_cfi, (struct cyg_flash_dev*, cyg_am29xxxxx_dev*, volatile S29_TYPE*));
static int  S29_RAMFNDECL(s29_hw_erase, (volatile S29_TYPE*, cyg_bool));
static int  S29_RAMFNDECL(s29_hw_program, (volatile S29_TYPE*, volatile S29_TYPE*, const cyg_uint8*, cyg_uint32 count, int retries));
static int  S29_RAMFNDECL(at49_hw_softlock,        (volatile S29_TYPE*));
static int  S29_RAMFNDECL(at49_hw_hardlock,        (volatile S29_TYPE*));
static int  S29_RAMFNDECL(at49_hw_unlock,          (volatile S29_TYPE*));


// ----------------------------------------------------------------------------

#ifdef CYGHWR_DEVS_FLASH_SPANSION_S29GLN_V2_RESET_NEEDS_RESUME
// With this flash component (e.g. AT49xxxxx), the reset does not
// cause a suspended erase/program to be aborted. Instead all we
// can do is resume any suspended operations. We do this on each
// block as some parts have different granularity.

static void
S29_FNNAME(s29_hw_force_all_suspended_resume)(struct cyg_flash_dev* dev, cyg_am29xxxxx_dev* s29_dev, volatile S29_TYPE* addr)
{
    cyg_ucount16 i,j;
    S29_TYPE datum1, datum2;

    S29_2RAM_ENTRY_HOOK();
    
    for (i=0; i<dev->num_block_infos; i++)
    {
        for (j=0; j<s29_dev->block_info[i].blocks; j++)
        {
            addr[S29_OFFSET_COMMAND] = S29_COMMAND_ERASE_RESUME;
            HAL_MEMORY_BARRIER();
            // We don't know if the suspended operation was an erase or
            // program, so just compare the whole word to spot _any_ toggling.
            do {
                datum1  = addr[S29_OFFSET_COMMAND];
                datum2  = addr[S29_OFFSET_COMMAND];
            } while (datum1 != datum2);

            addr += s29_dev->block_info[i].block_size/sizeof(S29_TYPE);
        }
    }
    
    S29_2RAM_EXIT_HOOK();
}
#endif // ifdef CYGHWR_DEVS_FLASH_SPANSION_S29GLN_V2_RESET_NEEDS_RESUME

// Read the device id. This involves a straightforward command
// sequence, followed by a reset to get back into array mode.
// All chips are accessed in parallel, but only the response
// from the least significant is used.
static int
S29_FNNAME(s29_hw_query)(volatile S29_TYPE* addr)
{
    int devid;
    cyg_uint32 onedevmask;

    S29_2RAM_ENTRY_HOOK();

    // Fortunately the compiler should optimise the below
    // tests such that onedevmask is a constant.
    if ( 1 == (sizeof(S29_TYPE) / S29_DEVCOUNT) )
        onedevmask = 0xFF;
    else if ( 2 == (sizeof(S29_TYPE) / S29_DEVCOUNT) )
        onedevmask = 0xFFFF;
    else {
        CYG_ASSERT( 4 == (sizeof(S29_TYPE) / S29_DEVCOUNT), 
                    "Unexpected flash width per device" );
        onedevmask = 0xFFFFFFFF;
    }
    
    addr[S29_OFFSET_COMMAND]   = S29_COMMAND_SETUP1;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND2]  = S29_COMMAND_SETUP2;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND]   = S29_COMMAND_AUTOSELECT;
    HAL_MEMORY_BARRIER();

    devid                       = S29_UNSWAP(addr[S29_OFFSET_DEVID]) & onedevmask;

//    spansion_diag("devid %x\n", devid );
//    spansion_dump_buf(addr, 64 );
    
    // The original SPANSION chips only used a single-byte device id, but
    // all codes have now been used up. Newer devices use a 3-byte
    // devid. The above devid read will have returned 0x007E. The
    // test allows for boards with a mixture of old and new chips.
    // The amount of code involved is too small to warrant a config
    // option.
    // FIXME by jifl: What happens when a single device is connected 16-bits
    // (or 32-bits) wide per device? Is the code still 0x7E, and all the
    // other devids are 8-bits only?
    if (0x007E == devid) {
        devid <<= 16;
        devid  |= ((S29_UNSWAP(addr[S29_OFFSET_DEVID2]) & 0x00FF) << 8);
        devid  |=  (S29_UNSWAP(addr[S29_OFFSET_DEVID3]) & 0x00FF);
    }
    addr[S29_OFFSET_COMMAND]   = S29_COMMAND_RESET;
    HAL_MEMORY_BARRIER();
    
//    spansion_diag("devid %x\n", devid );
    
    S29_2RAM_EXIT_HOOK();
    return devid;
}

// Perform a CFI query. This involves placing the device(s) into CFI
// mode, checking that this has really happened, and then reading the
// size and block info. The address corresponds to the start of the
// flash.
static int
S29_FNNAME(s29_hw_cfi)(struct cyg_flash_dev* dev, cyg_am29xxxxx_dev* s29_dev, volatile S29_TYPE* addr)
{
    int     dev_size;
    int     i;
    int     erase_regions;

    S29_2RAM_ENTRY_HOOK();
    
#ifdef CYGHWR_DEVS_FLASH_SPANSION_S29GLN_V2_CFI_BOGOSITY
    int     manufacturer_id;
    addr[S29_OFFSET_COMMAND]   = S29_COMMAND_SETUP1;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND2]  = S29_COMMAND_SETUP2;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND]   = S29_COMMAND_AUTOSELECT;
    HAL_MEMORY_BARRIER();

    manufacturer_id             = S29_UNSWAP(addr[S29_OFFSET_MANUFACTURER_ID]) & 0x00FF;
    addr[S29_OFFSET_COMMAND]   = S29_COMMAND_RESET;
    HAL_MEMORY_BARRIER();
#endif
    
    // Just a single write is needed to put the device into CFI mode
    addr[S29_OFFSET_CFI]   = S29_COMMAND_CFI;
    HAL_MEMORY_BARRIER();
//    spansion_diag("CFI data:\n");
//    spansion_dump_buf( addr, 256 );
    // Now check that we really are in CFI mode. There should be a 'Q'
    // at a specific address. This test is not 100% reliable, but should
    // be good enough.
    if ('Q' != (S29_UNSWAP(addr[S29_OFFSET_CFI_Q]) & 0x00FF)) {
        addr[S29_OFFSET_COMMAND]   = S29_COMMAND_RESET;
        HAL_MEMORY_BARRIER();
        S29_2RAM_EXIT_HOOK();
        return CYG_FLASH_ERR_PROTOCOL;
    }
    // Device sizes are always a power of 2, and the shift is encoded
    // in a single byte
    dev_size = 0x01 << (S29_UNSWAP(addr[S29_OFFSET_CFI_SIZE]) & 0x00FF);
    dev->end = dev->start + dev_size - 1;

    // The number of erase regions is also encoded in a single byte.
    // Usually this is no more than 4. A value of 0 indicates that
    // only chip erase is supported, but the driver does not cope
    // with that.
    erase_regions   = S29_UNSWAP(addr[S29_OFFSET_CFI_BLOCK_REGIONS]) & 0x00FF;
    if (erase_regions > CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_ERASE_REGIONS) {
        addr[S29_OFFSET_COMMAND]   = S29_COMMAND_RESET;
        HAL_MEMORY_BARRIER();
        S29_2RAM_EXIT_HOOK();
        return CYG_FLASH_ERR_PROTOCOL;
    }
    dev->num_block_infos    = erase_regions;

    for (i = 0; i < erase_regions; i++) {
        cyg_uint32 count, size;
        cyg_uint32 count_lsb   = S29_UNSWAP(addr[S29_OFFSET_CFI_BLOCK_COUNT_LSB(i)]) & 0x00FF;
        cyg_uint32 count_msb   = S29_UNSWAP(addr[S29_OFFSET_CFI_BLOCK_COUNT_MSB(i)]) & 0x00FF;
        cyg_uint32 size_lsb    = S29_UNSWAP(addr[S29_OFFSET_CFI_BLOCK_SIZE_LSB(i)]) & 0x00FF;
        cyg_uint32 size_msb    = S29_UNSWAP(addr[S29_OFFSET_CFI_BLOCK_SIZE_MSB(i)]) & 0x00FF;

        count = ((count_msb << 8) | count_lsb) + 1;
        size  = (size_msb << 16) | (size_lsb << 8);
        s29_dev->block_info[i].block_size  = (size_t) size * S29_DEVCOUNT;
        s29_dev->block_info[i].blocks      = count;
    }

#ifdef CYGHWR_DEVS_FLASH_SPANSION_S29GLN_V2_CFI_BOGOSITY

    // Some flash parts have a peculiar implementation of CFI. The
    // device erase regions may not be in the order specified in the
    // main CFI area. Instead the erase regions are given in a
    // manufacturer dependent fixed order, regardless of whether this
    // is a top or bottom boot block device. A vendor-specific
    // extended query block has an entry saying whether the boot
    // blocks are at the top or bottom. This code works out whether
    // the erase regions appear to be specified in the wrong order,
    // and then swaps them over.

    {
        enum { bottom, symmetric, top } boot_type = symmetric;
        cyg_uint32 vspec = S29_SWAP(addr[S29_OFFSET_CFI_DATA(0x15)]) & 0x00FF;

        // Take a look at the vendor specific area for the boot block
        // order.
        
        switch( manufacturer_id )
        {
            // Atmel appear to have their own layout for the vendor
            // specific area. Offset 0x06 of the vendor specific area
            // contains a single bit: 0x00 = top boot, 0x01 = bottom
            // boot. There appears to be no way of specifying
            // symmetric formats.
        case 0x1F:
            if( (addr[S29_OFFSET_CFI_DATA(vspec+0x06)] & S29_SWAP(0x1)) == S29_SWAP(0x1) )
                boot_type = bottom;
            else boot_type = top;
            break;

            // Most other manufacturers seem to follow the same layout
            // and encoding. Offset 0xF of the vendor specific area
            // contains the boot sector layout: 0x00 = uniform, 0x01 =
            // 8x8k top and bottom, 0x02 = bottom boot, 0x03 = top
            // boot, 0x04 = both top and bottom.
            //
            // The following manufacturers support this layout:
            // SPANSION, Spansion, ST, Macronix.
        default:
            if( (addr[S29_OFFSET_CFI_DATA(vspec+0xF)] == S29_SWAP(0x2)) )
                boot_type = bottom;                
            else if( (addr[S29_OFFSET_CFI_DATA(vspec+0xF)] == S29_SWAP(0x3)) )
                boot_type = top;
            // All other options are symmetric
            break;
        }

        // If the device is marked as top boot, but the erase region
        // list appears to be in bottom boot order, then reverse the
        // list. Also swap, if it is marked as bottom boot but the
        // erase regions appear to be in top boot order. This code
        // assumes that the first boot block is always smaller than
        // regular blocks; it is possible to imagine flash layouts for
        // which that is not true.
        
        if( ((boot_type == top) &&
             (s29_dev->block_info[0].block_size < s29_dev->block_info[erase_regions-1].block_size)) ||
            ((boot_type == bottom) &&
             (s29_dev->block_info[0].block_size > s29_dev->block_info[erase_regions-1].block_size)))
        {
            int lo, hi;

            for( lo = 0, hi = erase_regions-1 ; lo < hi ; lo++, hi-- )
            {
                size_t size                          = s29_dev->block_info[lo].block_size;
                cyg_uint32 count                     = s29_dev->block_info[lo].blocks;
                s29_dev->block_info[lo].block_size  = s29_dev->block_info[hi].block_size;
                s29_dev->block_info[lo].blocks      = s29_dev->block_info[hi].blocks;
                s29_dev->block_info[hi].block_size  = size;
                s29_dev->block_info[hi].blocks      = count;
            }
        }
    }
#endif
        
    // Get out of CFI mode
    addr[S29_OFFSET_COMMAND]   = S29_COMMAND_RESET;
    HAL_MEMORY_BARRIER();

    S29_2RAM_EXIT_HOOK();
    return CYG_FLASH_ERR_OK;
}

// Erase a single sector, or resume an earlier erase. There is no API
// support for chip-erase. The generic code operates one sector at a
// time, invoking the driver for each sector, so there is no
// opportunity inside the driver for erasing multiple sectors in a
// single call. The address argument points at the start of the
// sector.
static int
S29_FNNAME(s29_hw_erase)(volatile S29_TYPE* addr, cyg_bool first_call)
{
    int         retries;
    S29_TYPE   datum;

    S29_2RAM_ENTRY_HOOK();
    
    if (first_call) {
        // Start the erase operation
        addr[S29_OFFSET_COMMAND]   = S29_COMMAND_SETUP1;
        HAL_MEMORY_BARRIER();
        addr[S29_OFFSET_COMMAND2]  = S29_COMMAND_SETUP2;
        HAL_MEMORY_BARRIER();
        addr[S29_OFFSET_COMMAND]   = S29_COMMAND_ERASE;
        HAL_MEMORY_BARRIER();
        addr[S29_OFFSET_COMMAND]   = S29_COMMAND_SETUP1;
        HAL_MEMORY_BARRIER();
        addr[S29_OFFSET_COMMAND2]  = S29_COMMAND_SETUP2;
        HAL_MEMORY_BARRIER();
        addr[S29_OFFSET_COMMAND]   = S29_COMMAND_ERASE_SECTOR;
        HAL_MEMORY_BARRIER();
        // There is now a 50us window in which we could send additional
        // ERASE_SECTOR commands, but the driver API does not allow this
    } else {
        // Resume an earlier erase. The block should currently
        // be in erase-suspend mode. Just a single write is needed.
        addr[S29_OFFSET_COMMAND]   = S29_COMMAND_ERASE_RESUME;
        HAL_MEMORY_BARRIER();
    }

    // All chips are now erasing in parallel. Loop until all have
    // completed. This can be detected in a number of ways. The DQ7
    // bit will be 0 until the erase is complete, but there is a
    // problem if something went wrong (e.g. the sector is locked),
    // the erase has not actually started, and the relevant bit was 0
    // already. More useful is DQ6. This will toggle during the 50us
    // window and while the erase is in progress, then stop toggling.
    // If the erase does not actually start then the bit won't toggle
    // at all so the operation completes rather quickly.
    //
    // If at any time DQ5 is set (indicating a timeout inside the
    // chip) then a reset command must be issued and the erase is
    // aborted. It is not clear this can actually happen during an
    // erase, but just in case.
    for (retries = CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_ERASE_BURST_DURATION;
         retries > 0;
         retries--) {
        
        datum  = addr[S29_OFFSET_COMMAND];
        // The operation completes when all DQ7 bits are set.
        if ((datum & S29_STATUS_DQ7) == S29_STATUS_DQ7) {
            break;
        }
        // Otherwise, for any flash chips where DQ7 is still clear, it is
        // necessary to test DQ5.
        if (((datum ^ S29_STATUS_DQ7) >> 2) & datum & S29_STATUS_DQ5) {
            // DQ5 is set, indicating a hardware error. The calling code
            // will always verify that the erase really was successful
            // so we do not need to distinguish between error conditions.
            addr[S29_OFFSET_COMMAND] = S29_COMMAND_RESET;
            HAL_MEMORY_BARRIER();
            break;
        }
    }

#if (CYGNUM_DEVS_FLASH_AMD_AM29XXXXX_V2_ERASE_BURST_DURATION < CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_ERASE_TIMEOUT)
    if (retries == 0) {
        int suspend_retries;
        
        // We have timed out for now, so suspend the erase operation.
        // First write the suspend command
        addr[S29_OFFSET_COMMAND]   = S29_COMMAND_ERASE_SUSPEND;
        HAL_MEMORY_BARRIER();
        // The suspend does not take immediate effect, it make take
        // 20us or so. DQ7 will be 0 while the erase is in progress,
        // 1 when the suspend has taken effect.
        for (suspend_retries = CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_ERASE_BURST_DURATION;
             suspend_retries > 0;
             suspend_retries--) {
            S29_TYPE   datum;
            datum = addr[S29_OFFSET_COMMAND];
            if ((datum & S29_STATUS_DQ7) == S29_STATUS_DQ7) {
                break;
            }
            // Possibly check DQ5 here. It is not clear that is
            // necessary.
        }
        // If suspend_retries == 0 then things are messed up. The
        // block should be in erase-suspend mode, but is still being
        // erased. For now ignore this. Possibly we should be
        // returning a non-zero result, with the calling code
        // detecting that the erase operation has not completed.
    }
#endif

    // A result of 0 indicates a timeout, and depending on configury
    // the block is in erase_suspend mode. A non-zero result indicates
    // that the erase completed or there has been a fatal error.
    S29_2RAM_EXIT_HOOK();
    return retries;
}

// Write data to flash. At most one block will be processed at a time,
// but the write may be for a subset of the write. The destination
// address will be aligned in a way suitable for the bus. The source
// address need not be aligned. The count is in S29_TYPE's, i.e.
// as per the bus alignment, not in bytes.
static int
S29_FNNAME(s29_hw_program)(volatile S29_TYPE* block_start, volatile S29_TYPE* addr, const cyg_uint8* buf, cyg_uint32 count, int retries)
{
    int     i;

    S29_2RAM_ENTRY_HOOK();
    
    for (i = 0; (i < count) && (retries > 0); i++) {
        S29_TYPE   datum, current, active_dq7s;
        
        // We can only clear bits, not set them, so any bits that were
        // already clear need to be preserved.
        current = addr[i];
        datum   = S29_NEXT_DATUM(buf) & current;
        if (datum == current) {
            // No change, so just move on.
            continue;
        }
        
        block_start[S29_OFFSET_COMMAND]    = S29_COMMAND_SETUP1;
        HAL_MEMORY_BARRIER();
        block_start[S29_OFFSET_COMMAND2]   = S29_COMMAND_SETUP2;
        HAL_MEMORY_BARRIER();
        block_start[S29_OFFSET_COMMAND]    = S29_COMMAND_PROGRAM;
        HAL_MEMORY_BARRIER();
        addr[i] = datum;
        HAL_MEMORY_BARRIER();

        // The data is now being written. The official algorithm is
        // to poll either DQ7 or DQ6, checking DQ5 along the way for
        // error conditions. This gets complicated with parallel
        // flash chips because they may finish at different times.
        // The alternative approach is to ignore the status bits
        // completely and just look for current==datum until the
        // retry count is exceeded. However that does not cope
        // cleanly with cases where the flash chip reports an error
        // early on, e.g. because a flash block is locked.

        while (--retries > 0) {
#if CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_PROGRAM_DELAY > 0
            // Some chips want a delay between polling
            { int j; for( j = 0; j < CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_PROGRAM_DELAY; j++ ); }
#endif
            // While the program operation is in progress DQ7 will read
            // back inverted from datum.
            current = addr[i];
            if ((current & S29_STATUS_DQ7) == (datum & S29_STATUS_DQ7)) {
                // All DQ7 bits now match datum, so the operation has completed.
                // But not necessarily successfully. On some chips DQ7 may
                // toggle before DQ0-6 are valid, so we need to read the
                // data one more time.
                current = addr[i];
                if (current != datum) {
                    retries = 0;    // Abort this burst.
                }
                break;
            }

            // Now we want to check the DQ5 status bits, but only for those
            // chips which are still programming. ((current^datum) & DQ7) gives
            // ones for chips which are still programming, zeroes for chips when
            // the programming is complete.
            active_dq7s = (current ^ datum) & S29_STATUS_DQ7;
            
            if (0 != (current & (active_dq7s >> 2))) {
                // Unfortunately this is not sufficient to prove an error. On
                // some chips DQ0-6 switch to the data while DQ7 is still a
                // status flag, so the set DQ5 bit detected above may be data
                // instead of an error. Check again, this time DQ7 may
                // indicate completion.
                //
                // Next problem. Suppose chip A gave a bogus DQ5 result earlier
                // because it was just finishing. For this read chip A gives
                // back datum, but now chip B is finishing and has reported a
                // bogus DQ5.
                //
                // Solution: if any of the DQ7 lines have changed since the last
                // time, go around the loop again. When an error occurs DQ5
                // remains set and DQ7 remains toggled, so there is no harm
                // in one more polling loop.
                
                current = addr[i];
                if (((current ^ datum) & S29_STATUS_DQ7) != active_dq7s) {
                    continue;
                }

                // DQ5 has been set in a chip where DQ7 indicates an ongoing
                // program operation for two successive reads. That is an error.
                // The hardware is in a strange state so must be reset.
                block_start[S29_OFFSET_COMMAND]    = S29_COMMAND_RESET;
                HAL_MEMORY_BARRIER();
                retries = 0;
                break;
            }
            // No DQ5 bits set in chips which are still programming. Poll again.
        }   // Retry for current word
    }       // Next word

    // At this point retries holds the total number of retries left.
    //  0 indicates a timeout or fatal error.
    // >0 indicates success.
    S29_2RAM_EXIT_HOOK();
    return retries;
}

// FIXME: implement a separate program routine for buffered writes. 

#if 0
// Unused for now, but might be useful later
static int
S29_FNNAME(at49_hw_is_locked)(volatile S29_TYPE* addr)
{
    int result;
    S29_TYPE plane;

    S29_2RAM_ENTRY_HOOK();
    
    // Plane is bits A21-A20 for AT49BV6416
    // A more generic formula is needed.
    plane = S29_PARALLEL( ((((CYG_ADDRESS)addr)>>21) & 0x3) );
    addr[S29_OFFSET_COMMAND]         = S29_COMMAND_SETUP1;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND2]        = S29_COMMAND_SETUP2;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND + plane] = S29_COMMAND_AUTOSELECT;
    HAL_MEMORY_BARRIER();
    result          = addr[S29_OFFSET_AT49_LOCK_STATUS];
    addr[0]         = S29_COMMAND_RESET;
    HAL_MEMORY_BARRIER();
    // The bottom two bits hold the lock status, LSB indicates
    // soft lock, next bit indicates hard lock. We don't distinguish
    // in this function.
    S29_2RAM_EXIT_HOOK();
    return (0 != (result & S29_ID_LOCKED));
}
#endif

static int
S29_FNNAME(at49_hw_softlock)(volatile S29_TYPE* addr)
{
    int result  = CYG_FLASH_ERR_OK;

    S29_2RAM_ENTRY_HOOK();
    
    addr[S29_OFFSET_COMMAND]         = S29_COMMAND_SETUP1;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND2]        = S29_COMMAND_SETUP2;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND]         = S29_COMMAND_AT49_SOFTLOCK_BLOCK_0;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND]         = S29_COMMAND_SETUP1;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND2]        = S29_COMMAND_SETUP2;
    HAL_MEMORY_BARRIER();
    addr[0]                           = S29_COMMAND_AT49_SOFTLOCK_BLOCK_1;
    HAL_MEMORY_BARRIER();
    // not sure if this is required:
    addr[0]                           = S29_COMMAND_RESET;
    HAL_MEMORY_BARRIER();
    S29_2RAM_EXIT_HOOK();
    return result;
}

static int
S29_FNNAME(at49_hw_hardlock)(volatile S29_TYPE* addr)
{
    int result  = CYG_FLASH_ERR_OK;

    S29_2RAM_ENTRY_HOOK();
    
    addr[S29_OFFSET_COMMAND]         = S29_COMMAND_SETUP1;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND2]        = S29_COMMAND_SETUP2;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND]         = S29_COMMAND_AT49_HARDLOCK_BLOCK_0;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND]         = S29_COMMAND_SETUP1;
    HAL_MEMORY_BARRIER();
    addr[S29_OFFSET_COMMAND2]        = S29_COMMAND_SETUP2;
    HAL_MEMORY_BARRIER();
    addr[0]                           = S29_COMMAND_AT49_HARDLOCK_BLOCK_1;
    HAL_MEMORY_BARRIER();
    // not sure if this is required:
    addr[0]                           = S29_COMMAND_RESET;
    HAL_MEMORY_BARRIER();
    S29_2RAM_EXIT_HOOK();
    return result;
}

static int
S29_FNNAME(at49_hw_unlock)(volatile S29_TYPE* addr)
{
    int result  = CYG_FLASH_ERR_OK;

    S29_2RAM_ENTRY_HOOK();
    
    addr[S29_OFFSET_COMMAND]         = S29_COMMAND_SETUP1;
    HAL_MEMORY_BARRIER();
    addr[0]                           = S29_COMMAND_AT49_UNLOCK_BLOCK;
    HAL_MEMORY_BARRIER();
    // not sure if this is required:
    addr[0]                           = S29_COMMAND_RESET;
    HAL_MEMORY_BARRIER();
    S29_2RAM_EXIT_HOOK();
    return result;
}


// ----------------------------------------------------------------------------
// Exported code, mostly for placing in a cyg_flash_dev_funs structure.

// Just read the device id, either for sanity checking that the system
// has been configured for the right device, or for filling in the
// block info by a platform-specific init routine if the platform may
// be manufactured with one of several different chips.
int
S29_FNNAME(cyg_am29xxxxx_read_devid) (struct cyg_flash_dev* dev)
{
    int                 (*query_fn)(volatile S29_TYPE*);
    int                 devid;
    volatile S29_TYPE* addr;
    S29_INTSCACHE_STATE;

    CYG_CHECK_DATA_PTR(dev, "valid flash device pointer required");

    spansion_diag("\n");
    
    addr     = S29_UNCACHED_ADDRESS(dev->start);
    query_fn = (int (*)(volatile S29_TYPE*)) cyg_flash_anonymizer( & S29_FNNAME(s29_hw_query) );
    S29_INTSCACHE_BEGIN();
    devid    = (*query_fn)(addr);
    S29_INTSCACHE_END();
    return devid;
}

// Validate that the device statically configured is the one on the
// board.
int
S29_FNNAME(cyg_am29xxxxx_init_check_devid)(struct cyg_flash_dev* dev)
{
    cyg_am29xxxxx_dev*  s29_dev;
    int                 devid;

    spansion_diag("\n");
    
    s29_dev = (cyg_am29xxxxx_dev*) dev->priv;
    devid    = S29_FNNAME(cyg_am29xxxxx_read_devid)(dev);
    if (devid != s29_dev->devid) {
        return CYG_FLASH_ERR_DRV_WRONG_PART;
    }

#ifdef CYGHWR_DEVS_FLASH_SPANSION_S29GLN_V2_RESET_NEEDS_RESUME
    {
        volatile S29_TYPE *addr = S29_UNCACHED_ADDRESS(dev->start);
        void (*resume_fn)(struct cyg_flash_dev*, cyg_am29xxxxx_dev*, volatile S29_TYPE*);
        resume_fn = (void (*)(struct cyg_flash_dev*, cyg_am29xxxxx_dev*, volatile S29_TYPE*))
            cyg_flash_anonymizer( &S29_FNNAME(s29_hw_force_all_suspended_resume) );
        S29_INTSCACHE_STATE;

        S29_INTSCACHE_BEGIN();
        (*resume_fn)(dev, s29_dev, addr);
        S29_INTSCACHE_END();
    }
#endif

    // Successfully queried the device, and the id's match. That
    // should be a good enough indication that the flash is working.
    return CYG_FLASH_ERR_OK;
}

// Initialize via a CFI query, instead of statically specifying the
// boot block layout.
int
S29_FNNAME(cyg_am29xxxxx_init_cfi)(struct cyg_flash_dev* dev)
{
    int                 (*cfi_fn)(struct cyg_flash_dev*, cyg_am29xxxxx_dev*, volatile S29_TYPE*);
    volatile S29_TYPE* addr;
    cyg_am29xxxxx_dev*  s29_dev;
    int                 result;
    S29_INTSCACHE_STATE;
    
    spansion_diag("\n");
    
    CYG_CHECK_DATA_PTR(dev, "valid flash device pointer required");
    s29_dev    = (cyg_am29xxxxx_dev*) dev->priv;    // Remove const, only place where this is needed.
    addr        = S29_UNCACHED_ADDRESS(dev->start);
    cfi_fn      = (int (*)(struct cyg_flash_dev*, cyg_am29xxxxx_dev*, volatile S29_TYPE*))
        cyg_flash_anonymizer( & S29_FNNAME(s29_hw_cfi));

    S29_INTSCACHE_BEGIN();
    result      = (*cfi_fn)(dev, s29_dev, addr);
    S29_INTSCACHE_END();

    // Now calculate the device size, and hence the end field.
    if (CYG_FLASH_ERR_OK == result) {
        int i;
        int size    = 0;
        for (i = 0; i < dev->num_block_infos; i++) {
            spansion_diag("region %d: 0x%08x * %d\n", i, (int)dev->block_info[i].block_size, dev->block_info[i].blocks );
            size += (dev->block_info[i].block_size * dev->block_info[i].blocks);
        }
        dev->end = dev->start + size - 1;

#ifdef CYGHWR_DEVS_FLASH_SPANSION_S29GLN_V2_RESET_NEEDS_RESUME
    {
        void (*resume_fn)(struct cyg_flash_dev*, cyg_am29xxxxx_dev*, volatile S29_TYPE*);
        resume_fn = (void (*)(struct cyg_flash_dev*, cyg_am29xxxxx_dev*, volatile S29_TYPE*))
            cyg_flash_anonymizer( &S29_FNNAME(s29_hw_force_all_suspended_resume) );

        S29_INTSCACHE_BEGIN();
        (*resume_fn)(dev, s29_dev, addr);
        S29_INTSCACHE_END();
    }
#endif
    }
    return result;
}

// Erase a single block. The calling code will have supplied a pointer
// aligned to a block boundary.
int
S29_FNNAME(cyg_am29xxxxx_erase)(struct cyg_flash_dev* dev, cyg_flashaddr_t addr)
{
    int                 (*erase_fn)(volatile S29_TYPE*, cyg_bool);
    volatile S29_TYPE* block;
    cyg_flashaddr_t     block_start;
    size_t              block_size;
    int                 i;
    int                 result;
    S29_INTSCACHE_STATE;

    CYG_CHECK_DATA_PTR(dev, "valid flash device pointer required");
    CYG_ASSERT((addr >= dev->start) && (addr <= dev->end), "flash address out of device range");

    s29_get_block_info(dev, addr, &block_start, &block_size);
    CYG_ASSERT(addr == block_start, "erase address should be the start of a flash block");
    
    spansion_diag("addr %p block %p[%d]\n", (void*)addr, (void*)block_start, (int)block_size );
    
    block       = S29_UNCACHED_ADDRESS(addr);
    erase_fn    = (int (*)(volatile S29_TYPE*, cyg_bool)) cyg_flash_anonymizer( & S29_FNNAME(s29_hw_erase) );

    S29_INTSCACHE_BEGIN();
    result = (*erase_fn)(block, true);
#if (CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_ERASE_BURST_DURATION < CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_ERASE_TIMEOUT)
    if (0 == result) {
        // The retry count expired so the erase has not completed.
        // The block is currently in erase-suspend mode.
        // Ints and cache are still disabled.
        int retries_left;
        
        for (retries_left = CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_ERASE_TIMEOUT - CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_ERASE_BURST_DURATION;
             retries_left > 0;
             retries_left -= CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_ERASE_BURST_DURATION) {

            // Give other code a chance to run.
            S29_INTSCACHE_SUSPEND();
            S29_INTSCACHE_RESUME();
            result = (*erase_fn)(block, false);
            if (0 != result) {
                // The erase has finished so the block is no longer in
                // erase-suspend mode
                break;
            }
        }
    }
#endif
    S29_INTSCACHE_END();

    // The erase may have failed for a number of reasons, e.g. because
    // of a locked sector. The best thing to do here is to check that the
    // erase has succeeded.
    block = (S29_TYPE*) addr;
    for (i = 0; i < (block_size / sizeof(S29_TYPE)); i++) {
        if (block[i] != (S29_TYPE)~0) {
            // There is no easy way of detecting the specific error,
            // e.g. locked flash block, timeout, ... So return a
            // useless catch-all error.
            return CYG_FLASH_ERR_ERASE;
        }
    }
    return CYG_FLASH_ERR_OK;
}

// Write some data to the flash. The destination must be aligned
// appropriately for the bus width (not the device width).
int
S29_FNNAME(cyg_am29xxxxx_program)(struct cyg_flash_dev* dev, cyg_flashaddr_t dest, const void* src, size_t len)
{
    int                 (*program_fn)(volatile S29_TYPE*, volatile S29_TYPE*, const cyg_uint8*, cyg_uint32, int);
    volatile S29_TYPE* block;
    volatile S29_TYPE* addr; 
    cyg_flashaddr_t     block_start;
    size_t              block_size;
    const cyg_uint8*    data;
    int                 retries;
    int                 to_write;
    int                 i;

    S29_INTSCACHE_STATE;

    CYG_CHECK_DATA_PTR(dev, "valid flash device pointer required");

    spansion_diag("dest %p src %p len %d\n", (void*)dest, (void*)src, (int)len );
    
    // Only support writes that are aligned to the bus boundary. This
    // may be more restrictive than what the hardware is capable of.
    // However it ensures that the hw_program routine can write as
    // much data as possible each iteration, and hence significantly
    // improves performance. The length had better be a multiple of
    // the bus width as well
    if ((0 != ((CYG_ADDRWORD)dest & (sizeof(S29_TYPE) - 1))) ||
        (0 != (len & (sizeof(S29_TYPE) - 1)))) {
        return CYG_FLASH_ERR_INVALID;
    }

    addr        = S29_UNCACHED_ADDRESS(dest);
    CYG_ASSERT((dest >= dev->start) && (dest <= dev->end), "flash address out of device range");

    s29_get_block_info(dev, dest, &block_start, &block_size);
    CYG_ASSERT(((dest - block_start) + len) <= block_size, "write cannot cross block boundary");
    
    block       = S29_UNCACHED_ADDRESS(block_start);
    data        = (const cyg_uint8*) src;
    retries     = CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_PROGRAM_TIMEOUT;
    to_write    = len / sizeof(S29_TYPE);      // Number of words, not bytes

    program_fn  = (int (*)(volatile S29_TYPE*, volatile S29_TYPE*, const cyg_uint8*, cyg_uint32, int))
        cyg_flash_anonymizer( & S29_FNNAME(s29_hw_program) );

    S29_INTSCACHE_BEGIN();
    while ((to_write > 0) && (retries > 0)) {
        size_t this_write = (to_write < CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_PROGRAM_BURST_SIZE) ?
            to_write : CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_PROGRAM_BURST_SIZE;

        retries = (*program_fn)(block, addr, data, this_write, retries);
        to_write -= CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_PROGRAM_BURST_SIZE;
        if (to_write > 0) {
            // There is still more to be written. The last write must have been a burst size
            addr += CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_PROGRAM_BURST_SIZE;
            data += sizeof(S29_TYPE) * CYGNUM_DEVS_FLASH_SPANSION_S29GLN_V2_PROGRAM_BURST_SIZE;
            S29_INTSCACHE_SUSPEND();
            S29_INTSCACHE_RESUME();
        }
    }
    S29_INTSCACHE_END();

    spansion_diag("to_write %d retries %d\n", to_write, retries );
    
    // Too many things can go wrong when manipulating the h/w, so
    // verify the operation by actually checking the data.
    addr = (volatile S29_TYPE*) dest;
    data = (const cyg_uint8*) src;
    for (i = 0; i < (len / sizeof(S29_TYPE)); i++) {
        S29_TYPE   datum   = S29_NEXT_DATUM(data);
        S29_TYPE   current = addr[i];
        if ((datum & current) != current) {
            spansion_diag("data %p addr[i] %p datum %08x current %08x\n", data-sizeof(S29_TYPE), &addr[i], datum, current );
            return CYG_FLASH_ERR_PROGRAM;
        }
    }
    return CYG_FLASH_ERR_OK;
}

int
S29_FNNAME(cyg_at49xxxx_softlock)(struct cyg_flash_dev* dev, const cyg_flashaddr_t dest)
{
    volatile S29_TYPE*   uncached;
    int                     result;
    int (*lock_fn)(volatile S29_TYPE*);
    S29_INTSCACHE_STATE;

    spansion_diag("\n");
    
    CYG_CHECK_DATA_PTR(dev, "valid flash device pointer required");
    CYG_ASSERT((dest >= (cyg_flashaddr_t)dev->start) &&
               ((CYG_ADDRESS)dest <= dev->end), "flash address out of device range");

    uncached    = S29_UNCACHED_ADDRESS(dest);
    lock_fn     = (int (*)(volatile S29_TYPE*)) cyg_flash_anonymizer( & S29_FNNAME(at49_hw_softlock) );
    S29_INTSCACHE_BEGIN();
    result  = (*lock_fn)(uncached);
    S29_INTSCACHE_END();
    return result;
}

int
S29_FNNAME(cyg_at49xxxx_hardlock)(struct cyg_flash_dev* dev, const cyg_flashaddr_t dest)
{
    volatile S29_TYPE*   uncached;
    int                     result;
    int (*lock_fn)(volatile S29_TYPE*);
    S29_INTSCACHE_STATE;

    spansion_diag("\n");
    
    CYG_CHECK_DATA_PTR(dev, "valid flash device pointer required");
    CYG_ASSERT((dest >= (cyg_flashaddr_t)dev->start) &&
               ((CYG_ADDRESS)dest <= dev->end), "flash address out of device range");

    uncached    = S29_UNCACHED_ADDRESS(dest);
    lock_fn     = (int (*)(volatile S29_TYPE*)) cyg_flash_anonymizer( & S29_FNNAME(at49_hw_hardlock) );
    S29_INTSCACHE_BEGIN();
    result  = (*lock_fn)(uncached);
    S29_INTSCACHE_END();
    return result;
}

int
S29_FNNAME(cyg_at49xxxx_unlock)(struct cyg_flash_dev* dev, const cyg_flashaddr_t dest)
{
    volatile S29_TYPE* uncached;
    int                 result;
    int (*unlock_fn)(volatile S29_TYPE*);
    S29_INTSCACHE_STATE;

    spansion_diag("\n");
    
    CYG_CHECK_DATA_PTR(dev, "valid flash device pointer required");
    CYG_ASSERT((dest >= (cyg_flashaddr_t)dev->start) &&
               ((CYG_ADDRESS)dest <= dev->end), "flash address out of device range");

    uncached = S29_UNCACHED_ADDRESS(dest);
    unlock_fn = (int (*)(volatile S29_TYPE*)) cyg_flash_anonymizer( & S29_FNNAME(at49_hw_unlock) );

    S29_INTSCACHE_BEGIN();
    result  = (*unlock_fn)(uncached);
    S29_INTSCACHE_END();
    return result;
}

// ----------------------------------------------------------------------------
// Clean up the various #define's so this file can be #include'd again
#undef S29_FNNAME
#undef S29_RAMFNDECL
#undef S29_OFFSET_COMMAND
#undef S29_OFFSET_COMMAND2
#undef S29_OFFSET_DEVID
#undef S29_OFFSET_DEVID2
#undef S29_OFFSET_DEVID3
#undef S29_OFFSET_CFI
#undef S29_OFFSET_CFI_DATA
#undef S29_OFFSET_AT49_LOCK_STATUS
#undef S29_PARALLEL
