//==========================================================================
//
//      watchdog_at91rm9200.cxx
//
//      Watchdog implementation for Atmel AT91RM9200 variant CPUs
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2004, 2005, 2009 eCosCentric Limited                              
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005 Free Software Foundation, Inc.
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    jlarmour
// Contributors: 
// Date:         2005-08-15
// Purpose:      Watchdog class implementation
// Description:  Contains an implementation of the Watchdog class for use
//               with the Atmel AT91RM9200 on-chip watchdog timer.
//               Based on LPC2xxx watchdog driver.
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/kernel.h>
#include <pkgconf/infra.h>
#include <pkgconf/kernel.h>
#include <pkgconf/watchdog.h>
#include <pkgconf/devs_watchdog_arm_at91rm9200.h>

#include <cyg/infra/cyg_type.h>
#include <cyg/infra/cyg_ass.h>
#include <cyg/infra/cyg_trac.h>
#include <cyg/hal/hal_io.h>
#include <cyg/hal/hal_diag.h>

#include <cyg/io/watchdog.hxx>

#if !defined(CYGSEM_WATCHDOG_RESETS_ON_TIMEOUT)
#include <cyg/hal/hal_platform_ints.h>
#include <cyg/kernel/intr.hxx>
#endif

//==========================================================================

#define WD_TICK_FREQ            (32768/128)  // Slow clock divided by 128
#define WD_TIMER_CONSTANT_INIT  (((cyg_uint64)CYGNUM_DEVS_WATCHDOG_ARM_AT91RM9200_DESIRED_TIMEOUT_MS * WD_TICK_FREQ + 999)/1000)

//==========================================================================

#if defined(CYGSEM_WATCHDOG_RESETS_ON_TIMEOUT)

void
Cyg_Watchdog::init_hw(void)
{
  CYG_REPORT_FUNCTION();
  CYG_REPORT_FUNCARGVOID();
  resolution = (cyg_uint64)CYGNUM_DEVS_WATCHDOG_ARM_AT91RM9200_DESIRED_TIMEOUT_MS*1000000;
  CYG_REPORT_RETURN();
}

#else /* defined(CYGSEM_WATCHDOG_RESETS_ON_TIMEOUT) */

//==========================================================================

static Cyg_Watchdog *wd;

//==========================================================================

static cyg_uint32
isr(cyg_vector vector, CYG_ADDRWORD data)
{
    cyg_uint32 status;

    CYG_REPORT_FUNCTION();
    CYG_REPORT_FUNCARG2XV(vector, data);

    CYG_ASSERT( vector == CYGNUM_HAL_INTERRUPT_WDOVF, "WD ISR called with wrong vector");
    wd->trigger();
    Cyg_Interrupt::acknowledge_interrupt(CYGNUM_HAL_INTERRUPT_WDOVF);
    CYG_REPORT_RETVAL(Cyg_Interrupt::HANDLED);
    return Cyg_Interrupt::HANDLED;
}

//==========================================================================

// Grab the watchdog overflow interrupt, a "virtual" interrupt within the system timer interrupt

static Cyg_Interrupt CYGBLD_ATTRIB_INIT_PRI(CYG_INIT_INTERRUPTS) wdint(
    CYGNUM_HAL_INTERRUPT_WDOVF,
    1,
    0,
    isr,
    NULL
  );

//==========================================================================

void
Cyg_Watchdog::init_hw(void)
{
  CYG_REPORT_FUNCTION();
  CYG_REPORT_FUNCARGVOID();

  wd = this;
  resolution = (cyg_uint64)CYGNUM_DEVS_WATCHDOG_ARM_AT91RM9200_DESIRED_TIMEOUT_MS*1000000;
  CYG_REPORT_RETURN();
}

#endif	/* defined(CYGSEM_WATCHDOG_RESETS_ON_TIMEOUT) */

//==========================================================================
/*
 * Reset watchdog timer. This needs to be called regularly to prevent
 * the watchdog from firing.
 */

void
Cyg_Watchdog::reset(void)
{
  CYG_REPORT_FUNCTION();
  CYG_REPORT_FUNCARGVOID();

  /* Simple bit twiddle to reset the watchdog. */
  HAL_WRITE_UINT32( _ST_CR, _ST_CR_WDRST );
  CYG_REPORT_RETURN();
}

//==========================================================================
/*
 * Start watchdog to generate a hardware reset
 * or interrupt when expiring.
 */

void
Cyg_Watchdog::start(void)
{
  CYG_REPORT_FUNCTION();
  CYG_REPORT_FUNCARGVOID();
  cyg_uint32 wdmr;

  // The watchdog always runs but at power-on defaults to only
  // asserting an external signal, not a reset.

  // First reset watchdog to (present) max value to avoid any risk of a
  // watchdog reset happening while we're doing the setup, since it
  // is continuously running.
  HAL_WRITE_UINT32( _ST_CR, _ST_CR_WDRST );

  // Now set up watchdog properly

  // Set up counter, configuring for reset if required
  wdmr = _ST_WDMR_EXTEN | (WD_TIMER_CONSTANT_INIT & 0xffff);
#ifdef CYGSEM_WATCHDOG_RESETS_ON_TIMEOUT
  wdmr |= _ST_WDMR_RSTEN;
#endif
  // Set value
  HAL_WRITE_UINT32( _ST_WDMR, wdmr );

  // Enable/Reset
  HAL_WRITE_UINT32( _ST_CR, _ST_CR_WDRST );

  // Enable interrupt if needed
#ifndef CYGSEM_WATCHDOG_RESETS_ON_TIMEOUT
  wdint.attach();
  wdint.acknowledge_interrupt(CYGNUM_HAL_INTERRUPT_WDOVF);
  wdint.unmask_interrupt(CYGNUM_HAL_INTERRUPT_WDOVF);
  HAL_WRITE_UINT32( _ST_IER, _ST_SR_WDOVF );
#endif

  CYG_REPORT_RETURN();
}

//==========================================================================
// End of watchdog_at91rm9200.cxx
