#ifndef CYGONCE_FATFS_FATFS_H
#define CYGONCE_FATFS_FATFS_H
//==========================================================================
//
//      fatfs.h
//
//      FAT file system header 
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 2003, 2004, 2005, 2006, 2007 Free Software Foundation, Inc.
// Copyright (C) 2004, 2005, 2006, 2007 eCosCentric Limited                 
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):           savin
// Contributor(s):      nickg,jlarmour
// Date:                2003-06-29
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/fs_fat.h>
#include <cyg/infra/cyg_type.h>
#include <cyg/io/io.h>
#include <cyg/fileio/fileio.h>

#include <linux/rbtree.h>
#include <linux/list.h>

#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#include <dirent.h>

#include <stdlib.h>
#include <string.h>

#include <cyg/fileio/fileio.h>

// --------------------------------------------------------------------------

#define FATFS_HASH_TABLE_SIZE CYGNUM_FS_FAT_NODE_HASH_TABLE_SIZE 

#define FATFS_NODE_POOL_SIZE  CYGNUM_FS_FAT_NODE_POOL_SIZE

#ifdef CYGDBG_FS_FAT_NODE_CACHE_EXTRA_CHECKS
# define FATFS_NODE_CACHE_EXTRA_CHECKS 1
#endif

#ifdef CYGCFG_FS_FAT_LONG_FILE_NAMES
#define CYG_FAT_FILE_NAME_MAX   (CYGNUM_FS_FAT_LONG_FILE_NAME_MAX+1)
#else
#define CYG_FAT_FILE_NAME_MAX   13
#endif

// --------------------------------------------------------------------------

// Node cache tracing support
//#define FATFS_TRACE_NODE_CACHE 1

// FAT dir entry operations tracing support 
//#define FATFS_TRACE_DIR_ENTRY 1

// FAT clusters operations tracing support 
//#define FATFS_TRACE_CLUSTER 1

// FAT data tracing support 
//#define FATFS_TRACE_DATA 1

// FAT file operations tracing support 
//#define FATFS_TRACE_FILE_OP 1

// FAT filesystem operations tracing support 
//#define FATFS_TRACE_FS_OP 1

// --------------------------------------------------------------------------

struct fatfs_disk_s;
typedef struct fatfs_disk_s fatfs_disk_t;

// --------------------------------------------------------------------------

typedef enum fatfs_type_e
{
    FATFS_FAT12 = 0,    
    FATFS_FAT16,
    FATFS_FAT32
} fatfs_type_t;

typedef struct fatfs_data_pos_s
{
    cyg_uint32 cluster;      // Cluster number
    cyg_uint32 cluster_snum; // Cluster file seq number 
                             // (0 - first cluster of file,
                             //  1 - second cluster of file, ...)  
    cyg_uint32 cluster_pos;  // Position inside cluster
} fatfs_data_pos_t;

typedef struct {
    cyg_uint32  n_gets;     // number of block gets
    cyg_uint32  n_reads;    // number of block reads
    cyg_uint32  n_writes;   // number of block writes
} fatfs_blib_stat_t;

typedef struct {
    cyg_uint32            flags;           // mode flags
    struct list_head      list_head;       // head of block list 
    struct rb_root        rb_root;         // block red-black tree root
    fatfs_disk_t          *disk;           // Pointer to disk object
    cyg_uint32            block_size;      // block size
    cyg_uint32            block_size_log2; // block size log2
    cyg_uint32            sector_size;     // disk sector size
    cyg_uint8            *mem_base;        // memory base
    cyg_uint32            mem_size;        // memory size
    struct list_head      free_list_head;  // list of free blocks
#ifdef CYGIMP_BLOCK_LIB_STATISTICS
    fatfs_blib_stat_t     stat;            // statistics
#endif
    volatile cyg_int32    io_requests;     // number of outstanding IO requests
    cyg_int32             modified_blocks; // Number of modified blocks in cache
} fatfs_blib_t;

typedef struct fatfs_dir_entry_s
 {
    char              filename[12+1]; // File name
    mode_t            mode;           // Node type
    size_t            size;           // Size of file in bytes
    time_t            ctime;          // Creation timestamp
    time_t            atime;          // Last access timestamp
    time_t            mtime;          // Last write timestamp
    cyg_uint8         priv_data;      // Private data
    cyg_uint32        cluster;        // First cluster number
    cyg_uint32        parent_cluster; // First cluster of parent dentry
    fatfs_data_pos_t  disk_pos;       // Position of dir entry on disk
#ifdef CYGCFG_FS_FAT_LONG_FILE_NAMES
    fatfs_data_pos_t  start_pos;      // Position of start of dir entry on disk
    int               extras;         // Number of extra dir entries
#endif
#ifdef CYGCFG_FS_FAT_USE_ATTRIBUTES
    cyg_fs_attrib_t   attrib;         // Attribute bits for DOS compatability
#endif //CYGCFG_FS_FAT_USE_ATTRIBUTES
} fatfs_dir_entry_t;

typedef struct fatfs_node_s
{
    fatfs_dir_entry_t    dentry;     // Dir entry data
    cyg_ucount32         refcnt;     // Open file/current dir references
    cyg_bool             free_pending; // node_free is pending
 
    struct fatfs_node_s *list_prev;  // Next node in list
    struct fatfs_node_s *list_next;  // Prev node in list
    struct fatfs_node_s *hash_next;  // Next node in hash

    struct fatfs_node_list_s *list;  // Pointer to current list

    char                filename[CYG_FAT_FILE_NAME_MAX];
    
} fatfs_node_t;

typedef struct fatfs_hash_table_s 
{
    cyg_uint32     size;                         // Number of slots
    cyg_uint32     n;                            // Number of nodes 
    fatfs_node_t  *nodes[FATFS_HASH_TABLE_SIZE]; // Nodes slots
} fatfs_hash_table_t;

typedef struct fatfs_node_list_s
{
    cyg_uint32    size;          // Number of nodes in list
    fatfs_node_t *first;         // First node in list
    fatfs_node_t *last;          // Last node in list
} fatfs_node_list_t;

struct fatfs_disk_s
{
    cyg_uint32    sector_size;          // Sector size in bytes
    cyg_uint32    sector_size_log2;     // Sector size log2
    cyg_uint32    cluster_size;         // Cluster size in bytes
    cyg_uint32    cluster_size_log2;    // Cluster size log2 
    cyg_uint32    fat_tbl_pos;          // Position of the first FAT table
    cyg_uint32    fat_tbl_size;         // FAT table size in bytes
    cyg_uint32    fat_tbl_nents;        // Number of entries in FAT table
    cyg_uint32    fat_tbls_num;         // Number of FAT tables
    cyg_uint32    fat_root_dir_pos;     // Position of the root dir
    cyg_uint32    fat_root_dir_size;    // Root dir size in bytes 
    cyg_uint32    fat_root_dir_nents;   // Max number of entries in root dir
    cyg_uint32    fat_root_dir_cluster; // Cluster number of root dir (FAT32) 
    cyg_uint32    fat_data_pos;         // Position of data area
    fatfs_type_t  fat_type;             // Type of FAT - 12, 16 or 32 
     
    cyg_io_handle_t  dev_h;             // Disk device handle
    fatfs_node_t    *root;              // Root dir node

    cyg_mtab_entry  *mte;               // Mount table entry
#ifdef CYGPKG_KERNEL
    cyg_drv_mutex_t *lock;              // Filesystem lock
    cyg_drv_cond_t  pending;            // cond var for pending IO threads
#endif

    cyg_uint32       flags;             // mode flags
    cyg_bool         mounted;           // Filesystem mounted
    
    cyg_uint8       *bcache_mem;        // Block cache memory base
    fatfs_blib_t     blib;              // Block cache and access library instance
 
    fatfs_node_t  node_pool_base[FATFS_NODE_POOL_SIZE]; // Node pool base   
    fatfs_node_t *node_pool[FATFS_NODE_POOL_SIZE];      // Node pool 
    cyg_uint32    node_pool_free_cnt;                   // Node pool free cnt
    
    fatfs_node_list_t  live_nlist;      // List of nodes with refcnt > 0
    fatfs_node_list_t  dead_nlist;      // List of nodes with refcnt == 0
    fatfs_hash_table_t node_hash;       // Hash of nodes in live and dead lists

    struct cyg_fs_callback_info callback;      // Filesystem callbacks

#ifdef CYGCFG_FS_FAT_LONG_FILE_NAMES    
    struct cyg_fs_mbcs_translate mbcs_translate; // Character code translation
#endif
};

// Blib and Disk mode flags
#define FATFS_FLAGS_WRITETHROUGH        0x01    // Write through caching
#define FATFS_FLAGS_SYNCCLOSE           0x02    // Sync on file close
#define FATFS_FLAGS_WRITEDATA           0x04    // Write-through data to disk
#define FATFS_FLAGS_READONLY            0x10    // Mount Read-only

// --------------------------------------------------------------------------

int  fatfs_init(fatfs_disk_t *disk);

void fatfs_get_root_dir_entry(fatfs_disk_t *disk, fatfs_dir_entry_t *dentry);

bool fatfs_is_root_dir_dentry(fatfs_dir_entry_t *dentry);

int  fatfs_get_disk_usage(fatfs_disk_t *disk,
                          cyg_uint32   *total_clusters,
                          cyg_uint32   *free_clusters);

int  fatfs_update_fsinfo( fatfs_disk_t *disk );

int  fatfs_initpos(fatfs_disk_t      *disk,
                   fatfs_dir_entry_t *file,
                   fatfs_data_pos_t  *pos);

int  fatfs_setpos(fatfs_disk_t      *disk,
                  fatfs_dir_entry_t *file,
                  fatfs_data_pos_t  *pos,
                  cyg_uint32         offset);

cyg_uint32 fatfs_getpos(fatfs_disk_t      *disk, 
                        fatfs_dir_entry_t *file,
                        fatfs_data_pos_t  *pos);

#ifdef CYGCFG_FS_FAT_LONG_FILE_NAMES

cyg_bool fatfs_is_long_filename( const char         *name,
                                 int                 namelen );

int fatfs_make_shortname( const cyg_uint16   *name,
                          int                 namelen,
                          char               *shortname);
#endif

int  fatfs_read_dir_entry(fatfs_disk_t      *disk,
                          fatfs_dir_entry_t *dir,
                          fatfs_data_pos_t  *pos,
                          fatfs_dir_entry_t *dentry);

int  fatfs_read_dir_entry_name(fatfs_disk_t      *disk,
                               fatfs_dir_entry_t *dir,
                               fatfs_data_pos_t  *pos,
                               char              *name,
                               int                len);

int  fatfs_write_dir_entry(fatfs_disk_t *disk, fatfs_dir_entry_t *dentry);

int  fatfs_find_dir_entry(fatfs_disk_t      *disk,
                          fatfs_dir_entry_t *dir,
                          fatfs_data_pos_t  *pos,
                          const char        *name,
                          int                namelen,
                          fatfs_dir_entry_t *dentry);

int  fatfs_delete_file(fatfs_disk_t         *disk,
                       fatfs_dir_entry_t    *dir,
                       fatfs_dir_entry_t    *file,
                       const char           *name,
                       int                   namelen);

int  fatfs_create_file(fatfs_disk_t      *disk, 
                       fatfs_dir_entry_t *dir, 
                       const char        *name,
                       int                namelen,
                       fatfs_dir_entry_t *dentry);

int  fatfs_create_dir(fatfs_disk_t      *disk, 
                      fatfs_dir_entry_t *dir, 
                      const char        *name,
                      int                namelen,
                      fatfs_dir_entry_t *dentry);

int  fatfs_trunc_file(fatfs_disk_t *disk, fatfs_dir_entry_t *file);

int  fatfs_rename_file(fatfs_disk_t      *disk,
                       fatfs_dir_entry_t *dir1,
                       fatfs_dir_entry_t *target,
                       fatfs_dir_entry_t *dir2,
                       const char        *name,
                       int                namelen);

int  fatfs_read_data(fatfs_disk_t      *disk,
                     fatfs_dir_entry_t *file,
                     fatfs_data_pos_t  *pos,
                     void              *data,
                     cyg_uint32        *len);

int  fatfs_write_data(fatfs_disk_t      *disk,
                      fatfs_dir_entry_t *file,
                      fatfs_data_pos_t  *pos,
                      void              *data,
                      cyg_uint32        *len);

// --------------------------------------------------------------------------

void fatfs_node_cache_init(fatfs_disk_t *disk);

void fatfs_node_cache_flush(fatfs_disk_t *disk);

fatfs_node_t *fatfs_node_alloc(fatfs_disk_t      *disk,
                               fatfs_dir_entry_t *dentry,
                               const char        *name,
                               int               namelen);

void fatfs_node_ref(fatfs_disk_t *disk, fatfs_node_t *node);

void fatfs_node_unref(fatfs_disk_t *disk, fatfs_node_t *node);

void fatfs_node_touch(fatfs_disk_t *disk, fatfs_node_t *node);

void fatfs_node_rehash(fatfs_disk_t *disk, fatfs_node_t *node);

void fatfs_node_free(fatfs_disk_t *disk, fatfs_node_t *node);

void fatfs_node_free_unref(fatfs_disk_t *disk, fatfs_node_t *node);

fatfs_node_t* fatfs_node_find(fatfs_disk_t *disk, 
                              const char   *name, 
                              unsigned int  namelen, 
                              cyg_uint32    parent_cluster);

int  fatfs_get_live_node_count(fatfs_disk_t *disk);

int  fatfs_get_dead_node_count(fatfs_disk_t *disk);

// --------------------------------------------------------------------------
// Support routines
// These enable the definition of local versions of certain routines
// if the given packages are not present.

#ifndef CYGPKG_LIBC_I18N

__externC int toupper( int c );

#endif

#ifndef CYGFUN_LIBC_STRING_BSD_FUNCS

__externC int strcasecmp( const char *s1, const char *s2 );

__externC int strncasecmp( const char *s1, const char *s2, size_t n );

#endif


// --------------------------------------------------------------------
// Creates a block lib instance
// 
//   mem_base   - block cache memory base
//   mem_size   - block cache memory size
//   block_size - block size
//   bl         - block lib instance space holder
//   
//   returns ENOERR if create succeded
//

int fatfs_blib_create(fatfs_disk_t       *disk,                    
                      void               *mem_base,
                      cyg_uint32          mem_size,
                      cyg_uint32          block_size,
                      cyg_uint32          sector_size,
                      cyg_uint32          flags,
                      fatfs_blib_t       *bl);

// --------------------------------------------------------------------
// Creates a block lib instance on top of IO system 
//   (cyg_io_bread and cyg_io_bwrite)
// 
//   handle      - cyg_io_handle_t
//   mem_base    - block cache memory base
//   mem_size    - block cache memory size
//   block_size  - block size
//   sector_size - disk sector size
//   bl          - block lib instance space holder
//   
//   returns ENOERR if create succeded
//

int fatfs_blib_io_create(fatfs_disk_t       *disk,
                         void               *mem_base,
                         cyg_uint32          mem_size,
                         cyg_uint32          block_size,
                         cyg_uint32          sector_size,
                         cyg_uint32          flags,                       
                         fatfs_blib_t       *bl);

// --------------------------------------------------------------------
// Deletes a block lib instance
//   
//   bl    - block lib instance
//   force - force deletion without syncing blocks
//
//   The block cache is synced before
//
//   returns ENOERR if delete succeded
//

int fatfs_blib_delete(fatfs_blib_t *bl, cyg_bool force);

// --------------------------------------------------------------------
// Reads a number of blocks
//
//   bl  - block lib instance
//   buf - block buffer ptr 
//   len - number of blocks to read
//   pos - starting block number
//       
//   returns ENOERR if read succeded
//   
        
int fatfs_blib_bread(fatfs_blib_t *bl,
                     void         *buf,
                     cyg_uint32   *len,
                     cyg_uint32    pos);

// --------------------------------------------------------------------
// Writes a number of blocks
//
//   bl  - block lib instance
//   buf - block buffer ptr 
//   len - number of blocks to write 
//   pos - starting block number
//       
//   returns ENOERR if write succeded
//   
 
int fatfs_blib_bwrite(fatfs_blib_t *bl,
                      const void   *buf,
                      cyg_uint32   *len,
                      cyg_uint32    pos);

// --------------------------------------------------------------------
// Reads data
//
//   bl   - block lib instance
//   buf  - data buffer ptr 
//   len  - number of bytes to read
//   bnum - starting block number 
//   pos  - starting position inside starting block
//       
//   returns ENOERR if read succeded
//   
//   The block number is automatically adjusted if
//   position is greater than block size
//
 
int fatfs_blib_read(fatfs_blib_t *bl,
                    void        *buf,
                    cyg_uint32  *len,
                    cyg_uint32   bnum,
                    cyg_uint32   pos);

// --------------------------------------------------------------------
// Writes data
//
//   bl   - block lib instance
//   buf  - data buffer ptr 
//   len  - number of bytes to write 
//   bnum - starting block number 
//   pos  - starting position inside starting block
//       
//   returns ENOERR if write succeded
//
//   The block number is automatically adjusted if
//   position is greater than block size
//    
 
int fatfs_blib_write(fatfs_blib_t *bl,
                     const void   *buf,
                     cyg_uint32   *len,
                     cyg_uint32    bnum,
                     cyg_uint32    pos);

// --------------------------------------------------------------------
// Syncs block cache - ie write modified blocks
//
//   bl - block lib instance
//
//   returns ENOERR if sync succeded
//

int fatfs_blib_sync(fatfs_blib_t *bl);

// --------------------------------------------------------------------
// Syncs block - ie write if modified
//
//   bl  - block lib instance
//   num - block number to sync
//
//   returns ENOERR if sync succeded
//

int fatfs_blib_sync_block(fatfs_blib_t *bl, cyg_uint32 num);

// --------------------------------------------------------------------
// Flushes block cache 
//
//   bl  - block lib instance
//
//   returns ENOERR if flush succeded
//
//   The block cache is synced before
//

int fatfs_blib_flush(fatfs_blib_t *bl);

// --------------------------------------------------------------------
// Sets block size 
//
//   bl         - block lib instance
//   block_size - new block size
//
//   returns ENOERR if set succeded
//
//   The block cache is synced before
//

int fatfs_blib_set_block_size(fatfs_blib_t *bl, cyg_uint32 block_size);

// --------------------------------------------------------------------
// Gets block size 
//
//   bl  - block lib instance
//
//   returns the current block size

static inline cyg_uint32 fatfs_blib_get_block_size(fatfs_blib_t *bl)
{
    return bl->block_size;
}

// --------------------------------------------------------------------
// Gets log2 of block size 
//
//   bl  - block lib instance
//
//   returns log2 of the current block size 

static inline cyg_uint32 fatfs_blib_get_block_size_log2(fatfs_blib_t *bl)
{
    return bl->block_size_log2;
}

// --------------------------------------------------------------------
// Gets block cache statistics 
//
//   bl - block lib instance
//
//   returns ENOERR if get succeded
//

int fatfs_blib_get_stat(fatfs_blib_t *bl, fatfs_blib_stat_t *stat);


// --------------------------------------------------------------------------

#endif // CYGONCE_FATFS_FATFS_H

// --------------------------------------------------------------------------
// EOF fatfs.h
