//==========================================================================
//
//      jffs2-2t1f.c
//
//      Test multiple thread use of JFFS2 filesystem
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004, 2006 Free Software Foundation, Inc.
// Copyright (C) 2004,2006 eCosCentric Limited                              
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):           jlarmour
// Contributors:        nickg
// Date:                2004-12-14
// Purpose:             Test multiple thread use of JFFS2 filesystem
// Description:         This tests thread use of JFFS2 by having two threads
//                      operate on a single file, overlapping operations.
//                      This also consequently exercises the garbage collection.
//
//####DESCRIPTIONEND####
//
//==========================================================================

#define NLOOPS 3
#ifndef NLOOPS
#define NLOOPS (1000/NTHREADS)
#endif
// Make sure FILESIZE is a multiple of IOSIZE
#define IOSIZE 100
#define FILESIZE 20000

#ifndef NTHREADS
#define NTHREADS 2
#endif
#if NTHREADS > 99
#error Too many threads
#endif
#ifndef TESTNAME
#define TESTNAME "jffs2-2t1f"
#endif

#include <pkgconf/system.h>
#include <pkgconf/isoinfra.h>
#ifdef CYGPKG_KERNEL
#include <pkgconf/kernel.h>
#endif
#ifdef CYGPKG_IO_FLASH
#include <pkgconf/io_flash.h>
#endif

#ifndef CYGPKG_KERNEL
# define NA_MSG "No kernel"
#elif !defined(CYGPKG_IO_FLASH)
# define NA_MSG "No flash layer"
#elif !defined(CYGFUN_KERNEL_THREADS_TIMER)
# define NA_MSG "No kernel thread timers"
#elif !defined(CYGINT_ISO_STRING_STRFUNCS)
# define NA_MSG "No string str* functions"
#elif !defined(CYGINT_ISO_STRING_MEMFUNCS)
# define NA_MSG "No string mem* functions"
#endif

#ifdef NA_MSG
#include <cyg/infra/testcase.h>
externC
void cyg_start(void)
{
    CYG_TEST_INIT();
    CYG_TEST_NA(NA_MSG);
    CYG_TEST_FINISH("Done");
}
#else

#include <cyg/kernel/thread.hxx>
#include <cyg/kernel/thread.inl>
#include <cyg/kernel/sched.hxx>
#include <cyg/kernel/sched.inl>
#include <cyg/kernel/sema.hxx>

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include <errno.h>
#include <dirent.h>

#include <cyg/io/io.h>
#include <cyg/io/config_keys.h>
#include <cyg/fileio/fileio.h>
#include <cyg/io/flash.h>

#include <cyg/infra/testcase.h>
#include <cyg/infra/diag.h>            // HAL polled output
#include "testaux.hxx"

Cyg_Counting_Semaphore thrsema;

volatile int stopnow = 0;


//==========================================================================
// Mount details

#define stringify2(_x_) #_x_
#define stringify(_x_) stringify2(_x_)

#if defined(CYGDAT_IO_FLASH_BLOCK_DEVICE_NAME_1)
# define JFFS2_TEST_DEV CYGDAT_IO_FLASH_BLOCK_DEVICE_NAME_1
#elif defined(CYGFUN_IO_FLASH_BLOCK_FROM_FIS)
# define JFFS2_TEST_DEV "/dev/flash/fis/jffs2test"
#else
// fall back to using a user set area in the first device (only)
# define JFFS2_TEST_DEV "/dev/flash/0/" stringify(CYGNUM_FS_JFFS2_TEST_OFFSET) "," stringify(CYGNUM_FS_JFFS2_TEST_LENGTH)
// But just in case, we check it's blank.
# define CHECK_FLASH_BLANK
#endif

//==========================================================================

#define SHOW_RESULT( _fn, _res ) \
diag_printf("FAIL: <" #_fn "() returned %d %s>\n", _res, _res<0?strerror(errno):"");

//==========================================================================
// prepare_device
//
// This does a quick check for existence of the underlying device.
// It also checks the region is blank if required.

void prepare_device( const char *dev )
{
    Cyg_ErrNo err;
    cyg_io_handle_t h;
    cyg_io_flash_getconfig_devsize_t devsize;
#ifdef CYGHWR_IO_FLASH_BLOCK_LOCKING
    cyg_io_flash_getconfig_unlock_t unlockconf;
#endif
    cyg_uint32 conf_size = sizeof(devsize);
    cyg_bool isblank = false;

    err = cyg_io_lookup( dev, &h );
    if (err == -ENOENT)
    {
        CYG_TEST_NA("Flash test device not found, test environment not set up by user.");
    }
    else if (err)
    {
        diag_printf("cyg_io_lookup returned: %d\n", err );
        CYG_TEST_FAIL_FINISH( "Test device lookup failed" );
    }

    err = cyg_io_get_config( h, CYG_IO_GET_CONFIG_FLASH_DEVSIZE, &devsize,
                             &conf_size );
    if (err)
    {
        CYG_TEST_FAIL_FINISH("prepare_device: cyg_io_get_config devsize failed");
    }

    // this check is really for sanity with the CHECK_FLASH_BLANK code, but
    // we may as well always do it.
    if ((devsize.dev_size % 128) != 0)
    {
        diag_printf("device size == 0x%08x\n",(int)devsize.dev_size);
        CYG_TEST_FAIL_FINISH("prepare_device: bizarre size returned - not multiple of 128");
    }
    
#ifdef CHECK_FLASH_BLANK
    {
        cyg_uint32 pos;
        cyg_uint8 buf[128];

        for (pos = 0; pos < devsize.dev_size; pos += sizeof(buf) )
        {
            cyg_uint32 i;
            cyg_uint32 len = sizeof(buf);
            cyg_uint32 *buf32 = (cyg_uint32 *)&buf[0];

            err = cyg_io_bread(h, &buf[0], &len, pos );
            if (err || (len < sizeof(buf)))
            {
                diag_printf("Read failure at %u, err=%d, new len=%u\n", pos, err, len );
                CYG_TEST_FAIL_FINISH("prepare_device: read failed");
            }
            for (i=0; i<(sizeof(buf)/4); i++)
            {
                if (buf32[i] != 0xFFFFFFFF)
                {
                    CYG_TEST_FAIL_FINISH("Supplied test device not blank! Not erasing.");
                }
            } // for
        } // for
        isblank = true;
    }
#endif    

#ifdef CYGHWR_IO_FLASH_BLOCK_LOCKING
    // This device might need unlocking before it can be written to.
    conf_size = sizeof(unlockconf);

    // unlock the whole "device"
    unlockconf.offset = 0;
    unlockconf.len = devsize.dev_size;
    err = cyg_io_get_config( h, CYG_IO_GET_CONFIG_FLASH_UNLOCK, &unlockconf,
                             &conf_size );
    if (err)
    {
        CYG_TEST_FAIL_FINISH("prepare_device: cyg_io_get_config unlock failed");
    }
    if (unlockconf.flasherr != CYG_FLASH_ERR_OK)
    {
        diag_printf("flash error @0x%08x:%d:%s\n", unlockconf.err_address,
                    unlockconf.flasherr,
                    cyg_flash_errmsg(unlockconf.flasherr) );
        CYG_TEST_FAIL_FINISH("prepare_device: unlock failed");
    }
#endif // ifdef CYGHWR_IO_FLASH_BLOCK_LOCKING

    if (!isblank)
    {
        // erase - this is our test device and we need it to be clean, rather
        // than containing a potentially corrupt FS.
        cyg_io_flash_getconfig_erase_t eraseconf;
        conf_size = sizeof(eraseconf);
        
        // erase the whole "device"
        eraseconf.offset = 0;
        eraseconf.len = devsize.dev_size;
        CYG_TEST_INFO("Erasing Flash test region");
        err = cyg_io_get_config( h, CYG_IO_GET_CONFIG_FLASH_ERASE, &eraseconf,
                                 &conf_size );
        if (err)
        {
            CYG_TEST_FAIL_FINISH("prepare_device: get_config for erase failed");
        }
        if (eraseconf.flasherr != CYG_FLASH_ERR_OK)
        {
            diag_printf("flash error @0x%08x:%d:%s\n", eraseconf.err_address,
                        eraseconf.flasherr,
                        cyg_flash_errmsg(eraseconf.flasherr) );
            CYG_TEST_FAIL_FINISH("prepare_device: erase failed");
        }
        CYG_TEST_INFO("Flash test region erase complete");
    }

    // flash block layer is known to need close when last instance stops using
    // handle.
    err = cyg_io_set_config( h, CYG_IO_SET_CONFIG_CLOSE, NULL, NULL );
    if (err)
    {
        CYG_TEST_FAIL_FINISH("prepare_device: CLOSE via cyg_io_set_config failed");
    }
}

//==========================================================================

static void listdir( char *name, int statp, int numexpected, int *numgot )
{
    int err;
    DIR *dirp;
    int num=0;
    
    diag_printf("<INFO>: reading directory %s\n",name);
    
    dirp = opendir( name );
    if( dirp == NULL ) SHOW_RESULT( opendir, -1 );

    for(;;)
    {
        struct dirent *entry = readdir( dirp );
        
        if( entry == NULL )
            break;
        num++;
        diag_printf("<INFO>: entry %14s",entry->d_name);
        if( statp )
        {
            char fullname[PATH_MAX];
            struct stat sbuf;

            if( name[0] )
            {
                strcpy(fullname, name );
                if( !(name[0] == '/' && name[1] == 0 ) )
                    strcat(fullname, "/" );
            }
            else fullname[0] = 0;
            
            strcat(fullname, entry->d_name );
            
            err = stat( fullname, &sbuf );
            if( err < 0 )
            {
                if( errno == ENOSYS )
                    diag_printf(" <no status available>");
                else SHOW_RESULT( stat, err );
            }
            else
            {
                diag_printf(" [mode %08x ino %08x nlink %d size %d]",
                            sbuf.st_mode,sbuf.st_ino,sbuf.st_nlink,(int)sbuf.st_size);
            }
        }

        diag_printf("\n");
    }

    err = closedir( dirp );
    if( err < 0 ) SHOW_RESULT( stat, err );
    if (numexpected >= 0 && num != numexpected)
        CYG_TEST_FAIL("Wrong number of dir entries\n");
    if ( numgot != NULL )
        *numgot = num;
}

//==========================================================================

static void createfile( char *name, size_t size )
{
    unsigned char buf[IOSIZE];
    int fd;
    ssize_t wrote;
    int i;
    int err;

    err = access( name, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );
    
    for( i = 0; i < IOSIZE; i++ ) buf[i] = 0;
 
    fd = open( name, O_WRONLY|O_CREAT );
    if( fd < 0 ) SHOW_RESULT( open, fd );

    while( size > 0 )
    {
        ssize_t len = size;
        if ( len > IOSIZE ) len = IOSIZE;
        
        wrote = write( fd, buf, len );
        if( wrote != len ) SHOW_RESULT( write, (int)wrote );        

        size -= wrote;
    }

    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );
}

//==========================================================================

static void checkfile( char *name, int thr )
{
    unsigned char buf[IOSIZE];
    int fd;
    ssize_t done = 0;
    int err;
    off_t pos = 0;
    int i;

    err = access( name, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );

    fd = open( name, O_RDWR );
    if( fd < 0 ) SHOW_RESULT( open, fd );

    for ( pos = (thr*IOSIZE); pos < FILESIZE; pos += (NTHREADS*IOSIZE) )
    {
        pos = lseek(fd, pos, SEEK_SET );
        if( pos < 0 ) SHOW_RESULT( lseek, (int)done );
        //        diag_printf("pos=%d\n",pos);
        for (i=0;i<IOSIZE;i++)
        {
            buf[i] = 1 + ((pos+i+thr)%255);
        }
        done = write(fd, &buf, IOSIZE);
        if ((pos % (FILESIZE/10)) == 0) {
            int existingdirents = -1;
            listdir( "/", true, -1, &existingdirents );
            if ( existingdirents < 3 )
                CYG_TEST_FAIL("Not enough dir entries\n");
        }

        if( done < 0 ) SHOW_RESULT( write, (int)done );
    }
    for ( pos = (thr*IOSIZE); pos < FILESIZE; pos += (NTHREADS*IOSIZE) )
    {
        pos = lseek(fd, pos, SEEK_SET );
        if( pos < 0 ) SHOW_RESULT( lseek, (int)done );

        done = read( fd, &buf, IOSIZE );
        if( done < 0 ) SHOW_RESULT( read, (int)done );

        if( done == 0 )
        {
            diag_printf("Got EOF at %u, thr=%d\n", (unsigned)pos, thr);
            CYG_TEST_FAIL("Unexpected EOF");
            break;
        }

        for (i=0; i<IOSIZE; i++)
        {
            if (buf[i] != (1 + ((pos+i+thr)%255)))
            {
                    diag_printf("buf[%d] at pos %u (0x%02x) != 0x%02x (thr=%d)\n",
                                i, (unsigned)pos, buf[i], (unsigned)(1 + ((pos+i+thr)%255)), thr);
                    CYG_TEST_FAIL("Data read not equal to data written");
            }
        }
    }

    // Put back to 0s
    for ( pos = (thr*IOSIZE); pos < FILESIZE; pos += (NTHREADS*IOSIZE) )
    {
        pos = lseek(fd, pos, SEEK_SET );
        if( pos < 0 ) SHOW_RESULT( lseek, (int)done );
        memset(buf, 0, IOSIZE);
        done = write(fd, &buf, IOSIZE);
        if( done < 0 ) SHOW_RESULT( write, (int)done );
    }
    // may as well check
    for ( pos = (thr*IOSIZE); pos < FILESIZE; pos += (NTHREADS*IOSIZE) )
    {
        pos = lseek(fd, pos, SEEK_SET );
        if( pos < 0 ) SHOW_RESULT( lseek, (int)done );

        done = read( fd, &buf, IOSIZE );
        if( done < 0 ) SHOW_RESULT( read, (int)done );

        if( done == 0 )
        {
            diag_printf("Got EOF at %u, thr=%d\n", (unsigned)pos, thr);
            CYG_TEST_FAIL("Unexpected EOF reading zeros");
            break;
        }

        for (i=0; i<IOSIZE; i++)
        {
            if (buf[i] != 0)
            {
                diag_printf("buf[%d] at pos %u (%02x) != 0 (thr=%d)\n",
                            i, (unsigned)pos, buf[i], thr);
                CYG_TEST_FAIL("Data read not zeroed");
            }
        }
    }
    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );
}

//==========================================================================

void checkcwd( const char *cwd )
{
    static char cwdbuf[PATH_MAX];
    char *ret;

    ret = getcwd( cwdbuf, sizeof(cwdbuf));
    if( ret == NULL ) SHOW_RESULT( getcwd, 0 );    

    if( strcmp( cwdbuf, cwd ) != 0 )
    {
        diag_printf( "cwdbuf %s cwd %s\n",cwdbuf, cwd );
        CYG_TEST_FAIL( "Current directory mismatch");
    }
}

//==========================================================================


static void
thr( CYG_ADDRWORD data )
{
    int i;

    if (data == 0) // thread 0 does init in thread context
    {
        int err;
        int existingdirents = -1;

        prepare_device( JFFS2_TEST_DEV );

        CYG_TEST_INFO("mount /");    
        err = mount( JFFS2_TEST_DEV, "/", "jffs2" );
        
        if( err < 0 ) SHOW_RESULT( mount, err );    

        err = chdir( "/" );
        if( err < 0 ) SHOW_RESULT( chdir, err );
        
        checkcwd( "/" );
        
        // create file full of 0s
        createfile( "testfile", FILESIZE );

        listdir( "/", true, -1, &existingdirents );
        if ( existingdirents < 3 )
            CYG_TEST_FAIL("Not enough dir entries\n");

        for (i=1;i<NTHREADS; i++)
            thrsema.post(); // wake up others
    } else {
        // wait for thread 0
        thrsema.wait();
    }

    i=1;
    while (i++ <= NLOOPS)
    {
        checkfile( "testfile", (int)data );
        diag_printf("INFO: <Checked file in thread %d %d times>\n",
                        data, i );
    }
    if (data == 0)
    {
        // wait for other threads to be done
        cyg_ucount8 i;
        for (i=1; i<NTHREADS; i++)
            thrsema.wait();
        CYG_TEST_PASS_FINISH(TESTNAME);
    }
    else
    {
        thrsema.post();
    }
}

externC void
cyg_start(void)
{
    int i;
#ifdef CYGSEM_HAL_STOP_CONSTRUCTORS_ON_FLAG
    cyg_hal_invoke_constructors();
#endif
    CYG_TEST_INIT();
    CYG_TEST_INFO("Starting " TESTNAME  " test");

    for (i=0; i<NTHREADS; i++)
    {
        new_thread(thr, i);
    }

    Cyg_Scheduler::start();

    CYG_TEST_FAIL_FINISH("Not reached");
} // cyg_start()

#endif // ifndef NA_MSG

// EOF jffs2-2t1f.cxx
