#ifndef CYGONCE_HAL_SAM9_INIT_INC
#define CYGONCE_HAL_SAM9_INIT_INC
/*=============================================================================
//
//      sam9_init.inc
//
//      Common SAM9 HAL initialisation helper macros (assembly code)
//
//=============================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007 Free Software Foundation, Inc.
// Copyright (C) 2003, 2004, 2005, 2006, 2007 eCosCentric Limited           
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//=============================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    eCosCentric
// Contributors: 
// Date:         2006-05-23
// Purpose:      Common SAM9 HAL initialisation helper macros
// Description:  The idea with this file is not to provide macros that must be
//               made to work for every SAM9 platform, but just try to
//               increase the amount of shared code. Therefore this is just
//               an optional library of macros, and platforms are able to do
//               their own thing instead.
//
//               There is only one exception which is CYGSEM_HAL_ARM_VECTOR_IRQ
//               and CYGSEM_HAL_ARM_VECTOR_FIQ, which are defined here as their
//               support exists in this package, but are dependent on
//               CYGINT_HAL_SAM9_AIC_VSR in any case.
//
// Usage:        #include <cyg/hal/sam9_init.inc>
//               Only used by platform "hal_platform_setup.h"
//
//####DESCRIPTIONEND####
//
//===========================================================================*/


#include <pkgconf/system.h>                  // System-wide configuration info
#include <pkgconf/hal_arm.h>
#include <pkgconf/hal_arm_arm9.h>
#include <pkgconf/hal_arm_arm9_sam9.h>

#include CYGBLD_HAL_VARIANT_H                // Variant specific configuration
#include <cyg/hal/hal_mmu.h>                 // MMU definitions

//===========================================================================*/

// This is derived from the AT91RM9200 HAL, in turn derived from the kb9200 HAL with
// additional information gleaned from Atmel example startup code, particularly
// AT91_Microcontroller/AT91RM9200-BasicsCompilation-ADS1_2-2_0/AT91RM9200-BasicBoot/init_ram.c
// from the CD supplied with the AT91RM9200-EK board kits.

//--------------------------------------------------------------------------------
// Set up the CPU
        
        .macro sam9_init_cpu
#if !defined(CYG_HAL_STARTUP_RAM)
        // Disable MMU, caches, and thus write buffer. Also sets little endian.
        // We daren't do this for RAM startup even if USE_ROM_MONITOR unset,
        // in case the MMU _was_ set up. e.g. if loading RAM RedBoot from ROM
        // RedBoot. This should be ok with JTAG as generally the CPU is reset
        // anyway (and so the MMU/caches were disabled).
        
        ldr     r0, =0x00000078
        mcr     p15,0,r0,c1,c0,0

        ldr     r0, =0x00000000
        mcr     p15,0,r0,c8,c7,0      // invalidate both i and d TLB's 
        mcr     p15,0,r0,c7,c7,0      // invalidate both i and d Caches 
        mcr     p15,0,r0,c7,c10,4     // Drain Write Buffer
        
        mvn     r0, #0                // grant manager access to all domains
        mcr     p15,0,r0,c3,c0,0  
#endif
        .endm

//--------------------------------------------------------------------------------
// Get the clocks running

        .macro sam9_init_clocks plla_val, pllb_val
#if !defined(CYG_HAL_STARTUP_RAM) || !defined(CYGSEM_HAL_USE_ROM_MONITOR)
        // Switch to the slow clock unless we already are (warm reset)
        // Note that when changing to the slow clock,
        // we should set the CSS first, wait for MCKRDY and only then set the
        // prescaler.

        ldr     r1,=_PMC_MCKR
        ldr     r2,[r1]
        cmp     r2,#0
        beq     20f

        // Don't change clock if already slow clock - part of workaround for erratum 27
        tst     r2,#_PMC_MCKR_CSS
        beq     5f

        bic     r2,r2,#_PMC_MCKR_CSS
        str     r2,[r1]
5:      
        // Wait for main clock to report ready
        ldr     r1,=_PMC_SR
10:     ldr     r2,[r1]
        tst     r2,#_PMC_SR_MCKRDY
        beq     10b

        // Don't change prescaler if already 0 - part of workaround for erratum 27
        tst     r2,#_PMC_MCKR_PRES
        beq     5f
        bic     r2,r2,#_PMC_MCKR_PRES
        str     r2,[r1]
5:
        // errata 27/28 don't mention problems with MDIV so just set it
        bic     r2,r2,#_PMC_MCKR_MDIV
        str     r2,[r1]
                
        // Turn on the main oscillator and wait for it to stabilise
        // by checking the MCFR
        // We conservatively set OSCOUNT to its maximum value
20:     
        ldr     r1,=_CKGR_MOR
        ldr     r2,=(_CKGR_MOR_OSCOUNT|_CKGR_MOR_MOSCEN)
        str     r2,[r1]
        ldr     r1,=_CKGR_MCFR
20:     ldr     r2,[r1]
        tst     r2,#_CKGR_MCFR_MAINRDY
        beq     20b

        // Cannot write PLLAR/PLLBR with its own current value (SAM9 erratum 26)
        // so disable first as long as it wasn't already disabled.
        ldr     r0,=_CKGR_PLLAR_DIVA_0
        ldr     r1,=_CKGR_PLLAR
        ldr     r2,[r1]
        cmp     r0,r2
        beq     10f
        str     r0,[r1]
10:     
//        ldr     r0,=_CKGR_PLLAR_DIVB_0 redundant since DIVA_0 == DIVB_0
        ldr     r1,=_CKGR_PLLBR
        ldr     r2,[r1]
        cmp     r0,r2
        beq     10f
        str     r0,[r1]
10:     

        // Enable PLLA at chosen frequency
        ldr     r1,=_CKGR_PLLAR
        ldr     r2,=\plla_val
        str     r2,[r1]

#ifndef SAM9_NO_PLLB
        // Enable PLLB at chosen frequency
        ldr     r1,=_CKGR_PLLBR
        ldr     r2,=\pllb_val
        str     r2,[r1]
#endif

        // Wait for PLLA and PLLB lock bits
        ldr     r1,=_PMC_SR
22:     ldr     r2,[r1]
#ifdef SAM9_NO_PLLB
        and     r2,r2,#(_PMC_IER_LOCKA)
        cmp     r2,#(_PMC_IER_LOCKA)
#else
        and     r2,r2,#(_PMC_IER_LOCKA|_PMC_IER_LOCKB)
        cmp     r2,#(_PMC_IER_LOCKA|_PMC_IER_LOCKB)
#endif
        bne     22b

        // Set MCK (used for peripherals) as processor clock/CYGARC_HAL_ARM_ARM9_SAM9_MCK_DIV
        // i.e. 180MHz/CYGARC_HAL_ARM_ARM9_SAM9_MCK_DIV
        // We don't bother with prescaler

        ldr     r1,=_PMC_MCKR
        ldr     r2,=(((CYGARC_HAL_ARM_ARM9_SAM9_MCK_DIV-1)<<8)|_PMC_MCKR_CSS_PLLA_CLK)
        str     r2,[r1]

        // Wait for MCK to report ready
        ldr     r1,=_PMC_SR
10:     ldr     r2,[r1]
        tst     r2,#_PMC_SR_MCKRDY
        beq     10b

        // Enable system clocks: USB host clock, USB device clock. No programmable clocks needed.

#ifndef SAM9_NO_PLLB
        ldr     r1,=_PMC_SCER
        ldr     r2,=(_PMC_SCER_UHP|_PMC_SCER_UDP)
        str     r2,[r1]
#endif

        // Set asynchronous clock mode
        mrc     p15,0,r0,c1,c0,0       // read cp15 control register
        mov     r3,#0xc0000000         // set asynchronous clock, no fast bus
        orr     r0,r0,r3
        mcr     p15,0,r0,c1,c0,0       // set in cp15 control reg
        
#endif // if !defined(CYG_HAL_STARTUP_RAM) || !defined(CYGSEM_HAL_USE_ROM_MONITOR)

        // Enable the clocks to all the on-chip peripherals
        // Each bit maps to the respective peripheral ID.
        // Note that top 7 peripheral ID bits are in fact
        // for the AIC so not enabled as clocks.

        ldr     r1,=_PMC_PCER
        ldr     r2,=0x01FFFFFC;
        str     r2,[r1]
        mov     r1,#1000
28:     sub     r1,r1,#1
        cmp     r1,#0
        bne     28b

        .endm

//-----------------------------------------------------------------------------
// SDRAM Initialization
//
// This will probably be common to all targets using SDRAM, apart from
// a few magic values.

        .macro sam9_init_sdram sdrc_cr, sdram_base_addr, sdram_refresh, sdram_mode

        // Do not initialize the SDRAM if it is already running,
        // such as when we are loaded into ram via JTAG, or booted
        // via the AT91BootStrap second stage loader

        ldr     r1,=(SAM9_MATRIX+AT91_MATRIX_EBICSA)
        ldr     r2,[r1]
        tst     r2,#AT91_MATRIX_EBICSA_CS1A_SDRAMC
        bne     sdram_running
        
        // Otherwise we can now init. The first stage of
        // which is to assign EBI CS1 to SDRAM in bus matrix.
        orr     r2,r2,#AT91_MATRIX_EBICSA_CS1A_SDRAMC
        str     r2,[r1]

        // Write SDRAM configuration register.
        ldr     r1,=AT91_SDRAMC_CR
        ldr     r2,=\sdrc_cr
        str     r2,[r1]

        // Issue 2 nops. Atmel don't mention this stage, but it seems
        // harmless enough.

        ldr     r1,=AT91_SDRAMC_MR
        ldr     r2,=AT91_SDRAMC_MR_MODE_NOP_CMD
        ldr     r3,=\sdram_base_addr
        str     r2,[r1]
        mov     r4,#0
        str     r4,[r3]
        str     r4,[r3]
        
        mov     r5,#104         // delay for a bit
30:     subs    r5,r5,#1
        bne     30b

       // issue precharge all

        ldr     r2,=AT91_SDRAMC_MR_MODE_PRCGALL_CMD
        str     r2,[r1]
        str     r4,[r3]
        
        mov     r5,#105         // delay for a bit
32:     subs    r5,r5,#1
        bne     32b

        // issue 8 refresh cycles

        ldr     r2,=AT91_SDRAMC_MR_MODE_RFSH_CMD
        str     r2,[r1]
        str     r4,[r3]
        str     r4,[r3]
        str     r4,[r3]
        str     r4,[r3]
        str     r4,[r3]
        str     r4,[r3]
        str     r4,[r3]
        str     r4,[r3]
        mov     r5,#100         // delay for a bit
34:     subs    r5,r5,#1
        bne     34b

       // issue mode register set

        ldr     r2,=AT91_SDRAMC_MR_MODE_LMR_CMD
        str     r2,[r1]
        str     r4,[r3,#\sdram_mode]

        // set normal mode

        ldr     r2,=AT91_SDRAMC_MR_MODE_NORMAL_CMD
        str     r2,[r1]
        str     r4,[r3]

        // set refresh.

        ldr     r1,=AT91_SDRAMC_TR
        ldr     r2,=\sdram_refresh
        str     r2,[r1]

sdram_running:
        .endm

//--------------------------------------------------------------------------------
// MMU setup, and handle ROM->RAM copy in ROMRAM startups
// the argument values are irrelevant if ROMRAM unsupported
//
// It is assumed that:
//  * __startup_stack is defined and corresponds to the startup stack's address
//    after relocation, which will be in ram starting at 0x0 (after mmu setup).
//  * hal_mmu_init is to be called to do the real init
//  * __exception_handlers is defined and sensible and corresponds to ram
//    starting at 0x0 (after MMU setup)
//  * __rom_data_end is defined and sensible

#define CYGHWR_HAL_ARM_HAS_MMU

        .macro sam9_init_mmu image_base_mask, rom_image_virt_base, ram_phys_base
#ifdef CYG_HAL_STARTUP_ROMRAM
        // Compute [logical] base address of this image in ROM
        bl      10f
10:     mov     r9,lr
        ldr     r8,=~\image_base_mask        // Bits to ignore
        and     r9,r9,r8
        orr     r9,r9,#\rom_image_virt_base  // Turn into ROM address
#endif        

        // Test whether MMU is already running. If it is then we skip
        // over the setup code.
        ldr     r1,=MMU_Control_M
        mrc     MMU_CP,0,r2,MMU_Control,c0
        and     r1,r1,r2
        cmp     r1,#0
        bne     sam9_mmu_ok

        // Set up a stack [for calling C code]
        ldr     r1,=__startup_stack
        ldr     r2,=\ram_phys_base
        orr     sp,r1,r2

        // Create MMU tables
        bl      hal_mmu_init

        // Enable MMU
        ldr     r2,=10f
#ifdef CYG_HAL_STARTUP_ROMRAM
        ldr     r1,=__exception_handlers
        sub     r1,r2,r1
        add     r2,r9,r1        // r9 has ROM offset
#endif        
        ldr    r1,=MMU_Control_Init|MMU_Control_M
        mcr    MMU_CP,0,r1,MMU_Control,c0
        mov    pc,r2    /* Change address spaces */
        nop
        nop
        nop
10:

sam9_mmu_ok:
        
#ifdef CYG_HAL_STARTUP_ROMRAM
        mov     r0,r9                     // Relocate FLASH/ROM to RAM
        ldr     r1,=__exception_handlers  // ram base & length
        ldr     r2,=__rom_data_end
20:     ldr     r3,[r0],#4
        str     r3,[r1],#4
        cmp     r1,r2
        bne     20b
        ldr     r0,=30f
        mov     pc,r0
        nop
        nop
        nop
        nop
30:             
#endif
        .endm

//--------------------------------------------------------------------------------
// Set up PIO ports
//
//                 

        .macro  sam9_pio_set_port port
        ldr     r1,=\port               // Set up pointer to PIO port
        .endm
                
        .macro sam9_pio_set_output mask
        ldr     r2,=\mask
        str     r2,[r1,#_PIO_PER]       // Enable PIO mode on selected lines
        str     r2,[r1,#_PIO_OER]       // Enable output mode
        .endm

        .macro sam9_pio_set_input mask
        ldr     r2,=\mask
        str     r2,[r1,#_PIO_PER]       // Enable PIO mode on selected lines
        str     r2,[r1,#_PIO_ODR]       // Enable input mode
        .endm

        .macro sam9_pio_set_interrupt mask
        ldr     r2,=\mask
        str     r2,[r1,#_PIO_PER]       // Enable PIO mode on selected lines
        str     r2,[r1,#_PIO_ODR]       // Enable input mode
        str     r2,[r1,#_PIO_IER]       // Enable interrupt on selected lines
        .endm

        .macro sam9_pio_set_perA mask
        ldr     r2,=\mask
        str     r2,[r1,#_PIO_ASR]       // Enable peripheral A mode
        str     r2,[r1,#_PIO_PDR]       // Disble PIO mode on selected lines
        .endm

        .macro sam9_pio_set_perB mask
        ldr     r2,=\mask
        str     r2,[r1,#_PIO_BSR]       // Enable peripheral B mode
        str     r2,[r1,#_PIO_PDR]       // Disble PIO mode on selected lines
        .endm
        

//--------------------------------------------------------------------------------        
// Install macros to make use of hardware vectors
//
#if CYGINT_HAL_SAM9_AIC_VSR
        .macro  irq_vector
        ldr     pc,[pc, #-(_AIC_IVR_OFFSET+0x20)]
        .endm

        .macro  fiq_vector
        ldr     pc,[pc, #-(_AIC_FVR_OFFSET+0x24)]
        .endm

# define CYGSEM_HAL_ARM_VECTOR_IRQ
# define CYGSEM_HAL_ARM_VECTOR_FIQ
#endif

#endif // ifndef CYGONCE_HAL_SAM9_INIT_INC
//-----------------------------------------------------------------------------
// EOF sam9_init.inc
