/*=============================================================================
//
//      hal_diag.c
//
//      HAL diagnostic output code
//
//=============================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004, 2005, 2006, 2007, 2008 Free Software Foundation, Inc.
// Copyright (C) 2004, 2005, 2006, 2007, 2008 eCosCentric Limited           
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//=============================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):   jskov
// Contributors:jskov, gthomas
// Date:        2001-07-12
// Purpose:     HAL diagnostic output
// Description: Implementations of HAL diagnostic output support.
//
//####DESCRIPTIONEND####
//
//===========================================================================*/

#include <pkgconf/hal.h>
#include CYGBLD_HAL_PLATFORM_H

#include <cyg/infra/cyg_type.h>         // base types

#include <cyg/hal/hal_arch.h>           // SAVE/RESTORE GP macros
#include <cyg/hal/hal_io.h>             // IO macros
#include <cyg/hal/hal_if.h>             // interface API
#include <cyg/hal/hal_intr.h>           // HAL_ENABLE/MASK/UNMASK_INTERRUPTS
#include <cyg/hal/hal_misc.h>           // Helper functions
#include <cyg/hal/drv_api.h>            // CYG_ISR_HANDLED

#include <cyg/hal/var_io.h>             // USART registers

#include "hal_diag_dcc.h"               // DCC initialization file
//-----------------------------------------------------------------------------

#ifndef AT91_PDC_RX_ENABLE
#define AT91_PDC_RX_ENABLE( __base ) CYG_EMPTY_STATEMENT
#define AT91_PDC_RX_DISABLE( __base ) CYG_EMPTY_STATEMENT
#define AT91_PDC_TX_ENABLE( __base ) CYG_EMPTY_STATEMENT
#define AT91_PDC_TX_DISABLE( __base ) CYG_EMPTY_STATEMENT
#endif

#define NUMOF(t) (sizeof(t)/sizeof(t[0]))

//-----------------------------------------------------------------------------

typedef struct {
    cyg_uint8   *base;
    cyg_int32   msec_timeout;
    int         isr_vector;
    int         baud_rate;    
    char        rcv_buffer[2][1];
    int         rcv_current;
    int         rcv_pos;
    int         rcv_end;
} channel_data_t;

//-----------------------------------------------------------------------------

static channel_data_t at91_ser_channels[] = {
#if CYGINT_HAL_ARM_AT91_SERIAL_DEBUG
    { (cyg_uint8 *)AT91_DEBUG, 1000, CYGNUM_HAL_INTERRUPT_DEBUG,  CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD },
#endif    
#if CYGINT_HAL_ARM_AT91_SERIAL_UART0
    { (cyg_uint8*)AT91_USART0, 1000, CYGNUM_HAL_INTERRUPT_USART0, CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD },
#endif
#if CYGINT_HAL_ARM_AT91_SERIAL_UART1
    { (cyg_uint8*)AT91_USART1, 1000, CYGNUM_HAL_INTERRUPT_USART1, CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD },
#endif
#if CYGINT_HAL_ARM_AT91_SERIAL_UART2
    { (cyg_uint8*)AT91_USART2, 1000, CYGNUM_HAL_INTERRUPT_USART2, CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD },
#endif
#if CYGINT_HAL_ARM_AT91_SERIAL_UART3
    { (cyg_uint8*)AT91_USART3, 1000, CYGNUM_HAL_INTERRUPT_USART3, CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD },
#endif
};

//-----------------------------------------------------------------------------

void
cyg_hal_plf_serial_putc(void *__ch_data, char c);

static void
cyg_hal_plf_serial_init_channel(void* __ch_data)
{
    channel_data_t* chan = (channel_data_t*)__ch_data;    
    cyg_uint8 *base = ((channel_data_t*)__ch_data)->base;

    // Reset device
    HAL_WRITE_UINT32(base+AT91_US_CR, AT91_US_CR_RxRESET | AT91_US_CR_TxRESET);
    AT91_US_WRITE_IDR( base, 0xFFFFFFFF );
    HAL_WRITE_UINT32(base+AT91_US_IMR, 0x00000000);

    // 8-1-no parity.
    HAL_WRITE_UINT32(base+AT91_US_MR,
                     AT91_US_MR_CLOCK_MCK | AT91_US_MR_LENGTH_8 |
                     AT91_US_MR_PARITY_NONE | AT91_US_MR_STOP_1);

    HAL_WRITE_UINT32(base+AT91_US_BRG, AT91_US_BAUD(chan->baud_rate));

    // Set up PDC to receive data into the first buffer
    AT91_US_WRITE_RPR( base, (cyg_uint32)chan->rcv_buffer[0]);
    AT91_US_WRITE_RCR( base, sizeof(chan->rcv_buffer[0]));
    AT91_PDC_RX_ENABLE( base );
    chan->rcv_current = 1;
    chan->rcv_pos = 0;
    chan->rcv_end = 0;

    
    // Enable RX and TX
    HAL_WRITE_UINT32(base+AT91_US_CR, AT91_US_CR_RxENAB | AT91_US_CR_TxENAB | AT91_US_CR_RTSEN);
}

void
cyg_hal_plf_serial_putc(void *__ch_data, char c)
{
    cyg_uint8* base = ((channel_data_t*)__ch_data)->base;
    cyg_uint32 status, ch;
    CYGARC_HAL_SAVE_GP();

    do {
        HAL_READ_UINT32(base+AT91_US_CSR, status);
    } while ((status & AT91_US_CSR_TxRDY) == 0);

    ch = (cyg_uint32)c;
    HAL_WRITE_UINT32(base+AT91_US_THR, ch);

    CYGARC_HAL_RESTORE_GP();
}

static cyg_bool
cyg_hal_plf_serial_getc_nonblock(void* __ch_data, cyg_uint8* ch)
{
    channel_data_t* chan = (channel_data_t*)__ch_data;
    cyg_uint8* base = chan->base;
    cyg_uint8 *c;
    cyg_uint32 end;
    cyg_uint32 status;

    if( chan->rcv_pos >= chan->rcv_end )
    {
        // If there is no data waiting in the current buffer, try to
        // swap the buffers over to receive any data that is waiting
        // in the buffer the hardware is currently using.
        AT91_PDC_RX_DISABLE( base );
        AT91_US_READ_RPR(base, end);
        if( end == (cyg_uint32)chan->rcv_buffer[chan->rcv_current^1] )
        {
            // If there is no data in the buffer yet, don't swap and just
            // report no data available.
            AT91_PDC_RX_ENABLE( base );
            return false;
        }

        // Point the PDC at the current, now empty, buffer.
        AT91_US_WRITE_RPR( base, (cyg_uint32)chan->rcv_buffer[chan->rcv_current]);
        AT91_US_WRITE_RCR( base, sizeof(chan->rcv_buffer[chan->rcv_current]));
        AT91_PDC_RX_ENABLE( base );

        // Clear the status register
        HAL_READ_UINT32(base+AT91_US_CSR, status);
        
        // Set up to extract bytes from the buffer we just got back
        // from the hardware.
        chan->rcv_current ^= 1;
        chan->rcv_pos = 0;
        chan->rcv_end = end - (cyg_uint32)chan->rcv_buffer[chan->rcv_current];

        // If the buffer is empty, report no data available.
        if( chan->rcv_end == 0 )
            return false;
    }

    // Fetch a byte from the (uncached) version of the buffer and return it.
    c = &chan->rcv_buffer[chan->rcv_current][chan->rcv_pos];
    chan->rcv_pos++;
    
    *ch = *c;
    
    return true;
}

#if defined(CYGSEM_HAL_VIRTUAL_VECTOR_DIAG) || defined(CYGSEM_HAL_VIRTUAL_VECTOR_CLAIM_COMMS)
cyg_uint8
cyg_hal_plf_serial_getc(void* __ch_data)
{
    cyg_uint8 ch;
    CYGARC_HAL_SAVE_GP();

    while(!cyg_hal_plf_serial_getc_nonblock(__ch_data, &ch));

    CYGARC_HAL_RESTORE_GP();
    return ch;
}

static void
cyg_hal_plf_serial_write(void* __ch_data, const cyg_uint8* __buf, 
                         cyg_uint32 __len)
{
    CYGARC_HAL_SAVE_GP();

    while(__len-- > 0)
        cyg_hal_plf_serial_putc(__ch_data, *__buf++);

    CYGARC_HAL_RESTORE_GP();
}

static void
cyg_hal_plf_serial_read(void* __ch_data, cyg_uint8* __buf, cyg_uint32 __len)
{
    CYGARC_HAL_SAVE_GP();

    while(__len-- > 0)
        *__buf++ = cyg_hal_plf_serial_getc(__ch_data);

    CYGARC_HAL_RESTORE_GP();
}

cyg_bool
cyg_hal_plf_serial_getc_timeout(void* __ch_data, cyg_uint8* ch)
{
    int delay_count;
    channel_data_t* chan = (channel_data_t*)__ch_data;
    cyg_bool res;
    CYGARC_HAL_SAVE_GP();

    delay_count = chan->msec_timeout * 10; // delay in .1 ms steps

    for(;;) {
        res = cyg_hal_plf_serial_getc_nonblock(__ch_data, ch);
        if (res || 0 == delay_count--)
            break;
        
        CYGACC_CALL_IF_DELAY_US(100);
    }

    CYGARC_HAL_RESTORE_GP();
    return res;
}

static int
cyg_hal_plf_serial_control(void *__ch_data, __comm_control_cmd_t __func, ...)
{
    static int irq_state = 0;
    channel_data_t* chan = (channel_data_t*)__ch_data;
    cyg_uint8* base = ((channel_data_t*)__ch_data)->base;
    int ret = 0;
    CYGARC_HAL_SAVE_GP();

    switch (__func) {
    case __COMMCTL_IRQ_ENABLE:
        irq_state = 1;
        HAL_INTERRUPT_ACKNOWLEDGE(chan->isr_vector);
        HAL_INTERRUPT_UNMASK(chan->isr_vector);
        AT91_US_WRITE_IER( base, AT91_US_IER_ENDRX );                
        break;
    case __COMMCTL_IRQ_DISABLE:
        ret = irq_state;
        irq_state = 0;
        HAL_INTERRUPT_MASK(chan->isr_vector);
        AT91_US_WRITE_IDR( base, AT91_US_IER_ENDRX );        
        break;
    case __COMMCTL_GETBAUD:
        ret = chan->baud_rate;
        break;
    case __COMMCTL_SETBAUD:
    {
        cyg_uint32 b;
        va_list ap;

        va_start(ap, __func);
        b = va_arg(ap, cyg_int32);
        va_end(ap);

        chan->baud_rate = b;
        HAL_WRITE_UINT32(base+AT91_US_BRG, AT91_US_BAUD(b));
        ret = 1;
        break;
    }
    case __COMMCTL_DBG_ISR_VECTOR:
        ret = chan->isr_vector;
        break;
    case __COMMCTL_SET_TIMEOUT:
    {
        va_list ap;

        va_start(ap, __func);

        ret = chan->msec_timeout;
        chan->msec_timeout = va_arg(ap, cyg_uint32);

        va_end(ap);
    }        
    default:
        break;
    }
    CYGARC_HAL_RESTORE_GP();
    return ret;
}

static int
cyg_hal_plf_serial_isr(void *__ch_data, int* __ctrlc, 
                       CYG_ADDRWORD __vector, CYG_ADDRWORD __data)
{
    channel_data_t* chan = (channel_data_t*)__ch_data;    
    int res = 0;
    cyg_uint8 c;

    CYGARC_HAL_SAVE_GP();

    *__ctrlc = 0;
    if( cyg_hal_plf_serial_getc_nonblock( __ch_data, &c ) )
    {
        if( cyg_hal_is_break( &c , 1 ) )
            *__ctrlc = 1;
        res = CYG_ISR_HANDLED;
    }

    HAL_INTERRUPT_ACKNOWLEDGE(chan->isr_vector);

    CYGARC_HAL_RESTORE_GP();
    return res;
}

static void
cyg_hal_plf_serial_init(void)
{
    hal_virtual_comm_table_t* comm;
    int cur = CYGACC_CALL_IF_SET_CONSOLE_COMM(CYGNUM_CALL_IF_SET_COMM_ID_QUERY_CURRENT);
    int i;
    
    for (i = 0;  i < NUMOF(at91_ser_channels);  i++)
    {
        // Init channel
        cyg_hal_plf_serial_init_channel(&at91_ser_channels[i]);

        // Setup procs in the vector table
        CYGACC_CALL_IF_SET_CONSOLE_COMM(i);
        comm = CYGACC_CALL_IF_CONSOLE_PROCS();
        CYGACC_COMM_IF_CH_DATA_SET(*comm, &at91_ser_channels[i]);
        CYGACC_COMM_IF_WRITE_SET(*comm, cyg_hal_plf_serial_write);
        CYGACC_COMM_IF_READ_SET(*comm, cyg_hal_plf_serial_read);
        CYGACC_COMM_IF_PUTC_SET(*comm, cyg_hal_plf_serial_putc);
        CYGACC_COMM_IF_GETC_SET(*comm, cyg_hal_plf_serial_getc);
        CYGACC_COMM_IF_CONTROL_SET(*comm, cyg_hal_plf_serial_control);
        CYGACC_COMM_IF_DBG_ISR_SET(*comm, cyg_hal_plf_serial_isr);
        CYGACC_COMM_IF_GETC_TIMEOUT_SET(*comm, cyg_hal_plf_serial_getc_timeout);
    }

    // Restore original console
    CYGACC_CALL_IF_SET_CONSOLE_COMM(cur);
}

void
cyg_hal_plf_comms_init(void)
{
    static int initialized = 0;

    if (initialized)
        return;

    initialized = 1;

    cyg_hal_plf_serial_init();

#ifdef CYGBLD_HAL_ARM_AT91_DCC
    cyg_hal_plf_dcc_init(CYGBLD_HAL_ARM_AT91_DCC_CHANNEL);
#endif
}
#endif // if defined(CYGSEM_HAL_VIRTUAL_VECTOR_DIAG) || defined(CYGSEM_HAL_VIRTUAL_VECTOR_CLAIM_COMMS)

#ifndef CYGSEM_HAL_VIRTUAL_VECTOR_DIAG

#ifdef CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL
# define CHAN CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL
#else
# define CHAN 0
#endif

__externC void
hal_at91_diag_init(void)
{
    if (CHAN <= NUMOF(at91_ser_channels)) // this check should compile out
    {
        cyg_hal_plf_serial_init_channel(&at91_ser_channels[CHAN]);
    }
}

__externC void
hal_at91_diag_write_char(char c)
{
    if (CHAN <= NUMOF(at91_ser_channels)) // this check should compile out
    {
        cyg_hal_plf_serial_putc(&at91_ser_channels[CHAN], c);
    }
}

__externC void
hal_at91_diag_read_char(char *c)
{
    if (CHAN <= NUMOF(at91_ser_channels)) // this check should compile out
    {
        while (!cyg_hal_plf_serial_getc_nonblock(&at91_ser_channels[CHAN], (cyg_uint8*)c))
            CYG_EMPTY_STATEMENT;
    }
}
#endif

void
hal_diag_led(int mask)
{
    extern void hal_at91_set_leds(int);
    hal_at91_set_leds(mask);
}

//-----------------------------------------------------------------------------
// End of hal_diag.c
