/*==========================================================================
//
//      lpc2xxx_intr.c
//
//      HAL interrupt support code for Philips LPC2xxx variants.
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2008 Free Software Foundation, Inc.
// Copyright (C) 2004, 2008 eCosCentric Limited                             
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    jlarmour
// Contributors: gthomas, jskov, nickg, tkoeller
// Date:         2004-07-23
// Purpose:      HAL board support
// Description:  Implementations of HAL board interrupt interfaces
//
//####DESCRIPTIONEND####
//
//========================================================================*/

#include <cyg/infra/cyg_type.h>         // base types
#include <cyg/infra/cyg_trac.h>         // tracing macros
#include <cyg/infra/cyg_ass.h>          // assertion macros

#include <cyg/hal/hal_io.h>             // IO macros and variant registers
#include <cyg/hal/hal_arch.h>           // Register state info
//#include <cyg/hal/hal_diag.h>
#include <cyg/hal/hal_intr.h>           // interrupt defines

// -------------------------------------------------------------------------
// This routine is called to respond to a hardware interrupt (IRQ).  It
// should interrogate the hardware and return the IRQ vector number.

int hal_IRQ_handler(void)
{
    // We do not support vectored interrupts at this point.
    // So all we do is read the IRQ status register

    cyg_uint32 irq_status;
    cyg_uint32 irq;
    
    HAL_READ_UINT32( CYGARC_HAL_LPC2XXX_REG_VIC_BASE + CYGARC_HAL_LPC2XXX_REG_VICIRQSTAT,
                     irq_status );
    
    HAL_LSBIT_INDEX( irq, irq_status );
    // No valid interrrupt source, treat as spurious interrupt    
    if (irq < CYGNUM_HAL_ISR_MIN || irq > CYGNUM_HAL_ISR_MAX)
      irq = CYGNUM_HAL_INTERRUPT_NONE;
    return irq;
}

// -------------------------------------------------------------------------
// Interrupt control
//

void hal_interrupt_mask(int vector)
{
    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN , "Invalid vector");

    HAL_WRITE_UINT32( CYGARC_HAL_LPC2XXX_REG_VIC_BASE + CYGARC_HAL_LPC2XXX_REG_VICINTENCLEAR,
                      1<<vector );
}

void hal_interrupt_unmask(int vector)
{
    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN , "Invalid vector");

    HAL_WRITE_UINT32( CYGARC_HAL_LPC2XXX_REG_VIC_BASE + CYGARC_HAL_LPC2XXX_REG_VICINTENABLE,
                      1<<vector );
}

void hal_interrupt_acknowledge(int vector)
{
    // No check for valid vector here! Spurious interrupts
    // must be acknowledged, too.

    // For on-chip peripherals, clearing the int in the device should be sufficient.
    // External interrupts must be handled explicitly.
    if (vector >= CYGNUM_HAL_INTERRUPT_EINT0 &&
        vector <= CYGNUM_HAL_INTERRUPT_EINT3)
    {
        HAL_WRITE_UINT8( CYGARC_HAL_LPC2XXX_REG_SCB_BASE + CYGARC_HAL_LPC2XXX_REG_EXTINT,
                         1<<(vector - CYGNUM_HAL_INTERRUPT_EINT0) );
    }    

    // The VIC usage notes say that the internal priority hardware must be updated
    // after processing the ISR. Unclear whether this is required if not using
    // priority based (i.e. vectored) interrupts, but we write it here anyway.
    HAL_WRITE_UINT32( CYGARC_HAL_LPC2XXX_REG_VIC_BASE + CYGARC_HAL_LPC2XXX_REG_VICADDR,
                      1<<vector );
}

void hal_interrupt_configure(int vector, int level, int up)
{
    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN , "Invalid vector");
    
    // Only applies to external i.e. off-chip interrupts

#if !(defined(CYGHWR_HAL_ARM_LPC2XXX_LPC2468) || \
      defined(CYGHWR_HAL_ARM_LPC2XXX_LPC2478))

    if (vector >= CYGNUM_HAL_INTERRUPT_EINT0 &&
        vector <= CYGNUM_HAL_INTERRUPT_EINT3)
    {
        cyg_uint32 reg;
        cyg_uint32 eintnum = vector - CYGNUM_HAL_INTERRUPT_EINT0;
        cyg_uint32 vpbdiv;

        HAL_READ_UINT32( CYGARC_HAL_LPC2XXX_REG_SCB_BASE+CYGARC_HAL_LPC2XXX_REG_VPBDIV, vpbdiv );
        HAL_READ_UINT32( CYGARC_HAL_LPC2XXX_REG_SCB_BASE+CYGARC_HAL_LPC2XXX_REG_VPBDIV, vpbdiv );

        HAL_WRITE_UINT32( CYGARC_HAL_LPC2XXX_REG_SCB_BASE+CYGARC_HAL_LPC2XXX_REG_VPBDIV, 0 );

        // set level versus edge sensitive. read-modify-write
        HAL_READ_UINT32( CYGARC_HAL_LPC2XXX_REG_SCB_BASE + CYGARC_HAL_LPC2XXX_REG_EXTMODE,
                        reg );
        if ( level )
            reg &= ~(1<<eintnum);
        else
            reg |= (1<<eintnum);
        HAL_WRITE_UINT8( CYGARC_HAL_LPC2XXX_REG_SCB_BASE + CYGARC_HAL_LPC2XXX_REG_EXTMODE,
                         reg );
        HAL_WRITE_UINT32( CYGARC_HAL_LPC2XXX_REG_SCB_BASE+CYGARC_HAL_LPC2XXX_REG_VPBDIV, reg );
        

        HAL_WRITE_UINT32( CYGARC_HAL_LPC2XXX_REG_SCB_BASE+CYGARC_HAL_LPC2XXX_REG_VPBDIV, 0 );

        // set edge/level polarity. read-modify-write
        HAL_READ_UINT8( CYGARC_HAL_LPC2XXX_REG_SCB_BASE + CYGARC_HAL_LPC2XXX_REG_EXTPOLAR,
                        reg );

        if ( !up )
            reg &= ~(1<<eintnum);
        else
            reg |= (1<<eintnum);
        HAL_WRITE_UINT8( CYGARC_HAL_LPC2XXX_REG_SCB_BASE + CYGARC_HAL_LPC2XXX_REG_EXTPOLAR,
                         reg );
        HAL_WRITE_UINT32( CYGARC_HAL_LPC2XXX_REG_SCB_BASE+CYGARC_HAL_LPC2XXX_REG_VPBDIV, reg );

        HAL_WRITE_UINT32( CYGARC_HAL_LPC2XXX_REG_SCB_BASE+CYGARC_HAL_LPC2XXX_REG_VPBDIV, 0 );        
        HAL_WRITE_UINT32( CYGARC_HAL_LPC2XXX_REG_SCB_BASE+CYGARC_HAL_LPC2XXX_REG_VPBDIV, vpbdiv );        
    }

#else

    if (vector >= CYGNUM_HAL_INTERRUPT_EINT0 &&
        vector <= CYGNUM_HAL_INTERRUPT_EINT3)
    {
        cyg_uint32 reg;
        cyg_uint32 eintnum = vector - CYGNUM_HAL_INTERRUPT_EINT0;

        // set level versus edge sensitive. read-modify-write
        HAL_READ_UINT32( CYGARC_HAL_LPC2XXX_REG_SCB_BASE + CYGARC_HAL_LPC2XXX_REG_EXTMODE,
                        reg );
        if ( level )
            reg &= ~(1<<eintnum);
        else
            reg |= (1<<eintnum);
        HAL_WRITE_UINT8( CYGARC_HAL_LPC2XXX_REG_SCB_BASE + CYGARC_HAL_LPC2XXX_REG_EXTMODE,
                         reg );
        

        // set edge/level polarity. read-modify-write
        HAL_READ_UINT8( CYGARC_HAL_LPC2XXX_REG_SCB_BASE + CYGARC_HAL_LPC2XXX_REG_EXTPOLAR,
                        reg );

        if ( !up )
            reg &= ~(1<<eintnum);
        else
            reg |= (1<<eintnum);
        
        HAL_WRITE_UINT8( CYGARC_HAL_LPC2XXX_REG_SCB_BASE + CYGARC_HAL_LPC2XXX_REG_EXTPOLAR,
                         reg );

    }

#endif
    
}

void hal_interrupt_set_level(int vector, int level)
{
    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN , "Invalid vector");

#if defined(CYGHWR_HAL_ARM_LPC2XXX_LPC2468) || \
    defined(CYGHWR_HAL_ARM_LPC2XXX_LPC2478)

    HAL_WRITE_UINT32( CYGARC_HAL_LPC2XXX_REG_VIC_BASE + CYGARC_HAL_LPC2XXX_REG_VICVECTPRI(vector),
                      15-level );
    
#else
    // No vectored interrupts, so no support for priorities
#endif
}

void hal_show_IRQ(int vector, int data, int handler)
{
    //diag_printf("IRQ - vector: %x, data: %x, handler: %x\n", vector, data, handler);
}


#ifndef CYGSEM_HAL_VIRTUAL_VECTOR_SUPPORT
// Need our own default ISR

__externC cyg_uint32 hal_arch_default_isr(CYG_ADDRWORD vector, CYG_ADDRWORD data);

cyg_uint32
hal_default_isr(CYG_ADDRWORD vector, CYG_ADDRWORD data)
{
    return hal_arch_default_isr (vector, data);
}
#endif

//--------------------------------------------------------------------------
// EOF lpc2xxx_intr.c
