/*=============================================================================
//
//      hal_diag.c
//
//      HAL diagnostic output code
//
//=============================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004, 2005, 2006 Free Software Foundation, Inc.
// Copyright (C) 2004, 2005, 2006 eCosCentric Limited                       
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//=============================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):   nickg
// Contributors:jskov, gthomas, jlarmour
// Date:        2006-03-20
// Purpose:     HAL diagnostic output
// Description: Implementations of HAL diagnostic output support.
//
//####DESCRIPTIONEND####
//
//===========================================================================*/

#include <pkgconf/hal.h>
#include CYGBLD_HAL_PLATFORM_H

#include <cyg/infra/cyg_type.h>         // base types
#include <cyg/infra/cyg_trac.h>         // tracing

#include <cyg/hal/hal_arch.h>           // SAVE/RESTORE GP macros
#include <cyg/hal/hal_io.h>             // IO macros
#include <cyg/hal/hal_if.h>             // interface API
#include <cyg/hal/hal_intr.h>           // HAL_ENABLE/MASK/UNMASK_INTERRUPTS
#include <cyg/hal/hal_misc.h>           // Helper functions
#include <cyg/hal/drv_api.h>            // CYG_ISR_HANDLED

#include <cyg/hal/var_io.h>             // USART registers

//-----------------------------------------------------------------------------

#define BUF_SIZE        64

typedef struct {
    cyg_uint32          uart;
    cyg_uint8*          base;
    cyg_int32           msec_timeout;
    int                 isr_vector;
    int                 pio;
    int                 rxbit;
    int                 txbit;
    cyg_uint32          baud_rate;
    cyg_uint8           buf[BUF_SIZE];
    int                 buf_head;
    int                 buf_tail;
    int                 buf_count;
} channel_data_t;

static channel_data_t str7xx_ser_channels[] = {
#if CYGINT_HAL_STR7XX_UART0>0
    { 0, (cyg_uint8*)CYGARC_HAL_STR7XX_UART0_BASE, 1000, CYGNUM_HAL_INTERRUPT_UART0, 0,  8,  9 },
#endif
#if CYGINT_HAL_STR7XX_UART1>0    
    { 1, (cyg_uint8*)CYGARC_HAL_STR7XX_UART1_BASE, 1000, CYGNUM_HAL_INTERRUPT_UART1, 0, 10, 11 },
#endif
#if CYGINT_HAL_STR7XX_UART2>0    
    { 2, (cyg_uint8*)CYGARC_HAL_STR7XX_UART2_BASE, 1000, CYGNUM_HAL_INTERRUPT_UART2, 0, 13, 14 },
#endif
#if CYGINT_HAL_STR7XX_UART3>0    
    { 3, (cyg_uint8*)CYGARC_HAL_STR7XX_UART3_BASE, 1000, CYGNUM_HAL_INTERRUPT_UART3, 0, 1, 0 },
#endif
};

//-----------------------------------------------------------------------------

void
cyg_hal_plf_serial_putc(void *__ch_data, char c);

static void
cyg_hal_plf_serial_init_channel(void* __ch_data)
{
    cyg_uint32 cr;
    channel_data_t *chan = (channel_data_t*)__ch_data;
    cyg_uint8* base = chan->base; 

    // Enable the PIO lines for the serial channel

    CYGARC_HAL_STR7XX_PIO_SET( chan->pio, chan->txbit, CYGARC_HAL_STR7XX_PIO_PP_AF );
    CYGARC_HAL_STR7XX_PIO_SET( chan->pio, chan->rxbit, CYGARC_HAL_STR7XX_PIO_TRI_CMOS_IN );
//    CYGARC_HAL_STR7XX_PIO_SET( chan->pio, chan->rxbit, CYGARC_HAL_STR7XX_PIO_TRI_TTL_IN );    

    // Set up Baud rate
    
    chan->baud_rate = CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD;
//    update_baud_rate( chan );
    hal_str7xx_uart_setbaud( chan->uart, chan->baud_rate );
    
     // disable interrupts on device and disable device
     HAL_WRITE_UINT32( base+CYGARC_HAL_STR7XX_UART_IER, 0 );
     HAL_WRITE_UINT32( base+CYGARC_HAL_STR7XX_UART_CR, 0 );

     // Set serial configuration
     cr = CYGARC_HAL_STR7XX_UART_CR_BITS_8 |
          CYGARC_HAL_STR7XX_UART_CR_STOP_1_0;
     HAL_WRITE_UINT32( base+CYGARC_HAL_STR7XX_UART_CR, cr );

     // Set timeout register to 16 bit times.
     HAL_WRITE_UINT32( base+CYGARC_HAL_STR7XX_UART_TOR, 16 );

     // Enable the FIFO and receiver
     cr |= CYGARC_HAL_STR7XX_UART_CR_FIFO_ENABLE;
     HAL_WRITE_UINT32( base+CYGARC_HAL_STR7XX_UART_CR, cr );     
     cr |= CYGARC_HAL_STR7XX_UART_CR_RX_ENABLE;
     HAL_WRITE_UINT32( base+CYGARC_HAL_STR7XX_UART_CR, cr );     

     // Reset the FIFOs
     HAL_WRITE_UINT32( base+CYGARC_HAL_STR7XX_UART_TxRST, 0 );     
     HAL_WRITE_UINT32( base+CYGARC_HAL_STR7XX_UART_RxRST, 0 );     

     // Now enable the UART
     cr |= CYGARC_HAL_STR7XX_UART_CR_BRG_ENABLE;
     HAL_WRITE_UINT32( base+CYGARC_HAL_STR7XX_UART_CR, cr );

     chan->buf_head = 0;
     chan->buf_tail = 0;
     chan->buf_count = 0;
}

void
cyg_hal_plf_serial_putc(void *__ch_data, char c)
{
    cyg_uint8* base = ((channel_data_t*)__ch_data)->base;
    cyg_uint8 status;
    CYGARC_HAL_SAVE_GP();

     do {
         HAL_READ_UINT32( base + CYGARC_HAL_STR7XX_UART_SR, status );
     } while ((status & CYGARC_HAL_STR7XX_UART_IER_Tx_HALF) == 0);

     HAL_WRITE_UINT32( base + CYGARC_HAL_STR7XX_UART_TxBUF, c );

    CYGARC_HAL_RESTORE_GP();
}

static cyg_bool
cyg_hal_plf_serial_getc_nonblock1(void* __ch_data, cyg_uint8* ch)
{
    volatile channel_data_t* chan = (channel_data_t*)__ch_data;
    cyg_uint8* base = chan->base;
    volatile cyg_uint8 stat;
    cyg_uint32 c;

    // Start by fetching all available chars from the FIFO

    HAL_READ_UINT32( base + CYGARC_HAL_STR7XX_UART_SR, stat );
    while( (chan->buf_count < BUF_SIZE-1) &&
           ((stat & CYGARC_HAL_STR7XX_UART_IER_Rx_NOTEMPTY) != 0) )
    {
        HAL_READ_UINT32( base + CYGARC_HAL_STR7XX_UART_RxBUF, c);
        chan->buf[chan->buf_head++] = c&0xFF;
        chan->buf_count++;
        if( chan->buf_head == BUF_SIZE )
            chan->buf_head = 0;
        HAL_READ_UINT32( base + CYGARC_HAL_STR7XX_UART_SR, stat );
    }

    if( chan->buf_count == 0 )
        return false;

    
    *ch = chan->buf[chan->buf_tail++];
    chan->buf_count--;
    if( chan->buf_tail == BUF_SIZE )
        chan->buf_tail = 0;

    return true;
}

static cyg_bool
cyg_hal_plf_serial_getc_nonblock(void* __ch_data, cyg_uint8* ch)
{
    return cyg_hal_plf_serial_getc_nonblock1( __ch_data, ch );
}

cyg_uint8
cyg_hal_plf_serial_getc(void* __ch_data)
{
    cyg_uint8 ch;
    CYGARC_HAL_SAVE_GP();

    while(!cyg_hal_plf_serial_getc_nonblock(__ch_data, &ch));

    CYGARC_HAL_RESTORE_GP();
    return ch;
}

static void
cyg_hal_plf_serial_write(void* __ch_data, const cyg_uint8* __buf, 
                         cyg_uint32 __len)
{
    CYGARC_HAL_SAVE_GP();

    while(__len-- > 0)
        cyg_hal_plf_serial_putc(__ch_data, *__buf++);

    CYGARC_HAL_RESTORE_GP();
}

static void
cyg_hal_plf_serial_read(void* __ch_data, cyg_uint8* __buf, cyg_uint32 __len)
{
    CYGARC_HAL_SAVE_GP();

    while(__len-- > 0)
        *__buf++ = cyg_hal_plf_serial_getc(__ch_data);

    CYGARC_HAL_RESTORE_GP();
}

cyg_bool
cyg_hal_plf_serial_getc_timeout(void* __ch_data, cyg_uint8* ch)
{
    int delay_count;
    channel_data_t* chan = (channel_data_t*)__ch_data;
    cyg_bool res;
    CYGARC_HAL_SAVE_GP();

    delay_count = chan->msec_timeout * 10; // delay in .1 ms steps

    for(;;) {
        res = cyg_hal_plf_serial_getc_nonblock(__ch_data, ch);
        if (res || 0 == delay_count--)
            break;
        
        CYGACC_CALL_IF_DELAY_US(100);
    }

    CYGARC_HAL_RESTORE_GP();
    return res;
}

static int
cyg_hal_plf_serial_control(void *__ch_data, __comm_control_cmd_t __func, ...)
{
    static int irq_state = 0;
    channel_data_t* chan = (channel_data_t*)__ch_data;
    cyg_uint8* base = ((channel_data_t*)__ch_data)->base;
    int ret = 0;

    va_list ap;

    CYGARC_HAL_SAVE_GP();

    va_start(ap, __func);

    switch (__func) {
    case __COMMCTL_IRQ_ENABLE:
        irq_state = 1;
        HAL_INTERRUPT_ACKNOWLEDGE( chan->isr_vector );
        HAL_INTERRUPT_UNMASK( chan->isr_vector );
        {
            cyg_uint32 ier;
            HAL_READ_UINT32( base + CYGARC_HAL_STR7XX_UART_IER, ier );
            ier |= CYGARC_HAL_STR7XX_UART_IER_Rx_NOTEMPTY;
            HAL_WRITE_UINT32( base + CYGARC_HAL_STR7XX_UART_IER, ier );
        }
        break;
    case __COMMCTL_IRQ_DISABLE:
        ret = irq_state;
        irq_state = 0;
        HAL_INTERRUPT_MASK( chan->isr_vector );
        {
            cyg_uint32 ier;
            HAL_READ_UINT32( base + CYGARC_HAL_STR7XX_UART_IER, ier );
            ier &= ~CYGARC_HAL_STR7XX_UART_IER_Rx_NOTEMPTY;
            HAL_WRITE_UINT32( base + CYGARC_HAL_STR7XX_UART_IER, ier );
        }
        break;
    case __COMMCTL_DBG_ISR_VECTOR:
        ret = chan->isr_vector;
        break;
    case __COMMCTL_SET_TIMEOUT:
    {
        va_list ap;

        va_start(ap, __func);

        ret = chan->msec_timeout;
        chan->msec_timeout = va_arg(ap, cyg_uint32);

        va_end(ap);
    }        
    case __COMMCTL_GETBAUD:
        ret = chan->baud_rate;
        break;
    case __COMMCTL_SETBAUD:
        chan->baud_rate = va_arg(ap, cyg_int32);
        // Should we verify this value here?
//        update_baud_rate(chan);
        hal_str7xx_uart_setbaud( chan->uart, chan->baud_rate );        
        ret = 0;
        break;
    default:
        break;
    }
    va_end(ap);
    CYGARC_HAL_RESTORE_GP();
    return ret;
}

static int
cyg_hal_plf_serial_isr(void *__ch_data, int* __ctrlc, 
                       CYG_ADDRWORD __vector, CYG_ADDRWORD __data)
{
    channel_data_t* chan = (channel_data_t*)__ch_data;
    cyg_uint8 ch;

    CYGARC_HAL_SAVE_GP();


     *__ctrlc = 0;

     if( cyg_hal_plf_serial_getc_nonblock(__ch_data, &ch) )
     {
         if( cyg_hal_is_break( &ch , 1 ) )
             *__ctrlc = 1; 
     }

    HAL_INTERRUPT_ACKNOWLEDGE(chan->isr_vector);

    CYGARC_HAL_RESTORE_GP();
    return 1;
}

static void
cyg_hal_plf_serial_init(void)
{
    hal_virtual_comm_table_t* comm;
    int cur;
    int i;

    cur = CYGACC_CALL_IF_SET_CONSOLE_COMM(CYGNUM_CALL_IF_SET_COMM_ID_QUERY_CURRENT);

    for( i = 0; i < CYGNUM_HAL_VIRTUAL_VECTOR_COMM_CHANNELS ; i++ )
    {
        cyg_hal_plf_serial_init_channel(&str7xx_ser_channels[i]);

        CYGACC_CALL_IF_SET_CONSOLE_COMM(i);
        comm = CYGACC_CALL_IF_CONSOLE_PROCS();
        CYGACC_COMM_IF_CH_DATA_SET(*comm, &str7xx_ser_channels[i]);
        CYGACC_COMM_IF_WRITE_SET(*comm, cyg_hal_plf_serial_write);
        CYGACC_COMM_IF_READ_SET(*comm, cyg_hal_plf_serial_read);
        CYGACC_COMM_IF_PUTC_SET(*comm, cyg_hal_plf_serial_putc);
        CYGACC_COMM_IF_GETC_SET(*comm, cyg_hal_plf_serial_getc);
        CYGACC_COMM_IF_CONTROL_SET(*comm, cyg_hal_plf_serial_control);
        CYGACC_COMM_IF_DBG_ISR_SET(*comm, cyg_hal_plf_serial_isr);
        CYGACC_COMM_IF_GETC_TIMEOUT_SET(*comm, cyg_hal_plf_serial_getc_timeout);
    }

    // Restore original console
    CYGACC_CALL_IF_SET_CONSOLE_COMM(cur);

    // set debug channel baud rate if different
#if (CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD != CYGNUM_HAL_VIRTUAL_VECTOR_DEBUG_CHANNEL_BAUD)
    str7xx_ser_channels[CYGNUM_HAL_VIRTUAL_VECTOR_DEBUG_CHANNEL]->baud_rate = 
        CYGNUM_HAL_VIRTUAL_VECTOR_DEBUG_CHANNEL_BAUD;
    update_baud_rate( &str7xx_ser_channels[CYGNUM_HAL_VIRTUAL_VECTOR_DEBUG_CHANNEL] );
#endif

}

void
cyg_hal_plf_comms_init(void)
{
    static int initialized = 0;

    if (initialized)
        return;

    initialized = 1;

    cyg_hal_plf_serial_init();
}

void
hal_diag_led(int mask)
{
//    extern void hal_str7xx_set_leds(int);
//    hal_str7xx_set_leds(mask);
}

//-----------------------------------------------------------------------------
// End of hal_diag.c
