/*========================================================================
//
//      mathfp/s_frexp.c
//
//      Part of the standard mathematical function library
//
//========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004 Free Software Foundation, Inc.
// Copyright (C) 2004 eCosCentric Limited                                   
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):     jlarmour
// Contributors:  
// Date:          2004-02-30
// Purpose:       
// Description:   Derived from newlib libm
//                http://sources.redhat.com/newlib
//                in turn based on SunSoft fdlibm.
//                See copyright/license information below.
// Usage:         #include <math.h>
//
//####DESCRIPTIONEND####
//======================================================================*/


/* @(#)z_frexp.c 1.0 98/08/13 */

/*
FUNCTION
       <<frexp>>, <<frexpf>>---split floating-point number
INDEX
        frexp
INDEX
        frexpf

ANSI_SYNOPSIS
        #include <math.h>
        double frexp(double <[val]>, int *<[exp]>);
        float frexpf(float <[val]>, int *<[exp]>);

TRAD_SYNOPSIS
        #include <math.h>
        double frexp(<[val]>, <[exp]>)
        double <[val]>;
        int *<[exp]>;

        float frexpf(<[val]>, <[exp]>)
        float <[val]>;
        int *<[exp]>;


DESCRIPTION
        All nonzero, normal numbers can be described as <[m]> * 2**<[p]>.
        <<frexp>> represents the double <[val]> as a mantissa <[m]>
        and a power of two <[p]>. The resulting mantissa will always
        be greater than or equal to <<0.5>>, and less than <<1.0>> (as
        long as <[val]> is nonzero). The power of two will be stored
        in <<*>><[exp]>.

@ifnottex
<[m]> and <[p]> are calculated so that
<[val]> is <[m]> times <<2>> to the power <[p]>.
@end ifnottex
@tex
<[m]> and <[p]> are calculated so that
$ val = m \times 2^p $.
@end tex

<<frexpf>> is identical, other than taking and returning
floats rather than doubles.

RETURNS
<<frexp>> returns the mantissa <[m]>. If <[val]> is <<0>>, infinity,
or Nan, <<frexp>> will set <<*>><[exp]> to <<0>> and return <[val]>.

PORTABILITY
<<frexp>> is ANSI.
<<frexpf>> is an extension.


*/

/*****************************************************************
 * frexp
 *
 * Input:
 *   d   - floating point value
 *   exp - exponent value
 *
 * Output:
 *   A floating point value in the range [0.5, 1).
 *
 * Description:
 *   This routine breaks a floating point value into a number f and
 *   an exponent exp such that d = f * 2 ^ exp.
 *
 *****************************************************************/

#include "mathincl/fdlibm.h"
#include "zmath.h"

#ifndef _DOUBLE_IS_32BITS

double frexp (double d, int *exp)
{
  double f;
  __uint32_t hd, ld, hf, lf;

  EXTRACT_WORDS (hd, ld, d);

  /* Get the exponent. */
  *exp = ((hd & 0x7ff00000) >> 20) - 1022;

  /* Get the mantissa. */ 
  lf = ld;
  hf = hd & 0x800fffff;  
  hf |= 0x3fe00000;

  INSERT_WORDS (f, hf, lf);

  /* Check for special values. */
  switch (numtest (f))
    {
      case NAN:
      case INF:
        errno = EDOM;
        *exp = 0;
        return (f);
    }

  return (f);
}

#endif /* _DOUBLE_IS_32BITS */

// EOF s_frexp.c
