/*========================================================================
//
//      mathfp/s_ldexp.c
//
//      Part of the standard mathematical function library
//
//========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004 Free Software Foundation, Inc.
// Copyright (C) 2004 eCosCentric Limited                                   
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):     jlarmour
// Contributors:  
// Date:          2004-02-30
// Purpose:       
// Description:   Derived from newlib libm
//                http://sources.redhat.com/newlib
//                in turn based on SunSoft fdlibm.
//                See copyright/license information below.
// Usage:         #include <math.h>
//
//####DESCRIPTIONEND####
//======================================================================*/


/* @(#)z_ldexp.c 1.0 98/08/13 */

/*
FUNCTION
       <<ldexp>>, <<ldexpf>>---load exponent

INDEX
        ldexp
INDEX
        ldexpf

ANSI_SYNOPSIS
       #include <math.h>
       double ldexp(double <[val]>, int <[exp]>);
       float ldexpf(float <[val]>, int <[exp]>);

TRAD_SYNOPSIS
       #include <math.h>

       double ldexp(<[val]>, <[exp]>)
              double <[val]>;
              int <[exp]>;

       float ldexpf(<[val]>, <[exp]>)
              float <[val]>;
              int <[exp]>;

DESCRIPTION
<<ldexp>> calculates the value
@ifnottex
<[val]> times 2 to the power <[exp]>.
@end ifnottex
@tex
$val\times 2^{exp}$.
@end tex
<<ldexpf>> is identical, save that it takes and returns <<float>>
rather than <<double>> values.

RETURNS
<<ldexp>> returns the calculated value.

Underflow and overflow both set <<errno>> to <<ERANGE>>.
On underflow, <<ldexp>> and <<ldexpf>> return 0.0.
On overflow, <<ldexp>> returns plus or minus <<HUGE_VAL>>.

PORTABILITY
<<ldexp>> is ANSI, <<ldexpf>> is an extension.

*/

/******************************************************************
 * ldexp
 *
 * Input:
 *   d - a floating point value
 *   e - an exponent value
 *
 * Output:
 *   A floating point value f such that f = d * 2 ^ e.
 *
 * Description:
 *   This function creates a floating point number f such that
 *   f = d * 2 ^ e.
 *
 *****************************************************************/

#include <float.h>
#include "mathincl/fdlibm.h"
#include "zmath.h"

#ifndef _DOUBLE_IS_32BITS

#define DOUBLE_EXP_OFFS 1023

double
_DEFUN (ldexp, (double, int),
        double d _AND
        int e)
{
  int exp;
  __uint32_t hd;

  GET_HIGH_WORD (hd, d);

  /* Check for special values and then scale d by e. */
  switch (numtest (d))
    {
      case NAN:
        errno = EDOM;
        break;

      case INF:
        errno = ERANGE;
        break;

      case 0:
        break;

      default:
        exp = (hd & 0x7ff00000) >> 20;
        exp += e;

        if (exp > DBL_MAX_EXP + DOUBLE_EXP_OFFS)
         {
           errno = ERANGE;
           d = z_infinity.d;
         }
        else if (exp < DBL_MIN_EXP + DOUBLE_EXP_OFFS)
         {
           errno = ERANGE;
           d = -z_infinity.d;
         }
        else
         {
           hd &= 0x800fffff;
           hd |= exp << 20;
           SET_HIGH_WORD (d, hd);
         }
    }

    return (d);
}

#endif /* _DOUBLE_IS_32BITS */

// EOF s_ldexp.c
