//==========================================================================
//
//      jffs2-fileio1.c
//
//      Test JFFS2 using fileio system
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004, 2006 Free Software Foundation, Inc.
// Copyright (C) 2004,2006 eCosCentric Limited                              
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):           nickg
// Contributors:        jlarmour
// Date:                2000-05-25
// Purpose:             Test JFFS2 using fileio system
// Description:         This test uses the fileio system to provide testing
//                      for JFFS2. It was originally named fileio1.c
//                      and derives from the version in CYGPKG_FILEIO
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/hal.h>
#include <pkgconf/kernel.h>
#include <pkgconf/io_fileio.h>
#include <pkgconf/isoinfra.h>
#include <pkgconf/io_flash.h>

#include <cyg/infra/cyg_type.h>        // base types
#include <cyg/infra/cyg_trac.h>        // tracing macros
#include <cyg/infra/cyg_ass.h>         // assertion macros
#include <cyg/io/io.h>
#include <cyg/io/config_keys.h>
#include <cyg/io/flash.h>

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#include <dirent.h>
#include <stdio.h>

#include <stdlib.h>

#include <cyg/fileio/fileio.h>

#include <cyg/infra/testcase.h>
#include <cyg/infra/diag.h>            // HAL polled output

#include <pkgconf/fs_jffs2.h>	// Address of JFFS2

//==========================================================================
// Mount details

#define stringify2(_x_) #_x_
#define stringify(_x_) stringify2(_x_)

#if defined(CYGDAT_IO_FLASH_BLOCK_DEVICE_NAME_1)
# define JFFS2_TEST_DEV CYGDAT_IO_FLASH_BLOCK_DEVICE_NAME_1
#elif defined(CYGFUN_IO_FLASH_BLOCK_FROM_FIS)
# define JFFS2_TEST_DEV "/dev/flash/fis/jffs2test"
#else
// fall back to using a user set area in the first device (only)
# define JFFS2_TEST_DEV "/dev/flash/0/" stringify(CYGNUM_FS_JFFS2_TEST_OFFSET) "," stringify(CYGNUM_FS_JFFS2_TEST_LENGTH)
// But just in case, we check it's blank.
# define CHECK_FLASH_BLANK
#endif

// we could use an mtab but we don't in order to get better diagnostics
// by calling mount() directly.
#if 0
MTAB_ENTRY( jffs2_mte1,
                   "/",
                   "jffs2",
                   JFFS2_TEST_DEV,
                   0);
#endif

//==========================================================================

#define SHOW_RESULT( _fn, _res ) \
diag_printf("FAIL:<" #_fn "() returned %d %s>\n", _res, _res<0?strerror(errno):"");

//==========================================================================

#define IOSIZE  1000

#define LONGNAME1       "long_file_name_that_should_take_up_more_than_one_directory_entry_1"
#define LONGNAME2       "long_file_name_that_should_take_up_more_than_one_directory_entry_2"


//==========================================================================

#ifndef CYGINT_ISO_STRING_STRFUNCS

char *strcat( char *s1, const char *s2 )
{
    char *s = s1;
    while( *s1 ) s1++;
    while( (*s1++ = *s2++) != 0);
    return s;
}

#endif

//==========================================================================

static void listdir( char *name, int statp, int numexpected, int *numgot )
{
    int err;
    DIR *dirp;
    int num=0;
    
    diag_printf("<INFO>: reading directory %s\n",name);
    
    dirp = opendir( name );
    if( dirp == NULL ) SHOW_RESULT( opendir, -1 );

    for(;;)
    {
        struct dirent *entry = readdir( dirp );
        
        if( entry == NULL )
            break;
        num++;
        diag_printf("<INFO>: entry %14s",entry->d_name);
#ifdef CYGPKG_FS_JFFS2_RET_DIRENT_DTYPE
        diag_printf(" d_type %2d", entry->d_type);
#endif
        if( statp )
        {
            char fullname[PATH_MAX];
            struct stat sbuf;

            if( name[0] )
            {
                strcpy(fullname, name );
                if( !(name[0] == '/' && name[1] == 0 ) )
                    strcat(fullname, "/" );
            }
            else fullname[0] = 0;
            
            strcat(fullname, entry->d_name );
            
            err = stat( fullname, &sbuf );
            if( err < 0 )
            {
                if( errno == ENOSYS )
                    diag_printf(" <no status available>");
                else SHOW_RESULT( stat, err );
            }
            else
            {
                diag_printf(" [mode %08x ino %08x nlink %d size %d]",
                            sbuf.st_mode,sbuf.st_ino,sbuf.st_nlink,(int)sbuf.st_size);
            }
#ifdef CYGPKG_FS_JFFS2_RET_DIRENT_DTYPE
            if ((entry->d_type & S_IFMT) != (sbuf.st_mode & S_IFMT))
              CYG_TEST_FAIL("File mode's don't match between dirent and stat");
#endif
        }

        diag_printf("\n");
    }

    err = closedir( dirp );
    if( err < 0 ) SHOW_RESULT( stat, err );
    if (numexpected >= 0 && num != numexpected)
        CYG_TEST_FAIL("Wrong number of dir entries\n");
    if ( numgot != NULL )
        *numgot = num;
}

//==========================================================================

static void fillfile( int fd, size_t size )
{
    unsigned char buf[IOSIZE];
    ssize_t wrote;
    int i;

    for( i = 0; i < IOSIZE; i++ ) buf[i] = i%256;
 
    while( size > 0 )
    {
        ssize_t len = size;
        if ( len > IOSIZE ) len = IOSIZE;
        
        wrote = write( fd, buf, len );
        if( wrote != len ) SHOW_RESULT( write, (int)wrote );        

        size -= wrote;
    }
}

//==========================================================================

static void createfile( char *name, size_t size )
{
    int fd;
    int err;

    diag_printf("<INFO>: create file %s size %d\n",name,(int)size);

    err = access( name, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );
    
    fd = open( name, O_WRONLY|O_CREAT );
    if( fd < 0 ) SHOW_RESULT( open, fd );

    fillfile(fd, size);

    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );
}

//==========================================================================

#if 0
static void maxfile( char *name )
{
    char buf[IOSIZE];
    int fd;
    ssize_t wrote;
    int i;
    int err;
    size_t size = 0;
    
    diag_printf("<INFO>: create maximal file %s\n",name);

    err = access( name, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );
    
    for( i = 0; i < IOSIZE; i++ ) buf[i] = i%256;
 
    fd = open( name, O_WRONLY|O_CREAT );
    if( fd < 0 ) SHOW_RESULT( open, fd );

    do
    {
        wrote = write( fd, buf, IOSIZE );
        if( wrote < 0 ) SHOW_RESULT( write, wrote );        

        size += wrote;
        
    } while( wrote == IOSIZE );

    diag_printf("<INFO>: file size == %d\n",size);

    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );
}
#endif

//==========================================================================

static void checkfile( char *name )
{
    unsigned char buf[IOSIZE];
    int fd;
    ssize_t done;
    int i;
    int err;
    off_t pos = 0;

    diag_printf("<INFO>: check file %s\n",name);
    
    err = access( name, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );

    fd = open( name, O_RDONLY );
    if( fd < 0 ) SHOW_RESULT( open, fd );

    for(;;)
    {
        done = read( fd, buf, IOSIZE );
        if( done < 0 ) SHOW_RESULT( read, (int)done );

        if( done == 0 ) break;

        for( i = 0; i < done; i++ )
            if( buf[i] != i%256 )
            {
                diag_printf("buf[%d+%d](%02x) != %02x\n",(int)pos,i,buf[i],i%256);
                CYG_TEST_FAIL("Data read not equal to data written\n");
            }
        
        pos += done;
    }

    err = close( fd );
    if( err < 0 ) SHOW_RESULT( close, err );
}

//==========================================================================

static void copyfile( char *name2, char *name1 )
{

    int err;
    char buf[IOSIZE];
    int fd1, fd2;
    ssize_t done, wrote;

    diag_printf("<INFO>: copy file %s -> %s\n",name2,name1);

    err = access( name1, F_OK );
    if( err < 0 && errno != EACCES ) SHOW_RESULT( access, err );

    err = access( name2, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );
    
    fd1 = open( name1, O_WRONLY|O_CREAT );
    if( fd1 < 0 ) SHOW_RESULT( open, fd1 );

    fd2 = open( name2, O_RDONLY );
    if( fd2 < 0 ) SHOW_RESULT( open, fd2 );
    
    for(;;)
    {
        done = read( fd2, buf, IOSIZE );
        if( done < 0 ) SHOW_RESULT( read, (int)done );

        if( done == 0 ) break;

        wrote = write( fd1, buf, done );
        if( wrote != done ) SHOW_RESULT( write, (int)wrote );

        if( wrote != done ) break;
    }

    err = close( fd1 );
    if( err < 0 ) SHOW_RESULT( close, err );

    err = close( fd2 );
    if( err < 0 ) SHOW_RESULT( close, err );
    
}

//==========================================================================

static void comparefiles( char *name2, char *name1 )
{
    int err;
    char buf1[IOSIZE];
    char buf2[IOSIZE];
    int fd1, fd2;
    ssize_t done1, done2;
    int i;

    diag_printf("<INFO>: compare files %s == %s\n",name2,name1);

    err = access( name1, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );

    err = access( name1, F_OK );
    if( err != 0 ) SHOW_RESULT( access, err );
    
    fd1 = open( name1, O_RDONLY );
    if( fd1 < 0 ) SHOW_RESULT( open, fd1 );

    fd2 = open( name2, O_RDONLY );
    if( fd2 < 0 ) SHOW_RESULT( open, fd2 );
    
    for(;;)
    {
        done1 = read( fd1, buf1, IOSIZE );
        if( done1 < 0 ) SHOW_RESULT( read, (int)done1 );

        done2 = read( fd2, buf2, IOSIZE );
        if( done2 < 0 ) SHOW_RESULT( read, (int)done2 );

        if( done1 != done2 )
            diag_printf("Files different sizes\n");
        
        if( done1 == 0 ) break;

        for( i = 0; i < done1; i++ )
            if( buf1[i] != buf2[i] )
            {
                diag_printf("buf1[%d](%02x) != buf1[%d](%02x)\n",i,buf1[i],i,buf2[i]);
                CYG_TEST_FAIL("Data in files not equal\n");
            }
    }

    err = close( fd1 );
    if( err < 0 ) SHOW_RESULT( close, err );

    err = close( fd2 );
    if( err < 0 ) SHOW_RESULT( close, err );
    
}

//==========================================================================

void checkcwd( const char *cwd )
{
    static char cwdbuf[PATH_MAX];
    char *ret;

    ret = getcwd( cwdbuf, sizeof(cwdbuf));
    if( ret == NULL ) SHOW_RESULT( getcwd, 0 );    

    if( strcmp( cwdbuf, cwd ) != 0 )
    {
        diag_printf( "cwdbuf %s cwd %s\n",cwdbuf, cwd );
        CYG_TEST_FAIL( "Current directory mismatch");
    }
}

//==========================================================================
// prepare_device
//
// This does a quick check for existence of the underlying device.
// It also checks the region is blank if required.

void prepare_device( const char *dev )
{
    Cyg_ErrNo err;
    cyg_io_handle_t h;
    cyg_io_flash_getconfig_devsize_t devsize;
#if defined(CYGHWR_IO_FLASH_BLOCK_LOCKING)
    cyg_io_flash_getconfig_unlock_t unlockconf;
#endif
    cyg_uint32 conf_size = sizeof(devsize);
    cyg_bool isblank = false;

    err = cyg_io_lookup( dev, &h );
    if (err == -ENOENT)
    {
        CYG_TEST_NA("Flash test device not found, test environment not set up by user.");
    }
    else if (err)
    {
        diag_printf("cyg_io_lookup returned: %d\n", err );
        CYG_TEST_FAIL_FINISH( "Test device lookup failed" );
    }

    err = cyg_io_get_config( h, CYG_IO_GET_CONFIG_FLASH_DEVSIZE, &devsize,
                             &conf_size );
    if (err)
    {
        CYG_TEST_FAIL_FINISH("prepare_device: cyg_io_get_config failed");
    }

    // this check is really for sanity with the CHECK_FLASH_BLANK code, but
    // we may as well always do it.
    if ((devsize.dev_size % 128) != 0)
    {
        diag_printf("device size == 0x%08x\n",(unsigned)devsize.dev_size);
        CYG_TEST_FAIL_FINISH("prepare_device: bizarre size returned - not multiple of 128");
    }
    
#ifdef CHECK_FLASH_BLANK
    {
        cyg_uint32 pos;
        cyg_uint8 buf[128];

        for (pos = 0; pos < devsize.dev_size; pos += sizeof(buf) )
        {
            cyg_uint32 i;
            cyg_uint32 len = sizeof(buf);
            cyg_uint32 *buf32 = (cyg_uint32 *)&buf[0];

            err = cyg_io_bread(h, &buf[0], &len, pos );
            if (err || (len < sizeof(buf)))
            {
                diag_printf("Read failure at %u, err=%d, new len=%u\n", pos, err, len );
                CYG_TEST_FAIL_FINISH("prepare_device: read failed");
            }
            for (i=0; i<(sizeof(buf)/4); i++)
            {
                if (buf32[i] != 0xFFFFFFFF)
                {
                    CYG_TEST_FAIL_FINISH("Supplied test device not blank! Not erasing.");
                }
            } // for
        } // for
        isblank = true;
    }
#endif    

#if defined(CYGHWR_IO_FLASH_BLOCK_LOCKING)
    // This device might need unlocking before it can be written to.
    conf_size = sizeof(unlockconf);

    // unlock the whole "device"
    unlockconf.offset = 0;
    unlockconf.len = devsize.dev_size;
    err = cyg_io_get_config( h, CYG_IO_GET_CONFIG_FLASH_UNLOCK, &unlockconf,
                             &conf_size );
    if (err)
    {
        CYG_TEST_FAIL_FINISH("prepare_device: cyg_io_get_config unlock failed");
    }
    if (unlockconf.flasherr != CYG_FLASH_ERR_OK)
    {
        diag_printf("flash error @0x%08x:%d:%s\n", unlockconf.err_address,
                    unlockconf.flasherr,
                    cyg_flash_errmsg(unlockconf.flasherr) );
        CYG_TEST_FAIL_FINISH("prepare_device: unlock failed");
    }
#endif // if defined(CYGHWR_IO_FLASH_BLOCK_LOCKING)

    if (!isblank)
    {
        // erase - this is our test device and we need it to be clean, rather
        // than containing a potentially corrupt FS.
        cyg_io_flash_getconfig_erase_t eraseconf;
        conf_size = sizeof(eraseconf);
        
        // erase the whole "device"
        eraseconf.offset = 0;
        eraseconf.len = devsize.dev_size;
        CYG_TEST_INFO("Erasing Flash test region");
        err = cyg_io_get_config( h, CYG_IO_GET_CONFIG_FLASH_ERASE, &eraseconf,
                                 &conf_size );
        if (err)
        {
            CYG_TEST_FAIL_FINISH("prepare_device: get_config for erase failed");
        }
        if (eraseconf.flasherr != CYG_FLASH_ERR_OK)
        {
            diag_printf("flash error @0x%08x:%d:%s\n", eraseconf.err_address,
                        eraseconf.flasherr,
                        cyg_flash_errmsg(eraseconf.flasherr) );
            CYG_TEST_FAIL_FINISH("prepare_device: erase failed");
        }
        CYG_TEST_INFO("Flash test region erase complete");
    }

    // flash block layer is known to need close when last instance stops using
    // handle.
    err = cyg_io_set_config( h, CYG_IO_SET_CONFIG_CLOSE, NULL, NULL );
    if (err)
    {
        CYG_TEST_FAIL_FINISH("prepare_device: CLOSE via cyg_io_set_config failed");
    }
}

//==========================================================================
// main
void jffs2_fileio1(cyg_bool prep_device);

#if CYGINT_ISO_MAIN_STARTUP
int main( int argc, char **argv )
{
    CYG_TEST_INIT();

    jffs2_fileio1(true);
    jffs2_fileio1(false);
    CYG_TEST_PASS_FINISH("jffs2-fileio1");
    return 0;
}

#else
void cyg_start(void)
{
    CYG_TEST_INIT();

    jffs2_fileio1(true);
    jffs2_fileio1(false);
    CYG_TEST_PASS_FINISH("jffs2-fileio1");
}
#endif

void jffs2_fileio1(cyg_bool prep_device)
{
    int err, fd;
    //int i;
    int existingdirents=-1;

    struct mallinfo info;

    info =  mallinfo();
    diag_printf("arenasize %d, freeblocks %d, totalallocated %d, totalfree %d, maxfree %d\n",
                info.arena, info.ordblks, info.uordblks, info.fordblks, info.maxfree);
    
    

    // --------------------------------------------------------------

    if (prep_device)
        prepare_device( JFFS2_TEST_DEV );

    err = mount( JFFS2_TEST_DEV, "/j", "jffs2" );
    if( err < 0 ) SHOW_RESULT( mount, err );    

    err = chdir( "/j" );
    if( err < 0 ) SHOW_RESULT( chdir, err );

    checkcwd( "/j" );
    
    listdir( "/j", true, -1, &existingdirents );
    if ( existingdirents < 2 )
        CYG_TEST_FAIL("Not enough dir entries\n");

    // --------------------------------------------------------------

    createfile( "/j/foo", 202 );
    checkfile( "foo" );
    copyfile( "foo", "fee");
    checkfile( "fee" );
    comparefiles( "foo", "/j/fee" );
    diag_printf("<INFO>: mkdir bar\n");
    err = mkdir( "/j/bar", 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );

    listdir( "/j" , true, existingdirents+3, NULL );

    copyfile( "fee", "/j/bar/fum" );
    checkfile( "bar/fum" );
    comparefiles( "/j/fee", "bar/fum" );

    diag_printf("<INFO>: cd bar\n");
    err = chdir( "bar" );
    if( err < 0 ) SHOW_RESULT( chdir, err );

    checkcwd( "/j/bar" );
    
    diag_printf("<INFO>: rename /j/foo bundy\n");    
    err = rename( "/j/foo", "bundy" );
    if( err < 0 ) SHOW_RESULT( rename, err );
    
    listdir( "/j", true, existingdirents+2, NULL );
    listdir( "." , true, 4, NULL );

    checkfile( "/j/bar/bundy" );
    comparefiles("/j/fee", "bundy" );

    // --------------------------------------------------------------

    createfile( LONGNAME1, 123 );
    checkfile( LONGNAME1 );
    copyfile( LONGNAME1, LONGNAME2 );

    listdir( ".", false, 6, NULL );
    
    diag_printf("<INFO>: unlink " LONGNAME1 "\n");    
    err = unlink( LONGNAME1 );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: unlink " LONGNAME2 "\n");    
    err = unlink( LONGNAME2 );
    if( err < 0 ) SHOW_RESULT( unlink, err );
    
    
    // --------------------------------------------------------------

    diag_printf("<INFO>: unlink fee\n");    
    err = unlink( "/j/fee" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: unlink fum\n");        
    err = unlink( "fum" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: unlink /bar/bundy\n");        
    err = unlink( "/j/bar/bundy" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: cd /j\n");        
    err = chdir( "/j" );
    if( err < 0 ) SHOW_RESULT( chdir, err );

    checkcwd( "/j" );
    
    diag_printf("<INFO>: rmdir /j/bar\n");        
    err = rmdir( "/j/bar" );
    if( err < 0 ) SHOW_RESULT( rmdir, err );
    
    listdir( "/j", false, existingdirents, NULL );

    // --------------------------------------------------------------

    diag_printf("<INFO>: mount /jffs2 \n");
    err = mount( JFFS2_TEST_DEV, "/jffs2", "jffs2" );
    if( err < 0 ) SHOW_RESULT( mount, err );    

    CYG_TEST_INFO("Testing creation and copying of files");

    createfile( "/jffs2/tinky", 456 );
    copyfile( "/jffs2/tinky", "/jffs2/laalaa" );
    checkfile( "/jffs2/tinky");
    checkfile( "/jffs2/laalaa");
    comparefiles( "/jffs2/tinky", "/jffs2/laalaa" );

    CYG_TEST_INFO("Creating laalaa O_TRUNC");
    // Test O_TRUNC
    fd = open("laalaa", O_WRONLY|O_CREAT|O_TRUNC);
    if( fd < 0 ) SHOW_RESULT( open, fd );
    fillfile(fd, 321);
    err = close(fd);
    if( fd < 0 ) SHOW_RESULT( close, fd );
    checkfile("laalaa");
    CYG_TEST_INFO("Creating tubby tustard");
    createfile( "tubby tustard", 321);
    comparefiles( "laalaa", "tubby tustard" );

    diag_printf("<INFO>: unlink tubby tustard\n");    
    err = unlink( "tubby tustard" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: cd /jffs2\n");    
    err = chdir( "/jffs2" );
    if( err < 0 ) SHOW_RESULT( chdir, err );

    checkcwd( "/jffs2" );
        
    diag_printf("<INFO>: mkdir noonoo\n");    
    err = mkdir( "noonoo", 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );

    listdir( "." , true, existingdirents+3, NULL);

    diag_printf("<INFO>: cd noonoo\n");
    err = chdir( "noonoo" );
    if( err < 0 ) SHOW_RESULT( chdir, err );

    checkcwd( "/jffs2/noonoo" );
    
    createfile( "tinky", 678 );
    checkfile( "tinky" );

    createfile( "dipsy", 3456 );
    checkfile( "dipsy" );
    copyfile( "dipsy", "po" );
    checkfile( "po" );
    comparefiles( "dipsy", "po" );


    /*for(i=0;i<2048;i++) {
        diag_printf("<INFO>: churningchurningchurning................................ITERATION = %d\n", i);    
        createfile( "churningchurningchurning", 4096 );
        diag_printf("<INFO>: unlink churningchurningchurning\n");    
        err = unlink( "churningchurningchurning" );
        if( err < 0 ) SHOW_RESULT( unlink, err );
    }*/


    listdir( ".", true, 5, NULL );
    listdir( "..", true, existingdirents+3, NULL );

    // --------------------------------------------------------------

    diag_printf("<INFO>: unlink tinky\n");    
    err = unlink( "tinky" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: unlink dipsy\n");    
    err = unlink( "dipsy" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: unlink po\n");    
    err = unlink( "po" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: cd ..\n"); 
    err = chdir( ".." );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( "/jffs2" );
    
    diag_printf("<INFO>: rmdir noonoo\n"); 
    err = rmdir( "noonoo" );
    if( err < 0 ) SHOW_RESULT( rmdir, err );

    // --------------------------------------------------------------

    err = mkdir( "x", 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );
    
    err = mkdir( "x/y", 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );
    
    err = mkdir( "x/y/z", 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );

    err = mkdir( "x/y/z/w", 0 );
    if( err < 0 ) SHOW_RESULT( mkdir, err );
    
    diag_printf("<INFO>: cd /jffs2/x/y/z/w\n");
    err = chdir( "/jffs2/x/y/z/w" );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( "/jffs2/x/y/z/w" );

    diag_printf("<INFO>: cd ..\n");
    err = chdir( ".." );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( "/jffs2/x/y/z" );
    
    diag_printf("<INFO>: cd .\n");
    err = chdir( "." );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( "/jffs2/x/y/z" );

    diag_printf("<INFO>: cd ../../y\n");
    err = chdir( "../../y" );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( "/jffs2/x/y" );

    diag_printf("<INFO>: cd ../..\n");
    err = chdir( "../.." );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( "/jffs2" );

    diag_printf("<INFO>: rmdir x/y/z/w\n"); 
    err = rmdir( "x/y/z/w" );
    if( err < 0 ) SHOW_RESULT( rmdir, err );

    diag_printf("<INFO>: rmdir x/y/z\n"); 
    err = rmdir( "x/y/z" );
    if( err < 0 ) SHOW_RESULT( rmdir, err );

    diag_printf("<INFO>: rmdir x/y\n"); 
    err = rmdir( "x/y" );
    if( err < 0 ) SHOW_RESULT( rmdir, err );

    diag_printf("<INFO>: rmdir x\n"); 
    err = rmdir( "x" );
    if( err < 0 ) SHOW_RESULT( rmdir, err );
    
    // --------------------------------------------------------------

    diag_printf("<INFO>: unlink tinky\n");    
    err = unlink( "tinky" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: unlink laalaa\n");    
    err = unlink( "laalaa" );
    if( err < 0 ) SHOW_RESULT( unlink, err );

    diag_printf("<INFO>: cd /\n");    
    err = chdir( "/" );
    if( err < 0 ) SHOW_RESULT( chdir, err );
    checkcwd( "/" );
    
    diag_printf("<INFO>: umount /jffs2\n");    
    err = umount( "/jffs2" );
    if( err < 0 ) SHOW_RESULT( umount, err );    
    
#ifdef CYGDAT_IO_FLASH_BLOCK_DEVICE_NAME_2
    diag_printf("<INFO>: mounting second JFFS2 filesystem on /mnt\n");
    
    err = mount( CYGDAT_IO_FLASH_BLOCK_DEVICE_NAME_2, "/mnt", "jffs2" );
    if( err < 0 ) SHOW_RESULT( mount, err );    

    err = chdir( "/" );
    if( err < 0 ) SHOW_RESULT( chdir, err );

    checkcwd( "/" );
    
    listdir( "/", true, -1, &existingdirents );
    if ( existingdirents < 2 )
        CYG_TEST_FAIL("Not enough dir entries\n");

    listdir( "/mnt", true, -1, &existingdirents );
    if ( existingdirents < 2 )
        CYG_TEST_FAIL("Not enough dir entries\n");

    diag_printf("<INFO>: umount /mnt\n");    
    err = umount( "/mnt" );
#endif

    diag_printf("<INFO>: umount /j\n");    
    err = umount( "/j" );
    if( err < 0 ) SHOW_RESULT( umount, err );    
}

// -------------------------------------------------------------------------
// EOF fileio1.c
