/*=============================================================================
//
//      hal_diag.c
//
//      HAL diagnostic output code
//
//=============================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003, 2004 Free Software Foundation, Inc.
// Copyright (C) 2003, 2004 eCosCentric Limited                             
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//=============================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):   nickg
// Contributors:nickg
// Date:        2004-07-05
// Purpose:     HAL diagnostic output
// Description: Implementations of HAL diagnostic output support.
//
//####DESCRIPTIONEND####
//
//===========================================================================*/

#include <pkgconf/hal.h>
#include CYGBLD_HAL_VARIANT_H           // Variant specific configuration
#include CYGBLD_HAL_PLATFORM_H          // Platform specific configuration
#include <pkgconf/hal_arm_arm9_mc9328mxl.h>

#include <cyg/infra/cyg_type.h>         // base types
#include <cyg/infra/cyg_trac.h>         // tracing macros
#include <cyg/infra/cyg_ass.h>          // assertion macros

#include <cyg/hal/hal_arch.h>           // basic machine info
#include <cyg/hal/hal_intr.h>           // interrupt macros
#include <cyg/hal/hal_io.h>             // IO macros
#include <cyg/hal/hal_diag.h>
#include <cyg/hal/drv_api.h>
#include <cyg/hal/hal_if.h>             // interface API
#include <cyg/hal/hal_misc.h>           // Helper functions
#include <cyg/hal/mc9328mxl.h>         // platform definitions

//-----------------------------------------------------------------------------
typedef struct {
    cyg_uint32  base;
    cyg_int32   msec_timeout;
    int         isr_vector;
    int         baud_rate;
    int         int_state;
} channel_data_t;

static channel_data_t mc9328mxl_ser_channels[] = {
    { MC9328MXL_UART1, 1000, CYGNUM_HAL_INTERRUPT_UART1_RX, CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD },
    { MC9328MXL_UART2, 1000, CYGNUM_HAL_INTERRUPT_UART2_RX, CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD }
};
#define NUMOF(t) (sizeof(t)/sizeof(t[0]))

//-----------------------------------------------------------------------------

static void
cyg_hal_plf_serial_init_channel(void *__ch_data)
{
    cyg_uint32 base = ((channel_data_t*)__ch_data)->base;
    cyg_uint32 baud = ((channel_data_t*)__ch_data)->baud_rate;
    cyg_uint32 reg;
    
    // Disable UART
    HAL_WRITE_UINT32( base+MC9328MXL_UART_CR1, 0 );
    HAL_WRITE_UINT32( base+MC9328MXL_UART_CR3, 0 );
    
    // Set clock
    HAL_READ_UINT32( base+MC9328MXL_UART_CR4, reg );
    reg |= MC9328MXL_UART_CR4_CTSTL_32 |
           MC9328MXL_UART_CR4_REF16    |
           MC9328MXL_UART_CR4_DREN;
    HAL_WRITE_UINT32( base+MC9328MXL_UART_CR4, reg );
    
    // Enable FIFOs
    HAL_WRITE_UINT32( base+MC9328MXL_UART_FCR,
                      (1 << MC9328MXL_UART_FCR_RXTL_SHF) |
                      MC9328MXL_UART_FCR_RFDIV_1 |
                      (2 << MC9328MXL_UART_FCR_TXTL_SHF) );

    // Set 8N1
    HAL_WRITE_UINT32( base+MC9328MXL_UART_CR2,
                      MC9328MXL_UART_CR2_WS   |
                      MC9328MXL_UART_CR2_IRTS );

    // Enable UART
    HAL_WRITE_UINT32( base+MC9328MXL_UART_CR1, MC9328MXL_UART_CR1_UARTCLKEN | MC9328MXL_UART_CR1_UARTEN );
    HAL_READ_UINT32( base+MC9328MXL_UART_CR2, reg );
    reg |= MC9328MXL_UART_CR2_SRST_ | MC9328MXL_UART_CR2_RXEN  | MC9328MXL_UART_CR2_TXEN;
    HAL_WRITE_UINT32( base+MC9328MXL_UART_CR2, reg );

    // Set baud rate
    HAL_WRITE_UINT32( base+MC9328MXL_UART_BIR, baud/100 - 1 );
    HAL_WRITE_UINT32( base+MC9328MXL_UART_BRM, 10000 - 1 );
    
    // Clear all status flags
    HAL_WRITE_UINT32( base+MC9328MXL_UART_SR1, 0xFFFFFFFF );
    HAL_WRITE_UINT32( base+MC9328MXL_UART_SR2, 0xFFFFFFFF );

}

void
cyg_hal_plf_serial_putc(void *__ch_data, char c)
{
    cyg_uint32 base = ((channel_data_t*)__ch_data)->base;
    cyg_uint32 status;

    do {
        HAL_READ_UINT32(base+MC9328MXL_UART_TS, status);
    } while (status & MC9328MXL_UART_TS_TXFULL);

    HAL_WRITE_UINT32(base+MC9328MXL_UART_TXR, c);
}

static cyg_bool
cyg_hal_plf_serial_getc_nonblock(void *__ch_data, cyg_uint8 *ch)
{
    cyg_uint32 base = ((channel_data_t*)__ch_data)->base;
    cyg_uint32 status;
    cyg_uint32 c;

    HAL_READ_UINT32(base+MC9328MXL_UART_TS, status);
    if ((status & MC9328MXL_UART_TS_RXEMPTY) != 0)
        return false;

    HAL_READ_UINT32(base+MC9328MXL_UART_RXR, c);
    *ch = c;

    return true;
}

cyg_uint8
cyg_hal_plf_serial_getc(void *__ch_data)
{
    cyg_uint8 ch;

    while(!cyg_hal_plf_serial_getc_nonblock(__ch_data, &ch));
    return ch;
}

static void
cyg_hal_plf_serial_write(void *__ch_data, const cyg_uint8 *__buf, 
                         cyg_uint32 __len)
{
    while(__len-- > 0)
        cyg_hal_plf_serial_putc(__ch_data, *__buf++);
}

static void
cyg_hal_plf_serial_read(void *__ch_data, cyg_uint8 *__buf, cyg_uint32 __len)
{
    while(__len-- > 0)
        *__buf++ = cyg_hal_plf_serial_getc(__ch_data);
}

cyg_bool
cyg_hal_plf_serial_getc_timeout(void *__ch_data, cyg_uint8 *ch)
{
    int delay_count;
    channel_data_t *chan = (channel_data_t*)__ch_data;
    cyg_bool res;

    delay_count = chan->msec_timeout * 10; // delay in .1 ms steps

    for(;;) {
        res = cyg_hal_plf_serial_getc_nonblock(__ch_data, ch);
        if (res || 0 == delay_count--)
            break;
        
        CYGACC_CALL_IF_DELAY_US(100);
    }
    return res;
}

static int
cyg_hal_plf_serial_control(void *__ch_data, __comm_control_cmd_t __func, ...)
{
    channel_data_t *chan = (channel_data_t*)__ch_data;
    cyg_uint32 base = ((channel_data_t*)__ch_data)->base;
    int ret = 0;
    cyg_uint32 reg;
    va_list ap;

    CYGARC_HAL_SAVE_GP();
    
    va_start(ap, __func);
    
    switch (__func) {
    case __COMMCTL_IRQ_ENABLE:
        chan -> int_state = 1;
        HAL_READ_UINT32(base+MC9328MXL_UART_CR1, reg );
        reg |= MC9328MXL_UART_CR1_RRDYEN;
        HAL_WRITE_UINT32(base+MC9328MXL_UART_CR1, reg );
        HAL_INTERRUPT_UNMASK(chan->isr_vector);
        break;
    case __COMMCTL_IRQ_DISABLE:
        ret = chan->int_state;
        chan->int_state = 0;
        HAL_READ_UINT32(base+MC9328MXL_UART_CR1, reg );
        reg &= ~MC9328MXL_UART_CR1_RRDYEN;
        HAL_WRITE_UINT32(base+MC9328MXL_UART_CR1, reg );
        HAL_INTERRUPT_MASK(chan->isr_vector);
        break;
    case __COMMCTL_GETBAUD:
        ret = chan->baud_rate;
        break;
    case __COMMCTL_SETBAUD:
        chan->baud_rate = va_arg(ap, cyg_int32);
        cyg_hal_plf_serial_init_channel(chan);
        ret = 0;
        break;        
    case __COMMCTL_DBG_ISR_VECTOR:
        ret = chan->isr_vector;
        break;
    case __COMMCTL_SET_TIMEOUT:
        ret = chan->msec_timeout;
        chan->msec_timeout = va_arg(ap, cyg_uint32);
    default:
        break;
    }

    va_end(ap);

    CYGARC_HAL_RESTORE_GP();

    return ret;
}

static int
cyg_hal_plf_serial_isr(void *__ch_data, int *__ctrlc, 
                       CYG_ADDRWORD __vector, CYG_ADDRWORD __data)
{
    int res = 0;
    channel_data_t *chan = (channel_data_t *)__ch_data;
    cyg_uint32 base = ((channel_data_t *)__ch_data)->base;
    cyg_uint32 status;
    cyg_uint32 _c;
    char c;

    cyg_drv_interrupt_acknowledge(chan->isr_vector);
    *__ctrlc = 0;
    HAL_READ_UINT32(base+MC9328MXL_UART_TS, status);
    if ((status & MC9328MXL_UART_TS_RXEMPTY) == 0) {
        HAL_READ_UINT32(base+MC9328MXL_UART_RXR, _c);
        c = (char)_c;
        if( cyg_hal_is_break( &c , 1 ) )
            *__ctrlc = 1;
        res = CYG_ISR_HANDLED;
    }
    return res;
}

static void
cyg_hal_plf_serial_init(void)
{
    hal_virtual_comm_table_t *comm;
    int cur = CYGACC_CALL_IF_SET_CONSOLE_COMM(CYGNUM_CALL_IF_SET_COMM_ID_QUERY_CURRENT);
    int i;

    for (i = 0;  i < NUMOF(mc9328mxl_ser_channels);  i++) {
        // Init channels
        mc9328mxl_ser_channels[i].base = CYGARC_VIRTUAL_ADDRESS(mc9328mxl_ser_channels[i].base);
        cyg_hal_plf_serial_init_channel(&mc9328mxl_ser_channels[i]);
        // Setup procs in the vector table
        CYGACC_CALL_IF_SET_CONSOLE_COMM(i);
        comm = CYGACC_CALL_IF_CONSOLE_PROCS();
        CYGACC_COMM_IF_CH_DATA_SET(*comm, &mc9328mxl_ser_channels[i]);
        CYGACC_COMM_IF_WRITE_SET(*comm, cyg_hal_plf_serial_write);
        CYGACC_COMM_IF_READ_SET(*comm, cyg_hal_plf_serial_read);
        CYGACC_COMM_IF_PUTC_SET(*comm, cyg_hal_plf_serial_putc);
        CYGACC_COMM_IF_GETC_SET(*comm, cyg_hal_plf_serial_getc);
        CYGACC_COMM_IF_CONTROL_SET(*comm, cyg_hal_plf_serial_control);
        CYGACC_COMM_IF_DBG_ISR_SET(*comm, cyg_hal_plf_serial_isr);
        CYGACC_COMM_IF_GETC_TIMEOUT_SET(*comm, cyg_hal_plf_serial_getc_timeout);
    }

    // Restore original console
    CYGACC_CALL_IF_SET_CONSOLE_COMM(cur);
}

void
cyg_hal_plf_comms_init(void)
{
    static int initialized = 0;

    if (initialized)
        return;

    initialized = 1;

    cyg_hal_plf_serial_init();
}

/*---------------------------------------------------------------------------*/
/* End of hal_diag.c */
