/*=============================================================================
//
//      hal_diag.c
//
//      HAL diagnostic output code
//
//=============================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005 Free Software Foundation, Inc.
// Copyright (C) 2003, 2004, 2005 eCosCentric Limited                       
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//=============================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):   nickg, eCosCentric
// Contributors:nickg, eCosCentric
// Date:        2001-04-23
// Purpose:     HAL diagnostic output
// Description: Implementations of HAL diagnostic output support.
//
//####DESCRIPTIONEND####
//
//===========================================================================*/

#include <pkgconf/hal.h>
#include CYGBLD_HAL_VARIANT_H           // Variant specific configuration
#include CYGBLD_HAL_PLATFORM_H          // Platform specific configuration
#include <pkgconf/hal_arm_arm9_sam9.h>

#include <cyg/infra/cyg_type.h>         // base types
#include <cyg/infra/cyg_trac.h>         // tracing macros
#include <cyg/infra/cyg_ass.h>          // assertion macros

#include <cyg/hal/hal_arch.h>           // basic machine info
#include <cyg/hal/hal_intr.h>           // interrupt macros
#include <cyg/hal/hal_io.h>             // IO macros
#include <cyg/hal/hal_diag.h>
#include <cyg/hal/drv_api.h>
#include <cyg/hal/hal_if.h>             // interface API
#include <cyg/hal/hal_misc.h>           // Helper functions
#include <cyg/hal/sam9.h>         // platform definitions

#define BAUD_RATE(n) (((CYGARC_HAL_ARM_ARM9_SAM9_MCK)/16)/n)

#define RCV_TIMEOUT 10

//-----------------------------------------------------------------------------

typedef struct {
    cyg_uint8   *base;
    cyg_int32   msec_timeout;
    int         isr_vector;
    cyg_int32   baud_rate;
    int         int_state;
    char        rcv_buffer[2][1];
    int         rcv_current;
    int         rcv_pos;
    int         rcv_end;
} channel_data_t;

static channel_data_t sam9_ser_channels[] = {
#if CYGINT_HAL_ARM_AT91_SERIAL_DEBUG > 0
    { (cyg_uint8 *)SAM9_DEBUG, 1000, CYGNUM_HAL_INTERRUPT_DEBUG },
#endif
#if CYGINT_HAL_ARM_AT91_SERIAL_UART0 > 0
    { (cyg_uint8 *)SAM9_UART0, 1000, CYGNUM_HAL_INTERRUPT_US0 },
#endif
#if CYGINT_HAL_ARM_AT91_SERIAL_UART1 > 0
    { (cyg_uint8 *)SAM9_UART1, 1000, CYGNUM_HAL_INTERRUPT_US1 },
#endif
#if CYGINT_HAL_ARM_AT91_SERIAL_UART2 > 0
    { (cyg_uint8 *)SAM9_UART2, 1000, CYGNUM_HAL_INTERRUPT_US2 },
#endif
#if CYGINT_HAL_ARM_AT91_SERIAL_UART3 > 0
    { (cyg_uint8 *)SAM9_UART3, 1000, CYGNUM_HAL_INTERRUPT_US3 },
#endif
};
#define NUMOF(t) (sizeof(t)/sizeof(t[0]))

//-----------------------------------------------------------------------------

void
cyg_hal_plf_serial_putc(void *__ch_data, char c);

void cyg_hal_plf_serial_putc0(char c)
{
    cyg_hal_plf_serial_putc( &sam9_ser_channels[0], c );
}

static void
cyg_hal_plf_serial_init_channel(void *__ch_data)
{
    channel_data_t *chan = (channel_data_t*)__ch_data;
    cyg_uint8 *base = ((channel_data_t*)__ch_data)->base;

    HAL_WRITE_UINT32(base+_UART_CR, (_UART_CR_RSTRX|
                                     _UART_CR_RSTTX|
                                     _UART_CR_RSTSTA));
    HAL_WRITE_UINT32(base+_UART_IDR, 0xFFFFFFFF);
    HAL_INTERRUPT_MASK(chan->isr_vector);
    HAL_WRITE_UINT32(base+_UART_BRG, BAUD_RATE(chan->baud_rate));
    HAL_WRITE_UINT32(base+_UART_MR, (_UART_MR_USMODE_NORMAL|
                                     _UART_MR_CHRL_8_BITS|
                                     _UART_MR_PAR_NONE|
                                     _UART_MR_NBSTOP_1_BIT));

    // Set up PDC to receive data into the first buffer
    HAL_WRITE_UINT32(base + AT91_US_RPR, CYGARC_PHYSICAL_ADDRESS(chan->rcv_buffer[0]));
    HAL_WRITE_UINT32(base + AT91_US_RTO, RCV_TIMEOUT);
    HAL_WRITE_UINT32(base + AT91_US_RCR, sizeof(chan->rcv_buffer[0]));
    AT91_PDC_RX_ENABLE( base );

    HAL_WRITE_UINT32(base+_UART_CR, (_UART_CR_RXEN|_UART_CR_TXEN|_UART_CR_DTREN|_UART_CR_RTSEN));

    chan->rcv_current = 1;
    chan->rcv_pos = 0;
    chan->rcv_end = 0;
}

void
cyg_hal_plf_serial_putc(void *__ch_data, char c)
{
    cyg_uint8 *base = ((channel_data_t*)__ch_data)->base;
    cyg_uint32 status;

    do {
        HAL_READ_UINT32(base+_UART_CSR, status);
    } while ((status & _UART_CSR_TxRDY) == 0);

    HAL_WRITE_UINT32(base+_UART_THR, c);
}

static cyg_bool
cyg_hal_plf_serial_getc_nonblock(void *__ch_data, cyg_uint8 *ch)
{
    channel_data_t *chan = (channel_data_t*)__ch_data;
    cyg_uint8 *base = ((channel_data_t*)__ch_data)->base;
    cyg_uint8 *c;
    cyg_uint32 end;

    if( chan->rcv_pos >= chan->rcv_end )
    {
        // If there is no data waiting in the current buffer, try to
        // swap the buffers over to receive any data that is waiting
        // in the buffer the hardware is currently using.
        AT91_PDC_RX_DISABLE( base );
        HAL_READ_UINT32(base + AT91_US_RPR, end);
        if( end == CYGARC_PHYSICAL_ADDRESS(chan->rcv_buffer[chan->rcv_current^1]) )
        {
            // If there is no data in the buffer yet, don't swap and just
            // report no data available.
            AT91_PDC_RX_ENABLE( base );
            return false;
        }

        // Point the PDC at the current, now empty, buffer.
        HAL_WRITE_UINT32(base + AT91_US_RPR, CYGARC_PHYSICAL_ADDRESS(chan->rcv_buffer[chan->rcv_current]));
        HAL_WRITE_UINT32(base + AT91_US_RCR, sizeof(chan->rcv_buffer[chan->rcv_current]));
        AT91_PDC_RX_ENABLE( base );

        // Set up to extract bytes from the buffer we just got back
        // from the hardware.
        chan->rcv_current ^= 1;
        chan->rcv_pos = 0;
        chan->rcv_end = end - CYGARC_PHYSICAL_ADDRESS(chan->rcv_buffer[chan->rcv_current]);

        // If the buffer is empty, report no data available.
        if( chan->rcv_end == 0 )
            return false;
    }

    // Fetch a byte from the (uncached) version of the buffer and return it.
    c = (cyg_uint8 *)CYGARC_UNCACHED_ADDRESS(&chan->rcv_buffer[chan->rcv_current][chan->rcv_pos]);
    chan->rcv_pos++;
    
    *ch = *c;

    return true;
}

cyg_uint8
cyg_hal_plf_serial_getc(void *__ch_data)
{
    cyg_uint8 ch;

    while(!cyg_hal_plf_serial_getc_nonblock(__ch_data, &ch));
    return ch;
}

static void
cyg_hal_plf_serial_write(void *__ch_data, const cyg_uint8 *__buf, 
                         cyg_uint32 __len)
{
    while(__len-- > 0)
        cyg_hal_plf_serial_putc(__ch_data, *__buf++);
}

static void
cyg_hal_plf_serial_read(void *__ch_data, cyg_uint8 *__buf, cyg_uint32 __len)
{
    while(__len-- > 0)
        *__buf++ = cyg_hal_plf_serial_getc(__ch_data);
}

cyg_bool
cyg_hal_plf_serial_getc_timeout(void *__ch_data, cyg_uint8 *ch)
{
    int delay_count;
    channel_data_t *chan = (channel_data_t*)__ch_data;
    cyg_bool res;

    delay_count = chan->msec_timeout * 10; // delay in .1 ms steps

    for(;;) {
        res = cyg_hal_plf_serial_getc_nonblock(__ch_data, ch);
        if (res || 0 == delay_count--)
            break;
        
        CYGACC_CALL_IF_DELAY_US(100);
    }
    return res;
}

static int
cyg_hal_plf_serial_control(void *__ch_data, __comm_control_cmd_t __func, ...)
{
    channel_data_t *chan = (channel_data_t*)__ch_data;
    cyg_uint8 *base = ((channel_data_t*)__ch_data)->base;
    int ret = 0;

    switch (__func) {
    case __COMMCTL_IRQ_ENABLE:
        chan -> int_state = 1;
        HAL_WRITE_UINT32(base+_UART_IER, _UART_IER_ENDRX );
        HAL_INTERRUPT_UNMASK(chan->isr_vector);
        break;
    case __COMMCTL_IRQ_DISABLE:
        ret = chan->int_state;
        chan->int_state = 0;
        HAL_WRITE_UINT32(base+_UART_IDR, _UART_IER_ENDRX );
        HAL_INTERRUPT_MASK(chan->isr_vector);
        break;
    case __COMMCTL_GETBAUD:
        ret = chan->baud_rate;
        break;
    case __COMMCTL_SETBAUD:
    {
        cyg_uint32 b;
        va_list ap;

        va_start(ap, __func);
        b = va_arg(ap, cyg_int32);
        va_end(ap);

        chan->baud_rate = b;
        HAL_WRITE_UINT32(base+_UART_BRG, BAUD_RATE(b));
        ret = 1;
        break;
    }
    case __COMMCTL_DBG_ISR_VECTOR:
        ret = chan->isr_vector;
        break;
    case __COMMCTL_SET_TIMEOUT:
    {
        va_list ap;

        va_start(ap, __func);

        ret = chan->msec_timeout;
        chan->msec_timeout = va_arg(ap, cyg_uint32);

        va_end(ap);
    }        
    default:
        break;
    }
    return ret;
}

static int
cyg_hal_plf_serial_isr(void *__ch_data, int *__ctrlc, 
                       CYG_ADDRWORD __vector, CYG_ADDRWORD __data)
{
    int res = 0;    
    cyg_uint8 c;

    if( cyg_hal_plf_serial_getc_nonblock( __ch_data, &c ) )
    {
        if( cyg_hal_is_break( &c , 1 ) )
            *__ctrlc = 1;
        res = CYG_ISR_HANDLED;
    }
    return res;
}

static void
cyg_hal_plf_serial_init(void)
{
    hal_virtual_comm_table_t *comm;
    int cur = CYGACC_CALL_IF_SET_CONSOLE_COMM(CYGNUM_CALL_IF_SET_COMM_ID_QUERY_CURRENT);
    int i;

    for (i = 0;  i < NUMOF(sam9_ser_channels);  i++) {
        // Init channels
        sam9_ser_channels[i].baud_rate = CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD;
        cyg_hal_plf_serial_init_channel(&sam9_ser_channels[i]);
        // Setup procs in the vector table
        CYGACC_CALL_IF_SET_CONSOLE_COMM(i);
        comm = CYGACC_CALL_IF_CONSOLE_PROCS();
        CYGACC_COMM_IF_CH_DATA_SET(*comm, &sam9_ser_channels[i]);
        CYGACC_COMM_IF_WRITE_SET(*comm, cyg_hal_plf_serial_write);
        CYGACC_COMM_IF_READ_SET(*comm, cyg_hal_plf_serial_read);
        CYGACC_COMM_IF_PUTC_SET(*comm, cyg_hal_plf_serial_putc);
        CYGACC_COMM_IF_GETC_SET(*comm, cyg_hal_plf_serial_getc);
        CYGACC_COMM_IF_CONTROL_SET(*comm, cyg_hal_plf_serial_control);
        CYGACC_COMM_IF_DBG_ISR_SET(*comm, cyg_hal_plf_serial_isr);
        CYGACC_COMM_IF_GETC_TIMEOUT_SET(*comm, cyg_hal_plf_serial_getc_timeout);
    }

    // Restore original console
    CYGACC_CALL_IF_SET_CONSOLE_COMM(cur);
}

void
cyg_hal_plf_comms_init(void)
{
    static int initialized = 0;

    if (initialized)
        return;

    initialized = 1;

    cyg_hal_plf_serial_init();
}

/*---------------------------------------------------------------------------*/
/* End of hal_diag.c */
