#ifndef CYGONCE_HAL_CACHE_H
#define CYGONCE_HAL_CACHE_H

//=============================================================================
//
//      hal_cache.h
//
//      HAL cache control API
//
//=============================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2004, 2008 Free Software Foundation, Inc.
// Copyright (C) 2004, 2008 eCosCentric Limited                             
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//=============================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):   gthomas
// Contributors:hmt, jskov
//              Travis C. Furrer <furrer@mit.edu>
//              Tom Chase        <tomc@dtccom.com>
// Date:        2005-05-10
// Purpose:     Cache control API
// Description: The macros defined here provide the HAL APIs for handling
//              cache control operations.
// Usage:
//              #include <cyg/hal/hal_cache.h>
//              ...
//              
//
//####DESCRIPTIONEND####
//
//=============================================================================

#include <pkgconf/hal.h>
#include <cyg/infra/cyg_type.h>
#include <cyg/hal/hal_mmu.h>

//-----------------------------------------------------------------------------
// Cache dimensions

#if defined(CYGPKG_HAL_ARM_ARM9_ARM920T)
# define HAL_ICACHE_SIZE                 0x4000
# define HAL_ICACHE_LINE_SIZE            32
# define HAL_ICACHE_WAYS                 64
# define HAL_ICACHE_SETS (HAL_ICACHE_SIZE/(HAL_ICACHE_LINE_SIZE*HAL_ICACHE_WAYS))

# define HAL_DCACHE_SIZE                 0x4000
# define HAL_DCACHE_LINE_SIZE            32
# define HAL_DCACHE_WAYS                 64
# define HAL_DCACHE_SETS (HAL_DCACHE_SIZE/(HAL_DCACHE_LINE_SIZE*HAL_DCACHE_WAYS))

# define HAL_WRITE_BUFFER                64

# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX
# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_STEP  0x20
# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_LIMIT 0x100

#elif defined(CYGPKG_HAL_ARM_ARM9_ARM922T)
# define HAL_ICACHE_SIZE                 0x2000
# define HAL_ICACHE_LINE_SIZE            32
# define HAL_ICACHE_WAYS                 64
# define HAL_ICACHE_SETS (HAL_ICACHE_SIZE/(HAL_ICACHE_LINE_SIZE*HAL_ICACHE_WAYS))

# define HAL_DCACHE_SIZE                 0x2000
# define HAL_DCACHE_LINE_SIZE            32
# define HAL_DCACHE_WAYS                 64
# define HAL_DCACHE_SETS (HAL_DCACHE_SIZE/(HAL_DCACHE_LINE_SIZE*HAL_DCACHE_WAYS))

# define HAL_WRITE_BUFFER                64

# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX
# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_STEP  0x20
# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_LIMIT 0x80

#elif defined(CYGPKG_HAL_ARM_ARM9_ARM925T)
# define HAL_ICACHE_SIZE                 0x4000
# define HAL_ICACHE_LINE_SIZE            16
# define HAL_ICACHE_WAYS                 2
# define HAL_ICACHE_SETS (HAL_ICACHE_SIZE/(HAL_ICACHE_LINE_SIZE*HAL_ICACHE_WAYS))

# define HAL_DCACHE_SIZE                 0x2000
# define HAL_DCACHE_LINE_SIZE            16
# define HAL_DCACHE_WAYS                 2
# define HAL_DCACHE_SETS (HAL_DCACHE_SIZE/(HAL_DCACHE_LINE_SIZE*HAL_DCACHE_WAYS))

# define HAL_WRITE_BUFFER                64
// must flush everything manually
# define CYGHWR_HAL_ARM_ARM9_ALT_CLEAN_DCACHE 

#elif defined(CYGPKG_HAL_ARM_ARM9_ARM926EJ)
# ifndef HAL_ICACHE_SIZE
#  define HAL_ICACHE_SIZE                 0x4000
# endif
# define HAL_ICACHE_LINE_SIZE            32
# define HAL_ICACHE_WAYS                 4
# define HAL_ICACHE_SETS (HAL_ICACHE_SIZE/(HAL_ICACHE_LINE_SIZE*HAL_ICACHE_WAYS))

# ifndef HAL_DCACHE_SIZE
#  define HAL_DCACHE_SIZE                 0x2000
# endif
# define HAL_DCACHE_LINE_SIZE            32
# define HAL_DCACHE_WAYS                 4
# define HAL_DCACHE_SETS (HAL_DCACHE_SIZE/(HAL_DCACHE_LINE_SIZE*HAL_DCACHE_WAYS))

# define HAL_WRITE_BUFFER                64

#define CYGHWR_HAL_ARM_ARM926EJ_CLEAN_DCACHE //has instruction to clean D-cache

#elif defined(CYGPKG_HAL_ARM_ARM9_ARM926E)
# define HAL_ICACHE_SIZE                 0x8000
# define HAL_ICACHE_LINE_SIZE            32
# define HAL_ICACHE_WAYS                 4
# define HAL_ICACHE_SETS (HAL_ICACHE_SIZE/(HAL_ICACHE_LINE_SIZE*HAL_ICACHE_WAYS))

# define HAL_DCACHE_SIZE                 0x8000
# define HAL_DCACHE_LINE_SIZE            32
# define HAL_DCACHE_WAYS                 4
# define HAL_DCACHE_SETS (HAL_DCACHE_SIZE/(HAL_DCACHE_LINE_SIZE*HAL_DCACHE_WAYS))

# define HAL_WRITE_BUFFER                64

# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX
# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_STEP  0x0020
# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_LIMIT 0x2000

#elif defined(CYGPKG_HAL_ARM_ARM9_ARM940T)
# define HAL_ICACHE_SIZE                 0x1000
# define HAL_ICACHE_LINE_SIZE            16
# define HAL_ICACHE_WAYS                 4
# define HAL_ICACHE_SETS (HAL_ICACHE_SIZE/(HAL_ICACHE_LINE_SIZE*HAL_ICACHE_WAYS))

# define HAL_DCACHE_SIZE                 0x1000
# define HAL_DCACHE_LINE_SIZE            16
# define HAL_DCACHE_WAYS                 4
# define HAL_DCACHE_SETS (HAL_DCACHE_SIZE/(HAL_DCACHE_LINE_SIZE*HAL_DCACHE_WAYS))

# define HAL_WRITE_BUFFER                32

# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX
# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_STEP  0x10
# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_LIMIT 0x40

#elif defined(CYGPKG_HAL_ARM_ARM9_ARM946E)
# define HAL_ICACHE_SIZE                 0x4000
# define HAL_ICACHE_LINE_SIZE            32
# define HAL_ICACHE_WAYS                 4
# define HAL_ICACHE_SETS (HAL_ICACHE_SIZE/(HAL_ICACHE_LINE_SIZE*HAL_ICACHE_WAYS))

# define HAL_DCACHE_SIZE                 0x1000
# define HAL_DCACHE_LINE_SIZE            32
# define HAL_DCACHE_WAYS                 4
# define HAL_DCACHE_SETS (HAL_DCACHE_SIZE/(HAL_DCACHE_LINE_SIZE*HAL_DCACHE_WAYS))

# define HAL_WRITE_BUFFER                32

# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX
# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_STEP  (1<<5)
# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_LIMIT (1<<10)

#elif defined(CYGPKG_HAL_ARM_ARM9_ARM966E)
# define HAL_ICACHE_SIZE                 0
# define HAL_ICACHE_LINE_SIZE            0
# define HAL_ICACHE_WAYS                 0
# define HAL_ICACHE_SETS (HAL_ICACHE_SIZE/(HAL_ICACHE_LINE_SIZE*HAL_ICACHE_WAYS))

# define HAL_DCACHE_SIZE                 0
# define HAL_DCACHE_LINE_SIZE            0
# define HAL_DCACHE_WAYS                 0
# define HAL_DCACHE_SETS (HAL_DCACHE_SIZE/(HAL_DCACHE_LINE_SIZE*HAL_DCACHE_WAYS))

# define HAL_WRITE_BUFFER                32

# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX
# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_STEP  0
# define CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_LIMIT 0

#else
# error "No cache details defined"
#endif

// FIXME: much of the code below should make better use of
// the definitions from hal_mmu.h

//-----------------------------------------------------------------------------
// Global control of Instruction cache

#if HAL_ICACHE_SIZE != 0

// FIXME: disable/enable instruction streaming?

// Enable the instruction cache
__externC void hal_icache_enable(void);
#define HAL_ICACHE_ENABLE() hal_icache_enable()


// Disable the instruction cache (and invalidate it, required semanitcs)
__externC void hal_icache_disable(void);
#define HAL_ICACHE_DISABLE() hal_icache_disable()

// Query the state of the instruction cache
__externC cyg_uint32 hal_icache_is_enabled(void);
#define HAL_ICACHE_IS_ENABLED(_state_)                                  \
CYG_MACRO_START                                                         \
    register cyg_uint32 reg = hal_icache_is_enabled();                  \
    (_state_) = (0 != (0x1000 & reg)); /* Bit 12 is ICache enable */    \
CYG_MACRO_END

// Invalidate the entire cache
__externC void hal_icache_invalidate_all(void);    
#define HAL_ICACHE_INVALIDATE_ALL() hal_icache_invalidate_all()

// Synchronize the contents of the cache with memory.
// (which includes flushing out pending writes)
#define HAL_ICACHE_SYNC()                                       \
CYG_MACRO_START                                                 \
    HAL_DCACHE_SYNC(); /* ensure data gets to RAM */            \
    HAL_ICACHE_INVALIDATE_ALL(); /* forget all we know */       \
CYG_MACRO_END

#else

#define HAL_ICACHE_ENABLE()
#define HAL_ICACHE_DISABLE()
#define HAL_ICACHE_IS_ENABLED(_state_) ((_state_) = 0)
#define HAL_ICACHE_INVALIDATE_ALL()
#define HAL_ICACHE_SYNC()

#endif

// Set the instruction cache refill burst size
//#define HAL_ICACHE_BURST_SIZE(_size_)

// Load the contents of the given address range into the instruction cache
// and then lock the cache so that it stays there.
//#define HAL_ICACHE_LOCK(_base_, _size_)

// Undo a previous lock operation
//#define HAL_ICACHE_UNLOCK(_base_, _size_)

// Unlock entire cache
//#define HAL_ICACHE_UNLOCK_ALL()

//-----------------------------------------------------------------------------
// Instruction cache line control

// Invalidate cache lines in the given range without writing to memory.
//#define HAL_ICACHE_INVALIDATE( _base_ , _size_ )

//-----------------------------------------------------------------------------
// Global control of data cache

#if HAL_DCACHE_SIZE != 0

// Enable the data cache
__externC void hal_dcache_enable(void);
#define HAL_DCACHE_ENABLE() hal_dcache_enable()

// Disable the data cache (and invalidate it, required semanitcs)
__externC void hal_dcache_disable(void);
#define HAL_DCACHE_DISABLE() hal_dcache_disable()

// Query the state of the data cache
__externC cyg_uint32 hal_dcache_is_enabled(void);
#define HAL_DCACHE_IS_ENABLED(_state_)                          \
CYG_MACRO_START                                                 \
    register cyg_uint32 reg = hal_dcache_is_enabled();          \
    (_state_) = (0 != (4 & reg)); /* Bit 2 is DCache enable */  \
CYG_MACRO_END

// Flush the entire dcache (and then both TLBs, just in case)
__externC void hal_dcache_invalidate_all(void);
#define HAL_DCACHE_INVALIDATE_ALL() hal_dcache_invalidate_all()

// Synchronize the contents of the cache with memory.
#ifdef CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE
__externC void hal_dcache_sync_clean(void);
#define HAL_DCACHE_SYNC() hal_dcache_sync_clean()
#elif defined(CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX)
__externC void hal_dcache_sync_index(cyg_uint32 step, cyg_uint32 limit);
#define HAL_DCACHE_SYNC()                                                       \
        hal_dcache_sync_index(CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_STEP,      \
                              CYGHWR_HAL_ARM_ARM9_CLEAN_DCACHE_INDEX_LIMIT)
#elif defined(CYGHWR_HAL_ARM_ARM9_ALT_CLEAN_DCACHE)
__externC void hal_dcache_sync_alt_clean(void);
#define HAL_DCACHE_SYNC() hal_dcache_sync_alt_clean()
#elif defined(CYGHWR_HAL_ARM_ARM926EJ_CLEAN_DCACHE)
__externC void hal_dcache_sync_arm926ej_clean(void);
#define HAL_DCACHE_SYNC() hal_dcache_sync_arm926ej_clean()
#else
# error "Don't know how to sync Dcache"
#endif

#else

#define HAL_DCACHE_ENABLE()
#define HAL_DCACHE_DISABLE()
#define HAL_DCACHE_IS_ENABLED(_state_) ((_state_) = 0)
#define HAL_DCACHE_INVALIDATE_ALL()
#define HAL_DCACHE_SYNC()

#endif


// Set the data cache refill burst size
//#define HAL_DCACHE_BURST_SIZE(_size_)

// Set the data cache write mode
//#define HAL_DCACHE_WRITE_MODE( _mode_ )

#define HAL_DCACHE_WRITETHRU_MODE       0
#define HAL_DCACHE_WRITEBACK_MODE       1

// Get the current writeback mode - or only writeback mode if fixed
#define HAL_DCACHE_QUERY_WRITE_MODE( _mode_ ) CYG_MACRO_START           \
    _mode_ = HAL_DCACHE_WRITEBACK_MODE;                                 \
CYG_MACRO_END

// Load the contents of the given address range into the data cache
// and then lock the cache so that it stays there.
//#define HAL_DCACHE_LOCK(_base_, _size_)

// Undo a previous lock operation
//#define HAL_DCACHE_UNLOCK(_base_, _size_)

// Unlock entire cache
//#define HAL_DCACHE_UNLOCK_ALL()

//-----------------------------------------------------------------------------
// Data cache line control

// Allocate cache lines for the given address range without reading its
// contents from memory.
//#define HAL_DCACHE_ALLOCATE( _base_ , _size_ )

// Write dirty cache lines to memory and invalidate the cache entries
// for the given address range.
// ---- this seems not to work despite the documentation ---
//#define HAL_DCACHE_FLUSH( _base_ , _size_ )
//CYG_MACRO_START
//    HAL_DCACHE_STORE( _base_ , _size_ );
//    HAL_DCACHE_INVALIDATE( _base_ , _size_ );
//CYG_MACRO_END

// Invalidate cache lines in the given range without writing to memory.
// ---- this seems not to work despite the documentation ---
//#define HAL_DCACHE_INVALIDATE( _base_ , _size_ )
//CYG_MACRO_START
//    register int addr, enda;
//    for ( addr = (~(HAL_DCACHE_LINE_SIZE - 1)) & (int)(_base_),
//              enda = (int)(_base_) + (_size_);
//          addr < enda ;
//          addr += HAL_DCACHE_LINE_SIZE )
//    {
//        asm volatile (
//                      "mcr  p15,0,%0,c7,c6,1;" /* flush entry away */
//                      :
//                      : "r"(addr)
//                      : "memory"
//            );
//    }
//CYG_MACRO_END
                          
// Write dirty cache lines to memory for the given address range.
// ---- this seems not to work despite the documentation ---
//#define HAL_DCACHE_STORE( _base_ , _size_ )
//CYG_MACRO_START
//    register int addr, enda;
//    for ( addr = (~(HAL_DCACHE_LINE_SIZE - 1)) & (int)(_base_),
//              enda = (int)(_base_) + (_size_);
//          addr < enda ;
//          addr += HAL_DCACHE_LINE_SIZE )
//    {
//        asm volatile ("mcr  p15,0,%0,c7,c10,1;" /* push entry to RAM */
//                      :
//                      : "r"(addr)
//                      : "memory"
//            );
//    }
//CYG_MACRO_END

// Preread the given range into the cache with the intention of reading
// from it later.
//#define HAL_DCACHE_READ_HINT( _base_ , _size_ )

// Preread the given range into the cache with the intention of writing
// to it later.
//#define HAL_DCACHE_WRITE_HINT( _base_ , _size_ )

// Allocate and zero the cache lines associated with the given range.
//#define HAL_DCACHE_ZERO( _base_ , _size_ )

//-----------------------------------------------------------------------------
// Cache controls for safely disabling/reenabling caches around execution
// of relocated code.

#define HAL_FLASH_CACHES_OFF(_d_, _i_)          \
    CYG_MACRO_START                             \
    HAL_ICACHE_IS_ENABLED(_i_);                 \
    HAL_DCACHE_IS_ENABLED(_d_);                 \
    HAL_ICACHE_INVALIDATE_ALL();                \
    HAL_ICACHE_DISABLE();                       \
    HAL_DCACHE_SYNC();                          \
    HAL_DCACHE_INVALIDATE_ALL();                \
    HAL_DCACHE_DISABLE();                       \
    CYG_MACRO_END

#define HAL_FLASH_CACHES_ON(_d_, _i_)           \
    CYG_MACRO_START                             \
    if (_d_) HAL_DCACHE_ENABLE();               \
    if (_i_) HAL_ICACHE_ENABLE();               \
    CYG_MACRO_END

//-----------------------------------------------------------------------------
// Virtual<->Physical translations
#ifndef HAL_VIRT_TO_PHYS_ADDRESS
extern cyg_uint32 hal_virt_to_phys_address(cyg_uint32 phys);
#define HAL_VIRT_TO_PHYS_ADDRESS(_va, _pa) \
   _pa = hal_virt_to_phys_address(_va)
#endif
//-----------------------------------------------------------------------------
#endif // ifndef CYGONCE_HAL_CACHE_H
// End of hal_cache.h
