/*==========================================================================
//
//      at91_misc.c
//
//      HAL misc board support code for Atmel AT91
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2008 Free Software Foundation, Inc.
// Copyright (C) 2004, 2005, 2006, 2008 eCosCentric Limited                 
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    gthomas
// Contributors: gthomas, jskov, nickg, tkoeller
// Date:         2001-07-12
// Purpose:      HAL board support
// Description:  Implementations of HAL board interfaces
//
//####DESCRIPTIONEND####
//
//========================================================================*/

#include <pkgconf/system.h>
#include <pkgconf/hal.h>

#include <cyg/infra/cyg_type.h>         // base types
#include <cyg/infra/cyg_trac.h>         // tracing macros
#include <cyg/infra/cyg_ass.h>          // assertion macros

#include <cyg/hal/hal_io.h>             // IO macros
#include <cyg/hal/hal_arch.h>           // Register state info
#include <cyg/hal/hal_diag.h>
#include <cyg/hal/hal_intr.h>           // necessary?
#include <cyg/hal/hal_cache.h>
#include <cyg/hal/hal_if.h>             // calling interface
#include <cyg/hal/hal_misc.h>           // helper functions
#include <cyg/hal/drv_api.h>            // HAL ISR support
#include <cyg/hal/var_io.h>             // platform registers

// -------------------------------------------------------------------------
// Hardware init

void hal_hardware_init(void)
{
    unsigned i;

#ifdef HAL_PLF_HARDWARE_INIT
    // Perform any platform specific initializations
    HAL_PLF_HARDWARE_INIT();
#endif
        
#ifdef CYGSEM_HAL_VIRTUAL_VECTOR_SUPPORT
    // Set up eCos/ROM interfaces
    hal_if_init();
#endif

    // Reset all interrupts
    HAL_WRITE_UINT32(AT91_AIC+AT91_AIC_IDCR, 0xFFFFFFFF);  

    // Flush internal priority level stack
    for (i = 0; i < 8; ++i)
        HAL_WRITE_UINT32(AT91_AIC+AT91_AIC_EOI, 0xFFFFFFFF);

    // Disable watchdog - a driver can enable it later if needed
#if defined(CYGHWR_HAL_ARM_AT91_R40807) || \
    defined(CYGHWR_HAL_ARM_AT91_R40008) || \
    defined(CYGHWR_HAL_ARM_AT91_M55800A)
    {
        cyg_uint32 wdomr;
        HAL_READ_UINT32( AT91_WD + AT91_WD_OMR, wdomr );
        wdomr &= ~(AT91_WD_OMR_WDEN|AT91_WD_OMR_RSTEN|AT91_WD_OMR_EXTEN);
        wdomr |= AT91_WD_OMR_OKEY;
        HAL_WRITE_UINT32( AT91_WD + AT91_WD_OMR, wdomr );
    }
#elif defined(CYGHWR_HAL_ARM_AT91_M42800A)
    {
        cyg_uint32 wdmr;
        HAL_READ_UINT32( AT91_WD + AT91_ST_WDMR, wdmr );
        wdmr &= ~(AT91_ST_WDMR_EXTEN|AT91_ST_WDMR_RSTEN);
        HAL_WRITE_UINT32( AT91_WD + AT91_ST_WDMR, wdmr );
    }
#endif

#ifdef CYGHWR_HAL_ARM_AT91SAM7A
    {
        cyg_uint32 csr;
        HAL_READ_UINT32(AT91_USART0+AT91_US_CSR, csr);
        HAL_READ_UINT32(AT91_USART1+AT91_US_CSR, csr);
    }
#endif

#if !defined(CYGSEM_HAL_VIRTUAL_VECTOR_SUPPORT) || defined(CYGSEM_HAL_VIRTUAL_VECTOR_CLAIM_DELAY_US)
    // When this config is providing delay functionality, and the kernel is
    // included, then the kernel clock will be used. But use can happen before
    // the clock is initialised by the kernel. So we initialise it here.
    // Update: in fact we need to initialise this at any time we're handling
    // delays, so the PIT gets running.
    hal_clock_initialize( CYGNUM_HAL_RTC_PERIOD );
#endif    

/* Perform some platform specific bits to get the Ethernet hardware
   setup. Specifically if a specific phy is used and does not start in
   the correct mode a function needs to be supplied as part of the plf
   to do the necessary initializations.
*/
#ifdef CYGPKG_DEVS_ETH_ARM_AT91
#ifdef HAL_PLF_ETH_INIT
       HAL_PLF_ETH_INIT();
#endif
#endif
}

// -------------------------------------------------------------------------
// Decode a system interrupt. Not all systems have all interrupts. So
// code will only be generated for those interrupts which have a
// defined value.

#if CYGINT_HAL_ARM_AT91_SYS_INTERRUPT

static int sys_irq_handler(void)
{
  cyg_uint32 sr, mr;

#ifdef CYGNUM_HAL_INTERRUPT_PITC
  // Periodic Interrupt Timer Controller
  HAL_READ_UINT32((AT91_PITC+AT91_PITC_PISR), sr);
  if (sr & AT91_PITC_PISR_PITS) {
    return CYGNUM_HAL_INTERRUPT_PITC;
  }
#endif

#ifdef CYGNUM_HAL_INTERRUPT_DEBUG
  // Debug Unit
  HAL_READ_UINT32((AT91_DEBUG + AT91_DBG_CSR), sr);
  HAL_READ_UINT32((AT91_DEBUG + AT91_DBG_IMR), mr);
  if (sr & mr) {
    return CYGNUM_HAL_INTERRUPT_DEBUG;
  }
#endif

#ifdef CYGNUM_HAL_INTERRUPT_RTTC
  /* Real Time Timer. Check the interrupt is enabled, not that just
     the status indicates there is an interrupt. It takes a while for
     the status bit to clear. */
  HAL_READ_UINT32((AT91_RTTC+AT91_RTTC_RTSR), sr);
  HAL_READ_UINT32((AT91_RTTC+AT91_RTTC_RTMR), mr);
  if (((mr & AT91_RTTC_RTMR_ALMIEN) &&
       (sr & AT91_RTTC_RTSR_ALMS)) ||
      ((mr & AT91_RTTC_RTMR_RTTINCIEN) &&
       (sr & AT91_RTTC_RTSR_RTTINC))) {
    return CYGNUM_HAL_INTERRUPT_RTTC;
  }
#endif

#ifdef CYGNUM_HAL_INTERRUPT_PMC
  // Power Management Controller
  HAL_READ_UINT32((AT91_PMC+AT91_PMC_IMR), mr);
  HAL_READ_UINT32((AT91_PMC+AT91_PMC_SR), sr);
  if ((sr & mr) & 
      (AT91_PMC_SR_MOSCS   |
       AT91_PMC_SR_LOCK    |
       AT91_PMC_SR_MCKRDY  |
       AT91_PMC_SR_PCK0RDY |
       AT91_PMC_SR_PCK1RDY |
       AT91_PMC_SR_PCK2RDY |
       AT91_PMC_SR_PCK3RDY)) {
    return CYGNUM_HAL_INTERRUPT_PMC;
  }
#endif

#ifdef CYGNUM_HAL_INTERRUPT_MC
  // Memory controller
  HAL_READ_UINT32((AT91_MC+AT91_MC_EFC0+AT91_MC_EFCN(0)+AT91_MC_FMR), mr);
  HAL_READ_UINT32((AT91_MC+AT91_MC_EFC0+AT91_MC_EFCN(0)+AT91_MC_FSR), sr);
  if ((sr & mr) & 
      (AT91_MC_FSR_FRDY  |
       AT91_MC_FSR_LOCKE |
       AT91_MC_FSR_PROGE)) {
    return CYGNUM_HAL_INTERRUPT_MC;
  }
#ifdef CYGHWR_HAL_ARM_AT91SAM7_at91sam7s512
  HAL_READ_UINT32((AT91_MC+AT91_MC_EFC0+AT91_MC_EFCN(1)+AT91_MC_FMR), mr);
  HAL_READ_UINT32((AT91_MC+AT91_MC_EFC0+AT91_MC_EFCN(1)+AT91_MC_FSR), sr);
  if ((sr & mr) & 
      (AT91_MC_FSR_FRDY  |
       AT91_MC_FSR_LOCKE |
       AT91_MC_FSR_PROGE)) {
    return CYGNUM_HAL_INTERRUPT_MC;
  }
#endif
#endif

#ifdef CYGNUM_HAL_INTERRUPT_WDTC
  // Watchdog Timer Controller
  HAL_READ_UINT32((AT91_WDTC+AT91_WDTC_WDSR), sr);
  HAL_READ_UINT32((AT91_WDTC+AT91_WDTC_WDMR), mr);
  if ((mr & AT91_WDTC_WDMR_FIEN) &&
      sr & (AT91_WDTC_WDSR_UNDER |
            AT91_WDTC_WDSR_ERROR)) {
    return CYGNUM_HAL_INTERRUPT_WDTC;
  }
#endif

#ifdef CYGNUM_HAL_INTERRUPT_RSTC
  // Reset Controller
  HAL_READ_UINT32((AT91_RST + AT91_RST_RSR), sr);
  HAL_READ_UINT32((AT91_RST + AT91_RST_RMR), mr);
  if (((mr & AT91_RST_RMR_URSTIEN) && (sr & AT91_RST_RSR_USER)) ||  
      ((mr & AT91_RST_RMR_BODIEN) && (sr & AT91_RST_RSR_BROWN)))
    return CYGNUM_HAL_INTERRUPT_RSTC;
#endif
  
  return CYGNUM_HAL_INTERRUPT_NONE;
}

// -------------------------------------------------------------------------
// UART interrupt mask manipulation
//
// Write the IDR and IER of a UART. For the DEBUG UART, the values are
// preserved and used to implement the whole device mask/unmask
// functionality. This is not needed for the other UARTs since they
// also have their own vectors which can be masked; the DEBUG UART has
// to share a vector with several other devices.

static cyg_uint32 hal_debug_imr, hal_debug_masked;

__externC void hal_debug_write_idr( CYG_ADDRESS base, cyg_uint32 idr )
{
    if( base == AT91_DEBUG )
    {
        hal_debug_imr &= ~idr;
        if( hal_debug_masked )
            return;
    }
    HAL_WRITE_UINT32(base + AT91_US_IDR, idr);
}

__externC void hal_debug_write_ier( CYG_ADDRESS base, cyg_uint32 ier )
{
    if( base == AT91_DEBUG )
    {
        hal_debug_imr |= ier;
        if( hal_debug_masked )
            return;
    }
    HAL_WRITE_UINT32(base + AT91_US_IER, ier);
}

// -------------------------------------------------------------------------
// System Interrupt Masking
//
// This currently only deals with the PIT and DEBUG UART. The Watchdog
// cannot be changed since its Mode Register is write-once, so it is
// ignored. All others are either ignored or raise an assertion failure.

static void hal_sysirq_mask( int vector, int mask )
{
    cyg_uint32 mr;

#ifdef CYGNUM_HAL_INTERRUPT_PITC
    if( vector == CYGNUM_HAL_INTERRUPT_PITC )
    {
        // Periodic Interrupt Timer Controller
        HAL_READ_UINT32((AT91_PITC+AT91_PITC_PIMR), mr);
        mr &= ~AT91_PITC_PIMR_PITIEN;
        if( !mask )
            mr |= AT91_PITC_PIMR_PITIEN;
        HAL_WRITE_UINT32((AT91_PITC+AT91_PITC_PIMR), mr);
        return;
    }
#endif

#ifdef CYGNUM_HAL_INTERRUPT_DEBUG
    if( vector == CYGNUM_HAL_INTERRUPT_DEBUG )
    {
        // Debug Unit
        hal_debug_masked = mask;
        if( mask )
            HAL_WRITE_UINT32(AT91_DEBUG + AT91_US_IDR, hal_debug_imr);
        else
            HAL_WRITE_UINT32(AT91_DEBUG + AT91_US_IER, hal_debug_imr);
        return;
    }
#endif

#ifdef CYGNUM_HAL_INTERRUPT_WDTC
    if( vector == CYGNUM_HAL_INTERRUPT_WDTC )
    {
        // Watchdog Timer Controller
        return;
    }
#endif

#ifdef CYGNUM_HAL_INTERRUPT_RTTC
    if( vector == CYGNUM_HAL_INTERRUPT_RTTC )
    {
        // Real Time Clock
        CYG_FAIL("RTTC Interrupt mask not currently supported");
    }
#endif

#ifdef CYGNUM_HAL_INTERRUPT_PMC
    if( vector == CYGNUM_HAL_INTERRUPT_PMC )
    {
        // Power Management Controller
        CYG_FAIL("PMC Interrupt mask not currently supported");
    }
#endif

#ifdef CYGNUM_HAL_INTERRUPT_MC
    if( vector == CYGNUM_HAL_INTERRUPT_MC )
    {
        // Memory controller
        CYG_FAIL("MC Interrupt mask not currently supported");        
    }
#endif

#ifdef CYGNUM_HAL_INTERRUPT_RSTC
    if( vector == CYGNUM_HAL_INTERRUPT_RSTC )
    {
        // Reset Controller
        CYG_FAIL("RSTC Interrupt mask not currently supported");
    }
#endif

}

#endif

// -------------------------------------------------------------------------
// This routine is called to respond to a hardware interrupt (IRQ).  It
// should interrogate the hardware and return the IRQ vector number.

int hal_IRQ_handler(void)
{
    cyg_uint32 irq_num;
    cyg_uint32 ivr;
#ifdef CYGHWR_HAL_ARM_AT91_FIQ
    // handle fiq interrupts as irq 
    cyg_uint32 ipr,imr;

    HAL_READ_UINT32(AT91_AIC+AT91_AIC_IPR, ipr);
    HAL_READ_UINT32(AT91_AIC+AT91_AIC_IMR, imr);
    
    if (imr & ipr & (1 << CYGNUM_HAL_INTERRUPT_FIQ)) {
      HAL_WRITE_UINT32(AT91_AIC+AT91_AIC_ICCR, (1 << CYGNUM_HAL_INTERRUPT_FIQ));
      return CYGNUM_HAL_INTERRUPT_FIQ;
    }
#endif
    // Calculate active interrupt (updates ISR)
    HAL_READ_UINT32(AT91_AIC+AT91_AIC_IVR, ivr);
    HAL_READ_UINT32(AT91_AIC+AT91_AIC_ISR, irq_num);
    
#if CYGINT_HAL_ARM_AT91_SYS_INTERRUPT
    if (irq_num == CYGNUM_HAL_INTERRUPT_SYS) {
      // determine the source of the system interrupt
      irq_num = sys_irq_handler();
    }
#endif
    // An invalid interrupt source is treated as a spurious interrupt    
    if (irq_num < CYGNUM_HAL_ISR_MIN || irq_num > CYGNUM_HAL_ISR_MAX)
      irq_num = CYGNUM_HAL_INTERRUPT_NONE;
    return irq_num;
}

// -------------------------------------------------------------------------
// Interrupt control
//

void hal_interrupt_mask(int vector)
{
    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN , "Invalid vector");

#if CYGINT_HAL_ARM_AT91_SYS_INTERRUPT
    if (vector >= 32) {
        hal_sysirq_mask( vector, 1 );
      return;
    }
#endif
    HAL_WRITE_UINT32(AT91_AIC+AT91_AIC_IDCR, (1<<vector));
}

void hal_interrupt_unmask(int vector)
{
    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN , "Invalid vector");

#if CYGINT_HAL_ARM_AT91_SYS_INTERRUPT
    if (vector >= 32) {
      hal_interrupt_configure(CYGINT_HAL_ARM_AT91_SYS_INTERRUPT, true, true);
      HAL_WRITE_UINT32(AT91_AIC+AT91_AIC_IECR, 
                       (1 <<CYGINT_HAL_ARM_AT91_SYS_INTERRUPT));
        hal_sysirq_mask( vector, 0 );        
      return;
    }
#endif
    HAL_WRITE_UINT32(AT91_AIC+AT91_AIC_IECR, (1<<vector));
}

void hal_interrupt_acknowledge(int vector)
{
    // No check for valid vector here! Spurious interrupts
    // must be acknowledged, too.
    HAL_WRITE_UINT32(AT91_AIC+AT91_AIC_EOI, 0xFFFFFFFF);  
}

void hal_interrupt_configure(int vector, int level, int up)
{
    cyg_uint32 mode;

    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN , "Invalid vector");

#if CYGINT_HAL_ARM_AT91_SYS_INTERRUPT
    if (vector >= 32) 
      return;
#endif
    if (level) {
        if (up) {
            mode = AT91_AIC_SMR_LEVEL_HI;
        } else {
            mode = AT91_AIC_SMR_LEVEL_LOW;
        }
    } else {
        if (up) {
            mode = AT91_AIC_SMR_EDGE_POS;
        } else {
            mode = AT91_AIC_SMR_EDGE_NEG;
        }
    }
    mode |= 7;  // Default priority
    HAL_WRITE_UINT32(AT91_AIC+(AT91_AIC_SMR0+(vector*4)), mode);
}

void hal_interrupt_set_level(int vector, int level)
{
    cyg_uint32 mode;

    CYG_ASSERT(vector <= CYGNUM_HAL_ISR_MAX &&
               vector >= CYGNUM_HAL_ISR_MIN , "Invalid vector");
    CYG_ASSERT(level >= 0 && level <= 7, "Invalid level");

#if CYGINT_HAL_ARM_AT91_SYS_INTERRUPT
    if (vector >= 32) 
      return;
#endif
    HAL_READ_UINT32(AT91_AIC+(AT91_AIC_SMR0+(vector*4)), mode);
    mode = (mode & ~AT91_AIC_SMR_PRIORITY) | level;
    HAL_WRITE_UINT32(AT91_AIC+(AT91_AIC_SMR0+(vector*4)), mode);
}

void hal_show_IRQ(int vector, int data, int handler)
{
//    UNDEFINED(__FUNCTION__);  // FIXME
}

#ifndef CYGSEM_HAL_VIRTUAL_VECTOR_SUPPORT
// Need our own default ISR

__externC cyg_uint32 hal_arch_default_isr(CYG_ADDRWORD vector, CYG_ADDRWORD data);

cyg_uint32
hal_default_isr(CYG_ADDRWORD vector, CYG_ADDRWORD data)
{
    return hal_arch_default_isr (vector, data);
}
#endif


#ifndef AT91_RST
/* Use the watchdog to generate a reset */
void hal_at91_reset_cpu(void)
{
#if defined(CYGHWR_HAL_ARM_AT91_R40807) || \
    defined(CYGHWR_HAL_ARM_AT91_R40008) || \
    defined(CYGHWR_HAL_ARM_AT91_M55800A)
    HAL_WRITE_UINT32(AT91_WD + AT91_WD_OMR, AT91_WD_OMR_OKEY);
    HAL_WRITE_UINT32(AT91_WD + AT91_WD_CMR, AT91_WD_CMR_CKEY);
    HAL_WRITE_UINT32(AT91_WD + AT91_WD_CR, AT91_WD_CR_RSTKEY);
    HAL_WRITE_UINT32(AT91_WD + AT91_WD_OMR, AT91_WD_OMR_OKEY | AT91_WD_OMR_RSTEN | AT91_WD_OMR_WDEN);
#elif defined(CYGHWR_HAL_ARM_AT91_M42800A)
    // Enable external (in case attached) and internal reset, and set timer to just 1 slow clock tick
    HAL_WRITE_UINT32(AT91_WD + AT91_ST_WDMR, AT91_ST_WDMR_EXTEN | AT91_ST_WDMR_RSTEN | 0x00000001);
    // Be sure there isn't an old value in the counter by resetting
    HAL_WRITE_UINT32(AT91_WD + AT91_ST_CR, AT91_ST_CR_WDRST);
#endif
    while(1) CYG_EMPTY_STATEMENT;
}
#else
/* Use the Reset Controller to generate a reset */
void hal_at91_reset_cpu(void)
{
  HAL_WRITE_UINT32(AT91_RST + AT91_RST_RCR,
                   AT91_RST_RCR_PROCRST |                   
                   AT91_RST_RCR_ICERST  |
                   AT91_RST_RCR_PERRST  |
                   AT91_RST_RCR_KEY);
  while(1) CYG_EMPTY_STATEMENT;
}
#endif

//--------------------------------------------------------------------------
// EOF at91_misc.c
