//==========================================================================
//
//      ixp_ide.c
//
//      HAL support code for PCI IDE devices
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003, 2004 Free Software Foundation, Inc.
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    msalter
// Contributors: msalter, gthomas
// Date:         2002-01-04
// Purpose:      PCI support
// Description:  Implementations of HAL PCI interfaces
//
//####DESCRIPTIONEND####
//
//========================================================================*/

#include <pkgconf/hal.h>
#include <pkgconf/system.h>
#include CYGBLD_HAL_PLATFORM_H
#include CYGHWR_MEMORY_LAYOUT_H

#include <cyg/infra/cyg_type.h>         // base types
#include <cyg/infra/cyg_trac.h>         // tracing macros
#include <cyg/infra/cyg_ass.h>          // assertion macros
#include <cyg/infra/diag.h>             // diag_printf()

#include <cyg/hal/hal_io.h>             // IO macros
#include <cyg/hal/hal_if.h>             // calling interface API
#include <cyg/hal/hal_arch.h>           // Register state info
#include <cyg/hal/hal_diag.h>
#include <cyg/hal/hal_intr.h>           // Interrupt names
#include <cyg/hal/hal_cache.h>
#include <cyg/hal/hal_endian.h>
#include <cyg/io/pci_hw.h>
#include <cyg/io/pci.h>

// Debug control
//   0 = no messages
//   1 = identification/discovery
//   2 = detailed I/O
#define IDE_DEBUG 0

int _plf_ide_num_controllers = 0;

static struct {
    cyg_uint32 cmd_bar;
    cyg_uint32 ctl_bar;
} ide_ctrl[HAL_IDE_NUM_CONTROLLERS];

cyg_uint8 
cyg_hal_plf_ide_read_uint8(int ctlr, cyg_uint32 reg)
{
    cyg_uint8 val;
    HAL_READ_UINT8(ide_ctrl[ctlr].cmd_bar + reg, val);
#if (IDE_DEBUG & 0x02)
    diag_printf("IDE8[%d.%d] => %x\n", ctlr, reg, val);
#endif
    return val;
}

void 
cyg_hal_plf_ide_write_uint8(int ctlr, cyg_uint32 reg, cyg_uint8 val)
{
    HAL_WRITE_UINT8(ide_ctrl[ctlr].cmd_bar + reg, val);
#if (IDE_DEBUG & 0x02)
    diag_printf("IDE8[%d.%d] <= %x\n", ctlr, reg, val);
#endif
}

cyg_uint16 
cyg_hal_plf_ide_read_uint16(int ctlr, cyg_uint32 reg)
{
    cyg_uint16 val;
    HAL_READ_UINT16(ide_ctrl[ctlr].cmd_bar + reg, val);
#if (CYG_BYTEORDER == CYG_MSBFIRST)
    val = CYG_SWAP16(val);
#endif
#if (IDE_DEBUG & 0x02)
    diag_printf("IDE16[%d.%d] => %x\n", ctlr, reg, val);
#endif
    return val;
}

void 
cyg_hal_plf_ide_write_uint16(int ctlr, cyg_uint32 reg, cyg_uint16 val)
{
#if (CYG_BYTEORDER == CYG_MSBFIRST)
    val = CYG_SWAP16(val);
#endif
    HAL_WRITE_UINT16(ide_ctrl[ctlr].cmd_bar + reg, val);
#if (IDE_DEBUG & 0x02)
    diag_printf("IDE16[%d.%d] <= %x\n", ctlr, reg, val);
#endif
}

void 
cyg_hal_plf_ide_write_control(int ctlr, cyg_uint8 val)
{
    HAL_WRITE_UINT8(ide_ctrl[ctlr].ctl_bar + 2, val);
#if (IDE_DEBUG & 0x02)
    diag_printf("IDECTL[%d] <= %x\n", ctlr, val);
#endif
}

static cyg_bool
find_ide_match_func( cyg_uint16 v, cyg_uint16 d, cyg_uint32 c, void *p )
{
    // is it generic PCI-IDE
    if ((c & 0xFFFF00) == 0x010100)
	return 1;

    // is it Promise 20275 or 20269
    if (v == 0x105a && (d == 0x1275 || d == 0x4d69))
	return 1;

    return 0;
}

static int
init_generic(cyg_pci_device *dev)
{
    cyg_uint8 progif;

    if (dev->command & CYG_PCI_CFG_COMMAND_ACTIVE)
	return 2;

    cyg_pci_write_config_uint16(dev->devid,
				CYG_PCI_CFG_COMMAND,
				dev->command | CYG_PCI_CFG_COMMAND_IO
				| CYG_PCI_CFG_COMMAND_MEMORY
				| CYG_PCI_CFG_COMMAND_MASTER);
    cyg_pci_read_config_uint8(dev->devid, 9, &progif);
    if ((progif & 5) != 5) {
	if ((progif & 0xa) != 0xa) {
	    // can't run in native mode
	    return 0;
	}
	cyg_pci_write_config_uint8(dev->devid, 9, progif | 5);
	cyg_pci_read_config_uint8(dev->devid, 9, &progif);
	if ((progif & 5) != 5)
	    return 0;
    }
    return 2;
}

static int
init_promise(cyg_pci_device *dev)
{
    if (dev->command & CYG_PCI_CFG_COMMAND_ACTIVE)
	return 2;

    cyg_pci_write_config_uint16(dev->devid,
				CYG_PCI_CFG_COMMAND,
				dev->command | CYG_PCI_CFG_COMMAND_IO
				| CYG_PCI_CFG_COMMAND_MEMORY
				| CYG_PCI_CFG_COMMAND_MASTER);
    return 2;
}

int
cyg_hal_plf_ide_init(void)
{
    int i, n;
    cyg_pci_device_id ide_dev = CYG_PCI_NULL_DEVID;
    cyg_pci_device ide_info;

#if (IDE_DEBUG & 0x01)
    diag_printf("Initializing IDE controller\n");
#endif
    cyg_pci_init();

    if (cyg_pci_find_matching(&find_ide_match_func, NULL, &ide_dev)) {

        cyg_pci_get_device_info(ide_dev, &ide_info);

        if (cyg_pci_configure_device(&ide_info)) {
	    if (ide_info.command & CYG_PCI_CFG_COMMAND_ACTIVE) {
#if (IDE_DEBUG & 0x01)
	        diag_printf(" Note that board is active. Probed"
			    " sizes and CPU addresses invalid!\n");
#endif
	    }
	    if ((ide_info.vendor == 0x105a) 
		&& (ide_info.device == 0x1275 || ide_info.device == 0x4d69))
		n = init_promise(&ide_info);
	    else
		n = init_generic(&ide_info);

	    if (!n)
		return 0;
	}

#if (IDE_DEBUG & 0x01)
	diag_printf("header_type[%d] class_rev[%08x]\n", ide_info.header_type, ide_info.class_rev);
        for (i = 0;  i < 6;  i++) {
            diag_printf("IDE - base[%d]: %08p, size: %08p, map: %08p\n",
                        i, ide_info.base_address[i], ide_info.base_size[i], ide_info.base_map[i]);
        }
#endif
        for (i = 0;  i < HAL_IDE_NUM_CONTROLLERS;  i++) {
            ide_ctrl[i].cmd_bar = ide_info.base_map[(2*i)+0] & 0xFFFFFFFE;
            ide_ctrl[i].ctl_bar = ide_info.base_map[(2*i)+1] & 0xFFFFFFFE;
        }

        return n /*HAL_IDE_NUM_CONTROLLERS*/;
    } else {
#if (IDE_DEBUG & 0x01)
        diag_printf("Can't find IDE controller!\n");
#endif
        return 0;
    }
}
