//==========================================================================
//
//        tcdiag.cxx
//
//        Infrastructure diag test harness.
//
//==========================================================================
// ####ECOSGPLCOPYRIGHTBEGIN####                                            
// -------------------------------------------                              
// This file is part of eCos, the Embedded Configurable Operating System.   
// Copyright (C) 1998, 1999, 2000, 2001, 2002 Free Software Foundation, Inc.
//
// eCos is free software; you can redistribute it and/or modify it under    
// the terms of the GNU General Public License as published by the Free     
// Software Foundation; either version 2 or (at your option) any later      
// version.                                                                 
//
// eCos is distributed in the hope that it will be useful, but WITHOUT      
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or    
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License    
// for more details.                                                        
//
// You should have received a copy of the GNU General Public License        
// along with eCos; if not, write to the Free Software Foundation, Inc.,    
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.            
//
// As a special exception, if other files instantiate templates or use      
// macros or inline functions from this file, or you compile this file      
// and link it with other works to produce a work based on this file,       
// this file does not by itself cause the resulting work to be covered by   
// the GNU General Public License. However the source code for this file    
// must still be made available in accordance with section (3) of the GNU   
// General Public License v2.                                               
//
// This exception does not invalidate any other reasons why a work based    
// on this file might be covered by the GNU General Public License.         
// -------------------------------------------                              
// ####ECOSGPLCOPYRIGHTEND####                                              
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):     dsm
// Contributors:  dsm, jlarmour
// Date:          1999-02-16
// Description:   Test harness implementation that uses the infrastructure
//                diag channel.  This is intended for manual testing.
// 
//####DESCRIPTIONEND####

#include <pkgconf/infra.h>
#include <pkgconf/system.h>
#include CYGBLD_HAL_TARGET_H           // get initialization for
#include CYGBLD_HAL_PLATFORM_H         //   cyg_test_is_simulator

#include <cyg/infra/cyg_type.h>        // base types
#include <cyg/hal/hal_arch.h>          // any architecture specific stuff
#include <cyg/infra/diag.h>            // HAL polled output
#include <cyg/infra/testcase.h>        // what we implement

#include <cyg/hal/hal_intr.h>          // exit macro, if defined

#ifdef CYGHWR_TARGET_SIMULATOR_NO_GDB_WORKING
int cyg_test_is_simulator = 1;         // set this anyway
#else
int cyg_test_is_simulator = 0;         // infrastructure changes as necessary
#endif

//----------------------------------------------------------------------------
// Functions ensuring we get pretty printed assertion messages in the
// farm - regardless of configuration and GDB capabilities.

#ifdef CYGPKG_KERNEL
# include <pkgconf/kernel.h>
# include <cyg/kernel/thread.hxx>        // thread id to print
# include <cyg/kernel/thread.inl>        // ancillaries for above
#endif

#ifdef CYGSEM_HAL_VIRTUAL_VECTOR_SUPPORT
#include <cyg/hal/hal_if.h>
#endif

static inline const char *trim_file(const char *file)
{
    if ( NULL == file )
        file = "<nofile>";
    
    const char *f = file;
    
    while( *f ) f++;

    while( *f != '/' && f != file ) f--;

    return f==file?f:(f+1);
}

static inline const char *trim_func(const char *func)
{
    static char fbuf[100];
    int i;
    
    if ( NULL == func )
        func = "<nofunc>";

    for( i = 0; func[i] && func[i] != '(' ; i++ )
        fbuf[i] = func[i];

    fbuf[i++] = '(';
    fbuf[i++] = ')';
    fbuf[i  ] = 0;

    return &fbuf[0];
}

static inline
void write_lnum( cyg_uint32 lnum)
{
    diag_write_char('[');
    diag_write_dec(lnum);
    diag_write_char(']');
}

static inline
void write_thread_id()
{
#ifdef CYGPKG_KERNEL
    Cyg_Thread *t = Cyg_Thread::self();
    cyg_uint16 tid = 0xFFFF;

    if( t != NULL ) tid = t->get_unique_id();

    diag_write_char('<');
    diag_write_hex(tid);
    diag_write_char('>');
#endif
}

// Called from the CYG_ASSERT_DOCALL macro
externC void
cyg_assert_msg( const char *psz_func, const char *psz_file,
                cyg_uint32 linenum, const char *psz_msg ) __THROW
{
    cyg_uint32 old_ints;

    HAL_DISABLE_INTERRUPTS(old_ints);
    DIAG_DEVICE_START_SYNC();

#ifdef CYG_HAL_DIAG_LOCK
    CYG_HAL_DIAG_LOCK();
#endif    
    
#ifdef CYGSEM_HAL_VIRTUAL_VECTOR_SUPPORT
    {
        int cur_console;
        int i;
        struct cyg_fconfig fc;

        cur_console = CYGACC_CALL_IF_SET_CONSOLE_COMM(CYGNUM_CALL_IF_SET_COMM_ID_QUERY_CURRENT);
        fc.key = (char *)"info_console_force";
        fc.type = CYGNUM_FLASH_CFG_TYPE_CONFIG_BOOL;
        fc.val = (void *)&i;
        if (CYGACC_CALL_IF_FLASH_CFG_OP2(CYGNUM_CALL_IF_FLASH_CFG_GET, &fc)) {
            if (i) {
                fc.key = (char *)"info_console_number";
                fc.type = CYGNUM_FLASH_CFG_TYPE_CONFIG_INT;
                if (CYGACC_CALL_IF_FLASH_CFG_OP2(CYGNUM_CALL_IF_FLASH_CFG_GET, &fc)) {
                    // Then i is the console to force it to:
                    CYGACC_CALL_IF_SET_CONSOLE_COMM(i);
                }
            }
        }
#endif
    diag_write_string("ASSERT FAIL: ");
    write_thread_id();
    diag_write_string(trim_file(psz_file));
    write_lnum(linenum);
    diag_write_string(trim_func(psz_func));
    diag_write_char(' ');
    diag_write_string(psz_msg);
    diag_write_char('\n');

#ifdef CYGSEM_HAL_VIRTUAL_VECTOR_SUPPORT
        CYGACC_CALL_IF_SET_CONSOLE_COMM(cur_console);
    }
#endif
#ifdef CYG_HAL_DIAG_UNLOCK
    CYG_HAL_DIAG_UNLOCK();
#endif    
    
    DIAG_DEVICE_END_SYNC();
    HAL_RESTORE_INTERRUPTS(old_ints);
}

// Table of CRC constants - implements x^16+x^12+x^5+1
static const cyg_uint16 crc16_tab[] = {
    0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50a5, 0x60c6, 0x70e7, 
    0x8108, 0x9129, 0xa14a, 0xb16b, 0xc18c, 0xd1ad, 0xe1ce, 0xf1ef, 
    0x1231, 0x0210, 0x3273, 0x2252, 0x52b5, 0x4294, 0x72f7, 0x62d6, 
    0x9339, 0x8318, 0xb37b, 0xa35a, 0xd3bd, 0xc39c, 0xf3ff, 0xe3de, 
    0x2462, 0x3443, 0x0420, 0x1401, 0x64e6, 0x74c7, 0x44a4, 0x5485, 
    0xa56a, 0xb54b, 0x8528, 0x9509, 0xe5ee, 0xf5cf, 0xc5ac, 0xd58d, 
    0x3653, 0x2672, 0x1611, 0x0630, 0x76d7, 0x66f6, 0x5695, 0x46b4, 
    0xb75b, 0xa77a, 0x9719, 0x8738, 0xf7df, 0xe7fe, 0xd79d, 0xc7bc, 
    0x48c4, 0x58e5, 0x6886, 0x78a7, 0x0840, 0x1861, 0x2802, 0x3823, 
    0xc9cc, 0xd9ed, 0xe98e, 0xf9af, 0x8948, 0x9969, 0xa90a, 0xb92b, 
    0x5af5, 0x4ad4, 0x7ab7, 0x6a96, 0x1a71, 0x0a50, 0x3a33, 0x2a12, 
    0xdbfd, 0xcbdc, 0xfbbf, 0xeb9e, 0x9b79, 0x8b58, 0xbb3b, 0xab1a, 
    0x6ca6, 0x7c87, 0x4ce4, 0x5cc5, 0x2c22, 0x3c03, 0x0c60, 0x1c41, 
    0xedae, 0xfd8f, 0xcdec, 0xddcd, 0xad2a, 0xbd0b, 0x8d68, 0x9d49, 
    0x7e97, 0x6eb6, 0x5ed5, 0x4ef4, 0x3e13, 0x2e32, 0x1e51, 0x0e70, 
    0xff9f, 0xefbe, 0xdfdd, 0xcffc, 0xbf1b, 0xaf3a, 0x9f59, 0x8f78, 
    0x9188, 0x81a9, 0xb1ca, 0xa1eb, 0xd10c, 0xc12d, 0xf14e, 0xe16f, 
    0x1080, 0x00a1, 0x30c2, 0x20e3, 0x5004, 0x4025, 0x7046, 0x6067, 
    0x83b9, 0x9398, 0xa3fb, 0xb3da, 0xc33d, 0xd31c, 0xe37f, 0xf35e, 
    0x02b1, 0x1290, 0x22f3, 0x32d2, 0x4235, 0x5214, 0x6277, 0x7256, 
    0xb5ea, 0xa5cb, 0x95a8, 0x8589, 0xf56e, 0xe54f, 0xd52c, 0xc50d, 
    0x34e2, 0x24c3, 0x14a0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405, 
    0xa7db, 0xb7fa, 0x8799, 0x97b8, 0xe75f, 0xf77e, 0xc71d, 0xd73c, 
    0x26d3, 0x36f2, 0x0691, 0x16b0, 0x6657, 0x7676, 0x4615, 0x5634, 
    0xd94c, 0xc96d, 0xf90e, 0xe92f, 0x99c8, 0x89e9, 0xb98a, 0xa9ab, 
    0x5844, 0x4865, 0x7806, 0x6827, 0x18c0, 0x08e1, 0x3882, 0x28a3, 
    0xcb7d, 0xdb5c, 0xeb3f, 0xfb1e, 0x8bf9, 0x9bd8, 0xabbb, 0xbb9a, 
    0x4a75, 0x5a54, 0x6a37, 0x7a16, 0x0af1, 0x1ad0, 0x2ab3, 0x3a92, 
    0xfd2e, 0xed0f, 0xdd6c, 0xcd4d, 0xbdaa, 0xad8b, 0x9de8, 0x8dc9, 
    0x7c26, 0x6c07, 0x5c64, 0x4c45, 0x3ca2, 0x2c83, 0x1ce0, 0x0cc1, 
    0xef1f, 0xff3e, 0xcf5d, 0xdf7c, 0xaf9b, 0xbfba, 0x8fd9, 0x9ff8, 
    0x6e17, 0x7e36, 0x4e55, 0x5e74, 0x2e93, 0x3eb2, 0x0ed1, 0x1ef0, 
};

static cyg_uint16
cyg_crc16(unsigned char *buf, int len)
{
    int i;
    cyg_uint16 cksum;

    cksum = 0;
    for (i = 0;  i < len;  i++) {
        cksum = crc16_tab[((cksum>>8) ^ *buf++) & 0xFF] ^ (cksum << 8);
    }
    return cksum;
}

extern unsigned char _stext[], _etext[];
static cyg_uint16 code_checksum;

externC void
cyg_test_init(void)
{
    code_checksum = cyg_crc16(_stext, _etext - _stext);
    diag_printf("INFO: <code from %p -> %p, CRC %04x>\n", _stext, _etext, code_checksum);
}

externC void
cyg_test_output(Cyg_test_code status, const char *msg, int line,
                const char *file)
{
    const char *st;

    switch (status) {
    case CYGNUM_TEST_FAIL:
        st = "FAIL:";
        break;
    case CYGNUM_TEST_PASS:
        st = "PASS:";
        break;
    case CYGNUM_TEST_EXIT:
        st = "EXIT:";
        break;
    case CYGNUM_TEST_INFO:
        st = "INFO:";
        break;
    case CYGNUM_TEST_GDBCMD:
        st = "GDB:";
        break;
    case CYGNUM_TEST_NA:
        st = "NOTAPPLICABLE:";
        break;
    default:
        st = "UNKNOWN STATUS:";
        break;
    }

#ifdef CYG_HAL_DIAG_LOCK
    CYG_HAL_DIAG_LOCK();
#endif    
    
    diag_write_string(st);
    diag_write_char('<');
    diag_write_string(msg);
    diag_write_char('>');
    if( CYGNUM_TEST_FAIL == status ) {
        diag_write_string(" Line: ");
        diag_write_dec(line);
        diag_write_string(", File: ");
        diag_write_string(file);
    }
    diag_write_char('\n');

#ifdef CYG_HAL_DIAG_UNLOCK
    CYG_HAL_DIAG_UNLOCK();
#endif    
    
    
}

// This is an appropriate function to set a breakpoint on
externC void
cyg_test_exit(void)
{
    if (code_checksum != cyg_crc16(_stext, _etext - _stext)) {
        CYG_TEST_FAIL("self-modifying code detected");
    }
    
// workaround SH dwarf2 gen problem    
#if defined(CYGPKG_HAL_SH) && (__GNUC__ >= 3)
    static volatile int i;
    i++;
#endif
#ifdef CYGHWR_TEST_PROGRAM_EXIT
    CYGHWR_TEST_PROGRAM_EXIT();
#endif
#ifdef CYGSEM_INFRA_RESET_ON_TEST_EXIT
#ifdef HAL_PLATFORM_RESET
    HAL_PLATFORM_RESET();
#else
#warning "Reset selected for test case exit, but none defined"
#endif
#endif
    // Default behaviour - simply hang in a loop
    for(;;)
        ;
}
// EOF tcdiag.cxx
