#include <pkgconf/system.h>
#ifndef CYGPKG_RBL
# error The eCos configuration does not contain the RBL package.
#endif
#include <cyg/infra/cyg_type.h>
#include <cyg/infra/diag.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <net/if.h>
#include <network.h>
#include <tftp_support.h>
#include <cyg/rbl/rbl.h>

// ----------------------------------------------------------------------------
// Globals. The program supports either a telnet session or a tftp session,
// but not both at the same time. Hence there is no need for lots of locking
// around rbl accesses.
// FIXME: a mutex should be used to control updates to these globals. Right
// now there is a race condition if telnet and tftp sessions are started
// concurrently.
static int  tftpd_busy  = 0;
static int  telnet_busy = 0;

// ----------------------------------------------------------------------------
// Buffer space for code & data. This is dynamically allocated since the
// required sizes depend on how RedBoot was configured rather than on the
// application.
static int      code_buffer_size    = 0;
static int      data_buffer_size    = 0;
static char*    code_buffer         = (char*)0;
static char*    data_buffer         = (char*)0;

static cyg_bool
allocate_buffers(void)
{
    rbl_flash_details   flash_details;
    if (! rbl_get_flash_details(&flash_details)) {
        fputs("Error: failed to get RBL flash details\n", stderr);
        return false;
    }

    code_buffer_size = flash_details.rbl_flash_block_size * flash_details.rbl_code_num_blocks;
    code_buffer = malloc(code_buffer_size);
    if (code_buffer == (char*)0) {
        fprintf(stderr, "Error: failed to allocate a %dK buffer for new code\n",
                (flash_details.rbl_flash_block_size * flash_details.rbl_code_num_blocks) / 1024);
        return false;
    }
    if (0 != flash_details.rbl_data_num_blocks) {
        data_buffer_size = flash_details.rbl_flash_block_size * flash_details.rbl_data_num_blocks;
        data_buffer = malloc(data_buffer_size);
        if (data_buffer == (char*)0) {
            fprintf(stderr, "Error: failed to allocate a %dK buffer for data\n",
                    (flash_details.rbl_flash_block_size * flash_details.rbl_data_num_blocks) / 1024);
            return false;
        }
    }
    return true;
}

// ----------------------------------------------------------------------------
// Socket utilities
static int
create_tcp_server_socket(int service)
{
    struct sockaddr_in      addr;
    int                     fd;

    fd = socket(AF_INET, SOCK_STREAM, 0);
    if (fd < 0) {
        fprintf(stderr, "Error: failed to create server socket for service %d\n", service);
        return fd;
    }
    memset(&addr, 0, sizeof(struct sockaddr_in));
    addr.sin_family         = AF_INET;
    addr.sin_port           = htons(service);
    addr.sin_addr.s_addr    = htonl(INADDR_ANY);
    if (bind(fd, (struct sockaddr*) &addr, sizeof(struct sockaddr_in)) < 0) {
        fprintf(stderr, "Error: failed to bind server socket for service %d\n", service);
        close(fd);
        return -1;
    }
    if (listen(fd, 5) < 0) {
        fprintf(stderr, "Error: failed to listen on server socket for service %d\n", service);
        close(fd);
        return -1;
    }
    return fd;
}

// ----------------------------------------------------------------------------
// A telnet server.

static void
telnet_worker(int fd)
{
    FILE*   file;
    rbl_flash_details       flash_details;
    rbl_flash_block_purpose last_purpose, purpose;
    rbl_block_details       block_details;
    int                     i;
    
    file = fdopen(fd, "r+");
    if ((FILE*)0 == file) {
        fputs("telnet_worker: fdopen call failed", stderr);
        close(fd);
        return;
    }
    fputs("nettest: ", file);
#ifdef VERSION
# define VERSION_STRINGIFY1(a) # a    
# define VERSION_STRINGIFY(a) VERSION_STRINGIFY1(a)
    fprintf(file, "version %s, ", VERSION_STRINGIFY(VERSION));
# undef VERSION_STRINGIFY
# undef VERSION_STRINGIFY1
#endif
    fprintf(file, "built %s %s\n", __DATE__, __TIME__);

    if (! rbl_get_flash_details(&flash_details)) {
        fputs("Failed to get RBL flash details.\n", file);
        fclose(file);
        return;
    }
    fprintf(file, "Flash: %d blocks * %d K @ %p\n", flash_details.rbl_flash_num_blocks,
            flash_details.rbl_flash_block_size / 1024, flash_details.rbl_flash_base);
    fprintf(file, "Each code image uses %d flash block(s)\n", flash_details.rbl_code_num_blocks);
    if (0 == flash_details.rbl_data_num_blocks) {
        fputs("Persistent data support has been disabled.\n", file);
    } else {
        fprintf(file, "Each data image uses %d flash block(s)\n", flash_details.rbl_data_num_blocks);
    }
    last_purpose = rbl_flash_block_invalid;
    for (i = 0; i < flash_details.rbl_flash_num_blocks; i++) {
        purpose = rbl_get_flash_block_purpose(i);
        if (last_purpose != purpose) {
            char* tmp;
            switch(purpose) {
              case rbl_flash_block_reserved:
                tmp = "Reserved"; break;
              case rbl_flash_block_code_A:
                tmp = "Code A  "; break;
              case rbl_flash_block_code_B:
                tmp = "Code B  "; break;
              case rbl_flash_block_data_A:
                tmp = "Data A  "; break;
              case rbl_flash_block_data_B:
                tmp = "Data B  "; break;
              case rbl_flash_block_free:
                tmp = "Free    "; break;
              case rbl_flash_block_invalid:
                tmp = "Invalid "; break;
              default:
                tmp = "<UNKNOWN>"; break;
            }
            fprintf(file, "\n%s flash block(s): %d", tmp, i);
            last_purpose = purpose;
        } else {
            fprintf(file, ", %d", i);
        }
    }
    putc('\n', file);

    fputs("\nPrimary  code block: ", file);
    if (rbl_get_block_details(rbl_block_code_primary, &block_details)) {
        if (!block_details.rbl_valid) {
            fputs("not valid\n", file);
        } else {
            fprintf(file, "starts @ %p (flash block %d), len %d, sequence number %d\n", block_details.rbl_address,
                    block_details.rbl_first_flash_block, block_details.rbl_size, block_details.rbl_sequence_number);
        } 
    } else {
        fputs("failed to get details\n", file);
    }
    fputs("Backup   code block: ", file);
    if (rbl_get_block_details(rbl_block_code_backup, &block_details)) {
        if (!block_details.rbl_valid) {
            fputs("not valid\n", file);
        } else {
            fprintf(file, "starts @ %p (flash block %d), len %d, sequence number %d\n", block_details.rbl_address,
                    block_details.rbl_first_flash_block, block_details.rbl_size, block_details.rbl_sequence_number);
        } 
    } else {
        fputs("failed to get details\n", file);
    }
    if (0 != flash_details.rbl_data_num_blocks) {
        fputs("Primary  data block: ", file);
        if (rbl_get_block_details(rbl_block_data_primary, &block_details)) {
            if (!block_details.rbl_valid) {
                fputs("not valid\n", file);
            } else {
                fprintf(file, "starts @ %p (flash block %d), len %d, sequence number %d\n", block_details.rbl_address,
                    block_details.rbl_first_flash_block, block_details.rbl_size, block_details.rbl_sequence_number);
            } 
        } else {
            fputs("failed to get details\n", file);
        }
        fputs("Backup   data block: ", file);
        if (rbl_get_block_details(rbl_block_data_backup, &block_details)) {
            if (!block_details.rbl_valid) {
                fputs("not valid\n", file);
            } else {
                fprintf(file, "starts @ %p (flash block %d), len %d, sequence number %d\n", block_details.rbl_address,
                    block_details.rbl_first_flash_block, block_details.rbl_size, block_details.rbl_sequence_number);
            } 
        } else {
            fputs("failed to get details\n", file);
        }
    }

    if (0 != flash_details.rbl_data_num_blocks) {
        if (rbl_get_block_details(rbl_block_data_primary, &block_details)) {
            if (block_details.rbl_valid) {
                if (!rbl_load_data(data_buffer, block_details.rbl_size)) {
                    fputs("Failed to load primary data block.\n", file);
                } else {
                    for (i = 0; i < block_details.rbl_size; i++) {
                        if (! (isprint(data_buffer[i]) || ('\r' == data_buffer[i]) || ('\n' == data_buffer[i]))) {
                            break;
                        }
                        putc(data_buffer[i], file);
                    }
                    putc('\n', file);
                }
            }
        }
    }
    
    fclose(file);
}

static void
telnet_server(void)
{
    int server_fd   = create_tcp_server_socket(23);
    if (server_fd < 0) {
        fputs("nettest: failed to create telnet server socket", stderr);
        return;
    }
    for ( ; ; ) {
        int client_fd = accept(server_fd, NULL, NULL);
        if (client_fd < 0) {
            fputs("telnet_server: accept failed on server socket\n", stderr);
            continue;
        }
        if (tftpd_busy) {
            close(client_fd);
        } else {
            telnet_busy = 1;
            telnet_worker(client_fd);
            telnet_busy = 0;
        }
    }
}

// ----------------------------------------------------------------------------
// TFTP support. This allows the code and data to be updated, and the data
// to be retrieved. Only a single transfer at a time is supported. Transfers
// go via the global buffers.

static int  tftpd_read  = 0;
static int  tftpd_code  = 0;
static int  tftpd_index = 0;
static int  tftpd_max   = 0;

static int
nettest_tftpd_open(const char* filename, int flags)
{
    if (tftpd_busy || telnet_busy) {
        return -1;
    }
    if (0 == strcmp(filename, "code")) {
        if (0 != (flags ^ O_WRONLY)) {
            return -1;
        }
        tftpd_busy      = 1;
        tftpd_read      = 0;
        tftpd_code      = 1;
        tftpd_index     = 0;
        tftpd_max       = code_buffer_size;
        return 1;
    }
    if (0 == strcmp(filename, "data")) {
        if (0 == data_buffer_size) {
            // This RedBoot does not support persistent data blocks
            return -1;
        }
        
        tftpd_code  = 0;
        tftpd_read  = (0 == (flags & O_WRONLY));
        if (tftpd_read) {
            rbl_block_details   block_details;
            if (! rbl_get_block_details(rbl_block_data_primary, &block_details)) {
                return -1;
            }
            if (! block_details.rbl_valid) {
                return -1;
            }
            tftpd_max = block_details.rbl_size;
            if (! rbl_load_data(data_buffer, tftpd_max)) {
                return -1;
            }
        } else {
            tftpd_max = data_buffer_size;
        }
        tftpd_busy  = 1;
        tftpd_index = 0;
        return 1;
    }
    return -1;
}

static int
nettest_tftpd_close(int fd)
{
    if (!tftpd_busy) {
        return -1;
    }
    tftpd_busy = 0;

    if (!tftpd_read) {
        if (0 == tftpd_index) {
            diag_printf("tftpd close: no data provided, skipping flash update.\n");
            return 0;
        }
        
        if (tftpd_code) {
            if (! rbl_update_code(code_buffer, tftpd_index)) {
                diag_printf("tftpd close: failed to update RBL code block\n");
            }
        } else {
            if (! rbl_update_data(data_buffer, tftpd_index)) {
                diag_printf("tftpd close: failed to update RBL data block\n");
            }
        }
    }
    return 0;
}

static int
nettest_tftpd_write(int fd, const void* buf, int len)
{
    if ( !tftpd_busy || tftpd_read) {
        return -1;
    }
    if ((tftpd_index + len) > tftpd_max) {
        return -1;
    }
    if (tftpd_code) {
        memcpy(&(code_buffer[tftpd_index]), buf, len);
    } else {
        memcpy(&(data_buffer[tftpd_index]), buf, len);
    }
    tftpd_index += len;
    return len;
}

static int
nettest_tftpd_read(int fd, void* buf, int len)
{
    if (!tftpd_busy || !tftpd_read) {
        return -1;
    }
    if ((tftpd_index + len) > tftpd_max) {
        len = tftpd_max - tftpd_index;
        if (len < 0) {
            return -1;
        }
    }
    if (len > 0) {
        memcpy(buf, &(data_buffer[tftpd_index]), len);
        tftpd_index += len;
    }
    return len;
}

static struct tftpd_fileops nettest_tftpd_fileops = {
    &nettest_tftpd_open,
    &nettest_tftpd_close,
    &nettest_tftpd_write,
    &nettest_tftpd_read
};

// ----------------------------------------------------------------------------
int
main(int argc, char** argv)
{
    init_all_network_interfaces();
    signal(SIGPIPE, SIG_IGN);

    if (allocate_buffers()) {
        tftpd_start(69, &nettest_tftpd_fileops);
        telnet_server();
    }
    for ( ; ; ) ;
}
