#include <pkgconf/system.h>
#include <pkgconf/rbl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <cyg/infra/cyg_type.h>
#include <cyg/infra/testcase.h>
#include <cyg/rbl/rbl.h>
#include <cyg/hal/hal_if.h>

#if (1 == CYGNUM_RBL_VERSION)
int
main(int argc, char** argv)
{
    CYG_TEST_INIT();
    CYG_TEST_FAIL_FINISH("Test requires RBL V2");
}
#else

static rbl_flash_details    details;
static rbl_block_details    data_primary, data_backup;
static rbl_block_details    old_data_primary, old_data_backup;
static rbl_block_details    code_primary, code_backup;
static rbl_block_details    old_code_primary, old_code_backup;

static cyg_uint8*   buf0;
static cyg_uint8*   buf1;
static void*        dataV[1024];
static cyg_uint32   sizesV[1024];

static void
update_block_details(void)
{
    old_data_primary    = data_primary;
    old_data_backup     = data_backup;
    old_code_primary    = code_primary;
    old_code_backup     = code_backup;

    if (! (rbl_get_block_details(rbl_block_data_primary, &data_primary) &&
           rbl_get_block_details(rbl_block_data_backup,  &data_backup)  &&
           rbl_get_block_details(rbl_block_code_primary, &code_primary) &&
           rbl_get_block_details(rbl_block_code_backup,  &code_backup)) ) {
        CYG_TEST_FAIL_FINISH("Failed to get RBL block details");
    }
}

// ----------------------------------------------------------------------------

static void
test_load_dataV(cyg_uint32 initial_size, cyg_uint32 incr_size)
{
    cyg_uint32  current_size;
    cyg_uint32  total;
    cyg_uint8*  ptr;
    int         i;

    if (0 == incr_size) {
        printf("Using load_dataV to load all of data in %d byte chunks\n", initial_size);
    } else {
        printf("Using load_dataV to load all of data in variable size chunks (initial %d, increment %d)\n", initial_size, incr_size);
    }
    for (ptr = buf1, i = 0, current_size = initial_size, total = 0; total < data_primary.rbl_size; i++) {
        dataV[i]    = ptr;
        if ((total + current_size) < data_primary.rbl_size) {
            sizesV[i]   = current_size;
        } else {
            sizesV[i]   = data_primary.rbl_size - total;
        }
        ptr             += current_size;
        total           += current_size;
        current_size    += incr_size;
    }
    if (! rbl_load_dataV(i, dataV, sizesV) ) {
        CYG_TEST_FAIL_FINISH("  Data load failed");
    }
    if (0 != memcmp(buf0, buf1, data_primary.rbl_size)) {
        CYG_TEST_FAIL_FINISH("  Data mismatch");
    }
}

static void
test_transaction_load_data(cyg_uint32 initial_size, cyg_uint32 incr_size)
{
    cyg_uint32  current_size;
    cyg_uint32  total;
    cyg_uint8*  ptr;
    int         i;

    if (0 == incr_size) {
        printf("Using transaction load_data to load all of data in %d byte chunks\n", initial_size);
    } else {
        printf("Using transaction load_data to load all of data in variable size chunks (initial %d, increment %d)\n", initial_size, incr_size);
    }
    if (! rbl_load_data_begin()) {
        CYG_TEST_FAIL_FINISH("rbl_load_data_begin() failed");
    }
    for (ptr = buf1, i = 0, current_size = initial_size, total = 0; total < data_primary.rbl_size; i++) {
        
        if ((total + current_size) > data_primary.rbl_size) {
            current_size    = data_primary.rbl_size - total;
        }
        if (! rbl_load_data_block(ptr, current_size)) {
            CYG_TEST_FAIL_FINISH("rbl_load_data_block() failed");
        }
        ptr             += current_size;
        total           += current_size;
        current_size    += incr_size;
    }
    if (! rbl_load_data_end()) {
        CYG_TEST_FAIL_FINISH("rbl_load_data_end() failed");
    }
    if (0 != memcmp(buf0, buf1, data_primary.rbl_size)) {
        CYG_TEST_FAIL_FINISH("  Data mismatch");
    }
}

static void
test_load_data(void)
{
    cyg_uint32  data_size   = data_primary.rbl_size;
    int         i;

    printf("Testing data loads, loading all of primary data into buffer\n");
    if (! rbl_load_data(buf0, data_size) ) {
        CYG_TEST_FAIL_FINISH("Failed to load all of data");
    }

    printf("Testing data reads of varying lengths\n");
    for (i = 21; i < data_size; i *= 2) {
        printf("  Attempting single load of %d bytes\n", i);
        if (! rbl_load_data(buf1, i) ) {
            CYG_TEST_FAIL_FINISH("  Data load failed");
        }
        if (0 != memcmp(buf0, buf1, i)) {
            CYG_TEST_FAIL_FINISH("  Data mismatch");
        }
    }

    test_load_dataV(65536, 0);
    test_load_dataV( 4096, 0);
    test_load_dataV( 1000, 0);
    test_load_dataV(1024, 512);
    test_load_dataV(907, 1281);
    test_load_dataV(50000, 10000);

    test_transaction_load_data(65536, 0);
    test_transaction_load_data(1024, 0);
    test_transaction_load_data(720, 0);
    test_transaction_load_data(2048, 1024);
    test_transaction_load_data(509, 2173);
    test_transaction_load_data(34567, 128);
}

// ----------------------------------------------------------------------------

static void
test_update_dataV(cyg_uint32 total_size, cyg_uint32 initial_size, cyg_uint32 incr_size)
{
    cyg_uint32  current_size;
    cyg_uint32  total;
    cyg_uint8*  ptr;
    int         i;

    if (0 == incr_size) {
        printf("Using update_dataV to update all of data in %d byte chunks\n", initial_size);
    } else {
        printf("Using updata_dataV to update all of data in variable size chunks (initial %d, increment %d)\n", initial_size, incr_size);
    }
    for (ptr = buf0, i = 0, current_size = initial_size, total = 0; total < total_size; i++) {
        dataV[i]    = ptr;
        if ((total + current_size) > total_size) {
            current_size = total_size - total;
        }
        sizesV[i]        = current_size;
        ptr             += current_size;
        total           += current_size;
        current_size    += incr_size;
    }
    if (! rbl_update_dataV(i, dataV, sizesV) ) {
        CYG_TEST_FAIL_FINISH("  update_dataV failed");
    }
    update_block_details();
    if (!data_primary.rbl_valid || (data_primary.rbl_size != total_size)) {
        CYG_TEST_FAIL_FINISH("  Primary not updated correctly.");
    }
    if ((0 != memcmp(&code_primary, &old_code_primary, sizeof(rbl_block_details))) ||
        (0 != memcmp(&code_backup,  &old_code_backup, sizeof(rbl_block_details)))) {
        CYG_TEST_FAIL_FINISH("  Code blocks unexpectedly updated.");
    }
    if (!rbl_load_data(buf1, total_size)) {
        CYG_TEST_FAIL_FINISH("  Failed to read back data.");
    }
    if (0 != memcmp(buf0, buf1, data_primary.rbl_size)) {
        CYG_TEST_FAIL_FINISH("  Data mismatch");
    }
}

static void
test_transaction_update_data(cyg_uint32 total_size, cyg_uint32 initial_size, cyg_uint32 incr_size)
{
    cyg_uint32  current_size;
    cyg_uint32  total;
    cyg_uint8*  ptr;
    int         i;

    if (0 == incr_size) {
        printf("Using transaction update_data to update all of data in %d byte chunks\n", initial_size);
    } else {
        printf("Using transaction update_data to update all of data in variable size chunks (initial %d, increment %d)\n", initial_size, incr_size);
    }
    if (! rbl_update_data_begin()) {
        CYG_TEST_FAIL_FINISH("rbl_update_data_begin() failed");
    }
    for (ptr = buf0, i = 0, current_size = initial_size, total = 0; total < total_size; i++) {
        
        if ((total + current_size) > total_size) {
            current_size    = total_size - total;
        }
        if (! rbl_update_data_block(ptr, current_size)) {
            CYG_TEST_FAIL_FINISH("rbl_update_data_block() failed");
        }
        ptr             += current_size;
        total           += current_size;
        current_size    += incr_size;
    }
    if (! rbl_update_data_end()) {
        CYG_TEST_FAIL_FINISH("rbl_update_data_end() failed");
    }
    update_block_details();
    if (!data_primary.rbl_valid || (data_primary.rbl_size != total_size)) {
        CYG_TEST_FAIL_FINISH("  Primary not updated correctly.");
    }
    if ((0 != memcmp(&code_primary, &old_code_primary, sizeof(rbl_block_details))) ||
        (0 != memcmp(&code_backup,  &old_code_backup, sizeof(rbl_block_details)))) {
        CYG_TEST_FAIL_FINISH("  Code blocks unexpectedly updated.");
    }
    if (!rbl_load_data(buf1, total_size)) {
        CYG_TEST_FAIL_FINISH("  Failed to read back data.");
    }
    if (0 != memcmp(buf0, buf1, data_primary.rbl_size)) {
        CYG_TEST_FAIL_FINISH("  Data mismatch");
    }
}

static void
test_update_data(void)
{
    int         i;
    cyg_uint32  data_size   = data_primary.rbl_size;
    cyg_uint32  old_seqno   = data_primary.rbl_sequence_number;
    
    printf("Testing data updates, loading all of primary data into buffer\n");
    if (! rbl_load_data(buf0, data_size) ) {
        CYG_TEST_FAIL_FINISH("Failed to load all of data");
    }
    
    printf("Testing data updates of varying lengths\n");
    for (i = 24; i < data_size; i *= 8) {
        printf("  Attempting update of %d bytes\n", i);
        if (! rbl_update_data(&(buf0[data_size - i]), i) ) {
            CYG_TEST_FAIL_FINISH("  Data load failed");
        }
        update_block_details();
        if (!data_primary.rbl_valid || (data_primary.rbl_size != i)) {
            CYG_TEST_FAIL_FINISH("  Primary not updated.");
        }
        if (!rbl_load_data(buf1, i)) {
            CYG_TEST_FAIL_FINISH("  Failed to read back data.");
        }
        if (0 != memcmp(&(buf0[data_size - i]), buf1, i)) {
            CYG_TEST_FAIL_FINISH("  Data mismatch");
        }
    }

    test_update_dataV(data_size, 65536,     0);
    test_update_dataV(data_size,  4096,     0);
    test_update_dataV(data_size,  1000,     0);
    test_update_dataV(data_size,  1024,   512);
    test_update_dataV(data_size,   904,  1280);
    test_update_dataV(data_size, 50000, 10000);

    test_transaction_update_data(data_size, 65536,    0);
    test_transaction_update_data(data_size,  1024,    0);
    test_transaction_update_data(data_size,   720,    0);
    test_transaction_update_data(data_size,  2048, 1024);
    test_transaction_update_data(data_size,   508, 2172);
    test_transaction_update_data(data_size, 34568,  128);

    printf("Testing abort operation.\n");
    if (! rbl_update_data_begin()) {
        CYG_TEST_FAIL_FINISH("rbl_update_data_begin() failed.\n");
    }
    if (! rbl_update_data_abort()) {
        CYG_TEST_FAIL_FINISH("rbl_update_data_abort() failed.\n");
    }
    update_block_details();
    if (0 != memcmp(&data_primary, &old_data_primary, sizeof(rbl_block_details))) {
        CYG_TEST_FAIL_FINISH("  Primary data block incorrectly updated");
    }
    if (data_backup.rbl_valid) {
        CYG_TEST_FAIL_FINISH("  Backup data block should no longer be valid");
    }
    if ((0 != memcmp(&code_primary, &old_code_primary, sizeof(rbl_block_details))) ||
        (0 != memcmp(&code_backup,  &old_code_backup, sizeof(rbl_block_details)))) {
        CYG_TEST_FAIL_FINISH("  Code blocks unexpectedly updated.");
    }
    
    if (! rbl_update_data_begin()) {
        CYG_TEST_FAIL_FINISH("rbl_update_data_begin() failed.\n");
    }
    if (! rbl_update_data_block(buf0, data_size)) {
        CYG_TEST_FAIL_FINISH("rbl_update_data_block() failed.\n");
    }
    if (! rbl_update_data_abort()) {
        CYG_TEST_FAIL_FINISH("rbl_update_data_abort() failed.\n");
    }
    update_block_details();
    if (0 != memcmp(&data_primary, &old_data_primary, sizeof(rbl_block_details))) {
        CYG_TEST_FAIL_FINISH("  Primary data block incorrectly updated");
    }
    if (data_backup.rbl_valid) {
        CYG_TEST_FAIL_FINISH("  Backup data block should no longer be valid");
    }
    if ((0 != memcmp(&code_primary, &old_code_primary, sizeof(rbl_block_details))) ||
        (0 != memcmp(&code_backup,  &old_code_backup, sizeof(rbl_block_details)))) {
        CYG_TEST_FAIL_FINISH("  Code blocks unexpectedly updated.");
    }
    
    printf("Restoring original data.\n");
    if (! rbl_update_data(buf0, data_size)) {
        CYG_TEST_FAIL_FINISH("  Primary not restored");
    }
    update_block_details();
    printf("Primary data sequence number was %d, now %d\n", old_seqno, data_primary.rbl_sequence_number);
}

// ----------------------------------------------------------------------------

static void
test_update_codeV(cyg_uint32 initial_size, cyg_uint32 incr_size)
{
    cyg_uint32  current_size;
    cyg_uint32  total;
    cyg_uint8*  ptr;
    int         i;

    if (0 == incr_size) {
        printf("Using update_codeV to update current executable in %d byte chunks\n", initial_size);
    } else {
        printf("Using updata_codeV to update current executable in variable size chunks (initial %d, increment %d)\n", initial_size, incr_size);
    }
    for (ptr = buf0, i = 0, current_size = initial_size, total = 0; total < code_primary.rbl_size; i++) {
        dataV[i]    = ptr;
        if ((total + current_size) > code_primary.rbl_size) {
            current_size = code_primary.rbl_size - total;
        }
        sizesV[i]        = current_size;
        ptr             += current_size;
        total           += current_size;
        current_size    += incr_size;
    }
    if (! rbl_update_codeV(i, dataV, sizesV) ) {
        CYG_TEST_FAIL_FINISH("  update_codeV failed");
    }
    update_block_details();
    if (!code_primary.rbl_valid ||
        (code_primary.rbl_first_flash_block != old_code_backup.rbl_first_flash_block) ||
        (code_primary.rbl_size              != old_code_primary.rbl_size) ||
        (code_primary.rbl_sequence_number   != (old_code_primary.rbl_sequence_number + 1))) {
        CYG_TEST_FAIL_FINISH("  Primary not updated correctly.");
    }
    if ((0 != memcmp(&data_primary, &old_data_primary, sizeof(rbl_block_details))) ||
        (0 != memcmp(&data_backup,  &old_data_backup, sizeof(rbl_block_details)))) {
        CYG_TEST_FAIL_FINISH("  Data blocks unexpectedly updated.");
    }
}

static void
test_transaction_update_code(cyg_uint32 initial_size, cyg_uint32 incr_size)
{
    cyg_uint32  current_size;
    cyg_uint32  total;
    cyg_uint8*  ptr;
    int         i;

    if (0 == incr_size) {
        printf("Using transaction update_code to update current executable in %d byte chunks\n", initial_size);
    } else {
        printf("Using transaction update_code to update current executable in variable size chunks (initial %d, increment %d)\n", initial_size, incr_size);
    }
    if (! rbl_update_code_begin()) {
        CYG_TEST_FAIL_FINISH("rbl_update_code_begin() failed");
    }
    for (ptr = buf0, i = 0, current_size = initial_size, total = 0; total < code_primary.rbl_size; i++) {
        
        if ((total + current_size) > code_primary.rbl_size) {
            current_size    = code_primary.rbl_size - total;
        }
        if (! rbl_update_code_block(ptr, current_size)) {
            CYG_TEST_FAIL_FINISH("rbl_update_code_block() failed");
        }
        ptr             += current_size;
        total           += current_size;
        current_size    += incr_size;
    }
    if (! rbl_update_code_end()) {
        CYG_TEST_FAIL_FINISH("rbl_update_code_end() failed");
    }
    update_block_details();
    if (!code_primary.rbl_valid ||
        (code_primary.rbl_first_flash_block != old_code_backup.rbl_first_flash_block) ||
        (code_primary.rbl_size              != old_code_primary.rbl_size) ||
        (code_primary.rbl_sequence_number   != (old_code_primary.rbl_sequence_number + 1))) {
        CYG_TEST_FAIL_FINISH("  Primary not updated correctly.");
    }
    if ((0 != memcmp(&data_primary, &old_data_primary, sizeof(rbl_block_details))) ||
        (0 != memcmp(&data_backup,  &old_data_backup, sizeof(rbl_block_details)))) {
        CYG_TEST_FAIL_FINISH("  Data blocks unexpectedly updated.");
    }
}

static void
test_update_code(void)
{
    cyg_uint32  code_size   = code_primary.rbl_size;
    cyg_uint32  old_seqno   = code_primary.rbl_sequence_number;
    cyg_uint8*  src;
    cyg_uint8*  dst;
    int         i;
    cyg_uint32  total;
    rbl_flash_block_purpose purpose;
    
    printf("Testing code updates, loading existing executable into buffer\n");
    src = details.rbl_flash_base;
    src += (code_primary.rbl_first_flash_block * details.rbl_flash_block_size);
    purpose = rbl_get_flash_block_purpose(code_primary.rbl_first_flash_block);
    for (total = 0, i = code_primary.rbl_first_flash_block, dst = buf0;
         (i  < details.rbl_flash_num_blocks) && (total < code_size);
         i++, src += details.rbl_flash_block_size) {
        if (purpose == rbl_get_flash_block_purpose(i)) {
            printf("  Found next part of code in flash block %d @ %p\n", i, src);
            memcpy(dst, src, details.rbl_flash_block_size);
            dst   += details.rbl_flash_block_size;
            total += details.rbl_flash_block_size;
        }
    }
    if (i >= details.rbl_flash_num_blocks) {
        CYG_TEST_FAIL_FINISH("Failed to find all code blocks");
    }
    
    test_update_codeV(65536,     0);
    test_update_codeV( 4096,     0);
    test_update_codeV( 1000,     0);
    test_update_codeV( 1024,   512);
    test_update_codeV(  904,  1280);
    test_update_codeV(50000, 10000);

    test_transaction_update_code(65536,    0);
    test_transaction_update_code( 1024,    0);
    test_transaction_update_code(  720,    0);
    test_transaction_update_code( 2048, 1024);
    test_transaction_update_code(  508, 2172);
    test_transaction_update_code(34568,  128);

    printf("Testing abort operation.\n");
    if (! rbl_update_code_begin()) {
        CYG_TEST_FAIL_FINISH("rbl_update_code_begin() failed.\n");
    }
    if (! rbl_update_code_abort()) {
        CYG_TEST_FAIL_FINISH("rbl_update_code_abort() failed.\n");
    }
    update_block_details();
    if (0 != memcmp(&code_primary, &old_code_primary, sizeof(rbl_block_details))) {
        CYG_TEST_FAIL_FINISH("  Primary code block incorrectly updated");
    }
    if (code_backup.rbl_valid) {
        CYG_TEST_FAIL_FINISH("  Backup code block should no longer be valid");
    }
    if ((0 != memcmp(&data_primary, &old_data_primary, sizeof(rbl_block_details))) ||
        (0 != memcmp(&data_backup,  &old_data_backup, sizeof(rbl_block_details)))) {
        CYG_TEST_FAIL_FINISH("  Data blocks unexpectedly updated.");
    }
    
    if (! rbl_update_code_begin()) {
        CYG_TEST_FAIL_FINISH("rbl_update_code_begin() failed.\n");
    }
    if (! rbl_update_code_block(buf0, code_size)) {
        CYG_TEST_FAIL_FINISH("rbl_update_code_block() failed.\n");
    }
    if (! rbl_update_code_abort()) {
        CYG_TEST_FAIL_FINISH("rbl_update_code_abort() failed.\n");
    }
    update_block_details();
    if (0 != memcmp(&code_primary, &old_code_primary, sizeof(rbl_block_details))) {
        CYG_TEST_FAIL_FINISH("  Primary code block incorrectly updated");
    }
    if (code_backup.rbl_valid) {
        CYG_TEST_FAIL_FINISH("  Backup code block should no longer be valid");
    }
    if ((0 != memcmp(&data_primary, &old_data_primary, sizeof(rbl_block_details))) ||
        (0 != memcmp(&data_backup,  &old_data_backup, sizeof(rbl_block_details)))) {
        CYG_TEST_FAIL_FINISH("  Code blocks unexpectedly updated.");
    }
    
    printf("Restoring original code.\n");
    if (! rbl_update_code(buf0, code_size)) {
        CYG_TEST_FAIL_FINISH("  Primary not restored");
    }
    update_block_details();
    printf("Primary code sequence number was %d, now %d\n", old_seqno, code_primary.rbl_sequence_number);
}

int
main(int argc, char** argv)
{
    CYG_TEST_INIT();
    if (! rbl_get_flash_details(&details)) {
        CYG_TEST_FAIL_FINISH("rbl_get_flash_details failed");
    }
    printf("RBL: base %p, flash block size %d, %d flash blocks, code uses %d blocks, data uses %d blocks, trailer size %d bytes\n",
           details.rbl_flash_base, details.rbl_flash_block_size, details.rbl_flash_num_blocks,
           details.rbl_code_num_blocks, details.rbl_data_num_blocks, details.rbl_trailer_size);

    if (0 == details.rbl_data_num_blocks) {
        CYG_TEST_FAIL_FINISH("Test requires RBL data blocks");
    }
    
    if (details.rbl_code_num_blocks > details.rbl_data_num_blocks) {
        buf0    = malloc(details.rbl_code_num_blocks * details.rbl_flash_block_size);
        buf1    = malloc(details.rbl_code_num_blocks * details.rbl_flash_block_size);
    } else {
        buf0    = malloc(details.rbl_data_num_blocks * details.rbl_flash_block_size);
        buf1    = malloc(details.rbl_data_num_blocks * details.rbl_flash_block_size);
    }
    if ((NULL == buf0) || (NULL == buf1)) {
        CYG_TEST_FAIL_FINISH("Failed to allocate buffers");
    }

    update_block_details();
    printf("Code primary: valid %d, first flash block %d, address %p, size %d, seqno 0x%08x\n",
           code_primary.rbl_valid,   code_primary.rbl_first_flash_block,
           code_primary.rbl_address, code_primary.rbl_size,
           code_primary.rbl_sequence_number);
    printf("Data primary: valid %d, first flash block %d, address %p, size %d, seqno 0x%08x\n",
           data_primary.rbl_valid,   data_primary.rbl_first_flash_block,
           data_primary.rbl_address, data_primary.rbl_size,
           data_primary.rbl_sequence_number);

    if (! data_primary.rbl_valid ) {
        CYG_TEST_FAIL_FINISH("Test requires a valid primary data block.");
    }
    if (! code_primary.rbl_valid) {
        CYG_TEST_FAIL_FINISH("Test requires a valid primary code block.");
    }

    test_load_data();
    test_update_data();
    test_update_code();
    
    CYG_TEST_PASS_FINISH("Done");
}
#endif

