/******************************************************************************

  Copyright (C), 2001-2011, Hisilicon Tech. Co., Ltd.

 ******************************************************************************
  File Name     : vou_adp.c
  Version       : Initial Draft
  Author        : Hisilicon multimedia software group
  Created       : 2009/03/13
  Description   :
  History       :
  1.Date        : 2009/03/13
    Author      : x00100808
    Modification: Created file
  2.Date        : 2009/08/13
    Author      : w54723
    Modification: AE6D04063.
    			  adjust user setting alpha which range 0~255
    			  into chip range 0~128
******************************************************************************/

#include <common.h>
#include <command.h>
#include <malloc.h>
#include <devices.h>
#include <version.h>

#include <asm/arch/platform.h>
#include <asm/sizes.h>

#include "hi_type.h"
#include "vou_reg.h"
#include "vou_def.h"
#include "vou_hal.h"
#include "vou_adp.h"

#ifdef __cplusplus
#if __cplusplus
extern "C"{
#endif
#endif /* End of #ifdef __cplusplus */

#define VOU_DEV_MAX_NUM             3
#define VOU_MAX_LAYER_PRIORITY      4
#define VOU_BACKGROUD_DEFAULT       0x8080     /* 0x804D3A42(green) 0x8080(black)*/

#define VOU_VTTH_WATERLINE          100

#ifdef __LINUX_OS__
#define VO_LOCK_IRQ(flags)  local_irq_save(flags)
#define VO_UNLOCK_IRQ(flags)  local_irq_restore(flags)
#else
#define VO_LOCK_IRQ(flags)      do{flags = 0;}while(0)
#define VO_UNLOCK_IRQ(flags)    do{flags = 0;}while(0);
#endif


#define VO_SC_PERCTRL2     0x20050034   // VPPL0
#define VO_SC_PERCTRL3     0x20050038   // VPPL0
#define VO_SC_PERCTRL4     0x2005003c   // VPPL1
#define VO_SC_PERCTRL5     0x20050040   // VPPL1
#define VO_SC_PERCTRL8     0x20050050   // RESET
#define VO_SC_PERCTRL9     0x20050054   // CLK
#define VO_SC_PERCTRL12    0x20050060   // DAC


typedef enum hiVOU_MUX_E
{
    VOU_MUX_CVBS     = 0,                  /* for SD or AD mux */
    VOU_MUX_BT656    = 1,
    VOU_MUX_VGA      = 2,                  /* for HD mux */
    VOU_MUX_YPBPR    = 3,
    VOU_MUX_BT1120   = 4,
    VOU_MUX_LCD      = 5,
    VOU_MUX_BUTT

} VOU_MUX_E;

typedef enum hiVOU_OUT_MODE_E
{
    VOU_OUTPUT_PAL           = 0,
    VOU_OUTPUT_NTSC          = 1,

    VOU_OUTPUT_720P60        = 2,
    VOU_OUTPUT_1080I60       = 3,
    VOU_OUTPUT_1080P30       = 4,

    VOU_OUTPUT_800x600_60    = 5,
    VOU_OUTPUT_1024x768_60   = 6,
    VOU_OUTPUT_1280x1024_60  = 7,
    VOU_OUTPUT_1366x768_60   = 8,
    VOU_OUTPUT_1440x900_60   = 9,

    VOU_OUTPUT_USER          = 10,
    VOU_OUTPUT_BUTT

} VOU_OUT_MODE_E;

typedef enum hiVOU_PIXERL_FORMAT_E
{
    /* for device interface */
    VOU_PIXERL_FORMAT_YCBCR422          = 0x0,
    VOU_PIXERL_FORMAT_YCBCR444          = 0xC,
    VOU_PIXERL_FORMAT_RGB888            = 0xC,

    /* for video surface interface */
    VOU_PIXERL_FORMAT_SPYCbCr_420       = 0x3,
    VOU_PIXERL_FORMAT_SPYCbCr_422       = 0x4,
    VOU_PIXERL_FORMAT_BUTT

} VOU_PIXERL_FORMAT_E;

typedef struct hiHAL_CONFIG_S
{
    HI_U32          u32BkGrd;
    VOU_MUX_E       enMuxIntf;
    VOU_OUT_MODE_E  enOutSync;
    VOU_PIXERL_FORMAT_E enPixelFmt;

} HAL_CONFIG_S;


static HAL_DISP_SYNCINFO_S g_stSyncTiming[VO_OUTPUT_BUTT] =
{
/* |--INTFACE--|   |---TOP-----|  |--HORIZON--|  |---BOTTOM----| |-PULSE-| |-INVERSE-| */
/* Synm,Iop, Itf,  Vact,Vbb,Vfb,  Hact, Hbb,Hfb, Bvact,Bvbb,Bvfb, Hpw,Vpw, Idv,Ihs,Ivs */
    {0,  0,   0,    288, 22,  2,   720, 132, 12,  288,  23,   2,  128,  4,  0,  0,  0},/* 576I(PAL) */
    {0,  0,   0,    244, 16,  2,   720, 122, 16,  243,  17,   3,  128,  4,  0,  0,  0},/* 480I(NTSC) */
    {1,  1,   2,    720, 25,  5,  1280, 260,110,    1,   1,   1,  128,  4,  0,  0,  0},/* 720P@60Hz */
    {1,  0,   2,    540, 20,  3,  1920, 192, 528, 540,  20,   2,  128,  4,  0,  0,  0},/* 1080I@50Hz */
    {1,  0,   2,    540, 20,  3,  1920, 192, 88,  540,  20,   2,  128,  4,  0,  0,  0},/* 1080I@60Hz */
    {1,  1,   2,   1080, 41,  4,  1920, 192, 528,   1,   1,   1,  128,  4,  0,  0,  0},/* 1080P@25Hz */
    {1,  1,   2,   1080, 41,  4,  1920, 192, 88,    1,   1,   1,  128,  4,  0,  0,  0},/* 1080P@30Hz */
    {1,  1,   2,    600, 27,  1,   800, 216, 40,    1,   1,   1,  128,  4,  0,  0,  0},/* 800*600@60Hz */
    {1,  1,   2,    768, 35,  3,  1024, 296, 24,    1,   1,   1,  136,  6,  0,  0,  0},/* 1024x768@60Hz */
    {1,  1,   2,   1024, 41,  1,  1280, 360, 48,    1,   1,   1,  112,  3,  0,  0,  0},/* 1280x1024@60Hz */
    {1,  1,   2,    768, 27,  3,  1366, 356, 70,    1,   1,   1,  143,  3,  0,  0,  0},/* 1366x768@60Hz */
    {1,  1,   2,    900, 23,  3,  1440, 112, 48,    1,   1,   1,   32,  6,  0,  0,  0} /* 1440x900@60Hz@88.75MHz */
};

static HAL_CONFIG_S g_stHalCfg[VOU_DEV_MAX_NUM] =
{
    {
        .u32BkGrd       = VOU_BACKGROUD_DEFAULT,
        .enMuxIntf      = VO_INTF_VGA,
        .enOutSync      = VO_OUTPUT_1024x768_60,           /* for debug: VO_OUTPUT_800x600_60 */
        .enPixelFmt     = VOU_PIXERL_FORMAT_RGB888,        /* VOU_PIXERL_FORMAT_RGB888, VOU_PIXERL_FORMAT_YCBCR422*/
    },
    {
        .u32BkGrd       = VOU_BACKGROUD_DEFAULT,
        .enMuxIntf      = VO_INTF_CVBS,
        .enOutSync      = VO_OUTPUT_PAL,
        .enPixelFmt     = VOU_PIXERL_FORMAT_YCBCR422,
    },
    {
        .u32BkGrd       = VOU_BACKGROUD_DEFAULT,
        .enMuxIntf      = VO_INTF_CVBS,
        .enOutSync      = VO_OUTPUT_PAL,
        .enPixelFmt     = VOU_PIXERL_FORMAT_YCBCR422,
    }
};


#define VOU_ALPHA_ADJUST(u8Alpha)\
do{\
    if(255 == u8Alpha)\
    {\
        u8Alpha = 0x80;\
    }\
    else\
    {\
        u8Alpha = ((u8Alpha + 1)>>1); \
    }\
}while(0)


HI_VOID VOU_DRV_BoardInit(HI_VOID)
{
    HAL_VOU_Init();
    HAL_SYS_UndoVouSoftRst(VO_SC_PERCTRL8);
    HAL_SYS_Control();

    /* default bind relationship between graphic and device */
    VOU_DRV_LayerVBindMixer(VOU_LAYER_HD, VOU_MIXER_1);
    VOU_DRV_LayerVBindMixer(VOU_LAYER_AD, VOU_MIXER_2);

    /* default bind relation                                                */
    VOU_DRV_LayerBindMixer(GRPHIC_LAYER_G0, OUT_DEV_HD);
    VOU_DRV_LayerBindMixer(GRPHIC_LAYER_G1, OUT_DEV_HD);
    VOU_DRV_LayerBindMixer(GRPHIC_LAYER_G2, OUT_DEV_AD);
    VOU_DRV_LayerBindMixer(GRPHIC_LAYER_HC, OUT_DEV_AD);

    /* default priority                                                     */
    VOU_DRV_SetLayerPriority(HAL_MXIER_LAYER_HD, 0);    /* low priority, on mixer 1 */
    VOU_DRV_SetLayerPriority(HAL_MXIER_LAYER_G0, 1);
    VOU_DRV_SetLayerPriority(HAL_MXIER_LAYER_G1, 2);
    VOU_DRV_SetLayerPriority(HAL_MXIER_LAYER_AD, 3);    /* high priority, on mixer 2 */
    VOU_DRV_SetLayerPriority(HAL_MXIER_LAYER_G2, 4);

    /* SD and G3 priority settings                                          */
    VOU_DRV_SetLayerPriority(HAL_MXIER_LAYER_G3, 0);

    /* enable DAC */
    HAL_INTF_DacEnable(0, HI_TRUE, VO_SC_PERCTRL12);
    HAL_INTF_DacEnable(1, HI_TRUE, VO_SC_PERCTRL12);
}

/* device relative                                                          */

#ifdef hi3515

HI_VOID VOU_DRV_Open(VO_DEV_E VoDev)
{
    HAL_DISP_SYNCINFO_S stSyncInfo;
    VOU_PIXERL_FORMAT_E enPixFmt;

    /* before enable, we disable first and reset device attributes.         */
    HAL_DISP_IntfEnable(VoDev, HI_FALSE);

    if (VO_DEV_HD == VoDev)
    {
        HAL_CMB_SetMixerBackGroud(HAL_MIXER_1, g_stHalCfg[VoDev].u32BkGrd);
    }
    else if (VO_DEV_SD == VoDev)
    {
        HAL_CMB_SetMixerBackGroud(HAL_MIXER_2, g_stHalCfg[VoDev].u32BkGrd);
    }

    /* Timing Setting */
    memcpy(&stSyncInfo, &g_stSyncTiming[g_stHalCfg[VoDev].enOutSync], sizeof(stSyncInfo));

    /* Pin Mux */
    if (2 == VoDev)
    {
        switch(g_stHalCfg[VoDev].enMuxIntf)
        {
            case VO_INTF_BT656:
            {
                HI_U32 i, u32Value;
                for (i = 0x200f0070; i < 0x200f0094; i+=4)
                {
                    HAL_ReadReg(i, &u32Value);
                    u32Value = 0x1;
                    HAL_WriteReg(i, u32Value);
                }
                break;
            }
            default:
            {
                /* it seems we needn't care "recover" */
                HI_U32 i, u32Value;
                for (i = 0x200f0070; i < 0x200f0094; i+=4)
                {
                    HAL_ReadReg(i, &u32Value);
                    u32Value &= ~0x1;
                    HAL_WriteReg(i, u32Value);
                }
                break;
            }
        }
    }

    switch(g_stHalCfg[VoDev].enMuxIntf)
    {
        case VO_INTF_BT1120:
            stSyncInfo.bSynm   = HAL_DISP_SYNC_MODE_TIMING;
            stSyncInfo.u8Intfb = HAL_DISP_BIT_WIDTH_2;
            enPixFmt           = VOU_PIXERL_FORMAT_YCBCR422;
            break;
        case VO_INTF_BT656:
            stSyncInfo.bSynm   = HAL_DISP_SYNC_MODE_TIMING;
            stSyncInfo.u8Intfb = HAL_DISP_BIT_WIDTH_1;
            enPixFmt           = VOU_PIXERL_FORMAT_YCBCR422;
            break;
        case VO_INTF_VGA:
            enPixFmt           = VOU_PIXERL_FORMAT_RGB888;
            break;
        case VO_INTF_YPBPR:
            enPixFmt           = VOU_PIXERL_FORMAT_YCBCR444;
            break;
        default:
            enPixFmt = g_stHalCfg[VoDev].enPixelFmt;
            break;
    }

    HAL_DISP_SetIntfDataFmt(VoDev, enPixFmt);
    HAL_DISP_SetIntfSync(VoDev, stSyncInfo);

    /* CLOCK setting */
    if (VO_DEV_HD == VoDev || VO_INTF_VGA == g_stHalCfg[VoDev].enMuxIntf)
    {
        switch (g_stHalCfg[VoDev].enOutSync)
        {
            case VO_OUTPUT_720P60 :         // 74.25MHz
                HAL_WriteReg(VO_SC_PERCTRL4, 0x34400000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c104a);
                break;
            case VO_OUTPUT_1080I50 :        // 74.25MHz
                HAL_WriteReg(VO_SC_PERCTRL4, 0x34400000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c104a);
                break;
            case VO_OUTPUT_1080I60 :        // 74.25MHz
                HAL_WriteReg(VO_SC_PERCTRL4, 0x34400000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c104a);
                break;
            case VO_OUTPUT_1080P25 :        // 74.25MHz
                HAL_WriteReg(VO_SC_PERCTRL4, 0x34400000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c104a);
                break;
            case VO_OUTPUT_1080P30 :        // 74.25MHz
                HAL_WriteReg(VO_SC_PERCTRL4, 0x34400000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c104a);
                break;
            case VO_OUTPUT_800x600_60 :     // 40M
                HAL_WriteReg(VO_SC_PERCTRL4, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c1028);
                break;
            case VO_OUTPUT_1024x768_60 :    // 65M
                HAL_WriteReg(VO_SC_PERCTRL4, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c1041);
                break;
            case VO_OUTPUT_1280x1024_60 :   // 108M
                HAL_WriteReg(VO_SC_PERCTRL4, 0xa2000000);   // 0x9C000000 (another config)
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c1024);   // 0x006C1036
                break;
            case VO_OUTPUT_1366x768_60 :    // 86M
                HAL_WriteReg(VO_SC_PERCTRL4, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c1055);
                break;
            case VO_OUTPUT_1440x900_60 :    // 88M
                HAL_WriteReg(VO_SC_PERCTRL4, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c1058);   //  1440x900@60Hz
                break;
        default:                            // 65M
                HAL_WriteReg(VO_SC_PERCTRL4, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c1041);
        }

        /* PLLڱƵʱҪȴ0.1msȶʱӡ
            PLLƵֻϵͳSLOWģʽ½С*/
        /* we delay 200 microsecond, udelay unit is micro */
        udelay(200);
    }
    else
    {
        /* AD & SD clock configure                                          */
        HAL_INTF_DivSel(2, 0x01, VO_SC_PERCTRL9);
    #if 1
        /* another choice maybe: (test by chenping)
        	CVBSΪ54Mʱ
        	SC_REG(SC_PERCTRL2) = 0xab000000;//vpll0 clk=54M(ad & sd)
        	SC_REG(SC_PERCTRL3) = 0x006c2087;//vpll0 clk=54M(ad & sd)
         */

        HAL_WriteReg(VO_SC_PERCTRL2, 0x9c000000);       // vpll0 clk=54M(ad & sd)
        HAL_WriteReg(VO_SC_PERCTRL3, 0x006c1036);       // vpll0 clk=54M(ad & sd)
    #else
        HAL_WriteReg(VO_SC_PERCTRL2, 0xa5000000);       // cvbs clk=27M(ad & sd)
        HAL_WriteReg(VO_SC_PERCTRL3, 0x006c102d);       // cvbs clk=27M(ad & sd)
    #endif

        udelay(200);
    }

    /* DATE Setting */
    /* modify r/w mode? need confirm */
    switch (g_stHalCfg[VoDev].enMuxIntf)
    {
        case VO_INTF_YPBPR :
            /* Fall through */
        case VO_INTF_CVBS :
            VOU_DRV_DateSetting(VoDev, g_stHalCfg[VoDev].enOutSync);
            break;
        default:
            HAL_DATE_Enable(VoDev, HI_FALSE);
    }

    /* CLIP Setting */
    switch (g_stHalCfg[VoDev].enMuxIntf)
    {
        case VO_INTF_BT656 :
            /* Fall through */
        case VO_INTF_CVBS :
            /* Fall through */
        case VO_INTF_BT1120 :
            /* Fall through */
        case VO_INTF_YPBPR :
        {
            #if 0   /* should be 10-bit per member */
            HAL_DISP_CLIP_S stClipData = {0x41,0x0,0x40,0x3ac,0x3c0,0x3c0};
            #else
            HAL_DISP_CLIP_S stClipData = {0x40,0x40,0x40,0x3ac,0x3c0,0x3c0};
            #endif

            HAL_DISP_SetIntfClip(VoDev, HI_TRUE, stClipData);
            break;
        }
        default:
        {
            #if 0
            HAL_DISP_CLIP_S stClipData = {0x41,0x0,0x40,0x3ac,0x3c0,0x3c0};
            #else
            HAL_DISP_CLIP_S stClipData = {0x40,0x40,0x40,0x3ac,0x3c0,0x3c0};
            #endif

            HAL_DATE_Enable(VoDev, HI_FALSE);
            HAL_DISP_SetIntfClip(VoDev, HI_FALSE, stClipData);
        }
    }

    /* CSC setting */
    if (VO_INTF_VGA == g_stHalCfg[VoDev].enMuxIntf)
    {
        HAL_DISP_IntfCscEnable(VoDev, HI_TRUE);

        /* for demo special csc coefficient setting */
        #if 0
        HAL_WriteReg(0x20130c20, 0x7eb0180);
        HAL_WriteReg(0x20130c28, 0x129);
        HAL_WriteReg(0x20130c2c, 0x12901cb);
        HAL_WriteReg(0x20130c30, 0x1f781fca);
        HAL_WriteReg(0x20130c34, 0x21d0129);
        HAL_WriteReg(0x20130c38, 0x0);
        #endif

        /* new csc coefficient from A-Gao Asic board  2009-07-28 */
        /* Աȶ=50,Ͷ=50,ɫ=50,=0 */
        if (VO_DEV_HD == VoDev)
        {
            HAL_WriteReg(0x20130c20, 0x7c30180);
            HAL_WriteReg(0x20130c28, 0x129);
            HAL_WriteReg(0x20130c2c, 0x12901cb);
            HAL_WriteReg(0x20130c30, 0x1f781fca);
            HAL_WriteReg(0x20130c34, 0x21d0129);
            HAL_WriteReg(0x20130c38, 0x0);
        }
        else if (VO_DEV_AD == VoDev)            /* AD is VGA */
        {
            HAL_WriteReg(0x20130d20, 0x7c30180);
            HAL_WriteReg(0x20130d28, 0x129);
            HAL_WriteReg(0x20130d2c, 0x12901cb);
            HAL_WriteReg(0x20130d30, 0x1f781fca);
            HAL_WriteReg(0x20130d34, 0x21d0129);
            HAL_WriteReg(0x20130d38, 0x0);
        }
    }
    else if (VO_INTF_CVBS == g_stHalCfg[VoDev].enMuxIntf)
    {
        if (VO_DEV_AD == VoDev)
        {
            #if 0 /* XuCong ֵ */
            HAL_WriteReg(0x20130d20, 0x07c30180);
            HAL_WriteReg(0x20130d24, 0x00410080);
            HAL_WriteReg(0x20130d28, 0x00000100);
            HAL_WriteReg(0x20130d2c, 0x00000000);
            HAL_WriteReg(0x20130d30, 0x0000010f);
            HAL_WriteReg(0x20130d34, 0x00000000);
            HAL_WriteReg(0x20130d38, 0x0000010f);
            #else /* ɫ */
            HAL_WriteReg(0x20130d20, 0x07c30180);
            HAL_WriteReg(0x20130d24, 0x00410080);
            HAL_WriteReg(0x20130d28, 0x00000119);
            HAL_WriteReg(0x20130d2c, 0x00000000);
            HAL_WriteReg(0x20130d30, 0x00000113);
            HAL_WriteReg(0x20130d34, 0x00000000);
            HAL_WriteReg(0x20130d38, 0x00000113);
            #endif
            HAL_DISP_IntfCscEnable(VoDev, HI_TRUE);
        }
        else if (VO_DEV_SD == VoDev)
        {
            #if 0
            /* register value same as AD  */
            HAL_WriteReg(0x20130e20, 0x07c30180);
            HAL_WriteReg(0x20130e24, 0x00410080);
            HAL_WriteReg(0x20130e28, 0x00000119);
            HAL_WriteReg(0x20130e2c, 0x00000000);
            HAL_WriteReg(0x20130e30, 0x00000113);
            HAL_WriteReg(0x20130e34, 0x00000000);
            HAL_WriteReg(0x20130e38, 0x00000113);
            #else
            /* VM700T test best CSC value  */
            HAL_WriteReg(0x20130e20, 0x07c30180);
            HAL_WriteReg(0x20130e24, 0x00410080);
            HAL_WriteReg(0x20130e28, 0x00000100);
            HAL_WriteReg(0x20130e2c, 0x00000000);
            HAL_WriteReg(0x20130e30, 0x000000f0);
            HAL_WriteReg(0x20130e34, 0x00000000);
            HAL_WriteReg(0x20130e38, 0x000000f0);
            #endif
            HAL_DISP_IntfCscEnable(VoDev, HI_TRUE);
        }
    }
    else
    {
        HAL_DISP_IntfCscEnable(VoDev, HI_FALSE);
    }

    /* enable vo interface (kernel process)                         */
    HAL_DISP_IntfEnable(VoDev, HI_TRUE);
}

#else

/*
** DATEĴʱݵַĵ16bitֵ޸VOӦĴֵ
** ۲췢ARM11ʵtimerIPĵ16bitᳬ0x38ȱ0x38֮ǰ
** ĴĴҲҪĻ˳Ӽ
*/
static HI_U32 g_u32VouRegBackup[][2] =
{
    {0x20130000, 0},
    {0x20130008, 0},
    {0x2013001C, 0},
    {0x2013002C, 0},
    {0x20130030, 0},
    {0x20130034, 0},
    {0x20130038, 0}
};

static inline HI_VOID VOU_DRV_RegisterBackup(HI_VOID)
{
    HI_S32 i;

    for(i=0; i < (sizeof(g_u32VouRegBackup) >> 3); i++)
    {
        HAL_ReadReg(g_u32VouRegBackup[i][0], &g_u32VouRegBackup[i][1]);
    }
}

static inline HI_VOID VOU_DRV_RegisterRestore(HI_VOID)
{
    HI_S32 i;

    for(i=0; i < (sizeof(g_u32VouRegBackup) >> 3); i++)
    {

        HAL_WriteReg(g_u32VouRegBackup[i][0], g_u32VouRegBackup[i][1]);
    }
}

HI_VOID VOU_DRV_Open(VO_DEV_E VoDev)
{
    HAL_DISP_SYNCINFO_S stSyncInfo;
    VOU_PIXERL_FORMAT_E enPixFmt;

    /* before enable, we disable first and reset device attributes.         */
    HAL_DISP_IntfEnable(VoDev, HI_FALSE);

    /* DATE ÷ǰ棬ΪDATEVOĴĵĸ */
    switch (g_stHalCfg[VoDev].enMuxIntf)
    {
        case VO_INTF_YPBPR :
            /* Fall through */
        case VO_INTF_CVBS :
            /* һЩĴбһϵͳĸ */
            VOU_DRV_RegisterBackup();

            VOU_DRV_DateSetting(VoDev, g_stHalCfg[VoDev].enOutSync);

            VOU_DRV_RegisterRestore();
            break;
        default:
            #if 0
            AE6D04223 3520 VO DATEĴõĴ󣬷ֹVOĴ
            ϣӦ豸رʱҲرDATEDATEзգǲٹرDATE
            HAL_DATE_Enable(VoDev, HI_FALSE);
            #endif
            break;
    }

    HAL_CMB_SetMixerBackGroud(VoDev, g_stHalCfg[VoDev].u32BkGrd);

    /* Timing Setting */
    memcpy(&stSyncInfo, &g_stSyncTiming[g_stHalCfg[VoDev].enOutSync], sizeof(stSyncInfo));

    /* Pin Mux : ܽŸֱӸֵûнλ */
    if (0 == VoDev)
    {
        switch(g_stHalCfg[VoDev].enMuxIntf)
        {
            case VO_INTF_BT1120:
            {
                HI_U32 i, u32Value;
                for (i = 0x200f00b4; i < 0x200f00f8; i+=4)
                {
                    HAL_ReadReg(i, &u32Value);
                    u32Value = 0x1;
                    HAL_WriteReg(i, u32Value);
                }
                break;
            }
            default:
            {
                HI_U32 i, u32Value;
                for (i = 0x200f00b4; i < 0x200f00f8; i+=4)
                {
                    /* protect AD VGA output sync */
                    if (0x200f00ec == i || 0x200f00f0 == i) continue;

                    HAL_ReadReg(i, &u32Value);
                    u32Value &= ~0x1;
                    HAL_WriteReg(i, u32Value);
                }
                break;
            }
        }
    }
    else if (1 == VoDev)
    {
        switch(g_stHalCfg[VoDev].enMuxIntf)
        {
            case VO_INTF_BT656:
            {
                HI_U32 i, u32Value;
                for (i = 0x200f0090; i < 0x200f00b4; i+=4)
                {
                    HAL_ReadReg(i, &u32Value);
                    u32Value = 0x1;
                    HAL_WriteReg(i, u32Value);
                }
                break;
            }
            case VO_INTF_VGA:
            {
                HI_U32 i, u32Value;
                for (i = 0x200f00ec; i < 0x200f00f4; i+=4)
                {
                    HAL_ReadReg(i, &u32Value);
                    u32Value = 0x3;
                    HAL_WriteReg(i, u32Value);
                }
                break;
            }
            default:
            {
                HI_U32 i, u32Value;
                for (i = 0x200f0090; i < 0x200f00b4; i+=4)
                {
                    HAL_ReadReg(i, &u32Value);
                    u32Value &= ~0x1;
                    HAL_WriteReg(i, u32Value);
                }
                break;
            }
        }
    }

    switch(g_stHalCfg[VoDev].enMuxIntf)
    {
        case VO_INTF_BT1120:
            stSyncInfo.bSynm   = HAL_DISP_SYNC_MODE_TIMING;
            stSyncInfo.u8Intfb = HAL_DISP_BIT_WIDTH_2;
            enPixFmt           = VOU_PIXERL_FORMAT_YCBCR422;
            break;
        case VO_INTF_BT656:
            stSyncInfo.bSynm   = HAL_DISP_SYNC_MODE_TIMING;
            stSyncInfo.u8Intfb = HAL_DISP_BIT_WIDTH_1;
            enPixFmt           = VOU_PIXERL_FORMAT_YCBCR422;
            break;
        case VO_INTF_VGA:
            enPixFmt           = VOU_PIXERL_FORMAT_RGB888;
            break;
        case VO_INTF_YPBPR:
            enPixFmt           = VOU_PIXERL_FORMAT_YCBCR444;
            break;
        default:
            enPixFmt = g_stHalCfg[VoDev].enPixelFmt;
            break;
    }

    HAL_DISP_SetMux(VoDev, g_stHalCfg[VoDev].enMuxIntf);
    HAL_DISP_SetIntfDataFmt(VoDev, enPixFmt);
    HAL_DISP_SetIntfSync(VoDev, stSyncInfo);

    /* CLOCK setting */
    if (VO_DEV_HD == VoDev)
    {
        switch (g_stHalCfg[VoDev].enOutSync)
        {
            case VO_OUTPUT_720P60 :         // 74.25MHz
                HAL_WriteReg(VO_SC_PERCTRL4, 0x34400000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c104a);
                break;
            case VO_OUTPUT_1080I50 :        // 74.25MHz
                HAL_WriteReg(VO_SC_PERCTRL4, 0x34400000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c104a);
                break;
            case VO_OUTPUT_1080I60 :        // 74.25MHz
                HAL_WriteReg(VO_SC_PERCTRL4, 0x34400000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c104a);
                break;
            case VO_OUTPUT_1080P25 :        // 74.25MHz
                HAL_WriteReg(VO_SC_PERCTRL4, 0x34400000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c104a);
                break;
            case VO_OUTPUT_1080P30 :        // 74.25MHz
                HAL_WriteReg(VO_SC_PERCTRL4, 0x34400000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c104a);
                break;
            case VO_OUTPUT_800x600_60 :     // 40M
                HAL_WriteReg(VO_SC_PERCTRL4, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c1028);
                break;
            case VO_OUTPUT_1024x768_60 :    // 65M
                HAL_WriteReg(VO_SC_PERCTRL4, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c1041);
                break;
            case VO_OUTPUT_1280x1024_60 :   // 108M
                HAL_WriteReg(VO_SC_PERCTRL4, 0xa2000000);   // 0x9C000000 (another config)
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c1024);   // 0x006C1036
                break;
            case VO_OUTPUT_1366x768_60 :    // 86M
                HAL_WriteReg(VO_SC_PERCTRL4, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c1055);
                break;
            case VO_OUTPUT_1440x900_60 :    // 88M
                HAL_WriteReg(VO_SC_PERCTRL4, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c1058);   //  1440x900@60Hz
                break;
        default:                            // 65M
                HAL_WriteReg(VO_SC_PERCTRL4, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL5, 0x006c1041);
        }

        /* PLLڱƵʱҪȴ0.1msȶʱӡ
            PLLƵֻϵͳSLOWģʽ½С*/
        /* we delay 200 microsecond, udelay unit is micro */
        udelay(200);
    }
    else if (VO_DEV_AD == VoDev && VO_INTF_VGA == g_stHalCfg[VoDev].enMuxIntf)
    {
        /* VO_SD/AD DATE ʱVO ʱӵķƵȿ */
        HI_U32 u32FreqValue;
        HAL_ReadReg(VO_SC_PERCTRL9, &u32FreqValue);
        u32FreqValue = (u32FreqValue & (~0x0000c000));
        HAL_WriteReg(VO_SC_PERCTRL9, u32FreqValue);

        switch (g_stHalCfg[VoDev].enOutSync)
        {
            case VO_OUTPUT_800x600_60 :     // 40M
                HAL_WriteReg(VO_SC_PERCTRL2, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL3, 0x006c1028);
                break;
            case VO_OUTPUT_1024x768_60 :    // 65M
                HAL_WriteReg(VO_SC_PERCTRL2, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL3, 0x006c1041);
                break;
            case VO_OUTPUT_1280x1024_60 :   // 108M
                HAL_WriteReg(VO_SC_PERCTRL2, 0xa2000000);   // 0x9C000000 (another config)
                HAL_WriteReg(VO_SC_PERCTRL3, 0x006c1024);   // 0x006C1036
                break;
            case VO_OUTPUT_1366x768_60 :    // 86M
                HAL_WriteReg(VO_SC_PERCTRL2, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL3, 0x006c1055);
                break;
            case VO_OUTPUT_1440x900_60 :    // 88M
                HAL_WriteReg(VO_SC_PERCTRL2, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL3, 0x006c1058);   //  1440x900@60Hz
                break;
            default:                            // 40M
                HAL_WriteReg(VO_SC_PERCTRL2, 0xa6000000);
                HAL_WriteReg(VO_SC_PERCTRL3, 0x006c1028);
        }
    }
    else
    {
        /* AD & SD clock configure                                          */
        HAL_INTF_DivSel(VoDev, 0x01, VO_SC_PERCTRL9);

    #if 1
        HAL_WriteReg(VO_SC_PERCTRL2, 0xb4000000);       // vpll0 clk=54M(ad & sd)
        HAL_WriteReg(VO_SC_PERCTRL3, 0x006c1036);       // vpll0 clk=54M(ad & sd)
    #else
        HAL_WriteReg(VO_SC_PERCTRL2, 0xa5000000);       // cvbs clk=27M(ad & sd)
        HAL_WriteReg(VO_SC_PERCTRL3, 0x006c102d);       // cvbs clk=27M(ad & sd)
    #endif

        udelay(200);
    }

    /* CLIP Setting */
    switch (g_stHalCfg[VoDev].enMuxIntf)
    {
        case VO_INTF_BT656 :
            /* Fall through */
        case VO_INTF_CVBS :
            /* Fall through */
        case VO_INTF_BT1120 :
            /* Fall through */
        case VO_INTF_YPBPR :
        {
            #if 0   /* should be 10-bit per member */
            HAL_DISP_CLIP_S stClipData = {0x41,0x0,0x40,0x3ac,0x3c0,0x3c0};
            #else
            HAL_DISP_CLIP_S stClipData = {0x40,0x40,0x40,0x3ac,0x3c0,0x3c0};
            #endif

            HAL_DISP_SetIntfClip(VoDev, HI_TRUE, stClipData);
            break;
        }
        default:
        {
            #if 0
            HAL_DISP_CLIP_S stClipData = {0x41,0x0,0x40,0x3ac,0x3c0,0x3c0};
            #else
            HAL_DISP_CLIP_S stClipData = {0x40,0x40,0x40,0x3ac,0x3c0,0x3c0};
            #endif

            HAL_DISP_SetIntfClip(VoDev, HI_FALSE, stClipData);
        }
    }

    /* CSC setting */
    if (VO_INTF_VGA == g_stHalCfg[VoDev].enMuxIntf)
    {
        HAL_DISP_IntfCscEnable(VoDev, HI_TRUE);

        /* for demo special csc coefficient setting */
        #if 0
        HAL_WriteReg(0x20130c20, 0x7eb0180);
        HAL_WriteReg(0x20130c28, 0x129);
        HAL_WriteReg(0x20130c2c, 0x12901cb);
        HAL_WriteReg(0x20130c30, 0x1f781fca);
        HAL_WriteReg(0x20130c34, 0x21d0129);
        HAL_WriteReg(0x20130c38, 0x0);
        #endif

        /* new csc coefficient from A-Gao Asic board  2009-07-28 */
        /* Աȶ=50,Ͷ=50,ɫ=50,=0 */
        if (VO_DEV_HD == VoDev)
        {
            HAL_WriteReg(0x20130c20, 0x7c30180);
            HAL_WriteReg(0x20130c28, 0x129);
            HAL_WriteReg(0x20130c2c, 0x12901cb);
            HAL_WriteReg(0x20130c30, 0x1f781fca);
            HAL_WriteReg(0x20130c34, 0x21d0129);
            HAL_WriteReg(0x20130c38, 0x0);
        }
        else if (VO_DEV_AD == VoDev)            /* AD is VGA */
        {
            HAL_WriteReg(0x20130d20, 0x7c30180);
            HAL_WriteReg(0x20130d28, 0x129);
            HAL_WriteReg(0x20130d2c, 0x12901cb);
            HAL_WriteReg(0x20130d30, 0x1f781fca);
            HAL_WriteReg(0x20130d34, 0x21d0129);
            HAL_WriteReg(0x20130d38, 0x0);
        }
    }
    else if (VO_INTF_CVBS == g_stHalCfg[VoDev].enMuxIntf)
    {
        if (VO_DEV_AD == VoDev)
        {
            #if 0 /* XuCong ֵ */
            HAL_WriteReg(0x20130d20, 0x07c30180);
            HAL_WriteReg(0x20130d24, 0x00410080);
            HAL_WriteReg(0x20130d28, 0x00000100);
            HAL_WriteReg(0x20130d2c, 0x00000000);
            HAL_WriteReg(0x20130d30, 0x0000010f);
            HAL_WriteReg(0x20130d34, 0x00000000);
            HAL_WriteReg(0x20130d38, 0x0000010f);
            #else /* ɫ */
            HAL_WriteReg(0x20130d20, 0x07c30180);
            HAL_WriteReg(0x20130d24, 0x00410080);
            HAL_WriteReg(0x20130d28, 0x00000119);
            HAL_WriteReg(0x20130d2c, 0x00000000);
            HAL_WriteReg(0x20130d30, 0x00000113);
            HAL_WriteReg(0x20130d34, 0x00000000);
            HAL_WriteReg(0x20130d38, 0x00000113);
            #endif
            HAL_DISP_IntfCscEnable(VoDev, HI_TRUE);
        }
        else if (VO_DEV_SD == VoDev)
        {
            /* register value same as AD  */
            HAL_WriteReg(0x20130e20, 0x07c30180);
            HAL_WriteReg(0x20130e24, 0x00410080);
            HAL_WriteReg(0x20130e28, 0x00000119);
            HAL_WriteReg(0x20130e2c, 0x00000000);
            HAL_WriteReg(0x20130e30, 0x00000113);
            HAL_WriteReg(0x20130e34, 0x00000000);
            HAL_WriteReg(0x20130e38, 0x00000113);
            HAL_DISP_IntfCscEnable(VoDev, HI_TRUE);
        }
    }
    else
    {
        HAL_DISP_IntfCscEnable(VoDev, HI_FALSE);
    }

    /* enable vo interface (kernel process)                         */
    HAL_DISP_IntfEnable(VoDev, HI_TRUE);
}

#endif  /* hi3515 */

HI_VOID VOU_DRV_Close(VO_DEV_E VoDev)
{
    #if 0
    AE6D04223 3520 VO DATEĴõĴ󣬷ֹVOĴ
    ϣӦ豸رʱҲرDATEDATEзգǲٹرDATE
    HAL_DATE_Enable(VoDev, HI_FALSE);
    #endif

    HAL_DISP_IntfEnable(VoDev, HI_FALSE);
}

HI_VOID VOU_DRV_SetDevMux(VO_DEV_E VoDev, VO_INTF_TYPE_E enVoMux)
{
    g_stHalCfg[VoDev].enMuxIntf = enVoMux;
}

HI_VOID VOU_DRV_SetDevBgColor(VO_DEV_E VoDev, HI_U32 u32BgColor)
{
    g_stHalCfg[VoDev].u32BkGrd = u32BgColor;
}

HI_VOID VOU_DRV_SetDevNorm(VO_DEV_E VoDev, VO_INTF_SYNC_E enVoOutMode)
{
    g_stHalCfg[VoDev].enOutSync = enVoOutMode;
}

HI_U32 VOU_DRV_GetDevStatus(VO_DEV_E VoDev)
{
    return 0;
}

HI_BOOL VOU_DRV_GetDevEnable(VO_DEV_E VoDev)
{
	return HAL_DISP_GetIntfEnable(VoDev);
}

HI_VOID VOU_DRV_SetDevVTTH(VO_DEV_E VoDev, HI_U16 Threshold)
{
    HAL_DISP_SetVTThd(VoDev, Threshold);
}

HI_VOID VOU_DRV_SetSyncTiming(VO_DEV_E VoDev, HAL_DISP_SYNCINFO_S *pstCfg)
{
    HAL_DISP_SYNCINFO_S stSyncInfo;

    memcpy(&stSyncInfo, pstCfg, sizeof(HAL_DISP_SYNCINFO_S));

    HAL_DISP_SetIntfSync(VoDev, stSyncInfo);
}

HI_BOOL VOU_DRV_IsDevInterlaced(VO_DEV_E VoDev)
{
    return HAL_DISP_GetIntfIoP(VoDev);
}

HI_VOID VOU_DRV_LayerVBindMixer(VO_DEV_E VoDev, VOU_MIXER_E Mixer)
{
    /* we only care video layer bind relation */
    HAL_VIDEO_LayerBindMixer(VoDev, Mixer);
}

HI_VOID VOU_DRV_DateSetting(VOU_LAYER_E Layer, VO_INTF_SYNC_E enOutSync)
{
    U_HDATE_VIDEO_FORMAT VFormat;
    U_HDATE_OUT_CTRL OutCtrl;

    switch ( enOutSync )
    {
        case VO_OUTPUT_PAL :
            HAL_DATE_SetNorm(Layer, HAL_DISP_NORM_PAL);
            return;
        case VO_OUTPUT_NTSC :
            HAL_DATE_SetNorm(Layer, HAL_DISP_NORM_NTSC);
            return;
        case VO_OUTPUT_720P60 :
            VFormat.u32 = 0xa2;     // EIA-770.3/SMPTE 296M
            OutCtrl.u32 = 0x5b0;
            break;
        case VO_OUTPUT_1080I50 :
            VFormat.u32 = 0xa4;     // EIA-770.3/SMPTE 274M
            OutCtrl.u32 = 0x1b0;
            break;
        case VO_OUTPUT_1080I60 :
            VFormat.u32 = 0xa4;     // EIA-770.3/SMPTE 274M
            OutCtrl.u32 = 0x1b0;
            break;
        case VO_OUTPUT_1080P25 :
            VFormat.u32 = 0xa3;     // SMPTE 274M
            OutCtrl.u32 = 0x5b0;
            break;
        case VO_OUTPUT_1080P30 :
            VFormat.u32 = 0xa3;     // SMPTE 274M
            OutCtrl.u32 = 0x5b0;
            break;

        default:
            return;
    }

    HAL_DATE_Enable(Layer, HI_TRUE);
    HAL_DATE_PolaCtrl(Layer);               /* default: 0x3 H/Vͬ  0ͬͷ */
    HAL_DATE_VideoFmt(Layer, VFormat);
    HAL_DATE_OutCtrl(Layer, OutCtrl);
}

/*****************************************************************************
 Family          : Graphics Layer Operation

 Description     : special settings for graphics only

  History
  1.Date         : 2009/3/13
    Author       : x00100808
    Modification : Created function

*****************************************************************************/
HI_S32 VOU_DRV_LayerEnable(GRPHIC_LAYER_E Layer, HI_BOOL Enable)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_EnableLayer(Layer, Enable);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_LayerBindMixer(GRPHIC_LAYER_E Layer, OUT_DEV_E OutDev)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_LayerBindMixer(Layer, OutDev);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_LayerGetIntfSize(GRPHIC_LAYER_E Layer, HI_U32 *pu32Width, HI_U32 *pu32Height)
{
    HAL_MIXER_E enMixer = HAL_MIXER_1;
    HAL_DISP_CHANNEL_E enChan = HAL_DISP_CHANNEL_DHD;

    if (HI_TRUE != HAL_GRAPHIC_LayerGetBindMixer(Layer, &enMixer))
    {
        return HI_FAILURE;
    }

    if (HAL_MIXER_1 == enMixer)
    {
        enChan = HAL_DISP_CHANNEL_DHD;
    }
    else if(HAL_MIXER_2 == enMixer)
    {
        #ifdef hi3515
        enChan = HAL_DISP_CHANNEL_DSD;
        #else
        enChan = HAL_DISP_CHANNEL_DAD;
        #endif
    }
    else if (HAL_MIXER_3 == enMixer)
    {
        enChan = HAL_DISP_CHANNEL_DSD;
    }
    else
    {
        return HI_FAILURE;
    }

    if (HI_TRUE != HAL_DISP_GetIntfSyncSize(enChan, pu32Width, pu32Height))
    {
        return HI_FAILURE;
    }

    return HI_SUCCESS;
}


HI_S32 VOU_DRV_SetLayerAddr(GRPHIC_LAYER_E Layer, HI_U32 u32Addr)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_SetLayerAddr(Layer, u32Addr);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_SetLayerStride(GRPHIC_LAYER_E Layer, HI_U32 u32Stride)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_SetLayerStride(Layer, u32Stride);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_GetLayerStrideAlign(GRPHIC_LAYER_E Layer, HI_U32 *pu32StrideAlign)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_GetLayerStrideAlign(Layer, pu32StrideAlign);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_SetLayerRect(GRPHIC_LAYER_E Layer, RECT_S stRect)
{
    unsigned long flags;
    HAL_RECT_S Rect;

    Rect.s32X = stRect.s32X;
    Rect.s32Y = stRect.s32Y;
    Rect.u32Width = stRect.u32Width;
    Rect.u32Height = stRect.u32Height;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_SetLayerInReso(Layer, stRect.u32Width, stRect.u32Height);
    HAL_GRAPHIC_SetLayerOutReso(Layer, stRect.u32Width, stRect.u32Height);
#if 0
    HAL_GRAPHIC_SetLayerSrcStart(Layer, Rect.s32X, Rect.s32Y);
#else
    HAL_GRAPHIC_SetLayerSrcStart(Layer, 0, 0);
#endif

    #if 1   /* ֵ2봦 */

    if (Rect.s32Y & 0x1)         Rect.s32Y--;
    if (Rect.u32Height & 0x1)    Rect.u32Height--;

    #endif

    HAL_GRAPHIC_SetLayerDispRect(Layer, Rect);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_SetLayerAlpha(GRPHIC_LAYER_E Layer, HI_U32 Alpha)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);


	/*AE6D04063 adjust user setting alpha which range 0~255 into chip range */
	VOU_ALPHA_ADJUST(Alpha);
    HAL_GRAPHIC_SetLayerGalpha(Layer, Alpha);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_SetLayerPixelAlpha(GRPHIC_LAYER_E Layer, HI_BOOL bEnable, HI_U8 u8Alpha0, HI_U8 u8Alpha1)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_SetAlphaMode(Layer, HAL_GRAPHIC_ALPHA_ARANGE_256);
    HAL_GRAPHIC_SetPixelAlphaEnable(Layer, bEnable);
    HAL_GRAPHIC_SetPixelAlpha(Layer, u8Alpha0, u8Alpha1);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_SetLayerGlobalAlpha(GRPHIC_LAYER_E Layer, HI_U32 Alpha)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

	/*AE6D04063 adjust user setting alpha which range 0~255 into chip range */
	VOU_ALPHA_ADJUST(Alpha);
    HAL_GRAPHIC_SetLayerGalpha(Layer, Alpha);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;}


HI_S32 VOU_DRV_SetLayerPriority(GRPHIC_LAYER_E Layer, HI_U32 Priority)
{
    unsigned long flags;

    if (Priority > VOU_MAX_LAYER_PRIORITY)
    {
        return HI_FAILURE;
    }

    VO_LOCK_IRQ(flags);

    HAL_CMB_SetMixerPriority(Layer, Priority);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_SetLayerDataFmt(GRPHIC_LAYER_E Layer, LAYER_DTFMT_E DataFmt)
{
    unsigned long flags;
    HAL_PIXEL_FORMAT_E enPixelFmt;

    switch ( DataFmt )
    {
        case LAYER_DTFMT_KRGB444 :
            enPixelFmt = HAL_PIXEL_FORMAT_RGB_444;
            break;
        case LAYER_DTFMT_KRGB555 :
            enPixelFmt = HAL_PIXEL_FORMAT_RGB_555;
            break;
        case LAYER_DTFMT_RGB565 :
            enPixelFmt = HAL_PIXEL_FORMAT_RGB_565;
            break;
        case LAYER_DTFMT_KRGB888 :
            enPixelFmt = HAL_PIXEL_FORMAT_KRGB_888;
            break;
        case LAYER_DTFMT_ARGB4444 :
            enPixelFmt = HAL_PIXEL_FORMAT_ARGB_4444;
            break;
        case LAYER_DTFMT_ARGB1555 :
            enPixelFmt = HAL_PIXEL_FORMAT_ARGB_1555;

            break;
        case LAYER_DTFMT_ARGB8888 :
            enPixelFmt = HAL_PIXEL_FORMAT_ARGB_8888;
            break;
        default:
            return HI_FAILURE;
    }

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_SetLayerPixelFmt(Layer, enPixelFmt);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_LayerColorKeyEnable(GRPHIC_LAYER_E Layer, HI_BOOL Enable)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_SetColorKeyEnable(Layer, Enable);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_LayerColorKeyMode(GRPHIC_LAYER_E Layer, VOU_COLORKEY_MODE_E Mode)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_SetColorKeyMode(Layer, Mode);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_LayerColorKeyLimit(GRPHIC_LAYER_E Layer, HI_U32 MinRgb, HI_U32 MaxRgb)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_SetColorKeyValue(Layer, MinRgb, MaxRgb);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_LayerPixAlphaEnable(GRPHIC_LAYER_E Layer, HI_BOOL Enable)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_SetPixelAlphaEnable(Layer, Enable);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_SetLayerCSC(GRPHIC_LAYER_E Layer, HI_BOOL Enable, VOU_CSC_MODE_E Mode)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_SetCsc(Layer, Enable, Mode);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_SetLayerBitExtMode(GRPHIC_LAYER_E Layer, HI_BOOL Enable, VOU_BITEXT_MODE_E Mode)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_SetBitExt(Layer, Mode);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}

HI_S32 VOU_DRV_LayerRegUpdate(GRPHIC_LAYER_E Layer)
{
    unsigned long flags;

    VO_LOCK_IRQ(flags);

    HAL_GRAPHIC_SetRegUp(Layer);

    VO_UNLOCK_IRQ(flags);
    return HI_SUCCESS;
}


#ifdef __cplusplus
#if __cplusplus
}
#endif
#endif /* End of #ifdef __cplusplus */

