/*
 * drivers/mtd/maps/e4ve_nor.c
 *
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#include <asm/io.h>


#define WINDOW_ADDR 0x80000000
#define WINDOW_SIZE 0x02000000

static struct mtd_partition serenity_flash_partitions[] = {
        {
                .name = "all",
                .offset = 0x00000000,
                .size = 0x02000000,
        },
        {
                .name = "upgrade",
                .offset = 0x00100000,
                .size = 0x00e80000, /*TODO*/
        },
        {
                .name = "bootflag", /*TODO*/
                .offset = 0x01f80000,
                .size = 0x00080000,
        },
        {
                .name = "master_uboot",
                .offset = 0x00000000,
                .size = 0x00100000,
        },
        {
                .name = "master_kernel",
                .offset = 0x00100000,
                .size = 0x00200000,
        },
        {
                .name = "master_rootfs",
                .offset = 0x00300000,
                .size = 0x00c80000, /*12.5M*/
        }, 
        {
                .name = "etc",
                .offset = 0x00f80000,
                .size = 0x00200000, /*2M*/
        }, 
        {
                .name = "slave_uboot",
                .offset = 0x01180000,
                .size = 0x00100000,
        },
        {
                .name = "slave_kernel",
                .offset = 0x01280000,
                .size = 0x00200000,
        },
        {
                .name = "slave_rootfs",
                .offset = 0x01480000,
                .size = 0x00400000,
        },        
        {
                .name = "master_kernelback",
                .offset = 0x01880000,
                .size = 0x00200000,
        },
        {
                .name = "master_rootfsback",
                .offset = 0x01a80000,
                .size = 0x00500000,
        },
};

struct map_info serenity_flash_map = {
	.name = "HUSS-E4VE",
	.phys = WINDOW_ADDR,
	.size = WINDOW_SIZE,
	.bankwidth = 1,
};


#define NUM_SERENITY_FLASH_PARTITIONS ARRAY_SIZE(serenity_flash_partitions)

static struct mtd_info *serenity_mtd;

int __init init_serenity_flash(void)
{
	int err;

	printk(KERN_NOTICE "HUSS-E4 VE: flash mapping: %x at %x\n",
			WINDOW_SIZE, WINDOW_ADDR);

	serenity_flash_map.virt = ioremap(WINDOW_ADDR, WINDOW_SIZE);

	if (!serenity_flash_map.virt) {
		printk("init_huss_e4ve_flash: failed to ioremap\n");
		return -EIO;
	}
	simple_map_init(&serenity_flash_map);

	printk("init_huss_e4ve_flash: probing\n");
	serenity_mtd = do_map_probe("cfi_probe", &serenity_flash_map);

	printk("init_huss_e4ve_flash: adding partitions\n");

	if (serenity_mtd) {
	    serenity_mtd->owner = THIS_MODULE;
		err = add_mtd_partitions(serenity_mtd,
		    serenity_flash_partitions,
				NUM_SERENITY_FLASH_PARTITIONS);
	        printk("init_huss_e4ve_flash: adding %d partitions\n", NUM_SERENITY_FLASH_PARTITIONS);
		if (err) {
			printk("init_huss_e4ve_flash: add_mtd_partitions failed\n");
			iounmap(serenity_flash_map.virt);
		}
		return err;
	}
        else
        {
                printk("huss-e4 ve: no flash device found\n");
        }

	iounmap(serenity_flash_map.virt);
	return -ENXIO;
}

static void __exit cleanup_serenity_flash(void)
{
	if (serenity_mtd) {
		del_mtd_partitions(serenity_mtd);
		/* moved iounmap after map_destroy - armin */
		map_destroy(serenity_mtd);
		iounmap((void *)serenity_flash_map.virt);
	}
}

module_init(init_serenity_flash);
module_exit(cleanup_serenity_flash);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Honeywell Intergrited ");
MODULE_DESCRIPTION("MTD map driver for the Honeywell reference boards");
