/**
 * system/src/bld/timer.c
 *
 * History:
 *    2005/08/01 - [Charles Chiou] - created file
 *
 * Copyright (C) 2004-2005, Ambarella, Inc.
 *
 * All rights reserved. No Part of this file may be reproduced, stored
 * in a retrieval system, or transmitted, in any form, or by any means,
 * electronic, mechanical, photocopying, recording, or otherwise,
 * without the prior consent of Ambarella, Inc.
 */

#include <asm/types.h>
#include <asm/io.h>

#include <config.h>
#include <asm/arch/ambhw/chip.h>
#include <asm/arch/basedef.h>
#include <asm/arch/ambhw/eth.h>
#include <asm/arch/netdev/tftp.h>
#include <asm/arch/bldnet.h> 


#include <asm/arch/ambhw.h>
#include <asm/arch/netdev/tftp.h>
#include <asm/arch/bldfunc.h>

#define TIMER_CLOCK timer_clock
#define TIMER_MAX_VAL (0xFFFFFFFF)
#define regw32(a,v) writel(v, a)

static unsigned long lastinc;
static unsigned long timestamp;
static unsigned int timer_clock;

#ifdef IRQ
unsigned int timer1_count = 0;
unsigned int timer2_count = 0;
unsigned int timer3_count = 0;
#endif

unsigned long get_timer(unsigned long);

#ifdef IRQ
/**
 * Handler of timer 1 IRQ.
 */
void timer1_handler(void)
{
	timer1_count++;
	vic_ackint(TIMER1_INT_VEC);
}

/**
 * Handler of timer 2 IRQ.
 */
void timer2_handler(void)
{
	timer2_count++;
	vic_ackint(TIMER2_INT_VEC);
}

/**
 * Handler of timer 3 IRQ.
 */
void timer3_handler(void)
{
	timer3_count++;
	vic_ackint(TIMER3_INT_VEC);
}
#endif

/**
 * Initialize the timer to start ticking.
 */
int timer_init(void)
{
//	unsigned int cnt;
	
    #ifdef IRQ
	timer1_count = 0;
	timer2_count = 0;
	timer3_count = 0;
	#endif
	timer_clock = get_apb_bus_freq_hz();

	/* Reset all timers */
	regw32(TIMER_CTR_REG, 0x0);

	/* Setup VIC */
	vic_set_type(TIMER1_INT_VEC, VIRQ_RISING_EDGE);
	vic_enable(TIMER1_INT_VEC);

	vic_set_type(TIMER2_INT_VEC, VIRQ_RISING_EDGE);
	vic_enable(TIMER2_INT_VEC);

	/* Reset timer control register */
	regw32(TIMER_CTR_REG, 0x0);

	/* Setup timer 1: 100 millisecond */
	//cnt = apb_freq / 10;
	regw32(TIMER1_STATUS_REG, 0xffffffff);
	regw32(TIMER1_RELOAD_REG, 0xffffffff);
	regw32(TIMER1_MATCH1_REG, 0x0);
	regw32(TIMER1_MATCH2_REG, 0x0);

	/* Setup timer 2: 1 millisecond */
	//cnt = apb_freq / 1000;
	regw32(TIMER2_STATUS_REG, 0xffffffff);
	regw32(TIMER2_RELOAD_REG, 0xffffffff);
	regw32(TIMER2_MATCH1_REG, 0x0);
	regw32(TIMER2_MATCH2_REG, 0x0);

	/* Setup timer 3: only used for polling */
	regw32(TIMER3_STATUS_REG, 0xffffffff);
	regw32(TIMER3_RELOAD_REG, 0xffffffff);
	regw32(TIMER3_MATCH1_REG, 0x0);
	regw32(TIMER3_MATCH2_REG, 0x0);

	/* Start timer */
	regw32(TIMER_CTR_REG, 0x555);
	return 0;
}

static void timer_enable(int tmr_id)
{
	u32 val;

	val = readl(TIMER_CTR_REG);

	switch (tmr_id) {
	case TIMER1_ID:
		regw32(TIMER_CTR_REG, val | 0x5);
		break;
	case TIMER2_ID:
		regw32(TIMER_CTR_REG, val | 0x50);
		break;
	case TIMER3_ID:
		regw32(TIMER_CTR_REG, val | 0x500);
		break;
	default:
		K_ASSERT(0);
	}
}

static void timer_disable(int tmr_id)
{
	u32 val;

	val = readl(TIMER_CTR_REG);

	switch (tmr_id) {
	case TIMER1_ID:
		regw32(TIMER_CTR_REG, val & (~0xf));
		regw32(TIMER1_STATUS_REG, 0xffffffff);
		regw32(TIMER1_RELOAD_REG, 0xffffffff);
		break;
	case TIMER2_ID:
		regw32(TIMER_CTR_REG, val & (~0xf0));
		regw32(TIMER2_STATUS_REG, 0xffffffff);
		regw32(TIMER2_RELOAD_REG, 0xffffffff);
		break;
	case TIMER3_ID:
		regw32(TIMER_CTR_REG, val & (~0xf00));
		regw32(TIMER3_STATUS_REG, 0xffffffff);
		regw32(TIMER3_RELOAD_REG, 0xffffffff);
		break;
	default:
		K_ASSERT(0);
	}
}

#ifdef IRQ
/**
 * Reset the count.
 */
static void timer_reset_count(int tmr_id)
{
	switch (tmr_id) {
	case TIMER1_ID:
		timer1_count = 0;
		break;
	case TIMER2_ID:
		timer2_count = 0;
		break;
	case TIMER3_ID:
		timer3_count = 0;
		break;
	default:
		K_ASSERT(0);
	}
}

/**
 * Get current count.
 */
static u32 timer_get_count(int tmr_id)
{
	switch (tmr_id) {
	case TIMER1_ID:
		return timer1_count;
	case TIMER2_ID:
		return timer2_count;
	case TIMER3_ID:
		return timer3_count;
	default:
		K_ASSERT(0);
	}
}
#endif

#if 0
static u32 timer_get_tick(int tmr_id)
{
	switch (tmr_id) {
	case TIMER1_ID:
		return readl(TIMER1_STATUS_REG);
	case TIMER2_ID:
		return readl(TIMER2_STATUS_REG);
	case TIMER3_ID:
		return readl(TIMER3_STATUS_REG);
	default:
		K_ASSERT(0);
	}
}
#endif

static unsigned long get_timer_masked(void)
{
	/* current tick value */
	unsigned long now = readl(TIMER3_STATUS_REG) / (TIMER_CLOCK/CONFIG_SYS_HZ);

	if (now <= lastinc)	/* normal mode (non roll) */
		/* move stamp fordward with absoulte diff ticks */
		timestamp += (lastinc - now);
	else	/* we have rollover of incrementer */
		timestamp += (TIMER_MAX_VAL / (TIMER_CLOCK/CONFIG_SYS_HZ)
				- lastinc) + now;
	lastinc = now;

	return timestamp;
}
unsigned long get_timer(unsigned long base)
{
		return get_timer_masked() - base;
}

static inline u32 timer_tick2us(u32 s_tck, u32 e_tck)
{ 
	return (s_tck - e_tck) / (TIMER_CLOCK/1000000);
}

static void timer_dly_us(u32 dly_tim)
{
	u32 cur_tim;
	u32 s_tck, e_tck;

	timer_disable(TIMER1_ID);
	timer_enable(TIMER1_ID);
	s_tck = readl(TIMER1_STATUS_REG);
	while (1) {
		e_tck = readl(TIMER1_STATUS_REG);
		cur_tim = timer_tick2us(s_tck, e_tck);
		if (cur_tim >= dly_tim)
			break;
	}
}

int __udelay(unsigned long delay)
{
    timer_dly_us(delay);
    return 0;
}

void reset_timer(void)
{
    timer_disable(TIMER3_ID);
    timer_enable(TIMER3_ID);
    return;
}
