/**
 * system/src/bld/nand.c
 *
 * Flash controller functions with NAND chips.
 *
 * History:
 *    2005/02/15 - [Charles Chiou] created file
 *    2006/07/26 - [Charles Chiou] converted to DMA descriptor-mode
 *    2007/10/11 - [Charles Chiou] added PBA partition
 *    2008/11/18 - [Charles Chiou] added HAL and SEC partitions
 *
 * Copyright (C) 2004-2005, Ambarella, Inc.
 *
 * All rights reserved. No Part of this file may be reproduced, stored
 * in a retrieval system, or transmitted, in any form, or by any means,
 * electronic, mechanical, photocopying, recording, or otherwise,
 * without the prior consent of Ambarella, Inc.
 */

#define ENABLE_FLASH 1
#undef CONFIG_NAND_NONE

//#define CONFIG_NAND_K9F2G08 1
#define CONFIG_NAND_1DEVICE 

#include <asm/types.h>
#include <asm/io.h>
#include <exports.h>

#include <config.h>
#include <asm/arch/ambhw/chip.h>
#include <asm/arch/basedef.h>
#include <asm/arch/ambhw/eth.h>
#include <asm/arch/netdev/tftp.h>
#include <asm/arch/bldnet.h> 


#include <asm/arch/bldfunc.h>
#include <asm/arch/ambhw.h>

#define __FLDRV_IMPL__
#include <asm/arch/flash/nand_common.h>
#include <asm/arch/fio/ftl_const.h>
#include <asm/arch/fio/firmfl.h>
#include <asm/arch/flash/nanddb.h>

#undef DEBUG

#if (defined(ENABLE_FLASH) && !defined(CONFIG_NAND_NONE))
flnand_t flnand;
#define regw32(addr,v) writel(v, addr)


/**
 * DMA descriptor.
 */
struct fio_dmadesc_s
{
	u32	src_addr;	/**< Source address */
	u32	dst_addr;	/**< Destination address */
	u32	next;		/**< Next descriptor */
	u32	rpt_addr;	/**< Report address */
	u32	xfrcnt;		/**< Transfer count */
	u32	ctrl;		/**< Control */
	u32	rsv0;		/**< Reserved */
	u32	rsv1;		/**< Reserved */
	u32	rpt;		/**< Report */
	u32	rsv2;		/**< Reserved */
	u32	rsv3;		/**< Reserved */
	u32	rsv4;		/**< Reserved */
} __ATTRIB_PACK__ G_fio_dmadesc __GNU_ALIGN(32);


#define PAGE_SIZE_512	512
#define PAGE_SIZE_2K	2048

#define read_status()						\
{								\
	/* Read Status */					\
	regw32(NAND_CMD_REG, AMB_NAND_CMD_READSTATUS);		\
	while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0);	\
								\
	regw32(NAND_INT_REG, 0x0);				\
								\
	status = readl(NAND_STA_REG);				\
								\
	return (status & 0x1) ? -1 : 0;				\
}

static u8 buffer[PAGE_SIZE_2K*64] __attribute__ ((aligned(32)));

#define FWPROG_BUF_SIZE		0x04000		/* 16K */
/*static u8 page_data[FWPROG_BUF_SIZE]
__attribute__ ((aligned(32), section(".bss.noinit")));*/
static u8 check_buf[FWPROG_BUF_SIZE]
__attribute__ ((aligned(32), section(".bss.noinit")));

extern int amb_gpio_config_hw(int gpio);
extern int amb_gpio_direction_output(int gpio, int value);
extern int amb_gpio_set(int gpio);
extern int amb_gpio_clear(int gpio);

int a5s_nand_read_spare(u32 block, u32 page, u32 pages, const u8 *buf);
int a5s_nand_is_bad_block(u32 block);



int a5s_nand_is_busy(void)
{
    int status;
    read_status();
}

/**
 * Calculate address from the (block, page) pair.
 */
static u32 addr_from_block_page(u32 block, u32 page)
{
	u32 rval = ((block * flnand.pages_per_block) + page);

	if (flnand.main_size == PAGE_SIZE_512)
		return (rval << 9);
	else if (flnand.main_size == PAGE_SIZE_2K)
		return (rval << 11);
	return -1;
}

static int a5s_nand_probe_banks(u32 id, const struct nand_db_s *db)
{
	u32 gpio1_afsel_reg, gpio1_mask_reg, gpio1_data_reg;
	u64 addr64;
	u32 addrhi, addrlo;
	u32 probe;
	int i, banks = 0;;

	if (db == NULL)
		goto done;

	gpio1_afsel_reg = readl(GPIO1_AFSEL_REG);
	gpio1_mask_reg  = readl(GPIO1_MASK_REG);
	gpio1_data_reg  = readl(GPIO1_DATA_REG);

	/* Probe for banks */
	regw32(HOST_ENABLE_REG, readl(HOST_ENABLE_REG) | 0x04);
	amb_gpio_config_hw(NAND_CE1);
	amb_gpio_config_hw(NAND_CE2);
	amb_gpio_config_hw(NAND_CE3);

	for (i = 0; i < 4; i++) {
		addr64 = (u64) (db->devinfo.blocks_per_bank *
				db->devinfo.pages_per_block *
				db->devinfo.main_size * i);
		addrhi = (u32) (addr64 >> 32);
		addrlo = (u32) addr64;

		regw32(NAND_CTR_REG, (db->control & 0x0ffffff3) |
		       NAND_CTR_A(addrhi) | NAND_CTR_4BANK | NAND_CTR_I4);

		regw32(NAND_CMD_REG, AMB_NAND_CMD_READID | addrlo);
		while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0);

		regw32(NAND_INT_REG, 0x0);

		probe = readl(NAND_ID_REG);
		if (probe == id) {
			banks++;
		} else {
			break;
		}
	}

	/* Restore gpio settings. */
	regw32(GPIO1_AFSEL_REG, gpio1_afsel_reg);
	regw32(GPIO1_MASK_REG, gpio1_mask_reg);
	regw32(GPIO1_DATA_REG, gpio1_data_reg);

done:
	return banks;
}

/**
 * Initialize NAND parameters.
 */
int a5s_nand_init(void)
{
	u32 id;
	u8 manid, devid, id3, id4;
	int i;
	int match = 0;
	flnand_t *fn = &flnand;
	struct nand_db_s db;
	u32 t0_reg, t1_reg, t2_reg, t3_reg, t4_reg, t5_reg;
	u8 tcls, tals, tcs, tds;
	u8 tclh, talh, tch, tdh;
	u8 twp, twh, twb, trr;
	u8 trp, treh, trb, tceh;
	u8 trdelay, tclr, twhr, tir;
	u8 tww, trhz, tar; 

#if (DMA_SUPPORT_DMA_FIOS == 1)
	vic_set_type(DMA_FIOS_INT_VEC, VIRQ_LEVEL_HIGH);
#endif
	regw32(FIO_DMACTR_REG,
	       (readl(FIO_DMACTR_REG) & 0xcfffffff) | FIO_DMACTR_FL);

	/* Force ReadID with 4-cycles */
	regw32(NAND_CTR_REG, readl(NAND_CTR_REG) | NAND_CTR_I4);

	/* Reset chip */
	regw32(NAND_CMD_REG, AMB_NAND_CMD_RESET);
	while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0);
	regw32(NAND_INT_REG, 0x0);

	/* Read ID with maximun 5 times if id is 0. */
	for (i = 0; i < 5; i++) {
		regw32(NAND_CMD_REG, AMB_NAND_CMD_READID);
		while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0);
		regw32(NAND_INT_REG, 0x0);
		id = readl(NAND_ID_REG);
		if (id)
			break;
	}

	manid = (id >> 24);
	devid = (id >> 16);
	id3 = (id >> 8);
	id4 = id & 0xff;
	
#if 0
    printf("manid:%02x,devid:%02x,id3:%02x,id4:%02x\n",\
        manid,devid,id3,id4);

	if (id == NAND_ID) {
		match = 1;
		goto matched;
	} else if ((id >> 16) == (NAND_ID >> 16)) {
		match = 2;
		goto matched;
	}

#endif

	/* Search the NAND-DB for an exact match */
	for (i = 0; G_nand_db[i] != NULL; i++) {
		if (G_nand_db[i]->manid == manid &&
		    G_nand_db[i]->devid == devid &&
		    G_nand_db[i]->id3 == id3 &&
		    G_nand_db[i]->id4 == id4) {
			match = 3;
			if (id == NAND_ID_K9K8_SERIAL ||
		    	    id == NAND_ID_K9K4_SERIAL) {
		    	    	/* These kinds of NAND have the same ID and */
				/* need to probe the banks to match exactly. */
				goto probe_banks;
			} else {
				goto matched;
			}
		}
	}

	/* Search the NAND-DB for a match on manfid & devid */
	for (i = 0; G_nand_db[i] != NULL; i++) {
		if (G_nand_db[i]->manid == manid &&
		    G_nand_db[i]->devid == devid) {
			match = 4;
			goto matched;
		}
	}

probe_banks:
	if (match == 3) {
		/* Re-match K9K8 and K9K4 serial NAND. */
		match = 0;
	    	fn->banks = a5s_nand_probe_banks(id, G_nand_db[i]);
		/* Search the NAND-DB for an exact match with banks. */
		for (i = 0; G_nand_db[i] != NULL; i++) {
			if (G_nand_db[i]->manid == manid &&
			    G_nand_db[i]->devid == devid &&
			    G_nand_db[i]->id3 == id3 &&
			    G_nand_db[i]->id4 == id4 &&
			    G_nand_db[i]->total_banks == fn->banks) {
				match = 5;
				goto matched;
			}
		}

		/* Search the NAND-DB for an exact match with one bank. */
		for (i = 0; G_nand_db[i] != NULL; i++) {
			if (G_nand_db[i]->manid == manid &&
			    G_nand_db[i]->devid == devid &&
			    G_nand_db[i]->id3 == id3 &&
			    G_nand_db[i]->id4 == id4 &&
			    G_nand_db[i]->total_banks == 1) {
				match = 6;
				goto matched;
			}
		}
	}

matched:
	if (match <= 2) {
		db.name = NAND_DEFAULT_NAME;
		db.devices = NAND_DEVICES;
		db.total_banks = NAND_DEFAULT_TOTAL_BANKS;
		db.control = NAND_DEFAULT_CONTROL;
		db.manid = NAND_DEFAULT_MANID;
		db.id3 = NAND_DEFAULT_ID3;
		db.id4 = NAND_DEFAULT_ID4;

		db.devinfo.main_size = NAND_DEFAULT_MAIN_SIZE;
		db.devinfo.spare_size = NAND_DEFAULT_SPARE_SIZE;
		db.devinfo.page_size = NAND_DEFAULT_PAGE_SIZE;
		db.devinfo.pages_per_block = NAND_DEFAULT_PAGES_PER_BLOCK;
		db.devinfo.blocks_per_plane = NAND_DEFAULT_BLOCKS_PER_PLANE;
		db.devinfo.blocks_per_bank = NAND_DEFAULT_BLOCKS_PER_BANK;
		db.devinfo.planes_per_bank = NAND_DEFAULT_PLANES_PER_BANK;
		db.devinfo.total_blocks = NAND_DEFAULT_TOTAL_BLOCKS;
		db.devinfo.total_zones = NAND_DEFAULT_TOTAL_ZONES;
		db.devinfo.total_planes = NAND_DEFAULT_TOTAL_PLANES;
		db.devinfo.plane_mask = NAND_DEFAULT_PLANE_MASK;
		db.devinfo.plane_map = NAND_DEFAULT_PLANE_MAP;
		db.devinfo.column_cycles = NAND_DEFAULT_COLUMN_CYCLES;
		db.devinfo.page_cycles = NAND_DEFAULT_PAGE_CYCLES;
		db.devinfo.id_cycles = NAND_DEFAULT_ID_CYCLES;
		db.devinfo.chip_width = NAND_DEFAULT_CHIP_WIDTH;
		db.devinfo.chip_size = NAND_DEFAULT_CHIP_SIZE_MB;
		db.devinfo.bus_width = NAND_DEFAULT_BUS_WIDTH;
		db.devinfo.banks = NAND_DEFAULT_BANKS_PER_DEVICE;

		db.timing.tcls = NAND_DEFAULT_TCLS;
		db.timing.tals = NAND_DEFAULT_TALS;
		db.timing.tcs = NAND_DEFAULT_TCS;
		db.timing.tds = NAND_DEFAULT_TDS;
		db.timing.tclh = NAND_DEFAULT_TCLH;
		db.timing.talh = NAND_DEFAULT_TALH;
		db.timing.tch = NAND_DEFAULT_TCH;
		db.timing.tdh = NAND_DEFAULT_TDH;
		db.timing.twp = NAND_DEFAULT_TWP;
		db.timing.twh = NAND_DEFAULT_TWH;
		db.timing.twb = NAND_DEFAULT_TWB;
		db.timing.trr = NAND_DEFAULT_TRR;
		db.timing.trp = NAND_DEFAULT_TRP;
		db.timing.treh = NAND_DEFAULT_TREH;
		db.timing.trb = NAND_DEFAULT_TRB;
		db.timing.tceh = NAND_DEFAULT_TCEH;
		db.timing.trdelay = NAND_DEFAULT_TRDELAY;
		db.timing.tclr = NAND_DEFAULT_TCLR;
		db.timing.twhr = NAND_DEFAULT_TWHR;
		db.timing.tir = NAND_DEFAULT_TIR;
		db.timing.tww = NAND_DEFAULT_TWW;
		db.timing.trhz = NAND_DEFAULT_TRHZ;
		db.timing.tar = NAND_DEFAULT_TAR;
	} else {
		memcpy(&db, G_nand_db[i], sizeof(db));
		db.total_banks = a5s_nand_probe_banks(id, &db);
	}

	tcls	= FLASH_TIMING_MIN(db.timing.tcls);
	tals	= FLASH_TIMING_MIN(db.timing.tals);
	tcs	= FLASH_TIMING_MIN(db.timing.tcs);
	tds	= FLASH_TIMING_MIN(db.timing.tds);
	tclh	= FLASH_TIMING_MIN(db.timing.tclh);
	talh	= FLASH_TIMING_MIN(db.timing.talh);
	tch	= FLASH_TIMING_MIN(db.timing.tch);
	tdh	= FLASH_TIMING_MIN(db.timing.tdh);
	twp	= FLASH_TIMING_MIN(db.timing.twp);
	twh	= FLASH_TIMING_MIN(db.timing.twh);
	twb	= FLASH_TIMING_MAX(db.timing.twb);
	trr	= FLASH_TIMING_MIN(db.timing.trr);
	trp	= FLASH_TIMING_MIN(db.timing.trp);
	treh	= FLASH_TIMING_MIN(db.timing.treh);
	trb	= FLASH_TIMING_MAX(db.timing.trb);
	tceh	= FLASH_TIMING_MAX(db.timing.tceh);
	trdelay = FLASH_TIMING_MAX(db.timing.trdelay);
	tclr	= FLASH_TIMING_MIN(db.timing.tclr);
	twhr	= FLASH_TIMING_MIN(db.timing.twhr);
	tir	= FLASH_TIMING_MIN(db.timing.tir);
	tww	= FLASH_TIMING_MIN(db.timing.tww);
	trhz	= FLASH_TIMING_MAX(db.timing.trhz);
        tar	= FLASH_TIMING_MAX(db.timing.tar);

	t0_reg = NAND_TIM0_TCLS(tcls) |
		 NAND_TIM0_TALS(tals) |
		 NAND_TIM0_TCS(tcs)   |
		 NAND_TIM0_TDS(tds);

	t1_reg = NAND_TIM1_TCLH(tclh) |
		 NAND_TIM1_TALH(talh) |
		 NAND_TIM1_TCH(tch)   |
		 NAND_TIM1_TDH(tdh);

	t2_reg = NAND_TIM2_TWP(twp) |
		 NAND_TIM2_TWH(twh) |
		 NAND_TIM2_TWB(twb) |
		 NAND_TIM2_TRR(trr);

	t3_reg = NAND_TIM3_TRP(trp)   |
		 NAND_TIM3_TREH(treh) |
		 NAND_TIM3_TRB(trb)   |
		 NAND_TIM3_TCEH(tceh);

	t4_reg = NAND_TIM4_TRDELAY(trdelay) |
		 NAND_TIM4_TCLR(tclr) |
		 NAND_TIM4_TWHR(twhr) |
		 NAND_TIM4_TIR(tir);

	t5_reg = NAND_TIM5_TWW(tww) |
		 NAND_TIM5_TRHZ(trhz) |
		 NAND_TIM5_TAR(tar);

	fn->control = db.control;
	fn->timing0 = t0_reg;
	fn->timing1 = t1_reg;
	fn->timing2 = t2_reg;
	fn->timing3 = t3_reg;
	fn->timing4 = t4_reg;
	fn->timing5 = t5_reg;
	fn->main_size = db.devinfo.main_size;
	fn->spare_size = db.devinfo.spare_size;
	fn->blocks_per_bank = db.devinfo.blocks_per_bank;
	fn->pages_per_block = db.devinfo.pages_per_block;
	fn->block_size = fn->main_size * fn->pages_per_block;
	fn->banks = db.total_banks;

    debug("name:%s\n",db.name);
	return 0;
}

#if 0
static int a5s_nand_is_init(void)
{
	if (flnand.banks == 0 || flnand.banks > 4)
		return 0;
	else
		return 1;
}
#endif

/**
 * Reset NAND flash chip(s).
 */
int a5s_nand_reset(void)
{
	u32 ctr;
	u32 tmp;

	/* Setup Flash IO Control Register (exit random mode) */
	ctr = readl(FIO_CTR_REG);
	if (ctr & FIO_CTR_RR) {
		ctr &= ~FIO_CTR_RR;
		ctr |= FIO_CTR_XD;
		regw32(FIO_CTR_REG, ctr);
	}

	/* Clear the FIO DMA Status Register */
	regw32(FIO_DMASTA_REG, 0x0);

	/* Setup FIO DMA Control Register */
	regw32(FIO_DMACTR_REG, FIO_DMACTR_FL | FIO_DMACTR_TS4B);

	/* Enable NAND flash bank(s) */
	if (flnand.banks == 1) {
		flnand.control = flnand.control & (~0xc);
		tmp = readl(HOST_ENABLE_REG);
		regw32(HOST_ENABLE_REG, tmp & ~0x04);
	} else if (flnand.banks == 2) {
		flnand.control = (flnand.control & ~0xc) | NAND_CTR_2BANK;
		tmp = readl(HOST_ENABLE_REG);
		regw32(HOST_ENABLE_REG, tmp | 0x04);
		amb_gpio_config_hw(NAND_CE1);
	} else if (flnand.banks == 3) {
		flnand.control = (flnand.control & ~0xc) | NAND_CTR_4BANK;
		tmp = readl(HOST_ENABLE_REG);
		regw32(HOST_ENABLE_REG, tmp | 0x04);
		amb_gpio_config_hw(NAND_CE1);
		amb_gpio_config_hw(NAND_CE2);
	} else if (flnand.banks == 4) {
		flnand.control = (flnand.control & ~0xc) | NAND_CTR_4BANK;
		tmp = readl(HOST_ENABLE_REG);
		regw32(HOST_ENABLE_REG, tmp | 0x04);
		amb_gpio_config_hw(NAND_CE1);
		amb_gpio_config_hw(NAND_CE2);
		amb_gpio_config_hw(NAND_CE3);
	}

	/* Initialize write protect pin */
	/* Enable write protect */
	amb_gpio_direction_output(FL_WP,0); 

	/* Setup NAND Flash Control Register */
	regw32(NAND_CTR_REG, flnand.control);
	regw32(NAND_INT_REG, 0x0);

	/* Setup flash timing register */
	regw32(NAND_TIM0_REG, flnand.timing0);
	regw32(NAND_TIM1_REG, flnand.timing1);
	regw32(NAND_TIM2_REG, flnand.timing2);
	regw32(NAND_TIM3_REG, flnand.timing3);
	regw32(NAND_TIM4_REG, flnand.timing4);
	regw32(NAND_TIM5_REG, flnand.timing5);

	return 0;
}

/**
 * Read multiple pages from NAND flash with ecc check.
 */
int a5s_nand_read_pages(u32 block, u32 page, u32 pages, const u8 *buf, u32 enable_ecc)
{
    const char *p;
	int i;
	u32 status;
	u32 addr;
	u32 size = 0;
	u32 nand_ctr_reg = 0;

 	/* check parameters */
	if ((page < 0 || page >= flnand.pages_per_block)	||
	    (pages <= 0 || pages > flnand.pages_per_block)	||
	    ((page + pages) > flnand.pages_per_block)		||
	    (buf == NULL)) {
		printf("ERR: parameter error in nand_read_pages()");
		return -1;
	}

	for (i = 0; i < pages; i++)
		size += flnand.main_size;

	/* Setup FIO DMA Control Register */
	regw32(FIO_DMACTR_REG, FIO_DMACTR_FL | FIO_DMACTR_TS4B);

    //memcpy(buffer,buf,size);
    if ((((u32)buf)&0x1f)){
        p = buffer; 
    }else{
        p = buf;
    }
	clean_flush_d_cache((void *) p, size);

	/* Setup Flash Control Register */
#if (NAND_XD_SUPPORT_WAS == 0) && (NAND_SUPPORT_INTLVE == 0)
	nand_ctr_reg = flnand.control;
#elif (NAND_XD_SUPPORT_WAS >= 1) && (NAND_SUPPORT_INTLVE == 0)
	nand_ctr_reg = flnand.control | NAND_CTR_WAS;
#elif (NAND_XD_SUPPORT_WAS >= 1) && (NAND_SUPPORT_INTLVE == 1)
	nand_ctr_reg = flnand.control | NAND_CTR_WAS;

#if defined(NAND_K9K8_INTLVE)
	nand_ctr_reg = flnand.control | NAND_CTR_K9;
#endif

#endif
	if (enable_ecc) {
		/* Setup Flash IO Control Register */
		regw32(FIO_CTR_REG, FIO_CTR_XD | FIO_CTR_RS | FIO_CTR_CO);

		/* Setup Flash Control Register*/
		regw32(NAND_CTR_REG, nand_ctr_reg | NAND_CTR_SE | NAND_CTR_EC_MAIN);
	} else {
		/* NO ECC */
		regw32(FIO_CTR_REG, FIO_CTR_XD);

		regw32(NAND_CTR_REG, nand_ctr_reg);
	}

#if (DMA_SUPPORT_DMA_FIOS == 1)
	/* Setup external DMA engine transfer */
	regw32(DMA_FIOS_CHAN_STA_REG(FIO_DMA_CHAN), 0x0);
	regw32(DMA_FIOS_CHAN_SRC_REG(FIO_DMA_CHAN), FIO_FIFO_BASE);
	regw32(DMA_FIOS_CHAN_DST_REG(FIO_DMA_CHAN), (u32) p);
	regw32(DMA_FIOS_CHAN_CTR_REG(FIO_DMA_CHAN),
	       DMA_CHANX_CTR_EN		|
	       DMA_CHANX_CTR_WM		|
	       DMA_CHANX_CTR_NI		|
	       DMA_NODC_MN_BURST_SIZE	|
	       size);

#else /* DMA_SUPPORT_DMA_FIOS == 1 */

	/* Setup external DMA engine transfer */
	regw32(DMA_CHAN_STA_REG(FIO_DMA_CHAN), 0x0);
	regw32(DMA_CHAN_SRC_REG(FIO_DMA_CHAN), FIO_FIFO_BASE);
	regw32(DMA_CHAN_DST_REG(FIO_DMA_CHAN), (u32) p);
	regw32(DMA_CHAN_CTR_REG(FIO_DMA_CHAN),
	       DMA_CHANX_CTR_EN		|
	       DMA_CHANX_CTR_WM		|
	       DMA_CHANX_CTR_NI		|
	       DMA_NODC_MN_BURST_SIZE	|
	       size);
#endif

	/* Write start address for memory target to */
	/* FIO DMA Address Register. */
	addr = addr_from_block_page(block, page);
	regw32(FIO_DMAADR_REG, addr);

	/* Setup the Flash IO DMA Control Register */
	regw32(FIO_DMACTR_REG,
	       FIO_DMACTR_EN		|
	       FIO_DMACTR_FL		|
	       FIO_MN_BURST_SIZE	|
	       size);

	/* Wait for interrupt for DMA done */
#if (DMA_SUPPORT_DMA_FIOS == 1)
	while ((readl(DMA_FIOS_INT_REG) & DMA_INT_CHAN(FIO_DMA_CHAN)) == 0x0);

	regw32(DMA_FIOS_INT_REG, 0x0);	/* clear */
	regw32(DMA_FIOS_CHAN_STA_REG(FIO_DMA_CHAN), 0);
#else
	while ((readl(DMA_INT_REG) & DMA_INT_CHAN(FIO_DMA_CHAN)) == 0x0);

	regw32(DMA_INT_REG, 0x0);	/* clear */
#endif

	status = readl(FIO_DMASTA_REG);

	regw32(FIO_DMASTA_REG, 0x0);	/* clear */

	if (status & (FIO_DMASTA_RE | FIO_DMASTA_AE))
		return -1;

	/* Wait for interrupt for NAND operation done */
	while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0);

	regw32(NAND_INT_REG, 0x0);
	if ((((u32)buf)&0x1f)){
        memcpy((void *)buf,buffer,size);
    }
    
	return 0;
}

//char testtt[1024*38] = {0,1,2,3,4,5,6};
static int a5s_nand_read(u32 block, u32 page, u32 pages, u8 *buf)
{
	int rval = 0;
	u32 first_blk_pages, blocks, last_blk_pages;
	u32 bad_blks = 0;

	first_blk_pages = flnand.pages_per_block - page;
	if (pages > first_blk_pages) {
		pages -= first_blk_pages;
		blocks = pages / flnand.pages_per_block;
		last_blk_pages = pages % flnand.pages_per_block;
	} else {
		first_blk_pages = pages;
		blocks = 0;
		last_blk_pages = 0;
	}

	if (first_blk_pages) {
		while (a5s_nand_is_bad_block(block)) {
			/* if bad block, find next */
			block++;
			bad_blks++;
		}
		rval = a5s_nand_read_pages(block, page, first_blk_pages, buf, 1);
		if (rval < 0)
			return -1;
		block++;
		buf += first_blk_pages * flnand.main_size;
	}

	while (blocks > 0) {
		while (a5s_nand_is_bad_block(block)) {
			/* if bad block, find next */
			block++;
			bad_blks++;
		}
		rval = a5s_nand_read_pages(block, 0, flnand.pages_per_block, buf,1);
		if (rval < 0)
			return -1;
		block++;
		blocks--;
		buf += flnand.block_size;
	}

	if (last_blk_pages) {
		while (a5s_nand_is_bad_block(block)) {
			/* if bad block, find next */
			block++;
			bad_blks++;
		}
		rval = a5s_nand_read_pages(block, 0, last_blk_pages, buf, 1);
		if (rval < 0)
			return -1;
	}

	return bad_blks;
}

static void a5s_nand_get_offset_adr(u32 *block, u32 *page, u32 pages, u32 bad_blks)
{
	u32 blocks;

	blocks = pages / flnand.pages_per_block;
	pages  = pages % flnand.pages_per_block;

	*block =  *block + blocks;
	*page += pages;

	if (*page >= flnand.pages_per_block) {
		*page -= flnand.pages_per_block;
		*block += 1;
	}

	*block += bad_blks;
}

/**
 * Read data from NAND flash to memory.
 * dst - address in dram.
 * src - address in nand device.
 * len - length to be read from nand.
 * return - length of read data.
 */
int a5s_nand_read_byte(u8 *dst, u32 from, int len)
{
	u32 block, page, pages, pos;
	u32 first_ppage_size, last_ppage_size;
	u32 val;
	int rval = -1;

#if defined(DEBUG)
	printf("nand_read_data( 0x");
	puthex((u32)dst);
	printf(", 0x");
	puthex((u32)src);
	printf(", ");
	putdec(len);
	printf(" )\r\n");
#endif

	/* translate address to block, page, address */
	val =  from;
	block = val / flnand.block_size;
	val  -= block * flnand.block_size;
	page  = val / flnand.main_size;
	pos   = val % flnand.main_size;
	pages = len / flnand.main_size;

	if (pos == 0)
		first_ppage_size = 0;
	else
		first_ppage_size = flnand.main_size - pos;

	if (len >= first_ppage_size) {
		pages = (len - first_ppage_size) / flnand.main_size;

		last_ppage_size = (len - first_ppage_size) % flnand.main_size;
	} else {
		first_ppage_size = len;
		pages = 0;
		last_ppage_size = 0;
	}

	if (len !=
	    (first_ppage_size + pages * flnand.main_size + last_ppage_size)) {
		return -1;
	}

	len = 0;
	if (first_ppage_size) {
		rval = a5s_nand_read(block, page, 1, buffer);
		if (rval < 0)
			return len;

		memcpy(dst, (void *) (buffer + pos), first_ppage_size);
		dst += first_ppage_size;
		len += first_ppage_size;
		a5s_nand_get_offset_adr(&block, &page, 1, rval);
	}

	if (pages > 0) {
		rval = a5s_nand_read(block, page, pages, dst);
		if (rval < 0)
			return len;

		dst += pages * flnand.main_size;
		len += pages * flnand.main_size;
		a5s_nand_get_offset_adr(&block, &page, pages, rval);
	}

	if (last_ppage_size > 0) {
		rval = a5s_nand_read(block, page, 1, buffer);
		if (rval < 0)
			return len;

		memcpy(dst, (void *) buffer, last_ppage_size);
		len += last_ppage_size;
	}

	return len;
}

/**
 * Program a page to NAND flash.
 */
int a5s_nand_prog_pages(u32 block, u32 page, u32 pages, const u8 *buf)
{
    const char *p;
	int i;
	u32 status;
	u32 addr;
	u32 size = 0;

	/* check parameters */
	if ((page < 0 || page >= flnand.pages_per_block)	||
	    (pages <= 0 || pages > flnand.pages_per_block)	||
	    ((page + pages) > flnand.pages_per_block)		||
	    (buf == NULL)) {
		printf("ERR: parameter error in nand_prog_pages()");
		return -1;
	}

    debug("%s,page:%08x,buffer[0]:%02x\n",__func__,page,buffer[0]);
	for (i = 0; i < pages; i++)
		size += flnand.main_size;

	/* Setup FIO DMA Control Register */
	regw32(FIO_DMACTR_REG, FIO_DMACTR_FL | FIO_DMACTR_TS4B);

    if ((((u32)buf)&0x1f)){
        memcpy(buffer,buf,size);
        p = buffer;
    }else{
        p = buf;
    }
    
	clean_d_cache((void *) p, size);

	/* diable write protect */
	amb_gpio_set(FL_WP);

	/* Setup Flash IO Control Register */
	regw32(FIO_CTR_REG, FIO_CTR_RS | FIO_CTR_XD);

	/* Setup Flash Control Register */
	regw32(NAND_CTR_REG, flnand.control | NAND_CTR_SE | NAND_CTR_EG_MAIN);

#if (DMA_SUPPORT_DMA_FIOS == 1)

	/* Setup external DMA engine transfer */
	regw32(DMA_FIOS_CHAN_STA_REG(FIO_DMA_CHAN), 0x0);
	regw32(DMA_FIOS_CHAN_SRC_REG(FIO_DMA_CHAN), (u32) p);
	regw32(DMA_FIOS_CHAN_DST_REG(FIO_DMA_CHAN), FIO_FIFO_BASE);
	regw32(DMA_FIOS_CHAN_CTR_REG(FIO_DMA_CHAN),
	       DMA_CHANX_CTR_EN		|
	       DMA_CHANX_CTR_RM		|
	       DMA_CHANX_CTR_NI		|
	       DMA_NODC_MN_BURST_SIZE	|
	       size);

	/* Write start address for memory target to */
	/* FIO DMA Address Register. */
	addr =  addr_from_block_page(block, page);
	regw32(FIO_DMAADR_REG, addr);

	/* Setup the Flash IO DMA Control Register */
	regw32(FIO_DMACTR_REG,
	       FIO_DMACTR_EN		|
	       FIO_DMACTR_RM		|
	       FIO_DMACTR_FL		|
	       FIO_MN_BURST_SIZE	|
	       size);

	/* Wait for interrupt for NAND operation done and DMA done */
	while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0 ||
	       (readl(DMA_FIOS_INT_REG) & DMA_INT_CHAN(FIO_DMA_CHAN)) == 0x0 ||
	       (readl(DMA_FIOS_CHAN_CTR_REG(FIO_DMA_CHAN)) & DMA_CHANX_CTR_EN) ||
	       (readl(FIO_DMASTA_REG) & FIO_DMASTA_DN) == 0x0 ||
	       (readl(FIO_DMACTR_REG) & FIO_DMACTR_EN));
	status = readl(FIO_DMASTA_REG);

	regw32(NAND_INT_REG, 0x0);
	regw32(FIO_DMASTA_REG, 0x0);
	regw32(DMA_FIOS_INT_REG, 0x0);
	regw32(DMA_FIOS_CHAN_STA_REG(FIO_DMA_CHAN), 0);

#else /* DMA_SUPPORT_DMA_FIOS == 1 */

	/* Setup external DMA engine transfer */
	regw32(DMA_CHAN_STA_REG(FIO_DMA_CHAN), 0x0);
	regw32(DMA_CHAN_SRC_REG(FIO_DMA_CHAN), (u32) p);
	regw32(DMA_CHAN_DST_REG(FIO_DMA_CHAN), FIO_FIFO_BASE);
	regw32(DMA_CHAN_CTR_REG(FIO_DMA_CHAN),
	       DMA_CHANX_CTR_EN		|
	       DMA_CHANX_CTR_RM		|
	       DMA_CHANX_CTR_NI		|
	       DMA_NODC_MN_BURST_SIZE	|
	       size);

	/* Write start address for memory target to */
	/* FIO DMA Address Register. */
	addr =  addr_from_block_page(block, page);
	regw32(FIO_DMAADR_REG, addr);

	/* Setup the Flash IO DMA Control Register */
	regw32(FIO_DMACTR_REG,
	       FIO_DMACTR_EN		|
	       FIO_DMACTR_RM		|
	       FIO_DMACTR_FL		|
	       FIO_MN_BURST_SIZE	|
	       size);

	/* Wait for interrupt for NAND operation done and DMA done */
	while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0 ||
	       (readl(DMA_INT_REG) & DMA_INT_CHAN(FIO_DMA_CHAN)) == 0x0 ||
	       (readl(DMA_CHAN_CTR_REG(FIO_DMA_CHAN)) & DMA_CHANX_CTR_EN) ||
	       (readl(FIO_DMASTA_REG) & FIO_DMASTA_DN) == 0x0 ||
	       (readl(FIO_DMACTR_REG) & FIO_DMACTR_EN));

	status = readl(FIO_DMASTA_REG);

	regw32(NAND_INT_REG, 0x0);
	regw32(FIO_DMASTA_REG, 0x0);
	regw32(DMA_INT_REG, 0x0);

#endif /* DMA_SUPPORT_DMA_FIOS == 1 */

	/* Enable write protect */
	amb_gpio_clear(FL_WP);

	if (status & (FIO_DMASTA_RE | FIO_DMASTA_AE))
		return -1;

	read_status();
}

/**
 * Read spare area from NAND flash.
 */
int a5s_nand_read_spare(u32 block, u32 page, u32 pages, const u8 *buf)
{
	int i;
	u32 status;
	u32 addr, size = 0;

	/* check parameters */
	if ((page < 0 || page >= flnand.pages_per_block)	||
	    (pages <= 0 || pages > flnand.pages_per_block)	||
	    ((page + pages) > flnand.pages_per_block)		||
	    (buf == NULL)) {
		printf("ERR: parameter error in nand_read_spare()");
		return -1;
	}

	for (i = 0; i < pages; i++)
		size += flnand.spare_size;

	/* Setup DMA descriptor */
	G_fio_dmadesc.src_addr = FIO_FIFO_BASE;
	G_fio_dmadesc.dst_addr = (u32) buf;
	G_fio_dmadesc.next= 0x0;
	G_fio_dmadesc.rpt_addr = (u32) &G_fio_dmadesc.rpt;
	G_fio_dmadesc.xfrcnt = size;
	G_fio_dmadesc.ctrl =
		DMA_DESC_WM |
		DMA_DESC_EOC |
		DMA_DESC_NI |
		DMA_DESC_IE |
		DMA_DESC_ST |
		DMA_DESC_SP_BURST_SIZE;
	G_fio_dmadesc.rpt = 0x0;

	_clean_flush_d_cache();

	/* Setup Flash IO Control Register */
	regw32(FIO_CTR_REG, FIO_CTR_RS | FIO_CTR_XD);

	/* Setup Flash Control Register */
	regw32(NAND_CTR_REG, flnand.control | NAND_CTR_SE  | NAND_CTR_SA);

	/* Setup external DMA engine transfer */
#if (DMA_SUPPORT_DMA_FIOS == 1)
	regw32(DMA_FIOS_CHAN_DA_REG(FIO_DMA_CHAN), (u32) &G_fio_dmadesc);
	regw32(DMA_FIOS_CHAN_CTR_REG(FIO_DMA_CHAN),
	       DMA_CHANX_CTR_D |
	       DMA_CHANX_CTR_EN);
#else
	regw32(DMA_CHAN_DA_REG(FIO_DMA_CHAN), (u32) &G_fio_dmadesc);
	regw32(DMA_CHAN_CTR_REG(FIO_DMA_CHAN),
	       DMA_CHANX_CTR_D |
	       DMA_CHANX_CTR_EN);
#endif

	/* Write start address for memory target to */
	/* FIO DMA Address Register. */
	addr = addr_from_block_page(block, page);
	regw32(FIO_DMAADR_REG, addr);

	/* Setup the Flash IO DMA Control Register */
	regw32(FIO_DMACTR_REG,
	       FIO_DMACTR_EN		|
	       FIO_DMACTR_FL		|
	       FIO_SP_BURST_SIZE	|
	       size);

	/* Wait for interrupt for NAND operation done and DMA done */
	do {
		_clean_flush_d_cache();
	} while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0	||
		 readl(FIO_DMASTA_REG) == 0x0			||
		 (G_fio_dmadesc.rpt & DMA_CHANX_STA_DN) == 0x0	||
#if (DMA_SUPPORT_DMA_FIOS == 1)
		 (readl(DMA_FIOS_INT_REG) & DMA_INT_CHAN(FIO_DMA_CHAN)) == 0x0);
#else
		 (readl(DMA_INT_REG) & DMA_INT_CHAN(FIO_DMA_CHAN)) == 0x0);
#endif

	status = readl(FIO_DMASTA_REG);

	regw32(NAND_INT_REG, 0x0);
	regw32(FIO_DMASTA_REG, 0x0);
#if (DMA_SUPPORT_DMA_FIOS == 1)
	regw32(DMA_FIOS_INT_REG, 0x0);
	regw32(DMA_FIOS_CHAN_STA_REG(FIO_DMA_CHAN), 0x0);
#else
	regw32(DMA_INT_REG, 0x0);
	regw32(DMA_CHAN_STA_REG(FIO_DMA_CHAN), 0x0);
#endif

	if (status & (FIO_DMASTA_RE | FIO_DMASTA_AE))
		return -1;

	read_status();
}

/**
 * Program spare area to NAND flash.
 */
int a5s_nand_prog_spare(u32 block, u32 page, u32 pages, const u8 *buf)
{
	int i;
	u32 status;
	u32 addr, size = 0;
	const char *p;

	/* check parameters */
	if ((page < 0 || page >= flnand.pages_per_block)	||
	    (pages <= 0 || pages > flnand.pages_per_block)	||
	    ((page + pages) > flnand.pages_per_block)		||
	    (buf == NULL)) {
		printf("ERR: parameter error in nand_prog_spare()");
		return -1;
	}

        if ((((u32)buf)&0x1f)) {
            memcpy(buffer,buf,size);
            p = buffer;
        } else {
            p = buf;
        }

	for (i = 0; i < pages; i++)
		size += flnand.spare_size;

	/* Setup DMA descriptor */
	G_fio_dmadesc.src_addr = (u32) p;
	G_fio_dmadesc.dst_addr = FIO_FIFO_BASE;
	G_fio_dmadesc.next= 0x0;
	G_fio_dmadesc.rpt_addr = (u32) &G_fio_dmadesc.rpt;
	G_fio_dmadesc.xfrcnt = size;
	G_fio_dmadesc.ctrl =
		DMA_DESC_RM	|
		DMA_DESC_EOC	|
		DMA_DESC_NI	|
		DMA_DESC_IE	|
		DMA_DESC_ST	|
		DMA_DESC_SP_BURST_SIZE;
	G_fio_dmadesc.rpt = 0x0;

	_clean_d_cache();
	//_drain_write_buffer();

	/* Diable write protect */
	amb_gpio_set(FL_WP);

	/* Setup Flash IO Control Register */
	regw32(FIO_CTR_REG, FIO_CTR_RS);

	/* Setup Flash Control Register */
	regw32(NAND_CTR_REG, flnand.control | NAND_CTR_SE  | NAND_CTR_SA);
#if (DMA_SUPPORT_DMA_FIOS == 1)
	/* Setup external DMA engine transfer */
	regw32(DMA_FIOS_CHAN_DA_REG(FIO_DMA_CHAN), (u32) &G_fio_dmadesc);
	regw32(DMA_FIOS_CHAN_CTR_REG(FIO_DMA_CHAN),
	       DMA_CHANX_CTR_D |
	       DMA_CHANX_CTR_EN);
#else
	/* Setup external DMA engine transfer */
	regw32(DMA_CHAN_DA_REG(FIO_DMA_CHAN), (u32) &G_fio_dmadesc);
	regw32(DMA_CHAN_CTR_REG(FIO_DMA_CHAN),
	       DMA_CHANX_CTR_D |
	       DMA_CHANX_CTR_EN);
#endif

	/* Write start address for memory target to */
	/* FIO DMA Address Register. */
	addr =  addr_from_block_page(block, page);
	regw32(FIO_DMAADR_REG, addr);

	/* Setup the Flash IO DMA Control Register */
	regw32(FIO_DMACTR_REG,
	       FIO_DMACTR_EN		|
	       FIO_DMACTR_RM		|
	       FIO_DMACTR_FL		|
	       FIO_SP_BURST_SIZE	|
	       size);

	/* Wait for interrupt for NAND operation done and DMA done */
	do {
		_clean_flush_d_cache();
	} while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0	||
		 readl(FIO_DMASTA_REG) == 0x0			||
		 (G_fio_dmadesc.rpt & DMA_CHANX_STA_DN) == 0x0	||
#if (DMA_SUPPORT_DMA_FIOS == 1)
		 (readl(DMA_FIOS_INT_REG) & DMA_INT_CHAN(FIO_DMA_CHAN)) == 0x0);
#else
		 (readl(DMA_INT_REG) & DMA_INT_CHAN(FIO_DMA_CHAN)) == 0x0);
#endif

	status = readl(FIO_DMASTA_REG);

	regw32(NAND_INT_REG, 0x0);
	regw32(FIO_DMASTA_REG, 0x0);
#if (DMA_SUPPORT_DMA_FIOS == 1)
	regw32(DMA_FIOS_INT_REG, 0x0);
	regw32(DMA_FIOS_CHAN_STA_REG(FIO_DMA_CHAN), 0x0);
#else
	regw32(DMA_INT_REG, 0x0);
	regw32(DMA_CHAN_STA_REG(FIO_DMA_CHAN), 0x0);
#endif

	/* Enable write protect */
	amb_gpio_clear(FL_WP);

	if (status & (FIO_DMASTA_RE | FIO_DMASTA_AE))
		return -1;

	read_status();
}

/**
 * Erase a NAND flash block.
 */
int a5s_nand_erase_block(u32 block)
{
	u32 status;
	u32 addr;

	/* Disable write protect */
	amb_gpio_set(FL_WP);

	/* Setup FIO DMA Control Register */
	regw32(FIO_DMACTR_REG, FIO_DMACTR_FL | FIO_DMACTR_TS4B);

	/* Setup Flash IO Control Register */
	regw32(FIO_CTR_REG, FIO_CTR_XD);

	/* Setup Flash Control Register */
	regw32(NAND_CTR_REG, flnand.control);

	/* Erase block */
	addr = addr_from_block_page(block, 0);
	regw32(NAND_INT_REG, 0x0);
	regw32(NAND_CMD_REG, AMB_NAND_CMD_ERASE | addr);
	while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0);

	status = readl(FIO_DMASTA_REG);

	regw32(NAND_INT_REG, 0x0);
	regw32(FIO_DMASTA_REG, 0x0);

	/* Enable write protect */
	amb_gpio_clear(FL_WP);

	if (status & (FIO_DMASTA_RE | FIO_DMASTA_AE))
		return -1;

	/* Read Status */
	regw32(NAND_CMD_REG, AMB_NAND_CMD_READSTATUS);
	while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0);

	regw32(NAND_INT_REG, 0x0);
	status = readl(NAND_STA_REG);

	if ((status & 0x1)) {
		/* Reset chip */
		regw32(NAND_CMD_REG, AMB_NAND_CMD_RESET);
		while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0);
		regw32(NAND_INT_REG, 0x0);
		return -1;
	} else
		return 0;
}

#if 0
int a5s_nand_update_bb_info(void)
{
#ifdef NAND_BB_PRE_SCAN
	u32 i, blk, lim, bb_offset = 0;
	flnand_t *fn = &flnand;

	/* skip initial bad block */
	for (i = PART_BB_SKIP_START; i < TOTAL_FW_PARTS; i++) {
		if (fn->nblk[i] == 0)
			continue;
		fn->sblk[i] += bb_offset;
		lim = fn->sblk[i] + fn->nblk[i];
		for (blk = fn->sblk[i]; blk < lim; blk ++) {
			if (a5s_nand_is_bad_block(blk) == 0x1){
				fn->nblk[i] ++;
				bb_offset ++;
				lim ++;
			}
		}
	}
#endif
	return 0;
}

/**
 * Mark a bad block.
 */
int a5s_nand_mark_bad_block(u32 block)
{
	int rval = -1, i, j;
	u8 bi;

	for (i = AMB_BB_START_PAGE; i < BAD_BLOCK_PAGES; i++) {
		memset(check_buf, 0xaa, flnand.spare_size);
		if (flnand.main_size == 512) {
			*(check_buf + 5) = AMB_BAD_BLOCK_MARKER;
		} else { 
			for (j = 0; j < 4; j++) {
				*(check_buf + j) =0x55;
			}
		}

		rval = a5s_nand_prog_spare(block, i, 1, check_buf);
		if (rval < 0) {
			printf("mark bad block failed >> "
				"write spare data error.\r\n");
			return rval;
		}

		rval = a5s_nand_read_spare(block, i, 1, check_buf);
		if (rval < 0) {
			printf("mark bad block failed >> "
				"read spare data error.\r\n");
			return rval;
		}

		if (flnand.main_size == 512)
			bi = *(check_buf + 5);
		else
			bi = *check_buf;

		if (bi == 0xff) {
			printf("mark bad block failed >> "
				"verify failed at block %d\r\n",block);
			return -1;
		}
	}

#if defined(NAND_SUPPORT_BBT)
	//nand_update_bbt(block, 0);
#endif

	return 0;
}
#endif

int a5s_nand_is_bad_block(u32 block)
{
	int rval, i;
	u8 bi;

#if (CHIP_REV == A1)
	for (i = 0; i < BAD_BLOCK_PAGES; i++) {
		u8 *sbuf;

		if (flnand.main_size == 512)
			sbuf = check_buf + i * (1 << NAND_SPARE_16_SHF);
		else
			sbuf = check_buf + i * (1 << NAND_SPARE_64_SHF);

		rval = a5s_nand_read_spare(block, 0, 1, sbuf);
		if (rval < 0) {
			printf("check bad block failed >> "
					"read spare data error.\r\n");
			/* Treat as factory bad block */
			return NAND_INITIAL_BAD_BLOCK;
		}
	}
#else
	rval = a5s_nand_read_spare(block, 0, BAD_BLOCK_PAGES, check_buf);
	if (rval < 0) {
		printf("check bad block failed >> "
				"read spare data error.\r\n");
		/* Treat as factory bad block */
		return NAND_INITIAL_BAD_BLOCK;
	}
#endif

	for (i = 0; i < INIT_BAD_BLOCK_PAGES; i++) {
		if (flnand.main_size == 512)
			bi = *(check_buf + i * (1 << NAND_SPARE_16_SHF) + 5);
		else
			bi = *(check_buf + i * (1 << NAND_SPARE_64_SHF));

		if (bi != 0xff)
			break;
	}


	/* Good block */
	if (i == INIT_BAD_BLOCK_PAGES)
		return NAND_GOOD_BLOCK;

	for (i = INIT_BAD_BLOCK_PAGES; i < BAD_BLOCK_PAGES; i++) {
		if (flnand.main_size == 512)
			bi = *(check_buf + i * (1 << NAND_SPARE_16_SHF) + 5);
		else
			bi = *(check_buf + i * (1 << NAND_SPARE_64_SHF));

		if (bi != 0xff)
			break;
	}

	if (i < BAD_BLOCK_PAGES) {
		/* Late developed bad blocks. */
		return NAND_LATE_DEVEL_BAD_BLOCK;
	} else {
		/* Initial invalid blocks. */
		return NAND_INITIAL_BAD_BLOCK;
	}
}

int nand_amb_reset(void)
{
    debug("%s\n",__func__);
    a5s_nand_reset();
    return 0;
}

int nand_amb_read_id(u8 *buffer)
{
    int id;
    
    regw32(NAND_CMD_REG, AMB_NAND_CMD_READID);
	while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0);
	regw32(NAND_INT_REG, 0x0);
	id = readl(NAND_ID_REG);
	if (id){
	    id = be32_to_cpu(id);
	    memcpy(buffer,&id,sizeof(id));
	    return 0;
	}

	return 0;
}

int nand_amb_read_status(void)
{
    debug("%s\n",__func__);

    int status;
 
    regw32(NAND_CMD_REG, AMB_NAND_CMD_READSTATUS);
    while ((readl(NAND_INT_REG) & NAND_INT_DI) == 0x0);
    regw32(NAND_INT_REG, 0x0);
    status = readl(NAND_STA_REG);
    return status;
}

int nand_amb_erase(u32 page_addr)
{
    debug("%s\n",__func__);
    return a5s_nand_erase_block(page_addr/flnand.pages_per_block);
}

static int nand_amb_read_data(u32 page,u8 *buffer, u8 area)
{
    u32 block = page/flnand.pages_per_block;
    page = page%flnand.pages_per_block;
    int ret;
    
    debug("%s\n",__func__);
    switch(area)
    {
        case MAIN_ONLY:
            ret = a5s_nand_read_pages(block, page,1,buffer,0);
        break;

        case MAIN_ECC:
            ret = a5s_nand_read_pages(block, page,1,buffer,1);
        break;

        case SPARE_ECC:
        case SPARE_ONLY:
            ret = a5s_nand_read_spare(block,page,1,buffer);
        break;
        default:
            ret = -1;
    }
    return ret;
}			

int nand_amb_write_data(u32 page,const u8 *buffer, u8 area)
{
    debug("%s,page:%08x,buffer[0]:%02x\n",__func__,page,buffer[0]);
    u32 block = page/flnand.pages_per_block;
    page = page%flnand.pages_per_block;
    int ret;
    
    switch(area)
    {
        case MAIN_ONLY:
        case MAIN_ECC:
            ret = a5s_nand_prog_pages(block, page,1,buffer);
        break;  

        case SPARE_ECC:
        case SPARE_ONLY:
            ret = a5s_nand_prog_spare(block,page,1,buffer);
        break;
        default:
            ret = -1;
    }
    return ret;
}


#else

/**
 * Read data from NAND flash to memory.
 */
int a5s_nand_read_data(u8 *dst, u8 *src, int len)
{
#if defined(DEBUG)
	printf("nand_read_data( 0x");
	puthex(dst);
	printf(", 0x");
	puthex(src);
	printf(", ");
	putdec(len);
	printf(" )\r\n");
#endif

	return -1;
}

#endif

/*-----------------------------------------------------------------------------------------------*/

#include <linux/mtd/nand.h>
#include <nand.h>

#define MAIN_ONLY		0
#define SPARE_ONLY		1
#define MAIN_ECC		2
#define SPARE_ECC		3
#define SPARE_ONLY_BURST	4
#define SPARE_ECC_BURST		5
#define AMBARELLA_NAND_DMA_BUFFER_SIZE	4096


struct ambarella_g_nand_info {
	struct nand_chip			chip;
	struct mtd_info			mtd;
	struct nand_hw_control		controller;

	int				suspend;
	int				dma_irq;
	int				cmd_irq;
	int				wp_gpio;

	u8				*dmabuf;
	int				dma_bufpos;
	u32				dma_status;
	u32				fio_dma_sta;

	/* saved column/page_addr during CMD_SEQIN */
	int				seqin_column;
	int				seqin_page_addr;

	/* Operation parameters for nand controller register */
	int				err_code;
	u32				cmd;
	u32				control_reg;
	u32				addr_hi;
	u32				addr;
	u32				dst;
	u32				len;
	u32				area;
	u32				ecc;

	u32				origin_clk;	/* in Khz */
};

struct ambarella_g_nand_info		*g_nand_info;


/* ==========================================================================*/
static uint8_t amb_nand_read_byte(struct mtd_info *mtd)
{
	uint8_t					*data;
	//struct ambarella_g_nand_info *g_nand_info;
    debug("%s\n",__func__);

	//g_nand_info = (struct ambarella_g_nand_info *)mtd->priv;

	data = g_nand_info->dmabuf + g_nand_info->dma_bufpos;
	g_nand_info->dma_bufpos++;
    debug("%02x\n",*data);

	return *data;
}

static u16 amb_nand_read_word(struct mtd_info *mtd)
{
	//struct ambarella_g_nand_info		*g_nand_info;
	u16					*data;
    printf("%s\n",__func__);

	//g_nand_info = (struct ambarella_g_nand_info *)mtd->priv;

	data = (u16 *)(g_nand_info->dmabuf + g_nand_info->dma_bufpos);
	g_nand_info->dma_bufpos += 2;

	return *data;
}

static void amb_nand_read_buf(struct mtd_info *mtd, uint8_t *buf, int len)
{
	//struct ambarella_g_nand_info		*g_nand_info;
    debug("%s\n",__func__);

	//g_nand_info = (struct ambarella_g_nand_info *)mtd->priv;

	if ((g_nand_info->dma_bufpos >= AMBARELLA_NAND_DMA_BUFFER_SIZE) ||
		((g_nand_info->dma_bufpos + len) > AMBARELLA_NAND_DMA_BUFFER_SIZE))
		return;

	memcpy(buf, g_nand_info->dmabuf + g_nand_info->dma_bufpos, len);
	g_nand_info->dma_bufpos += len;
}

static void amb_nand_write_buf(struct mtd_info *mtd,
	const uint8_t *buf, int len)
{
    debug("%s\n",__func__);

	//g_nand_info = (struct ambarella_g_nand_info *)mtd->priv;

	if ((g_nand_info->dma_bufpos >= AMBARELLA_NAND_DMA_BUFFER_SIZE) ||
		((g_nand_info->dma_bufpos + len) > AMBARELLA_NAND_DMA_BUFFER_SIZE))
		return;

	memcpy(g_nand_info->dmabuf + g_nand_info->dma_bufpos, buf, len);
	g_nand_info->dma_bufpos += len;
}

#if 0
static int amb_nand_verify_buf(struct mtd_info *mtd,
	const uint8_t *buf, int len)
{
	//struct ambarella_g_nand_info		*g_nand_info;
    printf("%s\n",__func__);

	printf("%s: We don't implement the verify function\n", __func__);

	return 0;
}

static void amb_nand_cmd_ctrl(struct mtd_info *mtd, int dat, unsigned int ctrl)
{
    printf("%s\n",__func__);

}
#endif

static void amb_nand_select_chip(struct mtd_info *mtd, int chip)
{
	//struct ambarella_g_nand_info		*g_nand_info;
    debug("%s\n",__func__);

	//g_nand_info = (struct ambarella_g_nand_info *)mtd->priv;

	if (chip > 0) {
		printf("%s: Multi-Chip isn't supported yet.\n", __func__);
	}
}

static int amb_nand_dev_ready(struct mtd_info *mtd)
{
	struct nand_chip 			*chip = mtd->priv;
    printf("%s\n",__func__);

	chip->cmdfunc(mtd, NAND_CMD_STATUS, -1, -1);

	return (chip->read_byte(mtd) & NAND_STATUS_READY) ? 1 : 0;
}

static int amb_nand_waitfunc(struct mtd_info *mtd, struct nand_chip *chip)
{
	int					status = 0;
	
	status = nand_amb_read_status();

	return status | 0x80;
}

static void amb_nand_cmdfunc(struct mtd_info *mtd, unsigned command,
	int column, int page_addr)
{
	//struct ambarella_g_nand_info		*g_nand_info;
    debug("%s,cmd:%02x\n",__func__,command);

	//g_nand_info = (struct ambarella_g_nand_info *)mtd->priv;
	g_nand_info->err_code = 0;
	int status;

	switch(command) {
	case NAND_CMD_RESET:
		nand_amb_reset();
		break;
	case NAND_CMD_READID:
		g_nand_info->dma_bufpos = 0;
		nand_amb_read_id(g_nand_info->dmabuf);
		break;
	case NAND_CMD_STATUS:
		g_nand_info->dma_bufpos = 0;
		status = nand_amb_read_status();
		//memcpy(g_nand_info->dmabuf,&status,sizeof(status));
		g_nand_info->dmabuf[0] = (status&0xff)| NAND_STATUS_WP;
		break;
	case NAND_CMD_ERASE1:
    	debug("%s\n","NAND_CMD_ERASE1");
		nand_amb_erase(page_addr);
		break;
	case NAND_CMD_ERASE2:
		break;
	case NAND_CMD_READOOB:
	    debug("READOOB\n");
		g_nand_info->dma_bufpos = column;
		nand_amb_read_data(page_addr,g_nand_info->dmabuf, SPARE_ONLY);
		break;
	case NAND_CMD_READ0:
		g_nand_info->dma_bufpos = column;
		nand_amb_read_data(page_addr,
			g_nand_info->dmabuf, MAIN_ECC);
		nand_amb_read_data(page_addr,
			g_nand_info->dmabuf + mtd->writesize, SPARE_ONLY);
		break;
	case NAND_CMD_SEQIN:
		g_nand_info->dma_bufpos = column;
		g_nand_info->seqin_column = column;
		g_nand_info->seqin_page_addr = page_addr;
		debug("NAND_CMD_SEQIN pages:%08x\n",g_nand_info->seqin_page_addr);
		break;
	case NAND_CMD_PAGEPROG:
	    debug("NAND_CMD_PAGEPROG\n");
	    break;
		if (g_nand_info->seqin_column < mtd->writesize) {
			nand_amb_write_data(g_nand_info->seqin_page_addr,
				g_nand_info->dmabuf, MAIN_ECC);
		}
		nand_amb_write_data(g_nand_info->seqin_page_addr,
				g_nand_info->dmabuf + mtd->writesize,
				SPARE_ONLY);
		break;
	default:
		printf("%s: 0x%x, %d, %d\n",__func__, command, column, page_addr);
		break;
	}
}

#if 0
static int amb_nand_caculate_ecc(struct mtd_info *mtd,
	const u_char *dat, u_char *ecc_code)
{
    printf("%s\n",__func__);

	ecc_code[0] = 0xff;
	ecc_code[1] = 0xff;
	ecc_code[2] = 0xff;
	ecc_code[3] = 0xff;
	ecc_code[4] = 0xff;
	ecc_code[5] = 0xff;

	return 0;
}

static int amb_nand_correct_data(struct mtd_info *mtd, u_char *dat,
	u_char *read_ecc, u_char *calc_ecc)
{
	struct ambarella_g_nand_info		*g_nand_info;
    printf("%s\n",__func__);

	g_nand_info = (struct ambarella_g_nand_info *)mtd->priv;
	/*
	 * Any error include DMA error and FIO DMA error, we consider
	 * it as a ecc error which will tell the caller the read fail.
	 * We have distinguish all the errors, but the nand_read_ecc only
	 * check the return value by this function.
	 */
	return g_nand_info->err_code;
}

static int amb_nand_write_oob_std(struct mtd_info *mtd,
	struct nand_chip *chip, int page)
{
	int					i, status;
    printf("%s\n",__func__);

	/* Our nand controller will write the generated ECC code into spare
	  * area automatically, so we should mark the ECC code which located
	  * in the eccpos.
	  */
	for (i = 0; i < chip->ecc.total; i++)
		chip->oob_poi[chip->ecc.layout->eccpos[i]] = 0xFF;

	chip->cmdfunc(mtd, NAND_CMD_SEQIN, mtd->writesize, page);
	chip->write_buf(mtd, chip->oob_poi, mtd->oobsize);
	chip->cmdfunc(mtd, NAND_CMD_PAGEPROG, -1, -1);
	status = chip->waitfunc(mtd, chip);

	return status & NAND_STATUS_FAIL ? -1 : 0;
}

static int amb_nand_block_bad(struct mtd_info *mtd, loff_t ofs, int getchip)
{
    printf("%s\n",__func__);

    return 0;
}

static int amb_nand_block_markbad(struct mtd_info *mtd, loff_t ofs)
{
    printf("%s,ofs:%llx,block:%llx\n",__func__,ofs,ofs/mtd->erasesize);
    a5s_nand_mark_bad_block(ofs/mtd->erasesize);
    return 0;
}
#endif

/* ==========================================================================*/

#if 0
static int amb_nand_read_page_raw(struct mtd_info *mtd,
                     struct nand_chip *chip,
                     uint8_t *buf, int page)
{
    debug("%s mtd->writesize\n",__func__,mtd->writesize);
    nand_amb_read_data(page, buf, MAIN_ONLY); 
    nand_amb_read_data(page,g_nand_info->dmabuf, SPARE_ONLY);
    memcpy(chip->oob_poi,g_nand_info->dmabuf,mtd->oobsize);
    return 0;
}

static void amb_nand_write_page_raw(struct mtd_info *mtd,
                      struct nand_chip *chip,
                      const uint8_t *buf)
{
    debug("%s\n",__func__);
    nand_amb_write_data(g_nand_info->seqin_page_addr, buf, MAIN_ONLY);
    memcpy(g_nand_info->dmabuf,chip->oob_poi,mtd->oobsize);
    nand_amb_write_data(g_nand_info->seqin_page_addr, \
            g_nand_info->dmabuf, SPARE_ONLY);
    return ;
}
#endif

static int amb_nand_read_page(struct mtd_info *mtd,
                     struct nand_chip *chip,
                     uint8_t *buf, int page)
{
    return nand_amb_read_data(page, buf, MAIN_ECC);
}
                     
static void amb_nand_write_page(struct mtd_info *mtd,
                      struct nand_chip *chip,
                      const uint8_t *buf)
{
    nand_amb_write_data(g_nand_info->seqin_page_addr,buf,MAIN_ECC);
    return;                    
}
static int amb_nand_read_oob(struct mtd_info *mtd,
                    struct nand_chip *chip,
                    int page,
                    int sndcmd)
{
   debug("%s\n");
   if ( nand_amb_read_data(page,g_nand_info->dmabuf, SPARE_ONLY) <0){
        return -1;
   }
    memcpy(chip->oob_poi,g_nand_info->dmabuf,mtd->oobsize);
    return 0;
}
 
static int dump_oob(unsigned char *oob)
{
    int i;
    for (i = 0; i< 64; i++){
         if (i&& i%8 == 0)
                printf("\n");
        printf("%02x ",oob[i]); 
    }
    printf("\n\n");
    return 0;
}

static int amb_nand_write_oob(struct mtd_info *mtd,
                     struct nand_chip *chip,
                     int page)
{
    int i;
    debug("%s,ecc.total:%d,mtd->oobsize:%d\n",__func__,chip->ecc.total,mtd->oobsize); 
   //dump_oob(chip->oob_poi);

     /* Our nand controller will write the generated ECC code into spare
	  * area automatically, so we should mark the ECC code which located
	  * in the eccpos.
	  */
    //memset(chip->oob_poi,0xff,mtd->oobsize);	  
    for (i = 0; i < chip->ecc.total; i++){
        chip->oob_poi[chip->ecc.layout->eccpos[i]] = 0xff; 
    }
    memcpy(g_nand_info->dmabuf,chip->oob_poi,mtd->oobsize); 
    nand_amb_write_data(page,g_nand_info->dmabuf, SPARE_ONLY);
    
    memset(g_nand_info->dmabuf,0,mtd->oobsize);
    if (nand_amb_read_data(page,g_nand_info->dmabuf, SPARE_ONLY) < 0){//read back
        printf("read spare err\n");
        return -1;
    }
    for (i = 0; i < chip->ecc.total ; i++){
        chip->oob_poi[chip->ecc.layout->eccpos[i]]       = 0xFF;
        g_nand_info->dmabuf[chip->ecc.layout->eccpos[i]] = 0xFF;
    }
    if (memcmp(chip->oob_poi,g_nand_info->dmabuf,mtd->oobsize)){
        printf("write page[0x%08x].oob err\n",page);
        dump_oob(g_nand_info->dmabuf);
        return -1;
    }
    return 0;
}

/*================================================================================*/

static struct nand_ecclayout amb_oobinfo_512 = {
	.eccbytes = 6,
	.eccpos = {5, 6, 7, 8, 9, 10},
	.oobfree = {{0, 5}, {11, 5}}
};

static struct nand_ecclayout amb_oobinfo_2048 = {
	.eccbytes = 20,
	.eccpos = { 8, 9, 10,11, 12,
		24, 25, 26, 27, 28,
		40, 41, 42, 43, 44,
		56, 57, 58, 59, 60},
	.oobfree = {{0, 7}, {13, 3},
		{16, 7}, {29, 3},
		{36, 7}, {45, 3},
		{48, 7}, {61, 3}}
};


int board_nand_init(struct nand_chip *chip)
{
    if (a5s_nand_init()){
        printf("nand init err\n");
        return -1;
    }

    chip->options		= 0;
    /*
    * If you get more than 1 NAND-chip with different page-sizes on the
    * board one day, it will get more complicated...
    */
    chip->ecc.mode		= NAND_ECC_HW;

    //chip->ecc.hwctl     = amb_nand_enable_hwecc;
    //chip->ecc.calculate = amb_nand_calculate_ecc;
    //chip->ecc.correct   = amb_nand_correct_data;
    if (flnand.main_size == 2048){
        chip->ecc.layout = &amb_oobinfo_2048;
        chip->ecc.size		= flnand.main_size;
        chip->ecc.bytes		= amb_oobinfo_2048.eccbytes;
    }

    if (flnand.main_size == 512){
        chip->ecc.layout = &amb_oobinfo_512;
        chip->ecc.size		= flnand.main_size;
        chip->ecc.bytes		= amb_oobinfo_512.eccbytes;
    }

    //chip->ecc.read_page_raw = amb_nand_read_page_raw;
    //chip->ecc.write_page_raw = amb_nand_write_page_raw;
    chip->ecc.read_page = amb_nand_read_page;
    //chip->ecc.read_subpage = amb_nand_read_subpage;
    chip->ecc.write_page = amb_nand_write_page;
    chip->ecc.read_oob= amb_nand_read_oob;
    chip->ecc.write_oob= amb_nand_write_oob;
    chip->chip_delay = 20;

    //chip->cmd_ctrl		= amb_nand_cmd_ctrl;
    chip->dev_ready		= amb_nand_dev_ready;
    chip->cmdfunc = amb_nand_cmdfunc;
    chip->waitfunc = amb_nand_waitfunc;
    chip->select_chip = amb_nand_select_chip;
    chip->read_byte = amb_nand_read_byte;
    chip->read_word = amb_nand_read_word;
    //chip->block_bad = amb_nand_block_bad;
    //chip->block_markbad = amb_nand_block_markbad;
    chip->write_buf = amb_nand_write_buf;
    chip->read_buf = amb_nand_read_buf;
    //chip->verify_buf = amb_nand_verify_buf;
    g_nand_info = malloc(sizeof(struct ambarella_g_nand_info));
    if (!g_nand_info){
        return -1;
    }
    memset(g_nand_info,0,sizeof(struct ambarella_g_nand_info));
    g_nand_info->dmabuf = malloc(AMBARELLA_NAND_DMA_BUFFER_SIZE + 31);//must alin 32
    if (!g_nand_info->dmabuf){
        return -1;
    }
    g_nand_info->dmabuf = (unsigned char *)((unsigned int)\
                (g_nand_info->dmabuf) & ~(0x1f));
    return 0;
}

