/*  Kernel module help for SH.

    SHcompact version by Kaz Kojima and Paul Mundt.

    SHmedia bits:

	Copyright 2004 SuperH (UK) Ltd
	Author: Richard Curnow

	Based on the sh version, and on code from the sh64-specific parts of
	modutils, originally written by Richard Curnow and Ben Gaster.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <linux/moduleloader.h>
#include <linux/elf.h>
#include <linux/vmalloc.h>
#include <linux/bug.h>
#include <linux/fs.h>
#include <linux/string.h>
#include <linux/kernel.h>
#if defined(CONFIG_MODULES_BPA2)
#include <linux/io.h>
#include <linux/pfn.h>
#include <linux/bpa2.h>
#endif
#include <asm/unaligned.h>

#if defined(CONFIG_MODULES_BPA2)
static struct bpa2_part *modules_bpa2_part;

void *module_alloc(unsigned long size)
{
	unsigned long addr;
	unsigned long n_pages;

	if (unlikely(size == 0))
		return NULL;

	if (unlikely(modules_bpa2_part == NULL))
		 goto v_map;;

	n_pages = (size + PAGE_SIZE - 1) / PAGE_SIZE;

	addr = bpa2_alloc_pages(modules_bpa2_part, n_pages, 1, GFP_KERNEL);

	if (unlikely(!addr))
		 goto v_map;;

	return phys_to_virt(addr);

v_map:
	printk(KERN_WARNING "BPA2 module allocation failed\n");

#ifdef CONFIG_MODULES_BPA2_FALLBACK
	return vmalloc_exec(size);
#else
	return NULL;
#endif
}

void module_free(struct module *mod, void *module_region)
{
	unsigned long addr;

	/* Exit now on NULL address */
	if (unlikely(module_region == NULL))
		return;

	if (likely(modules_bpa2_part) &&
		  (module_region < (void *)VMALLOC_START)) {
		addr = (unsigned long) virt_to_phys(module_region);
		bpa2_free_pages(modules_bpa2_part, addr);
		return;
	}

	vfree(module_region);

	return;
}

static int __init modules_prepare_bpa2(void)
{
	struct bpa2_part *part = bpa2_find_part(CONFIG_MODULES_BPA2_PART_NAME);

	if (!part) {
		printk(KERN_WARNING "BPA2 module allocation: "
			"cannot find BPA2 partition \"%s\"\n",
			CONFIG_MODULES_BPA2_PART_NAME);
		return -1;
	}

	/* We need to check if BPA2 partition is in kernel logical
	 * memory.
	 */
	if (!bpa2_low_part(part)) {
		printk(KERN_WARNING "BPA2 module allocation: "
			"BPA2 partition \"%s\" is not in low memory\n",
			CONFIG_MODULES_BPA2_PART_NAME);
		return -1;
	}

	modules_bpa2_part = part;

	return 0;
}
late_initcall(modules_prepare_bpa2);
#else
void *module_alloc(unsigned long size)
{
	if (size == 0)
		return NULL;
	return vmalloc(size);
}

/* Free memory returned from module_alloc */
void module_free(struct module *mod, void *module_region)
{
	vfree(module_region);
}
#endif


/* We don't need anything special. */
int module_frob_arch_sections(Elf_Ehdr *hdr,
			      Elf_Shdr *sechdrs,
			      char *secstrings,
			      struct module *mod)
{
	return 0;
}

int apply_relocate_add(Elf32_Shdr *sechdrs,
		   const char *strtab,
		   unsigned int symindex,
		   unsigned int relsec,
		   struct module *me)
{
	unsigned int i;
	Elf32_Rela *rel = (void *)sechdrs[relsec].sh_addr;
	Elf32_Sym *sym;
	Elf32_Addr relocation;
	uint32_t *location;
	uint32_t value;

	pr_debug("Applying relocate section %u to %u\n", relsec,
		 sechdrs[relsec].sh_info);
	for (i = 0; i < sechdrs[relsec].sh_size / sizeof(*rel); i++) {
		/* This is where to make the change */
		location = (void *)sechdrs[sechdrs[relsec].sh_info].sh_addr
			+ rel[i].r_offset;
		/* This is the symbol it is referring to.  Note that all
		   undefined symbols have been resolved.  */
		sym = (Elf32_Sym *)sechdrs[symindex].sh_addr
			+ ELF32_R_SYM(rel[i].r_info);
		relocation = sym->st_value + rel[i].r_addend;

#ifdef CONFIG_SUPERH64
		/* For text addresses, bit2 of the st_other field indicates
		 * whether the symbol is SHmedia (1) or SHcompact (0).  If
		 * SHmedia, the LSB of the symbol needs to be asserted
		 * for the CPU to be in SHmedia mode when it starts executing
		 * the branch target. */
		relocation |= !!(sym->st_other & 4);
#endif

		switch (ELF32_R_TYPE(rel[i].r_info)) {
		case R_SH_DIR32:
			value = get_unaligned(location);
			value += relocation;
			put_unaligned(value, location);
			break;
		case R_SH_REL32:
			relocation = (relocation - (Elf32_Addr) location);
			value = get_unaligned(location);
			value += relocation;
			put_unaligned(value, location);
			break;
		case R_SH_IMM_LOW16:
			*location = (*location & ~0x3fffc00) |
				((relocation & 0xffff) << 10);
			break;
		case R_SH_IMM_MEDLOW16:
			*location = (*location & ~0x3fffc00) |
				(((relocation >> 16) & 0xffff) << 10);
			break;
		case R_SH_IMM_LOW16_PCREL:
			relocation -= (Elf32_Addr) location;
			*location = (*location & ~0x3fffc00) |
				((relocation & 0xffff) << 10);
			break;
		case R_SH_IMM_MEDLOW16_PCREL:
			relocation -= (Elf32_Addr) location;
			*location = (*location & ~0x3fffc00) |
				(((relocation >> 16) & 0xffff) << 10);
			break;
		default:
			printk(KERN_ERR "module %s: Unknown relocation: %u\n",
			       me->name, ELF32_R_TYPE(rel[i].r_info));
			return -ENOEXEC;
		}
	}
	return 0;
}

int apply_relocate(Elf32_Shdr *sechdrs,
		       const char *strtab,
		       unsigned int symindex,
		       unsigned int relsec,
		       struct module *me)
{
	printk(KERN_ERR "module %s: REL RELOCATION unsupported\n",
	       me->name);
	return -ENOEXEC;
}

int module_finalize(const Elf_Ehdr *hdr,
		    const Elf_Shdr *sechdrs,
		    struct module *me)
{
	return module_bug_finalize(hdr, sechdrs, me);
}

void module_arch_cleanup(struct module *mod)
{
	module_bug_cleanup(mod);
}
