/*=======================================================================
?Copyright (C) Microsoft Corporation.?All rights reserved.
?
  THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
  KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
  PARTICULAR PURPOSE.
=======================================================================*/

using System;
using System.Diagnostics;
using System.Globalization;
using System.Net;
using System.Net.Sockets;
using System.Runtime.InteropServices;
using System.Threading;
using System.Web;
using System.Collections.Generic;
using System.Collections.ObjectModel;

namespace IPSD.WebServer
{
    public delegate void UserPasswordChangedHandler(string userName);

    internal class Host : MarshalByRefObject
    {
        #region Private Fields

        private string _installPath;
        private string _lowerCasedClientScriptPathWithTrailingSlashV10;
        private string _lowerCasedClientScriptPathWithTrailingSlashV11;
        private string _lowerCasedVirtualPath;
        private string _lowerCasedVirtualPathWithTrailingSlash;
        private EventHandler _onAppDomainUnload;
        private WaitCallback _onSocketAccept;
        private WaitCallback _onStart;
        private string _physicalClientScriptPath;
        private string _physicalPath;
        private int _port;
        private Server _server;
        private Socket _socket;
        private bool _started;
        private bool _stopped;
        private string _virtualPath;
        private int keepAliveInterval;
        private SystemInfo systemInfo;
        public event UserPasswordChangedHandler UserPasswordChangedEvent;
        private int _monitorInteral;
        private int CPUOVERLOAD_THRESHOLD;
        private int MEMOVERLOAD_THRESHOLD;
        private Thread monitorThread = null;

        private bool m_CPUOverloaded = false;
        private bool m_MemOverloaded = false;

        private int m_CpuUsage;
        private int m_MemOccupied;

        #endregion

        #region Properties

        public string InstallPath
        {
            get
            {
                return this._installPath;
            }
        }

        public string NormalizedVirtualPath
        {
            get
            {
                return this._lowerCasedVirtualPathWithTrailingSlash;
            }
        }

        public string PhysicalClientScriptPath
        {
            get
            {
                return this._physicalClientScriptPath;
            }
        }

        public string PhysicalPath
        {
            get
            {
                return this._physicalPath;
            }
        }

        public int Port
        {
            get
            {
                return this._port;
            }
        }

        public string VirtualPath
        {
            get
            {
                return this._virtualPath;
            }
        }

        public int KeepAliveInterval
        {
            get
            {
                return keepAliveInterval;
            }
        }

        public SystemInfo SystemInfo
        {
            get
            {
                return systemInfo;
            }
        }

        /// <summary>
        /// Check whether cpu is overload
        /// </summary>
        public bool IsCPUOverload
        {
            get
            {
                return m_CPUOverloaded;
            }
        }

        /// <summary>
        /// Check whether memory is overload
        /// </summary>
        public bool IsMemoryOverload
        {
            get
            {
                return m_MemOverloaded;
            }
        }

        #endregion

        #region System Methods

        /// <summary>
        /// Configure the web server host
        /// </summary>
        /// <param name="server">Web Server</param>
        /// <param name="port">Port</param>
        /// <param name="virtualPath">virtualPath</param>
        /// <param name="physicalPath">physicalPath</param>
        /// <param name="installPath">installPath</param>
        public void Configure(Server server, int port, string virtualPath, string physicalPath, string installPath,int keepAliveInterval)
        {
            this._server = server;
            this._port = port;
            this._virtualPath = virtualPath;
            this.keepAliveInterval = keepAliveInterval;
            this._monitorInteral = int.Parse(System.Configuration.ConfigurationManager.AppSettings["MonitorInterval"]);
            this.CPUOVERLOAD_THRESHOLD = int.Parse(System.Configuration.ConfigurationManager.AppSettings["CPUOL"]);
            this.MEMOVERLOAD_THRESHOLD = int.Parse(System.Configuration.ConfigurationManager.AppSettings["Memory"]);
            this._lowerCasedVirtualPath = CultureInfo.InvariantCulture.TextInfo.ToLower(this._virtualPath);
            this._lowerCasedVirtualPathWithTrailingSlash = virtualPath.EndsWith("/") ? virtualPath : (virtualPath + "/");
            this._lowerCasedVirtualPathWithTrailingSlash = CultureInfo.InvariantCulture.TextInfo.ToLower(this._lowerCasedVirtualPathWithTrailingSlash);
            this._physicalPath = physicalPath;
            this._installPath = installPath;
            this.systemInfo = SystemInfo.GetInstance();
            this._physicalClientScriptPath = installPath + @"\asp.netclientfiles\";
            string fileVersion = FileVersionInfo.GetVersionInfo(typeof(HttpRuntime).Module.FullyQualifiedName).FileVersion;
            string str2 = fileVersion.Substring(0, fileVersion.LastIndexOf('.'));
            this._lowerCasedClientScriptPathWithTrailingSlashV10 = "/aspnet_client/system_web/" + fileVersion.Replace('.', '_') + "/";
            this._lowerCasedClientScriptPathWithTrailingSlashV11 = "/aspnet_client/system_web/" + str2.Replace('.', '_') + "/";
            this._onSocketAccept = new WaitCallback(this.OnSocketAccept);
            this._onStart = new WaitCallback(this.OnStart);
            this._onAppDomainUnload = new EventHandler(this.OnAppDomainUnload);
            Thread.GetDomain().DomainUnload += this._onAppDomainUnload;

            monitorThread = new Thread(new ThreadStart(RefreshSystemInfo));
        }

        /// <summary>
        /// Initial the lift time service
        /// </summary>
        /// <returns></returns>
        public override object InitializeLifetimeService()
        {
            return null;
        }
        
        /// <summary>
        /// Check whether path is virtual path
        /// </summary>
        /// <param name="path">path</param>
        /// <returns>True/False</returns>
        public bool IsVirtualPathAppPath(string path)
        {
            if (path == null)
            {
                return false;
            }
            path = CultureInfo.InvariantCulture.TextInfo.ToLower(path);
            return ((path == this._lowerCasedVirtualPath) || (path == this._lowerCasedVirtualPathWithTrailingSlash));
        }
         
        /// <summary>
        /// Check whether virtual path is in the application
        /// </summary>
        /// <param name="path">Virtual Path</param>
        /// <returns>True/False</returns>
        public bool IsVirtualPathInApp(string path)
        {
            bool flag;
            string str;
            return this.IsVirtualPathInApp(path, out flag, out str);
        }

        /// <summary>
        /// Check whether virtual path is in the application
        /// </summary>
        /// <param name="path">Virtual Path</param>
        /// <returns>True/False</returns>
        public bool IsVirtualPathInApp(string path, out bool isClientScriptPath, out string clientScript)
        {
            isClientScriptPath = false;
            clientScript = null;
            if (path != null)
            {
                if ((this._virtualPath == "/") && path.StartsWith("/"))
                {
                    if (path.StartsWith(this._lowerCasedClientScriptPathWithTrailingSlashV10))
                    {
                        isClientScriptPath = true;
                        clientScript = path.Substring(this._lowerCasedClientScriptPathWithTrailingSlashV10.Length);
                    }
                    if (path.StartsWith(this._lowerCasedClientScriptPathWithTrailingSlashV11))
                    {
                        isClientScriptPath = true;
                        clientScript = path.Substring(this._lowerCasedClientScriptPathWithTrailingSlashV11.Length);
                    }
                    return true;
                }
                path = CultureInfo.InvariantCulture.TextInfo.ToLower(path);
                if (path.StartsWith(this._lowerCasedVirtualPathWithTrailingSlash))
                {
                    return true;
                }
                if (path == this._lowerCasedVirtualPath)
                {
                    return true;
                }
                if (path.StartsWith(this._lowerCasedClientScriptPathWithTrailingSlashV10))
                {
                    isClientScriptPath = true;
                    clientScript = path.Substring(this._lowerCasedClientScriptPathWithTrailingSlashV10.Length);
                    return true;
                }
                if (path.StartsWith(this._lowerCasedClientScriptPathWithTrailingSlashV11))
                {
                    isClientScriptPath = true;
                    clientScript = path.Substring(this._lowerCasedClientScriptPathWithTrailingSlashV11.Length);
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// The unload event for the application domain
        /// </summary>
        /// <param name="unusedObject">unusedObject</param>
        /// <param name="unusedEventArgs">EventArgs</param>
        private void OnAppDomainUnload(object unusedObject, EventArgs unusedEventArgs)
        {
            Thread.GetDomain().DomainUnload -= this._onAppDomainUnload;
            if (!this._stopped)
            {
                this.Stop();
                //this._server.Restart();
                this._server = null;
            }
        }

        /// <summary>
        /// start the host
        /// </summary>
        /// <param name="unused"></param>
        private void OnStart(object unused)
        {
            Socket state = null;
            while (this._started)
            {
                try
                {
                    state = this._socket.Accept();
                    ThreadPool.QueueUserWorkItem(this._onSocketAccept, state);
                }
                catch(Exception)
                {
                    state = null;
                    Thread.Sleep(100);
                }
            }
            this._stopped = true;
        }

        private void OnSocketAccept(object acceptedSocket)
        {
            Socket socket = (Socket)acceptedSocket;
            if (socket != null)
            {
                try
                {
                    Connection newConnection = new Connection(this, socket);
                    newConnection.ProcessOneRequest();
                    newConnection = null;
                }
                catch (Exception)
                {

                }
                finally
                { 
                  GC.Collect();
                }
            }
        }

        /// <summary>
        /// Start the web server
        /// </summary>
        public void Start()
        {
            if (this._started)
            {
                throw new InvalidOperationException();
            }
            this._socket = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
            this._socket.Bind(new IPEndPoint(IPAddress.Any, this._port));
            this._socket.Listen(0x7fffffff);
            this._started = true;
            ThreadPool.QueueUserWorkItem(this._onStart);
            if (monitorThread != null)
            {
                if (monitorThread.ThreadState != System.Threading.ThreadState.Running)
                {
                    monitorThread.Start();
                }
            }
        }

        /// <summary>
        /// Stop web server
        /// </summary>
        public void Stop()
        {
            if (this._started)
            {
                this._started = false;
                try
                {
                    try
                    {
                        this._socket.Close();
                    }
                    catch
                    {
                    }
                    if (monitorThread != null && monitorThread.ThreadState == System.Threading.ThreadState.Running)
                    {
                        monitorThread.Abort();
                    }
                }
                finally
                {
                    this._socket = null;
                }
                while (!this._stopped)
                {
                    Thread.Sleep(100);
                }
            }
        }

        public void CloseAllAliveConnections(string name)
        {

            if (UserPasswordChangedEvent != null)
            {
                UserPasswordChangedEvent(name.ToLower());
            }
        }

        #endregion

        #region Extenal Request

        /// <summary>
        /// Notify the host client the data changed in the web server
        /// </summary>
        public void DataChangedNotification()
        {
            if (_server != null)
            {
                _server.DataChangedNotification();
            }
        }

        /// <summary>
        /// Send the message to client by HTTPWebRequest
        /// </summary>
        /// <param name="message">Message Body</param>
        /// <returns>Return Value</returns>
        public string SendMessageToClient(string message)
        {
            string result = string.Empty;
            if (_server != null)
            {
                result = _server.SendMessageToClient(message);
            }
            return result;
        }

        /// <summary>
        /// Send psia request
        /// </summary>
        /// <param name="request">PSIARequest</param>
        /// <returns>return xml value</returns>
        public string SendPSIARequest(PSIARequest psiaRequest)
        {
            string result = string.Empty;
            if (_server != null)
            {
                result = _server.SendPSIARequest(psiaRequest);
            }
            return result;
        }

        /// <summary>
        /// Push the data to client from webServer in keep-alive connection
        /// </summary>
        /// <param name="data">Pushed Data</param>
        public void PushDataToClient(string data)
        {
            if (!string.IsNullOrEmpty(data))
            {

            }
        }

        #endregion

        #region Monitor Thread

        /// <summary>
        /// Refresh the system info which contains CPU usage and memory usage
        /// </summary>
        private void RefreshSystemInfo()
        {
            bool alarm = false;
            bool lastAlarm = false;
            string template = "mode=Alarm&type={0}&value={1}";
            string alarmMessage = String.Empty;
            while (_started)
            {
                alarm = false;
                if (this.systemInfo != null)
                {
                    this.systemInfo.Refresh();
                    try
                    {
                        m_CpuUsage = systemInfo.GetCpuUsage();
                        m_CPUOverloaded = (m_CpuUsage >= CPUOVERLOAD_THRESHOLD);

                        m_MemOccupied = systemInfo.GetCurrentProcessMemOccupied();
                        m_MemOverloaded = (m_MemOccupied >= MEMOVERLOAD_THRESHOLD);
                    }
                    catch (Exception ex)
                    {
                        Debug.Assert(false, ex.Message);
                        WebEventLog.GetInstance().WriteException("Host.RefreshSystemInfo error", ex);
                    }
                    if (m_CPUOverloaded)
                    {
                        alarmMessage = string.Format(template, "CPU", "true");
                        alarm = true;
                    }
                    else if (IsMemoryOverload)
                    {
                        alarmMessage = string.Format(template, "Memory", "true");
                        alarm = true;
                    }

                    if (alarm)
                    {
                        SendMessageToClient(alarmMessage);
                    }
                    else
                    {
                        alarmMessage = string.Format(template, "null", "false");
                        SendMessageToClient(alarmMessage);
                    }
                    lastAlarm = alarm;
                }

                Thread.Sleep(this._monitorInteral);
            }
        
        }
    
        #endregion

    }
}
