/*=======================================================================
?Copyright (C) Microsoft Corporation.?All rights reserved.
?
  THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
  KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
  PARTICULAR PURPOSE.
=======================================================================*/

using System;
using System.Collections;
using System.Globalization;
using System.IO;
using System.Text;
using System.Web;
using System.Web.Hosting;
using System.Threading;
using System.Diagnostics;
using System.Collections.Specialized;
using System.Collections.Generic;

namespace IPSD.WebServer
{
    internal class Request : SimpleWorkerRequest
    {
        #region Private Fields

        private StringBuilder _allRawHeaders;
        private Connection _conn;
        private int _contentLength;
        private int _endHeadersOffset;
        private string _filePath;
        private byte[] _headerBytes;
        private List<ByteString>  _headerByteStrings;
        private bool _headersSent;
        private Host _host;
        private string[] _knownRequestHeaders;
        private string _path;
        private string _pathInfo;
        private string _pathTranslated;
        private byte[] _preloadedContent;
        private int _preloadedContentLength;
        private string _prot;
        private string _queryString;
        private byte[] _queryStringBytes;
        private ArrayList _responseBodyBytes;
        private StringBuilder _responseHeadersBuilder;
        private int _responseStatus;
        private bool _specialCaseStaticFileHeaders;
        private int _startHeadersOffset;
        private string[][] _unknownRequestHeaders;
        private string _url;
        private string _verb;
        private const int maxHeaderBytes = 0x8000;
        private static char[] s_badPathChars = new char[] { '%', '>', '<', '$', ':' };
        private static string[] s_defaultFilenames = new string[] { "default.aspx", "default.htm", "default.html", "index.ashx" };
        private const string PSIARequestUrl = "/PSIARequest.ashx";
        private const string ChangePWDURL = "/ChangePWD.ashx";
        private bool isPSIARequest = false;
        private PSIARequest psiaRequest;
        private bool isClose = false;
        private const string FirstAlertResponseHeader = "HTTP/1.1 200 OK \r\n MIME-Version: 1.0 \r\n Content-Type: multipart/mixed; boundary=\"<boundary>\"\r\n";
        private const string AlertResponseHeader = "Content-Type: application/xml; charset=\"UTF-8\" \r\n Content-Length: {0}\r\n";
        private const string HeartBeatResponse = "<?xml version=\"1.0\" encoding=\"UTF-8\" ?> \r\n<EventNotificationAlert version=\"1.0\" xmlns=\"urn:psialliance-org\" >\r\n<ipAddress>%IP%</ipAddress>\r\n<protocol>HTTP</protocol>\r\n<dateTime>%DateTime%</dateTime> \r\n<activePostCount>%Count%</activePostCount> \r\n<eventType>KeepAlive</eventType>\r\n</EventNotificationAlert>\r\n--<boundary>";
        private const string AlertResponse = "<?xml version=\"1.0\" encoding=\"UTF-8\"?> \r\n<EventNotificationAlert version=\"1.0\" xmlns=\"urn:psialliance-org\" > \r\n<ipAddress>%IP%</ipAddress>\r\n<protocol>HTTP</protocol>\r\n<dateTime>%DateTime%</dateTime> \r\n<activePostCount>%Count%</activePostCount> \r\n<eventType>CPUOL</eventType>\r\n<eventState>%active%</eventState>\r\n<eventDescription>CPU overload</eventDescription>\r\n</EventNotificationAlert>\r\n--<boundary>";
        private const string AlertMemoryResponse = "<?xml version=\"1.0\" encoding=\"UTF-8\"?> \r\n<EventNotificationAlert version=\"1.0\" xmlns=\"urn:psialliance-org\" > \r\n<ipAddress>%IP%</ipAddress>\r\n<protocol>HTTP</protocol>\r\n<dateTime>%DateTime%</dateTime> \r\n<activePostCount>%Count%</activePostCount> \r\n<eventType>Memory</eventType>\r\n<eventState>%active%</eventState>\r\n<eventDescription>Memory overload</eventDescription>\r\n</EventNotificationAlert>\r\n--<boundary>";

        /// <summary>
        /// This flag is to record whether there is any changes in web form or winForm.
        /// If it is <c>True</c>: It will send the notification to winForm Client while send the response back to client
        /// </summary>
        private bool dataChangedFlag = false;

        /// <summary>
        /// The client message is got from the request header or post data. 
        /// It is send from Winform client such as HUSClient by HttpWebRequest.
        /// </summary>
        private string inBoundData;

        /// <summary>
        /// The callback message will be added to the response header or psia response.
        /// </summary>
        private string returnResponseResult;
        private string aliveUserName;
        private bool pwdChanged = false;
        private bool m_AuthorizationCheck = false;

        #endregion

        #region Properties

        /// <summary>
        /// Whether is keep-alive request
        /// </summary>
        public bool KeepAlive
        {
            get
            {
                bool result = false;
                string keepAliveValue = this.GetKnownRequestHeader(3);
                if (!string.IsNullOrEmpty(keepAliveValue) && keepAliveValue.Equals("true", StringComparison.OrdinalIgnoreCase))
                {
                    result = true;
                }
                else
                {
                    result = false;
                }
                return result;
            }
        }

        #endregion

        #region Constructor

        public Request(Host host, Connection conn)
            : base(string.Empty, string.Empty, null)
        {
            this._host = host;
            this._conn = conn;
            if (this._host != null)
            {
                this._host.UserPasswordChangedEvent += new UserPasswordChangedHandler(_host_UserPasswordChangedEvent);
                //TODO: _host_UserPasswordChangedEvent need unregistered.
            }
        }

        void _host_UserPasswordChangedEvent(string userName)
        {
            if (!string.IsNullOrEmpty(userName) && !string.IsNullOrEmpty(aliveUserName) &&
                userName.Equals(aliveUserName, StringComparison.OrdinalIgnoreCase))
            {
                pwdChanged = true;
                //if (this._conn != null)
                //{
                //    this._conn.WriteErrorAndClose(400);
                //}
            }
        }

        #endregion

        #region System Methods - Get HttpRequest Info

        public override string GetAppPath()
        {
            return this._host.VirtualPath;
        }

        public override string GetAppPathTranslated()
        {
            return this._host.PhysicalPath;
        }

        public override string GetFilePath()
        {
            return this._filePath;
        }

        public override string GetFilePathTranslated()
        {
            return this._pathTranslated;
        }

        public override string GetHttpVerbName()
        {
            return this._verb;
        }

        public override string GetHttpVersion()
        {
            return this._prot;
        }

        public override string GetKnownRequestHeader(int index)
        {
            if (this._knownRequestHeaders != null)
            {
                return this._knownRequestHeaders[index];
            }
            else
            {
                return null;
            }
        }

        public bool HasKeepAliveHeader()
        {
            bool result = false;
            for (int i = 0; i < this._knownRequestHeaders.Length; i++)
            {
                if (!string.IsNullOrEmpty(this._knownRequestHeaders[i]) &&
                    this._knownRequestHeaders[i].Equals("KeepAlive", StringComparison.OrdinalIgnoreCase))
                {
                    result = true;
                    break;
                }
            }
            return result;
        }

        public override string GetLocalAddress()
        {
            return this._conn.LocalIP;
        }

        public override int GetLocalPort()
        {
            return this._host.Port;
        }

        public override string GetPathInfo()
        {
            return this._pathInfo;
        }

        public override byte[] GetPreloadedEntityBody()
        {
            return this._preloadedContent;
        }

        public override string GetQueryString()
        {
            return this._queryString;
        }

        public override byte[] GetQueryStringRawBytes()
        {
            return this._queryStringBytes;
        }

        public override string GetRawUrl()
        {
            return this._url;
        }

        public override string GetRemoteAddress()
        {
            return this._conn.RemoteIP;
        }

        public override int GetRemotePort()
        {
            return 0;
        }

        public override string GetServerVariable(string name)
        {
            string str = string.Empty;
            string str3 = name;
            if (str3 == null)
            {
                return str;
            }
            if (!(str3 == "ALL_RAW"))
            {
                if (str3 != "SERVER_PROTOCOL")
                {
                    if (str3 != "SERVER_SOFTWARE")
                    {
                        return str;
                    }
                    return ("IPSD - Web Server/" + Messages.VersionString);
                }
            }
            else
            {
                if (this._headerByteStrings.Count > 1)
                {
                    return Encoding.UTF8.GetString(this._headerBytes, this._startHeadersOffset, this._endHeadersOffset - this._startHeadersOffset);
                }
                else
                {
                    return string.Empty;
                }
            }
            return this._prot;
        }

        public override string GetUnknownRequestHeader(string name)
        {
            int length = this._unknownRequestHeaders.Length;
            for (int i = 0; i < length; i++)
            {
                if (string.Compare(name, this._unknownRequestHeaders[i][0], true, CultureInfo.InvariantCulture) == 0)
                {
                    return this._unknownRequestHeaders[i][1];
                }
            }
            return null;
        }

        public override string[][] GetUnknownRequestHeaders()
        {
            return this._unknownRequestHeaders;
        }

        public override string GetUriPath()
        {
            return this._path;
        }

        public override bool HeadersSent()
        {
            return this._headersSent;
        }

        private bool IsBadPath()
        {
            return ((this._path == null) || ((this._path.IndexOfAny(s_badPathChars) >= 0) || (this._path.IndexOf("..") >= 0)));
        }

        public override bool IsClientConnected()
        {
            bool bRet = false;
            if (_conn != null)
            {
                bRet = _conn.Connected;
            }
            return bRet;
        }

        public override bool IsEntireEntityBodyIsPreloaded()
        {
            return (this._contentLength == this._preloadedContentLength);
        }

        public override string MapPath(string path)
        {
            string physicalPath = string.Empty;
            if (((path == null) || (path.Length == 0)) || path.Equals("/"))
            {
                if (this._host.VirtualPath == "/")
                {
                    physicalPath = this._host.PhysicalPath;
                }
                else
                {
                    physicalPath = Environment.SystemDirectory;
                }
            }
            else if (this._host.IsVirtualPathAppPath(path))
            {
                physicalPath = this._host.PhysicalPath;
            }
            else if (this._host.IsVirtualPathInApp(path))
            {
                physicalPath = this._host.PhysicalPath + path.Substring(this._host.NormalizedVirtualPath.Length);
            }
            else if (path.StartsWith("/"))
            {
                physicalPath = this._host.PhysicalPath + path.Substring(1);
            }
            else
            {
                physicalPath = this._host.PhysicalPath + path;
            }
            physicalPath = physicalPath.Replace('/', '\\');
            if (!(!physicalPath.EndsWith(@"\") || physicalPath.EndsWith(@":\")))
            {
                physicalPath = physicalPath.Substring(0, physicalPath.Length - 1);
            }
            return physicalPath;
        }

        #endregion

        #region System Methods - Read/Parse Headers

        /// <summary>
        /// Parse headers
        /// </summary>
        private void ParseHeaders()
        {
            int num;
            this._knownRequestHeaders = new string[40];
            ArrayList list = new ArrayList();
            string str;
            string header;
            string str3;
            string userName;

            for (num = 1; num < this._headerByteStrings.Count; num++)
            {
                str = this._headerByteStrings[num].GetString();
                int index = str.IndexOf(':');
                if (index >= 0)
                {
                    header = str.Substring(0, index).Trim();
                    str3 = str.Substring(index + 1).Trim();
                    int knownRequestHeaderIndex = HttpWorkerRequest.GetKnownRequestHeaderIndex(header);
                    if (knownRequestHeaderIndex >= 0)
                    {
                        this._knownRequestHeaders[knownRequestHeaderIndex] = str3;
                    }
                    else
                    {
                        list.Add(header);
                        list.Add(str3);
                    }
                    if (header.Equals("Authorization", StringComparison.OrdinalIgnoreCase))
                    {
                        ListDictionary listDic = GetHeaderParts(str3);
                        userName = (string)listDic["username"];
                        if (isPSIARequest && psiaRequest != null)
                        {
                            psiaRequest.QueryString = this._queryString;
                            if (string.IsNullOrEmpty(psiaRequest.QueryString))
                            {
                                psiaRequest.QueryString = "user=" + userName;
                            }
                            else
                            {
                                psiaRequest.QueryString = psiaRequest.QueryString + "&user=" + userName;
                            }
                        }
                        else
                        {
                            aliveUserName = userName;
                        }
                    }
                }
            }
            int num4 = list.Count / 2;
            this._unknownRequestHeaders = new string[num4][];
            int num5 = 0;
            for (num = 0; num < num4; num++)
            {
                this._unknownRequestHeaders[num] = new string[] { (string)list[num5++], (string)list[num5++] };
            }
        }

        /// <summary>
        /// Convert the authorization from string to list dictionary
        /// </summary>
        /// <param name="authorization">string</param>
        /// <returns>ListDictionary</returns>
        private ListDictionary GetHeaderParts(string authorization)
        {
            ListDictionary listDic = new ListDictionary();
            string[] parts =
                authorization.Substring(7).Split(new char[] { ',' });
            string[] subParts;
            string key;
            string val;
            foreach (string part in parts)
            {
                subParts = part.Split(new char[] { '=' }, 2);
                key = subParts[0].Trim(new char[] { ' ', '\"' });
                val = subParts[1].Trim(new char[] { ' ', '\"' });
                listDic.Add(key, val);
            }

            return listDic;
        }

        /// <summary>
        /// Prase header URL
        /// </summary>
        /// <param name="url">The specified URL</param>
        private void PraseHeaderUrl(string url)
        {
            if (!string.IsNullOrEmpty(url))
            {
                string[] urlArray = url.Split(new char[] { '.' });
                if (urlArray != null && urlArray.Length > 0)
                {
                    string fileType = urlArray[urlArray.Length - 1];
                    if (fileType.Equals("ashx", StringComparison.OrdinalIgnoreCase))
                    {
                        if (!string.IsNullOrEmpty(inBoundData))
                        {
                            returnResponseResult = _host.SendMessageToClient(inBoundData);
                        }
                    }
                    else
                    {
                        string operation = string.Empty;
                        // People people = new People();
                        string[] data = url.Split(new char[] { '?' });
                        int length = data.Length;
                        if (length > 1)
                        {
                            operation = GetDataValue(data[length - 1]);

                            if (operation.Equals("Add", StringComparison.OrdinalIgnoreCase) ||
                                operation.Equals("Delete", StringComparison.OrdinalIgnoreCase) ||
                                operation.Equals("Update", StringComparison.OrdinalIgnoreCase))
                            {
                                dataChangedFlag = true;
                            }
                        }
                    }
                }
            }
        }

        private bool IsPSIARequest()
        {
            bool isPSIA = false;
            if (this._headerByteStrings != null && this._headerByteStrings.Count > 0)
            {
                ByteString[] strArray = this._headerByteStrings[0].Split(' ');

                if (((strArray != null) && (strArray.Length >= 2)) && (strArray.Length <= 3))
                {
                    this._verb = strArray[0].GetString();
                    ByteString str2 = strArray[1];
                    this._url = str2.GetString();

                    //Check whether this request is PSIA request
                    isPSIA = PSIAHelper.Instance.IsPSIARequest(this._url);
                }
            }
            return isPSIA;
        }

        private void ParseRequestLine()
        {
            ByteString[] strArray = this._headerByteStrings[0].Split(' ');

            if (((strArray != null) && (strArray.Length >= 2)) && (strArray.Length <= 3))
            {
                this._verb = strArray[0].GetString();
                ByteString str2 = strArray[1];
                this._url = str2.GetString();

                m_AuthorizationCheck = (String.Compare(this._url, "/PSIA/System/deviceInfo", true) == 0);

                //Check whether this request is PSIA request
                //isPSIARequest = PSIAHelper.Instance.IsPSIARequest(this._url);
                
                if (this._url.Contains(".aspx%23"))
                {
                    this._url = this._url.Replace(".aspx%23", ".aspx");
                }
                
                if (isPSIARequest)
                {
                    psiaRequest = GeneratePSIARequest(this._url);
                }
                else
                {
                    PraseHeaderUrl(this._url);
                }

                if (strArray.Length == 3)
                {
                    this._prot = strArray[2].GetString();
                }
                else
                {
                    this._prot = "HTTP/1.0";
                }
                int index = this._url.IndexOf('?');
                if (index > 0)
                {
                    this._queryStringBytes = System.Text.Encoding.Default.GetBytes(this._url.Substring(index + 1));
                }
                else
                {
                    this._queryStringBytes = new byte[0];
                }

                if (isPSIARequest)
                {
                    this._url = PSIARequestUrl;
                }
                else if (this._url.Contains("PSIA/Custom/Event/notification/alertStream"))
                {
                    this._url = this._url + "/Default.aspx";
                }
                index = this._url.IndexOf('?');
                if (index > 0)
                {
                    this._path = this._url.Substring(0, index);
                    this._queryString = this._url.Substring(index + 1);
                }
                else
                {
                    this._path = this._url;
                    this._queryStringBytes = new byte[0];
                }

                //Send talk setting to client
                if (this._url.Contains("SendTalkSettings.aspx"))
                {
                    if (!string.IsNullOrEmpty(this._queryString))
                    {
                        string value = this._host.SendMessageToClient(this._queryString);
                    }
                }

                if (this._url.Contains("ChangePWD.aspx"))
                {
                    if (!string.IsNullOrEmpty(this._queryString))
                    {
                        string[] queryArray = this._queryString.Split(new char[] { '=' });
                        if (queryArray.Length > 1)
                        {
                            this._host.CloseAllAliveConnections(queryArray[1]);
                        }
                    }
                }

                if (this._path.IndexOf('%') >= 0)
                {
                    this._path = HttpUtility.UrlDecode(this._path);
                }
                int startIndex = this._path.LastIndexOf('.');
                int num3 = this._path.LastIndexOf('/');
                if (((startIndex >= 0) && (num3 >= 0)) && (startIndex < num3))
                {
                    int length = this._path.IndexOf('/', startIndex);
                    this._filePath = this._path.Substring(0, length);
                    this._pathInfo = this._path.Substring(length);
                }
                else
                {
                    this._filePath = this._path;
                    this._pathInfo = string.Empty;
                }
                this._pathTranslated = this.MapPath(this._filePath);
            }
        }

        private void ReadAllHeaders()
        {
            this._headerBytes = null;

            // Ben: We need to reset the content length and header offset info for starting HTTP request command receiveing.
            this._startHeadersOffset = -1;
            this._endHeadersOffset = -1;
            this._contentLength = -1;

            while (this.TryReadAllHeaders() && (this._endHeadersOffset < 0))
            {
            }
        }

        private bool TryReadAllHeaders()
        {
            bool bRet = TryReadAllHeadersEx();
            isPSIARequest = IsPSIARequest();
            int i = 1;

            // Ben: Comment out below logic. Replace as another implementation.
            //bool hasMoreContent = true;
            //while (isPSIARequest &&
            //    //_headerBytes.Length >= 490 &&
            //    hasMoreContent &&
            //    string.Compare(this._verb, "PUT", true) == 0 &&
            //  (char)(_headerBytes[_headerBytes.Length - 2]) != '>')
            //{
            //    Debug.WriteLine(string.Format("TryReadAllHeadersEx for {0} time ...", i++));
            //    hasMoreContent = TryReadAllHeadersEx();
            //}
          
            // Ben: Here we need to confirm whether the fully HTTP request data has been fully received.
            //      1. Check whether the HTTP header has been received. This can be checked by the _contentLength >= 0?
            //      2. Check whether the content body has been fully received.This can be checked by the _headerBytes.length >= _contentLength + _endHeadersOffset?
            //      3. We cannot make sure when the subsequent request packet can be received. The reason can be that the client send the packet delayed or network is not so reliable.
            //         But we cannot infinitely wait the full HTTP request data. 
            //         So the strategy is that if the next packet can not income in 500ms after a valid packet receiving, 
            //         we can seen the HTTP request is timeout.

            if (isPSIARequest && string.Compare(this._verb, "PUT", true) == 0)
            {
                int nSleepInterval = 10; // Each failed receiving will sleep 5ms for next packet.
                int nSleepCount = 50;  // Total timeout wait count.
                int nCurrentCount = 0;  // Currently wait count for continuous invalid packet.
                while (this._contentLength < 0 || this._headerBytes.Length < this._endHeadersOffset + this._contentLength)
                {
                    Debug.WriteLine(string.Format("TryReadAllHeadersEx for {0} time ...", i++));

                    if (TryReadAllHeadersEx())
                    {
                        nCurrentCount = 0; // Reset the count for valid packet received.
                    }
                    else if (nCurrentCount >= nSleepCount)
                    {
                        // Timeout for continuous invalid packet over 100 times.
                        // Log the timeout issues.
                        Debug.WriteLine("The HTTP request data receive timeout!");
                        return false;
                    }
                    else
                    {
                        Thread.Sleep(nSleepInterval);
                        nCurrentCount++;
                    }
                }

                // Ben: Whether need to initialize the inBoundData?
                // Steven: Initial inBoundData
                ParseInBoundData();

            }

            return bRet;
        }

        private bool TryReadAllHeadersEx()
        {
            bool hasContent = false;

            byte[] src = this._conn.ReadRequestBytes(0x8000);

            if ((src != null && src.Length > 0))
            {
                if (this._headerBytes != null)
                {
                    int num = src.Length + this._headerBytes.Length;
                    if (num <= 0x8000)
                    {
                        hasContent = true;
                        byte[] dst = new byte[num];
                        Buffer.BlockCopy(this._headerBytes, 0, dst, 0, this._headerBytes.Length);
                        Buffer.BlockCopy(src, 0, dst, this._headerBytes.Length, src.Length);
                        this._headerBytes = dst;
                    }
                }
                else
                {
                    hasContent = true;
                    this._headerBytes = src;
                }

                if (hasContent)
                {
#if DEBUG
                    // The new string may consume big memory
                    string s = System.Text.Encoding.Default.GetString(_headerBytes);
                    s = String.Format(" ### {0} : Connection: {1} : headerbytes got in TryReadAllHeaders: {2}", 
                        DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss"), this._conn.GetHashCode().ToString(), s);
                    Debug.WriteLine(s);
#endif

                    // Ben: Check whether the HTTP header has been fully received.
                    if (this._contentLength < 0)
                    {
                        // Ben: Not yet been fully received.
                        //      Try parse the received content to found whether the HTTP header has been fully received.
                        ReadHeaderBytes();
                    }
                }
            }

            return hasContent;
        }

        private void ReadHeaderBytes()
        {
            this._startHeadersOffset = -1;
            this._endHeadersOffset = -1;
            this._headerByteStrings = new List<ByteString>();
            ByteParser parser = new ByteParser(this._headerBytes);
            ByteString str;
            while ((str = parser.ReadLine()) != null)
            {
                if (this._startHeadersOffset < 0)
                {
                    this._startHeadersOffset = parser.CurrentOffset;
                }
                if (str.IsEmpty)
                {
                    this._endHeadersOffset = parser.CurrentOffset;

                    // Ben: Here we have reached the HTTP Header part end.
                    //      So we can parse the HTTP Header content for further HTTP content receive.
                    this.ParseHeaders();
                    this.ParsePostedContent();
                    break;

                    // Ben: We cannot make sure the HTTP body has been fully received.
                    //      So just break the header parser part for further body receive.
                }
                this._headerByteStrings.Add(str);
            }
        }

        /// <summary>
        /// Parse InBoundData from fully header bytes
        /// </summary>
        private void ParseInBoundData()
        {
            ByteParser parser = new ByteParser(this._headerBytes);
            ByteString str;
            while ((str = parser.ReadLine()) != null)
            {
                if (str.IsEmpty)
                {
                    // Steven: HTTP body has been fully received.
                    //      So just parser in bound data
                    str = parser.ReadLine();
                    if (str != null && !str.IsEmpty)
                    {
                        inBoundData = str.GetString();
                    }
                    else
                    {
                        inBoundData = string.Empty;
                    }
                    break;
                }
                this._headerByteStrings.Add(str);
            }
        }

        #endregion

        #region System Methods - Parse Content

        private void ParsePostedContent()
        {
            this._contentLength = 0;
            this._preloadedContentLength = 0;
            string s = this._knownRequestHeaders[11];
            if (s != null)
            {
                try
                {
                    this._contentLength = int.Parse(s);
                }
                catch
                {
                }
            }

            if (this._headerBytes.Length > this._endHeadersOffset)
            {
                this._preloadedContentLength = this._headerBytes.Length - this._endHeadersOffset;
                if ((this._preloadedContentLength > this._contentLength) && (this._contentLength > 0))
                {
                    this._preloadedContentLength = this._contentLength;
                }
                this._preloadedContent = new byte[this._preloadedContentLength];
                Buffer.BlockCopy(this._headerBytes, this._endHeadersOffset, this._preloadedContent, 0, this._preloadedContentLength);
            }

            if (this._preloadedContent != null && this._preloadedContent.Length > 0)
            {
                //Prase the post data if its resquest to ashx file and send the data to host client
                if (!string.IsNullOrEmpty(this._url))
                {
                    string[] urlArray = this._url.Split(new char[] { '.' });
                    if (urlArray != null && urlArray.Length > 0)
                    {
                        string fileType = urlArray[urlArray.Length - 1];
                        if (fileType.Equals("ashx", StringComparison.OrdinalIgnoreCase))
                        {
                            System.Text.UTF8Encoding converter = new System.Text.UTF8Encoding();
                            inBoundData = converter.GetString(this._preloadedContent);

                            //if (!string.IsNullOrEmpty(inBoundData))
                            //{
                            //    returnResponseResult = _host.SendMessageToClient(inBoundData);
                            //}
                        }
                    }
                }
            }


        }

        private void PraseMessage(string message)
        {
            if (!string.IsNullOrEmpty(message))
            {
                if (_host != null)
                {
                    _host.SendMessageToClient(message);
                }
            }
        }

        /// <summary>
        /// Handle PSIA Request
        /// </summary>
        private void HandlePSIARequest()
        {
            if (isPSIARequest)
            {
                if (m_AuthorizationCheck)
                {
                    // returnResponseResult = "<ResourceList version=\"1.0\" xmlns=\"urn:psialliance-org:resourcelist\"><Resource></Resource></ResourceList>";
                    returnResponseResult = "<DeviceInfo version=\"1.0\" xmlns=\"urn:psialliance-org\"><deviceName></deviceName></DeviceInfo>";
                }
                else
                {
                    System.Text.UTF8Encoding converter = new System.Text.UTF8Encoding();

                    if (this._preloadedContent != null && this._preloadedContent.Length > 0)
                    {
                        inBoundData = converter.GetString(this._preloadedContent);
                    }
                    else
                    {
                        inBoundData = string.Empty;
                    }
                    if (psiaRequest != null)
                    {
                        psiaRequest.InBoundData = inBoundData;
                        
                        returnResponseResult = _host.SendPSIARequest(psiaRequest);

                        //StringBuilder sb = new StringBuilder();
                        //sb.Append("\r\nPSIA InBound Data:");
                        //sb.Append(inBoundData);
                        //sb.Append("\r\n Result:\r\n");
                        //sb.Append(returnResponseResult);
                        //WebEventLog.GetInstance().WriteInfomation(sb.ToString());
                    }
                }
            }
        }

        #endregion

        #region System Methods - Send/Flush Response

        public override void EndOfRequest()
        {
            if (this._conn.KeepAlive)
            {
                return;
            }
            else
            {
                base.EndOfRequest();
            }
        }

        public override void FlushResponse(bool finalFlush)
        {
            if (this.isPSIARequest || this.KeepAlive)
            {
                if (this._responseStatus != 200)
                {
                    UTF8Encoding en = new UTF8Encoding();
                    StringBuilder sb = new StringBuilder();
                    for (int i = 0; i < this._responseBodyBytes.Count; i++)
                    {
                        byte[] data = (byte[])this._responseBodyBytes[i];
                        sb.Append(en.GetString(data));
                    }

                    this._conn.WriteEntireResponseFromString(this._responseStatus, this._responseHeadersBuilder.ToString(), sb.ToString(), false);
                    this._conn.CloseAll();

                    if (!this._url.Contains("PSIARequest.ashx"))
                    {
                        StringBuilder sbLog = new StringBuilder();
                        sbLog.Append("Incorrect Request:\r\n");
                        sbLog.Append("url:");
                        sbLog.Append(this._url);
                        sbLog.Append("\r\n");
                        sbLog.Append("Status Code:" + this._responseStatus.ToString());
                        sbLog.Append("\r\n Request Header:");
                        sbLog.Append(Encoding.UTF8.GetString(this._headerBytes));
                        sbLog.Append("\r\n Response Header:");
                        sbLog.Append(this._responseHeadersBuilder.ToString());
                        WebEventLog.GetInstance().WriteInfomation(sbLog.ToString());
                    }
                }
                return;
            }

            if (!this._headersSent)
            {
                this._conn.WriteHeaders(this._responseStatus, this._responseHeadersBuilder.ToString());
                this._headersSent = true;
            }


            for (int i = 0; i < this._responseBodyBytes.Count; i++)
            {
                byte[] data = (byte[])this._responseBodyBytes[i];
                this._conn.WriteBody(data, 0, data.Length);
            }

            this._responseBodyBytes = new ArrayList();

            isPSIARequest = false;
            


            if (finalFlush)
            {
                if (dataChangedFlag)
                {
                    this._host.DataChangedNotification();
                    dataChangedFlag = false;
                }
                if (!this._conn.KeepAlive || pwdChanged)
                {
                    this._conn.Close();
                }
            }

        }

        public override int ReadEntityBody(byte[] buffer, int size)
        {
            int count = 0;
            byte[] src = this._conn.ReadRequestBytes(size);
            if ((src != null) && (src.Length > 0))
            {
                count = src.Length;
                Buffer.BlockCopy(src, 0, buffer, 0, count);
            }
            return count;
        }

        public override void SendCalculatedContentLength(int contentLength)
        {
            if (!this._headersSent)
            {
                this._responseHeadersBuilder.Append("Content-Length: ");
                this._responseHeadersBuilder.Append(contentLength.ToString());
                this._responseHeadersBuilder.Append("\r\n");
            }
        }

        public override void SendKnownResponseHeader(int index, string value)
        {
            if (!this._headersSent)
            {
                switch (index)
                {
                    case 1:
                    case 2:
                    //case 0x1a:
                    //    return;

                    case 0x12:
                    case 0x13:
                        if (this._specialCaseStaticFileHeaders)
                        {
                            return;
                        }
                        break;

                    case 20:
                        if (!(value == "bytes"))
                        {
                            break;
                        }
                        this._specialCaseStaticFileHeaders = true;
                        return;
                }
                this._responseHeadersBuilder.Append(HttpWorkerRequest.GetKnownResponseHeaderName(index));
                this._responseHeadersBuilder.Append(": ");
                this._responseHeadersBuilder.Append(value);
                this._responseHeadersBuilder.Append("\r\n");

            }
        }

        public override void SendResponseFromFile(IntPtr handle, long offset, long length)
        {
            if (length != 0L)
            {
                FileStream f = null;
                try
                {
                    //f = new FileStream(handle, FileAccess.Read, false);
                    f = new FileStream(handle, FileAccess.Read, false);
                    this.SendResponseFromFileStream(f, offset, length);
                }
                finally
                {
                    if (f != null)
                    {
                        f.Close();
                    }
                }
            }
        }

        public override void SendResponseFromFile(string filename, long offset, long length)
        {
            if (length != 0L)
            {
                FileStream f = null;
                try
                {
                    f = new FileStream(filename, FileMode.Open, FileAccess.Read, FileShare.Read);
                    this.SendResponseFromFileStream(f, offset, length);
                }
                finally
                {
                    if (f != null)
                    {
                        f.Close();
                    }
                }
            }
        }

        private void SendResponseFromFileStream(FileStream f, long offset, long length)
        {
            long num = f.Length;
            if (length == -1L)
            {
                length = num - offset;
            }
            if (((length != 0L) && (offset >= 0L)) && (length <= (num - offset)))
            {
                int num2;
                if (offset > 0L)
                {
                    f.Seek(offset, SeekOrigin.Begin);
                }
                if (length <= 0x10000L)
                {
                    byte[] buffer = new byte[(int)length];
                    num2 = f.Read(buffer, 0, (int)length);
                    this.SendResponseFromMemory(buffer, num2);
                }
                else
                {
                    byte[] buffer2 = new byte[0x10000];
                    int num3 = (int)length;
                    while (num3 > 0)
                    {
                        int count = (num3 < 0x10000) ? num3 : 0x10000;
                        num2 = f.Read(buffer2, 0, count);
                        this.SendResponseFromMemory(buffer2, num2);
                        num3 -= num2;
                        if ((num3 > 0) && (num2 > 0))
                        {
                            this.FlushResponse(false);
                        }
                    }
                }
            }
        }

        public override void SendResponseFromMemory(byte[] data, int length)
        {
            if (length > 0)
            {
                byte[] dst = new byte[length];
                Buffer.BlockCopy(data, 0, dst, 0, length);
                this._responseBodyBytes.Add(dst);
            }
        }

        public override void SendStatus(int statusCode, string statusDescription)
        {
            this._responseStatus = statusCode;
        }

        public override void SendUnknownResponseHeader(string name, string value)
        {
            if (!this._headersSent)
            {
                this._responseHeadersBuilder.Append(name);
                this._responseHeadersBuilder.Append(": ");
                this._responseHeadersBuilder.Append(value);
                this._responseHeadersBuilder.Append("\r\n");
            }
        }

        #endregion

        #region System Methods - Process Request

        public override void CloseConnection()
        {
            if (!this._conn.KeepAlive)
            {
                this._conn.Close();
            }
        }

        /// <summary>
        /// Generate the psia request by url
        /// </summary>
        /// <param name="url">Request URL</param>
        /// <returns>PSIARequest</returns>
        private PSIARequest GeneratePSIARequest(string url)
        {
            if (!string.IsNullOrEmpty(url))
            {
                string[] urlStr = url.Split(new char[] { '/' });
                if (urlStr != null && urlStr.Length > 1)
                {
                    string service = urlStr[1];
                    psiaRequest = PSIAHelper.Instance.CreatePSIARequest(service, this._url, this._verb);
                }
            }
            return psiaRequest;
        }

        public void Process()
        {
            // Thread.Sleep(500);
            this.ReadAllHeaders();
            if ((((this._headerBytes == null) || (this._endHeadersOffset < 0)) || (this._headerByteStrings == null)) || (this._headerByteStrings.Count == 0))
            {
                this._conn.WriteErrorAndClose(400);
            }
            else
            {
                this.ParseRequestLine();
                if (this.IsBadPath() || this._url.Contains("ChangePWD.aspx"))
                {
                    this._conn.WriteErrorAndClose(400);
                }
                else
                {
                    bool isClientScriptPath = false;
                    string clientScript = null;
                    if (!this._host.IsVirtualPathInApp(this._path, out isClientScriptPath, out clientScript))
                    {
                        this._conn.WriteErrorAndClose(0x194);
                    }
                    else
                    {
                        this.ParseHeaders();
                        this.ParsePostedContent();
                        if (((this._verb == "POST") && (this._contentLength > 0)) && (this._preloadedContentLength < this._contentLength))
                        {
                            this._conn.Write100Continue();
                        }
                        if (isClientScriptPath)
                        {
                            this._conn.WriteEntireResponseFromFile(this._host.PhysicalClientScriptPath + clientScript, false);
                        }
                        else if (isPSIARequest)
                        { 
                          //Handle the PSIA request
                          //1. Authentication request
                          this.PrepareResponse();
                          HttpRuntime.ProcessRequest(this);

                          if (this._responseStatus == 200)
                          {
                              HandlePSIARequest();
                              this._conn.WritePSIAResponse(returnResponseResult, false);
                              this._conn.CloseAll();
                          }
                        }
                        else if (!isPSIARequest && this.KeepAlive)
                        { 
                           //Handle heart bead and alarm request
                            this.PrepareResponse();
                            HttpRuntime.ProcessRequest(this);

                            if (this._responseStatus == 200)
                            {
                                HandleHeartBeatAndAlarmRequest();
                            }
                        }
                        else if (!this.ProcessDirectoryListingRequest())
                        {
                            this.PrepareResponse();

                            HttpRuntime.ProcessRequest(this);
                        }
                        this._host.UserPasswordChangedEvent -= new UserPasswordChangedHandler(_host_UserPasswordChangedEvent);
                    }
                }
            }
        }

        #endregion

        #region Private Methods

        /// <summary>
        /// Handle the heart beat and alarm request
        /// </summary>
        private void HandleHeartBeatAndAlarmRequest()
        {
            try
            {
                UTF8Encoding encoding = new UTF8Encoding();

                int memoryAlertCount = 0;
                int cupAlertCount = 0;
                int heartBeatCount = 0;
                bool isCPUAlert = false;
                bool isMemoryAlert = false;
                string keepAliveData = string.Empty;
                string keepAliveMemoryData = string.Empty;
                bool cpuActive = false;
                bool memoryActive = false;
                while (this.KeepAlive && this._conn != null)
                {
                    if (!this._conn.Connected || pwdChanged)
                    {
                        break;
                    }
                    if (cupAlertCount == 0 && heartBeatCount == 0 && memoryAlertCount == 0)
                    {
                        this._conn.WriteEntireResponseFromString(200, FirstAlertResponseHeader, "--<boundary>", true);
                        cupAlertCount++;
                        heartBeatCount++;
                        memoryAlertCount++;
                    }
                    else
                    {
                        isCPUAlert = false;
                        isMemoryAlert = false;
                        if (this._host.SystemInfo != null && this._host.SystemInfo.CPUEnabled)
                        {
                            //Handle CUP usg
                            if (this._host.IsCPUOverload)
                            {
                                keepAliveData = ConstructCPUAlertResponse(this._conn.LocalIP, cupAlertCount, true);
                                cupAlertCount++;
                                isCPUAlert = true;
                                cpuActive = true;
                            }
                            else
                            {
                                if (cpuActive)
                                {
                                    keepAliveData = ConstructCPUAlertResponse(this._conn.LocalIP, cupAlertCount, false);
                                    cupAlertCount++;
                                    isCPUAlert = true;
                                    cpuActive = false;
                                }
                            }

                            //Handle Memory usg
                            if (this._host.IsMemoryOverload)
                            {
                                keepAliveMemoryData = ConstructMemoryAlertResponse(this._conn.LocalIP, memoryAlertCount, true);
                                memoryAlertCount++;
                                isMemoryAlert = true;
                                memoryActive = true;
                            }
                            else
                            {
                                if (memoryActive)
                                {
                                    keepAliveMemoryData = ConstructMemoryAlertResponse(this._conn.LocalIP, memoryAlertCount, false);
                                    memoryAlertCount++;
                                    isMemoryAlert = true;
                                    memoryActive = false;
                                }
                            }
                        }

                        if (!isCPUAlert && !isMemoryAlert)
                        {
                            keepAliveData = ConstructHeartBeatResponse(this._conn.LocalIP, heartBeatCount);
                            heartBeatCount++;
                        }

                        this._conn.WriteEntireResponseFromString(200, string.Format(AlertResponseHeader, keepAliveData.Length), keepAliveData, true);

                        if (isMemoryAlert)
                        {
                            this._conn.WriteEntireResponseFromString(200, string.Format(AlertResponseHeader, keepAliveMemoryData.Length), keepAliveMemoryData, true);
                        }
                    }

                    Thread.Sleep(this._host.KeepAliveInterval);
                }
            }
            catch (Exception ex)
            { 
               
            }
        }

        private string GetDataValue(string data)
        {
            string value = string.Empty;
            if (!string.IsNullOrEmpty(data))
            {
                string[] values = data.Split(new char[] { '=' });
                if (values.Length == 2)
                {
                    value = values[1];
                }
            }
            return value;
        }

        private void PrepareResponse()
        {
            this._headersSent = false;
            this._responseStatus = 200;
            this._responseHeadersBuilder = new StringBuilder();
            this._responseBodyBytes = new ArrayList();
        }

        private bool ProcessDirectoryListingRequest()
        {
            //if (this._verb != "GET")
            //{
            //    return false;
            //}
            int startIndex = this._pathTranslated.LastIndexOf('\\');
            string lastFolder = this._pathTranslated.Substring(startIndex);
            if (!lastFolder.Equals(@"\IPSD.WebSite", StringComparison.OrdinalIgnoreCase))
            {
                if (this._pathTranslated.IndexOf('.', startIndex) >= startIndex)
                {
                    return false;
                }
            }
            if (!Directory.Exists(this._pathTranslated))
            {
                return false;
            }
            if (!this._path.EndsWith("/"))
            {
                string str = this._path + "/";
                string extraHeaders = "Location: " + str + "\r\n";
                string body = "<html><head><title>Object moved</title></head><body>\r\n<h2>Object moved to <a href='" + str + "'>here</a>.</h2>\r\n</body></html>\r\n";
                this._conn.WriteEntireResponseFromString(0x12e, extraHeaders, body, false);
                return true;
            }
            string str5 = string.Empty;
            foreach (string str4 in s_defaultFilenames)
            {
                str5 = this._pathTranslated + @"\" + str4;
                if (File.Exists(str5))
                {
                    this._path = this._path + str4;
                    this._filePath = this._path;
                    this._url = (this._queryString != null) ? (this._path + "?" + this._queryString) : this._path;
                    this._pathTranslated = str5;
                    return false;
                }
            }
            FileSystemInfo[] elements = null;
            try
            {
                elements = new DirectoryInfo(this._pathTranslated).GetFileSystemInfos();
            }
            catch
            {
            }
            string path = null;
            if (this._path.Length > 1)
            {
                int length = this._path.LastIndexOf('/', this._path.Length - 2);
                path = (length > 0) ? this._path.Substring(0, length) : "/";
                if (!this._host.IsVirtualPathInApp(path))
                {
                    path = null;
                }
            }
            this._conn.WriteEntireResponseFromString(200, "Content-type: text/html; charset=utf-8\r\n", Messages.FormatDirectoryListing(this._path, path, elements), false);
            return true;
        }

        /// <summary>
        /// Construct the heart beat response
        /// </summary>
        /// <param name="ip">heart address</param>
        /// <param name="count"></param>
        /// <returns></returns>
        private string ConstructHeartBeatResponse(string ip, int count)
        {
            string response = string.Empty;
            if (!string.IsNullOrEmpty(ip) && count > 0)
            {
                response = HeartBeatResponse.Replace("%IP%", ip).
                                             Replace("%DateTime%", DateTime.Now.ToString()).
                                             Replace("%Count%", count.ToString());

            }
            return response;
        }

        /// <summary>
        /// Construct Alert response
        /// </summary>
        /// <param name="ip">Ip address</param>
        /// <param name="count">alert count</param>
        /// <returns></returns>
        private string ConstructCPUAlertResponse(string ip, int count,bool active)
        {
            string response = string.Empty;
            if (!string.IsNullOrEmpty(ip) && count > 0)
            {
                string state = "inactive";
                if (active)
                {
                    state = "active";
                }
                response = AlertResponse.Replace("%IP%", ip).
                                             Replace("%DateTime%", DateTime.Now.ToString()).
                                             Replace("%Count%", count.ToString()).Replace("%active%", state);

            }
            return response;
        }

        /// <summary>
        /// Construct memory alert response
        /// </summary>
        /// <param name="ip">IP Address</param>
        /// <param name="count">count</param>
        /// <param name="active">active</param>
        /// <returns>the response</returns>
        private string ConstructMemoryAlertResponse(string ip, int count, bool active)
        {
            string response = string.Empty;
            if (!string.IsNullOrEmpty(ip) && count > 0)
            {
                string state = "inactive";
                if (active)
                {
                    state = "active";
                }
                response = AlertMemoryResponse.Replace("%IP%", ip).
                                             Replace("%DateTime%", DateTime.Now.ToString()).
                                             Replace("%Count%", count.ToString()).Replace("%active%", state);

            }
            return response;
        }

        #endregion

    }


}
