/*=======================================================================
?Copyright (C) Microsoft Corporation.?All rights reserved.
?
  THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
  KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
  PARTICULAR PURPOSE.
=======================================================================*/

using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace IPSD.WebServer
{
    public class SystemInfo
    {
        #region Win32 Api

        [StructLayout(LayoutKind.Sequential)]
        public struct MemoryStatus
        {
            public uint Length;
            public uint MemoryLoad;
            public ulong TotalPhysical;
            public ulong AvailablePhysical;
            public ulong TotalPageFile;
            public ulong AvailablePageFile;
            public ulong TotalVirtual;
            public ulong AvailableVirtual;
            public ulong ullAvailExtendedVirtual;
        }

        [DllImport("kernel32.dll")]
        public static extern void GlobalMemoryStatusEx(ref MemoryStatus stat);

        #endregion

        #region Fields

        internal static PerformanceCounter CPUIdle;// = new PerformanceCounter("Process", "% Processor Time", "Idle");
        internal static PerformanceCounter CPUTotal;// = new PerformanceCounter("Process", "% Processor Time", "_Total");

        private static SystemInfo s_SysInfo = new SystemInfo();
        private NetworkInfo m_NetworkInfo;
        private object m_objSync = new object();
        private bool m_Running = false;
        private bool m_firstTimePoll = true;
        //private float m_cpuIdle;
        private float m_cpuTotal;
        //private int m_countCPU = 1;

        private ulong m_totalPhysicalMem = 0;

        private object m_ObjSync = new object();
        private System.Timers.Timer m_TimerPeek;
        private DateTime m_dtLastRefresh = DateTime.MinValue;

        private uint m_intervalRefresh = 5000;

        //Eric Added
        private bool m_CPUEnabled = true;
        private bool m_MemoryEnabled = true;
        private bool m_NetworkEabled = false;
        //Eric Added end

        private Dictionary<int, ProcessInfo> m_HTID2PInfo = new Dictionary<int, ProcessInfo>();

        #endregion

        #region Properties

        /// <summary>
        /// interval in ms of refresh perf data
        /// </summary>
        public uint IntervalRefresh
        {
            get
            {
                return m_intervalRefresh;
            }
            set
            {
                if (value >= 1000 && value <= 5000)
                {
                    m_intervalRefresh = value;
                }
            }
        }

        public ulong TotalPhysicalMem
        {
            get
            {
                if (m_totalPhysicalMem == 0)
                {
                    MemoryStatus memStatus = new MemoryStatus();
                    memStatus.Length = sizeof(uint) * 2 + sizeof(ulong) * 7;
                    GlobalMemoryStatusEx(ref memStatus);
                    m_totalPhysicalMem = memStatus.TotalPhysical;
                }
                return m_totalPhysicalMem;
            }
        }

        //Eric Added

        /// <summary>
        /// Get to see whether the CPU performance counter is valid
        /// </summary>
        public bool CPUEnabled
        {
            get
            {
                return m_CPUEnabled;
            }
        }
        /// <summary>
        /// Get to see whether the Memory performance counter is valid
        /// </summary>
        public bool MemoryEnabled
        {
            get
            {
                return m_MemoryEnabled;
            }
        }
        /// <summary>
        /// Get to see whether the Network performance counter is valid
        /// </summary>
        public bool NetworkEnabled
        {
            get
            {
                return m_NetworkEabled;
            }
        }

        //Eric added end

        #endregion

        #region Constructors

        public static SystemInfo GetInstance()
        {
            return s_SysInfo;
        }

        private SystemInfo()
        {
            //Modified by Eric
            try
            {
                m_NetworkInfo = new NetworkInfo();
                m_NetworkInfo.Init();
                m_NetworkEabled = true;
            }
            catch
            {
                m_NetworkEabled = false;
            }
            CheckPerformanceCounterValidation();
            //Refresh();
            //m_dtLastRefresh = DateTime.MinValue;
        }
        //add by Eric
        /// <summary>
        /// Init the static field
        /// </summary>
        static SystemInfo()
        {
            try
            {
                // CPUIdle = new PerformanceCounter("Process", "% Processor Time", "Idle");
                CPUTotal = new PerformanceCounter("Processor", "% Processor Time", "_Total", true);
            }
            catch
            {
            }
        }

        #endregion

        #region Public helper functions
        
        public void Refresh()
        {
            lock (m_ObjSync)
            {
                TimeSpan diff = DateTime.Now.Subtract(m_dtLastRefresh);
                if (diff.TotalMilliseconds < m_intervalRefresh)
                {
                    return;
                }
            }

            if (CPUEnabled)
            {
                RefreshCpuData();
            }
           
            m_dtLastRefresh = DateTime.Now;
        }

        public int GetCpuUsage()
        {
            //int nRet = -1;

            //if (!m_firstTimePoll)
            //{
            //    nRet = (int)(100 - m_cpuIdle);
            //    if (nRet < 0)
            //    {
            //        nRet = 0;
            //    }
            //    else if (nRet > 100)
            //    {
            //        nRet = 100;
            //    }
            //}
            return (int)m_cpuTotal;
        }

        public int GetCurrentProcessMemOccupied()
        {
            int nRet = -1;
            Process process = Process.GetCurrentProcess();
            int phsyMemUsage = (int)(process.WorkingSet64 >> 20);
            int vitrualUsage = (int)(process.PrivateMemorySize64 >> 20);
             nRet = phsyMemUsage + vitrualUsage;
             return nRet;
        }

        public MemoryStatus GetMemoryStatus()
        {
            MemoryStatus memStatus = new MemoryStatus();
            memStatus.Length = sizeof(uint) * 2 + sizeof(ulong) * 7;
            GlobalMemoryStatusEx(ref memStatus);
            return memStatus;
        }

        #endregion

        #region Private helper functions

        /// <summary>
        /// Check whether the Process performance counter is valid.
        /// </summary>
        private void CheckPerformanceCounterValidation()
        {
            PerformanceCounter pc;
            try
            {
                pc = new PerformanceCounter("Process", "% Processor Time");
                if (pc != null)
                {
                    this.m_MemoryEnabled = true;
                    this.m_CPUEnabled = true;
                }
            }
            catch
            {
                this.m_MemoryEnabled = false;
                this.m_CPUEnabled = false;
            }
        }

        private Process GetProcessByName(string processName)
        {
            Process process = null;
            Process[] ps = Process.GetProcessesByName(processName);

            if (ps != null && ps.Length > 0)
            {
                process = ps[0];
            }

            return process;
        }

        void m_TimerPeek_Elapsed(object sender, System.Timers.ElapsedEventArgs e)
        {
            this.m_TimerPeek.Stop();
            lock (m_HTID2PInfo)
            {
                Refresh();
            }
            this.m_TimerPeek.Start();
        }

        private void RefreshProcessDatas()
        {
            if (m_HTID2PInfo.Count > 0)
            {
                Process process;
                IList<int> idsObsolete = new List<int>();
                foreach (ProcessInfo pInfo in m_HTID2PInfo.Values)
                {
                    try
                    {
                        process = Process.GetProcessById(pInfo.ID);
                        if (process != null && process.Id > 0)
                        {
                            pInfo.Udpate();
                            pInfo.PhysicalMemUsage = process.WorkingSet64;
                        }
                        else
                        {
                            idsObsolete.Add(pInfo.ID);
                        }
                    }
                    catch
                    {
                        idsObsolete.Add(pInfo.ID);
                    }
                }
                if (idsObsolete.Count > 0)
                {
                    foreach (int id in idsObsolete)
                    {
                        m_HTID2PInfo.Remove(id);
                    }
                }
            }
        }

        private void RefreshCpuData()
        {
            //m_cpuIdle = CPUIdle.NextValue();
            m_cpuTotal = CPUTotal.NextValue();

            if (m_firstTimePoll)
            {
                m_firstTimePoll = false;
            }

            //if (m_cpuIdle != 0 && m_cpuTotal != 0)
            //{
            //    // Not the first sampling
            //    if (m_cpuTotal > 100)
            //    {
            //        m_countCPU = (int)(m_cpuTotal + 100) / 100;
            //        m_cpuIdle = m_cpuIdle * 100 / m_cpuTotal;
            //    }
            //}
        }

        private void RefreshNetworkInfo()
        {
            m_NetworkInfo.Refresh();
        }

        #endregion
    }

    #region ProcessInfo class

    public class ProcessInfo
    {
        internal PerformanceCounter perfCounter;
        // private Process m_Process;
        internal int ID;
        internal string Name;
        internal float CpuUsage;
        internal long PhysicalMemUsage;

        public PerformanceCounter ProcessCounter
        {
            get
            {
                return perfCounter;
            }
        }

        internal ProcessInfo(Process p)
        {
            ID = p.Id;
            Name = p.ProcessName;
            perfCounter = new PerformanceCounter("Process", "% Processor Time");
            perfCounter.InstanceName = Name;

        }

        internal void Udpate()
        {
            CpuUsage = perfCounter.NextValue();
        }
    }

    #endregion

    #region NetworkInfo class

    public class NetworkInfo
    {
        internal IList<PerformanceCounter> perfCounters = new List<PerformanceCounter>();

        private float m_NetTransTotal = 0.0f;
        private float m_BandWidthTotal = 0.0f;
        private float m_NetworkUsagePercent = 0.0f;

        private bool m_Initialized = false;


        public float BandWidthTotal
        {
            get { return m_BandWidthTotal; }
            set { m_BandWidthTotal = value; }
        }

        public float NetTransTotal
        {
            get { return m_NetTransTotal; }
        }

        public float NetworkUsagePercent
        {
            get
            {
                return m_NetworkUsagePercent;
            }
        }

        public NetworkInfo()
        {

        }

        //Eric Added
        public int NetworkPerformanceCount
        {
            get
            {
                return perfCounters.Count;
            }
        }
        //Eric Added end

        private void BuildPCList()
        {
            string filter = "MS TCP Loopback interface";
            PerformanceCounterCategory category = new PerformanceCounterCategory("Network Interface");
            if (category != null)
            {
                if (category.GetInstanceNames() != null)
                {
                    PerformanceCounter pc;
                    foreach (string nic in category.GetInstanceNames())
                    {
                        if (!nic.Equals(filter, StringComparison.InvariantCultureIgnoreCase))
                        {
                            pc = new PerformanceCounter("Network interface", "Bytes Total/sec");
                            pc.InstanceName = nic;
                            perfCounters.Add(pc);
                        }
                    }
                }
            }
            else
            {
                //log4net.ILog loger = log4net.LogManager.GetLogger(this.GetType());
                //loger.Warn("Can not init Network Interface, Network performance counter is invalid.");

            }

        }

        private void CaculateTotalBandWidth()
        {
            if (perfCounters.Count > 0)
            {
                PerformanceCounter pcBandwidth;
                foreach (PerformanceCounter pc in perfCounters)
                {
                    pcBandwidth = new PerformanceCounter("Network interface", "Current Bandwidth");
                    pcBandwidth.InstanceName = pc.InstanceName;
                    m_BandWidthTotal += pcBandwidth.NextValue();
                }
            }
        }
        public void Init()
        {
            if (!m_Initialized)
            {
                try
                {
                    BuildPCList();
                    CaculateTotalBandWidth();
                }
                catch (Exception ex)
                {
                    throw new Exception("Network interface performance counter not supporte!", ex);
                }

            }
        }

        public void Refresh()
        {
            m_NetTransTotal = 0.0f;
            if (perfCounters != null && perfCounters.Count > 0)
            {
                foreach (PerformanceCounter pc in perfCounters)
                {
                    try
                    {
                        m_NetTransTotal += pc.NextValue();
                    }
                    catch
                    {

                    }
                }
            }

            if (m_BandWidthTotal > 0.0f && m_NetTransTotal > 0.0f)
            {
                m_NetworkUsagePercent = (8 * m_NetTransTotal / m_BandWidthTotal) * 100;
            }
            else
            {
                m_NetworkUsagePercent = 0.0f;
            }
        }
    }

    #endregion
}
