using System;
using System.Collections;
using System.Drawing;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Security.Permissions;
using System.Diagnostics.CodeAnalysis;
using HUS.Controls.Base;
using HUS.Controls.Helper;
using System.ComponentModel;
using System.Drawing.Drawing2D;

namespace HUS.Controls.TabDock
{
	internal class DefaultFloatWindow : FloatWindow
	{
	    internal protected DefaultFloatWindow(TabDockPanel dockPanel, DockPane pane):
            base(dockPanel,pane)
		{
            InitXNCForm();
            InitCForm();

            this.FormBorderStyle = FormBorderStyle.Sizable;
            EnableCloseBox = false;
        }

        internal protected DefaultFloatWindow(TabDockPanel dockPanel, DockPane pane, Rectangle bounds):
            base(dockPanel,pane,bounds)
		{
            InitXNCForm();
            InitCForm();
            this.FormBorderStyle = FormBorderStyle.Sizable;
            EnableCloseBox = false;
        }

        private bool _enableNCPaint = true;
        public override bool EnableNCPaint
        {
            get { return _enableNCPaint; }
            set 
            { 
                _enableNCPaint = value;
                InvalidateWindow();
            }
        }

        #region XNCForm Implement
        #region Properties
        private CaptionBox _maximizeBox;
        private CaptionBox _minimizeBox;
        private CaptionBox _closeBox;
        private CaptionMenuItem[] _menuItems;
        private CaptionBoxCollection _captionBoxCollection = new CaptionBoxCollection();


        private Icon _icon;
        private string _formStyleName = "";
        private bool _useFormStyleManager = false;

        private int _menuControlBoxOffSet = 50;

        private FormStyle _activeFormStyle;
        private FormStyle _formStyle;

        private CaptionBoxCollection CaptionBoxs
        {
            get { return _captionBoxCollection; }
        }

        [Browsable(false)]
        public CaptionMenuItem[] CaptionMenuItems
        {
            get { return _menuItems; }
            set 
            {
                if(_menuItems!=value)
                    _menuItems = value; 
            }
        }

        public new bool MaximizeBox
        {
            get { return base.MaximizeBox; }
            set
            {
                this._maximizeBox.Enabled = value;
                this._minimizeBox.Visible = this._maximizeBox.Visible
                    = this._maximizeBox.Enabled | this._minimizeBox.Enabled;
                base.MaximizeBox = value;
            }
        }

        public new bool MinimizeBox
        {
            get { return base.MinimizeBox; }
            set
            {
                this._minimizeBox.Enabled = value;
                this._minimizeBox.Visible = this._maximizeBox.Visible
                    = this._maximizeBox.Enabled | this._minimizeBox.Enabled;
                base.MinimizeBox = value;
            }
        }

        public new bool ControlBox
        {
            get { return base.ControlBox; }
            set
            {
                //this._closeBox.Enabled = value;
                this._closeBox.Visible = value;
                base.ControlBox = value;
            }
        }

        public bool EnableCloseBox
        {
            set 
            {
                if (_closeBox.Enabled != value)
                {
                    _closeBox.Enabled = value;
                }
            }
        }

        public new Icon Icon
        {
            get { return base.Icon; }
            set
            {
                if (value != Icon)
                {
                    if (this._icon != null)
                    {
                        _icon.Dispose();
                        _icon = null;
                    }
                    try
                    {
                        _icon = new Icon(value, SystemInformation.SmallIconSize);
                    }
                    catch{ }
                }
                base.Icon = value;
            }
        }



        public bool UseFormStyleManager
        {
            get { return _useFormStyleManager; }
            set
            {
                if (_useFormStyleManager != value)
                    _useFormStyleManager = value;
            }
        }

        public string FormStyleName
        {
            get { return _formStyleName; }
        }

        protected FormStyle ActiveFormStyle
        {
            get { return _activeFormStyle; }
            set 
            {
                if (_activeFormStyle != value)
                {
                    _activeFormStyle = value;

                    UpdateButtonAppearance();
                    InvalidateWindow();
                }
            }
        }

        
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public FormStyle FormStyle
        {
            get { return _formStyle; }
            set
            {
                if (_formStyle != value)
                {
                    _formStyle = value;
                    UpdateActiveFormStyle();
                }
            }
        }
        #endregion

        private void InitXNCForm()
        {
            _closeBox = new CaptionBox();
            _closeBox.Key = "close";
            _closeBox.Visible = true;
            _closeBox.HitTestCode = (int)Win32API.NCHITTEST.HTCLOSE;
            CaptionBoxs.Add(_closeBox);

            _maximizeBox = new CaptionBox();
            _maximizeBox.Key = "maximize";
            _maximizeBox.Enabled = this.MaximizeBox;
            _maximizeBox.HitTestCode = (int)Win32API.NCHITTEST.HTMAXBUTTON;
            CaptionBoxs.Add(_maximizeBox);

            _minimizeBox = new CaptionBox();
            _minimizeBox.Key = "minimize";
            _minimizeBox.Enabled = this.MinimizeBox;
            _minimizeBox.HitTestCode = (int)Win32API.NCHITTEST.HTMINBUTTON;
            CaptionBoxs.Add(_minimizeBox);

            this._minimizeBox.Visible = this._maximizeBox.Visible
                = this._maximizeBox.Enabled | this._maximizeBox.Enabled;

            UpdateActiveFormStyle();
            FormStyleManager.StyleChanged += new EventHandler(FormStyleManager_StyleChanged);
            
        }

        void FormStyleManager_StyleChanged(object sender, EventArgs e)
        {
            UpdateActiveFormStyle();
        }

        private void UpdateActiveFormStyle()
        {
            FormStyle style = null;
            if (UseFormStyleManager)
            {
                // try to load specified style
                if (!String.IsNullOrEmpty(FormStyleName))
                    style = FormStyleManager.GetStyle(FormStyleName);

                // if it wasn't found try to load default style
                if (style == null)
                    style = FormStyleManager.GetDefaultStyle();
            }
            else
            {
                style = FormStyle;
            }
            ActiveFormStyle = style;
        }

        private void UpdateButtonAppearance()
        {
            if (ActiveFormStyle != null)
            {
                _closeBox.Appearance = ActiveFormStyle.CloseButtonStyle;
                _minimizeBox.Appearance = ActiveFormStyle.MinimizeButtonStyle;
                _maximizeBox.Appearance = (this.WindowState == FormWindowState.Maximized) ?
                    ActiveFormStyle.RestoreButtonStyle : ActiveFormStyle.MaximizeButtonStyle;
            }
            else
            {
                _closeBox.Appearance = null;
                _minimizeBox.Appearance = null;
                _maximizeBox.Appearance = null;
            }
        }

        #region Process Windows Messages
        
        #region OnNonClientAreaCalcSize
        protected override void OnNonClientAreaCalcSize(ref Rectangle bounds, bool update)
        {
            if (ActiveFormStyle == null)
                return;

            if (update)
                UpdateCaptionBoxBounds(bounds);

            Padding clientPadding = ActiveFormStyle.ClientAreaPadding;
            bounds = new Rectangle(bounds.Left + clientPadding.Left, bounds.Top + clientPadding.Top,
                bounds.Width - clientPadding.Horizontal, bounds.Height - clientPadding.Vertical);
        }

        private void UpdateCaptionBoxBounds(Rectangle windowBounds)
        {
            // start from top-right corner
            int x = windowBounds.Width;

            foreach (CaptionBox button in this.CaptionBoxs)
            {
                if (button.Visible)
                {
                    int y = button.Appearance.Margin.Top;
                    x -= (button.Appearance.Size.Width + button.Appearance.Margin.Right);
                    button.Bounds = new Rectangle(x, y,
                        button.Appearance.Size.Width, button.Appearance.Size.Height);
                    x -= button.Appearance.Margin.Left;
                }

                // background will be recreated on next redraw
                button.BackgroundImage = null;
            }

            if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
            {
                x -= _menuControlBoxOffSet;
                for(int count = CaptionMenuItems.Length-1;count>=0;count--)
                {
                    CaptionMenuItem item = CaptionMenuItems[count];
                    if (item.Visible)
                    {
                        int y = item.Appearance.Margin.Top;
                        x -= (item.Appearance.Size.Width + item.Appearance.Margin.Right);
                        item.Bounds = new Rectangle(x, y, item.Appearance.Size.Width, item.Appearance.Size.Height);
                        x -= item.Appearance.Margin.Left;
                    }
                }

                // create menu item's background for next draw
                if (IsHandleCreated)
                {
                    // MSDN states that only WINDOW and INTERSECTRGN are needed,
                    // but other sources confirm that CACHE is required on Win9x
                    // and you need CLIPSIBLINGS to prevent painting on overlapping windows.
                    IntPtr hDC = Win32API.GetDCEx(this.Handle, (IntPtr)1,
                        (int)(Win32API.DCX.DCX_WINDOW | Win32API.DCX.DCX_INTERSECTRGN
                            | Win32API.DCX.DCX_CACHE | Win32API.DCX.DCX_CLIPSIBLINGS));
                    try
                    {
                        if (hDC != IntPtr.Zero)
                        {
                            IntPtr srcHDC = Win32API.CreateCompatibleDC(hDC);
                            IntPtr srcBitmap = Win32API.CreateCompatibleBitmap(hDC, windowBounds.Width, windowBounds.Height);

                            try
                            {
                                Win32API.SelectObject(srcHDC, srcBitmap);

                                using (Graphics srcG = Graphics.FromHdc(srcHDC))
                                {
                                    Rectangle tempRect = new Rectangle(0, 0, windowBounds.Width, windowBounds.Height);
                                    Region clipRegion = new Region(tempRect);
                                    clipRegion.Exclude(DrawUtil.ExcludePadding(tempRect, ActiveFormStyle.ClientAreaPadding));
                                    srcG.Clip = clipRegion;
                                    ActiveFormStyle.NormalState.Draw(srcG, tempRect);

                                    foreach (CaptionMenuItem item in CaptionMenuItems)
                                    {
                                        // background will be recreated on next redraw
                                        if (item.BackgroundImage != null)
                                        {
                                            item.BackgroundImage.Dispose();
                                            item.BackgroundImage = null;
                                        }
                                        Bitmap bkImg = new Bitmap(item.Bounds.Width, item.Bounds.Height);
                                        using (Graphics destG = Graphics.FromImage(bkImg))
                                        {
                                            IntPtr destDC = destG.GetHdc();
                                            Win32API.BitBlt(destDC, 0, 0, item.Bounds.Width, item.Bounds.Height, srcHDC, item.Bounds.X, item.Bounds.Y, Win32API.TernaryRasterOperations.SRCCOPY);
                                            destG.ReleaseHdc(destDC);
                                        }
                                        item.BackgroundImage = bkImg;
                                    }
                                }
                            }
                            finally
                            {
                                Win32API.DeleteObject(srcBitmap);
                                Win32API.DeleteDC(srcHDC);
                            }
                        }
                    }
                    finally
                    {
                        Win32API.ReleaseDC(this.Handle, hDC);
                    }
                }
                
            }

            // Should I move this where this actually changes WM_GETMINMAXINFO ??
            //maximizeButton.Appearance = (this.WindowState == FormWindowState.Maximized) ?
            //    _borderAppearance.RestoreButton : _borderAppearance.MaximizeButton;
        }
        #endregion

        #region MouseEvent
        protected override void OnNonClientMouseLeave(EventArgs args)
        {
            if (!trakingMouse)
                return;

            foreach (CaptionBox button in this.CaptionBoxs)
            {
                if (button.State != CaptionBoxState.Normal)
                {
                    button.State = CaptionBoxState.Normal;
                    DrawBox(button);
                    UnhookMouseEvent();
                }
            }

            if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
            {
                foreach (CaptionMenuItem item in CaptionMenuItems)
                {
                    if (item.State != CaptionBoxState.Normal)
                    {
                        item.State = CaptionBoxState.Normal;
                        DrawBox(item);
                        UnhookMouseEvent();
                    }
                }
            }
        }

        protected override void OnNonClientMouseDown(NonClientMouseEventArgs args)
        {
            if (args.Button != MouseButtons.Left)
                return;
            
            switch (args.HitTest)
            {
                //case (int)Win32API.NCHITTEST.HTSYSMENU:
                //    {
                //        IntPtr hSysMenu = Win32API.GetSystemMenu(this.Handle, 0);
                //        if (hSysMenu != IntPtr.Zero)
                //        {
                //            Point menuLocation = this.Location;
                //            int result = 0;
                //            if(Win32API.TrackPopupMenuEx(hSysMenu,
                //               (int)Win32API.TPM.TPM_RETURNCMD | (int)Win32API.TPM.TPM_LEFTBUTTON,
                //               menuLocation.X, menuLocation.Y, Handle, null))
                //                result = 1;
                //            Win32API.SendMessage(Handle,(int)Win32API.WindowMessages.WM_SYSCOMMAND,
                //               (uint)result,0); 

                //            args.Handled = true;
                //        }
                //        break;
                //    }
                case (int)Win32API.NCHITTEST.HTCLOSE:
                    {
                        //NativeMethods.POINT pt = NativeMethods.POINT.FromPoint(args.Location);
                        if (_closeBox.Enabled && DepressBox(_closeBox))
                            Win32API.SendMessage(this.Handle,
                                (int)Win32API.WindowMessages.WM_SYSCOMMAND,
                                (IntPtr)Win32API.SystemCommands.SC_CLOSE, IntPtr.Zero);
                        args.Handled = true;
                        break;
                    }
                case (int)Win32API.NCHITTEST.HTMINBUTTON:
                    {
                        //		NativeMethods.POINT pt = NativeMethods.POINT.FromPoint(args.Location);
                        if (_minimizeBox.Enabled && DepressBox(_minimizeBox))
                        {
                            Win32API.SendMessage(this.Handle,
                                (int)Win32API.WindowMessages.WM_SYSCOMMAND,
                                (IntPtr)Win32API.SystemCommands.SC_MINIMIZE, IntPtr.Zero);
                            MinimizeBox = false;
                        }
                        args.Handled = true;
                        break;
                    }
                case (int)Win32API.NCHITTEST.HTMAXBUTTON:
                    {
                        if (_maximizeBox.Enabled && DepressBox(_maximizeBox))
                        {
                            int sc = (WindowState == FormWindowState.Maximized||WindowState == FormWindowState.Minimized) ?
                                (int)Win32API.SystemCommands.SC_RESTORE : (int)Win32API.SystemCommands.SC_MAXIMIZE;

                            Win32API.SendMessage(this.Handle,
                                (int)Win32API.WindowMessages.WM_SYSCOMMAND,
                                (IntPtr)sc, IntPtr.Zero);
                            MinimizeBox = true;
                        }
                        args.Handled = true;
                        break;
                    }
            }

            int testHitResult = args.HitTest;
            if ((testHitResult & (int)Win32API.NCHITTEST.HTCAPTIONMENU) == (int)Win32API.NCHITTEST.HTCAPTIONMENU)
            {
                int index = testHitResult >> 8;

                CaptionMenuItem currentItem = null;
                if (CaptionMenuItems != null && CaptionMenuItems.Length > 0 && CaptionMenuItems.Length>index)
                {
                    currentItem = CaptionMenuItems[index];
                    if (currentItem != null &&currentItem.Enabled && DepressBox(currentItem))
                    {
                        currentItem.ShowDropdownMenu();
                    }
                }
                args.Handled = true;
            }
        }

        protected override void OnNonClientMouseMove(MouseEventArgs mouseEventArgs)
        {
            foreach (CaptionBox button in this.CaptionBoxs)
            {
                if (button.Visible && button.Bounds.Contains(mouseEventArgs.X, mouseEventArgs.Y) && button.HitTestCode > 0)
                {
                    if (button.State != CaptionBoxState.Over)
                    {
                        button.State = CaptionBoxState.Over;
                        DrawBox(button);
                        HookMouseEvent();
                    }
                }
                else
                {
                    if (button.State != CaptionBoxState.Normal)
                    {
                        button.State = CaptionBoxState.Normal;
                        DrawBox(button);
                        UnhookMouseEvent();
                    }
                }
            }

            if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
            {
                CaptionMenuItem shownMenuItem = null;
                foreach (CaptionMenuItem item in CaptionMenuItems)
                {
                    if (item.ContextMenu != null && item.ContextMenu.Visible)
                    {
                        shownMenuItem = item;
                        break;
                    }
                }

                foreach (CaptionMenuItem item in CaptionMenuItems)
                {
                    if (item.Visible && 
                        (item.Bounds.Contains(mouseEventArgs.X, mouseEventArgs.Y)
                        ||(item.ContextMenu!=null&& item.ContextMenu.Visible)))
                    {
                        if (item.State != CaptionBoxState.Over)
                        {
                            item.State = CaptionBoxState.Over;
                            DrawBox(item);

                            if (shownMenuItem != null && shownMenuItem != item)
                            {
                                shownMenuItem.HideDropdownMenu();
                                item.ShowDropdownMenu();
                            }
                            
                            HookMouseEvent();
                        }
                    }
                    else
                    {
                        if (item.State != CaptionBoxState.Normal)
                        {
                            item.State = CaptionBoxState.Normal;
                            DrawBox(item);
                            UnhookMouseEvent();
                        }
                    }
                }
            }
        }

        Win32API.TRACKMOUSEEVENT trackMouseEvent;
        bool trakingMouse = false;
        private void HookMouseEvent()
        {
            if (!trakingMouse)
            {
                trakingMouse = true;
                if (this.trackMouseEvent == null)
                {
                    this.trackMouseEvent = new Win32API.TRACKMOUSEEVENT();
                    this.trackMouseEvent.dwFlags =
                        (int)(Win32API.TrackMouseEventFalgs.TME_HOVER |
                              Win32API.TrackMouseEventFalgs.TME_LEAVE |
                              Win32API.TrackMouseEventFalgs.TME_NONCLIENT);

                    this.trackMouseEvent.hwndTrack = this.Handle;
                }
            }
        }

        private void UnhookMouseEvent()
        {
            trakingMouse = false;
        }
        #endregion

        #region OnNonClientAreaHitTest
        protected override int OnNonClientAreaHitTest(Point p)
        {
            if (ActiveFormStyle == null)
                return (int)Win32API.NCHITTEST.HTCLIENT;

            foreach (CaptionBox button in this.CaptionBoxs)
            {
                if (button.Visible && button.Bounds.Contains(p) && button.HitTestCode > 0)
                    return button.HitTestCode;
            }

            if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
            {
                for(int count =0;count< CaptionMenuItems.Length;count++)
                {
                    CaptionMenuItem item = CaptionMenuItems[count];
                    if (item.Visible && item.Bounds.Contains(p))
                    {
                        int hitResut = (int)Win32API.NCHITTEST.HTCAPTIONMENU;
                        int itemIndex = count << 8;
                        hitResut |= itemIndex;
                        return hitResut;
                    }
                }
            }

            if (FormBorderStyle != FormBorderStyle.FixedToolWindow &&
                FormBorderStyle != FormBorderStyle.SizableToolWindow &&
                FormBorderStyle != FormBorderStyle.Sizable)
            {
                if (GetIconRectangle().Contains(p))
                    return (int)Win32API.NCHITTEST.HTSYSMENU;
            }

            if (this.FormBorderStyle == FormBorderStyle.Sizable
                || this.FormBorderStyle == FormBorderStyle.SizableToolWindow)
            {
                #region Handle sizable window borders
                if (p.X <= ActiveFormStyle.SizingBorderWidth) // left border
                {
                    if (p.Y <= ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTTOPLEFT;
                    else if (p.Y >= this.Height - ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTBOTTOMLEFT;
                    else
                        return (int)Win32API.NCHITTEST.HTLEFT;
                }
                else if (p.X >= this.Width - ActiveFormStyle.SizingBorderWidth) // right border
                {
                    if (p.Y <= ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTTOPRIGHT;
                    else if (p.Y >= this.Height - ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTBOTTOMRIGHT;
                    else
                        return (int)Win32API.NCHITTEST.HTRIGHT;
                }
                else if (p.Y <= ActiveFormStyle.SizingBorderWidth) // top border
                {
                    if (p.X <= ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTTOPLEFT;
                    if (p.X >= this.Width - ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTTOPRIGHT;
                    else
                        return (int)Win32API.NCHITTEST.HTTOP;
                }
                else if (p.Y >= this.Height - ActiveFormStyle.SizingBorderWidth) // bottom border
                {
                    if (p.X <= ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTBOTTOMLEFT;
                    if (p.X >= this.Width - ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTBOTTOMRIGHT;
                    else
                        return (int)Win32API.NCHITTEST.HTBOTTOM;
                }
                #endregion
            }

            // title bar
            if (p.Y <= ActiveFormStyle.ClientAreaPadding.Top)
                return (int)Win32API.NCHITTEST.HTCAPTION;

            // rest of non client area
            if (p.X <= this.ActiveFormStyle.ClientAreaPadding.Left || p.X >= this.ActiveFormStyle.ClientAreaPadding.Right
                || p.Y >= this.ActiveFormStyle.ClientAreaPadding.Bottom)
                return (int)Win32API.NCHITTEST.HTBORDER;

            return (int)Win32API.NCHITTEST.HTCLIENT;
        }

        private Rectangle GetIconRectangle()
        {
            return new Rectangle(ActiveFormStyle.IconPadding.Left, ActiveFormStyle.IconPadding.Top, 16, 16);
        }
        #endregion

        #region OnNonClientAreaPaint
        protected override void OnNonClientAreaPaint(NonClientPaintEventArgs e)
        {

            if (ActiveFormStyle == null)
                return;

            // assign clip region to exclude client area
            Region clipRegion = new Region(e.Bounds);
            Rectangle clientRect = DrawUtil.ExcludePadding(e.Bounds, ActiveFormStyle.ClientAreaPadding);

            clipRegion.Exclude(clientRect);
            e.Graphics.Clip = clipRegion;

            // paint borders
            ActiveFormStyle.NormalState.Draw(e.Graphics, e.Bounds);

            int textOffset = 0;

            // paint icon
            if (ShowIcon &&
                FormBorderStyle != FormBorderStyle.FixedToolWindow &&
                FormBorderStyle != FormBorderStyle.SizableToolWindow)
            {
                Rectangle iconRect = GetIconRectangle();
                textOffset += iconRect.Right;

                if (_icon != null)
                    e.Graphics.DrawIconUnstretched(_icon, iconRect);
                else
                    e.Graphics.DrawIcon(Icon, iconRect);
            }


            // paint caption
            string text = this.Text;
            if (!String.IsNullOrEmpty(text))
            {
                // disable text wrapping and request elipsis characters on overflow
                using (StringFormat sf = new StringFormat())
                {
                    sf.Trimming = StringTrimming.EllipsisCharacter;
                    sf.FormatFlags = StringFormatFlags.NoWrap;
                    sf.LineAlignment = StringAlignment.Center;

                    // find position of the first button from left
                    int firstButton = e.Bounds.Width;
                    foreach (CaptionBox button in this.CaptionBoxs)
                        if (button.Visible)
                            firstButton = Math.Min(firstButton, button.Bounds.X);

                    if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
                    {
                        foreach (CaptionMenuItem item in CaptionMenuItems)
                        {
                            if (item.Visible)
                            {
                                firstButton = Math.Min(firstButton, item.Bounds.X);
                            }
                        }
                    }

                    Padding padding = ActiveFormStyle.TitlePadding;
                    Rectangle textRect = new Rectangle(textOffset + padding.Left,
                        padding.Top, firstButton - textOffset - padding.Horizontal,
                        ActiveFormStyle.ClientAreaPadding.Top - padding.Vertical);

                    Font textFont = this.Font;
                    if (ActiveFormStyle.TitleFont != null)
                        textFont = ActiveFormStyle.TitleFont;

                    if (!ActiveFormStyle.TitleShadowColor.IsEmpty)
                    {
                        Rectangle shadowRect = textRect;
                        shadowRect.Offset(1, 1);

                        // draw drop shadow
                        using (Brush b = new SolidBrush(ActiveFormStyle.TitleShadowColor))
                        {
                            e.Graphics.DrawString(text, textFont, b, shadowRect, sf);
                        }
                    }

                    if (!ActiveFormStyle.TitleColor.IsEmpty)
                    {
                        // draw text
                        using (Brush b = new SolidBrush(ActiveFormStyle.TitleColor))
                        {
                            e.Graphics.DrawString(text, textFont, b, textRect, sf);
                        }
                    }
                }
            }

            // paint buttons
            foreach (CaptionBox button in this.CaptionBoxs)
                button.Draw(e.Graphics, false);

            if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
            {
                foreach (CaptionMenuItem item in CaptionMenuItems)
                {
                    if (!e.Active)
                    {
                        item.State = CaptionBoxState.Normal;
                    }
                    item.Draw(e.Graphics, false);
                }
            }
        }
        #endregion

        protected override void UpdateWindowState()
        {
            if (ActiveFormStyle == null)
                return;

            CaptionBoxStyle newAppearance;
            if (this.WindowState == FormWindowState.Maximized)
                newAppearance = this.ActiveFormStyle.RestoreButtonStyle;
            else if(this.WindowState == FormWindowState.Minimized && !this.ShowInTaskbar)
                newAppearance = this.ActiveFormStyle.RestoreButtonStyle;
            else
                newAppearance = this.ActiveFormStyle.MaximizeButtonStyle;

            if (newAppearance != _maximizeBox.Appearance)
            {
                _maximizeBox.Appearance = newAppearance;
                DrawBox(_maximizeBox);
            }
        }

        protected override void UpdateCaptionMenuScreenPosition()
        {
            if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
            {
                for (int count = CaptionMenuItems.Length - 1; count >= 0; count--)
                {
                    CaptionMenuItem item = CaptionMenuItems[count];
                    if (item.Visible)
                    {
                        Point raw = PointToScreen(item.Bounds.Location);
                        raw.X -= 2;
                        raw.Y -= 4;
                        item.ScreenLocation = raw;
                    }
                }
            }
        }

        private void DrawBox(CaptionBox box)
        {
            if (IsHandleCreated)
            {
                // MSDN states that only WINDOW and INTERSECTRGN are needed,
                // but other sources confirm that CACHE is required on Win9x
                // and you need CLIPSIBLINGS to prevent painting on overlapping windows.
                IntPtr hDC = Win32API.GetDCEx(this.Handle, (IntPtr)1,
                    (int)(Win32API.DCX.DCX_WINDOW | Win32API.DCX.DCX_INTERSECTRGN
                        | Win32API.DCX.DCX_CACHE | Win32API.DCX.DCX_CLIPSIBLINGS));

                if (hDC != IntPtr.Zero)
                {
                    using (Graphics g = Graphics.FromHdc(hDC))
                    {
                        box.Draw(g, true);
                    }
                }

                Win32API.ReleaseDC(this.Handle, hDC);
            }
        }

        private bool DepressBox(CaptionBox currentBox)
        {
            bool result = false;
            try
            {
                // callect all mouse events until WL_BUTTONUP
                this.Capture = true;		// hopefully doeas the same as SetCapture(handle)

                // draw button in pressed mode
                currentBox.State = CaptionBoxState.Pressed;
                DrawBox(currentBox);

                // loop untill button is released
                bool done = false;
                while (!done)
                {
                    System.Threading.Thread.Sleep(1); 

                    // NOTE: This struct needs to be here. We had strange error (starting from Beta 2).
                    // when this was defined at begining of this method. also check if types are correct (no overlap). 
                    Message m = new Message();
                    if (Win32API.PeekMessage(ref m, IntPtr.Zero,
                        (int)Win32API.WindowMessages.WM_MOUSEFIRST, (int)Win32API.WindowMessages.WM_MOUSELAST,
                        (int)Win32API.PeekMessageOptions.PM_REMOVE))
                    {
                        switch (m.Msg)
                        {
                            case (int)Win32API.WindowMessages.WM_LBUTTONUP:
                                {
                                    if (currentBox.State == CaptionBoxState.Pressed)
                                    {
                                        currentBox.State = CaptionBoxState.Normal;
                                        DrawBox(currentBox);
                                    }
                                    Point pt = PointToWindow(PointToScreen(new Point(m.LParam.ToInt32())));
                                    result = currentBox.Bounds.Contains(pt);
                                    done = true;
                                    break;
                                }
                            case (int)Win32API.WindowMessages.WM_MOUSEMOVE:
                                {
                                    Point clientPoint = PointToWindow(PointToScreen(new Point(m.LParam.ToInt32())));
                                    if (currentBox.Bounds.Contains(clientPoint))
                                    {
                                        if (currentBox.State == CaptionBoxState.Normal)
                                        {
                                            currentBox.State = CaptionBoxState.Pressed;
                                            DrawBox(currentBox);
                                        }
                                    }
                                    else
                                    {
                                        if (currentBox.State == CaptionBoxState.Pressed)
                                        {
                                            currentBox.State = CaptionBoxState.Normal;
                                            DrawBox(currentBox);
                                        }
                                    }
                                    done = false;
                                    break;
                                }
                        }
                    }
                }
            }
            finally
            {
                this.Capture = false;
            }

            return result;
        }

        #endregion



        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);
            if (disposing)
            {
                if (_icon != null)
                    _icon.Dispose();

                FormStyleManager.StyleChanged -= new EventHandler(FormStyleManager_StyleChanged);
                ActiveFormStyle = null;
            }
        }


        private class CaptionBoxCollection : CollectionBase
        {
            public void Add(CaptionBox button)
            {
                this.List.Add(button);
            }
        }
        #endregion

        #region CForm Implement
        private void  InitCForm()
        {
            this.FormStyle = CreateFormStyle();
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            if (this.FormBorderStyle != FormBorderStyle.None)
            {
                int width = Width;
                int height = Height;

                int diam = 10;
                GraphicsPath path = new GraphicsPath();
                path.AddArc(0, 0, diam, diam, -90, -90);
                path.AddLines(new Point[] {new Point(0,diam), new Point(0, height), 
                new Point(width, height), new Point(width, diam)});
                path.AddArc(width - diam, 0, diam, diam, 0, -90);
                path.CloseFigure();

                this.Region = new Region(path);
            }
            else
            {
                if (this.WindowState == FormWindowState.Maximized)
                    this.Region = new Region(new Rectangle(0, 0, SystemInformation.PrimaryMonitorSize.Width, SystemInformation.PrimaryMonitorSize.Height));
            }
        }

        FormStyle CreateFormStyle()
        {
            FormStyle style = new FormStyle();
            ImageRenderer normalState = new ImageRenderer();
            if (RendererManager.Style == StyleEnum.Default)
                normalState.Image = Properties.Resources.border1;
            else
                normalState.Image = Properties.Resources.Border;

            normalState.Layout = ImageLayout.Stretch;
            normalState.StretchMargins = new Padding(8, 25, 8, 8);
            style.NormalState = normalState;


            style.CloseButtonStyle.Margin = new Padding(1, 5, 5, 0);
            ImageRenderer imgNRender = new ImageRenderer();
            ImageRenderer imgDRender = new ImageRenderer();
            ImageRenderer imgARender = new ImageRenderer();
            ImageRenderer imgHRender = new ImageRenderer();
            if (RendererManager.Style == StyleEnum.Default)
            {
                style.CloseButtonStyle.Size = Properties.Resources.close_1.Size;
                imgNRender.Image = Properties.Resources.close_1;
                imgDRender.Image = Properties.Resources.close_4;
                imgARender.Image = Properties.Resources.close_3;
                imgHRender.Image = Properties.Resources.close_2;
            }
            else
            {
                style.CloseButtonStyle.Size = Properties.Resources.CloseButton.Size;
                imgNRender.Image = Properties.Resources.CloseButton;
                imgDRender.Image = Properties.Resources.CloseButtonDisabled;
                imgARender.Image = Properties.Resources.CloseButtonPressed;
                imgHRender.Image = Properties.Resources.CloseButtonHover;
            }
            style.CloseButtonStyle.NormalState = imgNRender;
            style.CloseButtonStyle.DisabledState = imgDRender;
            style.CloseButtonStyle.ActiveState = imgARender;
            style.CloseButtonStyle.HoverState = imgHRender;


            style.MaximizeButtonStyle.Margin = new Padding(1, 5, 1, 0);
            imgNRender = new ImageRenderer();
            imgDRender = new ImageRenderer();
            imgARender = new ImageRenderer();
            imgHRender = new ImageRenderer();
            if (RendererManager.Style == StyleEnum.Default)
            {
                style.MaximizeButtonStyle.Size = Properties.Resources.max_1.Size;
                imgNRender.Image = Properties.Resources.max_1;
                imgDRender.Image = Properties.Resources.max_4;
                imgARender.Image = Properties.Resources.max_3;
                imgHRender.Image = Properties.Resources.max_2;
            }
            else
            {
                style.MaximizeButtonStyle.Size = Properties.Resources.MaximizeButton.Size;
                imgNRender.Image = Properties.Resources.MaximizeButton;
                imgDRender.Image = Properties.Resources.MaximizeButtonDisabled;
                imgARender.Image = Properties.Resources.MaximizeButtonPressed;
                imgHRender.Image = Properties.Resources.MaximizeButtonHover;
            }
            style.MaximizeButtonStyle.NormalState = imgNRender;
            style.MaximizeButtonStyle.DisabledState = imgDRender;
            style.MaximizeButtonStyle.ActiveState = imgARender;
            style.MaximizeButtonStyle.HoverState = imgHRender;

            style.MinimizeButtonStyle.Margin = new Padding(1, 5, 1, 0);
            imgNRender = new ImageRenderer();
            imgDRender = new ImageRenderer();
            imgARender = new ImageRenderer();
            imgHRender = new ImageRenderer();
            if (RendererManager.Style == StyleEnum.Default)
            {
                style.MinimizeButtonStyle.Size = Properties.Resources.min_1.Size;
                imgNRender.Image = Properties.Resources.min_1;
                imgDRender.Image = Properties.Resources.min_4;
                imgARender.Image = Properties.Resources.min_3;
                imgHRender.Image = Properties.Resources.min_2;
            }
            else
            {
                style.MinimizeButtonStyle.Size = Properties.Resources.MinimizeButton.Size;
                imgNRender.Image = Properties.Resources.MinimizeButton;
                imgDRender.Image = Properties.Resources.MinimizeButtonDisabled;
                imgARender.Image = Properties.Resources.MinimizeButtonPressed;
                imgHRender.Image = Properties.Resources.MinimizeButtonHover;
            }
            style.MinimizeButtonStyle.NormalState = imgNRender;
            style.MinimizeButtonStyle.DisabledState = imgDRender;
            style.MinimizeButtonStyle.ActiveState = imgARender;
            style.MinimizeButtonStyle.HoverState = imgHRender;


            style.RestoreButtonStyle.Margin = new Padding(1, 5, 1, 0);
            imgNRender = new ImageRenderer();
            imgDRender = new ImageRenderer();
            imgARender = new ImageRenderer();
            imgHRender = new ImageRenderer();
            if (RendererManager.Style == StyleEnum.Default)
            {
                style.RestoreButtonStyle.Size = Properties.Resources.return_1.Size;
                imgNRender.Image = Properties.Resources.return_1;
                imgDRender.Image = Properties.Resources.return_4;
                imgARender.Image = Properties.Resources.return_3;
                imgHRender.Image = Properties.Resources.return_2;
            }
            else
            {
                style.RestoreButtonStyle.Size = Properties.Resources.RestoreButton.Size;
                imgNRender.Image = Properties.Resources.RestoreButton;
                imgDRender.Image = Properties.Resources.RestoreButtonDisabled;
                imgARender.Image = Properties.Resources.RestoreButtonPressed;
                imgHRender.Image = Properties.Resources.RestoreButtonHover;
            }
            style.RestoreButtonStyle.NormalState = imgNRender;
            style.RestoreButtonStyle.DisabledState = imgDRender;
            style.RestoreButtonStyle.ActiveState = imgARender;
            style.RestoreButtonStyle.HoverState = imgHRender;

            style.TitleColor = Color.White;
            style.TitlePadding = new Padding(3, 0, 0, 0);
            style.TitleShadowColor = Color.DimGray;
            style.TitleFont = new System.Drawing.Font("Tahoma", 8.25F, System.Drawing.FontStyle.Bold);

            style.ClientAreaPadding = new Padding(8, 25, 8, 8);
            style.IconPadding = new Padding(3, 3, 2, 0);

            this.NonClientAreaDoubleBuffering = true;

            return style;
        }
        #endregion

    }

    internal class HUSD4FloatWindow : FloatWindow
    {
        internal protected HUSD4FloatWindow(TabDockPanel dockPanel, DockPane pane) :
            base(dockPanel, pane)
        {
            InitXNCForm();
            InitCForm();
            this.MinimizeBox = false;
            this.MaximizeBox = false;
            this.ShowIcon = false;
            this.FormBorderStyle = FormBorderStyle.FixedSingle;

        }

        internal protected HUSD4FloatWindow(TabDockPanel dockPanel, DockPane pane, Rectangle bounds) :
            base(dockPanel, pane, bounds)
        {
            
            InitXNCForm();
            InitCForm();

            this.MinimizeBox = false;
            this.MaximizeBox = false;
            this.ShowIcon = false;
            this.FormBorderStyle = FormBorderStyle.FixedSingle;
        }

        private bool _enableNCPaint = true;
        public override bool EnableNCPaint
        {
            get { return _enableNCPaint; }
            set
            {
                _enableNCPaint = value;
                InvalidateWindow();
            }
        }

        #region XNCForm Implement
        #region Properties
        private CaptionBox _maximizeBox;
        private CaptionBox _minimizeBox;
        private CaptionBox _closeBox;
        private CaptionMenuItem[] _menuItems;
        private CaptionBoxCollection _captionBoxCollection = new CaptionBoxCollection();


        private Icon _icon;
        private string _formStyleName = "";
        private bool _useFormStyleManager = false;

        private int _menuControlBoxOffSet = 50;

        private FormStyle _activeFormStyle;
        private FormStyle _formStyle;

        private CaptionBoxCollection CaptionBoxs
        {
            get { return _captionBoxCollection; }
        }

        [Browsable(false)]
        public CaptionMenuItem[] CaptionMenuItems
        {
            get { return _menuItems; }
            set
            {
                if (_menuItems != value)
                    _menuItems = value;
            }
        }

        public new bool MaximizeBox
        {
            get { return base.MaximizeBox; }
            set
            {
                this._maximizeBox.Enabled = value;
                this._minimizeBox.Visible = this._maximizeBox.Visible
                    = this._maximizeBox.Enabled | this._minimizeBox.Enabled;
                base.MaximizeBox = value;
            }
        }

        public new bool MinimizeBox
        {
            get { return base.MinimizeBox; }
            set
            {
                this._minimizeBox.Enabled = value;
                this._minimizeBox.Visible = this._maximizeBox.Visible
                    = this._maximizeBox.Enabled | this._minimizeBox.Enabled;
                base.MinimizeBox = value;
            }
        }

        public new bool ControlBox
        {
            get { return base.ControlBox; }
            set
            {
                //this._closeBox.Enabled = value;
                this._closeBox.Visible = value;
                base.ControlBox = value;
            }
        }

        public bool EnableCloseBox
        {
            set
            {
                if (_closeBox.Enabled != value)
                {
                    _closeBox.Enabled = value;
                }
            }
        }

        public new Icon Icon
        {
            get { return base.Icon; }
            set
            {
                if (value != Icon)
                {
                    if (this._icon != null)
                    {
                        _icon.Dispose();
                        _icon = null;
                    }
                    try
                    {
                        _icon = new Icon(value, SystemInformation.SmallIconSize);
                    }
                    catch { }
                }
                base.Icon = value;
            }
        }



        public bool UseFormStyleManager
        {
            get { return _useFormStyleManager; }
            set
            {
                if (_useFormStyleManager != value)
                    _useFormStyleManager = value;
            }
        }

        public string FormStyleName
        {
            get { return _formStyleName; }
        }

        protected FormStyle ActiveFormStyle
        {
            get { return _activeFormStyle; }
            set
            {
                if (_activeFormStyle != value)
                {
                    _activeFormStyle = value;

                    UpdateButtonAppearance();
                    InvalidateWindow();
                }
            }
        }


        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public FormStyle FormStyle
        {
            get { return _formStyle; }
            set
            {
                if (_formStyle != value)
                {
                    _formStyle = value;
                    UpdateActiveFormStyle();
                }
            }
        }
        #endregion

        private void InitXNCForm()
        {
            _closeBox = new CaptionBox();
            _closeBox.Key = "close";
            _closeBox.Visible = true;
            _closeBox.HitTestCode = (int)Win32API.NCHITTEST.HTCLOSE;
            CaptionBoxs.Add(_closeBox);

            _maximizeBox = new CaptionBox();
            _maximizeBox.Key = "maximize";
            _maximizeBox.Enabled = this.MaximizeBox;
            _maximizeBox.HitTestCode = (int)Win32API.NCHITTEST.HTMAXBUTTON;
            CaptionBoxs.Add(_maximizeBox);

            _minimizeBox = new CaptionBox();
            _minimizeBox.Key = "minimize";
            _minimizeBox.Enabled = this.MinimizeBox;
            _minimizeBox.HitTestCode = (int)Win32API.NCHITTEST.HTMINBUTTON;
            CaptionBoxs.Add(_minimizeBox);

            this._minimizeBox.Visible = this._maximizeBox.Visible
                = this._maximizeBox.Enabled | this._maximizeBox.Enabled;

            UpdateActiveFormStyle();
            FormStyleManager.StyleChanged += new EventHandler(FormStyleManager_StyleChanged);

        }

        void FormStyleManager_StyleChanged(object sender, EventArgs e)
        {
            UpdateActiveFormStyle();
        }

        private void UpdateActiveFormStyle()
        {
            FormStyle style = null;
            if (UseFormStyleManager)
            {
                // try to load specified style
                if (!String.IsNullOrEmpty(FormStyleName))
                    style = FormStyleManager.GetStyle(FormStyleName);

                // if it wasn't found try to load default style
                if (style == null)
                    style = FormStyleManager.GetDefaultStyle();
            }
            else
            {
                style = FormStyle;
            }
            ActiveFormStyle = style;
        }

        private void UpdateButtonAppearance()
        {
            if (ActiveFormStyle != null)
            {
                _closeBox.Appearance = ActiveFormStyle.CloseButtonStyle;
                _minimizeBox.Appearance = ActiveFormStyle.MinimizeButtonStyle;
                _maximizeBox.Appearance = (this.WindowState == FormWindowState.Maximized) ?
                    ActiveFormStyle.RestoreButtonStyle : ActiveFormStyle.MaximizeButtonStyle;
            }
            else
            {
                _closeBox.Appearance = null;
                _minimizeBox.Appearance = null;
                _maximizeBox.Appearance = null;
            }
        }

        #region Process Windows Messages

        #region OnNonClientAreaCalcSize
        protected override void OnNonClientAreaCalcSize(ref Rectangle bounds, bool update)
        {
            if (ActiveFormStyle == null)
                return;

            if (update)
                UpdateCaptionBoxBounds(bounds);

            Padding clientPadding = ActiveFormStyle.ClientAreaPadding;
            bounds = new Rectangle(bounds.Left + clientPadding.Left, bounds.Top + clientPadding.Top,
                bounds.Width - clientPadding.Horizontal, bounds.Height - clientPadding.Vertical);
        }

        private void UpdateCaptionBoxBounds(Rectangle windowBounds)
        {
            // start from top-right corner
            int x = windowBounds.Width;

            foreach (CaptionBox button in this.CaptionBoxs)
            {
                if (button.Visible)
                {
                    int y = button.Appearance.Margin.Top;
                    x -= (button.Appearance.Size.Width + button.Appearance.Margin.Right);
                    button.Bounds = new Rectangle(x, y,
                        button.Appearance.Size.Width, button.Appearance.Size.Height);
                    x -= button.Appearance.Margin.Left;
                }

                // background will be recreated on next redraw
                button.BackgroundImage = null;
            }

            if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
            {
                x -= _menuControlBoxOffSet;
                for (int count = CaptionMenuItems.Length - 1; count >= 0; count--)
                {
                    CaptionMenuItem item = CaptionMenuItems[count];
                    if (item.Visible)
                    {
                        int y = item.Appearance.Margin.Top;
                        x -= (item.Appearance.Size.Width + item.Appearance.Margin.Right);
                        item.Bounds = new Rectangle(x, y, item.Appearance.Size.Width, item.Appearance.Size.Height);
                        x -= item.Appearance.Margin.Left;
                    }
                }

                // create menu item's background for next draw
                if (IsHandleCreated)
                {
                    // MSDN states that only WINDOW and INTERSECTRGN are needed,
                    // but other sources confirm that CACHE is required on Win9x
                    // and you need CLIPSIBLINGS to prevent painting on overlapping windows.
                    IntPtr hDC = Win32API.GetDCEx(this.Handle, (IntPtr)1,
                        (int)(Win32API.DCX.DCX_WINDOW | Win32API.DCX.DCX_INTERSECTRGN
                            | Win32API.DCX.DCX_CACHE | Win32API.DCX.DCX_CLIPSIBLINGS));
                    try
                    {
                        if (hDC != IntPtr.Zero)
                        {
                            IntPtr srcHDC = Win32API.CreateCompatibleDC(hDC);
                            IntPtr srcBitmap = Win32API.CreateCompatibleBitmap(hDC, windowBounds.Width, windowBounds.Height);

                            try
                            {
                                Win32API.SelectObject(srcHDC, srcBitmap);

                                using (Graphics srcG = Graphics.FromHdc(srcHDC))
                                {
                                    Rectangle tempRect = new Rectangle(0, 0, windowBounds.Width, windowBounds.Height);
                                    Region clipRegion = new Region(tempRect);
                                    clipRegion.Exclude(DrawUtil.ExcludePadding(tempRect, ActiveFormStyle.ClientAreaPadding));
                                    srcG.Clip = clipRegion;
                                    ActiveFormStyle.NormalState.Draw(srcG, tempRect);

                                    foreach (CaptionMenuItem item in CaptionMenuItems)
                                    {
                                        // background will be recreated on next redraw
                                        if (item.BackgroundImage != null)
                                        {
                                            item.BackgroundImage.Dispose();
                                            item.BackgroundImage = null;
                                        }
                                        Bitmap bkImg = new Bitmap(item.Bounds.Width, item.Bounds.Height);
                                        using (Graphics destG = Graphics.FromImage(bkImg))
                                        {
                                            IntPtr destDC = destG.GetHdc();
                                            Win32API.BitBlt(destDC, 0, 0, item.Bounds.Width, item.Bounds.Height, srcHDC, item.Bounds.X, item.Bounds.Y, Win32API.TernaryRasterOperations.SRCCOPY);
                                            destG.ReleaseHdc(destDC);
                                        }
                                        item.BackgroundImage = bkImg;
                                    }
                                }
                            }
                            finally
                            {
                                Win32API.DeleteObject(srcBitmap);
                                Win32API.DeleteDC(srcHDC);
                            }
                        }
                    }
                    finally
                    {
                        Win32API.ReleaseDC(this.Handle, hDC);
                    }
                }

            }

            // Should I move this where this actually changes WM_GETMINMAXINFO ??
            //maximizeButton.Appearance = (this.WindowState == FormWindowState.Maximized) ?
            //    _borderAppearance.RestoreButton : _borderAppearance.MaximizeButton;
        }
        #endregion

        #region MouseEvent
        protected override void OnNonClientMouseLeave(EventArgs args)
        {
            if (!trakingMouse)
                return;

            foreach (CaptionBox button in this.CaptionBoxs)
            {
                if (button.State != CaptionBoxState.Normal)
                {
                    button.State = CaptionBoxState.Normal;
                    DrawBox(button);
                    UnhookMouseEvent();
                }
            }

            if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
            {
                foreach (CaptionMenuItem item in CaptionMenuItems)
                {
                    if (item.State != CaptionBoxState.Normal)
                    {
                        item.State = CaptionBoxState.Normal;
                        DrawBox(item);
                        UnhookMouseEvent();
                    }
                }
            }
        }

        protected override void OnNonClientMouseDown(NonClientMouseEventArgs args)
        {
            if (args.Button != MouseButtons.Left)
                return;

            switch (args.HitTest)
            {
                //case (int)Win32API.NCHITTEST.HTSYSMENU:
                //    {
                //        IntPtr hSysMenu = Win32API.GetSystemMenu(this.Handle, 0);
                //        if (hSysMenu != IntPtr.Zero)
                //        {
                //            Point menuLocation = this.Location;
                //            int result = 0;
                //            if(Win32API.TrackPopupMenuEx(hSysMenu,
                //               (int)Win32API.TPM.TPM_RETURNCMD | (int)Win32API.TPM.TPM_LEFTBUTTON,
                //               menuLocation.X, menuLocation.Y, Handle, null))
                //                result = 1;
                //            Win32API.SendMessage(Handle,(int)Win32API.WindowMessages.WM_SYSCOMMAND,
                //               (uint)result,0); 

                //            args.Handled = true;
                //        }
                //        break;
                //    }
                case (int)Win32API.NCHITTEST.HTCLOSE:
                    {
                        //NativeMethods.POINT pt = NativeMethods.POINT.FromPoint(args.Location);
                        if (_closeBox.Enabled && DepressBox(_closeBox))
                            Win32API.SendMessage(this.Handle,
                                (int)Win32API.WindowMessages.WM_SYSCOMMAND,
                                (IntPtr)Win32API.SystemCommands.SC_CLOSE, IntPtr.Zero);
                        args.Handled = true;
                        break;
                    }
                case (int)Win32API.NCHITTEST.HTMINBUTTON:
                    {
                        //		NativeMethods.POINT pt = NativeMethods.POINT.FromPoint(args.Location);
                        if (_minimizeBox.Enabled && DepressBox(_minimizeBox))
                            Win32API.SendMessage(this.Handle,
                                (int)Win32API.WindowMessages.WM_SYSCOMMAND,
                                (IntPtr)Win32API.SystemCommands.SC_MINIMIZE, IntPtr.Zero);
                        args.Handled = true;
                        break;
                    }
                case (int)Win32API.NCHITTEST.HTMAXBUTTON:
                    {
                        if (_maximizeBox.Enabled && DepressBox(_maximizeBox))
                        {
                            int sc = (WindowState == FormWindowState.Maximized || WindowState == FormWindowState.Minimized) ?
                                (int)Win32API.SystemCommands.SC_RESTORE : (int)Win32API.SystemCommands.SC_MAXIMIZE;

                            Win32API.SendMessage(this.Handle,
                                (int)Win32API.WindowMessages.WM_SYSCOMMAND,
                                (IntPtr)sc, IntPtr.Zero);
                        }
                        args.Handled = true;
                        break;
                    }
            }

            int testHitResult = args.HitTest;
            if ((testHitResult & (int)Win32API.NCHITTEST.HTCAPTIONMENU) == (int)Win32API.NCHITTEST.HTCAPTIONMENU)
            {
                int index = testHitResult >> 8;

                CaptionMenuItem currentItem = null;
                if (CaptionMenuItems != null && CaptionMenuItems.Length > 0 && CaptionMenuItems.Length > index)
                {
                    currentItem = CaptionMenuItems[index];
                    if (currentItem != null && currentItem.Enabled && DepressBox(currentItem))
                    {
                        currentItem.ShowDropdownMenu();
                    }
                }
                args.Handled = true;
            }
        }

        protected override void OnNonClientMouseMove(MouseEventArgs mouseEventArgs)
        {
            foreach (CaptionBox button in this.CaptionBoxs)
            {
                if (button.Visible && button.Bounds.Contains(mouseEventArgs.X, mouseEventArgs.Y) && button.HitTestCode > 0)
                {
                    if (button.State != CaptionBoxState.Over)
                    {
                        button.State = CaptionBoxState.Over;
                        DrawBox(button);
                        HookMouseEvent();
                    }
                }
                else
                {
                    if (button.State != CaptionBoxState.Normal)
                    {
                        button.State = CaptionBoxState.Normal;
                        DrawBox(button);
                        UnhookMouseEvent();
                    }
                }
            }

            if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
            {
                CaptionMenuItem shownMenuItem = null;
                foreach (CaptionMenuItem item in CaptionMenuItems)
                {
                    if (item.ContextMenu != null && item.ContextMenu.Visible)
                    {
                        shownMenuItem = item;
                        break;
                    }
                }

                foreach (CaptionMenuItem item in CaptionMenuItems)
                {
                    if (item.Visible &&
                        (item.Bounds.Contains(mouseEventArgs.X, mouseEventArgs.Y)
                        || (item.ContextMenu != null && item.ContextMenu.Visible)))
                    {
                        if (item.State != CaptionBoxState.Over)
                        {
                            item.State = CaptionBoxState.Over;
                            DrawBox(item);

                            if (shownMenuItem != null && shownMenuItem != item)
                            {
                                shownMenuItem.HideDropdownMenu();
                                item.ShowDropdownMenu();
                            }

                            HookMouseEvent();
                        }
                    }
                    else
                    {
                        if (item.State != CaptionBoxState.Normal)
                        {
                            item.State = CaptionBoxState.Normal;
                            DrawBox(item);
                            UnhookMouseEvent();
                        }
                    }
                }
            }
        }

        Win32API.TRACKMOUSEEVENT trackMouseEvent;
        bool trakingMouse = false;
        private void HookMouseEvent()
        {
            if (!trakingMouse)
            {
                trakingMouse = true;
                if (this.trackMouseEvent == null)
                {
                    this.trackMouseEvent = new Win32API.TRACKMOUSEEVENT();
                    this.trackMouseEvent.dwFlags =
                        (int)(Win32API.TrackMouseEventFalgs.TME_HOVER |
                              Win32API.TrackMouseEventFalgs.TME_LEAVE |
                              Win32API.TrackMouseEventFalgs.TME_NONCLIENT);

                    this.trackMouseEvent.hwndTrack = this.Handle;
                }
            }
        }

        private void UnhookMouseEvent()
        {
            trakingMouse = false;
        }
        #endregion

        #region OnNonClientAreaHitTest
        protected override int OnNonClientAreaHitTest(Point p)
        {
            if (ActiveFormStyle == null)
                return (int)Win32API.NCHITTEST.HTCLIENT;

            foreach (CaptionBox button in this.CaptionBoxs)
            {
                if (button.Visible && button.Bounds.Contains(p) && button.HitTestCode > 0)
                    return button.HitTestCode;
            }

            if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
            {
                for (int count = 0; count < CaptionMenuItems.Length; count++)
                {
                    CaptionMenuItem item = CaptionMenuItems[count];
                    if (item.Visible && item.Bounds.Contains(p))
                    {
                        int hitResut = (int)Win32API.NCHITTEST.HTCAPTIONMENU;
                        int itemIndex = count << 8;
                        hitResut |= itemIndex;
                        return hitResut;
                    }
                }
            }

            if (FormBorderStyle != FormBorderStyle.FixedToolWindow &&
                FormBorderStyle != FormBorderStyle.SizableToolWindow &&
                FormBorderStyle != FormBorderStyle.Sizable)
            {
                if (GetIconRectangle().Contains(p))
                    return (int)Win32API.NCHITTEST.HTSYSMENU;
            }

            if (this.FormBorderStyle == FormBorderStyle.Sizable
                || this.FormBorderStyle == FormBorderStyle.SizableToolWindow)
            {
                #region Handle sizable window borders
                if (p.X <= ActiveFormStyle.SizingBorderWidth) // left border
                {
                    if (p.Y <= ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTTOPLEFT;
                    else if (p.Y >= this.Height - ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTBOTTOMLEFT;
                    else
                        return (int)Win32API.NCHITTEST.HTLEFT;
                }
                else if (p.X >= this.Width - ActiveFormStyle.SizingBorderWidth) // right border
                {
                    if (p.Y <= ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTTOPRIGHT;
                    else if (p.Y >= this.Height - ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTBOTTOMRIGHT;
                    else
                        return (int)Win32API.NCHITTEST.HTRIGHT;
                }
                else if (p.Y <= ActiveFormStyle.SizingBorderWidth) // top border
                {
                    if (p.X <= ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTTOPLEFT;
                    if (p.X >= this.Width - ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTTOPRIGHT;
                    else
                        return (int)Win32API.NCHITTEST.HTTOP;
                }
                else if (p.Y >= this.Height - ActiveFormStyle.SizingBorderWidth) // bottom border
                {
                    if (p.X <= ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTBOTTOMLEFT;
                    if (p.X >= this.Width - ActiveFormStyle.SizingCornerOffset)
                        return (int)Win32API.NCHITTEST.HTBOTTOMRIGHT;
                    else
                        return (int)Win32API.NCHITTEST.HTBOTTOM;
                }
                #endregion
            }

            // title bar
            if (p.Y <= ActiveFormStyle.ClientAreaPadding.Top)
                return (int)Win32API.NCHITTEST.HTCAPTION;

            // rest of non client area
            if (p.X <= this.ActiveFormStyle.ClientAreaPadding.Left || p.X >= this.ActiveFormStyle.ClientAreaPadding.Right
                || p.Y >= this.ActiveFormStyle.ClientAreaPadding.Bottom)
                return (int)Win32API.NCHITTEST.HTBORDER;

            return (int)Win32API.NCHITTEST.HTCLIENT;
        }

        private Rectangle GetIconRectangle()
        {
            return new Rectangle(ActiveFormStyle.IconPadding.Left, ActiveFormStyle.IconPadding.Top, 16, 16);
        }
        #endregion

        #region OnNonClientAreaPaint
        protected override void OnNonClientAreaPaint(NonClientPaintEventArgs e)
        {

            if (ActiveFormStyle == null)
                return;

            // assign clip region to exclude client area
            Region clipRegion = new Region(e.Bounds);
            Rectangle clientRect = DrawUtil.ExcludePadding(e.Bounds, ActiveFormStyle.ClientAreaPadding);

            clipRegion.Exclude(clientRect);
            e.Graphics.Clip = clipRegion;

            // paint borders
            ActiveFormStyle.NormalState.Draw(e.Graphics, e.Bounds);

            int textOffset = 0;

            // paint icon
            if (ShowIcon &&
                FormBorderStyle != FormBorderStyle.FixedToolWindow &&
                FormBorderStyle != FormBorderStyle.SizableToolWindow)
            {
                Rectangle iconRect = GetIconRectangle();
                textOffset += iconRect.Right;

                if (_icon != null)
                    e.Graphics.DrawIconUnstretched(_icon, iconRect);
                else
                    e.Graphics.DrawIcon(Icon, iconRect);
            }


            // paint caption
            string text = this.Text;
            if (!String.IsNullOrEmpty(text))
            {
                // disable text wrapping and request elipsis characters on overflow
                using (StringFormat sf = new StringFormat())
                {
                    sf.Trimming = StringTrimming.EllipsisCharacter;
                    sf.FormatFlags = StringFormatFlags.NoWrap;
                    sf.LineAlignment = StringAlignment.Center;

                    // find position of the first button from left
                    int firstButton = e.Bounds.Width;
                    foreach (CaptionBox button in this.CaptionBoxs)
                        if (button.Visible)
                            firstButton = Math.Min(firstButton, button.Bounds.X);

                    if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
                    {
                        foreach (CaptionMenuItem item in CaptionMenuItems)
                        {
                            if (item.Visible)
                            {
                                firstButton = Math.Min(firstButton, item.Bounds.X);
                            }
                        }
                    }

                    Padding padding = ActiveFormStyle.TitlePadding;
                    Rectangle textRect = new Rectangle(textOffset + padding.Left,
                        padding.Top, firstButton - textOffset - padding.Horizontal,
                        ActiveFormStyle.ClientAreaPadding.Top - padding.Vertical);

                    Font textFont = this.Font;
                    if (ActiveFormStyle.TitleFont != null)
                        textFont = ActiveFormStyle.TitleFont;

                    if (!ActiveFormStyle.TitleShadowColor.IsEmpty)
                    {
                        Rectangle shadowRect = textRect;
                        shadowRect.Offset(1, 1);

                        // draw drop shadow
                        using (Brush b = new SolidBrush(ActiveFormStyle.TitleShadowColor))
                        {
                            e.Graphics.DrawString(text, textFont, b, shadowRect, sf);
                        }
                    }

                    if (!ActiveFormStyle.TitleColor.IsEmpty)
                    {
                        // draw text
                        using (Brush b = new SolidBrush(ActiveFormStyle.TitleColor))
                        {
                            e.Graphics.DrawString(text, textFont, b, textRect, sf);
                        }
                    }
                }
            }

            // paint buttons
            foreach (CaptionBox button in this.CaptionBoxs)
                button.Draw(e.Graphics, false);

            if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
            {
                foreach (CaptionMenuItem item in CaptionMenuItems)
                {
                    if (!e.Active)
                    {
                        item.State = CaptionBoxState.Normal;
                    }
                    item.Draw(e.Graphics, false);
                }
            }
        }
        #endregion

        protected override void UpdateWindowState()
        {
            if (ActiveFormStyle == null)
                return;

            CaptionBoxStyle newAppearance;
            if (this.WindowState == FormWindowState.Maximized)
                newAppearance = this.ActiveFormStyle.RestoreButtonStyle;
            else if (this.WindowState == FormWindowState.Minimized && !this.ShowInTaskbar)
                newAppearance = this.ActiveFormStyle.RestoreButtonStyle;
            else
                newAppearance = this.ActiveFormStyle.MaximizeButtonStyle;

            if (newAppearance != _maximizeBox.Appearance)
            {
                _maximizeBox.Appearance = newAppearance;
                DrawBox(_maximizeBox);
            }
        }

        protected override void UpdateCaptionMenuScreenPosition()
        {
            if (CaptionMenuItems != null && CaptionMenuItems.Length > 0)
            {
                for (int count = CaptionMenuItems.Length - 1; count >= 0; count--)
                {
                    CaptionMenuItem item = CaptionMenuItems[count];
                    if (item.Visible)
                    {
                        Point raw = PointToScreen(item.Bounds.Location);
                        raw.X -= 2;
                        raw.Y -= 4;
                        item.ScreenLocation = raw;
                    }
                }
            }
        }

        private void DrawBox(CaptionBox box)
        {
            if (IsHandleCreated)
            {
                // MSDN states that only WINDOW and INTERSECTRGN are needed,
                // but other sources confirm that CACHE is required on Win9x
                // and you need CLIPSIBLINGS to prevent painting on overlapping windows.
                IntPtr hDC = Win32API.GetDCEx(this.Handle, (IntPtr)1,
                    (int)(Win32API.DCX.DCX_WINDOW | Win32API.DCX.DCX_INTERSECTRGN
                        | Win32API.DCX.DCX_CACHE | Win32API.DCX.DCX_CLIPSIBLINGS));

                if (hDC != IntPtr.Zero)
                {
                    using (Graphics g = Graphics.FromHdc(hDC))
                    {
                        box.Draw(g, true);
                    }
                }

                Win32API.ReleaseDC(this.Handle, hDC);
            }
        }

        private bool DepressBox(CaptionBox currentBox)
        {
            bool result = false;
            try
            {
                // callect all mouse events until WL_BUTTONUP
                this.Capture = true;		// hopefully doeas the same as SetCapture(handle)

                // draw button in pressed mode
                currentBox.State = CaptionBoxState.Pressed;
                DrawBox(currentBox);

                // loop untill button is released
                bool done = false;
                while (!done)
                {
                    System.Threading.Thread.Sleep(1);

                    // NOTE: This struct needs to be here. We had strange error (starting from Beta 2).
                    // when this was defined at begining of this method. also check if types are correct (no overlap). 
                    Message m = new Message();
                    if (Win32API.PeekMessage(ref m, IntPtr.Zero,
                        (int)Win32API.WindowMessages.WM_MOUSEFIRST, (int)Win32API.WindowMessages.WM_MOUSELAST,
                        (int)Win32API.PeekMessageOptions.PM_REMOVE))
                    {
                        switch (m.Msg)
                        {
                            case (int)Win32API.WindowMessages.WM_LBUTTONUP:
                                {
                                    if (currentBox.State == CaptionBoxState.Pressed)
                                    {
                                        currentBox.State = CaptionBoxState.Normal;
                                        DrawBox(currentBox);
                                    }
                                    Point pt = PointToWindow(PointToScreen(new Point(m.LParam.ToInt32())));
                                    result = currentBox.Bounds.Contains(pt);
                                    done = true;
                                    break;
                                }
                            case (int)Win32API.WindowMessages.WM_MOUSEMOVE:
                                {
                                    Point clientPoint = PointToWindow(PointToScreen(new Point(m.LParam.ToInt32())));
                                    if (currentBox.Bounds.Contains(clientPoint))
                                    {
                                        if (currentBox.State == CaptionBoxState.Normal)
                                        {
                                            currentBox.State = CaptionBoxState.Pressed;
                                            DrawBox(currentBox);
                                        }
                                    }
                                    else
                                    {
                                        if (currentBox.State == CaptionBoxState.Pressed)
                                        {
                                            currentBox.State = CaptionBoxState.Normal;
                                            DrawBox(currentBox);
                                        }
                                    }
                                    done = false;
                                    break;
                                }
                        }
                    }
                }
            }
            finally
            {
                this.Capture = false;
            }

            return result;
        }

        #endregion



        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);
            if (disposing)
            {
                if (_icon != null)
                    _icon.Dispose();

                FormStyleManager.StyleChanged -= new EventHandler(FormStyleManager_StyleChanged);
                ActiveFormStyle = null;
            }
        }


        private class CaptionBoxCollection : CollectionBase
        {
            public void Add(CaptionBox button)
            {
                this.List.Add(button);
            }
        }
        #endregion

        #region CForm Implement
        private void InitCForm()
        {
            this.FormStyle = CreateFormStyle();
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            if (this.FormBorderStyle != FormBorderStyle.None)
            {
                int width = Width;
                int height = Height;

                int diam = 10;
                GraphicsPath path = new GraphicsPath();
                path.AddArc(0, 0, diam, diam, -90, -90);
                path.AddLines(new Point[] {new Point(0,diam), new Point(0, height), 
                new Point(width, height), new Point(width, diam)});
                path.AddArc(width - diam, 0, diam, diam, 0, -90);
                path.CloseFigure();

                this.Region = new Region(path);
            }
            else
            {
                if (this.WindowState == FormWindowState.Maximized)
                    this.Region = new Region(new Rectangle(0, 0, SystemInformation.PrimaryMonitorSize.Width, SystemInformation.PrimaryMonitorSize.Height));
            }
        }

        FormStyle CreateFormStyle()
        {
            FormStyle style = new FormStyle();
            ImageRenderer normalState = new ImageRenderer();
            if (RendererManager.Style == StyleEnum.Default)
                normalState.Image = Properties.Resources.border1;
            else
                normalState.Image = Properties.Resources.Border;

            normalState.Layout = ImageLayout.Stretch;
            normalState.StretchMargins = new Padding(8, 25, 8, 8);
            style.NormalState = normalState;


            style.CloseButtonStyle.Margin = new Padding(1, 5, 5, 0);
            ImageRenderer imgNRender = new ImageRenderer();
            ImageRenderer imgDRender = new ImageRenderer();
            ImageRenderer imgARender = new ImageRenderer();
            ImageRenderer imgHRender = new ImageRenderer();
            if (RendererManager.Style == StyleEnum.Default)
            {
                style.CloseButtonStyle.Size = Properties.Resources.close_1.Size;
                imgNRender.Image = Properties.Resources.close_1;
                imgDRender.Image = Properties.Resources.close_4;
                imgARender.Image = Properties.Resources.close_3;
                imgHRender.Image = Properties.Resources.close_2;
            }
            else
            {
                style.CloseButtonStyle.Size = Properties.Resources.CloseButton.Size;
                imgNRender.Image = Properties.Resources.CloseButton;
                imgDRender.Image = Properties.Resources.CloseButtonDisabled;
                imgARender.Image = Properties.Resources.CloseButtonPressed;
                imgHRender.Image = Properties.Resources.CloseButtonHover;
            }
            style.CloseButtonStyle.NormalState = imgNRender;
            style.CloseButtonStyle.DisabledState = imgDRender;
            style.CloseButtonStyle.ActiveState = imgARender;
            style.CloseButtonStyle.HoverState = imgHRender;


            style.MaximizeButtonStyle.Margin = new Padding(1, 5, 1, 0);
            imgNRender = new ImageRenderer();
            imgDRender = new ImageRenderer();
            imgARender = new ImageRenderer();
            imgHRender = new ImageRenderer();
            if (RendererManager.Style == StyleEnum.Default)
            {
                style.MaximizeButtonStyle.Size = Properties.Resources.max_1.Size;
                imgNRender.Image = Properties.Resources.max_1;
                imgDRender.Image = Properties.Resources.max_4;
                imgARender.Image = Properties.Resources.max_3;
                imgHRender.Image = Properties.Resources.max_2;
            }
            else
            {
                style.MaximizeButtonStyle.Size = Properties.Resources.MaximizeButton.Size;
                imgNRender.Image = Properties.Resources.MaximizeButton;
                imgDRender.Image = Properties.Resources.MaximizeButtonDisabled;
                imgARender.Image = Properties.Resources.MaximizeButtonPressed;
                imgHRender.Image = Properties.Resources.MaximizeButtonHover;
            }
            style.MaximizeButtonStyle.NormalState = imgNRender;
            style.MaximizeButtonStyle.DisabledState = imgDRender;
            style.MaximizeButtonStyle.ActiveState = imgARender;
            style.MaximizeButtonStyle.HoverState = imgHRender;

            style.MinimizeButtonStyle.Margin = new Padding(1, 5, 1, 0);
            imgNRender = new ImageRenderer();
            imgDRender = new ImageRenderer();
            imgARender = new ImageRenderer();
            imgHRender = new ImageRenderer();
            if (RendererManager.Style == StyleEnum.Default)
            {
                style.MinimizeButtonStyle.Size = Properties.Resources.min_1.Size;
                imgNRender.Image = Properties.Resources.min_1;
                imgDRender.Image = Properties.Resources.min_4;
                imgARender.Image = Properties.Resources.min_3;
                imgHRender.Image = Properties.Resources.min_2;
            }
            else
            {
                style.MinimizeButtonStyle.Size = Properties.Resources.MinimizeButton.Size;
                imgNRender.Image = Properties.Resources.MinimizeButton;
                imgDRender.Image = Properties.Resources.MinimizeButtonDisabled;
                imgARender.Image = Properties.Resources.MinimizeButtonPressed;
                imgHRender.Image = Properties.Resources.MinimizeButtonHover;
            }
            style.MinimizeButtonStyle.NormalState = imgNRender;
            style.MinimizeButtonStyle.DisabledState = imgDRender;
            style.MinimizeButtonStyle.ActiveState = imgARender;
            style.MinimizeButtonStyle.HoverState = imgHRender;


            style.RestoreButtonStyle.Margin = new Padding(1, 5, 1, 0);
            imgNRender = new ImageRenderer();
            imgDRender = new ImageRenderer();
            imgARender = new ImageRenderer();
            imgHRender = new ImageRenderer();
            if (RendererManager.Style == StyleEnum.Default)
            {
                style.RestoreButtonStyle.Size = Properties.Resources.return_1.Size;
                imgNRender.Image = Properties.Resources.return_1;
                imgDRender.Image = Properties.Resources.return_4;
                imgARender.Image = Properties.Resources.return_3;
                imgHRender.Image = Properties.Resources.return_2;
            }
            else
            {
                style.RestoreButtonStyle.Size = Properties.Resources.RestoreButton.Size;
                imgNRender.Image = Properties.Resources.RestoreButton;
                imgDRender.Image = Properties.Resources.RestoreButtonDisabled;
                imgARender.Image = Properties.Resources.RestoreButtonPressed;
                imgHRender.Image = Properties.Resources.RestoreButtonHover;
            }
            style.RestoreButtonStyle.NormalState = imgNRender;
            style.RestoreButtonStyle.DisabledState = imgDRender;
            style.RestoreButtonStyle.ActiveState = imgARender;
            style.RestoreButtonStyle.HoverState = imgHRender;

            style.TitleColor = Color.White;
            style.TitlePadding = new Padding(3, 0, 0, 0);
            style.TitleShadowColor = Color.DimGray;
            style.TitleFont = new System.Drawing.Font("Tahoma", 8.25F, System.Drawing.FontStyle.Bold);

            style.ClientAreaPadding = new Padding(8, 25, 8, 8);
            style.IconPadding = new Padding(3, 3, 2, 0);

            this.NonClientAreaDoubleBuffering = true;

            return style;
        }
        #endregion

    }
}
