/*
 * ckserial.c -- serial driver for CK510-EVB Board UARTS.
 *
 * Copyright (C) 2004 Li Chunqiang <chunqiang_li@c-sky.com>
 * Copyright (C) 2004 Kang Sun <sunk@vlsi.zju.edu.cn>
 * Copyright (c) 1999-2003 Greg Ungerer <gerg@snapgear.com>
 * Copyright (C) 2001-2003 SnapGear Inc. <www.snapgear.com> 
 * Copyright (c) 2000-2001 Lineo, Inc. <www.lineo.com> 
 *
 * Based on code from 68332serial.c which was:
 *
 * Copyright (C) 1995 David S. Miller (davem@caip.rutgers.edu)
 * Copyright (C) 1998 TSHG
 * Copyright (c) 1999 Rt-Control Inc. <jeff@uclinux.org>
 */
 
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/wait.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/config.h>
#include <linux/major.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/mm.h>
#include <linux/kernel.h>
#include <linux/serial.h>
#include <linux/serialP.h>
#ifdef CONFIG_LEDMAN
#include <linux/ledman.h>
#endif
#include <linux/console.h>
#include <linux/version.h>
#include <linux/init.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/system.h>
#include <asm/segment.h>
#include <asm/semaphore.h>
#include <asm/bitops.h>
#include <asm/delay.h>
#include <asm/ckcore.h>
#include <asm/cksim.h>
#include <asm/ckuart.h>
#include <asm/ckpic.h>
#include <asm/nap.h>
#include <asm/se1100.h>

#if LINUX_VERSION_CODE < 0x020100
#define queue_task_irq_off queue_task
#define copy_from_user(a,b,c) memcpy_fromfs(a,b,c)
#define copy_to_user(a,b,c) memcpy_tofs(a,b,c)
#else
#include <asm/uaccess.h>
#endif

#include "ck510evb_serial.h"

/*
 *	the only event we use
 */
#undef RS_EVENT_WRITE_WAKEUP
#define RS_EVENT_WRITE_WAKEUP 0
#undef RS_EVENT_READ_WAKEUP
#define RS_EVENT_READ_WAKEUP 1

#if LINUX_VERSION_CODE >= 0x020100
struct timer_list ck510evb_timer_struct;
#endif

/*
 *	Default console baud rate,  we use this as the default
 *	for all ports so init can just open /dev/console and
 *	keep going.  Perhaps one day the cflag settings for the
 *	console can be used instead.
 */
#if (defined(CONFIG_CK510) || defined(CONFIG_CSKY) || defined(CONFIG_CK510EVB))
#define	CONSOLE_BAUD_RATE	4800
#define	DEFAULT_CBAUD		B4800
#endif

#ifndef CONSOLE_BAUD_RATE
#define	CONSOLE_BAUD_RATE	4800
#define	DEFAULT_CBAUD		B4800
#endif

int ck510evb_console_inited = 0;
int ck510evb_console_port = -1;
int ck510evb_console_baud = CONSOLE_BAUD_RATE;
int ck510evb_console_cbaud = DEFAULT_CBAUD;

DECLARE_TASK_QUEUE(ck510evb_tq_serial);

/*
 *	Driver data structures.
 */
struct tty_driver	ck510evb_serial_driver, ck510evb_callout_driver;
static int		ck510evb_serial_refcount;

/* serial subtype definitions */
#define SERIAL_TYPE_NORMAL	1
#define SERIAL_TYPE_CALLOUT	2
  
/* number of characters left in xmit buffer before we ask for more */
#define WAKEUP_CHARS 256//128

/* Debugging...
 */
#undef SERIAL_DEBUG_OPEN
#undef SERIAL_DEBUG_FLOW

#define _INLINE_ inline

#define	IRQBASE	 (32 + 13)


/*
 *	Configuration table, UARTs to look for at startup.
 */
static struct ck510evb_serial ck510evb_table[] = {
  { 0, (CK510EVB_UART_BASE1), IRQBASE,   ASYNC_BOOT_AUTOCONF },  /* ttyS0 */
};


#define	NR_PORTS	(sizeof(ck510evb_table) / sizeof(struct ck510evb_serial))

static struct tty_struct	*ck510evb_serial_table[NR_PORTS];
static struct termios		*ck510evb_serial_termios[NR_PORTS];
static struct termios		*ck510evb_serial_termios_locked[NR_PORTS];

/*
 * This is used to figure out the divisor speeds and the timeouts.
 */
static int ck510evb_baud_table[] = {
	0, 50, 75, 110, 134, 150, 200, 300, 600, 1200, 1800, 2400, 4800,
	9600, 19200, 38400, 57600, 115200, 230400, 460800, 0
};
#define CK510EVB_BAUD_TABLE_SIZE \
			(sizeof(ck510evb_baud_table)/sizeof(ck510evb_baud_table[0]))


#ifndef MIN
#define MIN(a,b)	((a) < (b) ? (a) : (b))
#endif

#ifdef CONFIG_MAGIC_SYSRQ
/*
 *	Magic system request keys. Used for debugging...
 */
extern int	magic_sysrq_key(int ch);
#endif


/*
 * tmp_buf is used as a temporary buffer by serial_write.  We need to
 * lock it in case the copy_from_user blocks while swapping in a page,
 * and some other program tries to do a serial write at the same time.
 * Since the lock will only come under contention when the system is
 * swapping and available memory is low, it makes sense to share one
 * buffer across all the serial ports, since it significantly saves
 * memory if large numbers of serial ports are open.
 */
static unsigned char ck510evb_tmp_buf[4096]; /* This is cheating */
#if LINUX_VERSION_CODE < 0x020100
static struct semaphore ck510evb_tmp_buf_sem = MUTEX;
#else
static DECLARE_MUTEX(ck510evb_tmp_buf_sem);
#endif

/*
 *	Forward declarations...
 */
static void	ck510evb_change_speed(struct ck510evb_serial *info);
static void     ck510evb_wait_until_sent(struct tty_struct *tty, int timeout);


static inline int serial_paranoia_check(struct ck510evb_serial *info,
					dev_t device, const char *routine)
{
#ifdef SERIAL_PARANOIA_CHECK
	static const char *badmagic =
		"Warning: bad magic number for serial struct (%d, %d) in %s\n";
	static const char *badinfo =
		"Warning: null ck510evb_serial for (%d, %d) in %s\n";

	if (!info) {
		printk(badinfo, MAJOR(device), MINOR(device), routine);
		return 1;
	}
	if (info->magic != SERIAL_MAGIC) {
		printk(badmagic, MAJOR(device), MINOR(device), routine);
		return 1;
	}
#endif
	return 0;
}

/*
 *	Sets or clears DTR and RTS on the requested line.
 */
static void ck510evb_setsignals(struct ck510evb_serial *info, int dtr, int rts)
{
	volatile unsigned char	*uartp;
	unsigned long		flags;
#if 0
	
	printk("%s(%d): ck510evb_setsignals(info=%x,dtr=%d,rts=%d)\n",
		__FILE__, __LINE__, (int) info, dtr, rts);
#endif

	save_flags(flags); cli();
	if (dtr >= 0) {
#if defined(CONFIG_NETtel) && defined(CONFIG_M5307)
		if (dtr) {
			info->sigs |= TIOCM_DTR;
			ppdata &= ~(info->line ? NETtel_DTR1 : NETtel_DTR0);
		} else {
			info->sigs &= ~TIOCM_DTR;
			ppdata |= (info->line ? NETtel_DTR1 : NETtel_DTR0);
		}
		*((volatile unsigned short *) (MCF_MBAR+MCFSIM_PADAT)) = ppdata;
#endif
	}
	if (rts >= 0) {
		uartp = (volatile unsigned char *) info->addr;
		if (rts) {
			info->sigs |= TIOCM_RTS;
		} else {
			info->sigs &= ~TIOCM_RTS;
		}
	}
	restore_flags(flags);
	return;
}

/*
 *	Gets values of serial signals.
 */
static int ck510evb_getsignals(struct ck510evb_serial *info)
{
	volatile unsigned char	*uartp;
	unsigned long		flags;
	int			sigs;
#if 0
	printk("%s(%d): ck510evb_getsignals(info=%x)\n", __FILE__, __LINE__);
#endif
	save_flags(flags); cli();
	uartp = (volatile unsigned char *) info->addr;

#if defined(CONFIG_NETtel) && defined(CONFIG_M5307)
	ppdata = *((volatile unsigned short *) (MCF_MBAR+MCFSIM_PADAT));
	if (info->line == 0) {
		sigs |= (ppdata & NETtel_DCD0) ? 0 : TIOCM_CD;
		sigs |= (ppdata & NETtel_DTR0) ? 0 : TIOCM_DTR;
	} else if (info->line == 1) {
		sigs |= (ppdata & NETtel_DCD1) ? 0 : TIOCM_CD;
		sigs |= (ppdata & NETtel_DTR1) ? 0 : TIOCM_DTR;
	}
#endif
	restore_flags(flags);
	return(sigs);
}

/*
 * ------------------------------------------------------------
 * ck510evb_stop() and ck510evb_start()
 *
 * This routines are called before setting or resetting tty->stopped.
 * They enable or disable transmitter interrupts, as necessary.
 * ------------------------------------------------------------
 */
static void ck510evb_stop(struct tty_struct *tty)
{
	volatile unsigned char	*uartp;
	struct ck510evb_serial	*info = (struct ck510evb_serial *)tty->driver_data;
	unsigned long		flags;

	if (serial_paranoia_check(info, tty->device, "ck510evb_stop"))
		return;
	
	uartp = (volatile unsigned char *) info->addr;
	save_flags(flags); cli();
#if 0
	//Stop the uart
	info->cr &= ~CK510EVB_UART_LCR_TXCIE;
	uartp[CK510EVB_UART_CR] = info->cr;
#endif	//#if 0
	restore_flags(flags);
}

static void ck510evb_start(struct tty_struct *tty)
{
	volatile unsigned char	*uartp;
	struct ck510evb_serial	*info = (struct ck510evb_serial *)tty->driver_data;
	unsigned long		flags;
	
	if (serial_paranoia_check(info, tty->device, "ck510evb_start"))
		return;

	uartp = (volatile unsigned char *) info->addr;
	save_flags(flags); cli();
	if (info->xmit_cnt && info->xmit_buf) {
#if 0
		//Start the uart
		info->cr |= CK510EVB_UART_LCR_TXCIE;
		uartp[CK510EVB_UART_CR] = info->cr;
#endif //#if 0
	}
	restore_flags(flags);
}

/*
 * ----------------------------------------------------------------------
 *
 * Here starts the interrupt handling routines.  All of the following
 * subroutines are declared as inline and are folded into
 * ck510evb_uart_interrupt().  They were separated out for readability's sake.
 *
 * Note: ck510evb_uart_interrupt() is a "fast" interrupt, which means that it
 * runs with interrupts turned off.  People who may want to modify
 * ck510evb_uart_interrupt() should try to keep the interrupt handler as fast as
 * possible.  After you are done making modifications, it is not a bad
 * idea to do:
 * 
 * gcc -S -DKERNEL -Wall -Wstrict-prototypes -O6 -fomit-frame-pointer serial.c
 *
 * and look at the resulting assemble code in serial.s.
 *
 * 				- Ted Ts'o (tytso@mit.edu), 7-Mar-93
 * -----------------------------------------------------------------------
 */

/*
 * This routine is used by the interrupt handler to schedule
 * processing in the software interrupt portion of the driver.
 */
static _INLINE_ void ck510evb_sched_event(struct ck510evb_serial *info, int event)
{
	info->event |= 1 << event;
	queue_task(&info->tqueue, &ck510evb_tq_serial);
	mark_bh(CM206_BH);
}

static _INLINE_ void receive_chars(struct ck510evb_serial *info, volatile unsigned char *uartp)
{
	struct tty_struct	*tty = info->tty;
	unsigned char		status, ch;

	if (!tty)
		return;

#if defined(CONFIG_LEDMAN)
	ledman_cmd(LEDMAN_CMD_SET, info->line ? LEDMAN_COM2_RX : LEDMAN_COM1_RX);
#endif

	while ((status = uartp[CK510EVB_UART_LSR]) & CK510EVB_UART_LSR_DR) {

		if (tty->flip.count >= TTY_FLIPBUF_SIZE) {
			/*
			 * can't take any more data.  Turn off receiver
			 * so that the interrupt doesn't continually
			 * occur.  We have to make sure we turn it
			 * back on at some point though.
			 */
			break;
		}

		ch = uartp[CK510EVB_UART_RBR];
		info->stats.rx++;

#ifdef CONFIG_MAGIC_SYSRQ
		if (ck510evb_console_inited && (info->line == ck510evb_console_port)) {
			if (magic_sysrq_key(ch))
				continue;
		}
#endif

		tty->flip.count++;
		if (status & CK510EVB_UART_LSR_BI) {
			info->stats.rxbreak++;
			*tty->flip.flag_buf_ptr++ = TTY_BREAK;
		} else if (status & CK510EVB_UART_LSR_PERR) {
			info->stats.rxparity++;
			*tty->flip.flag_buf_ptr++ = TTY_PARITY;
		} else if (status & CK510EVB_UART_LSR_OVRERR) {
			info->stats.rxoverrun++;
			*tty->flip.flag_buf_ptr++ = TTY_OVERRUN;
		} else if (status & CK510EVB_UART_LSR_FERR) {
			info->stats.rxframing++;
			*tty->flip.flag_buf_ptr++ = TTY_FRAME;
		} else {
			*tty->flip.flag_buf_ptr++ = 0;
		}
		*tty->flip.char_buf_ptr++ = ch;
	}

	queue_task(&tty->flip.tqueue, &tq_timer);
	return;
	
	

}

#if 0
static _INLINE_ void transmit_chars(struct ck510evb_serial *info,
		volatile unsigned char *uartp)
{
#if defined(CONFIG_LEDMAN)
	ledman_cmd(LEDMAN_CMD_SET, info->line ? LEDMAN_COM2_TX : LEDMAN_COM1_TX);
#endif

	if (info->x_char) {
		/* Send special char - probably flow control */
		uartp[CK510EVB_UART_DR] = info->x_char;
		info->x_char = 0;
		info->stats.tx++;
	}

	if ((info->xmit_cnt <= 0) || info->tty->stopped) {
		info->cr &= ~CK510EVB_UART_CR_TXCIE;
		uartp[CK510EVB_UART_CR] = info->cr;
		return;
	}

	while (uartp[CK510EVB_UART_SR] & CK510EVB_UART_SR_TXC) {
		uartp[CK510EVB_UART_DR] = info->xmit_buf[info->xmit_tail++];
		info->xmit_tail = info->xmit_tail & (SERIAL_XMIT_SIZE-1);
		info->stats.tx++;
		if (--info->xmit_cnt <= 0)
			break;
	}

	if (info->xmit_cnt < WAKEUP_CHARS)
		ck510evb_sched_event(info, RS_EVENT_WRITE_WAKEUP);
	return;
}
#endif	//#if 0

/*
 * This is the serial driver's generic interrupt routine
 */
void ck510evb_uart_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct ck510evb_serial	*info;
	unsigned char			isr;
	volatile unsigned char  *uartp;

	info = &ck510evb_table[(irq - IRQBASE)];
	uartp = (volatile unsigned char *)info->addr;

	isr = uartp[CK510EVB_UART_IIR];

	if (isr & CK510EVB_UART_IIR_RDA)
	{
		receive_chars(info, uartp);
	}
	/*
	if (isr & CK510EVB_UART_IIR_THRE)
	{
		transmit_chars(info, uartp);
	}
	*/
	return;
}

/*
 * -------------------------------------------------------------------
 * Here ends the serial interrupt routines.
 * -------------------------------------------------------------------
 */

/*
 * This routine is used to handle the "bottom half" processing for the
 * serial driver, known also the "software interrupt" processing.
 * This processing is done at the kernel interrupt level, after the
 * ck510evb_uart_interrupt() has returned, BUT WITH INTERRUPTS TURNED ON.  This
 * is where time-consuming activities which can not be done in the
 * interrupt driver proper are done; the interrupt driver schedules
 * them using ck510evb_sched_event(), and they get done here.
 */
static void do_serial_bh(void)
{
	run_task_queue(&ck510evb_tq_serial);
}

static void do_softint(void *private_)
{
	struct ck510evb_serial	*info = (struct ck510evb_serial *) private_;
	struct tty_struct	*tty;
	
	tty = info->tty;
	if (!tty)
		return;

	if (test_and_clear_bit(RS_EVENT_WRITE_WAKEUP, &info->event)) {
		if ((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) &&
		    tty->ldisc.write_wakeup)
			(tty->ldisc.write_wakeup)(tty);
		wake_up_interruptible(&tty->write_wait);
	}

#if 0
	if (test_and_clear_bit(RS_EVENT_READ_WAKEUP, &info->event)) {
		if ((info->cr & CK510EVB_UART_CR_RXCIE) == 0) {
			info->cr |= CK510EVB_UART_CR_RXCIE;
			((volatile unsigned char *) info->addr)[CK510EVB_UART_CR] = info->cr;
		}
	}
#endif //#if 0
}


/*
 *	Change of state on a DCD line.
 */
void ck510evb_modem_change(struct ck510evb_serial *info, int dcd)
{
	if (info->count == 0)
		return;

	if (info->flags & ASYNC_CHECK_CD) {
		if (dcd) {
			wake_up_interruptible(&info->open_wait);
		} else if (!((info->flags & ASYNC_CALLOUT_ACTIVE) &&
		    (info->flags & ASYNC_CALLOUT_NOHUP))) {
			schedule_task(&info->tqueue_hangup);
		}
	}
}


#if defined(CK510EVB_HAVEDCD0) || defined(CK510EVB_HAVEDCD1)

unsigned int	ck510evb_olddcd0;

/*
 * This subroutine is called when the RS_TIMER goes off. It is used
 * to monitor the state of the DCD lines - since they have no edge
 * sensors and interrupt generators.
 */
#if LINUX_VERSION_CODE < 0x020100
static void ck510evb_timer(void)
#else
static void ck510evb_timer(unsigned long arg)
#endif
{
	unsigned int dcd;

	dcd = ck510evb_getppdcd(0);
	if (dcd != ck510evb_olddcd0)
		ck510evb_modem_change(&ck510evb_table[0], dcd);
	ck510evb_olddcd0 = dcd;

/*	dcd = ck510evb_getppdcd(1);
	if (dcd != ck510evb_olddcd1)
		ck510evb_modem_change(&ck510evb_table[1], dcd);
	ck510evb_olddcd1 = dcd;
*/
	/* Re-arm timer */
#if LINUX_VERSION_CODE < 0x020100
	timer_table[RS_TIMER].expires = jiffies + HZ/25;
	timer_active |= 1 << RS_TIMER;
#else
	ck510evb_timer_struct.expires = jiffies + HZ/25;
	add_timer(&ck510evb_timer_struct);
#endif
}

#endif	/* CK510EVB_HAVEDCD0 || CK510EVB_HAVEDCD1 */


/*
 * This routine is called from the scheduler tqueue when the interrupt
 * routine has signalled that a hangup has occurred.  The path of
 * hangup processing is:
 *
 * 	serial interrupt routine -> (scheduler tqueue) ->
 * 	do_serial_hangup() -> tty->hangup() -> ck510evb_hangup()
 * 
 */
static void do_serial_hangup(void *private_)
{
	struct ck510evb_serial	*info = (struct ck510evb_serial *) private_;
	struct tty_struct	*tty;
	
	tty = info->tty;
	if (!tty)
		return;

	tty_hangup(tty);
}

static int startup(struct ck510evb_serial * info)
{
	volatile unsigned char	*uartp;
	unsigned long		flags;
	
	if (info->flags & ASYNC_INITIALIZED)
		return 0;

	if (!info->xmit_buf) {
		info->xmit_buf = (unsigned char *) get_free_page(GFP_KERNEL);
		if (!info->xmit_buf)
			return -ENOMEM;
	}

	save_flags(flags); cli();

#ifdef SERIAL_DEBUG_OPEN
	printk("starting up ttyS%d (irq %d)...\n", info->line, info->irq);
#endif

	/*
	 *	Reset UART, get it into known state...
	 */
	uartp = (volatile unsigned char *) info->addr;
 	/* Cause no reset interface on ckcore uart, i have to reset it manually */
	/* select DLR */
	uartp[CK510EVB_UART_LCR] = CK510EVB_UART_LCR_DLAEN | CK510EVB_UART_LCR_WLEN8;
	uartp[CK510EVB_UART_DLL] = 0;       /* reset DLL to 0 */
	uartp[CK510EVB_UART_DLH] = 0;       /* reset DLH to 0 */
	uartp[CK510EVB_UART_LCR] = CK510EVB_UART_LCR_WLEN8; /* reset LCR to 0x03 */
	uartp[CK510EVB_UART_THR] = 0;       /* reset THR to 0 ?????????????????????????*/
	uartp[CK510EVB_UART_IER] = 0;       /* reset IER to 0 */

	ck510evb_setsignals(info, 1, 1);

	if (info->tty)
		clear_bit(TTY_IO_ERROR, &info->tty->flags);
	info->xmit_cnt = info->xmit_head = info->xmit_tail = 0;

	/*
	 * and set the speed of the serial port
	 */
	ck510evb_change_speed(info);

	/*
	 * Lastly enable the UART transmitter and receiver, and
	 * interrupt enables.
	 */
	info->imr = CK510EVB_UART_IER_ERDAI;
	uartp[CK510EVB_UART_IER] = info->imr;

	info->flags |= ASYNC_INITIALIZED;
	restore_flags(flags);
	return 0;
}

/*
 * This routine will shutdown a serial port; interrupts are disabled, and
 * DTR is dropped if the hangup on close termio flag is on.
 */
static void shutdown(struct ck510evb_serial * info)
{
	volatile unsigned char	*uartp;
	unsigned long		flags;

	if (!(info->flags & ASYNC_INITIALIZED))
		return;

#ifdef SERIAL_DEBUG_OPEN
	printk("Shutting down serial port %d (irq %d)....\n", info->line,
	       info->irq);
#endif
	
	uartp = (volatile unsigned char *) info->addr;
	save_flags(flags); cli(); /* Disable interrupts */

	uartp[CK510EVB_UART_IER] = 0;  /* mask all interrupts & reset the uart */

	if (!info->tty || (info->tty->termios->c_cflag & HUPCL))
		ck510evb_setsignals(info, 0, 0);

	if (info->xmit_buf) {
		free_page((unsigned long) info->xmit_buf);
		info->xmit_buf = 0;
	}

	if (info->tty)
		set_bit(TTY_IO_ERROR, &info->tty->flags);
	
	info->flags &= ~ASYNC_INITIALIZED;
	restore_flags(flags);
}


/*
 * This routine is called to set the UART divisor registers to match
 * the specified baud rate for a serial port.
 */
static void ck510evb_change_speed(struct ck510evb_serial *info)
{
	volatile unsigned char	*uartp;
	unsigned int		baudclk, cflag;
	unsigned long		flags;
	//unsigned char		cr1;
	int			i;

	if (!info->tty || !info->tty->termios)
		return;
	cflag = info->tty->termios->c_cflag;
	if (info->addr == 0)
		return;

#if 0
	printk("%s(%d): ck510evb_change_speed()\n", __FILE__, __LINE__);
#endif

	i = cflag & CBAUD;
	if (i & CBAUDEX) {
		i &= ~CBAUDEX;
		if (i < 1 || i > 4)
			info->tty->termios->c_cflag &= ~CBAUDEX;
		else
			i += 15;
	}
	if (i == 0) {
		ck510evb_setsignals(info, 0, -1);
		return;
	}

	/* Caculate the baudrate clock */
	baudclk = ((CK_BUSCLK / ck510evb_baud_table[i]) >> CK510EVB_UART_DIV) ;
	info->baud = ck510evb_baud_table[i];
#if 0
	cr1 = CK510EVB_UART_LCR_RXCIE;
	cr1 &= ~CK510EVB_UART_LCR_9BIT;    /* 8 bit character */
#endif


	uartp = (volatile unsigned char *) info->addr;

	save_flags(flags); cli();
#if 0
	printk("%s(%d): mr1=%x mr2=%x baudclk=%x\n", __FILE__, __LINE__,
		mr1, mr2, baudclk);
#endif
	uartp[CK510EVB_UART_LCR]  = CK510EVB_UART_LCR_DLAEN | CK510EVB_UART_LCR_WLEN8; /* select DLAR */
	uartp[CK510EVB_UART_DLL]  = *((unsigned char *)(&baudclk) + 3);
	if(baudclk & 0xff00)
	{
		uartp[CK510EVB_UART_DLH] = *((unsigned char *)(&baudclk) + 2);
	}
	uartp[CK510EVB_UART_LCR] = CK510EVB_UART_LCR_PEN | CK510EVB_UART_LCR_WLEN8;	/* Odd parity and 8 bits char */
	uartp[CK510EVB_UART_FCR] = CK510EVB_UART_FCR_FIFO1B; /* 1 byte FIFO */
	ck510evb_setsignals(info, 1, -1);
	restore_flags(flags);
	return;
}

static void ck510evb_flush_chars(struct tty_struct *tty)
{
	volatile unsigned char	*uartp;
	struct ck510evb_serial	*info = (struct ck510evb_serial *)tty->driver_data;
	unsigned long		flags;
        unsigned long           i;

	if (serial_paranoia_check(info, tty->device, "ck510evb_flush_chars"))
	{
		return;
	}
	/*
	 * re-enable receiver interrupt
	 */
#if 0
	uartp = (volatile unsigned char *) info->addr;
	save_flags(flags); cli();
	restore_flags(flags);
#endif //#if 0
	
	if (info->xmit_cnt <= 0 || tty->stopped || tty->hw_stopped ||
	    !info->xmit_buf)
		return;

	/* Enable transmitter */
	uartp = (volatile unsigned char *) info->addr;
	save_flags(flags); cli();
	/* Cause transmitter ready interrupt is not provided,
	we did not perform uart wrting in an asynchronized
	way. The write procedure is in ck510evb_wirte. */
	if (info->x_char){
		/* Send special char - probably flow control */
		uartp[CK510EVB_UART_THR] = info->x_char;
		info->x_char = 0;
		info->stats.tx++;
	}

	if ((info->xmit_cnt <= 0) || info->tty->stopped)
	{
		return;
	}
	/* use do-while for the case of LSR_THRE bit */
	while (1)
	{
		for (i = 0; i < 0x10000; i++)
		{
			if (uartp[CK510EVB_UART_LSR] & CK510EVB_UART_LSR_THRE)
			{
				break;
			}
		}
		uartp[CK510EVB_UART_THR] = info->xmit_buf[info->xmit_tail++];
		info->xmit_tail = info->xmit_tail & (SERIAL_XMIT_SIZE - 1);
		if (--info->xmit_cnt <= 0)
		{
			break;
		}
	}
	if (info->xmit_cnt < WAKEUP_CHARS)
		ck510evb_sched_event(info, RS_EVENT_WRITE_WAKEUP);
     /*	info->cr |= CK510EVB_UART_CR_TXCIE;
	uartp[CK510EVB_UART_CR] = info->cr; */
	restore_flags(flags);
}

static int ck510evb_write(struct tty_struct * tty, int from_user,
		    const unsigned char *buf, int count)
{
	volatile unsigned char	*uartp;
	struct ck510evb_serial	*info = (struct ck510evb_serial *)tty->driver_data;
	unsigned long			flags;
	int						c, total = 0;
	unsigned long			i;

#if 0
	printk("%s(%d): ck510evb_write(tty=%x,from_user=%d,buf=%x,count=%d)\n",
		__FILE__, __LINE__, tty, from_user, buf, count);
#endif

	if (serial_paranoia_check(info, tty->device, "ck510evb_write"))
		return 0;

	if (!tty || !info->xmit_buf)
		return 0;
	
	save_flags(flags);
	while (1) {
		cli();		
		c = MIN(count, MIN(SERIAL_XMIT_SIZE - info->xmit_cnt - 1,
				   SERIAL_XMIT_SIZE - info->xmit_head));

		if (c <= 0) {
			restore_flags(flags);
			break;
		}

		if (from_user) {
			down(&ck510evb_tmp_buf_sem);
			copy_from_user(ck510evb_tmp_buf, buf, c);
			restore_flags(flags);
			cli();		
			c = MIN(c, MIN(SERIAL_XMIT_SIZE - info->xmit_cnt - 1,
				       SERIAL_XMIT_SIZE - info->xmit_head));
			memcpy(info->xmit_buf + info->xmit_head, ck510evb_tmp_buf, c);
			up(&ck510evb_tmp_buf_sem);
		} else
			memcpy(info->xmit_buf + info->xmit_head, buf, c);

		info->xmit_head = (info->xmit_head + c) & (SERIAL_XMIT_SIZE-1);
		info->xmit_cnt += c;
		restore_flags(flags);

		buf += c;
		count -= c;
		total += c;
	}

	if (!tty->stopped) {
		uartp = (volatile unsigned char *) info->addr;
		cli();
		/* Cause transmitter ready interrupt is not provided,
		we did not perform uart wrting in an asynchronized
		way. The write procedure is in ck510evb_wirte.
		sunkang 2004.11.25 */
  		if (info->x_char){
			/* Send special char - probably flow control */
			uartp[CK510EVB_UART_THR] = info->x_char;
			info->x_char = 0;
			info->stats.tx++;
		}
       
		if ((info->xmit_cnt <= 0) || info->tty->stopped)
		{
			return 0;
		}
		/* use do-while for the case of LSR_THRE bit */
		while (1)
		{
			for (i = 0; i < 0x10000; i++)
			{
				if (uartp[CK510EVB_UART_LSR] & CK510EVB_UART_LSR_THRE)
				{
					break;
				}
			}
			uartp[CK510EVB_UART_THR] = info->xmit_buf[info->xmit_tail++];
			info->xmit_tail = info->xmit_tail & (SERIAL_XMIT_SIZE - 1);
			if (--info->xmit_cnt <= 0)
			{
				break;
			}
		}
		if (info->xmit_cnt < WAKEUP_CHARS)
		{
			ck510evb_sched_event(info, RS_EVENT_WRITE_WAKEUP);
		}
		restore_flags(flags);
	}

	return total;
}

static int ck510evb_write_room(struct tty_struct *tty)
{
	struct ck510evb_serial *info = (struct ck510evb_serial *)tty->driver_data;
	int	ret;
				
	if (serial_paranoia_check(info, tty->device, "ck510evb_write_room"))
		return 0;
	ret = SERIAL_XMIT_SIZE - info->xmit_cnt - 1;
	if (ret < 0)
		ret = 0;
	return ret;
}

static int ck510evb_chars_in_buffer(struct tty_struct *tty)
{
	struct ck510evb_serial *info = (struct ck510evb_serial *)tty->driver_data;
					
	if (serial_paranoia_check(info, tty->device, "ck510evb_chars_in_buffer"))
		return 0;

	return (info->xmit_cnt);
}

static void ck510evb_flush_buffer(struct tty_struct *tty)
{
	struct ck510evb_serial	*info = (struct ck510evb_serial *)tty->driver_data;
	unsigned long		flags;
				
	if (serial_paranoia_check(info, tty->device, "ck510evb_flush_buffer"))
		return;

	save_flags(flags); cli();
	info->xmit_cnt = info->xmit_head = info->xmit_tail = 0;
	restore_flags(flags);

	wake_up_interruptible(&tty->write_wait);
	if ((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) &&
	    tty->ldisc.write_wakeup)
		(tty->ldisc.write_wakeup)(tty);
}

/*
 * ------------------------------------------------------------
 * ck510evb_throttle()
 * 
 * This routine is called by the upper-layer tty layer to signal that
 * incoming characters should be throttled.
 * ------------------------------------------------------------
 */
static void ck510evb_throttle(struct tty_struct * tty)
{
	struct ck510evb_serial *info = (struct ck510evb_serial *)tty->driver_data;
#ifdef SERIAL_DEBUG_THROTTLE
	printk("ck510evb_throttle line=%d chars=%d\n", info->line,
	       tty->ldisc.chars_in_buffer(tty));
#endif

	if (serial_paranoia_check(info, tty->device, "ck510evb_throttle"))
		return;
	
	if (I_IXOFF(tty)) {
		/* Force STOP_CHAR (xoff) out */
		volatile unsigned char	*uartp;
		unsigned long			flags;
		info->x_char = STOP_CHAR(tty);
		uartp = (volatile unsigned char *) info->addr;
		save_flags(flags); cli();
		restore_flags(flags);
	}

	/* Turn off RTS line */
	if (tty->termios->c_cflag & CRTSCTS)
		ck510evb_setsignals(info, -1, 0);
}

static void ck510evb_unthrottle(struct tty_struct * tty)
{
	struct ck510evb_serial *info = (struct ck510evb_serial *)tty->driver_data;
#ifdef SERIAL_DEBUG_THROTTLE
	printk("ck510evb_unthrottle line=%d chars=%d\n", info->line,
	       tty->ldisc.chars_in_buffer(tty));
#endif

	if (serial_paranoia_check(info, tty->device, "ck510evb_unthrottle"))
		return;
	
	if (I_IXOFF(tty)) {
		if (info->x_char)
			info->x_char = 0;
		else {
			/* Force START_CHAR (xon) out */
			volatile unsigned char	*uartp;
			unsigned long		flags;
			info->x_char = START_CHAR(tty);
			uartp = (volatile unsigned char *) info->addr;
			save_flags(flags); cli();
			restore_flags(flags);
		}
	}

	/* Assert RTS line */
	if (tty->termios->c_cflag & CRTSCTS)
		ck510evb_setsignals(info, -1, 1);
}

/*
 * ------------------------------------------------------------
 * ck510evb_ioctl() and friends
 * ------------------------------------------------------------
 */

static int get_serial_info(struct ck510evb_serial * info,
			   struct serial_struct * retinfo)
{
	struct serial_struct tmp;
  
	if (!retinfo)
		return -EFAULT;
	memset(&tmp, 0, sizeof(tmp));
	tmp.type = info->type;
	tmp.line = info->line;
	tmp.port = info->addr;
	tmp.irq = info->irq;
	tmp.flags = info->flags;
	tmp.baud_base = info->baud_base;
	tmp.close_delay = info->close_delay;
	tmp.closing_wait = info->closing_wait;
	tmp.custom_divisor = info->custom_divisor;
	copy_to_user(retinfo,&tmp,sizeof(*retinfo));
	return 0;
}

static int set_serial_info(struct ck510evb_serial * info,
			   struct serial_struct * new_info)
{
	struct serial_struct new_serial;
	struct ck510evb_serial old_info;
	int 			retval = 0;

	if (!new_info)
		return -EFAULT;
	copy_from_user(&new_serial,new_info,sizeof(new_serial));
	old_info = *info;

	if (!suser()) {
		if ((new_serial.baud_base != info->baud_base) ||
		    (new_serial.type != info->type) ||
		    (new_serial.close_delay != info->close_delay) ||
		    ((new_serial.flags & ~ASYNC_USR_MASK) !=
		     (info->flags & ~ASYNC_USR_MASK)))
			return -EPERM;
		info->flags = ((info->flags & ~ASYNC_USR_MASK) |
			       (new_serial.flags & ASYNC_USR_MASK));
		info->custom_divisor = new_serial.custom_divisor;
		goto check_and_exit;
	}

	if (info->count > 1)
		return -EBUSY;

	/*
	 * OK, past this point, all the error checking has been done.
	 * At this point, we start making changes.....
	 */

	info->baud_base = new_serial.baud_base;
	info->flags = ((info->flags & ~ASYNC_FLAGS) |
			(new_serial.flags & ASYNC_FLAGS));
	info->type = new_serial.type;
	info->close_delay = new_serial.close_delay;
	info->closing_wait = new_serial.closing_wait;

check_and_exit:
	retval = startup(info);
	return retval;
}

/*
 * get_lsr_info - get line status register info
 *
 * Purpose: Let user call ioctl() to get info when the UART physically
 * 	    is emptied.  On bus types like RS485, the transmitter must
 * 	    release the bus after transmitting. This must be done when
 * 	    the transmit shift register is empty, not be done when the
 * 	    transmit holding register is empty.  This functionality
 * 	    allows an RS485 driver to be written in user space. 
 */
static int get_lsr_info(struct ck510evb_serial * info, unsigned int *value)
{
	volatile unsigned char	*uartp;
	unsigned long		flags;
	unsigned char		status;

	uartp = (volatile unsigned char *) info->addr;
	save_flags(flags); cli();
	status = (uartp[CK510EVB_UART_LSR] & CK510EVB_UART_LSR_THRE) ? TIOCSER_TEMT : 0;
	restore_flags(flags);

	put_user(status,value);
	return 0;
}

/*
 * This routine sends a break character out the serial port.
 */
static void send_break(	struct ck510evb_serial * info, int duration)
{
  // do not support this function in CK510-EVB board
}

static int ck510evb_ioctl(struct tty_struct *tty, struct file * file,
		    unsigned int cmd, unsigned long arg)
{
	struct ck510evb_serial * info = (struct ck510evb_serial *)tty->driver_data;
	unsigned int val;
	int retval, error;
	int dtr, rts;

	if (serial_paranoia_check(info, tty->device, "ck510evb_ioctl"))
		return -ENODEV;

	if ((cmd != TIOCGSERIAL) && (cmd != TIOCSSERIAL) &&
	    (cmd != TIOCSERCONFIG) && (cmd != TIOCSERGWILD)  &&
	    (cmd != TIOCSERSWILD) && (cmd != TIOCSERGSTRUCT)) {
		if (tty->flags & (1 << TTY_IO_ERROR))
		    return -EIO;
	}
	
	switch (cmd) {
		case TCSBRK:	/* SVID version: non-zero arg --> no break */
			retval = tty_check_change(tty);
			if (retval)
				return retval;
			tty_wait_until_sent(tty, 0);
			if (!arg)
				send_break(info, HZ/4);	/* 1/4 second */
			return 0;
		case TCSBRKP:	/* support for POSIX tcsendbreak() */
			retval = tty_check_change(tty);
			if (retval)
				return retval;
			tty_wait_until_sent(tty, 0);
			send_break(info, arg ? arg*(HZ/10) : HZ/4);
			return 0;
		case TIOCGSOFTCAR:
			error = verify_area(VERIFY_WRITE, (void *) arg,sizeof(long));
			if (error)
				return error;
			put_user(C_CLOCAL(tty) ? 1 : 0,
				    (unsigned long *) arg);
			return 0;
		case TIOCSSOFTCAR:
			get_user(arg, (unsigned long *) arg);
			tty->termios->c_cflag =
				((tty->termios->c_cflag & ~CLOCAL) |
				 (arg ? CLOCAL : 0));
			return 0;
		case TIOCGSERIAL:
			error = verify_area(VERIFY_WRITE, (void *) arg,
						sizeof(struct serial_struct));
			if (error)
				return error;
			return get_serial_info(info,
					       (struct serial_struct *) arg);
		case TIOCSSERIAL:
			return set_serial_info(info,
					       (struct serial_struct *) arg);
		case TIOCSERGETLSR: /* Get line status register */
			error = verify_area(VERIFY_WRITE, (void *) arg,
				sizeof(unsigned int));
			if (error)
				return error;
			else
			    return get_lsr_info(info, (unsigned int *) arg);

		case TIOCSERGSTRUCT:
			error = verify_area(VERIFY_WRITE, (void *) arg,
						sizeof(struct ck510evb_serial));
			if (error)
				return error;
			copy_to_user((struct mcf_serial *) arg,
				    info, sizeof(struct ck510evb_serial));
			return 0;
			
		case TIOCMGET:
			if ((error = verify_area(VERIFY_WRITE, (void *) arg,
                            sizeof(unsigned int))))
                                return(error);
			val = ck510evb_getsignals(info);
			put_user(val, (unsigned int *) arg);
			break;

                case TIOCMBIS:
			if ((error = verify_area(VERIFY_WRITE, (void *) arg,
                            sizeof(unsigned int))))
				return(error);

#if LINUX_VERSION_CODE < 0x020100
			val = get_user((unsigned int *) arg);
#else
			get_user(val, (unsigned int *) arg);
#endif
			rts = (val & TIOCM_RTS) ? 1 : -1;
			dtr = (val & TIOCM_DTR) ? 1 : -1;
			ck510evb_setsignals(info, dtr, rts);
			break;

                case TIOCMBIC:
			if ((error = verify_area(VERIFY_WRITE, (void *) arg,
                            sizeof(unsigned int))))
				return(error);
#if LINUX_VERSION_CODE < 0x020100
			val = get_user((unsigned int *) arg);
#else
			get_user(val, (unsigned int *) arg);
#endif
			rts = (val & TIOCM_RTS) ? 0 : -1;
			dtr = (val & TIOCM_DTR) ? 0 : -1;
			ck510evb_setsignals(info, dtr, rts);
			break;

                case TIOCMSET:
			if ((error = verify_area(VERIFY_WRITE, (void *) arg,
                            sizeof(unsigned int))))
				return(error);
#if LINUX_VERSION_CODE < 0x020100
			val = get_user((unsigned int *) arg);
#else
			get_user(val, (unsigned int *) arg);
#endif
			rts = (val & TIOCM_RTS) ? 1 : 0;
			dtr = (val & TIOCM_DTR) ? 1 : 0;
			ck510evb_setsignals(info, dtr, rts);
			break;

#ifdef TIOCSET422
		case TIOCSET422:
#if LINUX_VERSION_CODE < 0x020100
			val = get_user((unsigned int *) arg);
#else
			get_user(val, (unsigned int *) arg);
#endif
		/*	mcf_setpa(MCFPP_PA11, (val ? 0 : MCFPP_PA11));
		*/	break;
		case TIOCGET422:
		/*	val = (mcf_getpa() & MCFPP_PA11) ? 0 : 1;
		*/	put_user(val, (unsigned int *) arg);
			break;
#endif

		default:
			return -ENOIOCTLCMD;
		}
	return 0;
}

static void ck510evb_set_termios(struct tty_struct *tty, struct termios *old_termios)
{
	struct ck510evb_serial *info = (struct ck510evb_serial *)tty->driver_data;

	if (tty->termios->c_cflag == old_termios->c_cflag)
		return;

	ck510evb_change_speed(info);

	if ((old_termios->c_cflag & CRTSCTS) &&
	    !(tty->termios->c_cflag & CRTSCTS)) {
		tty->hw_stopped = 0;
		ck510evb_setsignals(info, -1, 1);
	}
}

/*
 * ------------------------------------------------------------
 * ck510evb_close()
 * 
 * This routine is called when the serial port gets closed.  First, we
 * wait for the last remaining data to be sent.  Then, we unlink its
 * S structure from the interrupt chain if necessary, and we free
 * that IRQ if nothing is left in the chain.
 * ------------------------------------------------------------
 */
static void ck510evb_close(struct tty_struct *tty, struct file * filp)
{
	volatile unsigned char	*uartp;
	struct ck510evb_serial	*info = (struct ck510evb_serial *)tty->driver_data;
	unsigned long		flags;

	if (!info || serial_paranoia_check(info, tty->device, "ck510evb_close"))
		return;
	
	save_flags(flags); cli();
	
	if (tty_hung_up_p(filp)) {
		restore_flags(flags);
		return;
	}
	
#ifdef SERIAL_DEBUG_OPEN
	printk("ck510evb_close ttyS%d, count = %d\n", info->line, info->count);
#endif
	if ((tty->count == 1) && (info->count != 1)) {
		/*
		 * Uh, oh.  tty->count is 1, which means that the tty
		 * structure will be freed.  Info->count should always
		 * be one in these conditions.  If it's greater than
		 * one, we've got real problems, since it means the
		 * serial port won't be shutdown.
		 */
		printk("ck510evb_close: bad serial port count; tty->count is 1, "
		       "info->count is %d\n", info->count);
		info->count = 1;
	}
	if (--info->count < 0) {
		printk("ck510evb_close: bad serial port count for ttyS%d: %d\n",
		       info->line, info->count);
		info->count = 0;
	}
	if (info->count) {
		restore_flags(flags);
		return;
	}
	info->flags |= ASYNC_CLOSING;

	/*
	 * Save the termios structure, since this port may have
	 * separate termios for callout and dialin.
	 */
	if (info->flags & ASYNC_NORMAL_ACTIVE)
		info->normal_termios = *tty->termios;
	if (info->flags & ASYNC_CALLOUT_ACTIVE)
		info->callout_termios = *tty->termios;

	/*
	 * Now we wait for the transmit buffer to clear; and we notify 
	 * the line discipline to only process XON/XOFF characters.
	 */
	tty->closing = 1;
	if (info->closing_wait != ASYNC_CLOSING_WAIT_NONE)
		tty_wait_until_sent(tty, info->closing_wait);

	/*
	 * At this point we stop accepting input.  To do this, we
	 * disable the receive line status interrupts, and tell the
	 * interrupt driver to stop checking the data ready bit in the
	 * line status register.
	 */
	info->imr &= ~CK510EVB_UART_IER_ERDAI;
	uartp = (volatile unsigned char *) info->addr;
	uartp[CK510EVB_UART_IER] = info->imr;

	shutdown(info);
	if (tty->driver.flush_buffer)
		tty->driver.flush_buffer(tty);
	if (tty->ldisc.flush_buffer)
		tty->ldisc.flush_buffer(tty);
	tty->closing = 0;
	info->event = 0;
	info->tty = 0;
	info->imr = 0;
	if (tty->ldisc.num != ldiscs[N_TTY].num) {
		if (tty->ldisc.close)
			(tty->ldisc.close)(tty);
		tty->ldisc = ldiscs[N_TTY];
		tty->termios->c_line = N_TTY;
		if (tty->ldisc.open)
			(tty->ldisc.open)(tty);
	}
	if (info->blocked_open) {
		if (info->close_delay) {
			current->state = TASK_INTERRUPTIBLE;
#if LINUX_VERSION_CODE < 0x020100
			current->timeout = jiffies + info->close_delay;
			schedule();
#else
			schedule_timeout(info->close_delay);
#endif
		}
		wake_up_interruptible(&info->open_wait);
	}
	info->flags &= ~(ASYNC_NORMAL_ACTIVE|ASYNC_CALLOUT_ACTIVE|
			 ASYNC_CLOSING);
	wake_up_interruptible(&info->close_wait);
	restore_flags(flags);
}

/*
 * ck510evb_wait_until_sent() --- wait until the transmitter is empty
 */
static void
ck510evb_wait_until_sent(struct tty_struct *tty, int timeout)
{

}
		
/*
 * ck510evb_hangup() --- called by tty_hangup() when a hangup is signaled.
 */
void ck510evb_hangup(struct tty_struct *tty)
{
	struct ck510evb_serial * info = (struct ck510evb_serial *)tty->driver_data;
	
	if (serial_paranoia_check(info, tty->device, "ck510evb_hangup"))
		return;
	
	ck510evb_flush_buffer(tty);
	shutdown(info);
	info->event = 0;
	info->count = 0;
	info->flags &= ~(ASYNC_NORMAL_ACTIVE|ASYNC_CALLOUT_ACTIVE);
	info->tty = 0;
	wake_up_interruptible(&info->open_wait);
}

/*
 * ------------------------------------------------------------
 * ck510evb_open() and friends
 * ------------------------------------------------------------
 */
static int block_til_ready(struct tty_struct *tty, struct file * filp,
			   struct ck510evb_serial *info)
{
#if LINUX_VERSION_CODE < 0x020100
	struct wait_queue wait = { current, NULL };
#else
	DECLARE_WAITQUEUE(wait, current);
#endif
	int		retval;
	int		do_clocal = 0;

	/*
	 * If the device is in the middle of being closed, then block
	 * until it's done, and then try again.
	 */
	if (info->flags & ASYNC_CLOSING) {
		if (filp->f_flags & O_NONBLOCK)
			return -EAGAIN;
		interruptible_sleep_on(&info->close_wait);
#ifdef SERIAL_DO_RESTART
		if (info->flags & ASYNC_HUP_NOTIFY)
			return -EAGAIN;
		else
			return -ERESTARTSYS;
#else
		return -EAGAIN;
#endif
	}

	/*
	 * If this is a callout device, then just make sure the normal
	 * device isn't being used.
	 */
	if (tty->driver.subtype == SERIAL_TYPE_CALLOUT) {
		if (info->flags & ASYNC_NORMAL_ACTIVE)
			return -EBUSY;
		if ((info->flags & ASYNC_CALLOUT_ACTIVE) &&
		    (info->flags & ASYNC_SESSION_LOCKOUT) &&
		    (info->session != current->session))
		    return -EBUSY;
		if ((info->flags & ASYNC_CALLOUT_ACTIVE) &&
		    (info->flags & ASYNC_PGRP_LOCKOUT) &&
		    (info->pgrp != current->pgrp))
		    return -EBUSY;
		info->flags |= ASYNC_CALLOUT_ACTIVE;
		return 0;
	}
	
	/*
	 * If non-blocking mode is set, or the port is not enabled,
	 * then make the check up front and then exit.
	 */
	if ((filp->f_flags & O_NONBLOCK) ||
	    (tty->flags & (1 << TTY_IO_ERROR))) {
		if (info->flags & ASYNC_CALLOUT_ACTIVE)
			return -EBUSY;
		info->flags |= ASYNC_NORMAL_ACTIVE;
		return 0;
	}

	if (info->flags & ASYNC_CALLOUT_ACTIVE) {
		if (info->normal_termios.c_cflag & CLOCAL)
			do_clocal = 1;
	} else {
		if (tty->termios->c_cflag & CLOCAL)
			do_clocal = 1;
	}
	
	/*
	 * Block waiting for the carrier detect and the line to become
	 * free (i.e., not in use by the callout).  While we are in
	 * this loop, info->count is dropped by one, so that
	 * ck510evb_close() knows when to free things.  We restore it upon
	 * exit, either normal or abnormal.
	 */
	retval = 0;
	add_wait_queue(&info->open_wait, &wait);
#ifdef SERIAL_DEBUG_OPEN
	printk("block_til_ready before block: ttyS%d, count = %d\n",
	       info->line, info->count);
#endif
	info->count--;
	info->blocked_open++;
	while (1) {
		cli();
		if (!(info->flags & ASYNC_CALLOUT_ACTIVE))
			ck510evb_setsignals(info, 1, 1);
		sti();
		current->state = TASK_INTERRUPTIBLE;
		if (tty_hung_up_p(filp) ||
		    !(info->flags & ASYNC_INITIALIZED)) {
#ifdef SERIAL_DO_RESTART
			if (info->flags & ASYNC_HUP_NOTIFY)
				retval = -EAGAIN;
			else
				retval = -ERESTARTSYS;	
#else
			retval = -EAGAIN;
#endif
			break;
		}
		if (!(info->flags & ASYNC_CALLOUT_ACTIVE) &&
		    !(info->flags & ASYNC_CLOSING) &&
		    (do_clocal || (ck510evb_getsignals(info) & TIOCM_CD)))
			break;
#if LINUX_VERSION_CODE < 0x020100
		if (current->signal & ~current->blocked) {
#else
		if (signal_pending(current)) {
#endif
			retval = -ERESTARTSYS;
			break;
		}
#ifdef SERIAL_DEBUG_OPEN
		printk("block_til_ready blocking: ttyS%d, count = %d\n",
		       info->line, info->count);
#endif
		schedule();
	}
	current->state = TASK_RUNNING;
	remove_wait_queue(&info->open_wait, &wait);
	if (!tty_hung_up_p(filp))
		info->count++;
	info->blocked_open--;
#ifdef SERIAL_DEBUG_OPEN
	printk("block_til_ready after blocking: ttyS%d, count = %d\n",
	       info->line, info->count);
#endif
	if (retval)
		return retval;
	info->flags |= ASYNC_NORMAL_ACTIVE;
	return 0;
}	

/*
 * This routine is called whenever a serial port is opened. It
 * enables interrupts for a serial port, linking in its structure into
 * the IRQ chain.   It also performs the serial-specific
 * initialization for the tty structure.
 */
int ck510evb_open(struct tty_struct *tty, struct file * filp)
{
	struct ck510evb_serial	*info;
	int 			retval, line;

	line = MINOR(tty->device) - tty->driver.minor_start;
	if ((line < 0) || (line >= NR_PORTS))
		return -ENODEV;
	info = ck510evb_table + line;
	if (serial_paranoia_check(info, tty->device, "ck510evb_open"))
		return -ENODEV;
#ifdef SERIAL_DEBUG_OPEN
	printk("ck510evb_open %s%d, count = %d\n", tty->driver.name, info->line,
	       info->count);
#endif
	info->count++;
	tty->driver_data = info;
	info->tty = tty;

	/*
	 * Start up serial port
	 */
	retval = startup(info);
	if (retval)
		return retval;

	retval = block_til_ready(tty, filp, info);
	if (retval) {
#ifdef SERIAL_DEBUG_OPEN
		printk("ck510evb_open returning after block_til_ready with %d\n",
		       retval);
#endif
		return retval;
	}

	if ((info->count == 1) && (info->flags & ASYNC_SPLIT_TERMIOS)) {
		if (tty->driver.subtype == SERIAL_TYPE_NORMAL)
			*tty->termios = info->normal_termios;
		else 
			*tty->termios = info->callout_termios;
		ck510evb_change_speed(info);
	}

	info->session = current->session;
	info->pgrp = current->pgrp;

#ifdef SERIAL_DEBUG_OPEN
	printk("ck510evb_open ttyS%d successful...\n", info->line);
#endif
	return 0;
}

/*
 *	Based on the line number set up the internal interrupt stuff.
 */
static void ck510evb_irqinit(struct ck510evb_serial *info)
{
	//volatile unsigned char *uartp;

	switch (info->line) {
	case 0:
		break;
	case 1:
		break;
	default:
		printk("SERIAL: don't know how to handle UART %d interrupt?\n",info->line);
		return;
	}
	if (request_irq(info->irq, ck510evb_uart_interrupt, SA_INTERRUPT,
		"CKcore for CK510EVB Board  UART", NULL))
	{
		printk("SERIAL: Unable to attach CK510EVB (Use CK510)  UART %d interrupt "
			"vector=%d\n", info->line, info->irq);
	}
	return;
}


char *ck510evb_drivername = "CK510EVB Board internal UART serial driver version 1.00\n";


/*
 * Serial stats reporting...
 */
int ck510evb_readproc(char *page, char **start, off_t off, int count,
		         int *eof, void *data)
{
	struct ck510evb_serial	*info;
	char					str[20];
	int						len, sigs, i;

	len = sprintf(page, ck510evb_drivername);
	for (i = 0; (i < NR_PORTS); i++) {
		info = &ck510evb_table[i];
		len += sprintf((page + len), "%d: port:%x irq=%d baud:%d ",
			i, info->addr, info->irq, info->baud);
		if (info->stats.rx || info->stats.tx)
			len += sprintf((page + len), "tx:%d rx:%d ",
			info->stats.tx, info->stats.rx);
		if (info->stats.rxframing)
			len += sprintf((page + len), "fe:%d ",
			info->stats.rxframing);
		if (info->stats.rxoverrun)
			len += sprintf((page + len), "oe:%d ",
			info->stats.rxoverrun);

		str[0] = str[1] = 0;
		if ((sigs = ck510evb_getsignals(info))) {
			if (sigs & TIOCM_RTS)
				strcat(str, "|RTS");
			if (sigs & TIOCM_CTS)
				strcat(str, "|CTS");
			if (sigs & TIOCM_DTR)
				strcat(str, "|DTR");
			if (sigs & TIOCM_CD)
				strcat(str, "|CD");
		}

		len += sprintf((page + len), "%s\n", &str[1]);
	}

	return(len);
}


/* Finally, routines used to initialize the serial driver. */

static void show_serial_version(void)
{
	printk(ck510evb_drivername);
}

/* ck510evb_init inits the driver */
static int __init
ck510evb_init(void)
{
	struct ck510evb_serial	*info;
	unsigned long			flags;
	int						i;

	init_bh(CM206_BH, do_serial_bh);

	/* Setup base handler, and timer table. */
#if defined(MCF_HAVEDCD0) || defined(MCF_HAVEDCD1)
#if LINUX_VERSION_CODE < 0x020100
	timer_table[RS_TIMER].fn = mcfrs_timer;
	timer_table[RS_TIMER].expires = 0;
#else
	init_timer(&mcfrs_timer_struct);
	mcfrs_timer_struct.function = mcfrs_timer;
	mcfrs_timer_struct.data = 0;
	mcfrs_timer_struct.expires = jiffies + HZ/25;
	add_timer(&mcfrs_timer_struct);
#endif
	mcfrs_olddcd0 = mcf_getppdcd(0);
	mcfrs_olddcd1 = mcf_getppdcd(1);
#endif /* MCF_HAVEDCD0 || MCF_HAVEDCD1 */

	show_serial_version();

	/* Initialize the tty_driver structure */
	memset(&ck510evb_serial_driver, 0, sizeof(struct tty_driver));
	ck510evb_serial_driver.magic = TTY_DRIVER_MAGIC;
#if defined(CONFIG_DEVFS_FS)
	ck510evb_serial_driver.name = "tts/%d";
#else
	ck510evb_serial_driver.name = "ttyS";
#endif  /* defined(CONFIG_DEVFS_FS) */
	ck510evb_serial_driver.major = TTY_MAJOR;
	ck510evb_serial_driver.minor_start = 64;
	ck510evb_serial_driver.num = NR_PORTS;
	ck510evb_serial_driver.type = TTY_DRIVER_TYPE_SERIAL;
	ck510evb_serial_driver.subtype = SERIAL_TYPE_NORMAL;
	ck510evb_serial_driver.init_termios = tty_std_termios;

	ck510evb_serial_driver.init_termios.c_cflag =
		ck510evb_console_cbaud | CS8 | CREAD | HUPCL | CLOCAL;
	ck510evb_serial_driver.flags = TTY_DRIVER_REAL_RAW;
	ck510evb_serial_driver.refcount = &ck510evb_serial_refcount;
	ck510evb_serial_driver.table = ck510evb_serial_table;
	ck510evb_serial_driver.termios = ck510evb_serial_termios;
	ck510evb_serial_driver.termios_locked = ck510evb_serial_termios_locked;

	ck510evb_serial_driver.open = ck510evb_open;
	ck510evb_serial_driver.close = ck510evb_close;
	ck510evb_serial_driver.write = ck510evb_write;
	ck510evb_serial_driver.flush_chars = ck510evb_flush_chars;
	ck510evb_serial_driver.write_room = ck510evb_write_room;
	ck510evb_serial_driver.chars_in_buffer = ck510evb_chars_in_buffer;
	ck510evb_serial_driver.flush_buffer = ck510evb_flush_buffer;
	ck510evb_serial_driver.ioctl = ck510evb_ioctl;
	ck510evb_serial_driver.throttle = ck510evb_throttle;
	ck510evb_serial_driver.unthrottle = ck510evb_unthrottle;
	ck510evb_serial_driver.set_termios = ck510evb_set_termios;
	ck510evb_serial_driver.stop = ck510evb_stop;
	ck510evb_serial_driver.start = ck510evb_start;
	ck510evb_serial_driver.hangup = ck510evb_hangup;
	ck510evb_serial_driver.read_proc = ck510evb_readproc;
	ck510evb_serial_driver.wait_until_sent = ck510evb_wait_until_sent;
	ck510evb_serial_driver.driver_name = "serial";

	/*
	 * The callout device is just like normal device except for
	 * major number and the subtype code.
	 */
	ck510evb_callout_driver = ck510evb_serial_driver;
#if defined(CONFIG_DEVFS_FS)
	ck510evb_callout_driver.name = "cua/%d";
#else
	ck510evb_callout_driver.name = "cua";
#endif  /* defined(CONFIG_DEVFS_FS) */
	ck510evb_callout_driver.major = TTYAUX_MAJOR;
	ck510evb_callout_driver.subtype = SERIAL_TYPE_CALLOUT;
	ck510evb_callout_driver.read_proc = 0;
	ck510evb_callout_driver.proc_entry = 0;

	if (tty_register_driver(&ck510evb_serial_driver)) {
		printk("%s: Couldn't register serial driver\n", __FUNCTION__);
		return(-EBUSY);
	}
	if (tty_register_driver(&ck510evb_callout_driver)) {
		printk("%s: Couldn't register callout driver\n", __FUNCTION__);
		return(-EBUSY);
	}
	
	save_flags(flags); cli();

	/*
	 *	Configure all the attached serial ports.
	 */
	for (i = 0, info = ck510evb_table; (i < NR_PORTS); i++, info++) {
		info->magic = SERIAL_MAGIC;
		info->line = i;
		info->tty = 0;
		info->custom_divisor = 16;
		info->close_delay = 50;
		info->closing_wait = 3000;
		info->x_char = 0;
		info->event = 0;
		info->count = 0;
		info->blocked_open = 0;
		info->tqueue.routine = do_softint;
		info->tqueue.data = info;
		info->tqueue_hangup.routine = do_serial_hangup;
		info->tqueue_hangup.data = info;
		info->callout_termios = ck510evb_callout_driver.init_termios;
		info->normal_termios = ck510evb_serial_driver.init_termios;
#if LINUX_VERSION_CODE < 0x020100
		info->open_wait = 0;
		info->close_wait = 0;
#else
		init_waitqueue_head(&info->open_wait);
		init_waitqueue_head(&info->close_wait);
#endif

		info->imr = 0;
		ck510evb_setsignals(info, 0, 0);
		ck510evb_irqinit(info);

#if defined(CONFIG_DEVFS_FS)
		printk("%.4s%d at 0x%04x (irq = %d)", ck510evb_serial_driver.name,
			info->line, info->addr, info->irq);
#else
		printk("%s%d at 0x%04x (irq = %d)", ck510evb_serial_driver.name,
			info->line, info->addr, info->irq);
#endif  /* defined(CONFIG_DEVFS_FS) */
		printk(" is a builtin CK510EVB UART\n");
	}

	restore_flags(flags);
	return 0;
}

module_init(ck510evb_init);
/* module_exit(ck510evb_fini); */

/****************************************************************************/
/*                          Serial Console                                  */
/****************************************************************************/
#ifdef CONFIG_CK510EVB_SERIAL_CONSOLE
/*
 *	Quick and dirty UART initialization, for console output.
 */

void ck510evb_init_console(void)
{
	volatile unsigned char	*uartp;
	unsigned long			divisor;

	/* Reset UART, get it into known state... */
	//uartp = (volatile unsigned char *) (ck510evb_console_port ? 0 : CK510EVB_UART_BASE1);
	uartp = (volatile unsigned char *) CK510EVB_UART_BASE1;

	/* set DLA bit in LCR and set word length to 8 bits */
	uartp[CK510EVB_UART_LCR] = CK510EVB_UART_LCR_DLAEN | CK510EVB_UART_LCR_WLEN8;

	/* calculate the baudrate clock */
	//divisor = (CK_BUSCLK / (ck510evb_console_baud)) >> CK510EVB_UART_DIV;
	divisor = (CK_BUSCLK / CONSOLE_BAUD_RATE) >> CK510EVB_UART_DIV;
	/* set low byte divisor */
	uartp[CK510EVB_UART_DLL] = *((unsigned char *)(&divisor) + 3);
	/* set high byte divisor */
	if(divisor & 0xff00)
	{
	    uartp[CK510EVB_UART_DLH] = *((unsigned char *)(&divisor) + 2);
	}
	/* clear CLA bit of LCR, and set CKUART_LCR_PEN and CKUART_LCR_WLEN8 */
	uartp[CK510EVB_UART_LCR] = CK510EVB_UART_LCR_PEN | CK510EVB_UART_LCR_WLEN8;
	/* set FIFO control 1 byte */
	uartp[CK510EVB_UART_FCR] = CK510EVB_UART_FCR_FIFO1B;

	ck510evb_console_inited++;
	return;
}


/*
 *	Setup for console. Argument comes from the boot command line.
 */

int ck510evb_console_setup(struct console *cp, char *arg)
{
	int	i, n = CONSOLE_BAUD_RATE;

	if (!cp)
		return(-1);

	if (!strncmp(cp->name, "ttyS", 4))
	{
		ck510evb_console_port = cp->index;
	}
	else if (!strncmp(cp->name, "cua", 3))
	{
		ck510evb_console_port = cp->index;
	}
	else
	{
		return(-1);
	}

	if (arg)
	{
		n = simple_strtoul(arg,NULL,0);
	}
	for (i = 0; i < CK510EVB_BAUD_TABLE_SIZE; i++)
	{
		if (ck510evb_baud_table[i] == n)
		{
			break;
		}
	}
	if (i < CK510EVB_BAUD_TABLE_SIZE) {
		ck510evb_console_baud = n;
		ck510evb_console_cbaud = 0;
		if (i > 15) {
			ck510evb_console_cbaud |= CBAUDEX;
			i -= 15;
		}
		ck510evb_console_cbaud |= i;
	}
	ck510evb_init_console(); /* make sure baud rate changes */
	return(0);
}


static kdev_t ck510evb_console_device(struct console *c)
{
	return MKDEV(TTY_MAJOR, 64 + c->index);
}


/*
 *	Output a single character, using UART polled mode.
 *	This is used for console output.
 */

void ck510evb_put_char(char ch)
{
	volatile unsigned char	*uartp;
	unsigned long		flags;
	int			i;

	uartp = (volatile unsigned char *) CK510EVB_UART_BASE1;

	save_flags(flags); cli();
	//Waiting for Transmitter Holding register empty(THRE)
	for (i = 0; (i < 0x10000); i++) {
		if (uartp[CK510EVB_UART_LSR] & CK510EVB_UART_LSR_THRE)
			break;
	}

	uartp[CK510EVB_UART_THR] = ch;
	restore_flags(flags);

	return;
}


/*
 * rs_console_write is registered for printk output.
 */

void ck510evb_console_write(struct console *cp, const char *p, unsigned len)
{
	if (!ck510evb_console_inited)
		ck510evb_init_console();
	while (len-- > 0) {
		if (*p == '\n')
			ck510evb_put_char('\r');
		ck510evb_put_char(*p++);
	}
}

/*
 * declare our consoles
 */

struct console ck510evb_console = {
	name:		"ttyS",
	write:		ck510evb_console_write,
	read:		NULL,
	device:		ck510evb_console_device,
	unblank:	NULL,
	setup:		ck510evb_console_setup,
	flags:		CON_PRINTBUFFER,
	index:		-1,
	cflag:		0,
	next:		NULL
};

void __init ck510evb_console_init(void)
{
	register_console(&ck510evb_console);
}
#endif //#ifdef CONFIG_CK510EVB_SERIAL_CONSOLE

/****************************************************************************/
