/*****************************************************************************/

/*
 *	pcf8563.c -- driver for PCF8563 Real Time Clock.
 *
 * 	(C) Copyright 2008, ZhengXJ (zheng_xingjian@dahuatech.com)
 */

/*****************************************************************************/
#include <linux/config.h>
#include <linux/types.h>
#include <linux/miscdevice.h>
#include <linux/i2c.h>
//#include <linux/bcd.h>
#include <linux/rtc.h>
#include <linux/init.h>
#include <linux/module.h> 
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/rtc.h>
#include <linux/proc_fs.h>
#include <asm/uaccess.h> 

#define DRV_VERSION "0.0.1"

/* Addresses to scan: none
 * This chip cannot be reliably autodetected. An empty eeprom
 * located at 0x51 will pass the validation routine due to
 * the way the registers are implemented.
 */
 
/* Module parameters */
//I2C_CLIENT_INSMOD;

#define PCF8563_DEBUG

#ifdef PCF8563_DEBUG
	#define pcfdbg(x)	x
#else
	#define pcfdbg(x)
#endif

#ifndef BCD_TO_BIN
#define BCD_TO_BIN(val) 	(((val)&15) + ((val)>>4)*10)
#endif

#ifndef BIN_TO_BCD
#define BIN_TO_BCD(val) 	( (((val)/10)<<4) + (val)%10)
#endif

#define PCF8563_REG_ST1		0x00 /* status */
#define PCF8563_REG_ST2		0x01

#define PCF8563_REG_SC		0x02 /* datetime */
#define PCF8563_REG_MN		0x03
#define PCF8563_REG_HR		0x04
#define PCF8563_REG_DM		0x05
#define PCF8563_REG_DW		0x06
#define PCF8563_REG_MO		0x07
#define PCF8563_REG_YR		0x08

#define PCF8563_REG_AMN		0x09 /* alarm */
#define PCF8563_REG_AHR		0x0A
#define PCF8563_REG_ADM		0x0B
#define PCF8563_REG_ADW		0x0C

#define PCF8563_REG_CLKO	0x0D /* clock out */
#define PCF8563_REG_TMRC	0x0E /* timer control */
#define PCF8563_REG_TMR		0x0F /* timer */

#define PCF8563_REG_MAX		0x10	/* 16 */

#define PCF8563_SC_LV		0x80 /* low voltage */
#define PCF8563_MO_C		0x80 /* century */


#define PCF8563_IS_OPEN		0x01	/* means /dev/rtc is in use	*/
#define PCF8563_TIMER_ON	0x02	/* missed irq timer active	*/

#define RTC_GETDATETIME		0
#define RTC_SETTIME			1
#define RTC_SETDATETIME		2
#define RTC_GETCTRL			3
#define RTC_SETCTRL			4
#define MEM_READ			5
#define MEM_WRITE			6

#define PCF8563_REG_TIME_LEN	(PCF8563_REG_YR - PCF8563_REG_SC + 1)

static unsigned long pcf8563_status = 0;	/* bitmapped status byte.	*/

static struct i2c_driver pcf8563_driver;
static struct i2c_client *pcf8563_c;
static unsigned int pcf8563_adapterid;

static unsigned long epoch = 1900;

static const unsigned char days_in_mo[] = 
{0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};


#define DAT(x) ((unsigned int)(x->data))

static unsigned short ignore[] = { I2C_CLIENT_END };
static unsigned short normal_addr[] = { 0x51, I2C_CLIENT_END };


static struct i2c_client_address_data addr_data = {
	normal_i2c:		normal_addr,
	normal_i2c_range:	ignore,
	probe:			ignore,
	probe_range:		ignore,
	ignore:			ignore,
	ignore_range:		ignore,
	force:			ignore,
};


static int pcf8563_attach(struct i2c_adapter *adap, int addr, unsigned short flags,
	       int kind)
{
	struct i2c_client *c;
	c = kmalloc(sizeof(*c), GFP_KERNEL);
	if (!c)
		return -ENOMEM;

	strcpy(c->name, "PCF8563");
	c->id		= pcf8563_driver.id;
	c->flags	= I2C_CLIENT_ALLOW_USE;
	c->addr		= addr;
	c->adapter	= adap;
	c->driver	= &pcf8563_driver;
	c->data		= NULL;
	
	//pcf8563_adapterid = adap->id;

	return i2c_attach_client(c);
}

static int pcf8563_probe(struct i2c_adapter *adap)
{
	return i2c_probe(adap, &addr_data, pcf8563_attach);
}

static int pcf8563_detach(struct i2c_client *client)
{
	i2c_detach_client(client);
	kfree(client);
	return 0;
}

static ssize_t pcf8563_readbus(int pos, char *buf, size_t count)
{
	int	i = 0;
	unsigned char i2cbuf[PCF8563_REG_MAX], ad[1];
	struct i2c_msg msgs[2] = {
		{ pcf8563_c->addr, 0,        1, ad  },
		{ pcf8563_c->addr, I2C_M_RD, 1, i2cbuf }
	};
	
	if( NULL !=  pcf8563_c)
	{
		//get address and count
		ad[0] = pos;
		msgs[1].len = count;
		
		//transfer
		if (i2c_transfer(pcf8563_c->adapter, msgs, 2) == 2)
		{
			for (i = 0; (i < count); i++,buf++)
			{
				put_user( i2cbuf[i], buf);
			}
		}
	}
	
	return i;
}

static ssize_t pcf8563_writebus(int pos, char *buf, size_t count)
{
	int	i = 0;
	unsigned char i2cbuf[ PCF8563_REG_MAX + 1 ];
	struct i2c_msg msgs[1];

	if( NULL ==  pcf8563_c)
		return -ENODEV;
		
	//get address
	i2cbuf[0] = pos;
	//copy buffer
	for (i = 0; (i < count); i++, buf++)
		get_user( i2cbuf[ i +1 ], buf );
				
	i++;
	msgs[0].addr = pcf8563_c->addr;
	msgs[0].flags = 0;
	msgs[0].len = i;	//count + 1
	msgs[0].buf = i2cbuf;
		
	//transfer
	if (i2c_transfer(pcf8563_c->adapter, msgs, 1) != 1)
		return 0;
	
	return(i);
}


static ssize_t pcf8563_read(struct file *fp, char *buf, size_t count, loff_t *ptr)
{
	int	total = 0;

	if( NULL ==  pcf8563_c)
		return -ENODEV;
		
	if ( fp->f_pos < PCF8563_REG_MAX )
	{
		if (count > (PCF8563_REG_MAX - fp->f_pos))
			count = PCF8563_REG_MAX - fp->f_pos;

		//transfer
		if( count == ( pcf8563_readbus( fp->f_pos, buf,count) ) )
			total = count;
		
		fp->f_pos += total;
	}
	return(total);
}

static int pcf8563_get_time( struct rtc_time *rtct  )
{
	unsigned char buffer[ PCF8563_REG_MAX + 1 ];
	int	moffset[] = { 0, 31, 59, 90, 120, 151,
						181, 212, 243, 273, 304, 334};
	
	if( NULL ==  pcf8563_c)
		return -ENODEV;
		
	//memset(buffer,0,PCF8563_REG_MAX + 1);
	
	if( PCF8563_REG_TIME_LEN == ( pcf8563_readbus( PCF8563_REG_SC, &buffer ,PCF8563_REG_TIME_LEN) ) )
	{
		rtct->tm_sec  = BCD_TO_BIN( (buffer[0] & 0x7F) );
		rtct->tm_min  = BCD_TO_BIN( (buffer[1] & 0x7F) );
		rtct->tm_hour = BCD_TO_BIN( (buffer[2] & 0x3F) );
		rtct->tm_mday = BCD_TO_BIN( (buffer[3] & 0x3F) );
		rtct->tm_mon  = BCD_TO_BIN( (buffer[5] & 0x1F) );
		rtct->tm_year = BCD_TO_BIN( (buffer[6] & 0xFF) );
		rtct->tm_wday = BCD_TO_BIN( (buffer[4] & 0x07) );
		
		//for linux
		rtct->tm_mon--;
		rtct->tm_yday	= moffset[rtct->tm_mon] + rtct->tm_mday - 1;
		rtct->tm_isdst	= 0;
		
		/*
		if ((rtct->tm_year += (epoch - 1900)) <= 69)
			rtct->tm_year += 100;
		*/
		
		if( 0 == ( buffer[5] & PCF8563_MO_C ) )
			rtct->tm_year += 100;
	}
	
	return 0;
		
	
}

static int pcf8563_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
	return -EINVAL;
}

/*****************************************************************************/

static ssize_t pcf8563_write(struct file *fp, const char *buf, size_t count, loff_t *ptr)
{
	int total = 0;

	if( NULL ==  pcf8563_c)
		return -ENODEV;
		
	if ( fp->f_pos < PCF8563_REG_MAX )
	{
		if (count > (PCF8563_REG_MAX - fp->f_pos))
			count = PCF8563_REG_MAX - fp->f_pos;
		
		//transfer
		if( ( count + 1 ) == pcf8563_writebus( fp->f_pos, buf, count ) )
		{
			total = count ;
			fp->f_pos += total;
			total ++;
		}
	}
	
	return(total);
}


/*
 * ioctl calls that are permitted to the /dev/rtc interface, if
 * any of the RTC drivers are enabled.
 */
static int pcf8563_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		     unsigned long arg)
{
	struct rtc_time wtime; 
	
	//no irq
	switch (cmd) 
	{
		case RTC_AIE_OFF:
		case RTC_AIE_ON:
		case RTC_PIE_OFF:
		case RTC_PIE_ON:
		case RTC_UIE_OFF:
		case RTC_UIE_ON:
		case RTC_IRQP_READ:
		case RTC_IRQP_SET:
			return -EINVAL;
	};

	switch (cmd) 
	{
		case RTC_ALM_READ:
		{
			break; 
		}
		case RTC_ALM_SET:
		{
			return 0;
		}
		case RTC_RD_TIME:	/* Read the time/date from RTC	*/
		{
			memset(&wtime, 0, sizeof(struct rtc_time));
			pcf8563_get_time(&wtime);
			break;
		}
		case RTC_SET_TIME:
		{
			unsigned char leap_yr;
			unsigned int yrs;
			unsigned char tmbuf[0x09];
			
			
			if (!capable(CAP_SYS_TIME))
				return -EACCES;

			if (copy_from_user(&wtime, (struct rtc_time*)arg,sizeof(struct rtc_time)))
				return -EFAULT;
			
			tmbuf[ PCF8563_REG_SC ] = wtime.tm_sec;
			tmbuf[ PCF8563_REG_MN ] = wtime.tm_min;
			tmbuf[ PCF8563_REG_HR ] = wtime.tm_hour;
			tmbuf[ PCF8563_REG_DM ] = wtime.tm_mday;
			tmbuf[ PCF8563_REG_DW ] = wtime.tm_wday;
			tmbuf[ PCF8563_REG_MO ] = wtime.tm_mon + 1;
			yrs = wtime.tm_year + 1900;
			
			//printk("tm_sec = 0x%02lx\n",wtime.tm_sec);

			if (yrs < 1970)
				return -EINVAL;
				
			leap_yr = ((!(yrs % 4) && (yrs % 100)) || !(yrs % 400));
			if ((tmbuf[ PCF8563_REG_MO ] > 12) || (tmbuf[ PCF8563_REG_DM ] == 0))
				return -EINVAL;
			

			if (tmbuf[ PCF8563_REG_DM ] > (days_in_mo[tmbuf[ PCF8563_REG_MO ]] + ((tmbuf[ PCF8563_REG_MO ] == 2) && leap_yr)))
				return -EINVAL;
			

			if ((tmbuf[ PCF8563_REG_HR ] >= 24) || (tmbuf[ PCF8563_REG_MN ] >= 60) || (tmbuf[ PCF8563_REG_SC ] >= 60))
				return -EINVAL;
			

			if ((yrs -= epoch) > 255)    
				return -EINVAL;
			
			tmbuf[ PCF8563_REG_YR ] = yrs;
			if (tmbuf[ PCF8563_REG_YR ] > 169)
				return -EINVAL;
			if (tmbuf[ PCF8563_REG_YR ] >= 100)
				tmbuf[ PCF8563_REG_YR ] -= 100;
			

			tmbuf[ PCF8563_REG_SC ] = BIN_TO_BCD(tmbuf[ PCF8563_REG_SC ]);
			tmbuf[ PCF8563_REG_MN ] = BIN_TO_BCD(tmbuf[ PCF8563_REG_MN ]);
			tmbuf[ PCF8563_REG_HR ] = BIN_TO_BCD(tmbuf[ PCF8563_REG_HR ]);
			tmbuf[ PCF8563_REG_DM ] = BIN_TO_BCD(tmbuf[ PCF8563_REG_DM ]);
			tmbuf[ PCF8563_REG_MO ] = BIN_TO_BCD(tmbuf[ PCF8563_REG_MO ]);
			tmbuf[ PCF8563_REG_YR ] = BIN_TO_BCD(tmbuf[ PCF8563_REG_YR ]);

			//printk("tmbuf = 0x%02lx\n",tmbuf[ PCF8563_REG_SC ]);
			//transfer
			if( ( PCF8563_REG_TIME_LEN + 1 ) == pcf8563_writebus( PCF8563_REG_SC, &(tmbuf[PCF8563_REG_SC]), PCF8563_REG_TIME_LEN ) )
				return 0;
			else
				return -EINVAL;
		}
		case RTC_EPOCH_READ:	/* Read the epoch.	*/
		{
			return put_user (epoch, (unsigned long *)arg);
		}
		case RTC_EPOCH_SET:	/* Set the epoch.	*/
		{
			/* 
			 * There were no RTC clocks before 1900.
			 */
			if (arg < 1900)
				return -EINVAL;
	
			if (!capable(CAP_SYS_TIME))
				return -EACCES;
	
			epoch = arg;
			return 0;
		}
		default:
			return -EINVAL;
	}
	
	return copy_to_user((void *)arg, &wtime, sizeof wtime) ? -EFAULT : 0;
}

static int pcf8563_open(struct inode *inode, struct file *file)
{
	if ( pcf8563_status & PCF8563_IS_OPEN )
		return -EBUSY;
	
	//check the i2c_client
	if( NULL ==  pcf8563_c)
	{
		pcfdbg( printk("finded client failed!\n"); )
		return -ENODEV;
	}

	MOD_INC_USE_COUNT;

	//set status to open
	pcf8563_status |= PCF8563_IS_OPEN;
	
	return 0;
}

static int pcf8563_release(struct inode *inode, struct file *file)
{
	if (pcf8563_status & PCF8563_IS_OPEN)
	{
		pcf8563_status &= ~PCF8563_IS_OPEN;
		MOD_DEC_USE_COUNT;
	}

	return 0;
}



static struct file_operations pcf8563_fops =
{
	owner:		THIS_MODULE, 
	read:		pcf8563_read,
	write:		pcf8563_write,
	ioctl:		pcf8563_ioctl,
	open:		pcf8563_open,
	release:	pcf8563_release,
};

static struct miscdevice pcf8563_dev =
{
	RTC_MINOR,
	"rtc",
	&pcf8563_fops
};

static struct i2c_driver pcf8563_driver = {
	name:		"PCF8563",
	id:		I2C_DRIVERID_PCF8563,
	flags:		I2C_DF_NOTIFY,
	attach_adapter:	pcf8563_probe,
	detach_client:	pcf8563_detach,
	command:	pcf8563_command
};

static __init int pcf8563_init(void)
{
	int rc;
	printk( "Add pcf8563 i2c driver\n" );
	
	rc = i2c_add_driver(&pcf8563_driver);
	
	if( 0 != rc )
	{
		printk( "Add pcf8563 i2c driver failed!\n" );
		return rc;
	}
	
	printk( "Register rtc device\n" );

	rc = misc_register(&pcf8563_dev);
	if( 0 != rc )
	{
		printk( "Register rtc device failed\n" );
		return rc;
	}
	
	printk( "get i2c client for rtc device\n" );
	pcf8563_c = i2c_get_client(I2C_DRIVERID_PCF8563,pcf8563_adapterid,0);
	if( NULL == pcf8563_c  )
	{
		printk("finded i2c client failed!\n");
		return -ENODEV;
	}
	
	pcf8563_status = 0;
	return 0;
}

static void __exit pcf8563_exit(void)
{
	pcf8563_c = NULL;
	misc_deregister(&pcf8563_dev);
	i2c_del_driver(&pcf8563_driver);
}


/*****************************************************************************/
module_init(pcf8563_init);
module_exit(pcf8563_exit);
