/*
 * mtdram - a test mtd device
 * $Id: mtdram.c,v 1.1.1.1 2005/01/17 01:39:56 licq Exp $
 * Author: Alexander Larsson <alex@cendio.se>
 *
 * Copyright (c) 1999 Alexander Larsson <alex@cendio.se>
 *
 * This code is GPL
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/ioport.h>
#include <linux/mtd/compatmac.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/kernel.h>

#include "./spi.h"
#include "./spi_flash.h"

#define MTDRAM_TOTAL_SIZE 0x800000

#define PER_READ_COUNT          128
#define PER_WRITE_COUNT         124
#define FLASH_PAGE_SIZE         256
#define FLASH_TOTAL_SIZE        4*1024*1024

#define FLASH_WRITE_BUFMAX      24

#define NUM_PARTITIONS          10
#define NUM_MTDSPI_DEV          ( 2 )


/* partition_info */
typedef struct mtdspi_part
{
    unsigned int mfr_id;
    unsigned int dev_id;
    char name[ 32 ];
    unsigned int size;
    struct mtd_partition partition_info[ NUM_PARTITIONS ];
}MTDSPI_PART_S;


MTDSPI_PART_S dh5k_mtdspi_part[ NUM_MTDSPI_DEV ] =
{
    {
        mfr_id: MXIC,
		dev_id: 0x1620,
		name: "MXIC 4MB",
		size: 0x00400000,
		#if 1
		partition_info: {
		    { name : "uboot",   offset: 0x000000, size: 0x1e000 },/* 120K RO*/
            { name : "upflag",  offset: 0x1e000 , size: 0x2000 }, /* 8K rw*/
		    { name : "hwid",    offset: 0x20000 , size: 0x2000 }, /* 8K rw*/
			{ name : "kernel",  offset: 0x22000,  size: 0xDE000 },/* 888K ro*/
			{ name : "user",    offset: 0x100000, size: 0x100000},/* 1M ro*/
			{ name : "app",     offset: 0x200000, size: 0x140000},/* 1M+256K ro*/
			{ name : "data",    offset: 0x340000, size: 0x60000 },/* 384K ro*/
			{ name : "product", offset: 0x3a0000, size: 0x20000 },/* 128K rw*/
			{ name : "config",  offset: 0x3c0000, size: 0x20000 },/* 128K rw*/
			{ name : "backup",  offset: 0x3e0000, size: 0x20000 } /* 128K rw*/
		}
		#else
		partition_info: {
			{ name : "kernel",  offset: 0x000000, size: 0x200000 },
			{ name : "user",    offset: 0x200000, size: 0x100000 },
			{ name : "config",  offset: 0x300000, size: 0x80000 },
			{ name : "backup",  offset: 0x380000, size: 0x80000 }
		}
		#endif
    },
   #if 0
    {
        mfr_id: WINBOND,
		dev_id: 0x1740,
		name: "AMD WINBOND 8MB",
		size: 0x00800000,
		partition_info: {
			{ name : "uboot",   offset: 0x000000, size: 0x20000 },/* 128K RO*/
		    { name : "hwid",    offset: 0x20000 , size: 0x2000 }, /* 8K rw*/
			{ name : "kernel",  offset: 0x22000,  size: 0xDE000 },/* 888K ro*/
			{ name : "user",    offset: 0x100000, size: 0x100000},/* 1M ro*/
			{ name : "app",     offset: 0x200000, size: 0x140000},/* 1M+256K ro*/
			{ name : "data",    offset: 0x340000, size: 0x60000 },/* 384K ro*/
			{ name : "product", offset: 0x3a0000, size: 0x20000 },/* 128K rw*/
			{ name : "config",  offset: 0x3c0000, size: 0x20000 },/* 128K rw*/
			{ name : "backup",  offset: 0x3e0000, size: 0x20000 } /* 128K rw*/
		}
    }
    #else
    {
        mfr_id: WINBOND,
		dev_id: 0x1740,
		name: "AMD WINBOND 4MB",
		size: 0x00400000,
		partition_info: {
		    { name : "uboot_120K_RO",      offset: 0x000000, size: 0x1e000 },/* 120K RO*/
            { name : "upflag_8K_RW",       offset: 0x1e000 , size: 0x2000 }, /* 8K rw*/
		    { name : "hwid_8K_RW",         offset: 0x20000 , size: 0x2000 }, /* 8K rw*/
			{ name : "kernel_888K_RO",     offset: 0x22000,  size: 0xDE000 },/* 888K ro*/
			{ name : "user+1M_RO",         offset: 0x100000, size: 0x100000},/* 1M ro*/
			{ name : "app_1M+256K_RO",     offset: 0x200000, size: 0x140000},/* 1M+256K ro*/
			{ name : "data_384K_RO",       offset: 0x340000, size: 0x60000 },/* 384K ro*/
			{ name : "product_128K_RW",    offset: 0x3a0000, size: 0x20000 },/* 128K rw*/
			{ name : "config_128K_RW",     offset: 0x3c0000, size: 0x20000 },/* 128K rw*/
			{ name : "backup_128K_RW",     offset: 0x3e0000, size: 0x20000 } /* 128K rw*/
		}
    }
    #endif
};

struct tag_mtd_partition sys_mtd_part; //add by lin zhuowei


// We could store these in the mtd structure, but we only support 1 device..
static struct mtd_info *mtd_info = NULL;

static int spi_flash_chk_busy( u32 u32TimeOut )
{
    u32 timeout, cmd, buf;
    int ret;
    
    /* check status */
    timeout = 0;
    ret = 0;
    
    while( 1 )
    {
        timeout ++;
        
        cmd = CMD_STATUS_READ << 24;
        spi_flash_mode_send_for_recv( SPI1TH,W25Q64CV_ADDR, cmd, 1, &buf, 1 );
        
        if ( 0 == ( buf & 0x1 ) )
        {
            break;
        }
		
		if( timeout > u32TimeOut )
		{
		    ret = -1;
			printk("time out %s function! status = %#x \n",__FUNCTION__, buf );
			
			break;
		}
    }
    
    return ret;
     
}




static int mtdspi_erase(struct mtd_info *mtd, struct erase_info *instr)
{
    u32 beg_sector, tmp;
    int ret;
    
    ret = 0;
    
    DEBUG(MTD_DEBUG_LEVEL2, "mtdspi_erase(pos:%ld, len:%ld)\n", (long)instr->addr, (long)instr->len);
    
    if ( ( ( instr->addr + instr->len ) > mtd->size )  || ( instr->addr & ( mtd->erasesize - 1 ) ) )
    {
        DEBUG(MTD_DEBUG_LEVEL1, "ram_erase() out of bounds (%ld > %ld)\n", (long)(instr->addr + instr->len), (long)mtd->size);
        return -EINVAL;
    }
   
    //printk("erase begin_addr = %#x, len = %#x\n", instr->addr, instr->len );

    tmp = CMD_WRITE_EN << 24;
    spi_normal_mode_send( SPI1TH,W25Q64CV_ADDR, &tmp, 1 ); 
    
    spi_flash_mode_send_cmd_addr( SPI1TH,W25Q64CV_ADDR, CMD_4K_ERASE, instr->addr );
    
    ret = spi_flash_chk_busy( 0xFFFFF );
        
    //clear_spi_dev_cs( 0, 1 );
    
    if ( 0 == ret )
    {
        instr->state = MTD_ERASE_DONE;

        if (instr->callback)
        {
            (*(instr->callback))(instr);
        }
    }

    return ret;
}

static int mtdspi_point (struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen, u_char **mtdbuf)
{
    if (from + len > mtd->size)
    return -EINVAL;
  	
    *mtdbuf = mtd->priv + from;
    *retlen = len;
 
    return 0;
}

static void mtdspi_unpoint (struct mtd_info *mtd, u_char *addr, loff_t from, size_t len)
{

    DEBUG(MTD_DEBUG_LEVEL2, "ram_unpoint\n");

}

static int mtdspi_read(struct mtd_info *mtd, loff_t from, size_t len,
	     size_t *retlen, u_char *buf)
{
    u32 baseaddr, precnt, cnt, senddata;
    u8 *p;
    
    DEBUG(MTD_DEBUG_LEVEL2, "ram_read(pos:%ld, len:%ld)\n", (long)from, (long)len);
    
    if (from + len > mtd->size) 
    {
        DEBUG(MTD_DEBUG_LEVEL1, "ram_read() out of bounds (%ld > %ld)\n", (long)(from + len), (long)mtd->size);
        return -EINVAL;
    }

    baseaddr = from;
    precnt = 0;
    cnt = len;
    p = buf;
    //senddata = 0;
    
    while( 1 )
    {
        
        if ( 0 == cnt )
        {
            break;
        }
    
        if ( cnt > PER_READ_COUNT )
        {
            precnt = PER_READ_COUNT;
        }
        else
        {
            precnt = cnt;
        }
        
        senddata = 0;
        senddata = CMD_READ << 24;
        senddata |= baseaddr;
    
        spi_flash_mode_send_for_recv( SPI1TH,W25Q64CV_ADDR, senddata, 4, (u32 *)p, precnt );
    
        /* inc env */
        p += precnt;
        baseaddr += precnt;
        cnt -= precnt;
        
    }

    *retlen = len;

    return 0;
}

static int mtdspi_write(struct mtd_info *mtd, loff_t to, size_t len,
	      size_t *retlen, const u_char *buf)
{
    u32 baseaddr, tmp;
    u32 precnt, pagecnt, cnt;
    u8 *p;

    
    DEBUG(MTD_DEBUG_LEVEL2, "ram_write(pos:%ld, len:%ld)\n", (long)to, (long)len);
    //printk("write begin_addr = %#x, len = %#x\n", to, len );
    if (to + len > mtd->size) 
    {
        DEBUG(MTD_DEBUG_LEVEL1, "ram_write() out of bounds (%ld > %ld)\n", (long)(to + len), (long)mtd->size);
        return -EINVAL;
    }

    baseaddr = to;
    precnt = 0;
    cnt = len;
    pagecnt = 0;
    p = buf;

    while( 1 )
    {
        if ( 0 == cnt )
        {
            break;
        }

        /* дַҳ룬ֹҳдflash */
        pagecnt = ( FLASH_PAGE_SIZE - ( baseaddr & ( FLASH_PAGE_SIZE - 1 ) ) );
        
        if ( pagecnt > PER_WRITE_COUNT )
        {
            precnt = PER_WRITE_COUNT;
        }
        else
        {
            precnt = pagecnt;
        }

        /* жʣֹд */
        if ( precnt > cnt )
        {
            precnt = cnt;
        }
        
        tmp = CMD_WRITE_EN << 24;
        spi_normal_mode_send( SPI1TH,W25Q64CV_ADDR, &tmp, 1 );
        
        spi_flash_mode_send_cmd_data( SPI1TH,W25Q64CV_ADDR, CMD_PAGE_PROG, baseaddr, ( u32 *)(p), precnt );
        
        spi_flash_chk_busy( 0xFFFFF );
        

        /* inc env */
        baseaddr += precnt;
        p += precnt;
        cnt -= precnt;
        
    }
    
    *retlen=len;
  
    return 0;
}

static void spi_flash_read_id(u32 *mf_id, u32 *dev_id)
{ 
	u32 buf = 0,cmd = 0,readdata = 0;
    u16 devid;

	cmd = CMD_READ_ID << 24;
	
	spi_flash_mode_send_for_recv(SPI1TH,W25Q64CV_ADDR, cmd,1,&buf, 3);
	*mf_id = buf & 0xFF;
	
	*dev_id = (buf >> 8) & 0xFFFF;
    
}


static int mtdspi_scan( MTDSPI_PART_S *psMtdSpiPart, u32 count )
{
    u32 mfr_id, dev_id, i, cnt;
    MTDSPI_PART_S *psPart = psMtdSpiPart;
    int retval;
    
    /* check input */
    if ( NULL == psPart )
    {
        return -1;
    }
    
    spi_flash_read_id( &mfr_id, &dev_id );
    
    
    
    retval = -1;
    
    for ( i = 0; i < count; i ++ )
    {
        printk("R:m=%#x d=%#x  Def:m=%#x d=%#x\n", mfr_id, dev_id, psMtdSpiPart->mfr_id, psMtdSpiPart->dev_id );
        if ( ( mfr_id == psMtdSpiPart->mfr_id ) && ( dev_id == psMtdSpiPart->dev_id )  )
        {
            retval = i;
            break;
        }
        
        psMtdSpiPart++;
    }
    

    return retval;
    
}



static void __exit cleanup_mtdspi(void)
{
    if (mtd_info) 
    {
        del_mtd_device(mtd_info);
        kfree(mtd_info);
    }
}

int mtdram_init_device( struct mtd_info *mtd, unsigned long size, char *name)
{
   memset(mtd, 0, sizeof(*mtd));

   /* Setup the MTD structure */
   mtd->name = name;
   mtd->type = MTD_NORFLASH;
   mtd->flags = MTD_CAP_NORFLASH;
   mtd->size = size;
   mtd->erasesize = 0x1000;
   mtd->priv = 0;

   mtd->module = THIS_MODULE;
   mtd->erase = mtdspi_erase;
   mtd->point = mtdspi_point;
   mtd->unpoint = mtdspi_unpoint;
   mtd->read = mtdspi_read;
   mtd->write = mtdspi_write;

/*
   if (add_mtd_device(mtd)) 
   {
     return -EIO;
   }*/
   
   return 0;
}

int __init init_mtdspi(void)
{
    int err;
    MTDSPI_PART_S *psPart;
    struct mtd_partition *mtd_parts = 0;
    int mtd_parts_nb = 0;
    const char *part_type = 0;
    int i = 0;
    /* scan device */
    err = mtdspi_scan( &dh5k_mtdspi_part, NUM_MTDSPI_DEV );
    if ( -1 == err )
    {
        return -EIO;
    }
    
    /* show */
    psPart = dh5k_mtdspi_part;
    psPart += err;
    
    printk("SPI FLASH NAME: %s\n", psPart->name );
    
    
    
    /* Allocate some memory */
    mtd_info = (struct mtd_info *)kmalloc(sizeof(struct mtd_info), GFP_KERNEL );
    if (!mtd_info)
        return -ENOMEM;
    
    err = mtdram_init_device(mtd_info,  MTDRAM_TOTAL_SIZE, "mtdspi for dh5k");
    
    if (err) 
    {
        mtd_info = NULL;
        return err;
    }
    
#ifdef CONFIG_MTD_CMDLINE_PARTS
	mtd_parts_nb = parse_cmdline_partitions(mtd_info, &mtd_parts, "dh5k mtdspi");
	if (mtd_parts_nb > 0)
	  part_type = "command line";
	else
	  mtd_parts_nb = 0;
#endif

	if (mtd_parts_nb == 0)
	{
		mtd_parts = psPart->partition_info;
		mtd_parts_nb = NUM_PARTITIONS;
		part_type = "static";
	}
	
	/* Register the partitions */
	printk(KERN_NOTICE "Using %s partition definition\n", part_type);

	sys_mtd_part.part_num = mtd_parts_nb; /* NUM_PARTITIONS  */
	for (i = 0; i < mtd_parts_nb; i++) 
	{
        strncpy(sys_mtd_part.parts[i].name, mtd_parts[i].name,strlen( mtd_parts[i].name));
        sys_mtd_part.parts[i].size   =  mtd_parts[i].size;
        sys_mtd_part.parts[i].offset =   mtd_parts[i].offset;
        sys_mtd_part.parts[i].cs = W25Q64CV_ADDR;/* Ƭѡ */
	}
	add_mtd_partitions( mtd_info, mtd_parts, mtd_parts_nb);
    
    return err;
}


module_init( init_mtdspi );
module_exit( cleanup_mtdspi );

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Alexander Larsson <alexl@redhat.com>");
MODULE_DESCRIPTION("Simulated MTD driver for testing");

