#include "./spi.h"
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/version.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/ioport.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <asm/ckcore.h>

void spi_init(void);

static spinlock_t spiMutx_lock = SPIN_LOCK_UNLOCKED;
unsigned long spiLockFlags;

void spi_kernelLock()
{
    spin_lock_irqsave(&spiMutx_lock, spiLockFlags);
   
}

void spi_kernelUnLock()
{
    spin_unlock_irqrestore(&spiMutx_lock, spiLockFlags);
}



static int Write_SPI_Reg(unsigned int ctr_num,unsigned int reg_offset,unsigned int val)
{
    unsigned int reg_addr;
    if(ctr_num == 0)
    {
        reg_addr = SPI0_BASE_ADDR + reg_offset;
    }else if(ctr_num == 1)
    {
        reg_addr = SPI1_BASE_ADDR + reg_offset;
    }else
    {
        printk("spi contrl number err! \n");
        return -1;
    }
    *(volatile unsigned int *)reg_addr = val;
    return 0;
}

static int Read_SPI_Reg(unsigned int ctr_num,unsigned int reg_offset)
{
    unsigned int reg_addr;
    if(ctr_num == 0)
    {
        reg_addr = SPI0_BASE_ADDR + reg_offset;
    }else if(ctr_num == 1)
    {
        reg_addr = SPI1_BASE_ADDR + reg_offset;
    }else
    {
        printk("spi contrl number err! \n");
        return -1;
    }
    return *(volatile unsigned int *)reg_addr;
}

void set_spi_dev_cs( unsigned int ctr_num, unsigned int dev_cs )
{
    unsigned int filter = 0x01;
    unsigned int reg_temp;
    if(dev_cs > 3 || dev_cs < 0)
    {
        printk("set slave device out of range\n");
        return;
    }
    spi_init();
    reg_temp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    reg_temp &= ~((filter) << (8+dev_cs));
	Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,reg_temp);
}

void clear_spi_dev_cs( unsigned int ctr_num, unsigned int dev_cs )
{
    unsigned int filter = 0x01;
    unsigned int reg_temp;
    if(dev_cs > 3 || dev_cs < 0)
    {
        printk("set slave device out of range\n");
        return;
    }
    reg_temp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    reg_temp |= ((filter) << (8+dev_cs));
	Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,reg_temp);
}

static void set_spi_mode(unsigned int ctr_num,int mode)
{
    unsigned int reg_temp;
    if(mode == SPI_FLASH_MODE)
    {
        reg_temp = Read_SPI_Reg(ctr_num,SPI_MODECONF_OFFSET);
        reg_temp |= 1 << 8;
        Write_SPI_Reg(ctr_num,SPI_MODECONF_OFFSET,reg_temp);

    }else if(mode == SPI_NORMAL_MODE)
    {
        reg_temp = Read_SPI_Reg(ctr_num,SPI_MODECONF_OFFSET);
        reg_temp &= ~(1 << 8);
        Write_SPI_Reg(ctr_num,SPI_MODECONF_OFFSET,reg_temp);
 
    }else
    {
        printk("set spi mode err!\n");
    }
    
}
void spi_init(void)
{
	unsigned int modecfg = 0;
    unsigned int irqcfg = 0;
    unsigned int contrlcfg = 0;
	modecfg = 	(PHASE_CLK << 31) +
				(BITS << 24) +
				(CLOCK_DELAY_TIMES << 16) +
				(RX_DELAY << 15) +
				(CPHA << 14) +
				(CPOL << 13) +
				(LSB << 12) +
				(PCK_MOD << 10) +
				(HOLD_EN << 9) +
				(FLASH_MOD << 8) +
				(TX_DELAY << 1) + 1;
	irqcfg	=  (TXDFFI_LEVEL <<16) + (RXDFFI_LEVEL << 8) + (COPMLETEIE << 2) + ( 1UL << 1) +1;
    contrlcfg = (REG_FF_ST_EN << 26) +
				(REG_CRC_EN << 25) +
			    (REG_BMULT << 24) +
			    (CR_CS << 8) + (DIRECTION << 1) + (STARTED);


    /*init the 1th spi*/
    Write_SPI_Reg(SPI1TH,SPI_MODECONF_OFFSET,modecfg);
    Write_SPI_Reg(SPI1TH,SPI_INTCONF_OFFSET,irqcfg);
    Write_SPI_Reg(SPI1TH,SPI_TC_OFFSET,TRANS_CNT);
    Write_SPI_Reg(SPI1TH,SPI_BOUND_OFFSET,BAUD);
    Write_SPI_Reg(SPI1TH,SPI_CTRL_OFFSET,contrlcfg);
    
	/*init the 2th spi*/

    Write_SPI_Reg(SPI2TH,SPI_MODECONF_OFFSET,modecfg);
    Write_SPI_Reg(SPI2TH,SPI_INTCONF_OFFSET,irqcfg);
    Write_SPI_Reg(SPI2TH,SPI_TC_OFFSET,TRANS_CNT);
    Write_SPI_Reg(SPI2TH,SPI_BOUND_OFFSET,BAUD);
    Write_SPI_Reg(SPI2TH,SPI_CTRL_OFFSET,contrlcfg);	  
}


static u32 spi_check_complete( u8 ctr_num, u32 u32TimeOut )
{
    u32 ret;
    u32 timeout;
    
    ret = 0;
    timeout = 0;
    
    while ( Read_SPI_Reg( ctr_num, SPI_STATE_OFFSET )  & ( 0x1 << 6) )
	{
		 timeout ++;
		 
		 if( timeout > u32TimeOut )
		 	{
                Write_SPI_Reg( ctr_num,SPI_STATE_OFFSET, 0 );
		 		printk("time out %s function! \n",__FUNCTION__);
                ret = -1;
		 		break;
		 	}
	}
	
	return ret;
    
}

int spi_normal_mode_send(unsigned char ctr_num,unsigned char chipselect,
                    unsigned int *tbuf, unsigned int tcnt )
{
	unsigned int i,tmp,ret = 0;
	unsigned int cnt;
	unsigned int timeout = 0;
	unsigned int *p = NULL;
	
	/* check parameter */
	if ( ( NULL == tbuf ) || ( 0 == tcnt ))
	{
		return -1;
	}
	p = tbuf;
	spi_kernelLock();
    set_spi_dev_cs(ctr_num,chipselect);
	/* set the spi mode to normal */
    set_spi_mode(ctr_num,SPI_NORMAL_MODE);
	cnt = tcnt >> 2;
	if( tcnt % 4 )
	{
		cnt++;
	}
	
    for ( i = 0; i < cnt; i ++ )
    {
        Write_SPI_Reg(ctr_num,SPI_TXDB_OFFSET,*p++);
    }
    Write_SPI_Reg(ctr_num,SPI_TC_OFFSET,tcnt - 1);

    tmp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    tmp |= 0x03;
    Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,tmp);
    
    ret = spi_check_complete( ctr_num, 0xffffff );

    tmp = Read_SPI_Reg(ctr_num,SPI_STATE_OFFSET);
    tmp |= (1UL << 20);;
    Write_SPI_Reg(ctr_num,SPI_STATE_OFFSET,tmp);
    clear_spi_dev_cs(ctr_num,chipselect);
    spi_kernelUnLock();
	//SPI_STR(spi_cs) |= (1UL << 20);

	/* set the spi mode from normal to flash */
	//SPI_MDR(spi_cs) |= ( 1 << AVI_SPI_MDR_FMODE );
	//set_spi_mode(ctr_num,SPI_FLASH_MODE);
	return ret;
}

int spi_normal_mode_recive( unsigned char ctr_num,unsigned char chipselect, unsigned int *rbuf, unsigned int rcnt )
{
    unsigned int i,ret = 0;
	unsigned int tmp;
	unsigned int cnt;
	unsigned int timeout = 0;
	unsigned int *p = NULL;
	timeout = 0;	
	
	if ( (NULL == rbuf) || ( 0 == rcnt ) )
	{
		return 0;
	}
	spi_kernelLock();
	set_spi_dev_cs(ctr_num,chipselect);
	/* set the spi mode  normal */
	set_spi_mode(ctr_num,SPI_NORMAL_MODE);
	Write_SPI_Reg(ctr_num,SPI_TC_OFFSET,rcnt - 1);
    tmp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    tmp &= SPI_CMDLEN_MASK;
	tmp |= 1UL;
    Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,tmp);

    p = rbuf;

    cnt = rcnt >> 2;
	if (  rcnt % 0x04  )
    {
        cnt ++;
    }
    
    ret = spi_check_complete( ctr_num, 0xffffff );

    for ( i = 0 ; i < cnt; i ++ )
    {
        *p ++ = Read_SPI_Reg(ctr_num,SPI_RXDB_OFFSET);
    }
    Write_SPI_Reg(ctr_num,SPI_STATE_OFFSET,Read_SPI_Reg(ctr_num,SPI_STATE_OFFSET)|(1UL << 20));
    clear_spi_dev_cs(ctr_num,chipselect);
    spi_kernelUnLock();
	return ret;
}


int spi_flash_mode_send_for_recv( u8 ctr_num,unsigned char chipselect, u32 cmd, u32 cmdcnt, u32 *recvbuf, u32 recvcnt )
{
    unsigned int i, ret = 0;
	unsigned int tmp;
	unsigned int cnt;
	
    unsigned char *pu8 = NULL;
    

	/* check input param */
	if( ctr_num > 1 || ctr_num < 0 )
    {
        printk("set spi contrl num out of range\n");
        return -1;
    }
    
	if ( ( NULL == cmd ) || ( 0 == cmdcnt ) )
	{
        printk("please check argument put in!\n");
        return -1;
	}
    
    if( ( 0 == recvcnt ) || ( NULL == recvbuf ) || ( recvcnt > SPI_READ_FIFO_MAX ) )
    {  
        printk("please check argument put in!\n");
        return -1;
    }
    spi_kernelLock();
    set_spi_dev_cs(ctr_num,chipselect);
    set_spi_mode( ctr_num,SPI_FLASH_MODE );
    

	/* send the command to the fifo */
    Write_SPI_Reg( ctr_num, SPI_TXDB_OFFSET, cmd );
    
    /* send the recv count */
    Write_SPI_Reg( ctr_num, SPI_TC_OFFSET, ( recvcnt - 1 ) );
 
    /* set ctrl */
    tmp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    tmp &= SPI_CMDLEN_MASK;
	tmp |= ( cmdcnt - 1 ) << 16;
	tmp |= 1UL;
    Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,tmp);
    	
    cnt = recvcnt >> 2;
    pu8 = ( u8 *)( recvbuf );

	ret = spi_check_complete( ctr_num, 0xffffff );
	
	for ( i = 0; i < cnt; i ++ )
	{
	     tmp = Read_SPI_Reg( ctr_num, SPI_RXDB_OFFSET );
        *pu8 ++ = ( ( tmp >> 24 ) & 0xFF );
        *pu8 ++ = ( ( tmp >> 16 ) & 0xFF );
        *pu8 ++ = ( ( tmp >> 8 ) & 0xFF );
        *pu8 ++ = ( ( tmp >> 0 ) & 0xFF );
	}
    
    /* read the left */
    cnt = recvcnt & 0x03;
    
    if ( cnt )
    {
        tmp = Read_SPI_Reg( ctr_num, SPI_RXDB_OFFSET );
        
        for ( i = 0; i < cnt; i ++ )
        {
            *pu8 ++ = ( ( tmp >> ( ( 3 - i ) * 8 ) ) & 0xFF );
        }
    }
	
	
	/* clear status */
    Write_SPI_Reg(ctr_num,SPI_STATE_OFFSET,Read_SPI_Reg(ctr_num,SPI_STATE_OFFSET)|(1UL << 20));
    clear_spi_dev_cs(ctr_num,chipselect);
    spi_kernelUnLock();
    
	return ret;
}

/*
    send to flash 
    
    8       24      32      32
    cmd     addr    dat0    dat1
*/
int spi_flash_mode_send_cmd_data( u8 ctr_num,unsigned char chipselect, u32 cmd, u32 addr, u32 *psendbuf, u32 sendcnt )
{
    u32 i, tmp, cnt;
	u32 ret;
	u8 *pu8 = NULL;
	
    /* check input param */
    if ( ( NULL == psendbuf ) || ( 0 == sendcnt ) )
    {
        return -1;
    }
    spi_kernelLock();
    set_spi_dev_cs(ctr_num,chipselect);
    /* set flash mode */
    set_spi_mode( ctr_num, SPI_FLASH_MODE );
    
    /* push command */
    tmp = ( ( cmd << 24 ) & 0xFF000000 );
    tmp |= ( addr & 0xFFFFFF );
    
    Write_SPI_Reg( ctr_num, SPI_TXDB_OFFSET, tmp );
    
    
    
    /* ΪSPIģflash ģʽ£ȷ͵Ǹ8λбֽڽ
        ֤ǰС˸ʽд뵽flashȥCPUֽͻ
     */
    cnt = ( sendcnt >> 2 );
    pu8 = ( u8 * )( psendbuf );
    
    for ( i = 0; i < cnt; i ++ )
    {
        tmp = ( ( *pu8++ ) << 24 );
        tmp |= ( ( *pu8++ ) << 16 );
        tmp |= ( ( *pu8++ ) << 8 );
        tmp |= ( ( *pu8++ ) << 0 );
        
        Write_SPI_Reg( ctr_num, SPI_TXDB_OFFSET, tmp );
    }
    
    /* left */
    cnt = ( sendcnt & 0x03 );
    tmp = 0;
    
    if ( cnt )
    {
        for ( i = 0; i < cnt; i ++ )
        {
            tmp |= ( *pu8++ ) << ( ( 3 - i ) * 8 );
        }
        
        Write_SPI_Reg( ctr_num, SPI_TXDB_OFFSET, tmp );
    }
    

	Write_SPI_Reg(ctr_num,SPI_TC_OFFSET, ( sendcnt - 1 ) );

    tmp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    tmp &= SPI_CMDLEN_MASK;
	tmp |= ( ( ( 4 - 1 ) << 16 ) + 0x3 );
    Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,tmp);

    /* wait for complete */
    ret = spi_check_complete( ctr_num, 0xffffff );

    /* clear status */
	Write_SPI_Reg(ctr_num,SPI_STATE_OFFSET,Read_SPI_Reg(ctr_num,SPI_STATE_OFFSET)|(1UL << 20));
    clear_spi_dev_cs(ctr_num,chipselect);
	spi_kernelUnLock();
	return ret;
}


int spi_flash_mode_send_cmd_addr( u8 ctr_num,unsigned char chipselect, u32 cmd, u32 addr )
{
    u32 tmp, ret = 0;
	spi_kernelLock();
    set_spi_dev_cs(ctr_num,chipselect);

    set_spi_mode(ctr_num,SPI_FLASH_MODE);
    
    tmp = ( ( cmd << 24 ) & 0xFF000000 );
    tmp |= ( addr & 0xFFFFFF );
    

    Write_SPI_Reg( ctr_num, SPI_TXDB_OFFSET, tmp );
    
	Write_SPI_Reg( ctr_num, SPI_TC_OFFSET, 3 - 1 );

    tmp = Read_SPI_Reg(ctr_num,SPI_CTRL_OFFSET);
    tmp &= SPI_CMDLEN_MASK;
	tmp |= ( ( ( 1 - 1 ) << 16 ) + 0x3 );
    Write_SPI_Reg(ctr_num,SPI_CTRL_OFFSET,tmp);

    ret = spi_check_complete( ctr_num, 0xffffff );

	Write_SPI_Reg(ctr_num,SPI_STATE_OFFSET,Read_SPI_Reg(ctr_num,SPI_STATE_OFFSET)|(1UL << 20));
    clear_spi_dev_cs(ctr_num,chipselect);
	spi_kernelUnLock();
	return ret;
}






