/* $Id: immac.c,v 1.4 2007/3/12 06:14:48  $
 *
 * immac.c: Driver for IM MAC 10/100baseT ethernet adapters.
 *
 * Copyright (C) 2007, Chen Xiaobo (chen_xiaobo@dahuatech.com)
 */

#include <linux/module.h>

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/crc32.h>
#include <linux/errno.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>

#include <asm/cksim.h>

#include <asm/cache.h>

#include "immac.h"


static struct cskymac *root_immac_dev;

static char version[] __initdata =
        "IM MAC driver v1.1 11/27/2008 zxj\n";
        
//const  unsigned long CSKYMACADDR[2] = { 0x00800000, 0x00000039 };
int tulip_rx_copybreak = 1518;
unsigned int tulip_max_interrupt_work = 25;
extern unsigned char mac_addr[0x08];
unsigned char *avi_basedesaddr = NULL;


#ifdef DEBUG_PROBE
#define DP(x)  printk x
#else
#define DP(x)
#endif

#ifdef DEBUG_TX
#define DTX(x)  printk x
#else
#define DTX(x)
#endif

#ifdef DEBUG_IRQ
#define DIRQ(x)  printk x
#else
#define DIRQ(x)
#endif

#ifdef DEBUG_RX
#define DRX(x) printk x
#else
#define DRX(x)
#endif

#define mdio_delay() (temp = CKMAC_REG_BASEADDR[CSR9])

#define MDIO_SHIFT_CLK		0x10000
#define MDIO_DATA_WRITE0	0x00000
#define MDIO_DATA_WRITE1	0x20000
#define MDIO_ENB			0x00000	/* Ignore the 0x02000 databook setting. */
#define MDIO_ENB_IN			0x40000
#define MDIO_DATA_READ		0x80000


static void immac_get_macaddr (unsigned char macaddr[])
{
	memcpy( macaddr, mac_addr, 0x08 );
}
static void immac_set_macaddr (struct net_device *dev, void *p)
{
	memcpy( mac_addr, p, 0x08 );
	immac_get_macaddr( dev->dev_addr );
}
static inline void immac_start_rxtx(struct immac *dp)
{
	CKMAC_REG_BASEADDR[CSR6] = dp->csr6|RxTx;
	//barrier();
}

static inline void immac_stop_rxtx(struct immac *dp)
{
	u32 csr6 = CKMAC_REG_BASEADDR[CSR6];
	
	if (csr6 & RxTx) 
	{
		unsigned i=1300/100;
		CKMAC_REG_BASEADDR[CSR6] = csr6&~RxTx;
		//barrier();
		
		/* wait until in-flight frame completes.
		 * Max time @ 10BT: 1500*8b/10Mbps == 1200us (+ 100us margin)
		 * Typically expect this loop to end in < 50 us on 100BT.
		 */
		while (--i && (CKMAC_REG_BASEADDR[CSR5]& (CSR5_TS|CSR5_RS)))
			udelay(10);
			
		if (!i)
			printk(KERN_DEBUG " immac_stop_rxtx() failed\n");
	}
}

static inline void immac_restart_rxtx(struct immac *dp)
{
	printk("immac restart \n");
	immac_stop_rxtx(dp);
	udelay(5);
	immac_start_rxtx(dp);
}

static unsigned long immac_phyr_read(struct immac *dp, unsigned char phyr)
{
	int i;
	int read_cmd = (0xf6 << 10) | ((dp->phys & 0x1f) << 5) | phyr;
	int retval = 0;
	int temp; //for mdio_delay
	unsigned long flags;

	if (phyr & ~0x1f)
		return 0xffff;

	spin_lock_irqsave(&dp->mii_lock, flags);
	

	/* Establish sync by sending at least 32 logic ones. */
	for (i = 32; i >= 0; i--) 
	{
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | MDIO_DATA_WRITE1;
		mdio_delay();
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | MDIO_DATA_WRITE1 | MDIO_SHIFT_CLK;
		mdio_delay();
	}
	
	/* Shift the read command bits out. */
	for (i = 15; i >= 0; i--) 
	{
		int dataval = (read_cmd & (1 << i)) ? MDIO_DATA_WRITE1 : 0;
		CKMAC_REG_BASEADDR[CSR9]  = MDIO_ENB | dataval;
		mdio_delay();
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | dataval | MDIO_SHIFT_CLK;
		mdio_delay();
	}
	/* Read the two transition, 16 data, and wire-idle bits. */
	for (i = 19; i > 0; i--) 
	{
		CKMAC_REG_BASEADDR[CSR9]  = MDIO_ENB_IN;
		mdio_delay();
		retval = (retval << 1) | ((CKMAC_REG_BASEADDR[CSR9]  & MDIO_DATA_READ) ? 1 : 0);
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB_IN | MDIO_SHIFT_CLK;
		mdio_delay();
	}

	spin_unlock_irqrestore(&dp->mii_lock, flags);
	return (retval>>1) & 0xffff;
}

static void immac_phyr_write(struct immac *dp, unsigned char phyr, unsigned long w_value)
{
	int i;
	int cmd = (0x5002 << 16) | ((dp->phys& 0x1f) << 23) | (phyr<<18) | (w_value& 0xffff);
	unsigned long flags;
	int temp; //for mdio_delay
	
	if (phyr& ~0x1f)
		return;

	spin_lock_irqsave(&dp->mii_lock, flags);
	
	/* Establish sync by sending 32 logic ones. */
	for (i = 32; i >= 0; i--) 
	{
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | MDIO_DATA_WRITE1;
		mdio_delay();
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | MDIO_DATA_WRITE1 | MDIO_SHIFT_CLK;
		mdio_delay();
	}
	
	/* Shift the command bits out. */
	for (i = 31; i >= 0; i--) 
	{
		int dataval = (cmd & (1 << i)) ? MDIO_DATA_WRITE1 : 0;
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | dataval;
		mdio_delay();
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB | dataval | MDIO_SHIFT_CLK;
		mdio_delay();
	}
	
	/* Clear out extra bits. */
	for (i = 2; i > 0; i--) 
	{
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB_IN;
		mdio_delay();
		CKMAC_REG_BASEADDR[CSR9] = MDIO_ENB_IN | MDIO_SHIFT_CLK;
		mdio_delay();
	}

	spin_unlock_irqrestore(&dp->mii_lock, flags);
}



static void immac_clean_stats(struct immac *dp)
{
    memset(&dp->enet_stats, 0, sizeof (struct net_device_stats));
}


static void immac_clean_rings(struct immac *dp)
{
	//by zxj 2008-11-26
	if( NULL != avi_basedesaddr )
	{
		kfree( avi_basedesaddr );
		avi_basedesaddr = NULL;
	}		
	
}

static void immac_init_rings(struct immac *dp, int from_irq)
{
	struct net_device *dev = dp->dev;
	int i, gfp_flags = GFP_KERNEL;
        
	printk("immac_init_rings\n");
	
	if (from_irq || in_interrupt())
		gfp_flags = GFP_ATOMIC;

	dp->rx_new = dp->rx_old = dp->tx_new = dp->tx_old = 0;

	//by zxj 2008-11-26
	immac_clean_rings(dp);
	
	i = PKT_BUF_SZ * RX_RING_SIZE;
	avi_basedesaddr = kmalloc( i, GFP_DMA );
	
	printk("desc_base = %#x, dp_base = %#x\n", avi_basedesaddr, dp );
		
	for (i = 0; i < RX_RING_SIZE; i++) 
	{
		dp->rx_ring[i].status = 0x00000000;
		dp->rx_ring[i].length = cpu_to_le32(PKT_BUF_SZ);
		dp->rx_ring[i].buffer2 =  cpu_to_le32(&(dp->rx_ring[i+1]));
	}
	
	/* Mark the last entry as wrapping the ring. */
	dp->rx_ring[i-1].length = cpu_to_le32(PKT_BUF_SZ | DESC_RING_WRAP);
	dp->rx_ring[i-1].buffer2 = cpu_to_le32(&(dp->rx_ring[0]));

	for (i = 0; i < RX_RING_SIZE; i++) 
	{
		dp->rx_ring[i].status = cpu_to_le32(DescOwned);	/* Owned by Tulip chip */
		dp->rx_ring[i].buffer1 = cpu_to_le32( avi_basedesaddr + i * PKT_BUF_SZ );
	}
	dp->dirty_rx = (unsigned int)(i - RX_RING_SIZE);

	/* The Tx buffer descriptor is filled in as needed, but we
	   do need to clear the ownership bit. */
	for (i = 0; i < TX_RING_SIZE; i++) 
	{
		dp->tx_skbs[i]= NULL;
		dp->tx_ring[i].status = 0x00000000;
		dp->tx_ring[i].buffer2 = cpu_to_le32(&(dp->tx_ring[i+1]));
	}
	dp->tx_ring[i-1].buffer2 = cpu_to_le32(&(dp->tx_ring[0]));
}


static int immac_init(struct immac *, int);

static int try_next_permutation(struct immac *dp)
{
	unsigned long tmp;
	int timeout;

	tmp = immac_phyr_read(dp, RTL8201_MODECTRL);
	if (tmp & RTL8201_MODECTRL_ANE)
	{
        // Reset the PHY.
        tmp = RTL8201_MODECTRL_RESET;
        immac_phyr_write(dp, RTL8201_MODECTRL, tmp);

        timeout = 64;
        while (--timeout)
        {
            tmp = immac_phyr_read(dp, RTL8201_MODECTRL);
            if ((tmp & RTL8201_MODECTRL_RESET) == 0)
            {
                    break;
            }
            udelay(20);
        }
        if (timeout == 0)
        {
			printk(KERN_ERR "%s: PHY reset failed.\n", dp->dev->name);
        }

        tmp = immac_phyr_read(dp, RTL8201_MODECTRL);

        // Now we try 100baseT. 
        tmp = RTL8201_MODECTRL_SPD100 | RTL8201_MODECTRL_DUPLEX | RTL8201_MODECTRL_LOOPBACK;
        immac_phyr_write(dp, RTL8201_MODECTRL, tmp);
        return 0;
	}
	else if (tmp & RTL8201_MODECTRL_SPD100) 
	{
		// Reset the PHY.
		tmp = RTL8201_MODECTRL_RESET;
		immac_phyr_write(dp, RTL8201_MODECTRL, tmp);

		timeout = 64;
		while (--timeout) 
		{
			tmp = immac_phyr_read(dp, RTL8201_MODECTRL);
			if ((tmp & RTL8201_MODECTRL_RESET) == 0)
			{
				break;
			}
			udelay(20);
		}
		if (timeout == 0)
		{
			printk(KERN_ERR "%s: PHY reset failed.\n", dp->dev->name);
		}

		tmp = immac_phyr_read(dp, RTL8201_MODECTRL);

		// Now we try 10baseT. 
		tmp = RTL8201_MODECTRL_DUPLEX | RTL8201_MODECTRL_LOOPBACK;
		immac_phyr_write(dp, RTL8201_MODECTRL, tmp);
		return 0;
	}

	return -1;
}


static void immac_timer(unsigned long data)
{
	struct immac	*dp = (struct immac *) data;
	int		restart_timer = 0;
	unsigned long	tmp;

	dp->timer_ticks++;
	if (dp->timer_state == ltrywait) 
	{
		tmp = immac_phyr_read(dp, RTL8201_MODESTS);
		//printk("tmp %x \n", tmp);
		if (tmp & RTL8201_MODESTS_LINKSTS)
		{
			printk(KERN_INFO "%s: Link is now up at %s.\n", dp->dev->name,
				(tmp & RTL8201_MODESTS_100BTXFD) ?  "100baseT" : "10baseT");
			dp->timer_state = asleep;
			restart_timer = 0;
		}
		else
		{
			if (dp->timer_ticks >= 4) 
			{
				int ret;

				ret = try_next_permutation(dp);
				if (ret == -1) 
				{
					printk(KERN_ERR "%s: Link down, cable problem?\n", dp->dev->name);
					ret = immac_init(dp, 0);
					if (ret) 
					{
						printk(KERN_ERR "%s: Error, cannot re-init the  MAC.\n", dp->dev->name);
					}
					return;
				}
				dp->timer_ticks = 0;
				restart_timer = 1;
			}
			else
			{
				restart_timer = 1;
			}
		}
	}
	else
	{
		// Can't happens.... 
		printk(KERN_ERR "%s: Aieee, link timer is asleep but we got one anyways!\n",
		       dp->dev->name);
		restart_timer = 0;
		dp->timer_ticks = 0;
		dp->timer_state = asleep; // foo on you 
	}

	if (restart_timer != 0)
	{
		dp->immac_timer.expires = jiffies + ((12 * HZ)/10); // 1.2 sec. 
		add_timer(&dp->immac_timer);
	}
}


/* Well, really we just force the chip into 100baseT then
 * 10baseT, each time checking for a link status.
 */

static void immac_begin_auto_negotiation(struct immac *dp)
{
	unsigned long	tmp;
	int timeout;

	// Reset the PHY. 
	tmp = RTL8201_MODECTRL_RESET;
	immac_phyr_write(dp, RTL8201_MODECTRL, tmp);

	timeout = 64;
	while (--timeout) 
	{
		tmp = immac_phyr_read(dp, RTL8201_MODECTRL);
		if ((tmp & RTL8201_MODECTRL_RESET) == 0)
		{
			break;
		}
		udelay(20);
	}
	if (timeout == 0)
	{
		printk(KERN_ERR "%s: PHY reset failed.\n", dp->dev->name);
	}

	tmp = immac_phyr_read(dp, RTL8201_MODECTRL);

	// First we try auto-negotiation (not 100baseT). 
	tmp = RTL8201_MODECTRL_ANE ;
	immac_phyr_write(dp, RTL8201_MODECTRL, tmp);

	tmp = immac_phyr_read(dp, RTL8201_MODESTS);

	if (tmp & RTL8201_MODESTS_LINKSTS)
	{
		timeout = 1000;
		while(--timeout)
		{
			tmp = immac_phyr_read(dp, RTL8201_MODESTS);
			if(tmp&RTL8201_MODESTS_ANC)
				break;
			udelay(2000);
		}
		
		if(tmp&RTL8201_MODESTS_100BTXFD)
		{
			printk("immac phy 100 fd\n");
			dp->csr6 &= ~CSR6_TTM;
			dp->csr6 |= CSR6_FD;
		}
		else if(tmp&RTL8201_MODESTS_100BTXHD)
		{
			printk("immac phy 100 hd\n");
			dp->csr6 &= ~CSR6_TTM;
			dp->csr6 &= ~CSR6_FD;
		}
		else if(tmp&RTL8201_MODESTS_10BTFD)
		{
			printk("immac phy 10 fd\n");
			dp->csr6 |= CSR6_TTM;
			dp->csr6 |= CSR6_FD;
		}
		else if(tmp&RTL8201_MODESTS_10BTHD)
		{
			printk("immac phy 10 hd\n");
			dp->csr6 |= CSR6_TTM;
			dp->csr6 &= ~CSR6_FD;
		}
		
		if(timeout==0)
		{
			printk("auto negotiation is time out \n");
			dp->csr6 |= CSR6_TTM;
			dp->csr6 &= ~CSR6_FD;
		}
	}
	else
	{
		printk("not link \n");
		dp->csr6 |= CSR6_TTM;
		dp->csr6 &= ~CSR6_FD;
	}
}

static void immac_up(struct immac *dp)
{
	int next_tick = 3*HZ;
	int i;
	
	printk("immac up\n", dp->rx_ring, dp->tx_ring,dp->setup_frame);
	printk("rx_ring = %#x tx_ring = %#x setup_frame = %#x\n", dp->rx_ring, dp->tx_ring,dp->setup_frame);
	
	CKMAC_REG_BASEADDR[CSR3] = dp->rx_ring;
	CKMAC_REG_BASEADDR[CSR4] = dp->tx_ring;
	
	printk("CSR3 = %#x CSR4 = %#x\n", CKMAC_REG_BASEADDR[CSR3],CKMAC_REG_BASEADDR[CSR4]);
	
	dp->cur_rx = dp->cur_tx = 0;
	dp->dirty_rx = dp->dirty_tx = 0;
	
	{
		/* This is set_rx_mode(), but without starting the transmitter. */
		u16 *eaddrs = (u16 *)dp->dev->dev_addr;
		u16 *setup_frm = &dp->setup_frame[15*6];
		
		/* 21140 bug: you must add the broadcast address. */
		memset(dp->setup_frame, 0xff, sizeof(dp->setup_frame));
		
		/* Fill the final entry of the table with our physical address. */
		*setup_frm++ = eaddrs[0]; *setup_frm++ = eaddrs[0];
		*setup_frm++ = eaddrs[1]; *setup_frm++ = eaddrs[1];
		*setup_frm++ = eaddrs[2]; *setup_frm++ = eaddrs[2];
		
		/* Put the setup frame on the Tx list. */
		dp->tx_ring[dp->cur_tx].length = cpu_to_le32(0x08000000 | 192); //0x08000000|192
		dp->tx_ring[dp->cur_tx].buffer1 = cpu_to_le32(dp->setup_frame);
		dp->tx_ring[dp->cur_tx].status = cpu_to_le32(DescOwned);
		
		dp->cur_tx++;
	}

	//media setup  immac
	/* Start the chip's Tx to process setup frame. */
	immac_stop_rxtx(dp);
	//barrier();
	udelay(5);
	
	CKMAC_REG_BASEADDR[CSR6] = CKMAC_REG_BASEADDR[CSR6]|TxOn; 
	
	CKMAC_REG_BASEADDR[CSR1] = 0; //George added
	
	/* Enable interrupts by setting the interrupt mask. */
	CKMAC_REG_BASEADDR[CSR5] = 0x0001c9e7;
	CKMAC_REG_BASEADDR[CSR7]  = 0x0001c9e7;
	immac_start_rxtx(dp);
	CKMAC_REG_BASEADDR[CSR2] = 0; 
}

static int immac_init(struct immac *dp, int from_irq)
{
	
	// Latch current counters into statistics.
	//immac_clean_stats(dp);
	
	// Alloc and reset the tx/rx descriptor chains.
	immac_begin_auto_negotiation(dp);
	
	immac_init_rings(dp, from_irq);
	
	immac_up(dp);
	
	netif_start_queue (dp->dev);
	
	// Stop transmitter and receiver.
	//immac_stop();
	
	// Set hardware ethernet address.
	//immac_set_macaddr(dp);

	return 0;
}

/* Error interrupts get sent here. */
static void immac_error(struct immac *dp, unsigned long mac_status)
{
	printk(KERN_ERR "cskymac_error: ");
	if (mac_status & CKMAC_INT_BER) 
	{
		printk("Bus error,");
		CKMAC_REG_BASEADDR[CKMAC_INT] |= CKMAC_INT_BER;
	}
	else if (mac_status & CKMAC_INT_RXE)
	{
		printk("Receive error,");
		CKMAC_REG_BASEADDR[CKMAC_INT] |= CKMAC_INT_RXE;
		// the detailed rx_errors status of BD
	}
	else if (mac_status & CKMAC_INT_TXE)
	{
		printk("Transmit error,");
		CKMAC_REG_BASEADDR[CKMAC_INT] |= CKMAC_INT_TXE;
		// the detailed tx_errors status of BD
	}

	printk(" reset mac\n");
	immac_init(dp, 1);
}

int immac_refill_rx(struct immac *dp)
{
	int entry;
	int refilled = 0;

	/* Refill the Rx ring buffers. */
	for (; dp->cur_rx - dp->dirty_rx > 0; dp->dirty_rx++) 
	{
		entry = dp->dirty_rx % RX_RING_SIZE;
		if ( dp->rx_ring[entry].buffer1 == NULL ) 
		{
			dp->rx_ring[entry].buffer1 = cpu_to_le32( avi_basedesaddr + entry * PKT_BUF_SZ );
			refilled++;
		}
		dp->rx_ring[entry].status = cpu_to_le32(DescOwned);
	}
	
	CKMAC_REG_BASEADDR[CSR1] = 0x55;
	CKMAC_REG_BASEADDR[CSR2] = 0x55;
	
	return refilled;
}


static int immac_rx(struct immac *dp)
{
	int entry = dp->cur_rx % RX_RING_SIZE;
	int rx_work_limit = dp->dirty_rx + RX_RING_SIZE - dp->cur_rx;
	int received = 0;

	/* If we own the next entry, it is a new packet. Send it up. */
	while ( ! (dp->rx_ring[entry].status & cpu_to_le32(DescOwned))) 
	{
		s32 status = le32_to_cpu(dp->rx_ring[entry].status);
		
		if (--rx_work_limit < 0)
			break;
			
		if ((status & 0x38008300) != 0x0300) 
		{
			if ((status & 0x38000300) != 0x0300) 
			{
				/* Ingore earlier buffers. */
				if ((status & 0xffff) != 0x7fff) 
				{
					dp->stats.rx_length_errors++;
				}
			}
			else if (status & RxDescFatalErr) 
			{
				/* There was a fatal error. */
				dp->stats.rx_errors++; /* end of a packet.*/
				if (status & 0x0890) dp->stats.rx_length_errors++;
				if (status & 0x0004) dp->stats.rx_frame_errors++;
				if (status & 0x0002) dp->stats.rx_crc_errors++;
				if (status & 0x0001) dp->stats.rx_fifo_errors++;
			}
		} 
		else 
		{
			/* Omit the four octet CRC from the length. */
			short pkt_len = ((status >> 16) & 0x7ff) - 4;
			struct sk_buff *skb;
			int rx_status;
			int ip_len;

#ifndef final_version
			if (pkt_len > 1518) 
			{
				printk(KERN_WARNING " Bogus packet size of %d (%#x).\n",
					  pkt_len, pkt_len);
				pkt_len = 1518;
				dp->stats.rx_length_errors++;
			}
#endif
			
			/* Check if the packet is long enough to accept without copying
			   to a minimally-sized skbuff. */
			if (pkt_len < tulip_rx_copybreak && (skb = dev_alloc_skb(pkt_len + 2)) != NULL) 
			{
				unsigned char *p;
				skb->dev = dp->dev;
				skb_reserve(skb, 2);	/* 16 byte align the IP header */
				
				p = le32_to_cpu( (unsigned char *)(dp->rx_ring[entry].buffer1) );
				//memcpy ( skb->data, p, pkt_len);
				eth_copy_and_sum(skb, p, pkt_len, 0);				
				
				skb_put(skb, pkt_len);
			}
			else 
			{
				printk("skb 1\n");
				goto _drop_it;
			}			
			
			rx_status = (skb->data[12] << 8) | skb->data[13];
			ip_len = ((skb->data[16] << 8) | skb->data[17]+14);
			
			if (rx_status == ETH_P_ARP)
			{
				
			}
			else if (rx_status == ETH_P_IP)
			{

			}
			else
			{ 

			}
			
			skb->protocol = eth_type_trans(skb, dp->dev);
			netif_rx(skb);

			dp->dev->last_rx = jiffies;
			dp->stats.rx_packets++;
			dp->stats.rx_bytes += pkt_len;
		}
		
_drop_it:
		received++;
		entry = (++dp->cur_rx) % RX_RING_SIZE;
		//InvCache();
		//wmb();
	}
	return received;
}

static void immac_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct immac *dp = (struct immac *)dev_id;
	unsigned int csr5, tmpcsr5;
	int entry;
	int missed;
	int rx = 0;
	int tx = 0;
	int oi = 0;
	int maxrx = RX_RING_SIZE;
	int maxtx = TX_RING_SIZE;
	int maxoi = TX_RING_SIZE;
	unsigned int work_count = tulip_max_interrupt_work;

	/* Let's see whether the interrupt really is for us */
//	InvCache();
	csr5 = CKMAC_REG_BASEADDR[CSR5];
	
/*
	tmpcsr5 = CKMAC_REG_BASEADDR[CSR5];
	
	if( csr5 != tmpcsr5 )
	{
		printk("err\n");
		csr5 = tmpcsr5;
	}
	*/

	if ((csr5 & (NormalIntr|AbnormalIntr)) == 0)
		return;

	dp->nir++;

	do 
	{
		/* Acknowledge all of the current interrupt sources ASAP. */
		CKMAC_REG_BASEADDR[CSR5] = csr5&0x0001ffff;
		if (csr5 & (RxIntr | RxNoBuf)) 
		{
			rx += immac_rx(dp);
			immac_refill_rx(dp);
		}
		
		if (csr5 & (TxNoBuf | TxDied | TxIntr | TimerInt)) 
		{
			unsigned int dirty_tx;
			spin_lock(&dp->lock);
			
			for (dirty_tx = dp->dirty_tx; dp->cur_tx - dirty_tx > 0; dirty_tx++) 
			{
				int entry = dirty_tx % TX_RING_SIZE;
				int status;
				
				status = le32_to_cpu(dp->tx_ring[entry].status);
				
				if (status < 0)
					break;			/* It still has not been Txed */

				/* Check for Rx filter setup frames. */
				if (dp->tx_skbs[entry] == NULL) 
				{
					continue;
				}

				if (status & 0x8000) 
				{
					/* There was an major error, log it. */
					dp->stats.tx_errors++;
					if (status & 0x4104) dp->stats.tx_aborted_errors++;
					if (status & 0x0C00) dp->stats.tx_carrier_errors++;
					if (status & 0x0200) dp->stats.tx_window_errors++;
					if (status & 0x0002) dp->stats.tx_fifo_errors++;
					if ((status & 0x0080) && dp->full_duplex == 0)
						dp->stats.tx_heartbeat_errors++;
				} else 
				{
					dp->stats.tx_bytes += dp->tx_skbs[entry]->len;
					dp->stats.collisions += (status >> 3) & 15;
					dp->stats.tx_packets++;
				}
				
				dev_kfree_skb_irq(dp->tx_skbs[entry]);
				dp->tx_skbs[entry] = NULL;
				dp->tx_ring[entry].status = 0x00000000;
				tx++;
			}

#ifndef final_version
			if (dp->cur_tx - dirty_tx > TX_RING_SIZE) {
				printk(KERN_ERR ": Out-of-sync dirty pointer, %d vs. %d.\n",
					   dirty_tx, dp->cur_tx);
				dirty_tx += TX_RING_SIZE;
			}
#endif
			
			if (dp->cur_tx - dirty_tx < TX_RING_SIZE - 2)
			{
				netif_wake_queue(dp->dev);
			}

			dp->dirty_tx = dirty_tx;
			if (csr5 & TxDied) 
			{
				immac_restart_rxtx(dp);
			}
			
			spin_unlock(&dp->lock);
		}

		/* Log errors. */
		if (csr5 & AbnormalIntr) 
		{	
			/* Abnormal error summary bit. */
			if (csr5 == 0xffffffff)
				break;
				
			printk("abnormalintr %x\n", csr5);
			
			if (csr5 & TxJabber) 
				dp->stats.tx_errors++;
				
			if (csr5 & TxFIFOUnderflow)
			{
				if ((dp->csr6 & 0xC000) != 0xC000)
					dp->csr6 += 0x4000;	/* Bump up the Tx threshold */
				else
					dp->csr6 |= 0x00200000;  /* Store-n-forward. */
					
				/* Restart the transmit process. */
				immac_restart_rxtx(dp);
				CKMAC_REG_BASEADDR[CSR1] = 0;
			}
			if (csr5 & RxDied) 
			{		
				/* Missed a Rx frame. */
				dp->stats.rx_missed_errors +=CKMAC_REG_BASEADDR[CSR8] & 0xffff;
				dp->stats.rx_errors++;
				immac_start_rxtx(dp);
			}
			
			/*
			 * NB: t21142_lnk_change() does a del_timer_sync(), so be careful if this
			 * call is ever done under the spinlock
			 */
			if (csr5 & (TPLnkPass | TPLnkFail | 0x08000000)) 
			{
				if (dp->link_change)
					(dp->link_change)(dp->dev, csr5);
			}
			
			if (csr5 & SytemError) 
			{
				int error = (csr5 >> 23) & 7;
				/* oops, we hit a PCI error.  The code produced corresponds
				 * to the reason:
				 *  0 - parity error
				 *  1 - master abort
				 *  2 - target abort
				 * Note that on parity error, we should do a software reset
				 * of the chip to get it back into a sane state (according
				 * to the 21142/3 docs that is).
				 *   -- rmk
				 */
			}
			
			/* Clear all error sources, included undocumented ones! */
			CKMAC_REG_BASEADDR[CSR5] =  0x0800f7ba;
			oi++;
		}
		
		if (csr5 & TimerInt) 
		{
			printk("TimerInt \n");
			CKMAC_REG_BASEADDR[CSR7]  = 0x0001c9e7;
			dp->ttimer = 0;
			oi++;
		}
		
		if (tx > maxtx || rx > maxrx || oi > maxoi) 
		{
			printk("tx error 2\n");
			CKMAC_REG_BASEADDR[CSR5] = 0x8001ffff;
			
          	if (dp->flags & HAS_INTR_MITIGATION) 
          	{
				/* Josip Loncaric at ICASE did extensive experimentation
				to develop a good interrupt mitigation setting.*/
				CKMAC_REG_BASEADDR[CSR11] = 0x8b240000;
			}  
			break;
		}
		work_count--;
		if (work_count == 0)
			break;
		csr5 = CKMAC_REG_BASEADDR[CSR5];
	} while ((csr5 & (NormalIntr|AbnormalIntr)) != 0);

	immac_refill_rx(dp);

	/* check if the card is in suspend mode */
	entry = dp->dirty_rx % RX_RING_SIZE;
	
	if ( dp->rx_ring[entry].buffer1 == NULL ) 
	{
		if (dp->ttimer == 0 || (CKMAC_REG_BASEADDR[CSR11]& 0xffff) == 0) 
		{
			CKMAC_REG_BASEADDR[CSR7] = 0x0001c9e7|TimerInt;
			CKMAC_REG_BASEADDR[CSR5] = TimerInt;
			CKMAC_REG_BASEADDR[CSR11] = 12; 
			dp->ttimer = 1;
		}
	}
	
	if ((missed = CKMAC_REG_BASEADDR[CSR8]& 0x1ffff)) 
	{
		dp->stats.rx_dropped += missed & 0x10000 ? 0x10000 : missed;
	}
}

static int immac_open(struct net_device *dev)
{
	struct immac	*dp = (struct immac *) dev->priv;
	int 		ret;
	volatile unsigned int * icrp;

	init_timer(&dp->immac_timer);

	printk("immac_open start ****\n");
	
	icrp = (volatile unsigned int *) (CKSIM_PIC_BASE);
	
	/* Set interrupt controller for uart */
   icrp[CKSIM_PIC_MODE] |= CKSIM_PIC_POS3; /* Pose level sensitive */
   icrp[CKSIM_PIC_MODE] |= CKSIM_PIC_EDGE3; /* Edge trigger */
   icrp[CKSIM_PIC_MASK] &= ~CKSIM_INT3_MASK; /* Unmask uart interrupt */
	
	ret = request_irq(dev->irq, &immac_interrupt, NULL, dev->name, dp);
	//ret = request_irq(34, &immac_interrupt, NULL, dev->name, dp);
	ret = immac_init(dp, 0);
	if (ret)
	{
		return ret;
	} 
	
	printk("irq ret = %#x CSR6 = %#x irq = %#x\n",ret,CKMAC_REG_BASEADDR[CSR6],dev->irq);
	
	return ret;
}

static int immac_close(struct net_device *dev)
{
	struct immac *dp = (struct immac *) dev->priv;

	//del_timer(&dp->immac_timer);
	//dp->timer_state = asleep;
	//dp->timer_ticks = 0;

	//immac_stop();
	//immac_clean_rings(dp);
	//free_irq(dev->irq, dp);
	return 0;
}

static void immac_tx_timeout(struct net_device *dev)
{
	unsigned int csr5;
	struct immac *dp = (struct immac *) dev->priv;

	printk("time out \n");
	
	csr5 = CKMAC_REG_BASEADDR[CSR5];
	printk("csr5 = %#x\n", csr5);
	
	dp->enet_stats.tx_errors++;
	immac_init(dp, 0);
	netif_wake_queue(dev);
}

/* 
Put a packet on the wire. 
Initial a transmision for device.
*/
static int immac_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct immac *dp = (struct immac *)dev->priv;
	int entry;
	u32 flag;
	dma_addr_t mapping;

	spin_lock_irq(&dp->lock);
	
	/* Calculate the next Tx descriptor entry. */
	entry = dp->cur_tx % TX_RING_SIZE;

	dp->tx_skbs[entry] = skb;
	dp->tx_ring[entry].buffer1 = cpu_to_le32(skb->data);
	
	
	//CKMAC_REG_BASEADDR[CSR5] = 0x0001c9e7 + 0x400;

#if 0
	flag = 0xe0000000; /* Tx-done intr. */

#else

	if (dp->cur_tx - dp->dirty_tx < TX_RING_SIZE/2) 
	{
		/* Typical path */
		flag = 0x60000000; /* No interrupt */
	} 
	else if (dp->cur_tx - dp->dirty_tx == TX_RING_SIZE/2) 
	{
		flag = 0xe0000000; /* Tx-done intr. */
	} 
	else if (dp->cur_tx - dp->dirty_tx < TX_RING_SIZE - 2) 
	{
		flag = 0x60000000; /* No Tx-done intr. */
	} 
	else 
	{		
		/* Leave room for set_rx_mode() to fill entries. */
		flag = 0xe0000000; /* Tx-done intr. */
		printk("ximit stop queue \n");
		netif_stop_queue(dev);
	}
#endif
	
	if (entry == TX_RING_SIZE-1)
		flag = 0xe0000000 | DESC_RING_WRAP;

	dp->tx_ring[entry].length = cpu_to_le32(skb->len | flag);
	
	/* if we were using Transmit Automatic Polling, we would need a
	 * wmb() here. */
	dp->tx_ring[entry].status = cpu_to_le32(DescOwned);
	//wmb();

	dp->cur_tx++;

	/* Trigger an immediate transmit demand. */
	CKMAC_REG_BASEADDR[CSR1] = 0x55;
	CKMAC_REG_BASEADDR[CSR2] = 0x55;
	
	spin_unlock_irq(&dp->lock);

	dp->dev->trans_start = jiffies;
	
	//printk("%#x %#x %#x %#x\n", skb->data[0], skb->data[1], skb->data[2], skb->data[3] );

	return 0;
}

static struct net_device_stats *immac_get_stats(struct net_device *dev)
{
	struct immac *dp = (struct immac *) dev->priv;

	return &dp->enet_stats;
}

#undef set_bit_le
#define set_bit_le(i,p) do { ((char *)(p))[(i)/8] |= (1<<((i)%8)); } while(0)

static void build_setup_frame_hash(u16 *setup_frm, struct net_device *dev)
{
	struct immac *dp =  (struct immac *)dev->priv;
	u16 hash_table[32];
	struct dev_mc_list *mclist;
	int i;
	u16 *eaddrs;

	memset(hash_table, 0, sizeof(hash_table));
	set_bit_le(255, hash_table); 			/* Broadcast entry */
	
	/* This should work on big-endian machines as well. */
	for (i = 0, mclist = dev->mc_list; mclist && i < dev->mc_count;
	     i++, mclist = mclist->next) 
	{
		int index = ether_crc_le(ETH_ALEN, mclist->dmi_addr) & 0x1ff;
		set_bit_le(index, hash_table);
	}
	
	for (i = 0; i < 32; i++) 
	{
		*setup_frm++ = hash_table[i];
		*setup_frm++ = hash_table[i];
	}
	
	setup_frm = &dp->setup_frame[13*6];

	/* Fill the final entry with our physical address. */
	eaddrs = (u16 *)dev->dev_addr;
	*setup_frm++ = eaddrs[0]; *setup_frm++ = eaddrs[0];
	*setup_frm++ = eaddrs[1]; *setup_frm++ = eaddrs[1];
	*setup_frm++ = eaddrs[2]; *setup_frm++ = eaddrs[2];
}

static void build_setup_frame_perfect(u16 *setup_frm, struct net_device *dev)
{
	struct immac *dp =  (struct immac *)dev->priv;
	struct dev_mc_list *mclist;
	int i;
	u16 *eaddrs;

	/* We have <= 14 addresses so we can use the wonderful
	   16 address perfect filtering of the Tulip. */
	for (i = 0, mclist = dev->mc_list; i < dev->mc_count;
	     i++, mclist = mclist->next) 
	{
		eaddrs = (u16 *)mclist->dmi_addr;
		*setup_frm++ = *eaddrs; *setup_frm++ = *eaddrs++;
		*setup_frm++ = *eaddrs; *setup_frm++ = *eaddrs++;
		*setup_frm++ = *eaddrs; *setup_frm++ = *eaddrs++;
	}
	
	/* Fill the unused entries with the broadcast address. */
	memset(setup_frm, 0xff, (15-i)*12);
	setup_frm = &dp->setup_frame[15*6];

	/* Fill the final entry with our physical address. */
	eaddrs = (u16 *)dev->dev_addr;
	*setup_frm++ = eaddrs[0]; *setup_frm++ = eaddrs[0];
	*setup_frm++ = eaddrs[1]; *setup_frm++ = eaddrs[1];
	*setup_frm++ = eaddrs[2]; *setup_frm++ = eaddrs[2];
}


static void immac_set_multicast(struct net_device *dev)
{
	struct immac *dp =  (struct immac *)dev->priv;
	int csr6;

	printk("set multicast \n");
	csr6 = CKMAC_REG_BASEADDR[CSR6] & ~0x00D5;
	dp->csr6 &= ~0x00D5;
	
	if (dev->flags & IFF_PROMISC) 
	{			/* Set promiscuous. */
		dp->csr6 |= AcceptAllMulticast | AcceptAllPhys;
		csr6 |= AcceptAllMulticast | AcceptAllPhys;
		/* Unconditionally log net taps. */
		printk(KERN_INFO "%s: Promiscuous mode enabled.\n", dev->name);
	} 
	else if ((dev->mc_count > 1000)  ||  (dev->flags & IFF_ALLMULTI)) 
	{
		/* Too many to filter well -- accept all multicasts. */
		dp->csr6 |= AcceptAllMulticast;
		csr6 |= AcceptAllMulticast;
		printk("se IFF_ALLMULTI \n");
	} 
	else	if (dp->flags & MC_HASH_ONLY) 
	{
		printk("MC_HASH_ONLY \n");
	} 
	else 
	{
		unsigned long flags;
		u32 tx_flags = 0x08000000 | 192;

		printk("build hash \n");
		
		/* Note that only the low-address shortword of setup_frame is valid!
		   The values are doubled for big-endian architectures. */
		if (dev->mc_count > 14) 
		{ 
			/* Must use a multicast hash table. */
			build_setup_frame_hash(dp->setup_frame, dev);
			tx_flags = 0x08400000 | 192;
		} 
		else 
		{
			build_setup_frame_perfect(dp->setup_frame, dev);
		}

		spin_lock_irqsave(&dp->lock, flags);

		if (dp->cur_tx - dp->dirty_tx > TX_RING_SIZE - 2) 
		{
			/* Same setup recently queued, we need not add it. */
		} 
		else 
		{
			unsigned int entry;
			int dummy = -1;

			/* Now add this frame to the Tx list. */
			entry = dp->cur_tx++ % TX_RING_SIZE;
			if (entry != 0)
			{
				/* Avoid a chip errata by prefixing a dummy entry. */
				dp->tx_ring[entry].length = (entry == TX_RING_SIZE-1) ? cpu_to_le32(DESC_RING_WRAP) : 0;
				dp->tx_ring[entry].buffer1 = 0;
				
				/* Must set DescOwned later to avoid race with chip */
				dummy = entry;
				entry = dp->cur_tx++ % TX_RING_SIZE;
			}	

			/* Put the setup frame on the Tx list. */
			if (entry == TX_RING_SIZE-1)
				tx_flags |= DESC_RING_WRAP;		/* Wrap ring. */
			dp->tx_ring[entry].length = cpu_to_le32(tx_flags);
			dp->tx_ring[entry].buffer1 = cpu_to_le32(dp->setup_frame);
			dp->tx_ring[entry].status = cpu_to_le32(DescOwned);
			if (dummy >= 0)
				dp->tx_ring[dummy].status = cpu_to_le32(DescOwned);
			if (dp->cur_tx - dp->dirty_tx >= TX_RING_SIZE - 2)
				netif_stop_queue(dev);

			/* Trigger an immediate transmit demand. */
			CKMAC_REG_BASEADDR[CSR1] = 0;
		}

		spin_unlock_irqrestore(&dp->lock, flags);
	}
	
	CKMAC_REG_BASEADDR[CSR6] = csr6;
}

static int __init immac_ether_init()
{
	struct immac *dp;
	struct net_device *dev;
	int	i;

	CKMAC_REG_BASEADDR[CSR0] =   0x100281;
	/* Get a new device struct for this interface. */
	dev = init_etherdev(NULL, sizeof(struct immac));
	if (!dev)
	{
		return -ENOMEM;
	}
	
	SET_MODULE_OWNER(dev);
	
	printk(KERN_INFO "%s: IM MAC 10/100baseT Ethernet\n", dev->name);
	printk(KERN_INFO "%s\n", version);

	/* Report what we have found to the user. */

	dev->base_addr = CKMAC_REG_BASEADDR;

	CKMAC_REG_BASEADDR[CSR0] =  0x100280; //set bus mode
	
	/* Clear the missed-packet counter. */
	i = CKMAC_REG_BASEADDR[CSR8] ;

	printk("CSR0 status %x\n", CKMAC_REG_BASEADDR[CSR0]);
	
	immac_get_macaddr(dev->dev_addr);
	
	dev->addr_len = 6;
	printk("MAC Address : ");
	for (i = 0; i < 6; i++)
	{
		printk("%2.2x%c", dev->dev_addr[i], i == 5 ? ' ' : ':');
	}
	printk("\n");

	/* Setup softc, with backpointers to QEC and CSKY MAC SBUS device structs. */
	dp = dev->priv;

	spin_lock_init(&dp->lock);
	spin_lock_init(&dp->mii_lock);
	/* Stop the CSKY MAC. */
	//immac_stop();

	/* Init auto-negotiation timer state. */
	init_timer(&dp->immac_timer);
	dp->timer_state = asleep;
	dp->timer_ticks = 0;

	/* Backlink to generic net device struct. */
	dp->dev = dev;

	/* Set links to our IM MAC open and close routines. */
	dev->open = &immac_open;
	dev->stop = &immac_close;
	dev->hard_start_xmit = &immac_start_xmit;

	/* Set links to IM MAC statistic and multi-cast loading code. */
	dev->get_stats = &immac_get_stats;
	dev->set_multicast_list = &immac_set_multicast;

	dev->tx_timeout = &immac_tx_timeout;
	dev->watchdog_timeo = 5 * HZ;

	/* Finish net device registration. */
	dev->irq = CKMAC_IRQBASE;
	dev->dma = 0;

	dp->csr6 = 0x4000;  ////128 bytes transmit fifo threshold

	ether_setup(dev);
	root_immac_dev = dp;

	dp->phys = CKMAC_PHY_ADDR;	
	
	/*
	i = immac_phyr_read(dp,0);
	printk("phys read %x \n", i);

	i = immac_phyr_read(dp,3);
	printk("phys read %x \n", i);
	*/

	return 0;

fail_and_cleanup:
	/* Something went wrong, undo whatever we did so far. */
	/* Free register mappings if any. */
	unregister_netdev(dev);
	kfree(dev);
	return -ENODEV;
}

static void __exit immac_cleanup(void)
{
	struct immac *dp = root_immac_dev;

	unregister_netdev(dp->dev);
	kfree(dp->dev);
}

//George added
#define module_init(x)	__initcall(x);
#define module_exit(x)	__exitcall(x);


module_init(immac_ether_init);
module_exit(immac_cleanup);
MODULE_LICENSE("GPL");
