#include "spi.h"
#include "spi_flash_winbond.h"
#include "armboot.h"
#include <config.h>
#define SPI_MOD

#ifdef SPI_MOD            
#  define PHMSG(fmt, args...) printf(fmt, ## args)
#else
#  define PHMSG(fmt, args...)
#endif

#define SPI_PCK_MOD      1

/*		$Function		:	void spi_initialize()
==	================================================================================
==		Description		:	initialize the spi Interface that access sd card
==		Argument		:	NULL
==		Return			:	NULL
==		Modification	:	2007-12-11	wangdx Create
==      Modification    :   2013-10-17  zhangjz 
==	================================================================================
*/
void spi_initialize()
{
	unsigned int modecfg = 0;
	int phase_clk = 1,bits = 0x7,clock_delay_times = 10,cpha = 1,cpol = 1,lsb = 0,pck_mod = SPI_PCK_MOD, hold_en = 0;
	int flash_mod = 1,tx_delay = 1,rx_delay = 1,reset = 1,send_adv = 1;
	int isr_enable = 0,txdffi_level = 0x10,rxdffi_level = 0x10,copmleteie = 1;
	int trans_cnt = 0;
	int baud = 0xf;
	int started = 0,cr_cs = 0xF,reg_ff_st_en = 0,reg_crc_en = 0,reg_bmult = 0,direction = 0;


	modecfg = 	(phase_clk << 31) +
				(bits << 24) +
				(clock_delay_times << 16) +
				(rx_delay << 15) +
				(cpha << 14) +
				(cpol << 13) +
				(lsb << 12) +
				(pck_mod << 10) +
				(hold_en << 9) +
				(flash_mod << 8) +
				(tx_delay << 1) + 1;
				
	/*init the 1th spi*/
	SPI_MDR(SPI1TH) = modecfg;
	SPI_IQR(SPI1TH) = (txdffi_level <<16) + (rxdffi_level << 8) + (copmleteie << 2) + ( 1UL << 1) +1;
	SPI_TCR(SPI1TH) = trans_cnt;
	SPI_BDR(SPI1TH) = baud;
	SPI_CTR(SPI1TH) = (reg_ff_st_en << 26) +
					  (reg_crc_en << 25) +
			          (reg_bmult << 24) +
			          (cr_cs << 8) + (direction << 1) + (started); 
	/*init the 2th spi*/
	SPI_MDR(SPI2TH) = modecfg;
	SPI_IQR(SPI2TH) = (txdffi_level <<16) + (rxdffi_level << 8) + (copmleteie << 2) + ( 1UL << 1) +1;
	SPI_TCR(SPI2TH) = trans_cnt;
	SPI_BDR(SPI2TH) = baud;
	SPI_CTR(SPI2TH) = (reg_ff_st_en << 26) +
					  (reg_crc_en << 25) +
					  (reg_bmult << 24) +
					  (cr_cs << 8) + (direction << 1) + (started);		
			  
}

/*		$Function		:	int spi_spi_normal_sendbytes()
==	================================================================================
==		Description		:	send a byte by SPI interface spi_cs is 0 or 1
==		Argument		:	UINT8 byte: the byte that will be send
==		Return			:	NULL
==		Modification	:	2007-12-11	wangdx Create
==      Modification    :   2013-10-17  zhangjz
==	================================================================================
*/
int spi_normal_sendbytes(unsigned spi_cs, unsigned int *tbuf, unsigned int tcnt )
{
	unsigned int i;
	unsigned int cnt;
	unsigned int timeout;
	unsigned int *p = NULL;
	
	/* check parameter */
	if ( ( NULL == tbuf ) || ( 0 == tcnt ))
	{
		return 0;
	}
	p = tbuf;
	
	/* set the spi mode from flash to normal */
	SPI_MDR(spi_cs) &= ~( 1 << AVI_SPI_MDR_FMODE );
	cnt = tcnt >> 2;
	if( tcnt % 4 )
	{
		cnt++;
	}
	
    for ( i = 0; i < cnt; i ++ )
    {
        SPI_SDR(spi_cs) = *p++;
    }

	SPI_TCR(spi_cs) = tcnt - 1;
    
    SPI_CTR(spi_cs) |= 0x03;
	
	while (((SPI_STR(spi_cs) >> 16) & 0x0f) != INT_GEN)
	{
		 timeout ++;
		 if( timeout > 0xffffff )
		 	{
		 		PHMSG("\n[ SPI_STR = %#x ]\n",  SPI_STR(spi_cs));
				SPI_STR(spi_cs) = 0;
				SPI_SDR(spi_cs) = 0;
		 		PHMSG("\n[ time out in the spi_sendbytes function! ]\n");
		 		break;
		 	}
	}
	SPI_STR(spi_cs) |= (1UL << 20);

	/* set the spi mode from normal to flash */
	SPI_MDR(spi_cs) |= ( 1 << AVI_SPI_MDR_FMODE );
	
	return tcnt;
}
/*		$Function		:	int spi_sendbyte()
==	================================================================================
==		Description		:	send a byte by SPI interface spi_cs is 0 or 1
==		Argument		:	UINT8 byte: the byte that will be send
==		Return			:	NULL
==		Modification	:	2007-12-11	wangdx Create
==      Modification    :   2013-10-17  zhangjz
==	================================================================================
*/
int spi_sendbytes( unsigned char spi_cs, unsigned int *tbuf, unsigned int cmdcnt, unsigned int tcnt )
{
	unsigned int *p = NULL;
	unsigned int i;
	unsigned int tmp;
	unsigned int cnt;
	unsigned int timeout = 0;
	
	if (( NULL == tbuf ) || ( 0 == tcnt ) || ( 0 == cmdcnt ))
	{
		return 0;
	}
	
	p = tbuf;
	cnt = ( cmdcnt + tcnt ) >> 2;
	if  ( ( cmdcnt + tcnt ) % 0x04 ) 
    {
        cnt ++;
    }
    for ( i = 0; i < cnt; i ++ )
    {
        SPI_SDR(spi_cs) = *p++;
    }

	if ( 0 == tcnt )
	{
		SPI_TCR(spi_cs) = 0;
	}
	else
	{
		SPI_TCR(spi_cs) = tcnt - 1 ;
	}
    
    tmp = SPI_CTR(spi_cs);
    tmp &= AVI_SPI_CMDLEN_MASK;
    tmp |= ( ( ( cmdcnt - 1 ) << 16 ) + 0x3 );
    
	SPI_CTR(spi_cs) = tmp;
	
	while (((SPI_STR(spi_cs) >> 16) & 0x0f) != INT_GEN)
	{
		 timeout ++;
		 if(timeout > 0xffffff )
		 	{
		 		PHMSG("\n[ SPI_STR = %#x ]\n",  SPI_STR(spi_cs));
				SPI_STR(spi_cs) = 0;
				SPI_SDR(spi_cs) = 0;
		 		PHMSG("\n[ time out in the spi_sendbytes function! ]\n");
		 		break;
		 	}
	}
	SPI_STR(spi_cs) |= (1UL << 20);
	return cnt;

}

/*		$Function		:	int spi_normal_readbytes()
==	================================================================================
==		Description		:	read a byte by SPI interface spi_cs is 0 or 1
==		Argument		:	UINT8 byte: the byte that will be read
==		Return			:	NULL
==		Modification	:	2007-12-11	wangdx Create
==      Modification    :   2013-10-17  zhangjz
==	================================================================================
*/
int spi_normal_readbytes( unsigned char spi_cs, unsigned int *rbuf, unsigned int rcnt )
{
    unsigned int i;
	unsigned int tmp;
	unsigned int cnt;
	unsigned int timeout;
	unsigned int *p = NULL;
	timeout = 0;	
	
	if ( (NULL == rbuf) || ( 0 == rcnt ) )
	{
		return 0;
	}
	/* set the spi mode from flash to normal */
	SPI_MDR(spi_cs) &= ~( 1 << AVI_SPI_MDR_FMODE );
	
	SPI_TCR(spi_cs) = ( rcnt - 1 );
    tmp = SPI_CTR(spi_cs);
    tmp &= AVI_SPI_CMDLEN_MASK;
	tmp |= 1UL;

	SPI_CTR(spi_cs) = tmp;

    p = rbuf;

    cnt = rcnt >> 2;
	if (  rcnt % 0x04  )
    {
        cnt ++;
    }

	while (((SPI_STR(spi_cs) >> 16) & 0x0f) != INT_GEN)
	{
		 timeout ++;
		 if( timeout > 0xffffff )
		 	{
		 		PHMSG("SPI_STR = %#x\n",  SPI_STR(spi_cs));
				SPI_STR(spi_cs) = 0;
				SPI_RDR(spi_cs) = 0;
		 		PHMSG("time out spi_readbytes function! \n");
		 		break;
		 	}
	}

    for ( i = 0 ; i < cnt; i ++ )
    {
        *p ++ = SPI_RDR(spi_cs);
    }
    
	SPI_STR(spi_cs) |= (1UL << 20);
	/* set the spi mode from normal to flash */
	SPI_MDR(spi_cs) |= ( 1 << AVI_SPI_MDR_FMODE );

	return rcnt;
}

/*		$Function		:	int spi_readbytes()
==	================================================================================
==		Description		:	read a byte by SPI interface spi_cs is 0 or 1
==		Argument		:	UINT8 byte: the byte that will be read
==		Return			:	NULL
==		Modification	:	2007-12-11	wangdx Create
==      Modification    :   2013-10-17  zhangjz
==	================================================================================
*/
int spi_readbytes( unsigned char spi_cs, unsigned int *rbuf, unsigned int cmdcnt, unsigned int rcnt )
{
    unsigned int i;
	unsigned int tmp;
	unsigned int cnt;
	unsigned int timeout;
	unsigned int *p = NULL;
	timeout = 0;	
	
	if ( (NULL == rbuf) || ( 0 == rcnt ) || ( 0 == cmdcnt ) )
	{
		return 0;
	}

	p = rbuf;
	
	cnt = cmdcnt >> 2;
	if ( cmdcnt % 0x04 )
    {
        cnt ++;
    }
	/* send the command to the fifo */
	for ( i = 0 ; i < cnt; i ++ )
	{
	    SPI_SDR(spi_cs) = *p ++;
	}
	
	SPI_TCR(spi_cs) = ( rcnt - 1 );
		
    tmp = SPI_CTR(spi_cs);
    tmp &= AVI_SPI_CMDLEN_MASK;
	tmp |= ( cmdcnt - 1 ) << 16;
	tmp |= 1UL;
		
	SPI_CTR(spi_cs) = tmp;
	
    p = rbuf;
    cnt = rcnt >> 2;
	if (  rcnt % 0x04  )
    {
        cnt ++;
    }
    
	while (((SPI_STR(spi_cs) >> 16) & 0x0f) != INT_GEN)
	{
		 timeout ++;
		 if( timeout > 0xffffff )
		 	{
		 		PHMSG("SPI_STR = %#x\n",  SPI_STR(spi_cs));
				SPI_STR(spi_cs) = 0;
				SPI_RDR(spi_cs) = 0;
		 		PHMSG("time out spi_readbytes function! \n");
		 		break;
		 	}
	}

    for ( i = 0 ; i < cnt; i ++ )
    {
        *p ++ = SPI_RDR(spi_cs);
        printf("%x ",*(p-1));
    }
	SPI_STR(spi_cs) |= (1UL << 20);

	return rcnt;
}
/*		$Function		:	void set_spi_dev_cs()
==	================================================================================
==		Description		:	set the SPI dev cs
==		Argument		:	cs is the cs of spi
                            value = 0/1
                            dev_cs is the cs of spi device 
                            value = 0 - 3
==		Return			:	NULL
==		Modification	:	2007-12-11	wangdx Create
==      Modification    :   2013-10-17  zhangjz
==	================================================================================
*/
static void set_spi_dev_cs( unsigned int cs, unsigned int dev_cs )
{
    unsigned int ctrlreg = 0x01;
    if(dev_cs > 3 || dev_cs < 0)
    {
        printf("set slave device out of range\n");
        return;
    }
	SPI_CTR(cs) &= ~((ctrlreg) << (8+dev_cs));	
}

static void clear_spi_dev_cs( unsigned int cs, unsigned int dev_cs )
{
	unsigned int ctrlreg = 0x01;
    if(dev_cs > 3 || dev_cs < 0)
    {
        printf("set slave device out of range\n");
        return;
    }
	SPI_CTR(cs) |= ((ctrlreg) << (8+dev_cs));	
}
int spi_read_flash_id( int *mf_id, int *dev_id )
{
	unsigned int buf = 0;
    set_spi_dev_cs(SPI1TH, 1);
	buf = SPI_CMD_READ_ID << 24;
	spi_readbytes( SPI1TH, &buf , 1 , 3);
	*mf_id = (buf >> 24) & 0xFF;
	*dev_id = (buf >> 8) & 0xFFFF;
    clear_spi_dev_cs(SPI1TH, 1);
	return 0;
}


/*		$Function		:	void set_spi_band()
==	================================================================================
==		Description		:	set the SPI band
==		Argument		:	cs is the cs of spi, dev_cs is the cs of spi device
==		Return			:	NULL
==		Modification	:	2007-12-11	wangdx Create
==      Modification    :   2013-10-17  zhangjz
==	================================================================================
*/
static void set_spi_band( unsigned int cs, unsigned int band )
{
	SPI_BDR(cs) = band;
}

/*		$Function		:	void set_spi_packbits()
==	================================================================================
==		Description		:	set the SPI pak bits
==		Argument		:	cs is the cs of spi, bits is the mode of pack
==		Return			:	NULL
==		Modification	:	2007-12-11	wangdx Create
==      Modification    :   2013-10-17  zhangjz
==	================================================================================
*/
static void set_spi_packbits( unsigned int cs, unsigned int bits )
{
	PHMSG("In the packbits!\n");
	SPI_MDR(cs) &= ~(0x1F << 24);
	SPI_MDR(cs) |= (bits << 24);
	PHMSG("%#x\n", SPI_MDR(cs));
}
/*		$Function		:	void set_spi_lsb()
==	================================================================================
==		Description		:	set the SPI lsb
==		Argument		:	cs is the cs of spi, dev_cs is the cs of spi device
==		Return			:	NULL
==		Modification	:	2007-12-11	wangdx Create
==      Modification    :   2013-10-17  zhangjz
==	================================================================================
*/
static void set_spi_lsb( unsigned int cs, unsigned int lsb )
{
	PHMSG("In the lsb!\n");
	SPI_MDR(cs) &= ~(1 << 12);
	SPI_MDR(cs) |= (lsb << 12);
	PHMSG("%#x\n", SPI_MDR(cs));	
}

