#include "types.h"
#include "hdcvi_upgrade.h"
#include "spi.h"
#include "w25Q64cv.h"
#include "armboot.h"

#define PER_READ_COUNT          128
#define PER_WRITE_COUNT         124
#define FLASH_PAGE_SIZE         256
#define FLASH_TOTAL_SIZE        4*1024*1024


static u8 g_u8FlashBuf[ 4 * 1024 ];




static int w25q64cv_chk_busy( u32 u32TimeOut )
{
    u32 timeout, cmd, buf;
    int ret;
    
    /* check status */
    timeout = 0;
    ret = 0;
    
    while( 1 )
    {
        timeout ++;
        
        cmd = CMD_STATUS_READ << 24;
        spi_flash_mode_send_for_recv( SPI1TH, cmd, 1, &buf, 1 );
        
        if ( 0 == ( buf & 0x1 ) )
        {
            break;
        }
		
		if( timeout > 0xFFFFF )
		{
		    ret = -1;
			printf("time out %s function! status = %#x \n",__FUNCTION__, buf );
			
			break;
		}
    }
    
    return ret;
     
}


int w25q64cv_read(u8 *buf,u32 addr,u32 count)
{  
    u32 baseaddr;
    u32 precnt, cnt;
    u8 *p;
    
    
    if(buf == NULL || addr > FLASH_TOTAL_SIZE || 0 == count)
    {
        return -1;
    }
    

    baseaddr = addr;
    precnt = 0;
    cnt = count;
    p = buf;
    
    //printf("r%#x->%#x\n", baseaddr, cnt );
    
    while( 1 )
    {
        
        if ( 0 == cnt )
        {
            break;
        }
    
        if ( cnt > PER_READ_COUNT )
        {
            precnt = PER_READ_COUNT;
        }
        else
        {
            precnt = cnt;
        }
        
        w25q64cv_page_read( p, baseaddr, precnt);

        /* inc env */
        p += precnt;
        baseaddr += precnt;
        cnt -= precnt;
        
    }
    
    return 0;
}


int w25q64cv_hdcvi_up_write(u8 *buf,u32 addr,u32 count,u8 endian)
{
    u32 baseaddr, writePageCnt = 0;
    u32 precnt, pagecnt, cnt;
    u8  *p;
    
    if(buf == NULL || addr > FLASH_TOTAL_SIZE || 0 == count)
    {
        return -1;
    }
    
    cnt = count;
    p = buf;
    baseaddr = addr;
    precnt = 0;
    pagecnt = 0;

    while( 1 )
    {
        if ( 0 == cnt )
        {
            break;
        }

        /* дַҳ룬ֹҳдflash */
        pagecnt = ( FLASH_PAGE_SIZE - ( baseaddr & ( FLASH_PAGE_SIZE - 1 ) ) );
        
        if (pagecnt > PER_WRITE_COUNT )
        {
            precnt = PER_WRITE_COUNT;
        }
        else
        {
            precnt = pagecnt;
        }


        /* жʣֹд */
        if ( precnt > cnt )
        {
            precnt = cnt;
        }
        
        if (w25q64cv_page_program( baseaddr, p, precnt, endian) == -1)
        {
            printf("w25q64cv_page_program error\n");
            return -1;
        }

        /* ÿ800 * 124 byte = 96 K ιһ, ظ,豣֤ʱ10 */
        if (WR_FLAH_WAIT_TIME == writePageCnt)
        { 
            hdcvi_wdt_feed();
            hdcvi_reply_up_result(STATE_UPGRADE_WR);
            writePageCnt = 0;
        }
        
        /* inc env */
        baseaddr += precnt;
        p += precnt;
        cnt -= precnt;
        writePageCnt++;
        
    }
   
    return 0;  
}


int w25q64cv_write(u8 *buf,u32 addr,u32 count,u8 endian)
{
    u32 baseaddr;
    u32 precnt, pagecnt, cnt;
    u8 *p;
    
    if(buf == NULL || addr > FLASH_TOTAL_SIZE || 0 == count)
    {
        //printf("please check argument in %s function! buf = 0x%08lx, addr = 0x%08lx,count = 0x%08lx\n",__FUNCTION__,buf,addr,count);
        return -1;
    }
    
    cnt = count;
    p = buf;
    baseaddr = addr;
    precnt = 0;
    pagecnt = 0;

    while( 1 )
    {
        if ( 0 == cnt )
        {
            break;
        }

        /* дַҳ룬ֹҳдflash */
        pagecnt = ( FLASH_PAGE_SIZE - ( baseaddr & ( FLASH_PAGE_SIZE - 1 ) ) );
        
        if ( pagecnt > PER_WRITE_COUNT )
        {
            precnt = PER_WRITE_COUNT;
        }
        else
        {
            precnt = pagecnt;
        }

        /* жʣֹд */
        if ( precnt > cnt )
        {
            precnt = cnt;
        }
        
        if (w25q64cv_page_program( baseaddr, p, precnt, endian) == -1)
        {
            printf("w25q64cv_page_program error\n");
            return -1;
        }
        
        /* inc env */
        baseaddr += precnt;
        p += precnt;
        cnt -= precnt;
        
    }
   
    return 0;  
}


int w25q64cv_page_program(u32 addr,u8 *data,u32 length,u8 endian)
{
    u32 cmd = 0;
    int ret;

    
    if( addr > FLASH_TOTAL_SIZE)
    {
        printf("program addr out of flash range!\n");
        return -1;
    }
    
    if(NULL == data || 0 == length || length > PER_WRITE_COUNT)
    {
        printf("input parameter err!\n");
        return -1;
    }
    
    set_spi_dev_cs(SPI1TH, W25Q64CV_ADDR);
    
    
    ret = w25q64cv_chk_busy( 0xFFFFF );
    
    if ( 0 != ret )
    {
        printf("flash is busy status!\n");
        clear_spi_dev_cs( SPI1TH, W25Q64CV_ADDR );
        
        return -1;
    }
    
    cmd = CMD_WRITE_EN << 24;
    spi_normal_mode_send( SPI1TH, &cmd, 1 );

    spi_flash_mode_send_cmd_data( SPI1TH, CMD_PAGE_PROG, addr, data, length, endian );
    
    ret = w25q64cv_chk_busy( 0xFFFFF );
    
    clear_spi_dev_cs( SPI1TH, W25Q64CV_ADDR );
    
    

    return 0;
}

int w25q64cv_page_read(u8 *buf,u32 addr,u32 count)
{  
    u32 senddata = 0;
    u32 wordcont;
    u32 temp = 0;
    u8 mod, i; 
    
    if(addr > FLASH_TOTAL_SIZE)
    {
        printf("program addr out of flash range!\n");
        return -1;
    }

    if(NULL == buf || 0 == count || count > PER_READ_COUNT)
    {
        printf("input parameter err!\n");
        return -1;
    }

    senddata = CMD_READ << 24;
    senddata |= addr;

  
    set_spi_dev_cs(SPI1TH, W25Q64CV_ADDR);

    spi_flash_mode_send_for_recv( SPI1TH, senddata, 4, (u32 *)buf, count );

    clear_spi_dev_cs(SPI1TH, W25Q64CV_ADDR);

    return 0;
}

int w25q64cv_erase(u32 startAddr,u32 sectorCont,u8 eraseMode)
{
    u32 i, j, cmd, baseaddr;
    u32 erase_block_size;
    u32 *pu32;
    u32 ret;
   
    if( startAddr % SECTOR_SIZE != 0 || 0 == sectorCont )
    {
        printf("erase sector addr invalud should be alignment sector,startAddr = 0x%08lx,sectorCont = 0x%x\n",startAddr,sectorCont);
        return -1;
    } 
    
    if( flash_erase_4k_mode == eraseMode )
    {
        erase_block_size = SECTOR_SIZE;
    }
    else if( flash_erase_32k_mode == eraseMode )
    {
        erase_block_size = BLOCK32_SIZE;      
    }
    else if( flash_erase_64k_mode == eraseMode )
    {
        erase_block_size = BLOCK64_SIZE;
    }else
    {
        printf(" erase block size err!\n");
        return -1;
    }
    
    
  
    /* loop to erase the blocks */
    baseaddr = startAddr;
    //for ( i = 0 ; i < sectorCont; i ++ )
    i = 0;
    
   
    while( 1 )
    {
        if ( i < sectorCont )
        {
            set_spi_dev_cs(SPI1TH, W25Q64CV_ADDR);   
             
            cmd = CMD_WRITE_EN << 24;
            spi_normal_mode_send(SPI1TH, &cmd, 1); 
            
            spi_flash_mode_send_cmd_addr( SPI1TH, eraseMode, baseaddr );
            
            w25q64cv_chk_busy( 0xFFFFF );
            
            clear_spi_dev_cs(0, 1);
            
            
            /* check */
            set_spi_dev_cs(SPI1TH, W25Q64CV_ADDR);  
            pu32 = ( u32 *)( g_u8FlashBuf );
            ret = 0;
            if (w25q64cv_read( pu32, baseaddr, SECTOR_SIZE )==-1)
            {
                printf("w25q64cv_read error\n");
                return -1;
            }
            clear_spi_dev_cs(0, 1);
            
            for ( j = 0; j < ( SECTOR_SIZE >> 2 ) ; j ++ )
            {
                if ( 0xFFFFFFFF != *pu32 ++ )
                {
                    ret = 1;
                    
                    printf("Erase Error\n");
                    
                    break;
                }
            }
            
            if ( 0 == ret )
            {
                baseaddr += erase_block_size;
                
                i ++;
            }
        }
        else
        {
            break;
        }
        //
        
    } 
    
    
    return 0;
}


int w25q64cv_chip_erase(void)
{
    u32 cmd,timeout,buf,*p;
    set_spi_dev_cs(SPI1TH, W25Q64CV_ADDR);
    cmd = CMD_WRITE_EN << 24;
    spi_normal_mode_send(SPI1TH, &cmd, 1);       
    cmd = CMD_CHIP_ERASE << 24;
    spi_normal_mode_send(SPI1TH, &cmd, 1);
    printf("chip erase...\n");
    
    /* check status */
    w25q64cv_chk_busy( 0xFFFFF );
    
    clear_spi_dev_cs(SPI1TH, W25Q64CV_ADDR);
    printf("Erase Done !\n"); 
    return 0;
}

int w25q64cv_read_device_id(u32 *mf_id, u32 *dev_id)
{ 
	u32 buf = 0,cmd = 0,readdata = 0;
    u16 devid;
    set_spi_dev_cs(SPI1TH, W25Q64CV_ADDR);
	cmd = CMD_READ_ID << 24;
	spi_flash_mode_send_for_recv(SPI1TH, cmd,1,&buf, 3);
	*mf_id = buf & 0xFF;
	devid = (buf >> 8) & 0xFFFF;
    *dev_id = SWAP16(devid);
    clear_spi_dev_cs(SPI1TH, W25Q64CV_ADDR);
	return 0;
}

