/*
 * (C) Copyright 2000
 * Wolfgang Denk, DENX Software Engineering, wd@denx.de.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

/*
 * FLASH support
 */
#include <armboot.h>
#include <command.h>
#include <cmd_boot.h>
#include <malloc.h>
#include "w25Q64cv.h"
#if (CONFIG_COMMANDS & CFG_CMD_FLASH)

extern flash_info_t flash_info[];	/* info for FLASH chips */
extern image_header_t header;	/* from cmd_bootm.c */
extern ulong _armboot_relocation;	//add by zxj 2008-07-28

ulong giFlashAppBaseAddr = 0;

/*
 * The user interface starts numbering for Flash banks with 1
 * for historical reasons.
 */

/*
 * this routine looks for an abbreviated flash range specification.
 * the syntax is B:SF[-SL], where B is the bank number, SF is the first
 * sector to erase, and SL is the last sector to erase (defaults to SF).
 * bank numbers start at 1 to be consistent with other specs, sector numbers
 * start at zero.
 *
 * returns:	1	- correct spec; *pinfo, *psf and *psl are
 *			  set appropriately
 *		0	- doesn't look like an abbreviated spec
 *		-1	- looks like an abbreviated spec, but got
 *			  a parsing error, a number out of range,
 *			  or an invalid flash bank.
 */
static int
abbrev_spec (char *str, flash_info_t ** pinfo, int *psf, int *psl)
{
  flash_info_t *fp;
  int bank, first, last;
  char *p, *ep;

  if ((p = strchr (str, ':')) == NULL)
    return 0;
  *p++ = '\0';

  bank = simple_strtoul (str, &ep, 10);
  if (ep == str || *ep != '\0' ||
      bank < 1 || bank > CFG_MAX_FLASH_BANKS ||
      (fp = &flash_info[bank - 1])->flash_id == FLASH_UNKNOWN)
    return -1;

  str = p;
  if ((p = strchr (str, '-')) != NULL)
    *p++ = '\0';

  first = simple_strtoul (str, &ep, 10);
  if (ep == str || *ep != '\0' || first >= fp->sector_count)
    return -1;

  if (p != NULL)
    {
      last = simple_strtoul (p, &ep, 10);
      if (ep == p || *ep != '\0' || last < first || last >= fp->sector_count)
	return -1;
    }
  else
    last = first;

  *pinfo = fp;
  *psf = first;
  *psl = last;

  return 1;
}
/************************************************************************************
 *
 *  added by chencb 060905
 *  save old environment to flash when you run da to rewrite armboot
 *  if return 0 success, else return -1 fail
 *
 ************************************************************************************/
static int do_save_env(bd_t * bd)
{
	extern int board_env_save(bd_t *bd, env_t *env, int size);

	setenv(bd, "restore", "1");  // set restore flag

	return board_env_save(bd, bd->bi_env, sizeof (env_t));
}
#if 0
int
do_flinfo (cmd_tbl_t * cmdtp, bd_t * bd, int flag, int argc, char *argv[])
{
  ulong bank;

  if (argc == 1)		/* print info for all FLASH banks */
    {
      for (bank = 0; bank < CFG_MAX_FLASH_BANKS; ++bank)
	{
	  printf ("\nBank # %ld: ", bank + 1);
	  flash_print_info (&flash_info[bank]);

	}
      return 0;
    }

  bank = simple_strtoul (argv[1], NULL, 16);
  if ((bank < 1) || (bank > CFG_MAX_FLASH_BANKS))
    {
      printf ("Only FLASH Banks # 1 ... # %d supported\n",
	      CFG_MAX_FLASH_BANKS);
      return 1;
    }
  printf ("\nBank # %ld: ", bank);
  flash_print_info (&flash_info[bank - 1]);
  return 0;
}

int
do_flerase (cmd_tbl_t * cmdtp, bd_t * bd, int flag, int argc, char *argv[])
{
  flash_info_t *info;
  ulong bank, addr_first, addr_last;
  int n, sect_first, sect_last;

  if (argc < 2)
    {
      printf ("Usage:\n%s\n", cmdtp->usage);
      return 1;
    }

  if (strcmp (argv[1], "all") == 0)
    {
      for (bank = 1; bank <= CFG_MAX_FLASH_BANKS; ++bank)
    	{
    	  printf ("Erase Flash Bank # %ld:\n", bank);
    	  info = &flash_info[bank - 1];
    	  n = flash_erase (info, 0, info->sector_count - 1);
    	  if (n < 0)
    	    {
    	      flash_perror (n);
    	      return 1;
    	    }
    	  else
    	    printf ("Done.\n");
    	}
      return 0;
    }

  if ((n = abbrev_spec (argv[1], &info, &sect_first, &sect_last)) != 0)
    {
      if (n < 0)
	{
	  printf ("Bad sector specification\n");
	  return 1;
	}
      printf ("Erase Flash Sectors %d-%d in Bank # %ld:\n",
	      sect_first, sect_last, (info - flash_info) + 1);

      n = flash_erase (info, sect_first, sect_last);
      if (n < 0)
	{
	  flash_perror (n);
	  return 1;
	}
      else
	printf ("Done.\n");

      return 0;
    }


  if (argc != 3)
    {
      printf ("Usage:\n%s\n", cmdtp->usage);
      return 1;
    }

  if (strcmp (argv[1], "bank") == 0)
    {
      bank = simple_strtoul (argv[2], NULL, 16);
      if ((bank < 1) || (bank > CFG_MAX_FLASH_BANKS))
	{
	  printf ("Only FLASH Banks # 1 ... # %d supported\n",
		  CFG_MAX_FLASH_BANKS);
	  return 1;
	}
      printf ("Erase Flash Bank # %ld:\n", bank);
      info = &flash_info[bank - 1];
      n = flash_erase (info, 0, info->sector_count - 1);
      if (n < 0)
	{
	  flash_perror (n);
	  return 1;
	}
      else
	printf ("done.\n");
      return 0;
    }

  addr_first = simple_strtoul (argv[1], NULL, 16);
  addr_last = simple_strtoul (argv[2], NULL, 16);
#if defined(AM29LV320D) || defined(AT49BV321)
  if (addr_first < 0x1000000 || addr_last >= 0x1400000)
#else
  //if (addr_first < 0x1000000 || addr_last >= 0x1200000)
  if (addr_first < 0 || addr_last >= 0x800000)
#endif
    {
      printf ("Address range error !!\n");
      printf ("Usage:\n%s\n", cmdtp->usage);
      return 1;
    }

  if (addr_first >= addr_last)
    {
      printf ("Usage:\n%s\n", cmdtp->usage);
      return 1;
    }

  printf ("Erase Flash from 0x%08lx to 0x%08lx... \n", addr_first, addr_last);
  n = flash_sect_erase (addr_first, addr_last);

  if (n < 0)
    {
      flash_perror (n);
      return 1;
    }
  else
    printf ("done.\nErased %d sectors.\n", n);
  return 0;
}

int
do_protect (cmd_tbl_t * cmdtp, bd_t * bd, int flag, int argc, char *argv[])
{
  flash_info_t *info;
  ulong bank, addr_first, addr_last;
  int i, p, n, sect_first, sect_last;

  if (argc < 3)
    {
      printf ("Usage:\n%s\n", cmdtp->usage);
      return 1;
    }

  if (strcmp (argv[1], "off") == 0)
    p = 0;
  else if (strcmp (argv[1], "on") == 0)
    p = 1;
  else
    {
      printf ("Usage:\n%s\n", cmdtp->usage);
      return 1;
    }

  if (strcmp (argv[2], "all") == 0)
    {
      for (bank = 1; bank <= CFG_MAX_FLASH_BANKS; ++bank)
	{
	  info = &flash_info[bank - 1];
	  if (info->flash_id == FLASH_UNKNOWN)
	    {
	      continue;
	    }
	  printf ("%sProtect Flash Bank # %ld\n", p ? "" : "Un-", bank);

	  for (i = 0; i < info->sector_count; ++i)
	    {
	      info->protect[i] = p;
	    }
	}

      return 0;
    }

  if ((n = abbrev_spec (argv[2], &info, &sect_first, &sect_last)) != 0)
    {
      if (n < 0)
	{
	  printf ("Bad sector specification\n");
	  return 1;
	}
      printf ("%sProtect Flash Sectors %d-%d in Bank # %ld\n",
	      p ? "" : "Un-", sect_first, sect_last, (info - flash_info) + 1);

      for (i = sect_first; i <= sect_last; i++)
	{
	  info->protect[i] = p;
	}
      return 0;
    }

  if (argc != 4)
    {
      printf ("Usage:\n%s\n", cmdtp->usage);
      return 1;
    }

  if (strcmp (argv[2], "bank") == 0)
    {
      bank = simple_strtoul (argv[3], NULL, 16);
      if ((bank < 1) || (bank > CFG_MAX_FLASH_BANKS))
	{
	  printf ("Only FLASH Banks # 1 ... # %d supported\n",
		  CFG_MAX_FLASH_BANKS);
	  return 1;
	}

      printf ("%sProtect Flash Bank # %ld\n", p ? "" : "Un-", bank);

      info = &flash_info[bank - 1];

      if (info->flash_id == FLASH_UNKNOWN)
	{
	  printf ("missing or unknown FLASH type\n");
	  return 1;
	}

      for (i = 0; i < info->sector_count; ++i)
	{
	  info->protect[i] = p;
	}
      return 0;
    }

  addr_first = simple_strtoul (argv[2], NULL, 16);
  addr_last = simple_strtoul (argv[3], NULL, 16);

  if (addr_first >= addr_last)
    {
      printf ("Usage:\n%s\n", cmdtp->usage);
      return 1;
    }

  return flash_sect_protect (p, addr_first, addr_last) ? 1 : 0;
}

int
flash_sect_protect (int p, ulong addr_first, ulong addr_last)
{
  flash_info_t *info;
  ulong bank;
  int s_first, s_last;
  int protection, i;
  int rc = ERR_OK;

  protection = 0;

  for (bank = 0, info = &flash_info[0]; bank < CFG_MAX_FLASH_BANKS;
       ++bank, ++info)
    {
      ulong b_end;
      int sect;
      
      if (info->flash_id == FLASH_UNKNOWN)
	{
	  continue;
	}

      b_end = info->start[0] + info->size - 1;	/* bank end addr */

      s_first = -1;		/* first sector to erase        */
      s_last = -1;		/* last  sector to erase        */

      for (sect = 0; sect < info->sector_count; ++sect)
	{
	  ulong end;		/* last address in current sect */
	  short s_end;

	  s_end = info->sector_count - 1;

	  end = (sect == s_end) ? b_end : info->start[sect + 1] - 1;

			
	  if (addr_first > end)
	    continue;
	  if (addr_last < info->start[sect])
	    continue;
					
	  if (addr_first == info->start[sect])
	    {
	      s_first = sect;
	    }
	  if (addr_last == end)
	    {
	      s_last = sect;
	    }
	}
    
      if (s_first >= 0 && s_first <= s_last)
	{
	  protection += s_last - s_first + 1;
	  for (i = s_first; i <= s_last; ++i)
	    {
	      info->protect[i] = p;
	    }
	}
    }

  if (protection)
    {
      printf ("%sProtected %d sectors\n", p ? "" : "Un-", protection);
    }
  else
    {
      rc = ERR_ALIGN;
      flash_perror (rc);
    }
  return rc;
}

// by laputa start
int
flash_protect_range (unsigned short *flashptr, int start, int end, int flag)
{

  //unsigned long from, to;
  ulong from, to;

  from = PHYS_FLASH_1 + (ulong) start;	// 0x8000 0000 + start
  to = PHYS_FLASH_1 + (ulong) end;	// 0x8000 0000 + end

  printf ("\nFlash  PROTECT ");
  (flag) ? printf ("SET  ") : printf ("CLEAR  ");

  printf (" [%#lx] ~ [%#lx]\n\n ", from, to);


  //flash_protect (flag, from, to, &flash_info[0]);
  if (flash_sect_protect (flag, from, to) == ERR_OK)
    return 1;			//PROTECT SET/CLEAR OK
  else
    return 0;			// SET/CLEAR ERROR
}


int
__do_flwrite (cmd_tbl_t * cmdtp, bd_t * bd, ulong data_addr, int argc, char *argv[])
{
	flash_info_t *info;
	ulong b_end;
	image_header_t *hdr = &header;
	int rc = ERR_OK;
	int n, sect;
	ulong len, len_mod, addr_last, src;
	ulong addr_last_erase;
	ulong data, checksum;
	int IncImgHeader, save_env_flag = 0;

	memcpy (&header, (char *) data_addr, sizeof (image_header_t));

	// Basically not including the image header.
	IncImgHeader = 0;

#if 0
	if (argc > 2)
	{
	    printf ("Usage:\n%s\n", cmdtp->usage);
		return 1;
	}
	else if (argc < 2)  // argc == 1
	{
		giFlashAppBaseAddr = SWAP32(hdr->ih_load);	/* Flash writing address */
	}
	else				// argc == 2
	{
		giFlashAppBaseAddr = simple_strtoul (argv[1], NULL, 16);
	}
#endif

	if (argc == 1)
	{
		giFlashAppBaseAddr = SWAP32(hdr->ih_load);	/* Flash writing address */
	}
	else /* argc > 1 */
	{
		giFlashAppBaseAddr = simple_strtoul (argv[1], NULL, 16);
	}

	// cheeck if DestAddr is valid , added by chencb 060831
#if defined(CONFIG_S3C2500) || defined(CONFIG_S3C2510)
	if ((giFlashAppBaseAddr < 0x80000000) || (giFlashAppBaseAddr > 0x80800000))
	{
		printf("DestAddr invalid!\nDestAddr: 0x80000000~0x80800000\n");
		return 1;
	}
#endif

	// avoid writing armboot region
#if defined(CONFIG_S3C2500) || defined(CONFIG_S3C2510)
	if ((giFlashAppBaseAddr > 0x80000000) && (giFlashAppBaseAddr < 0x80020000))
#elif defined(CONFIG_CK510)
	if ((giFlashAppBaseAddr > (CFG_FLASH_BASE + PHYS_FLASH_SIZE)) && (giFlashAppBaseAddr < CFG_FLASH_BASE))
#else
	if ((giFlashAppBaseAddr > 0x1000000) && (giFlashAppBaseAddr < 0x1020000))
#endif
	{
		rc = ERR_PROTECTED;
		flash_perror (rc);
		return 1;
	}

	printf ("\n## Checking Image at %#lx ...\n", data_addr);

	if (SWAP32 (hdr->ih_magic) != IH_MAGIC)
	{
		printf ("   Bad Magic Number\n");
		return 1;
	}

	/*
	 * if giFlashAppBaseAddr==0x80000000 && SWAP32(hdr->ih_load) == 0x80000000,
	 * it is new armboot, need to save enviroment
	 * if giFlashAppBaseAddr==0x80000000 && SWAP32(hdr->ih_load) == 0x80030000,
	 * it is old armboot, not need to save enviroment
	 */
	if ((giFlashAppBaseAddr == CFG_ENV_ADDR)
		&& (SWAP32(hdr->ih_load) == CFG_ENV_ADDR))
	{
		save_env_flag = 1;  //restore new armboot need to save enviroment
	}

	data = (ulong) & header;
	len = sizeof (image_header_t);
	checksum = SWAP32 (hdr->ih_hcrc);
	hdr->ih_hcrc = 0;
	printf ("   Header CRC Checking ... ");
	if (crc32 (0, (char *) data, len) != checksum)
	{
		printf ("ERROR\n");
		return 1;
	}
	else
	{
		printf("OK\n");
	}

	/* for multi-file images we need the data part, too */
	print_image_hdr ((image_header_t *) data_addr);

	data = data_addr + sizeof (image_header_t);
	len = SWAP32 (hdr->ih_size);

	printf ("   Data CRC Checking ... ");
	if (crc32 (0, (char *) data, len) != SWAP32 (hdr->ih_dcrc))
	{
		printf ("ERROR\n");
		return 1;
	}
	else
	{
		printf ("OK\n");
	}
	printf("Programing start at: 0x%08lx\n", giFlashAppBaseAddr);
	
	//add by zxj 2008-06-02
   	if ( 0 == strcmp(hdr->ih_name,"u-boot Updata Image") )
   	{
   		printf ("It's a image of u-boot updata \n");
   		IncImgHeader = 0;
   	}
   	else if( 0 == strcmp(hdr->ih_name,"fs Update Image") )
   	{
   		printf ("It's a image of fs updata \n");
   		IncImgHeader = 1;
   	}
   	else
	{
		printf ("It's a image of OS or FS\n");
		IncImgHeader = 2;
	}

	if ( IncImgHeader)
	{
		//for Linux imgae or fs image
		if ( 1 == IncImgHeader )
			src = data_addr + sizeof (image_header_t);
		else
		{
			src = data_addr;
			len = len + sizeof (image_header_t);
		}
		
		addr_last = giFlashAppBaseAddr + len;
	}
	else
	{
		//for u-boot updata
		src = data_addr + sizeof (image_header_t) - 4;	/* -4 for save the bin image to buffer header */
		*(u32 *)src = len;		/* load 0x100 code */
		
		len += 4;
		
		addr_last = giFlashAppBaseAddr + len;
	}

#if defined(AM29LV320D) || defined(AT49BV321)
	if ((len > 0x3e0000) || (addr_last > (CFG_FLASH_BASE + 0x3fffff)))
#else
	#if defined(S29GL064M)	/* ??? TimYu we need modify the length*/
		if (addr_last > (CFG_FLASH_BASE + 0x800000))
	#else
		if ((len > 0x1e0000) || (addr_last > (CFG_FLASH_BASE + 0x1fffff)))
	#endif
#endif
	{
		rc = ERR_SIZE;
		flash_perror (rc);
		return 1;
	}

	info = &flash_info[0];
	b_end = info->start[0] + info->size - 1;	/* bank end addr */
	
	//zxj 2008-10-15
	//addr_last_erase = addr_last;
	addr_last_erase = addr_last - 1;

	for (sect = 0; sect < info->sector_count; ++sect)
	{
		ulong end;		/* last address in current sect */
		short s_end;

		s_end = info->sector_count - 1;

		end = (sect == s_end) ? b_end : info->start[sect + 1] - 1;
		
		if( addr_last_erase <= end )
		//if ((signed long) ( - end) <= 0)
		{
			addr_last_erase = end;
			break;
		}
	}
	
	//
	printf ("Flash range : 0x%08lx -> 0x%08lx\n", giFlashAppBaseAddr, addr_last_erase);
	
	//while(1);

	n = flash_sect_erase (giFlashAppBaseAddr, addr_last_erase);
	if (n < 0)
	{
		flash_perror (n);
		return 1;
	}
	else
	{
		printf ("done.\nErased %d sectors.\n", n);
	}

	printf ("Saving Image to Flash ...\n");

	len_mod = len % 2;
	if (len_mod)
	{
		len += 1;
	}

	rc = flash_write ((char *) src, giFlashAppBaseAddr, len);
	if (rc < 0)
	{
		flash_perror (rc);
		return 1;
	}

	else
	{
		printf ("done.\n");
	}

	/* added by chencb 060920 */
	if (save_env_flag)
	{
		return do_save_env(bd);
	}

	return 0;
} /* end __do_flwrite */

/* modify by chencb 060914 */
int
do_flwrite (cmd_tbl_t * cmdtp, bd_t * bd, int flag, int argc, char *argv[])
{
	header_t *phdr = NULL;
	int total_len, head_len, img_len;
	ulong data;
	int ret;

	phdr = (header_t*)load_addr;
	ret = strncmp(phdr->id, PACK_ID, strlen(PACK_ID));

	/*
	 * normal image file
	 */
	if (ret != 0)
	{
		return 	__do_flwrite(cmdtp, bd, load_addr, argc, argv);
	}

	/*
	 * update.bin
	 */
#if 0
	ulong org_crc;

	org_crc = phdr->crc;
	phdr->crc = 0;
	if (crc32(0, phdr, phdr->total_len) != org_crc)
	{
		printf ("CRC ERROR\n");
		return 1;
	}
#endif

	total_len = phdr->total_len;
	head_len = phdr->head_len;
	img_len = 0;

	/*
	 * deal every image in update.bin
	 */
	for (data = load_addr + head_len;
	     data < load_addr + total_len;
	     data += img_len)
	{
		img_len = SWAP32(((image_header_t *)data)->ih_size) + sizeof(image_header_t);

		if (__do_flwrite(cmdtp, bd, data, argc, argv))
		{
			printf("flwrite error!\n");
			return 1;
		}
	}

	return 0;
} /* end do_flwrite */

//add by zxj 2008-07-28
int do_flrom(cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
	unsigned int size;
	flash_info_t *info;
	int rc = ERR_OK;
	unsigned char *p;
	
	int n, sect;
	ulong b_end, addr_last, addr_last_erase,len_mod;
	
	if (argc > 1)
	{
	    printf ("Usage:\n%s\n", cmdtp->usage);
		return 1;
	}
	
	giFlashAppBaseAddr = CFG_FLASH_BASE;
		
	size = _armboot_end - _armboot_start;
		
	// avoid writing armboot region
	if (( (giFlashAppBaseAddr + size) > (CFG_FLASH_BASE + PHYS_FLASH_SIZE)) && (giFlashAppBaseAddr < CFG_FLASH_BASE))
	{
		rc = ERR_SIZE;
		flash_perror (rc);
		return 1;
	}
	
	if (giFlashAppBaseAddr == CFG_ENV_ADDR)
	{
		rc = ERR_PROTECTED;
		flash_perror (rc);
		return 1;
	}
		
	//information
	printf("\nu-boot RAM to ROM now...\n");
	printf("source address = 0x%08lx size = 0x%08lx; to = 0x%08lx\n",0xc0000100,size,giFlashAppBaseAddr);
	
	//run
	//p = (unsigned char *)_armboot_relocation - 4;
	p = 0xc0000100 - 4;
	
	*(unsigned int *)p = size;
	
	size += 4;
	addr_last = giFlashAppBaseAddr + size;
	
	
	info = &flash_info[0];
	b_end = info->start[0] + info->size - 1;	/* bank end addr */

	addr_last_erase = addr_last;

	for (sect = 0; sect < info->sector_count; ++sect)
	{
		ulong end;		/* last address in current sect */
		short s_end;

		s_end = info->sector_count - 1;

		end = (sect == s_end) ? b_end : info->start[sect + 1] - 1;

		if ((signed long) (addr_last_erase - end) < 0)
		{
			addr_last_erase = end;
			break;
		}
	}
	
	//
	printf ("Flash range : 0x%08lx -> 0x%08lx\n", giFlashAppBaseAddr, addr_last_erase);

	n = flash_sect_erase (giFlashAppBaseAddr, addr_last_erase);
	if (n < 0)
	{
		flash_perror (n);
		return 1;
	}
	else
	{
		printf ("\nErased %d sectors done.\n", n);
	}

	printf ("Saving RAM to Flash ...\n");

	len_mod = size % 2;
	if ( len_mod )
		size += 1;

	rc = flash_write (p, giFlashAppBaseAddr, size);
	if (rc < 0)
	{
		flash_perror (rc);
		return 1;
	}

	printf("flrom done.\n\n");
	return 0;
} /* end do_flrom */

#endif
int do_read_flash (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
    flash_info_t *info;
    ulong  addr;
    u32 cont,i;
    if (argc < 2)
    {
        printf ("Usage:\n%s\n", cmdtp->usage);
        return 1;
    }

    if (argc != 3)
    {
        printf ("Usage:\n%s\n", cmdtp->usage);
        return 1;
    }

    addr = simple_strtoul (argv[1], NULL, 16);
    cont = simple_strtoul (argv[2], NULL, 16);
    
    for(i = 0; i < cont; i++)
    {
        u8 temp = 0;
        read_flash_byte(&temp,addr+i,1);
        printf("addr 0x%08lx  data 0x%02lx\n",addr+i,temp);
    }
    return 0;
}
int do_write_flash (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
    flash_info_t *info;
    ulong  addr;
    u8 data;
    if (argc < 2)
    {
        printf ("Usage:\n%s\n", cmdtp->usage);
        return 1;
    }
    if (argc != 3)
    {
        printf ("Usage:\n%s\n", cmdtp->usage);
        return 1;
    }
    addr = simple_strtoul (argv[1], NULL, 16);
    data = simple_strtoul (argv[2], NULL, 16);
    write_flash_byte(&data,addr,1);
    return 0;
}

int do_erase_flash (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
    flash_info_t *info;
    ulong bank, addr_first, addr_last;
    int n,mod, sect_first, sect_last;
    if (argc < 2)
    {
        printf ("Usage:\n%s\n", cmdtp->usage);
        return 1;
    }

    if (strcmp (argv[1], "all") == 0)
    {
        erase_flash_all();
        printf ("Done.\n");
        return 0;
    }

    if (argc > 3)
    {
        printf ("Usage:\n%s\n", cmdtp->usage);
        return 1;
    }

    addr_first = simple_strtoul (argv[1], NULL, 16);
    if(argc == 2)
    {
        if (addr_first < 0 || addr_first >= 0x800000)
        {
            printf ("Address range error !!\n");
            printf ("Usage:\n%s\n", cmdtp->usage);
            return 1;
        }
        sect_first = addr_first/4096;
        printf ("Erase Flash from 0x%08lx to 0x%08lx  ... \n", addr_first, 0x7FFFFF);
        erase_flash_sector ( sect_first, CFG_MAX_FLASH_SECT - 1);
        printf ("done.\nErased %d sectors.\n",CFG_MAX_FLASH_SECT - sect_first);
        return 0;
    }
    
    addr_last = simple_strtoul (argv[2], NULL, 16);
    if (addr_first < 0 || addr_last >= 0x800000)
    {
        printf ("Address range error !!\n");
        printf ("Usage:\n%s\n", cmdtp->usage);
        return 1;
    }

    if (addr_first >= addr_last)
    {
        printf ("Usage:\n%s\n", cmdtp->usage);
        return 1;
    }
    printf ("Erase Flash from 0x%08lx to 0x%08lx... \n", addr_first, addr_last);
    sect_first = addr_first/4096;
    sect_last  = (addr_last-1)/4096;
    erase_flash_sector ( sect_first, sect_last);
    printf ("done.\nErased %d sectors.\n",sect_last - sect_first + 1);
  return 0;
}

int do_write_uboot_to_flash (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
    flash_info_t *info;
    u32  file_addr,flash_addr,length;
    u8 data,*p;
    if (argc != 3)
    {
        printf ("Usage:\n%s\n", cmdtp->usage);
        return 1;
    }
    file_addr = simple_strtoul (argv[1], NULL, 16);
    length = simple_strtoul (argv[2], NULL, 16);
    flash_addr = CFG_UBOOT_ADDR_BEG;
    p = (u8 *)file_addr;
    printf("write uboot from memery adddr 0x%08lx write to flash addr 0x%08lx length 0x%08lx \n",file_addr,flash_addr,length);
    write_flash_byte(p,flash_addr,length);
    return 0;
}

int do_write_jffs2_to_flash (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
    return 0;
}


int do_write_kernel_to_flash (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
    flash_info_t *info;
    u32  file_addr,flash_addr,length;
    u8 data,*p;
    if (argc != 3)
    {
        printf ("Usage:\n%s\n", cmdtp->usage);
        return 1;
    }
    file_addr = simple_strtoul (argv[1], NULL, 16);
    length = simple_strtoul (argv[2], NULL, 16);
    flash_addr = CFG_KERNEL_ADDR_BEG;
    if(flash_addr + length > CFG_KERNEL_ADDR_END)
    {
        printf("kernel file too large!\n");
        return -1;
    }
    p = (u8 *)file_addr;
    printf("write kernel from memery adddr 0x%08lx write to flash addr 0x%08lx length 0x%08lx \n",file_addr,flash_addr,length);
    write_jffs2_byte(p,flash_addr,length);
}


#define DH5K_KNL_FS_LENGHT_OFF		2
#define DH5K_KNL_ADDR_OFF				3
#define DH5K_KNL_OFF					4
#define DH5K_KNL_LENGTH_OFF			5

#define DH5k_ROOTFS_ADDRESS			( 0xC0020000 )
#define DH5k_KNL_START_ADDRESS		( 0xC0100000 )

int do_load_kernel_from_flash (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
	ulong load_addr = CFG_LOAD_ADDR;
	ulong kernel_addr = CFG_KERNEL_ADDR_BEG;
	u32 size,*p;
	u32 kernel_header[ 8 ];
	u32 mem_addr, knl_length, i;
	
	if (argc != 1)
	{
		printf ("Usage:\n%s\n", cmdtp->usage);
		return -1;
	}

	/* read the header */
	read_flash_byte( ( u8 *)kernel_header, kernel_addr, 32 );

	/* get the kernel length */
	size  = kernel_header[ DH5K_KNL_FS_LENGHT_OFF ];
	mem_addr = kernel_header[ DH5K_KNL_ADDR_OFF ];
	knl_length =  kernel_header[ DH5K_KNL_LENGTH_OFF ];
	knl_length = knl_length - DH5k_KNL_START_ADDRESS;

	/* calc the rootfs size */
	size = size  - knl_length;
	p = ( u32 *)( mem_addr );
	
	/* copy the 16 bytes to memory */
	for ( i = 0; i < 4; i ++ )
	{
		*p ++ = kernel_header[ DH5K_KNL_OFF + i ];
	}
	
	kernel_addr += 32;
	knl_length -= 16;

	//printf ("load kernel...\n");

	
	read_flash_byte( (u8 *)p, kernel_addr, knl_length );

	
	//printf ("load rootfs...\n");
	p = ( u32 *)( DH5k_ROOTFS_ADDRESS );
	kernel_addr += knl_length;

	read_flash_byte( (u8 *)p, kernel_addr, size );

	
	printf("Done.\n");
}


static void do_erase_spi_sector( u32 u32StartSec,  u32 u32EndSec )
{
	u32 saddr, eaddr;

	/* align the sector address by sector_size */
	saddr = u32StartSec / SECTOR_SIZE;

	eaddr = u32EndSec / SECTOR_SIZE;
	
	printf ("Erase from 0x%08lx to 0x%08lx... \n", u32StartSec, u32EndSec);
	
	erase_flash_sector ( saddr, eaddr );

	printf ("done.\nErased %d sectors.\n", eaddr - saddr + 1);
	
}

int do_erase_uboot (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
	do_erase_spi_sector( CFG_UBOOT_ADDR_BEG, CFG_UBOOT_ADDR_END );
	return 0;
}


int do_erase_kernel (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
	do_erase_spi_sector( CFG_KERNEL_ADDR_BEG, CFG_KERNEL_ADDR_END );
	return 0;
}


int do_erase_user (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{

	do_erase_spi_sector( CFG_USER_ADDR_BEG, CFG_USER_ADDR_END );

	return 0;
}


#define DHC_DEBUG_CMD	20
#define DHC_DL_CMD		24

inline char serial_read_byte(void)
{
	char c = 0;

	c = serial_getc();
	
 	return (c);
}



static void do_load_from_uart(  u32 *pu32Addr, u32 *pu32Count )
{
	u32 i, length, addr;
	u8 tmp;
	u8 *pu8;

	printf ("## Ready for binary download ...\n");

	/* send start flag */
	serial_putc('@');

	/* read the ack */
	while( 1 )
	{
		tmp = serial_read_byte();

		if(  DHC_DL_CMD == tmp )
		{
			break;
		}
	}

	/* get the address */
	pu8 = ( u8 *)( &addr );
	pu8 += 3;
	for ( i = 0; i < 4; i ++ )
	{
		*pu8-- = serial_read_byte();
	}

	/* get the length */
	pu8 = ( u8 *)( &length );
	pu8 += 3;
    	for(i = 0;i < 4;i++)
    	{
        	*pu8-- =serial_read_byte();
    	}	

	/* now recv the data */
	pu8 = ( u8 * )( addr );
	for ( i = 0; i < length; i ++ )
	{
		*pu8++ = serial_read_byte();
	}

	if ( length & 0x03 )
	{
		/* for dw align */
		length &= 0xFFFFFFFC;
		length += 4;
	}

	/* return */
	*pu32Addr = addr;
	*pu32Count = length;

	printf("\nload Done!\n");

}


static void do_write_to_flash( u32 u32StartSec, u32 u32EndSec, u32 u32MemAddr, u32 u32FlashAddr, u32 u32Length, u8 u8endian )
{
	/* erase */
	do_erase_spi_sector( u32StartSec, u32EndSec );

	/* show current infomation */
	printf("Wr [%#x] -> [%#x], l = %#x\n", u32MemAddr, u32FlashAddr, u32Length );

	/* write to flash */
	//write_flash_byte( ( u8 *)( u32MemAddr ), u32FlashAddr,  u32Length );
	w25q64cv_write( ( u8 *)( u32MemAddr ), u32FlashAddr,  u32Length, u8endian );

	printf("write Done!\n");
}

int do_load_boot_from_uart (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
	u32 mem_addr, length, flash_addr;

	if (argc != 1)
	{
		printf("load boot cmd parameter err!\n");
		return 1;
	}
	
 
	do_load_from_uart(  &mem_addr , &length );

	if ( 0 == length )
	{
		return 0;
	}

	do_write_to_flash( CFG_UBOOT_ADDR_BEG, CFG_UBOOT_ADDR_END,  mem_addr, CFG_UBOOT_ADDR_BEG,  length,  little_endian  );

	return 0;

}


int do_load_kernel_from_uart (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
	u32 mem_addr, length, flash_addr;
	
	if (argc != 1)
	{
	    printf("load kernel cmd parameter err!\n");
	    return 1;
	}
	
	do_load_from_uart(  &mem_addr,  &length );

	if ( 0 == length )
	{
		return 0;
	}

	do_write_to_flash( CFG_KERNEL_ADDR_BEG, CFG_KERNEL_ADDR_END,  mem_addr, CFG_KERNEL_ADDR_BEG,  length, little_endian  );
	
	return 0;
    
}


int do_load_user_from_uart (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
    u32 mem_addr,length,flash_addr,i,count;
    u8 flag,dbg,*pLoadDDRaddr;
    u8 *paddr,*plength;
    u32 endFlashAddr = 0;
	
    if (argc != 2)
    {
        printf("load user  cmd parameter err!\n");
        printf("exaple:dl.user dstAddr(flashAddr)!\n");
        return 1;
    }

	do_load_from_uart(  &mem_addr,  &length );
	if ( 0 == length )
	{
		return 0;
	}
    
	flash_addr = simple_strtoul (argv[1], NULL, 16);

	if (flash_addr == 0x200000)
	{
		endFlashAddr = 0x2FFFFF;/*1M*/
	}
	else if(flash_addr == 0x300000)
	{
		endFlashAddr = 0x37FFFF;/*512k*/
	}
	else if(flash_addr == 0x380000)
	{
		endFlashAddr = 0x3FFFFF;/*512k*/
	}
	else
	{
		printf("flash addr should be 0x200000 or 0x300000 or 0x380000\n");
		return 1;
	}

	do_write_to_flash( flash_addr, endFlashAddr,  mem_addr, flash_addr,  length, big_endian  );
	
}


#define CONFIG_SYS_FLASH_BASE  0x0
#define CONFIG_SPI_FLASH_SIZE  0x400000
#define DATA_ALIGN_SZ 	0x10000


static int __do_check_addr(unsigned long addr)
{
	int rc = ERR_OK;

	if ((addr < CONFIG_SYS_FLASH_BASE) 
		|| (addr >= (CONFIG_SYS_FLASH_BASE+CONFIG_SPI_FLASH_SIZE)))
	{
		printf("Dest address:0x%lx invalid!\n", addr);
		rc = ERR_INVAL;
		goto err_out;
	}

	/* "run da"=CONFIG_SYS_FLASH_BASE, so here is right */
	if ((addr > CONFIG_SYS_FLASH_BASE) &&
		(addr < (CONFIG_SYS_FLASH_BASE + 128*1024)))
	{
		rc = ERR_PROTECTED;
		goto err_out;
	}
    printf("no check align\n");
#if 0
	/* align 128KB, for the old N5 */
	if (addr % DATA_ALIGN_SZ)
	{
		rc = ERR_ALIGN;
		goto err_out;
	}
#endif
	return rc;

err_out:
	flash_perror (rc);
	return rc;
}

static int __do_check_image(ulong addr)
{
	void *hdr = (void *)addr;

	printf ("\n## Checking Image at %08lx ...\n", addr);

	switch (genimg_get_format (hdr)) {
	case IMAGE_FORMAT_LEGACY:
		puts ("   Legacy image found\n");
		if (!image_check_magic (hdr)) {
			puts ("   Bad Magic Number\n");
			return 1;
		}

		if (!image_check_hcrc (hdr)) {
			puts ("   Bad Header Checksum\n");
			return 1;
		}

		image_print_contents (hdr);

		puts ("   Verifying Checksum ... ");
		if (!image_check_dcrc (hdr)) {
			puts ("   Bad Data CRC\n");
			return 1;
		}
		puts ("OK\n");
		return 0;
#if defined(CONFIG_FIT)
	case IMAGE_FORMAT_FIT:
		puts ("   FIT image found\n");

		if (!fit_check_format (hdr)) {
			puts ("Bad FIT image format!\n");
			return 1;
		}

		fit_print_contents (hdr);

		if (!fit_all_image_check_hashes (hdr)) {
			puts ("Bad hash in FIT image!\n");
			return 1;
		}

		return 0;
#endif
	default:
		puts ("Unknown image format!\n");
		break;
	}

	return 0;

}

int __do_flwrite (cmd_tbl_t * cmdtp, ulong img_addr, int argc, char *argv[])
{
    void *data_addr = NULL;
    int data_len = 0;
    uint data_end;
    image_header_t hdr; //img文件的header

	ulong flash_baseaddr; //要写入的位置
	printf("argc:%d  img_addr:0x%x\n",argc,img_addr);
	/* Flash writing address */
	if (argc == 1) {
		/* copy image hader */
		memcpy (&hdr, (char *)img_addr, sizeof (image_header_t));
		flash_baseaddr = SWAP32((&hdr)->ih_load);
	} else {/* argc > 1 */
	    memcpy (&hdr, (char *)img_addr, sizeof (image_header_t));
		flash_baseaddr = simple_strtoul (argv[1], NULL, 16);
	}

	data_addr = (void *)img_addr + sizeof(image_header_t);
	data_len = SWAP32((&hdr)->ih_size);
	data_end = SWAP32((&hdr)->ih_ep);


	/* check flash address */
	if (__do_check_addr(flash_baseaddr)) {
		return 1;
	}

	/* check image data */
	if (__do_check_image(img_addr)) 
    {
		printf ("   Bad Image Info\n");
		return 1;
	}


	/* write data to flash */
	printf("Programing start at: 0x%08lx,datalen=%d\n", flash_baseaddr,data_len);
    w25q64cv_erase(flash_baseaddr,data_len/SECTOR_SIZE+1,flash_erase_4k_mode);
   
    w25q64cv_write(data_addr,flash_baseaddr,data_len,little_endian);/* С˸ʽת */
    

	return 0;
} /* end __do_flwrite */


#define SCRIPT_IMG_NAME     "CmdScript"
#define SCRIPT_FILE_END     "SCRIPT_FILE_END"
/* 单条配置信息字符串最小长?*/
#define MIN_SCRIPT_LEN    (1)
/* 单条配置信息字符串最大长?*/
#define MAX_SCRIPT_LEN    (256)



static inline uint32_t image_get_size(const image_header_t *hdr)
{ 
    return SWAP32(hdr->ih_size);
}

static inline uint32_t image_get_header_size (void)
{
	return (sizeof (image_header_t));
}

static inline char *image_get_name (const image_header_t *hdr)
{
	return (char *)hdr->ih_name;
}

static inline ulong image_get_data (const image_header_t *hdr)
{
	return ((ulong)hdr + image_get_header_size ());
}

static inline uint32_t image_get_data_size (const image_header_t *hdr)
{
	return image_get_size (hdr);
}


static inline uint32_t image_get_image_size (const image_header_t *hdr)
{
	return (image_get_size (hdr) + image_get_header_size ());
}

/* 从文件中读取一?*/
static char *get_line(char **str)
{
    char *p, *end, *tmp;
    char *pStr;

    /* 非法指针 */
    if ((str == NULL) || (*str == NULL))
    {
        return NULL;
    }

    /* 空字符串 */
    if (**str == 0)
    {
        return NULL;
    }
    
    p = pStr = *str;
    do
    {
        if ((*pStr == '\n') 
		     || ((*pStr == '\r') && (*(pStr + 1) == '\n')))
        {
            *pStr = 0;
            pStr++;
            break;
        }

        pStr++;
    } while (*pStr);
    *str = pStr;

    return p;
}

/* 解析并执行配置脚?*/
static int __do_script(ulong img_addr,bd_t *bd)
{
    char *img_data, *new_line, *buf;
    ulong len, img_size;
    printf("__do_script\n");
    /* 检查镜像文?*/
    if (__do_check_image(img_addr)) 
    {
        printf ("   Bad Image Info\n");
        return 1;
    }
#if 1
    /* 获取配置信息 */
    img_data = image_get_data ((image_header_t *)img_addr);   
    img_size = image_get_data_size ((image_header_t *)img_addr);
    printf("img_data0x%x img_size%d\n",img_data,img_size);
    buf  = malloc(img_size + 4);
    if (!buf)
    {
        printf ("Fail to malloc buffer for CmdScript!\n");
        return 1;
    }
    memcpy(buf, img_data, img_size);
    buf[img_size] = 0;
    img_data = buf;

    printf ("exce update config script start!\n");
    
    /* 逐行执行配置脚本 */
    new_line = get_line(&img_data);
    while (new_line)
    {
        if (new_line[0] == '#')
        {
            new_line = get_line(&img_data);
            continue;
        }
        
        if (strstr(new_line, SCRIPT_FILE_END))
        {
            break;
        }

        len = strlen(new_line);
        if ((len < MIN_SCRIPT_LEN) || (len >= MAX_SCRIPT_LEN))
        {
            new_line = get_line(&img_data);
            continue;
        }


        run_command (new_line,bd, 0);


        new_line = get_line(&img_data);
    }

    free(buf);
    
    printf ("exce update config script complete!\n");
#endif
    return 0;

}
int do_flread (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
    unsigned long srcAddr,disAddr;
    unsigned long len;
    if(argc != 4)
    {
        printf("param is error %d\n",argc);
        printf("flread srcAddr disAddr len\n");
        return -1;
    }
    srcAddr = simple_strtoul (argv[1], NULL, 16);
    disAddr = simple_strtoul (argv[2], NULL, 16);
    len = simple_strtoul (argv[3], NULL, 16);
    printf("read 0x%x bytes from flash addr 0x%x to ram addr 0x%x\n",len,srcAddr,disAddr);
    read_flash_byte((u8 *)disAddr,srcAddr,len);
    printf("Done.\n");

    return 0;
    

}

/* __do_flwrite ķģʽ汾 */
static int flwrite(ulong img_addr)
{
    void            *data_addr = NULL;
    int             data_len = 0;
    uint            data_end;
    image_header_t  hdr;
    ulong           flash_baseaddr; 

    printf("img_addr: 0x%x\n", img_addr);

    /* Get Flash writing address */
    /* Copy image hader To */
    memcpy (&hdr, (char *)img_addr, sizeof (image_header_t));
    flash_baseaddr = SWAP32((&hdr)->ih_load); 

    data_addr = (void *)img_addr + sizeof(image_header_t);
    data_len = SWAP32((&hdr)->ih_size);
    data_end = SWAP32((&hdr)->ih_ep);


    /* check flash address */
    if (__do_check_addr(flash_baseaddr))
    {
        return 1;
    }

    /* check image data */
    if (__do_check_image(img_addr)) 
    {
        printf (" Bad Image Info\n");
        return 1;
    }


    /* write data to flash */
    printf("Programing start at: 0x%08lx, datalen=%d\n", 
    flash_baseaddr, data_len);

    w25q64cv_erase(flash_baseaddr,
                   data_len/SECTOR_SIZE + 1,
                   flash_erase_4k_mode);

    /* С˸ʽת */
    w25q64cv_hdcvi_up_write(data_addr, flash_baseaddr, data_len, little_endian);

    return 0;
} /* end __do_flwrite */




/*******************************************************************************
*   : pack_write
*     : дFLASHģʽð汾 
*     : 
*     : 
* ֵ  : < 0:ʧܡ
*           ==0:ɹ 
*******************************************************************************/
int pack_write(ulong mem_addr, bd_t *bd)
{
    int             ret = 0;
    u32             i;
    header_t        *phdr = NULL;
    int             total_len, head_len, img_len;
    ulong           data;
    image_header_t  tmp_header;

    /* check: is update img ? or special update file?
    *  Ext:romfs-x.cramfs.img 
    */
    phdr = (header_t *)(mem_addr);
    ret  = strncmp((char *)phdr->id, PACK_ID, strlen(PACK_ID));
    if (ret != 0)
    {  
        /* ļ */  
        ret = flwrite(mem_addr);
        if (ret) 
        {
            printf("flwrite error 1!\n");
            return (-1);
        }

        return (0);
    }

    phdr      = (header_t*)(mem_addr);
    total_len = phdr->total_len;
    head_len  = phdr->head_len;
    img_len   = 0;

    printf("head_len:%d\n", head_len);
    printf("mem_addr = %#x,total_len:%d\n", mem_addr, total_len);

    /* deal every image in update.img */
    for (data = mem_addr + head_len;
         data < mem_addr + total_len;
         data += img_len)
    {
        /* עdataַһ64ֽڶ룬data=0xc0114527,ֱ
        * ((image_header_t*)data)->ih_size
        * Ҫtmp_headerʱ,image_header_tСֽڿ
        * ʱٶȡļȣԺþðӵ!!!!!
        */

        memcpy(&tmp_header, data, sizeof(image_header_t)); 
        img_len = SWAP32(tmp_header.ih_size) + sizeof(image_header_t);

        if (strncmp(SCRIPT_IMG_NAME, image_get_name((image_header_t *)data),\
            sizeof(SCRIPT_IMG_NAME)) == 0)
        {
            if (NULL == bd)
            {
                printf("flwrite fail to run config script\n");
                return  (-1);
            }

            if (__do_script(data, bd))
            {
                printf("flwrite fail to run config script\n");
                return  (-1);
            }
        } 
        else
        { 
            if (flwrite(data))
            {
                printf("flwrite error 2!\n");
                return  (-1);
            }
        } 
    } 

    return (0);
}


int do_flwrite (cmd_tbl_t *cmdtp, bd_t *bd, int flag, int argc, char *argv[])
{
	header_t *phdr = NULL;
	int total_len, head_len, img_len;
	u32 mem_addr, length, flash_addr;
	ulong data;
	int ret;
    int i;
    image_header_t tmp_header;
    
 
	do_load_from_uart(  &mem_addr , &length );

	if ( 0 == length )
	{
		return 0;
	}
	
	/* special update file, ex: romfs-x.cramfs.img */
	phdr = (header_t*)(mem_addr);
	ret = strncmp((char *)phdr->id, PACK_ID, strlen(PACK_ID));
    if (ret != 0)
	{   /* ļ */
		ret = __do_flwrite(cmdtp, mem_addr, argc, argv);
		if (ret) 
            printf("flwrite error 1!\n");
        return 0;
	}

    phdr = (header_t*)(mem_addr);
    total_len = phdr->total_len;
    head_len = phdr->head_len;
    img_len = 0;
    printf("head_len:%d\n",head_len);//64
    printf("mem_addr=%#x,total_len:%d\n",mem_addr,total_len);//64
    /* deal every image in update.img */
    for (data = mem_addr + head_len;
         data < mem_addr + total_len;
         data += img_len)
    {
        /* עdataַһ64ֽڶ룬data=0xc0114527,ֱ((image_header_t*)data)->ih_size
           * Ҫtmp_headerʱ,image_header_tСֽڿʱٶȡ
           * ļȣԺþðӵ!!!!!
           */
        memcpy(&tmp_header,data,sizeof(image_header_t)); 
        img_len = SWAP32(tmp_header.ih_size) + sizeof(image_header_t);

        /* check Ctrl-C */
        ctrlc();
        if ((had_ctrlc())) {
            
            goto out;
        }
        if (strncmp(SCRIPT_IMG_NAME, image_get_name((image_header_t *)data), 
            sizeof(SCRIPT_IMG_NAME)) == 0)
        {
            if (__do_script(data,bd))
            {
                printf("flwrite fail to run config script\n");
                ret = 1;
                goto out;
            }
        } 
        else
        {
            if (__do_flwrite(cmdtp, data, argc, argv))
            {
                printf("flwrite error 2!\n");
                ret = 1;
            }
        }
        
    }
out:         
    return 0;
}
#endif /* CFG_CMD_FLASH */
