/* s3c2510.h - SAMSUNG DSL MCU S3C2510 header file */

/*
This file contains I/O address and related constants for the ARM PID board.
*/
#ifndef INCs3c2510h
#define	 INCs3c2510h
#include "config.h"
#ifdef INCLUDE_FLASH
#define FLASH_ADRS		0x80000000	/* Base address of Flash part */
#define FLASH_WIDTH		2		/* 1 byte-wide part */

/*
 * If the assummptions below are not met, the definitions will
 * probably need to be changed appropriately (see FlashMem.c) and then
 * calibrated using timexN.
 */
#define SYS_FLASH_BOARD_DELAY
#define SYS_FLASH_DELAY_SHIFT		0
#define SYS_FLASH_DELAY_ADJ		0
#define SYS_FLASH_DELAY_INCR		7

#endif /* INCLUDE_FLASH */

/*************************************************************************
* SYSTEM CLOCK 
*/

#define CPU_SPEED	166000000	/* CPU clocked at 72 MHz. The timer */

#define MHz 		1000000

#define fMCLK_MHz 	133000000	/* 133MHz Bus clock, S3C2510 */
#define fMCLK_MHz2	(fMCLK_MHz / 2)
#define fMCLK 		(fMCLK_MHz / MHz)

/*************************************************************************
* changes made from here
*/

/*************************************************************************
* S3C2510 SPECIAL REGISTERS 
*
*/
#define ASIC_BASE		0xf0000000

/* ----------------------------------------------------------------------------- */
/* System PLL Control Registers (SYSCFG)								        */
/* ----------------------------------------------------------------------------- */

#define CORE			1
#define BUS			2
#define USB			3
#define PHY			4

#define CPLLREN			0x80000000	/* CPLLCON register enable */
#define SPLLREN     		0x40000000	/* SPLLCON register enable */
#define UPLLREN			0x20000000	/* UPLLCON register enable */
#define PPLLREN    		0x10000000	/* PPLLCON register enable */

#define CPLLRCE			0x08000000	/* CPLL clock enable */
#define SPLLRCE   		0x04000000	/* SPLL clock enable */
#define UPLLRCE			0x02000000	/* UPLL clock enable */
#define PPLLRCE    		0x01000000	/* PPLL clock enable */

#define BIG		0x10000	/* Read only, 0: Little endian, 1: Big */
#define REMAP		0x100	/* 0: Remap disable, 1: Enable */
#define HCLKO_DIS	0x10	/* HCLKO output disable */
										/* 0: Enable always, 1: Enable during SDRAM access */
#define ARB	0x1	/* 0: round robin bus arbitration, 1: Fixed priority */

#define FastBusMode	0x00
#define Sync0BusMode	0x01
#define Sync1BusMode	0x02
#define AsyncBusMode	0x03

/* ----------------------------------------------------------------------------- */
/* Peripheral Clock Disable Registers (PCLKDIS)									*/
/* ----------------------------------------------------------------------------- */

#define GDMA0CLKDIS		(0x1<<0)	/* GDMA ch 0 clock disable */
#define GDMA1CLKDIS		(0x1<<1)	/* GDMA ch 1 clock disable */
#define GDMA2CLKDIS		(0x1<<2)	/* GDMA ch 2 clock disable */
#define GDMA3CLKDIS		(0x1<<3)	/* GDMA ch 3 clock disable */
#define GDMA4CLKDIS		(0x1<<4)	/* GDMA ch 4 clock disable */
#define GDMA5CLKDIS		(0x1<<5)	/* GDMA ch 5 clock disable */

#define MAC0CLKDIS		(0x1<<6)	/* ETHERC0 clock disable */
#define MAC1CLKDIS		(0x1<<7)	/* ETHERC1 clock disable */
#define USBCLKDIS		(0x1<<8)	/* USB       clock disable */
#define CUARTCLKDIS		(0x1<<9)	/* CUART clock disable */
#define HUART0CLKDIS		(0x1<<10)	/* HUART clock disable */
#define HUART1CLKDIS		(0x1<<11)	/* HUART clock disable */

#define TIMER0CLKDIS		(0x1<<12)	/* TIMER0 clock disable */
#define TIMER1CLKDIS		(0x1<<13)	/* TIMER1 clock disable */
#define TIMER2CLKDIS		(0x1<<14)	/* TIMER2 clock disable */
#define TIMER3CLKDIS		(0x1<<15)	/* TIMER3 clock disable */
#define TIMER4CLKDIS		(0x1<<16)	/* TIMER4 clock disable */
#define TIMER5CLKDIS		(0x1<<17)	/* TIMER5 clock disable */

#define WDTCLKDIS	(0x1<<18)	/* Watch dog timer clock disable */
#define IOPCCLKDIS		(0x1<<19)	/* IOPC clock disable */
#define IICCCLKDIS		(0x1<<20)	/* IICC clock disable */
#define DESCLKDIS		(0x1<<21)	/* DES clock disable */
#define MEMCONCLKDIS		(0x1<<22)	/* MEMCON clock disable */
#define SDRAMCCLKDIS		(0x1<<23)	/* SDRAMC clock disable */

#define HDLC0CLKDIS		(0x1<<24)	/* HDLC0 clock disable */
#define HDLC1CLKDIS		(0x1<<25)	/* HDLC1 clock disable */
#define HDLC2CLKDIS		(0x1<<26)	/* HDLC2 clock disable */
#define IOM2CLKDIS		(0x1<<27)	/* IOM2 clock disable */

#define SRack   (0x1<<30)	/* Self-refresh acknowledge ( Read only) */
#define SRreq   (0x1<<31)	/* Self-refresh request, 1 : Disable, 0: Enable */

#define ALLCLKENABLE	0x0
#define ALLCLKDISABLE	0xCFFFFFFF

/* ----------------------------------------------------------------------------- */
/* SYATEM PROTECTION AREA SETTING (CP15, C6)									*/
/* ----------------------------------------------------------------------------- */
#define	PROTECTBASE_SHIFT			(12)
#define	PROTECTAREA_EN				(0x1)
#define	PROTECTAREA_4KB				(0x0b<<1)
#define	PROTECTAREA_8KB				(0x0c<<1)
#define	PROTECTAREA_16KB			(0x0d<<1)
#define	PROTECTAREA_32KB			(0x0e<<1)
#define	PROTECTAREA_64KB			(0x0f<<1)
#define	PROTECTAREA_128KB			(0x10<<1)
#define	PROTECTAREA_256KB			(0x11<<1)
#define	PROTECTAREA_512KB			(0x12<<1)
#define	PROTECTAREA_1MB				(0x13<<1)
#define	PROTECTAREA_2MB				(0x14<<1)
#define	PROTECTAREA_4MB				(0x15<<1)
#define	PROTECTAREA_8MB				(0x16<<1)
#define	PROTECTAREA_16MB			(0x17<<1)
#define	PROTECTAREA_32MB			(0x18<<1)
#define	PROTECTAREA_64MB			(0x19<<1)
#define	PROTECTAREA_128MB			(0x1a<<1)
#define	PROTECTAREA_256MB			(0x1b<<1)
#define	PROTECTAREA_512MB			(0x1c<<1)
#define	PROTECTAREA_1GB				(0x1d<<1)
#define	PROTECTAREA_2GB				(0x1e<<1)
#define	PROTECTAREA_4GB				(0x1f<<1)

#define PROTECT_AREA0	0x1
#define PROTECT_AREA1	0x2
#define PROTECT_AREA2	0x4
#define PROTECT_AREA3	0x8
#define PROTECT_AREA4	0x10
#define PROTECT_AREA5	0x20
#define PROTECT_AREA6	0x40
#define PROTECT_AREA7	0x80

/* Permission Encoding */
#define NO_ACCESS	0
#define PRIV_ONLY	1
#define USER_READ	2
#define FULL_ACCESS	3

/* ----------------------------------------------------------------------------- */
/* SYATEM Configuration Registers (0xF0000000 ~ 0xF000FFFF)						*/
/* ----------------------------------------------------------------------------- */

#define SYSCFG			(ASIC_BASE+0x0000)	/* System configuration */
#define PDCODE			(ASIC_BASE+0x0004)	/* Product code and revision number */
#define CLKCON			(ASIC_BASE+0x0008)	/* System clock control */
#define PCLKDIS			(ASIC_BASE+0x000c)	/* Peripheral clock disable */
#define CLKST			(ASIC_BASE+0x0010)	/* Clock Status */
#define HPRIF			(ASIC_BASE+0x0014)	/* AHB bus master fixed priority */
#define HPRIR			(ASIC_BASE+0x0018)	/* AHB bus master round-robin priority */
#define CPLL			(ASIC_BASE+0x001C)	/* Core PLL Configuration */
#define SPLL			(ASIC_BASE+0x0020)	/* System PLL Configuration */
#define UPLL			(ASIC_BASE+0x0024)	/* USB PLL Configuration */
#define PPLL			(ASIC_BASE+0x0028)	/* PHY PLL Configuration */

/* ----------------------------------------------------------------------------- */
/* MEMORY Control Registers    (0xF0010000 ~ 0xF001FFFF)						*/
/* ----------------------------------------------------------------------------- */

#define B0CON			(ASIC_BASE+0x10000)	/* Bank 0 control */
#define B1CON			(ASIC_BASE+0x10004)	/* Bank 1 control */
#define B2CON			(ASIC_BASE+0x10008)	/* Bank 2 control */
#define B3CON			(ASIC_BASE+0x1000C)	/* Bank 3 control */
#define B4CON			(ASIC_BASE+0x10010)	/* Bank 4 control */
#define B5CON			(ASIC_BASE+0x10014)	/* Bank 5 control */
#define B6CON			(ASIC_BASE+0x10018)	/* Bank 6 control */
#define B7CON			(ASIC_BASE+0x1001C)	/* Bank 7 control */
#define MUXBCON			(ASIC_BASE+0x10020)	/* Muxed bus control */
#define WAITCON			(ASIC_BASE+0x10024)	/* Wait control */
#define WDSCON			(ASIC_BASE+0x10028)	/* Bank access after read wait control */

/* ----------------------------------------------------------------------------- */
/* SDRAM  Control Registers    (0xF0020000 ~ 0xF002FFFF)	                    */
/* ----------------------------------------------------------------------------- */

#define CFGREG   		(ASIC_BASE+0x20000)	/* Configuration register 0 */
#define CMDREG			(ASIC_BASE+0x20004)	/* Configuration register 1 */
#define REFREG			(ASIC_BASE+0x20008)	/* Refresh timer register */
#define WBTOREG			(ASIC_BASE+0x2000c)	/* Write buffer time-out register */

/* Used in DiagCode */
#define SDRAMCONFIG0   	CFGREG
#define SDRAMCONFIG1	CMDREG
#define SDRAMREFRESH	REFREG
#define SDRAMWRIEBUF	WBTOREG

/* ----------------------------------------------------------------------------- */
/* GPIO   control Registers           (0xF0030000 ~ 0xF003FFFF)		            */
/* ----------------------------------------------------------------------------- */

#define IOPMODE1		(ASIC_BASE+0x30000)	/* I/O port mode select lower register for PP[0]~PP[31] */
#define IOPMODE2		(ASIC_BASE+0x30004)	/* I/O port mode select upper register for PP[32]~PP[63] */
#define IOPCON1			(ASIC_BASE+0x30008)	/* I/O port select lower register for PP[0]~PP[31] */
#define IOPCON2			(ASIC_BASE+0x3000C)	/* I/O port select lower register for PP[32]~PP[63] */
#define IOPGDMA			(ASIC_BASE+0x30010)	/* I/O port special function register for DMA */
#define IOPEXTINT		(ASIC_BASE+0x30014)	/* I/O port special function register for external interrupt */
#define IOPEXTINTPND		(ASIC_BASE+0x30018)	/* External Interrupt clear register */
#define IOPDATA1		(ASIC_BASE+0x3001C)	/* I/O port data register */
#define IOPDATA2		(ASIC_BASE+0x30020)	/* I/O port data register */
#define IOPDRV1			(ASIC_BASE+0x30024)	/* I/O port drive control register */
#define IOPDRV2			(ASIC_BASE+0x30028)	/* I/O port drive control register */

/* ----------------------------------------------------------------------------- */
/* TIMER  Control Registers           (0xF0040000 ~ 0xF004FFFF) 		        */
/* ----------------------------------------------------------------------------- */

#define TIMER_TMOD			(ASIC_BASE+0x40000)	/* Timer mode register */
#define TIMER_TIC			(ASIC_BASE+0x40004)	/* Timer Interrupt Clear */
#define TIMER_WDT			(ASIC_BASE+0x40008)	/* Watchdog Timer Register */
#define TIMER_TDATA_0			(ASIC_BASE+0x40010)	/* Timer0 data register */
#define TIMER_TCNT_0			(ASIC_BASE+0x40014)	/* Timer0 counter register */
#define TIMER_TDATA_1			(ASIC_BASE+0x40018)	/* Timer1 data register */
#define TIMER_TCNT_1			(ASIC_BASE+0x4001c)	/* Timer1 counter register */
#define TIMER_TDATA_2			(ASIC_BASE+0x40020)	/* Timer2 data register */
#define TIMER_TCNT_2			(ASIC_BASE+0x40024)	/* Timer2 counter register */
#define TIMER_TDATA_3			(ASIC_BASE+0x40028)	/* Timer3 data register */
#define TIMER_TCNT_3			(ASIC_BASE+0x4002c)	/* Timer3 counter register */
#define TIMER_TDATA_4			(ASIC_BASE+0x40030)	/* Timer4 data register */
#define TIMER_TCNT_4			(ASIC_BASE+0x40034)	/* Timer4 counter register */
#define TIMER_TDATA_5			(ASIC_BASE+0x40038)	/* Timer5 data register */
#define TIMER_TCNT_5			(ASIC_BASE+0x4003c)	/* Timer5 counter register */

/* Used in DiagCode */
#define TIMER_TDATA(ch)		(ASIC_BASE+0x40010 + ch*0x8)	/* Timer  data register */
#define TIMER_TCNT(ch)		(ASIC_BASE+0x40014 + ch*0x8)	/* Timer  counter register */
#define WATCHDOG		WDT

/* ----------------------------------------------------------------------------- */
/* GDMA Ch 0/1/2/3/4/5   Control Registers        (0xF0050000 ~ 0xF005FFFF)		*/
/* ----------------------------------------------------------------------------- */

#define DPRIC			(ASIC_BASE+0x51000)	/* GDMA priority configuration register */
#define DPRIF			(ASIC_BASE+0x52000)	/* GDMA programmable priority register for fixed */
#define DPRIR			(ASIC_BASE+0x53000)	/* GDMA programmable priority register for round-robin */
#define DCON0			(ASIC_BASE+0x50000 + 0*0x20)	/* GDMA ch control register */
#define DSAR0			(ASIC_BASE+0x50004 + 0*0x20)	/* GDMA ch source address register */
#define DDAR0			(ASIC_BASE+0x50008 + 0*0x20)	/* GDMA ch destination address register */
#define DTCR0			(ASIC_BASE+0x5000C + 0*0x20)	/* GDMA ch transfer count register */
#define DRER0			(ASIC_BASE+0x50010 + 0*0x20)	/* GDMA ch run enable register */
#define DIPR0			(ASIC_BASE+0x50014 + 0*0x20)	/* GDMA ch interrupt pending register */
#define DCON1			(ASIC_BASE+0x50000 + 1*0x20)	/* GDMA ch control register */
#define DSAR1			(ASIC_BASE+0x50004 + 1*0x20)	/* GDMA ch source address register */
#define DDAR1			(ASIC_BASE+0x50008 + 1*0x20)	/* GDMA ch destination address register */
#define DTCR1			(ASIC_BASE+0x5000C + 1*0x20)	/* GDMA ch transfer count register */
#define DRER1			(ASIC_BASE+0x50010 + 1*0x20)	/* GDMA ch run enable register */
#define DIPR1			(ASIC_BASE+0x50014 + 1*0x20)	/* GDMA ch interrupt pending register */
#define DCON2			(ASIC_BASE+0x50000 + 2*0x20)	/* GDMA ch control register */
#define DSAR2			(ASIC_BASE+0x50004 + 2*0x20)	/* GDMA ch source address register */
#define DDAR2			(ASIC_BASE+0x50008 + 2*0x20)	/* GDMA ch destination address register */
#define DTCR2			(ASIC_BASE+0x5000C + 2*0x20)	/* GDMA ch transfer count register */
#define DRER2			(ASIC_BASE+0x50010 + 2*0x20)	/* GDMA ch run enable register */
#define DIPR2			(ASIC_BASE+0x50014 + 2*0x20)	/* GDMA ch interrupt pending register */
#define DCON3			(ASIC_BASE+0x50000 + 3*0x20)	/* GDMA ch control register */
#define DSAR3			(ASIC_BASE+0x50004 + 3*0x20)	/* GDMA ch source address register */
#define DDAR3			(ASIC_BASE+0x50008 + 3*0x20)	/* GDMA ch destination address register */
#define DTCR3			(ASIC_BASE+0x5000C + 3*0x20)	/* GDMA ch transfer count register */
#define DRER3			(ASIC_BASE+0x50010 + 3*0x20)	/* GDMA ch run enable register */
#define DIPR3			(ASIC_BASE+0x50014 + 3*0x20)	/* GDMA ch interrupt pending register */
#define DCON4			(ASIC_BASE+0x50000 + 4*0x20)	/* GDMA ch control register */
#define DSAR4			(ASIC_BASE+0x50004 + 4*0x20)	/* GDMA ch source address register */
#define DDAR4			(ASIC_BASE+0x50008 + 4*0x20)	/* GDMA ch destination address register */
#define DTCR4			(ASIC_BASE+0x5000C + 4*0x20)	/* GDMA ch transfer count register */
#define DRER4			(ASIC_BASE+0x50010 + 4*0x20)	/* GDMA ch run enable register */
#define DIPR4			(ASIC_BASE+0x50014 + 4*0x20)	/* GDMA ch interrupt pending register */
#define DCON5			(ASIC_BASE+0x50000 + 5*0x20)	/* GDMA ch control register */
#define DSAR5			(ASIC_BASE+0x50004 + 5*0x20)	/* GDMA ch source address register */
#define DDAR5			(ASIC_BASE+0x50008 + 5*0x20)	/* GDMA ch destination address register */
#define DTCR5			(ASIC_BASE+0x5000C + 5*0x20)	/* GDMA ch transfer count register */
#define DRER5			(ASIC_BASE+0x50010 + 5*0x20)	/* GDMA ch run enable register */
#define DIPR5			(ASIC_BASE+0x50014 + 5*0x20)	/* GDMA ch interrupt pending register */

/* Used in DiagCode */
#define DCON(ch)		(ASIC_BASE+0x50000 + ch*0x20)	/* GDMA ch control register */
#define DSAR(ch)		(ASIC_BASE+0x50004 + ch*0x20)	/* GDMA ch source address register */
#define DDAR(ch)		(ASIC_BASE+0x50008 + ch*0x20)	/* GDMA ch destination address register */
#define DTCR(ch)		(ASIC_BASE+0x5000C + ch*0x20)	/* GDMA ch transfer count register */
#define DRER(ch)		(ASIC_BASE+0x50010 + ch*0x20)	/* GDMA ch run enable register */
#define DIPR(ch)		(ASIC_BASE+0x50014 + ch*0x20)	/* GDMA ch interrupt pending register */

/* ----------------------------------------------------------------------------- */
/* Console UART Control Registers     (0xF0060000 ~ 0xF006FFFF)					*/
/* ----------------------------------------------------------------------------- */

#define CUCON			(ASIC_BASE+0x60000)	/* Console UART control register */
#define CUSTAT			(ASIC_BASE+0x60004)	/* Console UART status register */
#define CUINT			(ASIC_BASE+0x60008)	/* Console UART interrupt enable register */
#define CUTXBUF			(ASIC_BASE+0x6000c)	/* Console UART transmit data register */
#define CURXBUF			(ASIC_BASE+0x60010)	/* Console UART receive data register */
#define CUBRD			(ASIC_BASE+0x60014)	/* Console UART baud rate divisor register */
#define CUCHAR1			(ASIC_BASE+0x60018)	/* Console UART control character register1 */
#define CUCHAR2			(ASIC_BASE+0x6001c)	/* Console UART control character register2 */

/* Used in DiagCode */
#define UCON			(ASIC_BASE+0x60000)	/* Console UART control register */
#define USTAT			(ASIC_BASE+0x60004)	/* Console UART status register */
#define UINTEN			(ASIC_BASE+0x60008)	/* Console UART interrupt enable register */
#define UTXDAT			(ASIC_BASE+0x6000c)	/* Console UART transmit data register */
#define URXDAT			(ASIC_BASE+0x60010)	/* Console UART receive data register */
#define UBRDIV			(ASIC_BASE+0x60014)	/* Console UART baud rate divisor register */
#define CONCHAR1		(ASIC_BASE+0x60018)	/* Console UART control character register */
#define CONCHAR2		(ASIC_BASE+0x6001c)	/* Console UART control character register */

/* ----------------------------------------------------------------------------- */
/* High-Speed UART Ch 0/1 control Registers (0xF0070000 ~ 0xF008FFFF)			*/
/* ----------------------------------------------------------------------------- */

#define HUCON0			(ASIC_BASE + 0x70000 + 0*0x10000)	/* R/W  High-Speed UART control register */
#define HUSTAT0			(ASIC_BASE + 0x70004 + 0*0x10000)	/* High-Speed UART status register */
#define HUINT0	    	(ASIC_BASE + 0x70008 + 0*0x10000)	/* High-Speed UART interrupt enable register */
#define HUTXBUF0		(ASIC_BASE + 0x7000C + 0*0x10000)	/* High-Speed UART transmit data register */
#define HURXBUF0		(ASIC_BASE + 0x70010 + 0*0x10000)	/* High-Speed UART receive data register */
#define HUBRD0	    	(ASIC_BASE + 0x70014 + 0*0x10000)	/* High-Speed UART baud rate divisor register */
#define HUCHAR10		(ASIC_BASE + 0x70018 + 0*0x10000)	/* High-Speed UART control character register 1 */
#define HUCHAR20		(ASIC_BASE + 0x7001C + 0*0x10000)	/* High-Speed UART control character register 2 */
#define HUABB0			(ASIC_BASE + 0x70100 + 0*0x10000)	/* High-Speed UART autobaud boundary register */
#define HUABT0			(ASIC_BASE + 0x70104 + 0*0x10000)	/* High-Speed UART autobaud table register */

#define HUCON1			(ASIC_BASE + 0x70000 + 1*0x10000)	/* R/W  High-Speed UART control register */
#define HUSTAT1			(ASIC_BASE + 0x70004 + 1*0x10000)	/* High-Speed UART status register */
#define HUINT1	    	(ASIC_BASE + 0x70008 + 1*0x10000)	/* High-Speed UART interrupt enable register */
#define HUTXBUF1		(ASIC_BASE + 0x7000C + 1*0x10000)	/* High-Speed UART transmit data register */
#define HURXBUF1		(ASIC_BASE + 0x70010 + 1*0x10000)	/* High-Speed UART receive data register */
#define HUBRD1	    	(ASIC_BASE + 0x70014 + 1*0x10000)	/* High-Speed UART baud rate divisor register */
#define HUCHAR11		(ASIC_BASE + 0x70018 + 1*0x10000)	/* High-Speed UART control character register 1 */
#define HUCHAR21		(ASIC_BASE + 0x7001C + 1*0x10000)	/* High-Speed UART control character register 2 */
#define HUABB1			(ASIC_BASE + 0x70100 + 1*0x10000)	/* High-Speed UART autobaud boundary register */
#define HUABT1			(ASIC_BASE + 0x70104 + 1*0x10000)	/* High-Speed UART autobaud table register */

/* ----------------------------------------------------------------------------- */
/* DES Control Registers              (0xF0090000 ~ 0xF009FFFF)					*/
/* ----------------------------------------------------------------------------- */

#define DESCON			(ASIC_BASE+0x90000)	/* DES/3DES control register */
#define DESSTA			(ASIC_BASE+0x90004)	/* DES/3DES status register */
#define DESINT			(ASIC_BASE+0x90008)	/* DES/3DES interrupt enable register */
#define DESRUN			(ASIC_BASE+0x9000C)	/* DES/3DES run enable register */
#define DESKEY1L		(ASIC_BASE+0x90010)	/* Key 1 left half */
#define DESKEY1R		(ASIC_BASE+0x90014)	/* Key 1 right half */
#define DESKEY2L		(ASIC_BASE+0x90018)	/* Key 2 left half */
#define DESKEY2R		(ASIC_BASE+0x9001C)	/* Key 2 right half */
#define DESKEY3L		(ASIC_BASE+0x90020)	/* Key 3 left half */
#define DESKEY3R		(ASIC_BASE+0x90024)	/* Key 3 right half */
#define DESIVL			(ASIC_BASE+0x90028)	/* IV left half */
#define DESIVR			(ASIC_BASE+0x9002C)	/* IV right half */
#define DESINFIFO		(ASIC_BASE+0x90030)	/* DES/3DES input FIFO */
#define DESOUTFIFO		(ASIC_BASE+0x90034)	/* DES/3DES output FIFO */

/* ----------------------------------------------------------------------------- */
/* ETHERNET Ch 0/1  Control Registers    (0xF00A0000 ~ 0xF00DFFFF)				*/
/* ----------------------------------------------------------------------------- */
#if 0				/* defined also in MacApi.h */
#define BDMATXCON(ch)	(ASIC_BASE+0xA0000 + ch*0x20000)	/* Buffered DMA transmit control register */
#define BDMARXCON(ch)	(ASIC_BASE+0xA0004 + ch*0x20000)	/* Buffered DMA receive control register */
#define BDMATXDPTR(ch)	(ASIC_BASE+0xA0008 + ch*0x20000)	/* Transmit buffer descriptor start address */
#define BDMARXDPTR(ch)	(ASIC_BASE+0xA000C + ch*0x20000)	/* Receive buffer descriptor start address */
#define BTXBDCNT(ch)	(ASIC_BASE+0xA0010 + ch*0x20000)	/* BDMA Tx Buffer descriptor Counter */
#define BRXBDCNT(ch)	(ASIC_BASE+0xA0014 + ch*0x20000)	/* BDMA Rx Buffer descriptor Counter */
#define BMTXINTEN(ch)	(ASIC_BASE+0xA0018 + ch*0x20000)	/* BDMA/MAC Tx Interrupt enable register */
#define BMRXINTEN(ch)	(ASIC_BASE+0xA001C + ch*0x20000)	/* BDMA/MAC Rx Interrupt enable register */
#define BMTXSTAT(ch)	(ASIC_BASE+0xA0020 + ch*0x20000)	/* BDMA/MAC Tx Status register */
#define BMRXSTAT(ch)	(ASIC_BASE+0xA0024 + ch*0x20000)	/* BDMA/MAC Rx Status register */
#define BDMARXLEN(ch)	(ASIC_BASE+0xA0028 + ch*0x20000)	/* Receive Frame Size */
#define CFTXSTATA(ch)	(ASIC_BASE+0xA0030 + ch* 0x2000)	/* Transmit control frame status */
#define MACCON(ch)	(ASIC_BASE+0xB0000 + ch*0x20000)	/* MAC control */
#define CAMCON(ch)	(ASIC_BASE+0xB0004 + ch*0x20000)	/* CAM control */
#define MACTXCON(ch)	(ASIC_BASE+0xB0008 + ch*0x20000)	/* Transmit control */
#define MACTXSTAT(ch)	(ASIC_BASE+0xB000C + ch*0x20000)	/* Transmit status */
#define MACRXCON(ch)	(ASIC_BASE+0xB0010 + ch*0x20000)	/* Receive control */
#define MACRXSTAT(ch)	(ASIC_BASE+0xB0014 + ch*0x20000)	/* Receive status */
#define STADATA(ch)	(ASIC_BASE+0xB0018 + ch*0x20000)	/* Station management data */
#define STACON(ch)	(ASIC_BASE+0xB001C + ch*0x20000)	/* Station management control and address */
#define CAMEN(ch)	(ASIC_BASE+0xB0028 + ch*0x20000)	/* CAM enable */
#define MISSCNT(ch)	(ASIC_BASE+0xB003C + ch*0x20000)	/* Missed error count */
#define PZCNT(ch)	(ASIC_BASE+0xB0040 + ch*0x20000)	/* Pause count */
#define RMPZCNT(ch)	(ASIC_BASE+0xB0044 + ch*0x20000)	/* Remote pause count */

#define CAM(ch)		(ASIC_BASE+0xB0080 + ch*0x20000)	/* CAM content (32 words) Undefined */
#define CAM_Reg(ch,x)	(CAM(ch) + (x*0x4))	/* CAM content (32 words) Undefined */
#else
#define BMTXSTATA	(ASIC_BASE+0xA0020 + 0*0x20000)	/* BDMA/MAC Tx Status register */
#define BMRXSTATA	(ASIC_BASE+0xA0024 + 0*0x20000)	/* BDMA/MAC Rx Status register */

#define BMTXSTATB	(ASIC_BASE+0xA0020 + 1*0x20000)	/* BDMA/MAC Tx Status register */
#define BMRXSTATB	(ASIC_BASE+0xA0024 + 1*0x20000)	/* BDMA/MAC Rx Status register */
#endif
/* ----------------------------------------------------------------------------- */
/* USB Control Registers              (0xF00E0000 ~ 0xF00EFFFF)					*/
/* ----------------------------------------------------------------------------- */

#define USBFA			(ASIC_BASE+0xE0000)	/* USB function address register */
#define USBPM			(ASIC_BASE+0xE0004)	/* USB power management register */
#define USBINTR			(ASIC_BASE+0xE0008)	/* USB interrupt register */
#define USBINTRE		(ASIC_BASE+0xE000C)	/* USB interrupt enable register */
#define USBFN			(ASIC_BASE+0xE0010)	/* USB frame number register */
#define USBDISCONN		(ASIC_BASE+0xE0014)	/* USB disconnect timer register */
#define USBEP0CSR		(ASIC_BASE+0xE0018)	/* USB endpoint 0 common status register */
#define USBEP1CSR		(ASIC_BASE+0xE001C)	/* USB endpoint 1 common status register */
#define USBEP2CSR		(ASIC_BASE+0xE0020)	/* USB endpoint 2 common status register */
#define USBEP3CSR		(ASIC_BASE+0xE0024)	/* USB endpoint 3 common status register */
#define USBEP4CSR		(ASIC_BASE+0xE0028)	/* USB endpoint 4 common status register */
#define USBWCEP0		(ASIC_BASE+0xE0030)	/* USB write count register for endpoint 0 */
#define USBWCEP1		(ASIC_BASE+0xE0034)	/* USB write count register for endpoint 1 */
#define USBWCEP2		(ASIC_BASE+0xE0038)	/* USB write count register for endpoint 2 */
#define USBWCEP3		(ASIC_BASE+0xE003C)	/* USB write count register for endpoint 3 */
#define USBWCEP4		(ASIC_BASE+0xE0040)	/* USB write count register for endpoint 4 */
#define USBEP0			(ASIC_BASE+0xE0080)	/* USB endpoint 0 FIFO */
#define USBEP1			(ASIC_BASE+0xE0084)	/* USB endpoint 1 FIFO */
#define USBEP2			(ASIC_BASE+0xE0088)	/* USB endpoint 2 FIFO */
#define USBEP3			(ASIC_BASE+0xE008C)	/* USB endpoint 3 FIFO */
#define USBEP4			(ASIC_BASE+0xE0090)	/* USB endpoint 4 FIFO */

/* ----------------------------------------------------------------------------- */
/* IIC Control Registers              (0xF00F0000 ~ 0xF00FFFFF)					*/
/* ----------------------------------------------------------------------------- */

#define IICCON			(ASIC_BASE+0xF0000)	/* Control status register */
#define IICBUF			(ASIC_BASE+0xF0004)	/* Shift buffer register */
#define IICPS			(ASIC_BASE+0xF0008)	/* Prescaler register */
#define IICCNT			(ASIC_BASE+0xF000C)	/* Prescaler counter register */
#define IICPND			(ASIC_BASE+0xF0010)	/* Interrupt pending register */

/* ----------------------------------------------------------------------------- */
/* HDLC Ch 0/1/2  Control Registers      (0xF0100000 ~ 0xF012FFFF)				*/
/* ----------------------------------------------------------------------------- */
#define HMODE0 			(ASIC_BASE+0x100000 + 0*0x10000)	/* HDLC mode register */
#define HCON0 			(ASIC_BASE+0x100004 + 0*0x10000)	/* HDLC control register */
#define HSTAT0  		(ASIC_BASE+0x100008 + 0*0x10000)	/* HDLC status register */
#define HINTEN0 		(ASIC_BASE+0x10000c + 0*0x10000)	/* HDLC interrupt enable register */
#define HTXFIFOC0 		(ASIC_BASE+0x100010 + 0*0x10000)	/* HTxFIFO frame continue register */
#define HTXFIFOT0 		(ASIC_BASE+0x100014 + 0*0x10000)	/* HTxFIFO frame terminate register */
#define HRXFIFO0 		(ASIC_BASE+0x100018 + 0*0x10000)	/* HRxFIFO entry register */
#define HBRGTC0			(ASIC_BASE+0x10001c + 0*0x10000)	/* HDLC BRG time constant register */
#define HPRMB0	 		(ASIC_BASE+0x100020 + 0*0x10000)	/* HDLC preamble register */
#define HSAR00 			(ASIC_BASE+0x100024 + 0*0x10000)	/* HDLC station address 0 */
#define HSAR10	 		(ASIC_BASE+0x100028 + 0*0x10000)	/* HDLC station address 1 */
#define HSAR20	 		(ASIC_BASE+0x10002c + 0*0x10000)	/* HDLC station address 2 */
#define HSAR30	 		(ASIC_BASE+0x100030 + 0*0x10000)	/* HDLC station address 3 */
#define HMASK0 			(ASIC_BASE+0x100034 + 0*0x10000)	/* HDLC mask register */
#define HDMATXPTR0		(ASIC_BASE+0x100038 + 0*0x10000)	/* DMA Tx buffer descriptor pointer */
#define HDMARXPTR0		(ASIC_BASE+0x10003c + 0*0x10000)	/* DMA Rx buffer descriptor pointer */
#define HMFLR0 			(ASIC_BASE+0x100040 + 0*0x10000)	/* Maximum frame length register */
#define HRBSR0 			(ASIC_BASE+0x100044 + 0*0x10000)	/* Receive buffer size register */
#define HSYNC0			(ASIC_BASE+0x100048 + 0*0x10000)	/* HDLC Sync Register */
#define TCON0			(ASIC_BASE+0x10004c + 0*0x10000)	/* Transparent Control regiseter */
#define HTXBDCNT0		(ASIC_BASE+0x1000c0 + 0*0x10000)	/* Tx buffer descriptor count register */
#define HRXBDCNT0		(ASIC_BASE+0x1000c4 + 0*0x10000)	/* Rx buffer descriptor count register */
#define HTXMAXBDCNT0	(ASIC_BASE+0x1000c8 + 0*0x10000)	/* Tx buffer descriptor count mask register */
#define HRXMAXBDCNT0	(ASIC_BASE+0x1000cc + 0*0x10000)	/* Rx buffer descriptor count mask register */

#define HMODE1	 		(ASIC_BASE+0x100000 + 1*0x10000)	/* HDLC mode register */
#define HCON1 			(ASIC_BASE+0x100004 + 1*0x10000)	/* HDLC control register */
#define HSTAT1  		(ASIC_BASE+0x100008 + 1*0x10000)	/* HDLC status register */
#define HINTEN1 		(ASIC_BASE+0x10000c + 1*0x10000)	/* HDLC interrupt enable register */
#define HTXFIFOC1 		(ASIC_BASE+0x100010 + 1*0x10000)	/* HTxFIFO frame continue register */
#define HTXFIFOT1 		(ASIC_BASE+0x100014 + 1*0x10000)	/* HTxFIFO frame terminate register */
#define HRXFIFO1 		(ASIC_BASE+0x100018 + 1*0x10000)	/* HRxFIFO entry register */
#define HBRGTC1			(ASIC_BASE+0x10001c + 1*0x10000)	/* HDLC BRG time constant register */
#define HPRMB1	 		(ASIC_BASE+0x100020 + 1*0x10000)	/* HDLC preamble register */
#define HSAR01 			(ASIC_BASE+0x100024 + 1*0x10000)	/* HDLC station address 0 */
#define HSAR11	 		(ASIC_BASE+0x100028 + 1*0x10000)	/* HDLC station address 1 */
#define HSAR21	 		(ASIC_BASE+0x10002c + 1*0x10000)	/* HDLC station address 2 */
#define HSAR31	 		(ASIC_BASE+0x100030 + 1*0x10000)	/* HDLC station address 3 */
#define HMASK1 			(ASIC_BASE+0x100034 + 1*0x10000)	/* HDLC mask register */
#define HDMATXPTR1		(ASIC_BASE+0x100038 + 1*0x10000)	/* DMA Tx buffer descriptor pointer */
#define HDMARXPTR1		(ASIC_BASE+0x10003c + 1*0x10000)	/* DMA Rx buffer descriptor pointer */
#define HMFLR1 			(ASIC_BASE+0x100040 + 1*0x10000)	/* Maximum frame length register */
#define HRBSR1 			(ASIC_BASE+0x100044 + 1*0x10000)	/* Receive buffer size register */
#define HSYNC1			(ASIC_BASE+0x100048 + 1*0x10000)	/* HDLC Sync Register */
#define TCON1			(ASIC_BASE+0x10004c + 1*0x10000)	/* Transparent Control regiseter */
#define HTXBDCNT1		(ASIC_BASE+0x1000c0 + 1*0x10000)	/* Tx buffer descriptor count register */
#define HRXBDCNT1		(ASIC_BASE+0x1000c4 + 1*0x10000)	/* Rx buffer descriptor count register */
#define HTXMAXBDCNT1	(ASIC_BASE+0x1000c8 + 1*0x10000)	/* Tx buffer descriptor count mask register */
#define HRXMAXBDCNT1	(ASIC_BASE+0x1000cc + 1*0x10000)	/* Rx buffer descriptor count mask register */

#define HMODE2	 		(ASIC_BASE+0x100000 + 2*0x10000)	/* HDLC mode register */
#define HCON2 			(ASIC_BASE+0x100004 + 2*0x10000)	/* HDLC control register */
#define HSTAT2  		(ASIC_BASE+0x100008 + 2*0x10000)	/* HDLC status register */
#define HINTEN2 		(ASIC_BASE+0x10000c + 2*0x10000)	/* HDLC interrupt enable register */
#define HTXFIFOC2 		(ASIC_BASE+0x100010 + 2*0x10000)	/* HTxFIFO frame continue register */
#define HTXFIFOT2 		(ASIC_BASE+0x100014 + 2*0x10000)	/* HTxFIFO frame terminate register */
#define HRXFIFO2 		(ASIC_BASE+0x100018 + 2*0x10000)	/* HRxFIFO entry register */
#define HBRGTC2			(ASIC_BASE+0x10001c + 2*0x10000)	/* HDLC BRG time constant register */
#define HPRMB2		 	(ASIC_BASE+0x100020 + 2*0x10000)	/* HDLC preamble register */
#define HSAR02 			(ASIC_BASE+0x100024 + 2*0x10000)	/* HDLC station address 0 */
#define HSAR12	 		(ASIC_BASE+0x100028 + 2*0x10000)	/* HDLC station address 1 */
#define HSAR22	 		(ASIC_BASE+0x10002c + 2*0x10000)	/* HDLC station address 2 */
#define HSAR32	 		(ASIC_BASE+0x100030 + 2*0x10000)	/* HDLC station address 3 */
#define HMASK2 			(ASIC_BASE+0x100034 + 2*0x10000)	/* HDLC mask register */
#define HDMATXPTR2		(ASIC_BASE+0x100038 + 2*0x10000)	/* DMA Tx buffer descriptor pointer */
#define HDMARXPTR2		(ASIC_BASE+0x10003c + 2*0x10000)	/* DMA Rx buffer descriptor pointer */
#define HMFLR2 			(ASIC_BASE+0x100040 + 2*0x10000)	/* Maximum frame length register */
#define HRBSR2 			(ASIC_BASE+0x100044 + 2*0x10000)	/* Receive buffer size register */
#define HSYNC2			(ASIC_BASE+0x100048 + 2*0x10000)	/* HDLC Sync Register */
#define TCON2			(ASIC_BASE+0x10004c + 2*0x10000)	/* Transparent Control regiseter */
#define HTXBDCNT2		(ASIC_BASE+0x1000c0 + 2*0x10000)	/* Tx buffer descriptor count register */
#define HRXBDCNT2		(ASIC_BASE+0x1000c4 + 2*0x10000)	/* Rx buffer descriptor count register */
#define HTXMAXBDCNT2	(ASIC_BASE+0x1000c8 + 2*0x10000)	/* Tx buffer descriptor count mask register */
#define HRXMAXBDCNT2	(ASIC_BASE+0x1000cc + 2*0x10000)	/* Rx buffer descriptor count mask register */


/* ----------------------------------------------------------------------------- */
/* IOM2  Control Registers            ((0xF0130000 ~ 0xF013FFFF))	            */
/* ----------------------------------------------------------------------------- */

#define IOM2CON			(ASIC_BASE+0x130000)	/* IOM2 Control Register */
#define IOM2STAT		(ASIC_BASE+0x130004)	/* Status Register */
#define IOM2INTEN		(ASIC_BASE+0x130008)	/* Interrupt Enable Register */
#define IOM2TBA			(ASIC_BASE+0x13000C)	/* TIC Bus Address */
#define IOM2ICTD		(ASIC_BASE+0x130010)	/* IC ch Tx Buffer */
#define IOM2ICRD		(ASIC_BASE+0x130014)	/* IC ch Rx Buffer */
#define IOM2CITD0		(ASIC_BASE+0x130018)	/* C/I0 ch Tx Buffer */
#define IOM2CIRD0		(ASIC_BASE+0x13001C)	/* C/I0 ch Rx Buffer */
#define IOM2CITD1		(ASIC_BASE+0x130020)	/* C/I1 ch Tx Buffer */
#define IOM2CIRD1		(ASIC_BASE+0x130024)	/* C/I1 ch Rx Buffer */
#define IOM2MTD			(ASIC_BASE+0x130028)	/* Monitor ch Tx Buffer */
#define IOM2MRD			(ASIC_BASE+0x13002C)	/* Monitor ch Rx Buffer */
#define TSAACON			(ASIC_BASE+0x130030)	/* TSA A Configuration Register */
#define TSABCON			(ASIC_BASE+0x130034)	/* TSA B Configuration Register */
#define TSACCON			(ASIC_BASE+0x130038)	/* TSA C Configuration Register */
#define IOM2STRB		(ASIC_BASE+0x13003C)	/* IOM2 Strobe Set Register */

/* Used in DiagCode */
#define IOM2CITD(c)		(ASIC_BASE+0x130018 + c*0x8)	/* C/I ch Tx Buffer */
#define IOM2CIRD(c)		(ASIC_BASE+0x13001C + c*0x8)	/* C/I ch Rx Buffer */

#define TSAACFG			(ASIC_BASE+0x130030)	/* TSA A Configuration Register */
#define TSABCFG			(ASIC_BASE+0x130034)	/* TSA B Configuration Register */
#define TSACCFG			(ASIC_BASE+0x130038)	/* TSA C Configuration Register */

/* ----------------------------------------------------------------------------- */
/* INTERRUPT  Control Registers            (0xF0140000 ~ 0xF014FFFF)		    */
/* ----------------------------------------------------------------------------- */

#define INTMOD			(ASIC_BASE+0x140000)	/* Internal interrupt mode register */
#define EXTMOD			(ASIC_BASE+0x140004)	/* External interrupt mode register */
#define INTMASK			(ASIC_BASE+0x140008)	/* Internal Interrupt mask register */
#define EXTMASK			(ASIC_BASE+0x14000C)	/* External Interrupt mask register */
#define IPRIORHI		(ASIC_BASE+0x140010)	/* High bits,5~0 bit, Interrupt by priority register */
#define IPRIORLO		(ASIC_BASE+0x140014)	/* Low bits, 31~0 bit, Interrupt by priority register */
#define INTOFFSET_FIQ		(ASIC_BASE+0x140018)	/* FIQ interrupt offset register */
#define INTOFFSET_IRQ		(ASIC_BASE+0x14001C)	/* IRQ interrupt offset register */
#define INTPRIOR0		(ASIC_BASE+0x140020)	/* Interrupt priority register 0 */
#define INTPRIOR1		(ASIC_BASE+0x140024)	/* Interrupt priority register 1 */
#define INTPRIOR2		(ASIC_BASE+0x140028)	/* Interrupt priority register 2 */
#define INTPRIOR3		(ASIC_BASE+0x14002C)	/* Interrupt priority register 3 */
#define INTPRIOR4		(ASIC_BASE+0x140030)	/* Interrupt priority register 4 */
#define INTPRIOR5		(ASIC_BASE+0x140034)	/* Interrupt priority register 5 */
#define INTPRIOR6		(ASIC_BASE+0x140038)	/* Interrupt priority register 6 */
#define INTPRIOR7		(ASIC_BASE+0x14003C)	/* Interrupt priority register 7 */
#define INTPRIOR8		(ASIC_BASE+0x140040)	/* Interrupt priority register 8 */
#define INTPRIOR9		(ASIC_BASE+0x140044)	/* Interrupt priority register 9 */
#define INTTST			(ASIC_BASE+0x140048)	/* Interrupt test register */
#define EXTTST			(ASIC_BASE+0x14004C)	/* Interrupt test register */

/* ----------------------------------------------------------------------------- */
/* UART STATUS BITS																*/
/* ----------------------------------------------------------------------------- */
/* CUCON: Console UART control defines */

/* bit 1:0 */
#define	TxModeCpu		0x01	// UART Transmit mode interrupt request
#define	TxModeGDMA		0x02	// UART Transmit mode GDMA (Only for High Speed UART)
/* bit 3:2 */
#define	RxModeCpu		0x04	// UART Receive mode  interrupt request
#define	RxModeGDMA		0x08	// UART Receive mode  GDMA (Use High Speed UART)
/* bit 4 */
#define	Send_Break		0x010	// Send Break
#define	No_Break		0x000
/* bit 5 */
#define	Serial_Clock	0x000	// Serial Clock Internal
#define	Serial_ExtClock	0x020	// Serial Clock External
/* bit 6 */
#define	ABRD			0x040	// Auto Baud Rate Detect(Use High Speed UART)
/* bit 7 */
#define	Loop_Back		0x080	// Loop-back mode
/* bit 10:8 */
#define	Parity_Mask			0x700	// Parity mode
#define	Parity_None			0x000
#define	Parity_Odd			0x400
#define	Parity_Even			0x500
#define	Parity_1			0x600
#define	Parity_0			0x700
/* bit 11 */
#define	Stop_Bits_Mask		0x00000800	// Number of Stop bits
#define	One_Stop_Bit		0x00000000
#define	Two_Stop_Bits		0x00000800
/* bit 13:12 */
#define	Data_Length_5		0x0000	// Word Length
#define	Data_Length_6		0x1000
#define	Data_Length_7		0x2000
#define	Data_Length_8		0x3000
#define	Data_Length_Mask	0x3000
/* bit 14 */
#define	Infra_Red_Mode		0x4000	// Infra-red mode
/* bit 29 */
#define	Flow_Software		(0x1<<29)	// S/W Flow control
/* bit 30 */
#define	Uart_Echo			(0x1<<30)	// Echo RX data to TX port directly

/* CUSTAT: Console UART status register */

/* Rx error */
#define RX_ERROR			(0x1c)	// OVERRUN/ PARITY/ FRAMING
/* bit 0 */
#define Uart_Rx_Data_Ready	(0x1<<0)	// Receive Data Valid (RDV)
/* bit 1 */
#define Uart_Break_Detected	(0x1<<1)	// Break Signal Detected (BSD)
/* bit 2 */
#define Uart_Frame_Err		(0x1<<2)	// Frame Error (FER)
/* bit 3 */
#define Uart_Parity_Err		(0x1<<3)	// Parity Error (PER)
/* bit 4 */
#define Uart_Overrun_Err	(0x1<<4)	// Overrun Error(OER)
/* bit 5 */
#define Uart_Control_Detect	(0x1<<5)	// Control Caracter Detect (CCD)
/* bit 11 */
#define Uart_Rx_Idle		(0x1<<11)	// Receiver in idle (RXIDLE)
/* bit 17 */
#define Uart_Tx_Complete	(0x1<<17)	// Transmitter Idle (TI)
/* bit 18 */
#define Uart_Tx_Data_Ready	(0x1<<18)	// Transmit Holding Register Empty (THE)

/* CUINT: Console UART interrupt register */

#define	ENABLE_TX_RX		(Uart_Rx_Data_Ready | Uart_Tx_Data_Ready)
/* bit 0 */
#define	UART_RDVIE			Uart_Rx_Data_Ready	// Receive Data Valid interrupt enable
/* bit 1 */
#define	UART_BSDIE			Uart_Break_Detected	// Break Signal Detected interrupt enable
/* bit 2 */
#define	UART_FERIE			Uart_Frame_Err	// Frame Error interrupt enable
/* bit 3 */
#define	UART_PERIE			Uart_Parity_Err	// Parity Error interrupt enable
/* bit 4 */
#define	UART_OERIE			Uart_Overrun_Err	// Overrun Error interrupt enable
/* bit 5 */
#define	UART_CCDIE			Uart_Control_Detect	// Control Character Detect interrupt enable
/* bit 17 */
#define	UART_TIIE			Uart_Tx_Complete	// Transmitter idle interrupt enable
/* bit 18 */
#define	UART_THEIE			Uart_Tx_Data_Ready	// Transmit Holding Register Empty interrupt enable

#ifndef REG_READ
#define REG_READ(reg, result) \
	((result) = *(volatile ulong *)(reg))
#endif /*READ REG */

#ifndef REG_WRITE
#define REG_WRITE(reg, data) \
	(*((volatile ulong *)(reg)) = (data))
#endif /*WRITE REG */

/*************************************************************************
*
* DRAM Memory Bank 0 area MAP for Exception vector table 
* and Stack, User code area. 
*
*/

#define DRAM_BASE  			0x0	/* Final start address of DRAM */
#define DRAM_LIMIT 			0x2000000	/* 32MByte */
#define RESET_DRAM_START	0x40000000	/* Start of DRAM on power-up */
#define RESET_ROM_START		0x0	/* Start od ROM on power-up */

/****************************************************************************
*
* Format of the Program Status Register 
*/

#define FBit 	 	0x40
#define IBit  	 	0x80
#define LOCKOUT  	0xC0	/* Interrupt lockout value */
#define LOCK_MSK 	0xC0	/* Interrupt lockout mask value */
#define MOD_MASK 	0x1F	/* Processor Mode Mask */
#define UDF_MODE 	0x1B	/* Undefine Mode(UDF) */
#define ABT_MODE 	0x17	/* Abort Mode(ABT) */
#define SUP_MODE 	0x13	/* Supervisor Mode (SVC) */
#define IRQ_MODE 	0x12	/* Interrupt Mode (IRQ) */
#define FIQ_MODE 	0x11	/* Fast Interrupt Mode (FIQ) */
#define USR_MODE 	0x10	/* User Mode(USR) */

/* ----------------------------------------------------------------------------- */
/*  Define ARM CPSR and MMU Value										        */
/* ----------------------------------------------------------------------------- */
/* REGISTER 1 */
#define PROTECTION_ON	(0x1<<0)
#define DC_ON			(0x1<<2)
#define BIG_END			(0x1<<7)
#define IC_ON			(0x1<<12)
#define nFASTBUS		(0x1<<30)
#define ASYNC_CLK		(0x1<<31)

/* REGISTER 6 */
#define REGION_ENABLE	(0x1<<0)

/*************************************************************************
* SYSTEM MEMORY CONTROL REGISTER EQU TABLES 
*/
/* ----------------------------------------------------------------------------- */
/* EXTERNAL I/O BANK CONTROL SPECIAL REGISTER EQU TABLES
* ----------------------------------------------------------------------------- */

/*   BANK CONTROL REGISTER      */

#define DW_SFT		30	/* Physical memory data bus width */
									/*      0=Reserved,1=8it, 2=16bit, 3=32bit */
#define PMC_SFT		28	/* Page mode configuration */
									/*      0=Normal ROM or External I/O, 1=4word, 2=8word, 3=16word */
#define BS_SFT		24	/* Bank Sizeize0 */
									/*      0=Disable, 1=1M, 2=2M, 3=4M, 4=8M, 5=16M */
#define IS_SFT		23	/* nWBE or nBE/DQM selection (nWBE is only for writing operation, nBE/DQM is for reading/writing) */
									/*      0=nWBE function (8-bit), 1=nBE function (16-bit) */
#define TACC_SFT	16	/* Access cycles (nOE low time) */
									/*      0x3 = 3cycles, 0x4 = 4cycles, 0x5 = 5cycles, 0x6 = 6cycles, 0x7 = 7cycles */
									/*      0x8 = 8cycles, 0x9 = 9cycles, 0xA =10cycles, 0xB =11cycles, 0xC =12cycles */
									/*      0xD =13cycles, 0xE =14cycles, 0xF =15cycles, 0x10=16cycles, 0x11=17cycles */
									/*      0x12=18cycles, 0x13=19cycles, 0x14=20cycles, 0x15=21cycles, 0x16=22cycles */
									/*      0x17=23cycles, 0x18=24cycles, 0x19=25cycles, 0x1A=26cycles, 0x1B=27cycles */
									/*      0x1C=28cycles, 0x1D=29cycles, 0x1E=30cycles, 0x1F=31cycles */
#define TPA_SFT		12	/* Page address access cycle */
#define TACS_SFT	8	/* Address setup time */
#define TCOS_SFT	4	/* Chip selection setup time on nOE */
#define TCOH_SFT	0	/* Chip selection hold time on nOE */
									/*      0x3 = 3cycles, 0x4 = 4cycles ~ 0xF =15cycles */

#define rB0CON_0	0x14<<TACC_SFT+ 0xe<<TPA_SFT+ 0x4<<TACS_SFT+ 0x8<<TCOS_SFT+ 0x8<<TCOH_SFT
#define rB0CON   	1<<DW_SFT + 0<<PMC_SFT + 5<<BS_SFT + 0<<IS_SFT + rB0CON_0

#define rB1CON_0	0x14<<TACC_SFT + 0xe<<TPA_SFT + 0x4<<TACS_SFT + 0x8<<TCOS_SFT + 0x8<<TCOH_SFT
#define rB1CON   	2<<DW_SFT + 0<<PMC_SFT + 5<<BS_SFT + 0<<IS_SFT + rB1CON_0

#define rB2CON_0	((0x14<<TACC_SFT)+(0xe<<TPA_SFT)+(0x4<<TACS_SFT)+(0x8<<TCOS_SFT)+(0x8<<TCOH_SFT))
#define rB2CON   	((3<<DW_SFT)+(0<<PMC_SFT)+(5<<BS_SFT)+(1<<IS_SFT)+rB2CON_0)

#define rB3CON_0	((0xa<<TACC_SFT)+(0xe<<TPA_SFT)+(0x0<<TACS_SFT)+(0x0<<TCOS_SFT)+(0x1<<TCOH_SFT))
#define rB3CON   	((3<<DW_SFT)+(0<<PMC_SFT)+(5<<BS_SFT)+(1<<IS_SFT)+rB3CON_0)

#define rB4CON_0	((0xa<<TACC_SFT)+(0xe<<TPA_SFT)+(0x0<<TACS_SFT)|(0x0<<TCOS_SFT)+(0x1<<TCOH_SFT))
#define rB4CON   	((3<<DW_SFT)+(0<<PMC_SFT)+(5<<BS_SFT)+(1<<IS_SFT)+rB4CON_0)

#define rB5CON_0	((0xa<<TACC_SFT)+(0xe<<TPA_SFT)+(0x0<<TACS_SFT)+(0x0<<TCOS_SFT)+(0x1<<TCOH_SFT))
#define rB5CON   	((3<<DW_SFT)+(0<<PMC_SFT)+(5<<BS_SFT)+(1<<IS_SFT)+rB5CON_0)

#define rB6CON_0	((0xa<<TACC_SFT)+(0xe<<TPA_SFT)+(0x0<<TACS_SFT)+(0x1<<TCOS_SFT)+(0x1<<TCOH_SFT))
#define rB6CON   	((3<<DW_SFT)+(0<<PMC_SFT)+(5<<BS_SFT)+(1<<IS_SFT)+rB6CON_0)

#define rB7CON_0	((0xa<<TACC_SFT)+(0xe<<TPA_SFT)+(0x0<<TACS_SFT)+(0x1<<TCOS_SFT)+(0x1<<TCOH_SFT))
#define rB7CON   	((3<<DW_SFT)+(0<<PMC_SFT)+(5<<BS_SFT)+(1<<IS_SFT)+rB7CON_0)


/*	MUXBCON: Muxed Bus Control Register  */

#define MBE7		0x0<<31	/* 0x0=disable, 0x1=enable */
#define MBE6		0x0<<30
#define MBE5		0x0<<29
#define MBE4		0x0<<28
#define MBE3		0x0<<27
#define MBE2		0x0<<26
#define MBE1		0x0<<25
#define MBE0		0x0<<24

#define TMA7		0x3<<21	/* 0x0=2cycles, 0x1=3cycle, 0x2=4cycle, 0x3=5cycle */
#define TMA6		0x3<<18	/* 0x4=6cycles, 0x5=7cycle, 0x6=8cycle, 0x9=5cycle */
#define TMA5		0x3<<15
#define TMA4		0x3<<12
#define TMA3		0x3<<9
#define TMA2		0x3<<6
#define TMA1		0x3<<3
#define TMA0		0x3<<0

#define ALLMUXEN	(0xff<<24)
#define AULLMUXDIS	(0x0<<0)

#define rMUXBCON   	MBE7+MBE6+MBE5+MBE4+MBE3+MBE2+MBE1+MBE0+TMA7+TMA6+TMA5+TMA4+TMA3+TMA2+TMA1+TMA0


/*	WAITCON: Wait Control Register       */

#define COHDIS7		0x1<<23	/* This forces TCOH to '0' for read to read, write to write, */
				    /* and write to read access in bank 7 */
#define COHDIS6		0x1<<22	/* 0: disable  1: enable */
#define COHDIS5		0x1<<21
#define COHDIS4		0x1<<20
#define COHDIS3		0x1<<19
#define COHDIS2		0x1<<18
#define COHDIS1		0x1<<17
#define COHDIS0		0x1<<16

#define EWAITEN7	0x0<<15	/* External wait enable for bank(n) 0 : disable,  1 : enable */
#define EWAITEN6	0x0<<14
#define EWAITEN5	0x0<<13
#define EWAITEN4	0x0<<12
#define EWAITEN3	0x0<<11
#define EWAITEN2	0x0<<10
#define EWAITEN1	0x0<<9
#define EWAITEN0	0x0<<8

#define NREADY7		0x0<<7	/* nWait / nReady select for bank(n) 0 = nWait,   1 = nReady */
#define NREADY6		0x0<<6
#define NREADY5		0x0<<5
#define NREADY4		0x0<<4
#define NREADY3		0x0<<3
#define NREADY2		0x0<<2
#define NREADY1		0x0<<1
#define NREADY0		0x0<<0

#define rCOHDIS     COHDIS7+COHDIS6+COHDIS5+COHDIS4+COHDIS3+COHDIS2+COHDIS1+COHDIS0
#define rEWAITEN	EWAITEN7+EWAITEN6+EWAITEN5+EWAITEN4+EWAITEN3+EWAITEN2+EWAITEN1+EWAITEN0
#define rNREADY		NREADY7+NREADY6+NREADY5+NREADY4+NREADY3+NREADY2+NREADY1+NREADY0

#define rWAITCON	rCOHDIS+rEWAITEN+rNREADY
									/* Extern wait enable for bank (n)= disable */
									/* Bank (n) Wait single  : nWait */

/* ----------------------------------------------------------------------------- */
/* SDRAM CONTROL SPECIAL REGISTER TABLS										 */
/* ----------------------------------------------------------------------------- */

/*	CFGREG: SDRAM CONFIGURATION REGISTER 0	*/

#define SDR_XW		(0x0<<0)	/* External data bus Width, 0x0=32bit, 0x1=16it */
#define SDR_AP		(0x1<<1)	/* Auto Pre-charge control for SDRAM accesses, 0x0=Auto Pre-charge, 0x1=No Auto Pre-charge */
#define SDR_CL		(0x3<<2)	/* CAS Latency, 0x0=Reserved 0x1=1cycle, 0x2=2cycles, 0x3=3cycles */
#define SDR_D1		(0x3<<4)	/* SDRAM divice Density of bank 1, 0x0=16Mbit,  0x1=64Mbit, 0x2=128Mbit, 0x3=256Mbit */
#define SDR_D0		(0x3<<6)	/* SDRAM divice Density of bank 1, 0x0=16Mbit,  0x1=64Mbit, 0x2=128Mbit, 0x3=256Mbit */
#define SDR_RP		(0x2<<8)	/* Row Pre-charge time, 0x0=1cycle, 0x1=2cycle, 0x2=3cycles, 0x3=4cycles */
#define SDR_RCD		(0x2<<10)	/* RAS to CAS delay, 0x0=1cycle   0x1=2cycle, 0x2=3cycles, 0x3=4cycles */
#define SDR_RC		(0x8<<12)	/* Row Cycle, 0x0=1cycle, 0x1=2cycle ~ 0xf=16cycles */
#define SDR_RAS		(0x5<<16)	/* Row Active time, 0x0=1cycle, 0x1=2cycle ~  0xf=16cycles */

#define rSDRAM64MCFGREG	SDR_XW+SDR_AP+SDR_CL+SDR_D1+SDR_D0+SDR_RP+SDR_RCD+SDR_RC+SDR_RAS

/*
 * 	TimYu modified 20051013 for support 32M sdram
 */
#define	SDR_D1_32M	(0x2<<4)
#define	SDR_D0_32M	(0x2<<6)
#define rSDRAM32MCFGREG	SDR_XW+SDR_AP+SDR_CL+SDR_D1_32M+SDR_D0_32M+SDR_RP+SDR_RCD+SDR_RC+SDR_RAS

/*	CMDREG: SDRAM CONFIGURATION REGISTER 1  */

#define SDR_INITS		(0x0<<0)	/* Control bits for SDRAM Device initialization */
									/*      0x0=Normal operation 0x3=Reserved */
									/*      0x1=Auto issue a PALL to the SDRAM */
									/*      0x2=Auto issue a MRS  tot the SDRAM */
#define SDR_NORMAL		(0x0<<0)
#define SDR_PALL		(0x1<<0)
#define SDR_MRS 		(0x2<<0)
//#define SDR_WBUF              (0x0<<2)        /* Write Buffer Enable */       // 2030302 remarked by drsohn
#define SDR_WBUF		(0x1<<2)	/* Write Buffer Enable */
									/*      0x0=Merging Write buffer disable */
									/*      0x1=Merging Write buffer enable */
#define SDR_BUSY		(0x1<<3)	/* SDRAM controller status bit (READ ONLY) */
									/*      0x0: IDLE, 0x1: BUSY */

/*	CMDREG: SDRAM CONFIGURATION REGISTER 1  */

#define rSDRAMCMDREG	SDR_INITS+SDR_WBUF
										/* Refresh      =       (15.6*(fMCLK/Mhz)) */
										/* SDRAMCONFIG1 = (fMCLK/Mhz)*15 + (6*(fMCLK/Mhz))/10 */
										/* Clock >= 64/(15.6*Mhz) = 4.3Mh */
										/* Common refresh time period of 15.6us , SYSTEM CLK 66Mhz */
										/*      15.6Mhz X 66Mhz = 1029  refresh time */
#define rSDRAMWBTO		0x0

/* ----------------------------------------------------------------------------- */
/* S3C2510X Interrupt Number (Index value of interrupt sources)					*/
/* ----------------------------------------------------------------------------- */

#define INT_NUM_LEVELS	39
#define INT_ALL_MASK	0xffffffff
#define EXT_ALL_MASK	0x8000007f

#define	NR_EXT_IRQS	7	/* External interrupt sources number */

/* External Interrupt */
#define INT_LVL_EXT0			0	/* External Interrupt 0 */
#define INT_LVL_EXT1			1	/* External Interrupt 1 */
#define INT_LVL_EXT2			2	/* External Interrupt 2 */
#define INT_LVL_EXT3			3	/* External Interrupt 3 */
#define INT_LVL_EXT4			4	/* External Interrupt 4 */
#define INT_LVL_EXT5			5	/* External Interrupt 5 */
#define INT_LVL_IOM2			6	/* IOM2 Interrupt */

#define NR_INT_IRQS	32	/* Internal interrupt sources number */

/* Internal Interrupt */
#define INT_LVL_IICC			7	/* IICC Interrupt */
#define INT_LVL_HUART0_TX		8	/* HUART0 TX Interrupt */
#define INT_LVL_HUART0_RX		9	/* HUART0 RX Interrupt */
#define INT_LVL_HUART1_TX		10	/* HUART1 TX Interrupt */
#define INT_LVL_HUART1_RX		11	/* HUART1 RX Interrupt */
#define INT_LVL_CUART_TX		12	/* CUART TX Interrupt */
#define INT_LVL_CUART_RX		13	/* CUART RX Interrupt */
#define INT_LVL_USB				14	/* USB Interrupt */
#define INT_LVL_HDLC_TX0		15	/* HDLC Tx0 Interrupt */
#define INT_LVL_HDLC_RX0		16	/* HDLC Rx0 Interrupt */
#define INT_LVL_HDLC_TX1		17	/* HDLC Tx1 Interrupt */
#define INT_LVL_HDLC_RX1		18	/* HDLC Rx1 Interrupt */
#define INT_LVL_HDLC_TX2		19	/* HDLC Tx2 Interrupt */
#define INT_LVL_HDLC_RX2		20	/* HDLC Rx2 Interrupt */
#define INT_LVL_ETHTxA			21	/* Ethernet TX0 Interrupt */
#define INT_LVL_ETHRxA			22	/* Ethernet RX0 Interrupt */
#define INT_LVL_ETHTxB			23	/* Ethernet Tx1 Interrupt */
#define INT_LVL_ETHRxB			24	/* Ethernet Rx1 Interrupt */
#define INT_LVL_DES				25	/* DES Interrupt */
#define INT_LVL_GDMA0			26	/* GDMA Channel 0 Interrupt */
#define INT_LVL_GDMA1			27	/* GDMA Channel 1 Interrupt */
#define INT_LVL_GDMA2			28	/* GDMA Channel 2 Interrupt */
#define INT_LVL_GDMA3			29	/* GDMA Channel 3 Interrupt */
#define INT_LVL_GDMA4			30	/* GDMA Channel 4 Interrupt */
#define INT_LVL_GDMA5			31	/* GDMA Channel 5 Interrupt */
#define INT_LVL_TIMER0			32	/* 32-bit Timer 0 Interrupt */
#define INT_LVL_TIMER1			33	/* 32-bit Timer 1 Interrupt */
#define INT_LVL_TIMER2			34	/* 32-bit Timer 2 Interrupt */
#define INT_LVL_TIMER3			35	/* 32-bit Timer 3 Interrupt */
#define INT_LVL_TIMER4			36	/* 32-bit Timer 4 Interrupt */
#define INT_LVL_TIMER5			37	/* 32-bit Timer 5 Interrupt */
#define INT_LVL_WATCHDOG		38	/* Watchdog Timer Interrupt */

/* 20020612 by drsohn */
#define VPint	*(volatile unsigned int *)
#define UARTSTAT0           (VPint(ASIC_BASE+0x60004))
#define UARTTXH0            (VPint(ASIC_BASE+0x6000c))
#define UARTRXB0            (VPint(ASIC_BASE+0x60010))
/* drsohn end */

//-----------------------------------------------------------------------------
// Declare function protype
//-----------------------------------------------------------------------------
/*
extern void reset_cpu(ulong addr);
extern void intDisable(int vect);
*/
#define CONFIG_UCBOOTSTRAP	/* Use uCboot emulation */

#endif
